<?php
/**
 * Optimized Database Class with Performance Monitoring
 */

require_once __DIR__ . '/../includes/performance_monitor.php';

class DatabaseOptimized {
    private $host = 'localhost';
    private $db_name = 'boxxenstopp';
    private $username = 'root';
    private $password = '';
    private $conn;
    private static $instance = null;
    private $cache = [];
    private $query_count = 0;

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function getConnection() {
        if ($this->conn === null) {
            $start_time = microtime(true);
            
            try {
                $this->conn = new PDO(
                    "mysql:host=" . $this->host . ";dbname=" . $this->db_name . ";charset=utf8mb4",
                    $this->username,
                    $this->password,
                    [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                        PDO::ATTR_PERSISTENT => true,
                        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci",
                        PDO::ATTR_EMULATE_PREPARES => false,
                        PDO::ATTR_STRINGIFY_FETCHES => false,
                        PDO::MYSQL_ATTR_USE_BUFFERED_QUERY => true
                    ]
                );
                
                $connection_time = microtime(true) - $start_time;
                PerformanceMonitor::logQuery("CONNECTION", [], $connection_time);
                
            } catch(PDOException $exception) {
                error_log("Database connection error: " . $exception->getMessage());
                throw $exception;
            }
        }
        return $this->conn;
    }

    public function execute($sql, $params = []) {
        $start_time = microtime(true);
        $this->query_count++;
        
        try {
            $stmt = $this->getConnection()->prepare($sql);
            $result = $stmt->execute($params);
            
            $execution_time = microtime(true) - $start_time;
            PerformanceMonitor::logQuery($sql, $params, $execution_time);
            
            return $stmt;
        } catch (PDOException $e) {
            error_log("Query error: " . $e->getMessage() . " SQL: " . $sql);
            throw $e;
        }
    }

    public function fetchAll($sql, $params = []) {
        $stmt = $this->execute($sql, $params);
        return $stmt->fetchAll();
    }

    public function fetch($sql, $params = []) {
        $stmt = $this->execute($sql, $params);
        return $stmt->fetch();
    }

    public function fetchColumn($sql, $params = []) {
        $stmt = $this->execute($sql, $params);
        return $stmt->fetchColumn();
    }

    // Cached methods for frequently accessed data
    public function getCachedData($key, $callback, $ttl = 300) {
        if (isset($this->cache[$key]) && 
            isset($this->cache[$key]['timestamp']) && 
            (time() - $this->cache[$key]['timestamp']) < $ttl) {
            return $this->cache[$key]['data'];
        }

        $data = $callback();
        $this->cache[$key] = [
            'data' => $data,
            'timestamp' => time()
        ];

        return $data;
    }

    public function clearCache($key = null) {
        if ($key === null) {
            $this->cache = [];
        } else {
            unset($this->cache[$key]);
        }
    }

    public function getQueryCount() {
        return $this->query_count;
    }

    // Optimized user data retrieval
    public function getUserData($user_id) {
        return $this->getCachedData("user_data_$user_id", function() use ($user_id) {
            return $this->fetch("
                SELECT u.full_name, u.username, u.role, u.autohaus_id, u.can_switch_autohaus, u.theme, a.name as autohaus_name
                FROM admin_users u
                LEFT JOIN autohaus a ON u.autohaus_id = a.id
                WHERE u.id = ?
            ", [$user_id]);
        }, 600); // Cache for 10 minutes
    }

    // Optimized dashboard stats
    public function getDashboardStats($autohaus_id = null, $user_role = 'admin', $can_switch_autohaus = false) {
        $cacheKey = "dashboard_stats_" . ($autohaus_id ?? 'all') . "_" . $user_role . "_" . ($can_switch_autohaus ? '1' : '0');
        
        return $this->getCachedData($cacheKey, function() use ($autohaus_id, $user_role, $can_switch_autohaus) {
            $where_clause = "WHERE e.is_active = 1";
            $params = [];
            
            if ($user_role !== 'admin' || !$can_switch_autohaus) {
                if ($autohaus_id) {
                    $where_clause .= " AND e.autohaus_id = ?";
                    $params[] = $autohaus_id;
                } else {
                    $where_clause .= " AND e.autohaus_id IS NULL";
                }
            }
            
            $sql = "
                SELECT 
                    (SELECT COUNT(*) FROM events e $where_clause) as total_events,
                    (SELECT COUNT(*) FROM appointments a WHERE 1=1" . 
                    ($user_role !== 'admin' || !$can_switch_autohaus ? 
                        ($autohaus_id ? " AND a.autohaus_id = ?" : " AND a.autohaus_id IS NULL") : "") . 
                    ") as total_appointments,
                    (SELECT COUNT(*) FROM appointments a WHERE a.slot_date = CURDATE()" . 
                    ($user_role !== 'admin' || !$can_switch_autohaus ? 
                        ($autohaus_id ? " AND a.autohaus_id = ?" : " AND a.autohaus_id IS NULL") : "") . 
                    ") as today_appointments,
                    (SELECT COUNT(*) FROM appointments WHERE MONTH(slot_date) = MONTH(CURDATE()) AND YEAR(slot_date) = YEAR(CURDATE())" . 
                    ($user_role !== 'admin' || !$can_switch_autohaus ? 
                        ($autohaus_id ? " AND autohaus_id = ?" : " AND autohaus_id IS NULL") : "") . 
                    ") as month_appointments
            ";
            
            $all_params = array_merge($params, $params, $params, $params);
            return $this->fetch($sql, $all_params);
        }, 300); // Cache for 5 minutes
    }
}
?>