/**
 * Boxenstop - Booking JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    const bookingForm = document.getElementById('booking-form');
    const dateInput = document.getElementById('booking_date');
    const timeSlotsContainer = document.getElementById('time-slots');
    const submitBtn = document.getElementById('submit-btn');
    const bringsWheelsRadios = document.querySelectorAll('input[name="brings_wheels"]');
    const storageNumberGroup = document.getElementById('storage_number_group');
    const storageNumberInput = document.getElementById('storage_number');
    const tireTypeGroup = document.getElementById('tire_type_group');
    
    let selectedSlot = null;
    let eventId = getEventIdFromDom();
    
    // Add event listeners to service checkboxes
    const serviceCheckboxes = document.querySelectorAll('input[name^="service_"]');
    if (serviceCheckboxes && serviceCheckboxes.length > 0) {
    serviceCheckboxes.forEach(checkbox => {
            if (checkbox) {
        checkbox.addEventListener('change', updatePricing);
            }
    });
    }
    let pricing = {};
    
    // Initialize booking form
    initBookingForm();
    
    /**
     * Initialize booking form
     */
    function initBookingForm() {
        // Load time slots when date changes
        if (dateInput) {
        dateInput.addEventListener('change', loadTimeSlots);
        // For single-day events, lock date to event date
        if (dateInput.hasAttribute('max') && dateInput.getAttribute('max') === dateInput.getAttribute('min')) {
            dateInput.value = dateInput.getAttribute('min');
            }
        }
        
        // Handle brings wheels radio change
        if (bringsWheelsRadios && bringsWheelsRadios.length > 0) {
        bringsWheelsRadios.forEach(radio => {
                if (radio) {
            radio.addEventListener('change', handleBringsWheelsChange);
                }
        });
        }
        
        // Handle tire type radio change - verhindere Buchung bei "Lose Reifen"
        const tireTypeRadios = document.querySelectorAll('input[name="tire_type"]');
        if (tireTypeRadios && tireTypeRadios.length > 0) {
            tireTypeRadios.forEach(radio => {
                if (radio) {
                    radio.addEventListener('change', checkTireTypeAndDisableBooking);
                }
            });
        }
        
        // Wiper blades checkbox wurde entfernt
        
        // Handle seasonal check radio change
        const seasonalCheckRadios = document.querySelectorAll('input[name="seasonal_check"]');
        if (seasonalCheckRadios && seasonalCheckRadios.length > 0) {
        seasonalCheckRadios.forEach(radio => {
                if (radio) {
            radio.addEventListener('change', updatePricing);
                }
        });
        }
        
        // Add event listeners to all form inputs for real-time pricing updates
        const allInputs = document.querySelectorAll('input[type="radio"], input[type="checkbox"], select');
        if (allInputs && allInputs.length > 0) {
        allInputs.forEach(input => {
                if (input) {
            input.addEventListener('change', updatePricing);
                }
        });
        }
        
        // Speziell für wheels_storage_action Select
        const wheelsStorageAction = document.querySelector('select[name="wheels_storage_action"]');
        if (wheelsStorageAction) {
            wheelsStorageAction.addEventListener('change', updatePricing);
        }
        
        // Handle form submission
        if (bookingForm) {
        bookingForm.addEventListener('submit', handleFormSubmit);
        }
        
        // Load initial time slots
        loadTimeSlots();
        
        // Load pricing settings
        loadPricingSettings();
        
        // Handle brings wheels change on page load
        handleBringsWheelsChange();
        
        // Prüfe tire_type beim Laden der Seite
        checkTireTypeAndDisableBooking();
        
        // Update pricing on page load
        updatePricing();
    }
    
    /**
     * Handle brings wheels radio change
     */
    function handleBringsWheelsChange() {
        const checkedRadio = document.querySelector('input[name="brings_wheels"]:checked');
        if (!checkedRadio) {
            return;
        }
        const bringsWheels = checkedRadio.value === '1';
        
        if (bringsWheels) {
            // If bringing wheels, hide storage number and show tire type
            if (storageNumberGroup) storageNumberGroup.style.display = 'none';
            if (storageNumberInput) storageNumberInput.required = false;
            if (tireTypeGroup) tireTypeGroup.style.display = 'block';
        } else {
            // If not bringing wheels, show storage number and hide tire type
            if (storageNumberGroup) storageNumberGroup.style.display = 'block';
            if (storageNumberInput) {
            storageNumberInput.required = true;
            // Auto-generate storage number if empty
            if (!storageNumberInput.value) {
                generateStorageNumber();
            }
            }
            if (tireTypeGroup) tireTypeGroup.style.display = 'none';
        }
        
        // Prüfe tire_type nach Änderung (wichtig, da tire_type_group jetzt sichtbar/unsichtbar ist)
        checkTireTypeAndDisableBooking();
        
        // Update pricing
        updatePricing();
    }
    
    /**
     * Check tire type and disable booking if "Lose Reifen" is selected
     */
    function checkTireTypeAndDisableBooking() {
        const looseTireSelected = document.querySelector('input[name="tire_type"][value="loose"]:checked');
        const tireTypeGroup = document.getElementById('tire_type_group');
        
        if (looseTireSelected) {
            // Deaktiviere Submit-Button
            submitBtn.disabled = true;
            
            // Entferne Slot-Auswahl falls vorhanden
            if (selectedSlot) {
                document.querySelectorAll('.time-slot.selected').forEach(el => {
                    el.classList.remove('selected');
                });
                selectedSlot = null;
                
                // Entferne hidden input für slot_time
                const slotTimeInput = document.querySelector('input[name="slot_time"]');
                if (slotTimeInput) {
                    slotTimeInput.remove();
                }
                
                // Update summary
                const summary = document.getElementById('summary-slot');
                if (summary) summary.textContent = 'Bitte Slot wählen';
            }
            
            // Zeige Warnung in der tire_type_group
            let warningDiv = tireTypeGroup?.querySelector('.loose-tire-warning');
            if (!warningDiv && tireTypeGroup) {
                warningDiv = document.createElement('div');
                warningDiv.className = 'loose-tire-warning';
                warningDiv.style.cssText = 'background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px; padding: 1rem; margin-top: 1rem; color: #856404;';
                warningDiv.innerHTML = '<i class="fas fa-exclamation-triangle"></i> <strong>Hinweis:</strong> Lose Reifen können am Tag des Services nicht gewechselt werden. Bitte wählen Sie "Kompletträder / Radsatz" aus, um einen Termin zu buchen.';
                tireTypeGroup.appendChild(warningDiv);
            }
        } else {
            // Aktiviere Submit-Button nur wenn ein Slot ausgewählt ist
            if (selectedSlot) {
                submitBtn.disabled = false;
            }
            
            // Entferne Warnung
            const warningDiv = tireTypeGroup?.querySelector('.loose-tire-warning');
            if (warningDiv) {
                warningDiv.remove();
            }
        }
    }
    
    /**
     * Generate a storage number
     */
    function generateStorageNumber() {
        // Generate a random 12-digit number
        const randomNumber = Math.floor(100000000000 + Math.random() * 900000000000);
        // Format as 9.xxx.xxx.xxx
        const formattedNumber = '9.' + randomNumber.toString().substring(0, 3) + '.' + randomNumber.toString().substring(3, 6) + '.' + randomNumber.toString().substring(6, 9);
        storageNumberInput.value = formattedNumber;
    }
    
    /**
     * Get event ID from URL
     */
    function getEventIdFromDom() {
        const hidden = document.getElementById('event_id');
        if (hidden && hidden.value) return hidden.value;
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.get('event');
    }
    
    /**
     * Load available time slots
     */
    function loadTimeSlots() {
        const selectedDate = dateInput.value;
        
        if (!selectedDate || !eventId) {
            return;
        }
        
        timeSlotsContainer.innerHTML = '<p class="loading">Lade verfügbare Zeitslots...</p>';
        
        // Use the new dedicated API endpoint for slots
        fetch(`api/get_available_slots.php?event_id=${eventId}&date=${selectedDate}`)
            .then(response => response.json())
                .then(data => {
                if (data.success) {
                    console.log('API Response:', data);
                    console.log('Available Slots:', data.available_slots);
                    console.log('Total Booked:', data.total_booked);
                    console.log('Total Capacity:', data.total_capacity);
                    
                    // Debug: Zeige Debug-Informationen
                    if (data.debug) {
                        console.log('=== DEBUG INFORMATION ===');
                        console.log('Event ID:', data.debug.event_id);
                        console.log('Request Date:', data.debug.request_date);
                        console.log('Event Date:', data.debug.event_date);
                        console.log('Date Match:', data.debug.date_match ? '✅ YES' : '❌ NO - THIS IS THE PROBLEM!');
                        console.log('Booked Slots Raw Count:', data.debug.booked_slots_raw_count);
                        console.log('Booked Slots Raw:', data.debug.booked_slots_raw);
                        console.log('Booked Slots Normalized Count:', data.debug.booked_slots_normalized_count);
                        console.log('Booked Slots Normalized:', data.debug.booked_slots_normalized);
                        if (data.debug.debug_info) {
                            console.log('Debug Info (Total/For Date/With Status):', data.debug.debug_info);
                        }
                        if (data.debug.sample_appointments) {
                            console.log('Sample Appointments (if no matches found):', data.debug.sample_appointments);
                            console.log('⚠️ PROBLEM: Termine existieren, aber Datum stimmt nicht!');
                            console.log('   Gesuchtes Datum:', data.debug.date);
                            console.log('   Tatsächliche Termine:');
                            data.debug.sample_appointments.forEach((apt, idx) => {
                                console.log(`   ${idx + 1}. ID: ${apt.id}, Datum: ${apt.slot_date}, Zeit: ${apt.slot_time}, Status: ${apt.status}`);
                            });
                        }
                        console.log('=== END DEBUG ===');
                    }
                    
                    // Debug: Zeige erste Slots mit booked-Werten
                    if (data.available_slots && data.available_slots.length > 0) {
                        console.log('First 3 slots:', data.available_slots.slice(0, 3).map(s => ({
                            time: s.time,
                            booked: s.booked,
                            available_count: s.available_count,
                            max_count: s.max_count
                        })));
                    }
                    
                    displayTimeSlots(data.available_slots);
                    renderOccupancy(eventId, selectedDate, data);
                } else {
                    console.error('API Error:', data.error);
                    showError('Fehler beim Laden der Zeitslots: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error loading time slots:', error);
                showError('Fehler beim Laden der Zeitslots');
            });
    }
    
    /**
     * Display time slots
     */
    function displayTimeSlots(slots) {
        if (slots.length === 0) {
            timeSlotsContainer.innerHTML = '<p class="no-slots">Keine verfügbaren Zeitslots für dieses Datum.</p>';
            return;
        }
        
        timeSlotsContainer.innerHTML = '';
        
        slots.forEach(slot => {
            const slotElement = createSlotElement(slot);
            timeSlotsContainer.appendChild(slotElement);
        });
    }
    
    /**
     * Create slot element
     */
    function createSlotElement(slot) {
        const slotDiv = document.createElement('div');
        const isAvailable = slot.available && slot.available_count > 0;
        slotDiv.className = `time-slot ${!isAvailable ? 'unavailable' : ''}`;
        
        if (isAvailable) {
            slotDiv.addEventListener('click', () => selectSlot(slot, slotDiv));
        }
        
        // Berechne Auslastung
        const booked = slot.booked || 0;
        const maxCount = slot.max_count || 0;
        const availableCount = slot.available_count || 0;
        const occupancyPercentage = maxCount > 0 ? Math.round((booked / maxCount) * 100) : 0;
        
        // Erstelle Dots für die Auslastung
        let dotsHTML = '';
        for (let i = 0; i < maxCount; i++) {
            const isBooked = i < booked;
            dotsHTML += `<span class="occupancy-dot ${isBooked ? 'booked' : 'free'}" title="${isBooked ? 'Belegt' : 'Frei'}"></span>`;
        }
        
        slotDiv.innerHTML = `
            <div class="time">${slot.formatted_time}</div>
            <div class="availability" style="display: none;">
                ${isAvailable ? `${slot.available_count}/${slot.max_count} frei` : 'Ausgebucht'}
            </div>
            <div class="occupancy-indicator">
                <i class="fas fa-users" title="Belegung"></i>
                <div class="occupancy-dots">${dotsHTML}</div>
                <div class="occupancy-numbers">${booked}/${maxCount}</div>
            </div>
        `;
        
        return slotDiv;
    }
    
    /**
     * Select time slot
     */
    function selectSlot(slot, element) {
        // Remove previous selection
        document.querySelectorAll('.time-slot.selected').forEach(el => {
            el.classList.remove('selected');
        });
        
        // Select new slot
        element.classList.add('selected');
        selectedSlot = slot;
        
        // Prüfe ob "Lose Reifen" ausgewählt ist - wenn ja, Button nicht aktivieren
        const looseTireSelected = document.querySelector('input[name="tire_type"][value="loose"]:checked');
        if (looseTireSelected) {
            // Zeige Warnung und deaktiviere Button
            showError('Lose Reifen können am Tag des Services nicht gewechselt werden. Bitte wählen Sie "Kompletträder / Radsatz" aus.');
            submitBtn.disabled = true;
        } else {
            // Enable submit button nur wenn nicht "Lose Reifen" ausgewählt
            submitBtn.disabled = false;
        }
        
        // Add hidden input for slot time
        let slotTimeInput = document.querySelector('input[name="slot_time"]');
        if (!slotTimeInput) {
            slotTimeInput = document.createElement('input');
            slotTimeInput.type = 'hidden';
            slotTimeInput.name = 'slot_time';
            bookingForm.appendChild(slotTimeInput);
        }
        slotTimeInput.value = slot.time;
        
        // Show success message nur wenn nicht "Lose Reifen" ausgewählt
        if (!looseTireSelected) {
            showSuccess(`Zeitslot ${slot.formatted_time} ausgewählt`);
        }

        // Update summary card
        const summary = document.getElementById('summary-slot');
        if (summary) summary.textContent = slot.formatted_time;
    }
    
    /**
     * Wiper blades functionality wurde entfernt
     */
    
    /**
     * Load pricing settings
     */
    function loadPricingSettings() {
        console.log('Loading autohaus pricing settings...');
        
        // Verwende Autohaus-spezifische Preise falls verfügbar
        if (window.autohausPricing) {
            pricing = window.autohausPricing;
            console.log('Autohaus pricing loaded successfully:', pricing);
            updatePricing();
            return;
        }
        
        // Fallback zu globalen Preisen
        console.log('Loading global pricing settings...');
        fetch('api/settings.php')
            .then(response => {
                console.log('Settings API response status:', response.status);
                return response.json();
            })
            .then(data => {
                console.log('Settings API response data:', data);
                if (data.success) {
                    pricing = data.settings;
                    console.log('Global pricing loaded successfully:', pricing);
                    updatePricing();
                } else {
                    console.error('Settings API returned error:', data.error);
                    setFallbackPricing();
                }
            })
            .catch(error => {
                console.error('Error loading pricing settings:', error);
                setFallbackPricing();
            });
    }
    
    /**
     * Set fallback pricing
     */
    function setFallbackPricing() {
        pricing = {
            price_wheel_change: 'auf_anfrage',
            price_storage: 'auf_anfrage',
            price_spring_check: 'auf_anfrage',
            price_winter_check: 'auf_anfrage'
        };
        console.log('Using fallback pricing:', pricing);
        updatePricing();
    }
    
    function loadServices() {
        if (window.autohausServices && window.autohausServices.length > 0) {
            console.log('Loaded services:', window.autohausServices);
            return window.autohausServices;
        }
        return [];
    }
    
    /**
     * Update pricing display - Wird sofort bei jeder Änderung aufgerufen
     */
    function updatePricing() {
        console.log('updatePricing called, pricing object:', pricing);
        
        if (!pricing || !pricing.price_wheel_change) {
            console.log('Pricing not loaded yet, using fallback');
            // Use fallback pricing if not loaded yet
            pricing = {
                price_wheel_change: 'auf_anfrage',
                price_storage: 'auf_anfrage',
                price_spring_check: 'auf_anfrage',
                price_winter_check: 'auf_anfrage'
            };
        }
        
        const summaryPricing = document.getElementById('summary-pricing');
        const summaryTotal = document.getElementById('summary-total');
        const priceBreakdown = document.getElementById('price-breakdown');
        const totalPrice = document.getElementById('total-price');
        
        if (!summaryPricing || !summaryTotal || !priceBreakdown || !totalPrice) {
            console.log('Required elements not found:', {
                summaryPricing: !!summaryPricing,
                summaryTotal: !!summaryTotal,
                priceBreakdown: !!priceBreakdown,
                totalPrice: !!totalPrice
            });
            return;
        }
        
        let total = 0;
        let breakdown = [];
        
        // Base price for wheel change
        if (pricing.price_wheel_change === 'auf_anfrage') {
            breakdown.push(`Räderwechsel: ${pricing.price_wheel_change}`);
        } else {
            total += parseFloat(pricing.price_wheel_change);
            breakdown.push(`Räderwechsel: ${pricing.price_wheel_change}€`);
        }
        console.log('Base wheel change price added:', pricing.price_wheel_change);
        
        // Storage price logic
        // Einlagerung wird NUR berechnet, wenn die abmontierten Räder eingelagert werden sollen
        const wheelsStorageAction = document.querySelector('select[name="wheels_storage_action"]');
        
        console.log('Wheels storage action:', wheelsStorageAction ? wheelsStorageAction.value : 'not found');
        
        // Einlagerung NUR berechnen wenn "Einlagern" für die abmontierten Räder ausgewählt ist
        if (wheelsStorageAction && wheelsStorageAction.value === 'store') {
            if (pricing.price_storage === 'auf_anfrage') {
                breakdown.push(`Einlagerung: ${pricing.price_storage}`);
            } else {
                total += parseFloat(pricing.price_storage);
                breakdown.push(`Einlagerung: ${pricing.price_storage}€`);
            }
            console.log('Added storage price for wheels to be stored:', pricing.price_storage);
        }
        
        // Zeige an, was mit abmontierten Rädern gemacht wird (nur wenn Räder mitgebracht werden)
        const bringsWheels = document.querySelector('input[name="brings_wheels"]:checked');
        if (bringsWheels && bringsWheels.value === '1' && wheelsStorageAction && wheelsStorageAction.value) {
            const storageActionText = wheelsStorageAction.value === 'store' ? 'Einlagern' : 'In Fahrzeug legen';
            breakdown.push(`Abmontierte Räder: ${storageActionText}`);
            console.log('Added wheels storage action:', storageActionText);
        }
        
        // Dynamische Services - SOFORT anzeigen wenn ausgewählt
        const services = loadServices();
        services.forEach(service => {
            const serviceCheckbox = document.querySelector(`input[name="service_${service.id}"]`);
            if (serviceCheckbox && serviceCheckbox.checked) {
                if (service.show_price && service.price) {
                    total += parseFloat(service.price);
                    breakdown.push(`${service.service_name}: ${service.price}€`);
                } else {
                    breakdown.push(`${service.service_name}: auf Anfrage`);
                }
                console.log('Added service:', service.service_name, service.price);
            }
        });
        
        // Seasonal check - check both radio buttons and select dropdown - SOFORT anzeigen
        const seasonalCheckRadio = document.querySelector('input[name="seasonal_check"]:checked');
        const seasonalCheckSelect = document.querySelector('select[name="seasonal_check"]');
        const seasonalCheckValue = seasonalCheckRadio ? seasonalCheckRadio.value : (seasonalCheckSelect ? seasonalCheckSelect.value : 'none');
        
        console.log('Seasonal check value:', seasonalCheckValue);
        if (seasonalCheckValue && seasonalCheckValue !== 'none') {
            const checkPrice = seasonalCheckValue === 'spring' ? pricing.price_spring_check : pricing.price_winter_check;
            const checkName = seasonalCheckValue === 'spring' ? 'Frühjahres-Check' : 'Winter-Check';
            
            if (checkPrice === 'auf_anfrage') {
                breakdown.push(`${checkName}: ${checkPrice}`);
            } else {
                total += parseFloat(checkPrice);
                breakdown.push(`${checkName}: ${checkPrice}€`);
            }
            console.log('Added seasonal check price:', checkPrice);
        }
        
        // Update display - IMMER aktualisieren, auch wenn nur eine Option geändert wurde
        console.log('Final breakdown:', breakdown);
        console.log('Final total:', total);
        
        // Zeige Zusammenfassung an, sobald mindestens ein Eintrag vorhanden ist
        if (breakdown.length > 0) {
            priceBreakdown.innerHTML = breakdown.map(item => `<div class="price-item">${item}</div>`).join('');
            summaryPricing.style.display = 'block';
            summaryTotal.style.display = 'block';
            
            // Check if all prices are "auf_anfrage"
            const hasAufAnfrage = breakdown.some(item => item.includes('auf_anfrage'));
            if (hasAufAnfrage && total === 0) {
                totalPrice.textContent = 'auf Anfrage';
            } else {
                totalPrice.textContent = `${total.toFixed(2)}€`;
            }
        } else {
            // Zeige zumindest die Basis-Informationen
            summaryPricing.style.display = 'block';
            summaryTotal.style.display = 'block';
            priceBreakdown.innerHTML = '<div class="price-item">Bitte Optionen auswählen</div>';
            totalPrice.textContent = 'auf Anfrage';
        }
    }
    
    /**
     * Format storage number as user types (9.xxx.xxx.xxx)
     */
    if (storageNumberInput) {
        storageNumberInput.addEventListener('input', function(e) {
            // Remove all non-digit characters
            let value = this.value.replace(/\D/g, '');
            
            // Add dots after first digit and after next 3 digits
            if (value.length > 1) {
                value = value.charAt(0) + '.' + value.substring(1, 4) + (value.length > 4 ? '.' + value.substring(4, 7) : '') + (value.length > 7 ? '.' + value.substring(7, 10) : '');
            }
            
            // Update the input value
            this.value = value;
        });
    }
    
    /**
     * Handle form submission
     */
    function handleFormSubmit(e) {
        e.preventDefault();
        if (!selectedSlot) {
            showError('Bitte wählen Sie einen Zeitslot aus');
            return;
        }
        
        // Prüfe ob "Lose Reifen" ausgewählt ist - verhindere Buchung
        const looseTireSelected = document.querySelector('input[name="tire_type"][value="loose"]:checked');
        if (looseTireSelected) {
            showError('Lose Reifen können am Tag des Services nicht gewechselt werden. Bitte wählen Sie "Kompletträder / Radsatz" aus.');
            return;
        }
        
        // Prüfe ob "In Fahrzeug legen" ausgewählt ist - zeige Überzeugungs-Modal
        const bringsWheels = document.querySelector('input[name="brings_wheels"]:checked');
        const wheelsStorageAction = document.querySelector('select[name="wheels_storage_action"]');
        
        // Nur prüfen wenn Räder mitgebracht werden
        if (bringsWheels && bringsWheels.value === '1' && wheelsStorageAction && wheelsStorageAction.value === 'vehicle') {
            // Prüfe ob Modal existiert und öffne es direkt (falls Funktion noch nicht verfügbar)
            const modal = document.getElementById('storageConvictionModal');
            if (modal) {
                // Öffne Modal direkt
                modal.style.display = 'block';
                document.body.style.overflow = 'hidden';
                return; // Stoppe Submit, Modal wird weiterleiten
            } else if (window.openStorageConvictionModal) {
                // Fallback: Verwende globale Funktion falls verfügbar
                window.openStorageConvictionModal();
                return; // Stoppe Submit, Modal wird weiterleiten
            }
        }
        
        // Normale Submit-Logik
        proceedWithFormSubmission();
    }
    
    /**
     * Proceed with form submission (called after modal confirmation or directly)
     */
    function proceedWithFormSubmission() {
        // Validate form before submission
        if (!validateForm()) {
            return;
        }
        
        // Set loading state
        setButtonState('loading');
        
        // Collect form data
        const formData = new FormData(bookingForm);
        const data = Object.fromEntries(formData.entries());
        
        // Convert boolean values
        data.brings_wheels = data.brings_wheels === '1';
        data.has_wheel_lock = data.has_wheel_lock === '1';
        
        // Email is no longer required since we removed the email system

        // Submit form
        // Ensure event_id, slot_date and slot_time are present
        data.event_id = eventId;
        data.slot_date = data.slot_date || dateInput.value;
        data.slot_time = selectedSlot.time;
        
        // CSRF-Token hinzufügen
        const csrfMeta = document.querySelector('meta[name="csrf-token"]');
        if (!csrfMeta || !csrfMeta.content || csrfMeta.content.trim() === '') {
            console.error('CSRF Token nicht gefunden! Token:', csrfMeta ? csrfMeta.content : 'Meta-Tag nicht gefunden');
            console.error('Versuch, Token aus Session zu laden...');
            
            // Versuche Token aus Session zu laden (falls verfügbar)
            const tokenFromSession = sessionStorage.getItem('csrf_token');
            if (tokenFromSession) {
                console.log('Token aus Session geladen');
                data.csrf_token = tokenFromSession;
            } else {
                showError('Sicherheitstoken fehlt. Seite wird neu geladen...');
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
                return;
            }
        } else {
            data.csrf_token = csrfMeta.content.trim();
            // Speichere Token in Session Storage für Fallback
            sessionStorage.setItem('csrf_token', data.csrf_token);
        }
        
        console.log('CSRF Token wird gesendet:', data.csrf_token ? 'Token vorhanden (' + data.csrf_token.length + ' Zeichen)' : 'FEHLT');

        fetch('api/appointments.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(data)
        })
        .then(response => {
            // Prüfe ob 403 Forbidden (gesperrter Account)
            if (response.status === 403) {
                return response.json().then(result => {
                    throw new Error(result.error || 'Ihr Konto wurde gesperrt. Sie können keine Termine buchen.');
                });
            }
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(result => {
            if (result.success) {
                // Neues CSRF-Token speichern falls zurückgegeben
                if (result.csrf_token) {
                    const metaToken = document.querySelector('meta[name="csrf-token"]');
                    if (metaToken) {
                        metaToken.setAttribute('content', result.csrf_token);
                    }
                }
                
                setButtonState('success');
                
                // Redirect to index with success parameter
                window.location.href = 'index.php?success=request';
            } else {
                setButtonState('error');
                showError(result.error || 'Fehler beim Buchen des Termins');
                
                // Bei CSRF-Fehler Seite neu laden
                if (result.code === 'CSRF_TOKEN_INVALID') {
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                }
            }
        })
        .catch(error => {
            console.error('Error booking appointment:', error);
            setButtonState('error');
            
            // Spezielle Behandlung für gesperrte Accounts
            if (error.message && error.message.includes('gesperrt')) {
                showError(error.message);
                // Optional: Weiterleitung zur Startseite nach 3 Sekunden
                setTimeout(() => {
                    window.location.href = 'index.php?error=account_suspended';
                }, 3000);
            } else {
            showError('Fehler beim Buchen des Termins. Bitte versuchen Sie es erneut.');
            }
        });
    }

    /**
     * Render simple occupancy indicator for selected date
     */
    function renderOccupancy(eventId, date, apiData) {
        const container = document.getElementById('occupancy-calendar');
        if (!container) return;

        // Use API data for occupancy calculation
        const slots = apiData.available_slots || [];
        let totalPlaces = 0;
        let bookedPlaces = 0;
        let availableSlots = 0;
        let totalSlots = slots.length;
        
        console.log('renderOccupancy called with:', {
            slotsCount: slots.length,
            totalBookedFromAPI: apiData.total_booked,
            totalCapacityFromAPI: apiData.total_capacity
        });
        
        // Calculate total capacity and booked slots
        slots.forEach(slot => {
            const maxCount = parseInt(slot.max_count) || 0;
            const booked = parseInt(slot.booked) || 0;
            
            totalPlaces += maxCount;
            bookedPlaces += booked;
            
            if (slot.available && slot.available_count > 0) {
                availableSlots++;
            }
        });
        
        console.log('Calculated values:', {
            totalPlaces,
            bookedPlaces,
            availableSlots,
            totalSlots
        });
        
        // Calculate occupancy ratio
        const occupancyRatio = totalPlaces > 0 ? bookedPlaces / totalPlaces : 0;
        const occupancyPercentage = Math.round(occupancyRatio * 100);
        
        // Calculate slot utilization
        const slotUtilization = totalSlots > 0 ? (totalSlots - availableSlots) / totalSlots : 0;
        const slotUtilizationPercentage = Math.round(slotUtilization * 100);
        
        // Get status based on occupancy
        const statusClass = getLoadClass(occupancyRatio);
        const statusText = getOccupancyStatusText(occupancyPercentage);
        
        container.innerHTML = `
            <div class="occupancy-card">
                <div class="occupancy-header">
                    <h4><i class="fas fa-calendar-check"></i> Auslastung</h4>
                    <span class="occupancy-badge ${statusClass}">${occupancyPercentage}%</span>
                </div>
                
                <div class="occupancy-stats">
                    <div class="stat-item">
                        <div class="stat-label">
                            <i class="fas fa-users"></i>
                            Plätze
                        </div>
                        <div class="stat-value">
                            <span class="booked">${bookedPlaces}</span> / 
                            <span class="total">${totalPlaces}</span> belegt
                        </div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-label">
                            <i class="fas fa-clock"></i>
                            Zeitslots
                        </div>
                        <div class="stat-value">
                            <span class="available">${availableSlots}</span> von 
                            <span class="total">${totalSlots}</span> verfügbar
                        </div>
                    </div>
                </div>
                
                <div class="occupancy-bar">
                    <div class="occupancy-fill ${statusClass}" style="width:${occupancyPercentage}%"></div>
                </div>
                
                <div class="occupancy-status">
                    <i class="fas fa-${getOccupancyIcon(occupancyPercentage)}"></i>
                    <span class="status-text">${statusText}</span>
                </div>
            </div>
        `;
    }
    
    /**
     * Get occupancy status text
     */
    function getOccupancyStatusText(percentage) {
        if (percentage === 0) return 'Keine Buchungen';
        if (percentage < 25) return 'Geringe Auslastung';
        if (percentage < 50) return 'Moderate Auslastung';
        if (percentage < 75) return 'Hohe Auslastung';
        if (percentage < 100) return 'Sehr hohe Auslastung';
        return 'Ausgebucht';
    }
    
    /**
     * Get occupancy icon
     */
    function getOccupancyIcon(percentage) {
        if (percentage === 0) return 'leaf';
        if (percentage < 25) return 'check-circle';
        if (percentage < 50) return 'exclamation-circle';
        if (percentage < 75) return 'clock';
        if (percentage < 100) return 'exclamation-triangle';
        return 'times-circle';
    }

    function getLoadClass(ratio){
        if (ratio === 1) return 'full';
        if (ratio >= 0.75) return 'high';
        if (ratio >= 0.4) return 'medium';
        return 'low';
    }

    /**
     * Set button state with visual feedback
     */
    function setButtonState(state) {
        const icon = submitBtn.querySelector('i');
        const text = submitBtn.querySelector('span');
        
        // Remove all state classes
        submitBtn.classList.remove('btn-loading', 'btn-success', 'btn-error');
        
        switch (state) {
            case 'loading':
                submitBtn.disabled = true;
                submitBtn.classList.add('btn-loading');
                icon.className = 'fas fa-spinner fa-spin';
                text.textContent = 'Sende Anfrage...';
                break;
                
            case 'success':
                submitBtn.disabled = true;
                submitBtn.classList.add('btn-success');
                icon.className = 'fas fa-check-circle';
                text.textContent = 'Anfrage gesendet!';
                break;
                
            case 'error':
                submitBtn.disabled = false;
                submitBtn.classList.add('btn-error');
                icon.className = 'fas fa-exclamation-circle';
                text.textContent = 'Fehler - Erneut versuchen';
                // Reset to normal state after 3 seconds
                setTimeout(() => {
                    setButtonState('normal');
                }, 3000);
                break;
                
            case 'normal':
            default:
                submitBtn.disabled = false;
                icon.className = 'fas fa-paper-plane';
                text.textContent = 'Terminanfrage senden';
                break;
        }
    }
    
    /**
     * Disable form after successful booking
     */
    function disableForm() {
        const formInputs = bookingForm.querySelectorAll('input, select, textarea, button');
        formInputs.forEach(input => {
            if (input !== submitBtn) {
                input.disabled = true;
            }
        });
        
        // Disable time slot selection
        const timeSlots = document.querySelectorAll('.time-slot');
        timeSlots.forEach(slot => {
            slot.style.pointerEvents = 'none';
            slot.style.opacity = '0.6';
        });
    }
    
    /**
     * Show success message
     */
    function showSuccess(message) {
        const alert = document.createElement('div');
        alert.className = 'alert alert-success';
        alert.innerHTML = `<i class="fas fa-check-circle"></i> ${message}`;
        
        const container = document.querySelector('.booking-section .section-container');
        const existingAlert = container.querySelector('.alert');
        if (existingAlert) {
            existingAlert.remove();
        }
        
        container.insertBefore(alert, container.querySelector('.booking-layout'));
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (alert.parentNode) {
                alert.remove();
            }
        }, 5000);
    }
    
    /**
     * Show success message with calendar download option
     */
    function showSuccessWithCalendar(eventDetails) {
        const container = document.querySelector('.booking-section .section-container');
        const existingAlert = container.querySelector('.alert');
        if (existingAlert) {
            existingAlert.remove();
        }
        
        // Create success alert with calendar download
        const alert = document.createElement('div');
        alert.className = 'alert alert-success';
        alert.style.cssText = `
            background: #d1ecf1;
            border: 2px solid #bee5eb;
            border-radius: 12px;
            padding: 2rem;
            margin-bottom: 2rem;
            text-align: center;
        `;
        
        alert.innerHTML = `
            <div style="margin-bottom: 1.5rem;">
                <i class="fas fa-check-circle" style="font-size: 3rem; color: #28a745; margin-bottom: 1rem;"></i>
                <h3 style="color: #155724; margin-bottom: 0.5rem;">Termin erfolgreich gebucht!</h3>
                <p style="color: #0c5460;">Sie erhalten eine Bestätigung per E-Mail.</p>
            </div>
            
            <div style="background: white; padding: 1.5rem; border-radius: 8px; margin-bottom: 1.5rem; text-align: left;">
                <h4 style="color: #003057; margin-bottom: 1rem;"><i class="fas fa-calendar-alt"></i> Termin-Details</h4>
                <div style="display: grid; gap: 0.75rem;">
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Event:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.title}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Datum:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.date}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Zeit:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.time}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Kennzeichen:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.license_plate}</span>
                    </div>
                </div>
            </div>
            
            <div style="background: #fff3cd; border: 2px solid #ffc107; border-radius: 8px; padding: 1.25rem; margin-bottom: 1.5rem;">
                <h4 style="color: #856404; margin-bottom: 0.75rem;"><i class="fas fa-download"></i> In Kalender eintragen</h4>
                <p style="color: #856404; margin: 0 0 1rem 0;">Laden Sie die Kalenderdatei herunter und fügen Sie den Termin in Ihren Kalender ein (Google Calendar, Outlook, iCal, etc.).</p>
                <button onclick="downloadCalendar(${eventDetails.appointment_id})" style="
                    background: linear-gradient(135deg, #003057 0%, #0056b3 100%);
                    color: white;
                    border: none;
                    padding: 0.75rem 1.5rem;
                    border-radius: 8px;
                    cursor: pointer;
                    font-weight: 600;
                    font-size: 1rem;
                    transition: all 0.3s ease;
                    box-shadow: 0 4px 12px rgba(0, 48, 87, 0.3);
                    display: inline-flex;
                    align-items: center;
                    gap: 0.5rem;
                " onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 6px 16px rgba(0, 48, 87, 0.4)'" onmouseout="this.style.transform=''; this.style.boxShadow='0 4px 12px rgba(0, 48, 87, 0.3)'">
                    <i class="fas fa-download"></i>
                    Kalenderdatei herunterladen
                </button>
            </div>
            
            <div style="display: flex; gap: 1rem; justify-content: center;">
                <button onclick="window.location.href='index.php'" style="
                    background: #6c757d;
                    color: white;
                    border: none;
                    padding: 0.75rem 1.5rem;
                    border-radius: 8px;
                    cursor: pointer;
                    font-weight: 500;
                ">
                    <i class="fas fa-home"></i> Zur Startseite
                </button>
                <button onclick="window.location.href='profile.php'" style="
                    background: #0056b3;
                    color: white;
                    border: none;
                    padding: 0.75rem 1.5rem;
                    border-radius: 8px;
                    cursor: pointer;
                    font-weight: 500;
                ">
                    <i class="fas fa-user"></i> Zum Profil
                </button>
            </div>
        `;
        
        container.insertBefore(alert, container.querySelector('.booking-layout'));
        
        // Disable form
        disableForm();
        
        // Hide booking form
        bookingForm.style.display = 'none';
    }
    
    /**
     * Download calendar file
     */
    window.downloadCalendar = function(appointmentId) {
        window.location.href = `download_appointment_ics.php?id=${appointmentId}`;
    }
    
    /**
     * Show alert message - Höchste Priorität (höchster z-index)
     * Z-Index Hierarchie:
     * 9999999: Alert/Notification System (höchste Priorität)
     * 9999998: Loading Text
     * 9999997: Modal Content
     * 9999996: Modal Container
     * 9999995: Modal Backdrop
     */
    function showAlert(message, type = 'error') {
        // Entferne vorhandene Alerts
        const existingAlert = document.getElementById('booking-alert-container');
        if (existingAlert) {
            existingAlert.remove();
        }
        
        // Erstelle Alert-Container mit höchstem z-index
        const alertContainer = document.createElement('div');
        alertContainer.id = 'booking-alert-container';
        alertContainer.style.cssText = `
            position: fixed;
            top: 80px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 9999999 !important;
            max-width: 600px;
            width: 90%;
            animation: slideDownAlert 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        `;
        
        // Bestimme Farben basierend auf Typ
        const alertStyles = {
            'error': {
                background: '#f8d7da',
                color: '#721c24',
                border: '2px solid #f5c6cb',
                icon: 'fa-exclamation-circle'
            },
            'success': {
                background: '#d4edda',
                color: '#155724',
                border: '2px solid #c3e6cb',
                icon: 'fa-check-circle'
            },
            'warning': {
                background: '#fff3cd',
                color: '#856404',
                border: '2px solid #ffeaa7',
                icon: 'fa-exclamation-triangle'
            },
            'info': {
                background: '#d1ecf1',
                color: '#0c5460',
                border: '2px solid #bee5eb',
                icon: 'fa-info-circle'
            }
        };
        
        const style = alertStyles[type] || alertStyles['error'];
        
        alertContainer.innerHTML = `
            <div style="
                background: ${style.background};
                color: ${style.color};
                border: ${style.border};
                border-radius: 8px;
                padding: 1.25rem 1.5rem;
                box-shadow: 0 8px 24px rgba(0, 0, 0, 0.15);
                display: flex;
                align-items: center;
                gap: 1rem;
                font-weight: 500;
            ">
                <i class="fas ${style.icon}" style="font-size: 1.5rem; flex-shrink: 0;"></i>
                <span style="flex: 1;">${message}</span>
                <button onclick="this.closest('#booking-alert-container').remove()" style="
                    background: none;
                    border: none;
                    color: ${style.color};
                    cursor: pointer;
                    font-size: 1.25rem;
                    padding: 0;
                    width: 24px;
                    height: 24px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    opacity: 0.7;
                    transition: opacity 0.2s;
                " onmouseover="this.style.opacity='1'" onmouseout="this.style.opacity='0.7'">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        document.body.appendChild(alertContainer);
        
        // Auto remove nach 8 Sekunden (bei error) oder 5 Sekunden (bei anderen)
        const timeout = type === 'error' ? 8000 : 5000;
        setTimeout(() => {
            if (alertContainer.parentNode) {
                alertContainer.style.animation = 'slideUpAlert 0.3s cubic-bezier(0.4, 0, 0.2, 1)';
                setTimeout(() => {
                    alertContainer.remove();
                }, 300);
            }
        }, timeout);
    }
    
    /**
     * Show loading text directly under alert
     */
    function showLoadingText(text) {
        // Entferne vorhandenen Loading-Text
        const existingLoading = document.getElementById('booking-loading-text');
        if (existingLoading) {
            existingLoading.remove();
        }
        
        // Berechne Position basierend auf Alert
        const alertContainer = document.getElementById('booking-alert-container');
        let topPosition = 160; // Fallback
        
        if (alertContainer) {
            const alertRect = alertContainer.getBoundingClientRect();
            topPosition = alertRect.bottom + 10; // 10px Abstand unter dem Alert
        }
        
        // Erstelle Loading-Text direkt unter dem Alert
        const loadingText = document.createElement('div');
        loadingText.id = 'booking-loading-text';
        loadingText.style.cssText = `
            position: fixed;
            top: ${topPosition}px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 9999998 !important;
            max-width: 600px;
            width: 90%;
            background: rgba(0, 86, 179, 0.95);
            color: white;
            border-radius: 8px;
            padding: 1rem 1.5rem;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.2);
            display: flex;
            align-items: center;
            gap: 1rem;
            font-weight: 500;
            animation: slideDownAlert 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            backdrop-filter: blur(10px);
        `;
        
        loadingText.innerHTML = `
            <i class="fas fa-spinner fa-spin" style="font-size: 1.25rem;"></i>
            <span>${text}</span>
        `;
        
        document.body.appendChild(loadingText);
        
        // Update Position wenn Alert sich ändert
        if (alertContainer) {
            const observer = new MutationObserver(() => {
                const alertRect = alertContainer.getBoundingClientRect();
                loadingText.style.top = (alertRect.bottom + 10) + 'px';
            });
            observer.observe(alertContainer, { attributes: true, attributeFilter: ['style'] });
        }
    }
    
    /**
     * Hide loading text
     */
    function hideLoadingText() {
        const loadingText = document.getElementById('booking-loading-text');
        if (loadingText) {
            loadingText.style.animation = 'slideUpAlert 0.3s cubic-bezier(0.4, 0, 0.2, 1)';
            setTimeout(() => {
                loadingText.remove();
            }, 300);
        }
    }
    
    /**
     * Show error message (Legacy - verwendet jetzt showAlert)
     */
    function showError(message) {
        showAlert(message, 'error');
    }
    
    /**
     * Validate form before submission
     */
    function validateForm() {
        const requiredFields = bookingForm.querySelectorAll('[required]');
        let isValid = true;
        
        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                field.classList.add('error');
                isValid = false;
            } else {
                field.classList.remove('error');
            }
        });
        
        if (!selectedSlot) {
            showError('Bitte wählen Sie einen Zeitslot aus');
            isValid = false;
        }
        
        return isValid;
    }
    
    // Add CSS for booking specific styles
    const bookingStyle = document.createElement('style');
    bookingStyle.textContent = `
        .no-slots {
            text-align: center;
            color: #666;
            font-style: italic;
            padding: 2rem;
            background: #f8f9fa;
            border-radius: 8px;
            border: 2px dashed #dee2e6;
        }
        
        .time-slot {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .time-slot:hover:not(.unavailable) {
            transform: translateY(-2px);
        }
        
        .time-slot.unavailable {
            cursor: not-allowed;
            opacity: 0.6;
        }
        
        .time-slot.selected {
            transform: scale(1.05);
        }
        
        /* Occupancy Indicator - Dots und Zahlen */
        .occupancy-indicator {
            margin-top: 0.5rem;
            display: flex;
            flex-direction: row;
            align-items: center;
            justify-content: center;
            gap: 0.4rem;
        }
        
        .occupancy-indicator i {
            font-size: 0.7rem;
            color: #6b7280;
            opacity: 0.4;
        }
        
        .time-slot.selected .occupancy-indicator i {
            color: rgba(255, 255, 255, 0.6);
        }
        
        .occupancy-dots {
            display: flex;
            gap: 0.25rem;
            justify-content: center;
            flex-wrap: wrap;
            align-items: center;
        }
        
        .occupancy-dot {
            width: 6px;
            height: 6px;
            border-radius: 50%;
            display: inline-block;
            transition: all 0.2s ease;
        }
        
        .occupancy-dot.free {
            background: #d1d5db;
            opacity: 0.4;
        }
        
        .occupancy-dot.booked {
            background: #6b7280;
            opacity: 0.6;
        }
        
        .time-slot.selected .occupancy-dot.free {
            background: rgba(255, 255, 255, 0.5);
        }
        
        .time-slot.selected .occupancy-dot.booked {
            background: rgba(255, 255, 255, 0.8);
        }
        
        .occupancy-numbers {
            font-size: 0.7rem;
            color: #6b7280;
            opacity: 0.5;
            font-weight: 400;
            letter-spacing: 0.02em;
            white-space: nowrap;
        }
        
        .time-slot.selected .occupancy-numbers {
            color: rgba(255, 255, 255, 0.7);
        }
        
        .time-slot.unavailable .occupancy-numbers {
            opacity: 0.3;
        }
        
        /* Mobile Optimierung für Occupancy Indicator */
        @media (max-width: 768px) {
            .occupancy-dot {
                width: 5px;
                height: 5px;
            }
            
            .occupancy-dots {
                gap: 0.2rem;
            }
            
            .occupancy-numbers {
                font-size: 0.65rem;
            }
            
            .occupancy-indicator {
                gap: 0.3rem;
            }
            
            .occupancy-indicator i {
                font-size: 0.65rem;
            }
        }
        
        .field-error {
            color: #e10600;
            font-size: 0.875rem;
            margin-top: 0.25rem;
        }
        
        input.error {
            border-color: #e10600;
            box-shadow: 0 0 0 3px rgba(225, 6, 0, 0.1);
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .btn-loading {
            background: linear-gradient(135deg, #007bff, #0056b3);
            color: white;
            position: relative;
            overflow: hidden;
        }
        
        .btn-loading::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            animation: loading-shimmer 1.5s infinite;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #28a745, #1e7e34);
            color: white;
            border-color: #28a745;
            animation: success-pulse 0.6s ease-out;
        }
        
        .btn-error {
            background: linear-gradient(135deg, #dc3545, #c82333);
            color: white;
            border-color: #dc3545;
            animation: error-shake 0.5s ease-in-out;
        }
        
        .fa-spin {
            animation: fa-spin 1s infinite linear;
        }
        
        @keyframes fa-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        @keyframes loading-shimmer {
            0% { left: -100%; }
            100% { left: 100%; }
        }
        
        @keyframes success-pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }
        
        @keyframes error-shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }
        
        .alert {
            margin-bottom: 1rem;
            padding: 1rem;
            border-radius: 8px;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-weight: 500;
            animation: slideDown 0.3s ease-out;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Alert Animationen - Höchste Priorität */
        @keyframes slideDownAlert {
            from {
                opacity: 0;
                transform: translateX(-50%) translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateX(-50%) translateY(0);
            }
        }
        
        @keyframes slideUpAlert {
            from {
                opacity: 1;
                transform: translateX(-50%) translateY(0);
            }
            to {
                opacity: 0;
                transform: translateX(-50%) translateY(-20px);
            }
        }
    `;
    document.head.appendChild(bookingStyle);
    
    // Mache Funktionen global verfügbar
    window.showAlert = showAlert;
    window.showLoadingText = showLoadingText;
    window.hideLoadingText = hideLoadingText;
    window.proceedWithFormSubmission = proceedWithFormSubmission;
});
