// Push-Benachrichtigungen für B&E Boxenstop PWA
// Unterstützt alle Geräte: Mobile, Tablet, Desktop

let pushSubscription = null;
let isSubscribed = false;

// Prüfe ob Push-Benachrichtigungen unterstützt werden
function isPushNotificationSupported() {
  return 'serviceWorker' in navigator && 'PushManager' in window && 'Notification' in window;
}

// Prüfe ob Benachrichtigungen erlaubt sind
async function checkNotificationPermission() {
  if (!('Notification' in window)) {
    return 'not-supported';
  }
  
  let permission = Notification.permission;
  
  // Wenn noch nicht angefragt, fragen wir nicht automatisch
  if (permission === 'default') {
    return 'default';
  }
  
  return permission;
}

// Frage Benachrichtigungsberechtigung an
async function requestNotificationPermission() {
  if (!('Notification' in window)) {
    throw new Error('Benachrichtigungen werden von diesem Browser nicht unterstützt');
  }
  
  const permission = await Notification.requestPermission();
  
  if (permission === 'granted') {
    return true;
  } else if (permission === 'denied') {
    throw new Error('Benachrichtigungen wurden abgelehnt. Bitte in den Browser-Einstellungen aktivieren.');
  } else {
    return false;
  }
}

// VAPID Public Key (muss vom Server bereitgestellt werden)
// Dieser Key wird für die Verschlüsselung der Push-Benachrichtigungen verwendet
async function getVAPIDPublicKey() {
  try {
    const response = await fetch('/api/get_vapid_key.php');
    const data = await response.json();
    
    if (data.success && data.publicKey) {
      return data.publicKey;
    } else {
      throw new Error('VAPID Key konnte nicht geladen werden');
    }
  } catch (error) {
    console.error('[Push] Fehler beim Laden des VAPID Keys:', error);
    throw error;
  }
}

// Konvertiere Base64 URL zu Uint8Array
function urlBase64ToUint8Array(base64String) {
  const padding = '='.repeat((4 - base64String.length % 4) % 4);
  const base64 = (base64String + padding)
    .replace(/\-/g, '+')
    .replace(/_/g, '/');
  
  const rawData = window.atob(base64);
  const outputArray = new Uint8Array(rawData.length);
  
  for (let i = 0; i < rawData.length; ++i) {
    outputArray[i] = rawData.charCodeAt(i);
  }
  
  return outputArray;
}

// Abonniere Push-Benachrichtigungen
async function subscribeToPushNotifications() {
  if (!isPushNotificationSupported()) {
    throw new Error('Push-Benachrichtigungen werden von diesem Browser nicht unterstützt');
  }
  
  try {
    // Prüfe Berechtigung
    const permission = await checkNotificationPermission();
    if (permission !== 'granted') {
      const granted = await requestNotificationPermission();
      if (!granted) {
        throw new Error('Benachrichtigungen wurden nicht erlaubt');
      }
    }
    
    // Registriere Service Worker
    const registration = await navigator.serviceWorker.ready;
    
    // Hole VAPID Public Key
    const publicKey = await getVAPIDPublicKey();
    const applicationServerKey = urlBase64ToUint8Array(publicKey);
    
    // Erstelle Push-Subscription
    pushSubscription = await registration.pushManager.subscribe({
      userVisibleOnly: true,
      applicationServerKey: applicationServerKey
    });
    
    // Sende Subscription an Server
    await sendSubscriptionToServer(pushSubscription);
    
    isSubscribed = true;
    console.log('[Push] Erfolgreich abonniert');
    
    return pushSubscription;
  } catch (error) {
    console.error('[Push] Fehler beim Abonnieren:', error);
    throw error;
  }
}

// Kündige Push-Benachrichtigungen
async function unsubscribeFromPushNotifications() {
  if (!pushSubscription) {
    return;
  }
  
  try {
    const registration = await navigator.serviceWorker.ready;
    const subscription = await registration.pushManager.getSubscription();
    
    if (subscription) {
      await subscription.unsubscribe();
      
      // Entferne Subscription vom Server
      await removeSubscriptionFromServer(subscription);
      
      pushSubscription = null;
      isSubscribed = false;
      console.log('[Push] Abonnement gekündigt');
    }
  } catch (error) {
    console.error('[Push] Fehler beim Kündigen:', error);
    throw error;
  }
}

// Sende Subscription an Server
async function sendSubscriptionToServer(subscription) {
  try {
    const response = await fetch('/api/save_push_subscription.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        subscription: subscription
      })
    });
    
    const data = await response.json();
    
    if (!data.success) {
      throw new Error(data.message || 'Fehler beim Speichern der Subscription');
    }
    
    console.log('[Push] Subscription erfolgreich gespeichert');
    return data;
  } catch (error) {
    console.error('[Push] Fehler beim Senden der Subscription:', error);
    throw error;
  }
}

// Entferne Subscription vom Server
async function removeSubscriptionFromServer(subscription) {
  try {
    const response = await fetch('/api/remove_push_subscription.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        endpoint: subscription.endpoint
      })
    });
    
    const data = await response.json();
    
    if (!data.success) {
      console.warn('[Push] Fehler beim Entfernen der Subscription:', data.message);
    }
    
    return data;
  } catch (error) {
    console.error('[Push] Fehler beim Entfernen der Subscription:', error);
    throw error;
  }
}

// Prüfe ob bereits abonniert
async function checkSubscriptionStatus() {
  if (!isPushNotificationSupported()) {
    return false;
  }
  
  try {
    const registration = await navigator.serviceWorker.ready;
    const subscription = await registration.pushManager.getSubscription();
    
    if (subscription) {
      pushSubscription = subscription;
      isSubscribed = true;
      
      // Prüfe ob Subscription noch auf dem Server ist
      try {
        const response = await fetch('/api/check_push_subscription.php', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            endpoint: subscription.endpoint
          })
        });
        
        const data = await response.json();
        
        if (!data.exists) {
          // Subscription existiert nicht auf Server, neu speichern
          await sendSubscriptionToServer(subscription);
        }
      } catch (error) {
        console.warn('[Push] Fehler beim Prüfen der Subscription:', error);
      }
      
      return true;
    }
    
    return false;
  } catch (error) {
    console.error('[Push] Fehler beim Prüfen des Status:', error);
    return false;
  }
}

// Initialisiere Push-Benachrichtigungen
async function initializePushNotifications() {
  if (!isPushNotificationSupported()) {
    console.log('[Push] Push-Benachrichtigungen werden nicht unterstützt');
    return false;
  }
  
  try {
    // Prüfe ob bereits abonniert
    const subscribed = await checkSubscriptionStatus();
    
    if (subscribed) {
      console.log('[Push] Bereits abonniert');
      return true;
    }
    
    // Automatisches Abonnieren nur wenn Benutzer eingeloggt ist
    // (kann später durch User-Aktion ausgelöst werden)
    return false;
  } catch (error) {
    console.error('[Push] Fehler bei der Initialisierung:', error);
    return false;
  }
}

// Zeige Benachrichtigungs-Einstellungen UI
function showNotificationSettings() {
  // Diese Funktion kann in profile.php verwendet werden
  // um ein UI für Push-Benachrichtigungen anzuzeigen
  return {
    isSupported: isPushNotificationSupported(),
    isSubscribed: isSubscribed,
    permission: Notification.permission
  };
}

// Exportiere Funktionen für globale Verwendung
window.PushNotifications = {
  subscribe: subscribeToPushNotifications,
  unsubscribe: unsubscribeFromPushNotifications,
  checkStatus: checkSubscriptionStatus,
  initialize: initializePushNotifications,
  isSupported: isPushNotificationSupported,
  checkPermission: checkNotificationPermission,
  requestPermission: requestNotificationPermission,
  getSettings: showNotificationSettings
};

// Auto-Initialisierung wenn Seite geladen ist
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initializePushNotifications);
} else {
  initializePushNotifications();
}

