document.addEventListener('DOMContentLoaded', function() {
    // Entferne alle vorhandenen Burger-Menü-Buttons
    const existingMobileBtns = document.querySelectorAll('.mobile-menu-btn');
    existingMobileBtns.forEach(btn => btn.remove());
    
    // Stelle sicher, dass nav-menu immer sichtbar ist
    const navMenu = document.querySelector('.nav-menu');
    if (navMenu) {
        navMenu.classList.remove('mobile-open');
        navMenu.style.display = '';
        navMenu.style.visibility = '';
        navMenu.style.opacity = '';
        navMenu.style.transform = '';
    }
    
    // Initialize all components
    initAnimations();
    initFormValidation();
    initSmoothScrolling();
    // initMobileMenu(); // DEAKTIVIERT - Burger-Menü wird nicht mehr verwendet
    initProfessionalEffects();
    initScrollEffects();
});

/**
 * Initialize smooth animations
 */
function initAnimations() {
    // Add animation classes to elements when they come into view
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
            }
        });
    }, observerOptions);

    // Observe elements for animation
    document.querySelectorAll('.event-card, .feature-card, .stat-card, .hero-visual').forEach(el => {
        observer.observe(el);
    });
}

/**
 * Initialize form validation
 */
function initFormValidation() {
    const forms = document.querySelectorAll('form');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
            }
        });
        
        // Real-time validation
        const inputs = form.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            input.addEventListener('blur', function() {
                validateField(this);
            });
        });
    });
}

/**
 * Validate form
 */
function validateForm(form) {
    let isValid = true;
    const requiredFields = form.querySelectorAll('[required]');
    
    requiredFields.forEach(field => {
        if (!validateField(field)) {
            isValid = false;
        }
    });
    
    return isValid;
}

/**
 * Validate individual field
 */
function validateField(field) {
    const value = field.value.trim();
    const fieldName = field.name || field.id;
    let isValid = true;
    let errorMessage = '';
    
    // Remove existing error
    removeFieldError(field);
    
    // Required field validation
    if (field.hasAttribute('required') && !value) {
        errorMessage = `${getFieldLabel(field)} ist erforderlich`;
        isValid = false;
    }
    
    // Email validation
    if (field.type === 'email' && value) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(value)) {
            errorMessage = 'Bitte geben Sie eine gültige E-Mail-Adresse ein';
            isValid = false;
        }
    }
    
    // Phone validation
    if (field.type === 'tel' && value) {
        const phoneRegex = /^[\+]?[0-9\s\-\(\)]{10,}$/;
        if (!phoneRegex.test(value)) {
            errorMessage = 'Bitte geben Sie eine gültige Telefonnummer ein';
            isValid = false;
        }
    }
    
    // License plate validation
    if (fieldName === 'license_plate' && value) {
        const licenseRegex = /^[A-ZÄÖÜ]{1,3}[\s\-]?[A-Z]{1,2}[\s\-]?[0-9]{1,4}[H]?$/i;
        if (!licenseRegex.test(value)) {
            errorMessage = 'Bitte geben Sie ein gültiges Kennzeichen ein (z.B. M-AB 123)';
            isValid = false;
        }
    }
    
    if (!isValid) {
        showFieldError(field, errorMessage);
    }
    
    return isValid;
}

/**
 * Show field error
 */
function showFieldError(field, message) {
    field.classList.add('error');
    
    const errorDiv = document.createElement('div');
    errorDiv.className = 'field-error';
    errorDiv.textContent = message;
    
    field.parentNode.insertBefore(errorDiv, field.nextSibling);
}

/**
 * Remove field error
 */
function removeFieldError(field) {
    field.classList.remove('error');
    
    const existingError = field.parentNode.querySelector('.field-error');
    if (existingError) {
        existingError.remove();
    }
}

/**
 * Get field label
 */
function getFieldLabel(field) {
    const label = field.parentNode.querySelector('label');
    return label ? label.textContent.replace('*', '').trim() : fieldName;
}

/**
 * Initialize smooth scrolling
 */
function initSmoothScrolling() {
    const links = document.querySelectorAll('a[href^="#"]');
    
    links.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            const targetId = this.getAttribute('href').substring(1);
            const targetElement = document.getElementById(targetId);
            
            if (targetElement) {
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
}

/**
 * Initialize mobile menu
 * DEAKTIVIERT - Burger-Menü wird nicht mehr verwendet, User-Profile-Dropdown ist direkt sichtbar
 */
function initMobileMenu() {
    // Funktion deaktiviert - Burger-Menü wird nicht mehr erstellt
    // Das User-Profile-Dropdown ist jetzt direkt in der Navbar sichtbar
    return;
}

/**
 * Initialize professional effects
 */
function initProfessionalEffects() {
    // Add professional glow to hero section
    const hero = document.querySelector('.hero');
    if (hero) {
        hero.addEventListener('mouseenter', function() {
            this.classList.add('professional-glow');
        });
        
        hero.addEventListener('mouseleave', function() {
            this.classList.remove('professional-glow');
        });
    }
    
    // Add trust pulse effect to stat cards
    const statCards = document.querySelectorAll('.stat-card');
    statCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.classList.add('trust-pulse');
        });
        
        card.addEventListener('mouseleave', function() {
            this.classList.remove('trust-pulse');
        });
    });
    
    // Add professional glow to event cards
    const eventCards = document.querySelectorAll('.event-card');
    eventCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.classList.add('professional-glow');
        });
        
        card.addEventListener('mouseleave', function() {
            this.classList.remove('professional-glow');
        });
    });
}

/**
 * Initialize scroll effects
 */
function initScrollEffects() {
    let lastScrollTop = 0;
    const navbar = document.querySelector('.navbar');
    
    // Nur ausführen wenn navbar existiert
    if (!navbar) {
        return;
    }
    
    window.addEventListener('scroll', throttle(function() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        // Add/remove scrolled class to navbar
        if (scrollTop > 100) {
            navbar.classList.add('scrolled');
        } else {
            navbar.classList.remove('scrolled');
        }
        
        // Hide/show navbar on scroll
        if (scrollTop > lastScrollTop && scrollTop > 200) {
            navbar.style.transform = 'translateY(-100%)';
        } else {
            navbar.style.transform = 'translateY(0)';
        }
        
        lastScrollTop = scrollTop;
    }, 100));
}

/**
 * Show notification
 */
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            <span>${message}</span>
            <button class="notification-close" onclick="this.parentElement.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

/**
 * Format time for display
 */
function formatTime(timeString) {
    const time = new Date(`2000-01-01T${timeString}`);
    return time.toLocaleTimeString('de-DE', {
        hour: '2-digit',
        minute: '2-digit'
    });
}

/**
 * Format date for display
 */
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('de-DE', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

/**
 * Debounce function
 */
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

/**
 * Throttle function
 */
function throttle(func, limit) {
    let inThrottle;
    return function() {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// Add CSS for animations and mobile menu
// Prüfe, ob mainStyle bereits existiert (verhindert doppelte Deklaration)
if (typeof mainStyle === 'undefined') {
    var mainStyle = document.createElement('style');
}
mainStyle.textContent = `
    .animate-in {
        animation: slideInUp 0.6s ease-out forwards;
    }
    
    @keyframes slideInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    .field-error {
        color: #e10600;
        font-size: 0.875rem;
        margin-top: 0.25rem;
        display: flex;
        align-items: center;
        gap: 0.25rem;
    }
    
    .field-error::before {
        content: '⚠';
        font-size: 0.75rem;
    }
    
    input.error,
    select.error,
    textarea.error {
        border-color: #e10600;
        box-shadow: 0 0 0 3px rgba(225, 6, 0, 0.1);
    }
    
    .notification {
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 1000;
        max-width: 400px;
        background: white;
        border-radius: 8px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
        border-left: 4px solid #1e3c72;
        animation: slideInRight 0.3s ease-out;
    }
    
    .notification-success {
        border-left-color: #28a745;
    }
    
    .notification-error {
        border-left-color: #e10600;
    }
    
    .notification-content {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        padding: 1rem;
    }
    
    .notification-close {
        background: none;
        border: none;
        cursor: pointer;
        color: #666;
        margin-left: auto;
    }
    
    @keyframes slideInRight {
        from {
            opacity: 0;
            transform: translateX(100%);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }
    
    /* Mobile Menu Button - ENTFERNT, nicht mehr verwendet */
    .mobile-menu-btn {
        display: none !important;
    }
    
    .navbar.scrolled {
        background: rgba(0, 48, 87, 0.95);
        backdrop-filter: blur(20px);
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
    }
    
    @media (max-width: 768px) {
        .nav-container {
            position: relative;
        }
        
        /* Nav-Menu ist immer sichtbar auf mobilen Geräten */
        .nav-menu {
            position: static;
            flex-direction: row;
            padding: 0;
            gap: 0.5rem;
            box-shadow: none;
            border: none;
            transform: none;
            opacity: 1;
            visibility: visible;
            order: 2;
        }
        
        /* User-Profile-Dropdown ist immer sichtbar */
        .user-profile-dropdown {
            display: flex;
        }
        
        /* User-Name kann auf sehr kleinen Bildschirmen ausgeblendet werden */
        .user-name-nav {
            display: none;
        }
        
        /* Nav-Links bleiben kompakt */
        .nav-link {
            padding: 0.5rem 0.75rem;
            font-size: 0.85rem;
            margin-bottom: 0;
        }
        
        .nav-link span {
            display: none;
        }
        
        /* User Avatar kleiner auf mobilen Geräten */
        .user-avatar-nav {
            width: 32px;
            height: 32px;
        }
        
        .user-profile-trigger {
            padding: 0.375rem 0.5rem;
            min-width: auto;
        }
    }
`;
// Prüfe, ob mainStyle bereits im DOM ist
if (!document.getElementById('main-style-injected')) {
    mainStyle.id = 'main-style-injected';
    document.head.appendChild(mainStyle);
}
