/**
 * Three.js 3D-Viewer für Showroom mit Auto
 * Wird von booking.php verwendet
 */

(function () {
  // Namespace
  window.BE3D = window.BE3D || {};
  const overlay = document.getElementById('be3d-overlay');
  const canvas  = document.getElementById('be3d-canvas');
  const closeBtn= document.getElementById('be3d-close');
  const rotateHint = document.getElementById('be3d-rotate-hint');
  const debugPanel = document.getElementById('be3d-debug-panel');
  const debugContent = document.getElementById('be3d-debug-content');
  const debugToggle = document.getElementById('be3d-debug-toggle');

  // Prüfe ob benötigte Elemente vorhanden sind
  if(!overlay || !canvas || !closeBtn) {
    console.warn('3D-Viewer: Benötigte DOM-Elemente nicht gefunden');
    return;
  }

  let renderer=null, scene=null, camera=null, controls=null;
  let carRoot=null, showroomRoot=null, animationId=null;
  let cubeCamera=null, envMap=null, dirLight=null;
  
  // Projektpfade
  const BASE_PATH = 'assets/3dmodel/';
  
  // Verfügbare Modelle aus dem Projekt
  const AVAILABLE_MODELS = [
    'universal_car',
    'volvo_xc60',
    'volvo_s90_recharge',
    'universal_car1'
  ];

  // -------- Helpers --------
  const modelToBasename = s => (s||'').toLowerCase().replace(/\s+/g,'').replace(/-/g,'');
  
  function getSelectedVehicleModel(){
    const sel = document.getElementById('selected_vehicle');
    if(!sel) return '';
    const opt = sel.options[sel.selectedIndex];
    return opt ? (opt.getAttribute('data-vehicle-model')||opt.text||'') : '';
  }

  async function importGlb(url){
    return new Promise((resolve, reject) => {
      const loader = new THREE.GLTFLoader();
      
      // DRACO-Loader konfigurieren (falls verfügbar)
      if (typeof THREE.DRACOLoader !== 'undefined') {
        const dracoLoader = new THREE.DRACOLoader();
        // DRACO-Decoder von Google CDN (zuverlässiger)
        dracoLoader.setDecoderPath('https://www.gstatic.com/draco/v1/decoders/');
        // Alternative: jsDelivr (falls Google CDN nicht verfügbar)
        // dracoLoader.setDecoderPath('https://cdn.jsdelivr.net/npm/three@0.128.0/examples/js/libs/draco/gltf/');
        loader.setDRACOLoader(dracoLoader);
        console.log('DRACOLoader konfiguriert');
      } else {
        console.warn('DRACOLoader nicht verfügbar - Modelle mit DRACO-Kompression können nicht geladen werden');
      }
      
      console.log('Lade Modell:', url);
      loader.load(
        url,
        (gltf) => {
          // Die gesamte Scene als Root verwenden
          const root = gltf.scene;
          root.name = url + '_root';
          root.userData = { src: url };
          
          resolve(root);
        },
        (progress) => {
          // Progress-Callback (optional)
          if (progress.lengthComputable) {
            const percentComplete = (progress.loaded / progress.total) * 100;
            console.log('Lade Fortschritt:', Math.round(percentComplete) + '%');
          }
        },
        (error) => {
          console.error('Fehler beim Laden von', url, ':', error);
          // Prüfe ob es ein DRACO-Fehler ist
          if (error.message && error.message.includes('DRACO')) {
            reject(new Error('Das Modell verwendet DRACO-Kompression. DRACOLoader wird konfiguriert...'));
          } else {
            reject(error);
          }
        }
      );
    });
  }

  // Laden des Showrooms (optimized → normal)
  async function loadShowroom(){
    const cand = [
      `${BASE_PATH}car-showroom_2_optimized.glb`, 
      `${BASE_PATH}car-showroom_2.glb`
    ];
    let last=null;
    for(const url of cand){
      try{
        // Prüfe ob Datei existiert
        const response = await fetch(url, { method: 'HEAD' });
        if (!response.ok) {
          console.warn('Showroom-Datei nicht gefunden:', url);
          continue;
        }
        
        if(showroomRoot){ 
          scene.remove(showroomRoot);
          showroomRoot.traverse((child) => {
            if(child.geometry) child.geometry.dispose();
            if(child.material) {
              if(Array.isArray(child.material)) {
                child.material.forEach(mat => mat.dispose());
              } else {
                child.material.dispose();
              }
            }
          });
          showroomRoot=null; 
        }
        showroomRoot = await importGlb(url);
        scene.add(showroomRoot);
        console.log('Showroom geladen:', url);
        return url;
      }catch(e){ 
        console.warn('Showroom-Ladeversuch fehlgeschlagen:', url, e);
        // Wenn DRACO-Fehler, versuche es erneut mit DRACO-Setup
        if(e.message && e.message.includes('DRACO')){
          try{
            // Warte kurz und versuche erneut (DRACO könnte noch laden)
            await new Promise(resolve => setTimeout(resolve, 500));
            if(showroomRoot){ 
              scene.remove(showroomRoot);
              showroomRoot.traverse((child) => {
                if(child.geometry) child.geometry.dispose();
                if(child.material) {
                  if(Array.isArray(child.material)) {
                    child.material.forEach(mat => mat.dispose());
                  } else {
                    child.material.dispose();
                  }
                }
              });
              showroomRoot=null; 
            }
            showroomRoot = await importGlb(url);
            scene.add(showroomRoot);
            console.log('Showroom geladen (nach DRACO-Retry):', url);
            return url;
          }catch(retryError){
            console.warn('Showroom-Ladeversuch nach DRACO-Retry fehlgeschlagen:', url, retryError);
            last=retryError;
          }
        }else{
          last=e;
        }
      }
    }
    throw last||new Error('Showroom konnte nicht geladen werden.');
  }

  // Laden des Autos (optimized → normal → universal → volvo_xc60)
  async function loadCar(base){
    const cand=[];
    
    // Wenn ein Modellname gefunden wurde, versuche zuerst dieses
    if(base){
      const normalizedBase = modelToBasename(base);
      // Prüfe ob es zu einem verfügbaren Modell passt
      const matchingModel = AVAILABLE_MODELS.find(m => 
        normalizedBase.includes(modelToBasename(m)) || 
        modelToBasename(m).includes(normalizedBase)
      );
      
      if(matchingModel){
        cand.push(
          `${BASE_PATH}${matchingModel}_optimized.glb`, 
          `${BASE_PATH}${matchingModel}.glb`
        );
      } else {
        // Versuche direkt mit dem übergebenen Namen
        cand.push(
          `${BASE_PATH}${base}_optimized.glb`, 
          `${BASE_PATH}${base}.glb`
        );
      }
    }
    
    // Fallback-Modelle in Reihenfolge
    cand.push(
      `${BASE_PATH}universal_car.glb`,
      `${BASE_PATH}volvo_xc60.glb`
    );
    
    let last=null;
    for(const url of cand){
      try{
        // Prüfe ob Datei existiert
        const response = await fetch(url, { method: 'HEAD' });
        if (!response.ok) {
          console.warn('Fahrzeug-Datei nicht gefunden:', url);
          continue;
        }
        
        if(carRoot){ 
          scene.remove(carRoot);
          carRoot.traverse((child) => {
            if(child.geometry) child.geometry.dispose();
            if(child.material) {
              if(Array.isArray(child.material)) {
                child.material.forEach(mat => mat.dispose());
              } else {
                child.material.dispose();
              }
            }
          });
          carRoot=null; 
        }
        carRoot = await importGlb(url);
        scene.add(carRoot);
        console.log('Fahrzeug geladen:', url);
        return url;
      }catch(e){ 
        console.warn('Fahrzeug-Ladeversuch fehlgeschlagen:', url, e);
        // Wenn DRACO-Fehler, versuche es erneut mit DRACO-Setup
        if(e.message && e.message.includes('DRACO')){
          try{
            // Warte kurz und versuche erneut (DRACO könnte noch laden)
            await new Promise(resolve => setTimeout(resolve, 500));
            if(carRoot){ 
              scene.remove(carRoot);
              carRoot.traverse((child) => {
                if(child.geometry) child.geometry.dispose();
                if(child.material) {
                  if(Array.isArray(child.material)) {
                    child.material.forEach(mat => mat.dispose());
                  } else {
                    child.material.dispose();
                  }
                }
              });
              carRoot=null; 
            }
            carRoot = await importGlb(url);
            scene.add(carRoot);
            console.log('Fahrzeug geladen (nach DRACO-Retry):', url);
            return url;
          }catch(retryError){
            console.warn('Fahrzeug-Ladeversuch nach DRACO-Retry fehlgeschlagen:', url, retryError);
            last=retryError;
          }
        }else{
          last=e;
        }
      }
    }
    throw last||new Error('Kein Fahrzeugmodell geladen.');
  }

  // CubeCamera so konfigurieren, dass sie NUR den Showroom rendert
  function setupCubeCamera(){
    if(cubeCamera){ 
      scene.remove(cubeCamera);
      if(cubeCamera.renderTarget) cubeCamera.renderTarget.dispose();
      cubeCamera=null; 
    }
    
    // CubeCamera für Environment Map (nur Showroom)
    const cubeRenderTarget = new THREE.WebGLCubeRenderTarget(512, {
      format: THREE.RGBAFormat,
      generateMipmaps: true,
      minFilter: THREE.LinearMipmapLinearFilter
    });
    
    cubeCamera = new THREE.CubeCamera(0.1, 1000, cubeRenderTarget);
    
    // Position am Auto (für korrekte Perspektive)
    if(carRoot) {
      const box = new THREE.Box3().setFromObject(carRoot);
      const center = box.getCenter(new THREE.Vector3());
      cubeCamera.position.copy(center);
    } else {
      cubeCamera.position.set(0, 1.2, 0);
    }
    
    scene.add(cubeCamera);
    envMap = cubeRenderTarget.texture;
    
    console.log('CubeCamera konfiguriert');
  }

  // Environment Map NUR auf Car-Materialien anwenden (OHNE Reifen)
  function applyEnvMapOnlyToCar(intensity=0.5){
    if(!carRoot || !envMap) return;
    let appliedCount = 0;
    let tireCount = 0;
    
    carRoot.traverse((child) => {
      if(child.isMesh && child.material){
        // Handle Array of materials
        const materials = Array.isArray(child.material) ? child.material : [child.material];
        
        materials.forEach((mat) => {
          // Nur PBR-Materialien (MeshStandardMaterial)
          if(mat.isMeshStandardMaterial){
            // Prüfe ob es ein Reifen-Material ist
            const matName = (mat.name || '').toLowerCase();
            const meshName = (child.name || '').toLowerCase();
            const isTire = matName.includes('tire') || matName.includes('wheel') || 
                          matName.includes('reifen') || matName.includes('rad') ||
                          matName.includes('tyre') || meshName.includes('tire') ||
                          meshName.includes('wheel') || meshName.includes('reifen') ||
                          meshName.includes('rad') || meshName.includes('tyre') ||
                          matName.includes('rim') || meshName.includes('rim');
            
            if(isTire){
              // Reifen: KEINE Reflexionen, komplett matt
              mat.envMap = null;
              mat.envMapIntensity = 0;
              mat.roughness = 1.0;  // Maximale Rauheit (matt)
              mat.metalness = 0.0;  // Kein Metall
              mat.needsUpdate = true;
              tireCount++;
            } else {
              // Andere Teile: Environment Map anwenden (reduzierte Intensität)
              mat.envMap = envMap;
              mat.envMapIntensity = intensity;
              mat.needsUpdate = true;
              appliedCount++;
            }
          }
        });
      }
    });
    
    console.log('Environment Map auf', appliedCount, 'Car-Materialien angewendet (Intensität:', intensity, ')');
    console.log(tireCount, 'Reifen-Materialien wurden matt gemacht (keine Reflexionen)');
  }
  
  // CubeCamera rendern (nur Showroom, ohne Auto)
  function updateCubeCamera(){
    if(!cubeCamera || !carRoot || !showroomRoot) return;
    
    // Auto temporär ausblenden
    const carVisible = carRoot.visible;
    carRoot.visible = false;
    
    // CubeCamera rendern (nur Showroom sichtbar)
    cubeCamera.update(renderer, scene);
    
    // Auto wieder einblenden
    carRoot.visible = carVisible;
  }

  // Car-Recolor (alle Karosserie-Teile, außer Reifen)
  function recolorCarPaint(hex='#F0F0F0'){
    if(!carRoot) {
      console.error('recolorCarPaint: carRoot ist null!');
      return;
    }
    const col = new THREE.Color(hex);
    let paintCount = 0;
    let skippedCount = 0;
    let nonStandardCount = 0;
    let allMaterials = [];
    
    carRoot.traverse((child) => {
      if(child.isMesh && child.material){
        // Handle Array of materials
        const materials = Array.isArray(child.material) ? child.material : [child.material];
        
        materials.forEach((mat) => {
          const matName = (mat.name || '').toLowerCase();
          const meshName = (child.name || '').toLowerCase();
          allMaterials.push({matName, meshName, type: mat.type});
          
          // Prüfe Material-Typ
          if (!mat.isMeshStandardMaterial && !mat.isMeshPhysicalMaterial && !mat.isMeshPhongMaterial) {
            nonStandardCount++;
            return;
          }
          
          // Prüfe ob es ein Reifen-Material ist
          const isTire = matName.includes('tire') || matName.includes('wheel') || 
                        matName.includes('reifen') || matName.includes('rad') ||
                        matName.includes('tyre') || meshName.includes('tire') ||
                        meshName.includes('wheel') || meshName.includes('reifen') ||
                        meshName.includes('rad') || meshName.includes('tyre') ||
                        matName.includes('rim') || meshName.includes('rim');
          
          // Reifen überspringen
          if(isTire){
            skippedCount++;
            return;
          }
          
          // Alle anderen Materialien als Karosserie behandeln
          mat.color.copy(col);
          if(mat.isMeshStandardMaterial || mat.isMeshPhysicalMaterial){
            mat.metalness = 0.85;
            mat.roughness = 0.25;
          }
          mat.needsUpdate = true;
          paintCount++;
        });
      }
    });
    
    console.log('=== RECOLOR DEBUG ===');
    console.log('Lack auf', paintCount, 'Materialien angepasst (Farbe:', hex, ')');
    if(skippedCount > 0) console.log(skippedCount, 'Reifen-Materialien übersprungen');
    if(nonStandardCount > 0) console.log(nonStandardCount, 'Nicht-Standard-Materialien gefunden');
    console.log('Alle Materialien:', allMaterials.slice(0, 10)); // Erste 10 für Debug
  }

  // Fallback-Lichter, falls Showroom keine Lichter enthält
  function ensureLights(){
    // Prüfe ob Showroom Lichter enthält
    const lights = [];
    scene.traverse((child) => {
      if(child.isLight) lights.push(child);
    });
    
    if(lights.length === 0){
      console.log('Keine Lichter im Showroom gefunden - füge Fallback-Lichter hinzu');
      
      // Hemisphere Light für allgemeine Beleuchtung
      const hemi = new THREE.HemisphereLight(0xffffff, 0x444444, 0.35);
      hemi.position.set(0, 1, 0);
      scene.add(hemi);
      
      // Directional Light für Schatten
      dirLight = new THREE.DirectionalLight(0xffffff, 0.9);
      dirLight.position.set(-4, 10, -2.5);
      dirLight.castShadow = true;
      dirLight.shadow.mapSize.width = 4096;
      dirLight.shadow.mapSize.height = 4096;
      dirLight.shadow.camera.near = 0.5;
      dirLight.shadow.camera.far = 50;
      dirLight.shadow.camera.left = -10;
      dirLight.shadow.camera.right = 10;
      dirLight.shadow.camera.top = 10;
      dirLight.shadow.camera.bottom = -10;
      scene.add(dirLight);
      
      // Showroom-Boden empfängt Schatten
      scene.traverse((child) => {
        if(child.isMesh){
          const n = (child.name||'').toLowerCase();
          if(n.includes('floor') || n.includes('ground') || n.includes('boden')){
            child.receiveShadow = true;
          }
        }
      });
      
      // Car wirft Schatten
      if(carRoot){
        carRoot.traverse((child) => {
          if(child.isMesh){
            child.castShadow = true;
          }
        });
      }
      
      // Renderer Shadow-Map aktivieren
      renderer.shadowMap.enabled = true;
      renderer.shadowMap.type = THREE.PCFSoftShadowMap;
    } else {
      console.log('Showroom enthält bereits', lights.length, 'Lichter');
    }
  }

  // Auto im Showroom positionieren: optional Locator "CarAnchor"/"car_anchor"
  function placeCarInShowroom(){
    if(!carRoot || !showroomRoot) return;
    
    // Suche Anchor
    let anchor = null;
    showroomRoot.traverse((child) => {
      if(child.isMesh || child.isObject3D){
        const n = (child.name||'').toLowerCase();
        if(n==='caranchor' || n==='car_anchor' || n.includes('caranchor')){
          anchor = child;
        }
      }
    });
    
    if(anchor){
      // Anchor World-Position holen
      const worldPos = new THREE.Vector3();
      anchor.getWorldPosition(worldPos);
      carRoot.position.copy(worldPos);
      
      // Rotation vom Anchor übernehmen
      const worldQuat = new THREE.Quaternion();
      anchor.getWorldQuaternion(worldQuat);
      carRoot.quaternion.copy(worldQuat);
      
      // Scaling vom Anchor übernehmen
      const worldScale = new THREE.Vector3();
      anchor.getWorldScale(worldScale);
      carRoot.scale.copy(worldScale);
      
      console.log('Auto an Anchor positioniert:', anchor.name);
    }else{
      // Fallback: Mittelpunkt der Szene knapp über dem Boden
      carRoot.position.set(0, 0.01, 0);
      console.log('Auto an Standard-Position platziert (kein Anchor gefunden)');
    }
  }

  // Gespeicherte Kamera-Position laden
  function loadSavedCameraPosition(){
    try {
      const saved = localStorage.getItem('be3d_camera_position');
      if(saved){
        const data = JSON.parse(saved);
        console.log('Gespeicherte Kamera-Position geladen:', data);
        return data;
      }
    } catch(e) {
      console.warn('Fehler beim Laden der gespeicherten Kamera-Position:', e);
    }
    return null;
  }

  // Kamera auf 3/4-Front - nur horizontale Rotation
  function setupCamera(){
    const aspect = canvas.clientWidth / canvas.clientHeight;
    camera = new THREE.PerspectiveCamera(75, aspect, 0.1, 1000);
    
    // Versuche gespeicherte Position zu laden
    const savedPos = loadSavedCameraPosition();
    
    if(savedPos && savedPos.position && savedPos.target){
      // Gespeicherte Position verwenden
      camera.position.set(
        savedPos.position.x,
        savedPos.position.y,
        savedPos.position.z
      );
      camera.lookAt(savedPos.target.x, savedPos.target.y, savedPos.target.z);
      console.log('Kamera mit gespeicherter Position initialisiert');
    } else {
      // Standard 3/4-Front Position (leicht erhöht) - Startposition
      const radius = 12;
      const alpha = -35 * Math.PI / 180; // Horizontal
      const beta = 22 * Math.PI / 180;    // Vertikal
      
      camera.position.set(
        radius * Math.sin(beta) * Math.cos(alpha),
        radius * Math.cos(beta),
        radius * Math.sin(beta) * Math.sin(alpha)
      );
      camera.lookAt(0, 1.2, 0);
    }
    
    // OrbitControls
    controls = new THREE.OrbitControls(camera, canvas);
    
    // Controls konfigurieren
    controls.enableRotate = true;   // Rotation aktivieren (nur horizontal)
    controls.enableZoom = false;    // Zoom deaktiviert (Mausrad)
    controls.enablePan = false;     // Pan deaktiviert (horizontale Verschiebung)
    controls.enableKeys = true;     // Tastatur-Steuerung aktivieren
    
    // Target aus gespeicherten Werten oder Standard
    if(savedPos && savedPos.target){
      controls.target.set(savedPos.target.x, savedPos.target.y, savedPos.target.z);
    } else {
      controls.target.set(0, 1.2, 0);
    }
    
    controls.enableDamping = true;
    controls.dampingFactor = 0.05;
    
    // Keine Distanz-Limits
    controls.minDistance = 0.5;
    controls.maxDistance = 1000;
    
    // Berechne aktuellen Polar-Winkel aus der Kamera-Position
    // Wenn gespeicherte Position vorhanden, berechne den Winkel daraus
    let fixedPolarAngle;
    if(savedPos && savedPos.position){
      // Berechne Polar-Winkel aus gespeicherter Position
      const pos = savedPos.position;
      const target = savedPos.target || {x: 0, y: 1.2, z: 0};
      const direction = new THREE.Vector3(
        pos.x - target.x,
        pos.y - target.y,
        pos.z - target.z
      );
      const distance = direction.length();
      if(distance > 0){
        // Polar-Winkel = Winkel zwischen Y-Achse und Richtungsvektor
        fixedPolarAngle = Math.acos(direction.y / distance);
      } else {
        fixedPolarAngle = 22 * Math.PI / 180; // Fallback
      }
    } else {
      // Berechne Polar-Winkel aus aktueller Kamera-Position
      const direction = new THREE.Vector3().subVectors(camera.position, controls.target);
      const distance = direction.length();
      if(distance > 0){
        fixedPolarAngle = Math.acos(direction.y / distance);
      } else {
        fixedPolarAngle = 22 * Math.PI / 180; // Standard-Winkel
      }
    }
    
    // Vertikale Bewegung deaktiviert (nur horizontale Rotation)
    controls.minPolarAngle = fixedPolarAngle;
    controls.maxPolarAngle = fixedPolarAngle;
    
    // Keine Azimuth-Limits (horizontale Rotation erlaubt)
    controls.minAzimuthAngle = -Infinity;
    controls.maxAzimuthAngle = Infinity;
    
    // Geschwindigkeiten
    controls.zoomSpeed = 1.0;
    controls.rotateSpeed = 1.0;
    controls.panSpeed = 1.0;
    
    // Touch-Support für mobile Geräte
    if (typeof THREE.TOUCH !== 'undefined') {
      controls.touches = {
        ONE: THREE.TOUCH.ROTATE,
        TWO: THREE.TOUCH.DOLLY_PAN
      };
    }
    
    // Renderer Tonemapping
    renderer.toneMapping = THREE.ACESFilmicToneMapping;
    renderer.toneMappingExposure = 1.05;
    
    // Gespeicherte Controls-Einstellungen anwenden (ABER Zoom/Pan bleiben IMMER deaktiviert)
    if(savedPos && savedPos.controls){
      if(savedPos.controls.rotateSpeed !== undefined) controls.rotateSpeed = savedPos.controls.rotateSpeed;
      // Zoom und Pan bleiben IMMER deaktiviert, auch wenn gespeichert
      controls.enableZoom = false;
      controls.enablePan = false;
      // Vertikale Bewegung bleibt immer fixiert (mit berechnetem Winkel)
      controls.minPolarAngle = fixedPolarAngle;
      controls.maxPolarAngle = fixedPolarAngle;
    }
    
    // Debug: Log Controls-Status (NACH allen Einstellungen)
    console.log('OrbitControls konfiguriert:', {
      enableRotate: controls.enableRotate,
      enableZoom: controls.enableZoom,
      enablePan: controls.enablePan,
      minDistance: controls.minDistance,
      maxDistance: controls.maxDistance,
      minPolarAngle: controls.minPolarAngle,
      maxPolarAngle: controls.maxPolarAngle
    });
    
    return camera;
  }

  // Debug-Panel aktualisieren
  function updateDebugPanel(){
    if(!camera || !controls) {
      console.warn('updateDebugPanel: Kamera oder Controls nicht verfügbar');
      return;
    }
    
    try {
      // Position
      const posX = document.getElementById('debug-pos-x');
      const posY = document.getElementById('debug-pos-y');
      const posZ = document.getElementById('debug-pos-z');
      if(posX) posX.textContent = camera.position.x.toFixed(3);
      if(posY) posY.textContent = camera.position.y.toFixed(3);
      if(posZ) posZ.textContent = camera.position.z.toFixed(3);
      
      // Target
      const targetX = document.getElementById('debug-target-x');
      const targetY = document.getElementById('debug-target-y');
      const targetZ = document.getElementById('debug-target-z');
      if(targetX) targetX.textContent = controls.target.x.toFixed(3);
      if(targetY) targetY.textContent = controls.target.y.toFixed(3);
      if(targetZ) targetZ.textContent = controls.target.z.toFixed(3);
      
      // Rotation (Euler)
      const euler = new THREE.Euler().setFromQuaternion(camera.quaternion);
      const rotX = document.getElementById('debug-rot-x');
      const rotY = document.getElementById('debug-rot-y');
      const rotZ = document.getElementById('debug-rot-z');
      if(rotX) rotX.textContent = (euler.x * 180 / Math.PI).toFixed(2) + '°';
      if(rotY) rotY.textContent = (euler.y * 180 / Math.PI).toFixed(2) + '°';
      if(rotZ) rotZ.textContent = (euler.z * 180 / Math.PI).toFixed(2) + '°';
      
      // Kamera-Einstellungen
      const fov = document.getElementById('debug-fov');
      const near = document.getElementById('debug-near');
      const far = document.getElementById('debug-far');
      if(fov) fov.textContent = camera.fov.toFixed(1);
      if(near) near.textContent = camera.near.toFixed(2);
      if(far) far.textContent = camera.far.toFixed(0);
      
      // Distance berechnen
      const distance = camera.position.distanceTo(controls.target);
      const distEl = document.getElementById('debug-distance');
      if(distEl) distEl.textContent = distance.toFixed(3);
      
      // Controls
      const zoomSpeed = document.getElementById('debug-zoom-speed');
      const rotateSpeed = document.getElementById('debug-rotate-speed');
      const panSpeed = document.getElementById('debug-pan-speed');
      if(zoomSpeed) zoomSpeed.textContent = controls.zoomSpeed.toFixed(2);
      if(rotateSpeed) rotateSpeed.textContent = controls.rotateSpeed.toFixed(2);
      if(panSpeed) panSpeed.textContent = controls.panSpeed.toFixed(2);
    } catch(error) {
      console.error('Fehler beim Aktualisieren des Debug-Panels:', error);
    }
  }

  // Kamera-Werte als JSON kopieren
  function copyCameraValues(){
    if(!camera || !controls) return;
    
    const data = {
      position: {
        x: camera.position.x,
        y: camera.position.y,
        z: camera.position.z
      },
      target: {
        x: controls.target.x,
        y: controls.target.y,
        z: controls.target.z
      },
      rotation: {
        x: camera.rotation.x,
        y: camera.rotation.y,
        z: camera.rotation.z
      },
      camera: {
        fov: camera.fov,
        near: camera.near,
        far: camera.far
      },
      controls: {
        zoomSpeed: controls.zoomSpeed,
        rotateSpeed: controls.rotateSpeed,
        panSpeed: controls.panSpeed
      },
      distance: camera.position.distanceTo(controls.target)
    };
    
    const jsonString = JSON.stringify(data, null, 2);
    
    // In Zwischenablage kopieren
    if(navigator.clipboard && navigator.clipboard.writeText){
      navigator.clipboard.writeText(jsonString).then(() => {
        alert('Kamera-Werte wurden in die Zwischenablage kopiert!');
      }).catch(err => {
        console.error('Fehler beim Kopieren:', err);
        fallbackCopy(jsonString);
      });
    } else {
      fallbackCopy(jsonString);
    }
  }

  // Fallback für ältere Browser
  function fallbackCopy(text){
    const textarea = document.createElement('textarea');
    textarea.value = text;
    textarea.style.position = 'fixed';
    textarea.style.opacity = '0';
    document.body.appendChild(textarea);
    textarea.select();
    try {
      document.execCommand('copy');
      alert('Kamera-Werte wurden in die Zwischenablage kopiert!');
    } catch(err) {
      console.error('Fehler beim Kopieren:', err);
      alert('Fehler beim Kopieren. Bitte manuell kopieren:\n\n' + text);
    }
    document.body.removeChild(textarea);
  }

  // Kamera-Position als Standard speichern
  function saveCameraPosition(){
    if(!camera || !controls) return;
    
    const data = {
      position: {
        x: camera.position.x,
        y: camera.position.y,
        z: camera.position.z
      },
      target: {
        x: controls.target.x,
        y: controls.target.y,
        z: controls.target.z
      },
      controls: {
        zoomSpeed: controls.zoomSpeed,
        rotateSpeed: controls.rotateSpeed,
        panSpeed: controls.panSpeed
      },
      timestamp: new Date().toISOString()
    };
    
    try {
      localStorage.setItem('be3d_camera_position', JSON.stringify(data));
      alert('Kamera-Position wurde als Standard gespeichert!');
      console.log('Kamera-Position gespeichert:', data);
    } catch(e) {
      console.error('Fehler beim Speichern:', e);
      alert('Fehler beim Speichern der Kamera-Position.');
    }
  }

  // Kamera-Position zurücksetzen
  function resetCameraPosition(){
    if(!camera || !controls) return;
    
    if(confirm('Möchten Sie die gespeicherte Kamera-Position wirklich löschen?')){
      localStorage.removeItem('be3d_camera_position');
      alert('Gespeicherte Kamera-Position wurde gelöscht. Bitte Seite neu laden.');
      console.log('Kamera-Position zurückgesetzt');
    }
  }

  // Drehen (nur Auto, Showroom bleibt statisch)
  let lastTime = 0;
  let frameCount = 0; // Frame-Zähler für Debug-Panel Updates
  
  function startTurntable(){
    lastTime = performance.now();
    frameCount = 0;
    animate();
  }
  
  function stopTurntable(){ 
    if(animationId){
      cancelAnimationFrame(animationId);
      animationId = null;
    }
  }
  
  function animate(){
    animationId = requestAnimationFrame(animate);
    frameCount++;
    
    const currentTime = performance.now();
    const deltaTime = currentTime - lastTime;
    lastTime = currentTime;
    
    // Auto rotieren (langsamer)
    if(carRoot) {
      carRoot.rotation.y += deltaTime * 0.0002; // Reduziert von 0.00035
    }
    
    // Controls updaten - WICHTIG: Zoom/Pan bleiben deaktiviert, vertikale Bewegung fixiert
    if(controls) {
      // Sicherstellen dass Zoom/Pan deaktiviert bleiben
      controls.enableZoom = false;
      controls.enablePan = false;
      
      // Berechne aktuellen Polar-Winkel aus der Kamera-Position
      const direction = new THREE.Vector3().subVectors(camera.position, controls.target);
      const distance = direction.length();
      let fixedPolarAngle;
      if(distance > 0){
        fixedPolarAngle = Math.acos(direction.y / distance);
      } else {
        fixedPolarAngle = 22 * Math.PI / 180; // Fallback
      }
      
      // Vertikale Bewegung fixieren (nur horizontale Rotation erlaubt)
      controls.minPolarAngle = fixedPolarAngle;
      controls.maxPolarAngle = fixedPolarAngle;
      
      // Horizontale Rotation erlauben
      controls.minAzimuthAngle = -Infinity;
      controls.maxAzimuthAngle = Infinity;
      
      controls.update();
    }
    
    // Debug-Panel aktualisieren (alle 5 Frames für Performance, aber immer beim ersten Frame)
    if(frameCount === 1 || frameCount % 5 === 0){
      updateDebugPanel();
    }
    
    // CubeCamera regelmäßig updaten für dynamische Reflexionen
    if(cubeCamera && carRoot && showroomRoot) {
      // Nur alle paar Frames updaten (Performance)
      if(Math.random() < 0.1) { // 10% Chance pro Frame
        updateCubeCamera();
      }
    }
    
    // Render
    renderer.render(scene, camera);
  }

  // Open / Dispose
  async function openViewer(){
    overlay.style.display='block'; 
    overlay.setAttribute('aria-hidden','false'); 
    if(rotateHint) rotateHint.style.display='block';

    const isTouch = 'ontouchstart' in window || navigator.maxTouchPoints>0;
    if(!isTouch && overlay.requestFullscreen){ 
      try{ 
        await overlay.requestFullscreen(); 
      }catch(_){
        console.warn('Fullscreen nicht verfügbar');
      } 
    }

    // Renderer initialisieren
    renderer = new THREE.WebGLRenderer({ 
      canvas: canvas,
      antialias: true,
      alpha: true,
      preserveDrawingBuffer: true
    });
    renderer.setSize(canvas.clientWidth, canvas.clientHeight);
    renderer.setPixelRatio(window.devicePixelRatio);
    renderer.outputEncoding = THREE.sRGBEncoding;
    renderer.toneMapping = THREE.ACESFilmicToneMapping;
    renderer.toneMappingExposure = 1.05;
    
    // Scene initialisieren
    scene = new THREE.Scene();
    scene.background = new THREE.Color(0x1a1a1a); // Dunkler Hintergrund

    // Canvas-Element prüfen
    console.log('Canvas-Element:', canvas);
    console.log('Canvas clientWidth:', canvas.clientWidth, 'clientHeight:', canvas.clientHeight);
    console.log('Canvas pointer-events:', window.getComputedStyle(canvas).pointerEvents);

    setupCamera();

    try {
      // 1) Showroom laden
      await loadShowroom();

      // 2) Fahrzeugwahl & laden
      const base = modelToBasename(getSelectedVehicleModel());
      await loadCar(base);

      // 3) Auto platzieren
      placeCarInShowroom();

      // 4) CubeCamera konfigurieren (Showroom → aufs Auto)
      setupCubeCamera();
      updateCubeCamera(); // Erste Render
      
      // 5) Lack anpassen (VOR Environment Map, damit Farbe gesetzt ist)
      console.log('=== STARTE FARB-ANPASSUNG ===');
      recolorCarPaint('#F0F0F0');
      
      // 6) Environment Map anwenden (NACH Farbe)
      console.log('=== STARTE ENVIRONMENT MAP ===');
      applyEnvMapOnlyToCar(0.5); // Reduzierte Reflexionen (vorher 1.25)

      // 7) Fallback-Lichter nur wenn nötig
      ensureLights();

      // 8) Controls nochmal explizit konfigurieren (nach dem Laden) - WICHTIG!
      if(controls){
        controls.enableRotate = true;
        controls.enableZoom = false;  // Mausrad deaktiviert
        controls.enablePan = false;   // Horizontale Verschiebung deaktiviert
        controls.enableKeys = true;
        
        // Berechne Polar-Winkel aus aktueller Kamera-Position
        const direction = new THREE.Vector3().subVectors(camera.position, controls.target);
        const distance = direction.length();
        let fixedPolarAngle;
        if(distance > 0){
          fixedPolarAngle = Math.acos(direction.y / distance);
        } else {
          fixedPolarAngle = 22 * Math.PI / 180; // Fallback
        }
        
        // Vertikale Bewegung fixieren (nur horizontale Rotation)
        controls.minPolarAngle = fixedPolarAngle;
        controls.maxPolarAngle = fixedPolarAngle;
        
        // Horizontale Rotation erlauben
        controls.minAzimuthAngle = -Infinity;
        controls.maxAzimuthAngle = Infinity;
        
        // Event-Listener für Mausrad blockieren
        canvas.addEventListener('wheel', (e) => {
          e.preventDefault();
          e.stopPropagation();
          return false;
        }, { passive: false });
        
        // Event-Listener für Pan blockieren (Rechtsklick + Drag)
        canvas.addEventListener('contextmenu', (e) => {
          e.preventDefault();
          return false;
        });
        
        controls.update();
        console.log('Controls nach dem Laden konfiguriert (Zoom/Pan deaktiviert, nur horizontale Rotation)');
        console.log('enableZoom:', controls.enableZoom, 'enablePan:', controls.enablePan);
      }

      // 9) Debug-Panel Buttons initialisieren (nach dem Öffnen)
      initializeDebugButtons();

      // Resize Handler
      window.addEventListener('resize', onWindowResize);
      
      // Animation starten
      startTurntable();
      
      // Debug-Panel sofort aktualisieren
      updateDebugPanel();
      
      console.log('3D-Viewer erfolgreich initialisiert');
    } catch(error) {
      console.error('Fehler beim Laden des 3D-Viewers:', error);
      alert('Fehler beim Laden des 3D-Viewers. Bitte versuchen Sie es erneut.');
      disposeViewer();
    }
  }
  
  function onWindowResize(){
    if(!camera || !renderer) return;
    camera.aspect = canvas.clientWidth / canvas.clientHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(canvas.clientWidth, canvas.clientHeight);
  }

  function disposeViewer(){
    stopTurntable();
    window.removeEventListener('resize', onWindowResize);
    
    // Cleanup
    if(controls){
      controls.dispose();
      controls = null;
    }
    
    if(cubeCamera){
      scene.remove(cubeCamera);
      if(cubeCamera.renderTarget) cubeCamera.renderTarget.dispose();
      cubeCamera = null;
    }
    
    if(carRoot){
      scene.remove(carRoot);
      carRoot.traverse((child) => {
        if(child.geometry) child.geometry.dispose();
        if(child.material) {
          if(Array.isArray(child.material)) {
            child.material.forEach(mat => mat.dispose());
          } else {
            child.material.dispose();
          }
        }
      });
      carRoot = null;
    }
    
    if(showroomRoot){
      scene.remove(showroomRoot);
      showroomRoot.traverse((child) => {
        if(child.geometry) child.geometry.dispose();
        if(child.material) {
          if(Array.isArray(child.material)) {
            child.material.forEach(mat => mat.dispose());
          } else {
            child.material.dispose();
          }
        }
      });
      showroomRoot = null;
    }
    
    if(scene){
      scene.traverse((child) => {
        if(child.geometry) child.geometry.dispose();
        if(child.material) {
          if(Array.isArray(child.material)) {
            child.material.forEach(mat => mat.dispose());
          } else {
            child.material.dispose();
          }
        }
      });
      scene = null;
    }
    
    if(renderer){
      renderer.dispose();
      renderer = null;
    }
    
    camera = null;
    envMap = null;
    dirLight = null;
    
    overlay.style.display='none'; 
    overlay.setAttribute('aria-hidden','true');
    if(rotateHint) rotateHint.style.display='none';
    
    if(document.fullscreenElement && document.exitFullscreen){ 
      document.exitFullscreen().catch(()=>{}); 
    }
  }

  // Debug-Buttons initialisieren
  function initializeDebugButtons(){
    console.log('Initialisiere Debug-Buttons...');
    
    // Debug-Panel Toggle
    const toggle = document.getElementById('be3d-debug-toggle');
    const content = document.getElementById('be3d-debug-content');
    
    if(toggle && content){
      let isCollapsed = false;
      
      // Entferne alte Listener (falls vorhanden)
      const newToggle = toggle.cloneNode(true);
      toggle.parentNode.replaceChild(newToggle, toggle);
      
      newToggle.addEventListener('click', (e) => {
        e.stopPropagation();
        isCollapsed = !isCollapsed;
        if(isCollapsed){
          content.classList.add('collapsed');
          newToggle.textContent = '▶';
        } else {
          content.classList.remove('collapsed');
          newToggle.textContent = '▼';
        }
      });
      
      // Header klickbar machen
      const debugHeader = newToggle.parentElement;
      if(debugHeader){
        debugHeader.addEventListener('click', () => {
          newToggle.click();
        });
      }
      
      console.log('Debug-Toggle initialisiert');
    } else {
      console.warn('Debug-Toggle oder Content nicht gefunden');
    }
    
    // Debug-Buttons
    const debugCopyBtn = document.getElementById('be3d-debug-copy');
    const debugSaveBtn = document.getElementById('be3d-debug-save');
    const debugResetBtn = document.getElementById('be3d-debug-reset');
    
    if(debugCopyBtn){
      // Entferne alte Listener
      const newCopyBtn = debugCopyBtn.cloneNode(true);
      debugCopyBtn.parentNode.replaceChild(newCopyBtn, debugCopyBtn);
      newCopyBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        copyCameraValues();
      });
      console.log('Copy-Button initialisiert');
    } else {
      console.warn('Copy-Button nicht gefunden');
    }
    
    if(debugSaveBtn){
      // Entferne alte Listener
      const newSaveBtn = debugSaveBtn.cloneNode(true);
      debugSaveBtn.parentNode.replaceChild(newSaveBtn, debugSaveBtn);
      newSaveBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        saveCameraPosition();
      });
      console.log('Save-Button initialisiert');
    } else {
      console.warn('Save-Button nicht gefunden');
    }
    
    if(debugResetBtn){
      // Entferne alte Listener
      const newResetBtn = debugResetBtn.cloneNode(true);
      debugResetBtn.parentNode.replaceChild(newResetBtn, debugResetBtn);
      newResetBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        resetCameraPosition();
      });
      console.log('Reset-Button initialisiert');
    } else {
      console.warn('Reset-Button nicht gefunden');
    }
  }

  // Globale Funktionen exportieren
  window.BE3D.open3DViewer = openViewer;
  window.BE3D.dispose3DViewer = disposeViewer;
  
  closeBtn.addEventListener('click', disposeViewer);
  
  // ESC-Taste zum Schließen
  document.addEventListener('keydown', (e) => {
    if(e.key === 'Escape' && overlay.style.display === 'block') {
      disposeViewer();
    }
  });
})();
