<?php
/**
 * Wheelset Angebote API
 * CRUD-Operationen für Radsatz-Angebote
 */

// Unterdrücke Warnings/Notices für saubere JSON-Antworten
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);
ini_set('display_errors', 0);

header('Content-Type: application/json; charset=UTF-8');
session_start();

// Prüfe Authentifizierung nur für bestimmte Aktionen
$action = $_GET['action'] ?? '';
$publicActions = ['get_by_token', 'save_customer_selection'];

// Für öffentliche Aktionen keine Admin-Authentifizierung erforderlich
if (!in_array($action, $publicActions)) {
    if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
        exit;
    }
}

require_once dirname(__DIR__) . '/config/database.php';

$db = new Database();
$conn = $db->getConnection();
$current_user_id = $_SESSION['admin_user_id'] ?? null;

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'get':
            // GET - Angebote für einen Reifensatz laden
            $wheelsetNumber = $_GET['wheelset_number'] ?? '';
            $autohausId = $_GET['autohaus_id'] ?? '';
            
            if (!$wheelsetNumber || !$autohausId) {
                throw new Exception('wheelset_number und autohaus_id sind erforderlich');
            }
            
            $stmt = $conn->prepare("
                SELECT 
                    id, wheelset_number, autohaus_id, customer_name, customer_no, license_plate, vin,
                    offer_data, letter_content, employee_name, token, token_expires_at, token_used_at,
                    email_sent, email_sent_at, pdf_filename, status, customer_selection, total_amount,
                    created_by, created_at, updated_at
                FROM wheelset_offers
                WHERE wheelset_number = ? AND autohaus_id = ?
                ORDER BY created_at DESC
            ");
            $stmt->execute([$wheelsetNumber, $autohausId]);
            $offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Decode JSON fields
            foreach ($offers as &$offer) {
                if ($offer['offer_data']) {
                    $offer['offer_data'] = json_decode($offer['offer_data'], true);
                }
                if ($offer['customer_selection']) {
                    $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
                }
            }
            
            echo json_encode([
                'success' => true,
                'offers' => $offers,
                'count' => count($offers)
            ]);
            break;
            
        case 'get_all':
            // GET - Alle Angebote für ein Autohaus laden (für Benachrichtigungen)
            $autohausId = $_GET['autohaus_id'] ?? '';
            $filter = $_GET['filter'] ?? ''; // 'pending', 'accepted', 'declined', 'new_request', 'all'
            $search = $_GET['search'] ?? '';
            
            if (!$autohausId) {
                throw new Exception('autohaus_id ist erforderlich');
            }
            
            // Basis-Query - Status können sein: 'draft', 'sent', 'accepted', 'declined', 'new_request', 'viewed', 'processed'
            $sql = "
                SELECT 
                    id, wheelset_number, autohaus_id, customer_name, customer_no, license_plate, vin,
                    offer_data, letter_content, employee_name, token, token_expires_at, token_used_at,
                    email_sent, email_sent_at, pdf_filename, status, customer_selection, total_amount,
                    created_by, created_at, updated_at
                FROM wheelset_offers
                WHERE autohaus_id = ?
            ";
            
            $params = [$autohausId];
            
            // Filter basierend auf Parameter
            if ($filter === 'pending') {
                // "Noch Offen" = noch nicht entschieden (sent oder viewed, aber nicht accepted/declined/new_request)
                $sql .= " AND status IN ('sent', 'viewed')";
            } elseif ($filter === 'accepted') {
                $sql .= " AND status = 'accepted'";
            } elseif ($filter === 'declined') {
                $sql .= " AND status = 'declined'";
            } elseif ($filter === 'new_request') {
                $sql .= " AND status = 'new_request'";
            } elseif ($filter === 'all') {
                // "Alle" = alle Status die relevant sind (accepted, declined, new_request, sent, viewed)
                $sql .= " AND status IN ('accepted', 'declined', 'new_request', 'sent', 'viewed')";
            } else {
                // Default: accepted, declined, new_request (alte Logik für Kompatibilität)
                $sql .= " AND status IN ('accepted', 'declined', 'new_request')";
            }
            
            // Suchfilter
            if (!empty($search)) {
                $sql .= " AND (customer_name LIKE ? OR license_plate LIKE ? OR wheelset_number LIKE ?)";
                $searchParam = '%' . $search . '%';
                $params[] = $searchParam;
                $params[] = $searchParam;
                $params[] = $searchParam;
            }
            
            $sql .= " ORDER BY updated_at DESC, created_at DESC";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Decode JSON fields
            foreach ($offers as &$offer) {
                if ($offer['offer_data']) {
                    $offer['offer_data'] = json_decode($offer['offer_data'], true);
                }
                if ($offer['customer_selection']) {
                    $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
                }
            }
            
            echo json_encode([
                'success' => true,
                'offers' => $offers,
                'count' => count($offers)
            ]);
            break;
            
        case 'get_by_id':
            // GET - Einzelnes Angebot per ID laden
            $offerId = $_GET['offer_id'] ?? '';
            $autohausId = $_GET['autohaus_id'] ?? '';
            
            if (!$offerId || !$autohausId) {
                throw new Exception('offer_id und autohaus_id sind erforderlich');
            }
            
            $stmt = $conn->prepare("
                SELECT 
                    id, wheelset_number, autohaus_id, customer_name, customer_no, license_plate, vin,
                    offer_data, letter_content, employee_name, token, token_expires_at, token_used_at,
                    email_sent, email_sent_at, pdf_filename, status, customer_selection, total_amount,
                    created_by, created_at, updated_at
                FROM wheelset_offers
                WHERE id = ? AND autohaus_id = ?
                LIMIT 1
            ");
            $stmt->execute([$offerId, $autohausId]);
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$offer) {
                throw new Exception('Angebot nicht gefunden');
            }
            
            // Decode JSON fields
            if ($offer['offer_data']) {
                $offer['offer_data'] = json_decode($offer['offer_data'], true);
            }
            if ($offer['customer_selection']) {
                $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
            }
            
            echo json_encode([
                'success' => true,
                'offer' => $offer
            ]);
            break;
            
        case 'create':
        case 'save':
            // POST - Angebot erstellen
            $data = json_decode(file_get_contents('php://input'), true);
            
            $wheelsetNumber = $data['wheelset_number'] ?? '';
            $autohausId = $data['autohaus_id'] ?? '';
            $customerName = $data['customer_name'] ?? '';
            $customerNo = $data['customer_no'] ?? '';
            $licensePlate = $data['license_plate'] ?? '';
            $vin = $data['vin'] ?? '';
            $offerData = $data['offer_data'] ?? [];
            $letterContent = $data['letter_content'] ?? '';
            $employeeName = $data['employee_name'] ?? '';
            $newTires = $data['new_tires'] ?? [];
            $selectedServices = $data['selected_services'] ?? [];
            
            if (!$wheelsetNumber || !$autohausId) {
                throw new Exception('wheelset_number und autohaus_id sind erforderlich');
            }
            
            // Generiere Token für Kundenauswahl
            $token = bin2hex(random_bytes(32));
            $tokenExpiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
            
            // Berechne Gesamtsumme
            $totalAmount = 0;
            foreach ($newTires as $tire) {
                $totalAmount += floatval($tire['total'] ?? 0);
            }
            foreach ($selectedServices as $serviceId) {
                $stmt = $conn->prepare("SELECT price FROM autohaus_services WHERE id = ? AND autohaus_id = ?");
                $stmt->execute([$serviceId, $autohausId]);
                $service = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($service) {
                    $totalAmount += floatval($service['price']);
                }
            }
            
            // Debug: Log Reifen-Daten
            error_log("Saving offer with " . count($newTires) . " tires");
            if (count($newTires) > 0) {
                error_log("First tire: " . print_r($newTires[0], true));
            }
            
            // Speichere Angebot
            $offerDataJson = json_encode([
                'new_tires' => $newTires,
                'selected_services' => $selectedServices,
                'total_amount' => $totalAmount
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            
            // Debug: Prüfe JSON
            if (json_last_error() !== JSON_ERROR_NONE) {
                error_log("JSON Encode Error: " . json_last_error_msg());
                error_log("Data: " . print_r(['new_tires' => $newTires, 'selected_services' => $selectedServices], true));
            }
            
            $stmt = $conn->prepare("
                INSERT INTO wheelset_offers 
                (wheelset_number, autohaus_id, customer_name, customer_no, license_plate, vin,
                 offer_data, letter_content, employee_name, token, token_expires_at,
                 total_amount, created_by, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
            ");
            
            $stmt->execute([
                $wheelsetNumber,
                $autohausId,
                $customerName,
                $customerNo,
                $licensePlate,
                $vin,
                $offerDataJson,
                $letterContent,
                $employeeName,
                $token,
                $tokenExpiresAt,
                $totalAmount,
                $current_user_id
            ]);
            
            $offerId = $conn->lastInsertId();
            
            // Debug: Verifiziere gespeicherte Daten
            $stmt = $conn->prepare("SELECT offer_data FROM wheelset_offers WHERE id = ?");
            $stmt->execute([$offerId]);
            $savedOffer = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($savedOffer) {
                $savedData = json_decode($savedOffer['offer_data'], true);
                error_log("Saved offer_data contains " . count($savedData['new_tires'] ?? []) . " tires");
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Angebot erstellt',
                'offer_id' => $offerId,
                'token' => $token,
                'debug' => [
                    'tires_count' => count($newTires),
                    'services_count' => count($selectedServices)
                ]
            ]);
            break;
            
        case 'send_email':
            // POST - Email mit Angebot versenden
            $data = json_decode(file_get_contents('php://input'), true);
            
            $offerId = $data['offer_id'] ?? null;
            $recipientEmail = $data['recipient_email'] ?? '';
            
            if (!$offerId || empty($recipientEmail)) {
                throw new Exception('offer_id und recipient_email sind erforderlich');
            }
            
            // Lade Angebot
            $stmt = $conn->prepare("SELECT * FROM wheelset_offers WHERE id = ?");
            $stmt->execute([$offerId]);
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$offer) {
                throw new Exception('Angebot nicht gefunden');
            }
            
            // Lade Autohaus-Daten für Email - verwende gleiche Logik wie admin_approve_user.php
            // Erst versuchen das spezifische Autohaus zu laden, sonst Fallback auf erstes aktives Autohaus
            $stmt = $conn->prepare("
                SELECT name, email_smtp_host, email_smtp_port, email_smtp_username, 
                       email_smtp_password, email_smtp_encryption, email_from_name
                FROM autohaus WHERE id = ? AND email_smtp_host IS NOT NULL AND email_smtp_host != '' AND email_smtp_password IS NOT NULL AND email_smtp_password != ''
            ");
            $stmt->execute([$offer['autohaus_id']]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Falls kein Autohaus mit SMTP-Daten gefunden, versuche erstes aktives Autohaus
            if (!$autohaus || empty($autohaus['email_smtp_host']) || empty($autohaus['email_smtp_password'])) {
                $stmt = $conn->prepare("
                    SELECT name, email_smtp_host, email_smtp_port, email_smtp_username, 
                           email_smtp_password, email_smtp_encryption, email_from_name
                    FROM autohaus 
                    WHERE is_active = 1 
                      AND email_smtp_host IS NOT NULL 
                      AND email_smtp_host != '' 
                      AND email_smtp_password IS NOT NULL 
                      AND email_smtp_password != ''
                    ORDER BY id LIMIT 1
                ");
                $stmt->execute();
                $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            
            // Prüfe ob Autohaus mit SMTP-Daten gefunden wurde
            if (!$autohaus || empty($autohaus['email_smtp_host']) || empty($autohaus['email_smtp_password'])) {
                throw new Exception('SMTP-Konfiguration fehlt. Bitte konfigurieren Sie die SMTP-Einstellungen für das Autohaus in den Einstellungen.');
            }
            
            // Generiere PDF (vereinfacht - sollte in separater Datei sein)
            $pdfFilename = 'angebot_' . $offer['wheelset_number'] . '_' . date('Ymd_His') . '.pdf';
            
            // Erstelle Email-Inhalt
            require_once 'send_wheelset_offer_email.php';
            $emailResult = sendWheelsetOfferEmail($offer, $autohaus, $recipientEmail, $offer['token']);
            
            if ($emailResult['success']) {
                // Update Angebot Status - wenn es ein Entwurf war, ändere Status zu 'sent'
                $newStatus = $offer['status'] === 'draft' ? 'sent' : $offer['status'];
                
                $stmt = $conn->prepare("
                    UPDATE wheelset_offers 
                    SET email_sent = 1, email_sent_at = NOW(), status = ?, pdf_filename = ?
                    WHERE id = ?
                ");
                $stmt->execute([$newStatus, $pdfFilename, $offerId]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Email erfolgreich versendet'
                ]);
            } else {
                throw new Exception($emailResult['error'] ?? 'Fehler beim Versenden der Email');
            }
            break;
            
        case 'get_by_token':
            // GET - Angebot per Token laden (für Kundenauswahl)
            $token = $_GET['token'] ?? '';
            
            if (empty($token)) {
                throw new Exception('Token ist erforderlich');
            }
            
            $stmt = $conn->prepare("
                SELECT * FROM wheelset_offers
                WHERE token = ? AND token_expires_at > NOW() AND token_used_at IS NULL
            ");
            $stmt->execute([$token]);
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$offer) {
                // Prüfe ob Token abgelaufen oder verwendet wurde
                $stmt = $conn->prepare("SELECT * FROM wheelset_offers WHERE token = ?");
                $stmt->execute([$token]);
                $expiredOffer = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($expiredOffer) {
                    if ($expiredOffer['token_used_at']) {
                        throw new Exception('Token bereits verwendet');
                    } else {
                        throw new Exception('Token abgelaufen');
                    }
                } else {
                    throw new Exception('Ungültiger Token');
                }
            }
            
            // Decode JSON fields
            if ($offer['offer_data']) {
                $offer['offer_data'] = json_decode($offer['offer_data'], true);
            }
            if ($offer['customer_selection']) {
                $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
            }
            
            echo json_encode([
                'success' => true,
                'offer' => $offer
            ]);
            break;
            
        case 'save_customer_selection':
            // POST - Kundenauswahl speichern
            $data = json_decode(file_get_contents('php://input'), true);
            
            $token = $data['token'] ?? '';
            $customerSelection = $data['customer_selection'] ?? [];
            
            if (empty($token)) {
                throw new Exception('Token ist erforderlich');
            }
            
            // Lade Angebot
            $stmt = $conn->prepare("
                SELECT * FROM wheelset_offers
                WHERE token = ? AND token_expires_at > NOW() AND token_used_at IS NULL
            ");
            $stmt->execute([$token]);
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$offer) {
                throw new Exception('Token ungültig oder abgelaufen');
            }
            
            // Bestimme Status basierend auf Auswahl
            $status = 'viewed'; // Default
            if (!empty($customerSelection['action'])) {
                $status = $customerSelection['action']; // 'accepted', 'declined', 'new_request'
            } else if (!empty($customerSelection['tires']) || !empty($customerSelection['services'])) {
                $status = 'accepted';
            }
            
            // Markiere Token als verwendet und speichere Auswahl
            $stmt = $conn->prepare("
                UPDATE wheelset_offers 
                SET customer_selection = ?, token_used_at = NOW(), status = ?
                WHERE id = ?
            ");
            $stmt->execute([json_encode($customerSelection), $status, $offer['id']]);
            
            // Sende Teams-Benachrichtigung wenn konfiguriert
            if (in_array($status, ['accepted', 'declined', 'new_request'])) {
                require_once 'send_wheelset_offer_teams_notification.php';
                $teamsResult = sendWheelsetOfferTeamsNotification($offer['id'], $status);
                // Log aber wirf keinen Fehler wenn Teams nicht funktioniert
                if (!$teamsResult['success']) {
                    error_log("Teams Notification Failed: " . $teamsResult['error']);
                }
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Auswahl gespeichert',
                'status' => $status
            ]);
            break;
            
        case 'update':
            // POST - Angebot aktualisieren (nur Entwürfe)
            $data = json_decode(file_get_contents('php://input'), true);
            
            $offerId = $data['offer_id'] ?? null;
            $newTires = $data['new_tires'] ?? [];
            $selectedServices = $data['selected_services'] ?? [];
            $letterContent = $data['letter_content'] ?? '';
            $status = $data['status'] ?? 'draft';
            $totalAmount = $data['total_amount'] ?? 0;
            
            if (!$offerId) {
                throw new Exception('offer_id ist erforderlich');
            }
            
            // Prüfe ob Angebot existiert und Entwurf ist
            $stmt = $conn->prepare("SELECT id, status FROM wheelset_offers WHERE id = ?");
            $stmt->execute([$offerId]);
            $existingOffer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$existingOffer) {
                throw new Exception('Angebot nicht gefunden');
            }
            
            if ($existingOffer['status'] !== 'draft') {
                throw new Exception('Nur Entwürfe können bearbeitet werden');
            }
            
            // Update Angebot
            $stmt = $conn->prepare("
                UPDATE wheelset_offers 
                SET offer_data = ?, 
                    letter_content = ?, 
                    status = ?,
                    total_amount = ?,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            
            $offerData = [
                'new_tires' => $newTires,
                'selected_services' => $selectedServices,
                'total_amount' => $totalAmount
            ];
            
            $stmt->execute([
                json_encode($offerData),
                $letterContent,
                $status,
                $totalAmount,
                $offerId
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Angebot aktualisiert',
                'offer_id' => $offerId
            ]);
            break;
            
        default:
            throw new Exception('Ungültige Action');
    }
    
} catch (Exception $e) {
    // Stelle sicher dass nur JSON ausgegeben wird
    http_response_code(400);
    
    // Bereinige Output-Buffer falls PHP-Warnings ausgegeben wurden
    if (ob_get_level() > 0) {
        ob_clean();
    }
    
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
    exit;
}

