-- Users Table für B&E Service App
-- Erstellt die Tabelle für Benutzerregistrierung und -verwaltung

-- Zuerst prüfen wir, ob die Autohäuser-Tabelle existiert
-- Falls nicht, erstellen wir eine einfache Referenztabelle
CREATE TABLE IF NOT EXISTS autohauser (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    short_name VARCHAR(50) NOT NULL,
    address VARCHAR(500) NOT NULL,
    phone VARCHAR(50) NOT NULL,
    email VARCHAR(255) NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Einfügen von Beispieldaten für Autohäuser
INSERT IGNORE INTO autohauser (id, name, short_name, address, phone, email) VALUES
(1, 'B&E Automobile Kiel', 'Kiel', 'Kieler Straße 123, 24103 Kiel', '+49 431 123456', 'kiel@be-automobile.de'),
(2, 'B&E Automobile Norderstedt', 'Norderstedt', 'Norderstedter Straße 456, 22846 Norderstedt', '+49 40 654321', 'norderstedt@be-automobile.de'),
(3, 'B&E Automobile Glinde', 'Glinde', 'Glinder Straße 789, 21509 Glinde', '+49 40 987654', 'glinde@be-automobile.de');

CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    email VARCHAR(255) UNIQUE NOT NULL,
    phone_number VARCHAR(20) NOT NULL,
    license_plate VARCHAR(20) NOT NULL,
    preferred_dealership_id INT NULL,
    profile_image_url VARCHAR(500) NULL,
    is_leasing_vehicle BOOLEAN DEFAULT FALSE,
    is_contract_vehicle BOOLEAN DEFAULT FALSE,
    password_hash VARCHAR(255) NULL, -- NULL für Apple Sign In Benutzer
    apple_user_id VARCHAR(255) NULL UNIQUE, -- Für Apple Sign In
    auth_token VARCHAR(500) NULL,
    token_expires_at DATETIME NULL,
    is_active BOOLEAN DEFAULT TRUE,
    email_verified BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Foreign Key zu Autohäusern (jetzt sicher, da die Tabelle existiert)
    FOREIGN KEY (preferred_dealership_id) REFERENCES autohauser(id) ON DELETE SET NULL,
    
    -- Indizes für bessere Performance
    INDEX idx_email (email),
    INDEX idx_apple_user_id (apple_user_id),
    INDEX idx_auth_token (auth_token),
    INDEX idx_license_plate (license_plate),
    INDEX idx_preferred_dealership (preferred_dealership_id)
);

-- Tabelle für Benutzer-Sessions (optional für erweiterte Sicherheit)
CREATE TABLE IF NOT EXISTS user_sessions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    session_token VARCHAR(500) NOT NULL UNIQUE,
    device_info VARCHAR(500) NULL,
    ip_address VARCHAR(45) NULL,
    expires_at DATETIME NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_session_token (session_token),
    INDEX idx_user_id (user_id),
    INDEX idx_expires_at (expires_at)
);

-- Tabelle für Profilbilder (optional für erweiterte Bildverwaltung)
CREATE TABLE IF NOT EXISTS user_profile_images (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    image_url VARCHAR(500) NOT NULL,
    image_size INT NULL, -- Größe in Bytes
    mime_type VARCHAR(100) NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_is_active (is_active)
);

-- Beispiel-Daten für Tests (optional)
INSERT INTO users (
    first_name, 
    last_name, 
    email, 
    phone_number, 
    license_plate, 
    preferred_dealership_id, 
    is_leasing_vehicle, 
    is_contract_vehicle, 
    password_hash,
    email_verified
) VALUES (
    'Max', 
    'Mustermann', 
    'max.mustermann@example.com', 
    '+49 123 456789', 
    'K-AB 123', 
    1, 
    FALSE, 
    TRUE, 
    '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', -- password
    TRUE
);

-- Trigger für automatische Token-Generierung (optional)
DELIMITER //
CREATE TRIGGER generate_auth_token 
BEFORE INSERT ON users
FOR EACH ROW
BEGIN
    IF NEW.auth_token IS NULL THEN
        SET NEW.auth_token = SHA2(CONCAT(NEW.email, NOW(), RAND()), 256);
        SET NEW.token_expires_at = DATE_ADD(NOW(), INTERVAL 30 DAY);
    END IF;
END//
DELIMITER ;
