<?php
/**
 * User Verification API
 * Für Benutzer: Dokument-Upload und Status-Abfrage
 */

session_start();
require_once '../config/database.php';

$db = new Database();
$conn = $db->getConnection();
$method = $_SERVER['REQUEST_METHOD'];

// Aktion aus GET oder POST lesen (für konsistente Unterstützung)
$action = $_GET['action'] ?? $_POST['action'] ?? '';

header('Content-Type: application/json');


try {
    switch ($method) {
        case 'GET':
            handleGetRequest($action, $conn);
            break;
        case 'POST':
            handlePostRequest($action, $conn);
            break;
        default:
            // 405 für nicht unterstützte Methoden
            http_response_code(405);
            echo json_encode(['success' => false, 'message' => 'Methode nicht unterstützt']);
            exit;
    }
} catch (Exception $e) {
    // Fehlercodes differenzieren je nach Exception-Message
    $msg = $e->getMessage();
    if ($msg === 'Token erforderlich') {
        http_response_code(400);
    } elseif ($msg === 'Ungültiger Token') {
        http_response_code(401);
    } elseif ($msg === 'Methode nicht unterstützt') {
        http_response_code(405);
    } elseif ($msg === 'Unbekannte Aktion') {
        http_response_code(404); // Alternativ: 400, aber hier 404 für Konsistenz
    } else {
        http_response_code(500);
    }
    echo json_encode(['success' => false, 'message' => $msg]);
}


// GET-Handler: user_status, notifications, verification_documents
function handleGetRequest($action, $conn) {
    switch ($action) {
        case 'user_status':
            getUserStatus($conn);
            break;
        case 'notifications':
            getUserNotifications($conn);
            break;
        case 'verification_documents':
            getUserVerificationDocuments($conn);
            break;
        default:
            // 404 für unbekannte Aktion
            throw new Exception('Unbekannte Aktion');
    }
}


// POST-Handler: upload_document, mark_notification_read, user_status
function handlePostRequest($action, $conn) {
    switch ($action) {
        case 'upload_document':
            uploadVerificationDocument($conn);
            break;
        case 'mark_notification_read':
            markNotificationRead($conn);
            break;
        case 'user_status':
            // Änderung: user_status jetzt auch per POST möglich
            getUserStatus($conn);
            break;
        default:
            // 404 für unbekannte Aktion
            throw new Exception('Unbekannte Aktion');
    }
}

/**
 * Benutzerstatus abrufen (für eingeloggte Benutzer)
 */

/**
 * Benutzerstatus abrufen (GET oder POST, Token aus beiden Quellen)
 * Änderungen: Token wird aus GET oder POST gelesen, Fehlercodes werden im Catch-Block gesetzt
 */
function getUserStatus($conn) {
    // Token aus GET oder POST lesen (für beide Methoden nutzbar)
    $token = $_GET['token'] ?? $_POST['token'] ?? null;
    if (!$token) {
        // Fehlercode wird im Catch-Block gesetzt
        throw new Exception('Token erforderlich');
    }
    // Token validieren und Benutzer abrufen
    $stmt = $conn->prepare("
        SELECT u.*, a.name as dealership_name
        FROM users u
        LEFT JOIN autohaus a ON u.preferred_dealership_id = a.id
        WHERE u.auth_token = ? AND u.is_active = 1
    ");
    $stmt->execute([$token]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        throw new Exception('Ungültiger Token');
    }
    // Verifikationsdokumente zählen
    $stmt = $conn->prepare("
        SELECT COUNT(*) as document_count,
               COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved_count,
               COUNT(CASE WHEN status = 'rejected' THEN 1 END) as rejected_count
        FROM verification_documents 
        WHERE user_id = ?
    ");
    $stmt->execute([$user['id']]);
    $document_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    // Ungelesene Benachrichtigungen zählen
    $stmt = $conn->prepare("
        SELECT COUNT(*) as unread_count
        FROM user_notifications 
        WHERE user_id = ? AND is_read = 0
    ");
    $stmt->execute([$user['id']]);
    $notification_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    echo json_encode([
        'success' => true,
        'user' => $user,
        'document_stats' => $document_stats,
        'notification_stats' => $notification_stats
    ]);
}

/**
 * Benutzerbenachrichtigungen abrufen
 */

/**
 * Benutzerbenachrichtigungen abrufen (Token aus GET oder POST)
 */
function getUserNotifications($conn) {
    $token = $_GET['token'] ?? $_POST['token'] ?? null;
    if (!$token) {
        throw new Exception('Token erforderlich');
    }
    // Token validieren
    $stmt = $conn->prepare("SELECT id FROM users WHERE auth_token = ? AND is_active = 1");
    $stmt->execute([$token]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        throw new Exception('Ungültiger Token');
    }
    $stmt = $conn->prepare("
        SELECT * FROM user_notifications 
        WHERE user_id = ? 
        ORDER BY created_at DESC 
        LIMIT 20
    ");
    $stmt->execute([$user['id']]);
    $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
    echo json_encode(['success' => true, 'notifications' => $notifications]);
}

/**
 * Verifikationsdokumente des Benutzers abrufen
 */

/**
 * Verifikationsdokumente des Benutzers abrufen (Token aus GET oder POST)
 */
function getUserVerificationDocuments($conn) {
    $token = $_GET['token'] ?? $_POST['token'] ?? null;
    if (!$token) {
        throw new Exception('Token erforderlich');
    }
    // Token validieren
    $stmt = $conn->prepare("SELECT id FROM users WHERE auth_token = ? AND is_active = 1");
    $stmt->execute([$token]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        throw new Exception('Ungültiger Token');
    }
    $stmt = $conn->prepare("
        SELECT vd.*, au.username as reviewed_by_name
        FROM verification_documents vd
        LEFT JOIN admin_users au ON vd.reviewed_by = au.id
        WHERE vd.user_id = ?
        ORDER BY vd.upload_date DESC
    ");
    $stmt->execute([$user['id']]);
    $documents = $stmt->fetchAll(PDO::FETCH_ASSOC);
    echo json_encode(['success' => true, 'documents' => $documents]);
}

/**
 * Verifikationsdokument hochladen
 */
function uploadVerificationDocument($conn) {
    $token = $_POST['token'] ?? null;
    $document_type = $_POST['document_type'] ?? 'invoice';
    
    if (!$token) {
        throw new Exception('Token erforderlich');
    }
    
    // Token validieren
    $stmt = $conn->prepare("SELECT id FROM users WHERE auth_token = ? AND is_active = 1");
    $stmt->execute([$token]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        throw new Exception('Ungültiger Token');
    }
    
    if (!isset($_FILES['document']) || $_FILES['document']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Keine Datei hochgeladen');
    }
    
    $file = $_FILES['document'];
    $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'application/pdf'];
    $max_size = 5 * 1024 * 1024; // 5MB
    
    if (!in_array($file['type'], $allowed_types)) {
        throw new Exception('Dateityp nicht erlaubt. Nur JPG, PNG, GIF und PDF erlaubt.');
    }
    
    if ($file['size'] > $max_size) {
        throw new Exception('Datei zu groß. Maximum 5MB erlaubt.');
    }
    
    // Upload-Verzeichnis erstellen
    $upload_dir = '../uploads/verification_documents/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Eindeutigen Dateinamen generieren
    $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $file_name = 'doc_' . $user['id'] . '_' . time() . '_' . uniqid() . '.' . $file_extension;
    $file_path = $upload_dir . $file_name;
    
    if (!move_uploaded_file($file['tmp_name'], $file_path)) {
        throw new Exception('Fehler beim Hochladen der Datei');
    }
    
    // In Datenbank speichern
    $stmt = $conn->prepare("
        INSERT INTO verification_documents 
        (user_id, document_type, file_name, file_path, file_size, mime_type)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $user['id'],
        $document_type,
        $file['name'],
        $file_path,
        $file['size'],
        $file['type']
    ]);
    
    // Benachrichtigung erstellen mit Push
    require_once __DIR__ . '/../includes/push_notification_helper.php';
    $result = createNotificationWithPush(
        $conn,
        $user['id'],
        'Dokument hochgeladen',
        'Ihr Verifikationsdokument wurde erfolgreich hochgeladen und wird geprüft.',
        'info',
        '/profile.php'
    );
    
    echo json_encode(['success' => true, 'message' => 'Dokument erfolgreich hochgeladen']);
}

// ...existing code...

/**
 * Hinweis: Hier könnte eine serverseitige Origin-Prüfung für CORS/403 eingebaut werden,
 * z.B. um bestimmte Domains explizit zuzulassen oder zu blockieren.
 * Beispiel (auskommentiert):
 * // if (!in_array($_SERVER['HTTP_ORIGIN'] ?? '', ['https://deine-app.com'])) {
 * //     http_response_code(403);
 * //     exit(json_encode(['success' => false, 'message' => 'Origin nicht erlaubt']));
 * // }
 */
?>
