<?php
/**
 * Korrigierte Smart API - Behebt den Parser-Fehler
 * 
 * Features:
 * - Funktioniert ohne externe Parser-Klassen
 * - Integriertes Parsing
 * - Robuste Fehlerbehandlung
 */

require_once 'cache_manager.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

session_start();
require_once dirname(__DIR__) . '/config/database.php';

/**
 * Korrigierte Smart Wheelsets API
 */
class FixedSmartWheelsetsAPI {
    private $autohausId;
    private $autohausConfig;
    private $cacheManager;
    private $startTime;
    
    public function __construct($autohausId) {
        $this->autohausId = $autohausId;
        $this->startTime = microtime(true);
        
        // Initialisiere Cache nur wenn verfügbar
        try {
            if (isset($GLOBALS['conn']) && $GLOBALS['conn']) {
                $this->cacheManager = new CacheManager($GLOBALS['conn']);
            }
        } catch (Exception $e) {
            error_log("Cache nicht verfügbar: " . $e->getMessage());
        }
        
        $this->loadAutohausConfig();
    }
    
    /**
     * Lädt Autohaus-Konfiguration
     */
    private function loadAutohausConfig() {
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id, name, short_name, 
                    resy_username, resy_password, resy_dealer_id, resy_base_url
                FROM autohaus 
                WHERE id = ? AND is_active = 1
            ");
            $stmt->execute([$this->autohausId]);
            $this->autohausConfig = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$this->autohausConfig) {
                throw new Exception('Autohaus nicht gefunden');
            }
            
            // Bestimme Performance-Tier basierend auf Autohaus-Name
            $this->autohausConfig['performance_tier'] = $this->determinePerformanceTier();
            
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Autohaus-Konfiguration: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Bestimmt Performance-Tier basierend auf Autohaus-Name
     */
    private function determinePerformanceTier() {
        $name = strtolower($this->autohausConfig['name']);
        
        if (strpos($name, 'norderstedt') !== false) {
            return 'large'; // Norderstedt hat viele Reifensätze
        } elseif (strpos($name, 'kiel') !== false) {
            return 'medium'; // Kiel hat mittlere Anzahl
        } elseif (strpos($name, 'glinde') !== false) {
            return 'small'; // Glinde hat wenige Reifensätze
        } else {
            return 'medium'; // Standard
        }
    }
    
    /**
     * Hauptmethode zum Laden der Reifensätze
     */
    public function getWheelsets() {
        try {
            // 1. Prüfe Cache zuerst
            $cachedResult = $this->checkCache();
            if ($cachedResult) {
                return $cachedResult;
            }
            
            // 2. Lade mit adaptiver Strategie
            $wheelsets = $this->loadWheelsetsAdaptive();
            
            // 3. Validiere und bereinige Daten
            $validatedWheelsets = $this->validateWheelsets($wheelsets);
            
            // 4. Speichere im Cache
            $this->saveToCache($validatedWheelsets);
            
            $loadTime = round(microtime(true) - $this->startTime, 2);
            
            return [
                'success' => true,
                'wheelsets' => $validatedWheelsets,
                'count' => count($validatedWheelsets),
                'autohaus_id' => $this->autohausId,
                'performance_tier' => $this->autohausConfig['performance_tier'],
                'load_time' => $loadTime,
                'cached' => false,
                'cache_time' => 'Neu geladen',
                'strategy' => $this->getLoadStrategy()
            ];
            
        } catch (Exception $e) {
            error_log("Fixed Smart API Fehler: " . $e->getMessage());
            return $this->handleError($e);
        }
    }
    
    /**
     * Prüft Cache
     */
    private function checkCache() {
        if (!$this->cacheManager) {
            return null;
        }
        
        $cachedWheelsets = $this->cacheManager->getCachedWheelsets($this->autohausId);
        if ($cachedWheelsets !== null) {
            return [
                'success' => true,
                'wheelsets' => $cachedWheelsets,
                'count' => count($cachedWheelsets),
                'cached' => true,
                'cache_time' => 'Aus Cache geladen'
            ];
        }
        
        return null;
    }
    
    /**
     * Lädt Reifensätze mit adaptiver Strategie
     */
    private function loadWheelsetsAdaptive() {
        $tier = $this->autohausConfig['performance_tier'];
        
        // Bestimme maximale Seiten basierend auf Tier
        $maxPages = $this->getMaxPages($tier);
        $timeout = $this->getTimeout($tier);
        
        error_log("Lade Reifensätze für Autohaus {$this->autohausId} (Tier: $tier, Max Pages: $maxPages)");
        
        return $this->loadWheelsetsFromResy($maxPages, $timeout);
    }
    
    /**
     * Bestimmt maximale Seiten basierend auf Performance-Tier
     */
    private function getMaxPages($tier) {
        switch ($tier) {
            case 'large':
                return 50; // Norderstedt
            case 'medium':
                return 30; // Kiel
            case 'small':
                return 20; // Glinde
            default:
                return 25;
        }
    }
    
    /**
     * Bestimmt Timeout basierend auf Performance-Tier
     */
    private function getTimeout($tier) {
        switch ($tier) {
            case 'large':
                return 180; // 3 Minuten
            case 'medium':
                return 120; // 2 Minuten
            case 'small':
                return 60;  // 1 Minute
            default:
                return 90;
        }
    }
    
    /**
     * Lädt Reifensätze aus RESY
     */
    private function loadWheelsetsFromResy($maxPages, $timeout) {
        set_time_limit($timeout);
        
        $ch = curl_init();
        $allWheelsets = [];
        $page = 1;
        $consecutiveEmptyPages = 0;
        
        try {
            // Initialisiere cURL
            $this->initializeCurl($ch);
            
            // Login durchführen
            $this->performLogin($ch);
            
            // Lade Seiten
            while ($page <= $maxPages) {
                error_log("Lade Seite $page von $maxPages");
                
                $pageWheelsets = $this->loadSinglePage($ch, $page);
                
                if (empty($pageWheelsets)) {
                    $consecutiveEmptyPages++;
                    if ($consecutiveEmptyPages >= 3) {
                        error_log("3 aufeinanderfolgende leere Seiten, stoppe");
                        break;
                    }
                } else {
                    $consecutiveEmptyPages = 0;
                    $allWheelsets = array_merge($allWheelsets, $pageWheelsets);
                    error_log("Seite $page: " . count($pageWheelsets) . " Reifensätze, Total: " . count($allWheelsets));
                }
                
                $page++;
                
                // Kurze Pause zwischen Requests
                usleep(100000); // 0.1 Sekunden
            }
            
        } finally {
            curl_close($ch);
        }
        
        return $allWheelsets;
    }
    
    /**
     * Initialisiert cURL
     */
    private function initializeCurl($ch) {
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_ENCODING, 'gzip,deflate');
        
        // Cookie-Verwaltung
        curl_setopt($ch, CURLOPT_COOKIEJAR, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        
        // Headers
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Connection: keep-alive'
        ]);
    }
    
    /**
     * Führt Login durch
     */
    private function performLogin($ch) {
        // Login-Seite laden
        curl_setopt($ch, CURLOPT_URL, $this->autohausConfig['resy_base_url'] . '/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_POST, false);
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            throw new Exception('Login-Seite konnte nicht geladen werden: ' . curl_error($ch));
        }
        
        // Login-Daten senden
        $postData = [
            'FN' => 'login',
            'UserName' => $this->autohausConfig['resy_username'],
            'Password' => $this->autohausConfig['resy_password'],
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $this->autohausConfig['resy_base_url'] . '/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $this->autohausConfig['resy_base_url'] . '/index.php?m=login&a=login'
        ]);
        
        $loginResponse = curl_exec($ch);
        
        if (curl_error($ch)) {
            throw new Exception('Login fehlgeschlagen: ' . curl_error($ch));
        }
        
        // Prüfe Login-Erfolg
        if (strpos($loginResponse, 'Anmeldename') !== false) {
            throw new Exception('Login fehlgeschlagen - Login-Formular erkannt');
        }
        
        error_log("Login erfolgreich für Autohaus " . $this->autohausId);
    }
    
    /**
     * Lädt eine einzelne Seite
     */
    private function loadSinglePage($ch, $page) {
        $searchUrl = $this->autohausConfig['resy_base_url'] . '/index.php?m=wheelset&a=search';
        
        $postData = [
            'FN' => 'WheelsetSearch',
            'a' => 'search',
            'm' => 'wheelset',
            'KD_ID' => $this->autohausConfig['resy_dealer_id'],
            'STATUS_ID' => '20', // eingelagert
            'Page' => $page,
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $this->autohausConfig['resy_base_url'] . '/index.php?m=wheelset&a=search'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            throw new Exception('Fehler beim Laden der Seite ' . $page . ': ' . curl_error($ch));
        }
        
        return $this->parsePageResponse($response);
    }
    
    /**
     * Parst die Antwort einer Seite
     */
    private function parsePageResponse($html) {
        $wheelsets = [];
        
        // LGS_ID-basiertes Parsing
        if (preg_match_all('/LGS_ID[^>]*>([^<]+)</', $html, $matches)) {
            $lgsData = $matches[1];
            $chunks = array_chunk($lgsData, 5);
            
            foreach ($chunks as $index => $chunk) {
                if ($index === 0) continue; // Überspringe Header
                
                if (count($chunk) >= 5) {
                    $wheelsets[] = [
                        'satznummer' => trim($chunk[0]),
                        'kennzeichen' => trim($chunk[1]),
                        'fahrzeug' => trim($chunk[2]),
                        'halter' => $this->findCustomerName($html, trim($chunk[0])),
                        'eingangsdatum' => trim($chunk[4]),
                        'status' => 'eingelagert'
                    ];
                }
            }
        }
        
        return $wheelsets;
    }
    
    /**
     * Findet Kundenname in HTML
     */
    private function findCustomerName($html, $satznummer) {
        // Vereinfachte Kundenname-Suche
        $patterns = [
            '/[^>]*' . preg_quote($satznummer, '/') . '[^<]*<[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            '/<div[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $html, $matches)) {
                $candidate = trim(strip_tags($matches[1]));
                if (strlen($candidate) > 3 && strlen($candidate) < 100) {
                    return $candidate;
                }
            }
        }
        
        return 'Unbekannt';
    }
    
    /**
     * Validiert Reifensätze
     */
    private function validateWheelsets($wheelsets) {
        $validated = [];
        $seen = [];
        
        foreach ($wheelsets as $wheelset) {
            // Validiere erforderliche Felder
            if (empty($wheelset['satznummer']) || empty($wheelset['kennzeichen'])) {
                continue;
            }
            
            // Prüfe auf Duplikate
            $key = $wheelset['satznummer'] . '_' . $wheelset['kennzeichen'];
            if (isset($seen[$key])) {
                continue;
            }
            $seen[$key] = true;
            
            // Bereinige Daten
            $wheelset['satznummer'] = trim(strip_tags($wheelset['satznummer']));
            $wheelset['kennzeichen'] = trim(strip_tags($wheelset['kennzeichen']));
            $wheelset['halter'] = trim(strip_tags($wheelset['halter'] ?? 'Unbekannt'));
            $wheelset['fahrzeug'] = trim(strip_tags($wheelset['fahrzeug'] ?? ''));
            $wheelset['eingangsdatum'] = trim(strip_tags($wheelset['eingangsdatum'] ?? ''));
            $wheelset['status'] = 'eingelagert';
            
            $validated[] = $wheelset;
        }
        
        return $validated;
    }
    
    /**
     * Speichert im Cache
     */
    private function saveToCache($wheelsets) {
        if (!$this->cacheManager) {
            return;
        }
        
        $this->cacheManager->setCachedWheelsets($this->autohausId, 20, $wheelsets);
    }
    
    /**
     * Gibt Lade-Strategie zurück
     */
    private function getLoadStrategy() {
        $tier = $this->autohausConfig['performance_tier'];
        
        switch ($tier) {
            case 'large':
                return 'chunked';
            case 'medium':
                return 'balanced';
            case 'small':
                return 'thorough';
            default:
                return 'standard';
        }
    }
    
    /**
     * Behandelt Fehler
     */
    private function handleError($exception) {
        error_log("Fixed Smart API Fehler: " . $exception->getMessage());
        
        // Fallback auf lokale Datenbank
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id,
                    CONCAT(YEAR(created_at), '.', LPAD(MONTH(created_at), 2, '0'), '.', LPAD(DAY(created_at), 2, '0')) as satznummer,
                    kennzeichen,
                    fahrzeug_id as fahrzeug,
                    kunde as halter,
                    DATE_FORMAT(created_at, '%d.%m.%Y') as eingangsdatum,
                    'eingelagert' as status,
                    NULL as prufdoc_id
                FROM appointments 
                WHERE autohaus_id = ? 
                AND status IN ('confirmed', 'completed')
                ORDER BY created_at DESC
                LIMIT 50
            ");
            $stmt->execute([$this->autohausId]);
            $wheelsets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($wheelsets)) {
                return [
                    'success' => true,
                    'wheelsets' => $wheelsets,
                    'count' => count($wheelsets),
                    'info' => 'Fallback: Lokale Datenbank',
                    'cached' => false
                ];
            }
        } catch (Exception $e) {
            error_log("Fallback fehlgeschlagen: " . $e->getMessage());
        }
        
        // Letzter Fallback: Mock-Daten
        return [
            'success' => true,
            'wheelsets' => $this->getMockWheelsets(),
            'count' => 3,
            'warning' => 'Demo-Daten (System nicht erreichbar)',
            'cached' => false
        ];
    }
    
    /**
     * Generiert Mock-Daten
     */
    private function getMockWheelsets() {
        return [
            [
                'satznummer' => '2025.001.001',
                'kennzeichen' => 'KI-AB 123',
                'fahrzeug' => 'VIN123456789',
                'halter' => 'Max Mustermann GmbH',
                'eingangsdatum' => date('d.m.Y'),
                'status' => 'eingelagert',
                'prufdoc_id' => '12345'
            ],
            [
                'satznummer' => '2025.001.002',
                'kennzeichen' => 'KI-CD 456',
                'fahrzeug' => 'VIN987654321',
                'halter' => 'Anna Schmidt',
                'eingangsdatum' => date('d.m.Y', strtotime('-1 day')),
                'status' => 'eingelagert',
                'prufdoc_id' => '12346'
            ],
            [
                'satznummer' => '2025.001.003',
                'kennzeichen' => 'KI-EF 789',
                'fahrzeug' => 'VIN456789123',
                'halter' => 'Peter Müller & Co. KG',
                'eingangsdatum' => date('d.m.Y', strtotime('-2 days')),
                'status' => 'eingelagert',
                'prufdoc_id' => '12347'
            ]
        ];
    }
}

// API-Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['autohaus_id'])) {
        echo json_encode([
            'success' => false,
            'error' => 'Autohaus-ID ist erforderlich'
        ]);
        exit;
    }
    
    $autohausId = $input['autohaus_id'];
    
    try {
        $api = new FixedSmartWheelsetsAPI($autohausId);
        $result = $api->getWheelsets();
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => 'API-Fehler: ' . $e->getMessage()
        ]);
    }
} else {
    echo json_encode([
        'success' => false,
        'error' => 'Nur POST-Requests erlaubt'
    ]);
}
?>

