<?php
/**
 * Sendet Push-Benachrichtigungen an alle Geräte eines Benutzers
 * Wird aufgerufen wenn eine neue Benachrichtigung in user_notifications erstellt wird
 */

require_once __DIR__ . '/../config/database.php';

// Lade Composer Autoloader für web-push Library
if (file_exists(__DIR__ . '/../vendor/autoload.php')) {
    require_once __DIR__ . '/../vendor/autoload.php';
}

// Workaround für OpenSSL 3.0 auf Windows/XAMPP
// Setze OPENSSL_CONF Umgebungsvariable, damit openssl_pkey_new() funktioniert
if (PHP_OS_FAMILY === 'Windows') {
    $opensslConfigPaths = [
        'C:\\xampp\\apache\\conf\\openssl.cnf',
        'C:\\xampp\\php\\extras\\ssl\\openssl.cnf',
    ];
    
    foreach ($opensslConfigPaths as $path) {
        if (file_exists($path)) {
            putenv("OPENSSL_CONF=$path");
            break;
        }
    }
}

// Diese Funktion kann von anderen Scripts aufgerufen werden
function sendPushNotificationToUser($user_id, $title, $message, $url = '/profile.php', $notificationId = null, $type = 'info') {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Prüfe welche Spaltennamen existieren
        $stmt = $conn->query("DESCRIBE push_subscriptions");
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $useNewNames = in_array('p256dh', $columns) && in_array('auth', $columns);
        $p256dhColumn = $useNewNames ? 'p256dh' : 'p256dh_key';
        $authColumn = $useNewNames ? 'auth' : 'auth_key';
        
        // Hole alle aktiven Push-Subscriptions des Users
        $stmt = $conn->prepare("
            SELECT endpoint, $p256dhColumn as p256dh, $authColumn as auth 
            FROM push_subscriptions 
            WHERE user_id = ? AND is_active = 1
        ");
        $stmt->execute([$user_id]);
        $subscriptions = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($subscriptions)) {
            return ['success' => false, 'message' => 'Keine aktiven Subscriptions gefunden'];
        }
        
        // VAPID Keys laden
        $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'vapid_public_key'");
        $stmt->execute();
        $vapidPublicKey = $stmt->fetchColumn();
        
        $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'vapid_private_key'");
        $stmt->execute();
        $vapidPrivateKey = $stmt->fetchColumn();
        
        if (!$vapidPublicKey || !$vapidPrivateKey) {
            return ['success' => false, 'message' => 'VAPID Keys nicht konfiguriert'];
        }
        
        // Web Push Library verwenden (muss installiert sein: composer require minishlink/web-push)
        // Für jetzt: Einfache HTTP-Implementierung
        
        $results = [];
        foreach ($subscriptions as $subscription) {
            $result = sendWebPush(
                $subscription['endpoint'],
                $subscription['p256dh'],
                $subscription['auth'],
                [
                    'title' => $title,
                    'message' => $message,
                    'url' => $url,
                    'notificationId' => $notificationId,
                    'type' => $type
                ],
                $vapidPublicKey,
                $vapidPrivateKey
            );
            
            $results[] = $result;
        }
        
        return [
            'success' => true,
            'sent' => count(array_filter($results, function($r) { return $r['success']; })),
            'total' => count($subscriptions),
            'results' => $results
        ];
        
    } catch (Exception $e) {
        error_log('Push Notification Error: ' . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Sendet eine Web Push Benachrichtigung
 * Verwendet die web-push PHP Library falls verfügbar, sonst Fallback
 */
function sendWebPush($endpoint, $p256dh, $auth, $payload, $vapidPublicKey, $vapidPrivateKey) {
    // Prüfe ob web-push Library verfügbar ist
    if (class_exists('\Minishlink\WebPush\WebPush')) {
        try {
            // Validiere Keys
            if (empty($p256dh) || empty($auth)) {
                error_log('Push notification: Empty keys - p256dh: ' . (empty($p256dh) ? 'empty' : 'ok') . ', auth: ' . (empty($auth) ? 'empty' : 'ok'));
                return ['success' => false, 'message' => 'Subscription-Keys fehlen'];
            }
            
            // Prüfe Key-Längen (p256dh sollte ~88 Zeichen, auth sollte ~24 Zeichen sein)
            if (strlen($p256dh) < 50 || strlen($auth) < 10) {
                error_log('Push notification: Invalid key lengths - p256dh: ' . strlen($p256dh) . ', auth: ' . strlen($auth));
                return ['success' => false, 'message' => 'Ungültige Key-Längen'];
            }
            
            $webPush = new \Minishlink\WebPush\WebPush([
                'VAPID' => [
                    'subject' => 'mailto:info@be-gruppe.de', // E-Mail für VAPID
                    'publicKey' => $vapidPublicKey,
                    'privateKey' => $vapidPrivateKey,
                ],
            ]);
            
            $subscription = \Minishlink\WebPush\Subscription::create([
                'endpoint' => $endpoint,
                'keys' => [
                    'p256dh' => $p256dh,
                    'auth' => $auth
                ]
            ]);
            
            $result = $webPush->queueNotification(
                $subscription,
                json_encode($payload)
            );
            
            // Sende alle Benachrichtigungen
            $successCount = 0;
            $errorMessages = [];
            foreach ($webPush->flush() as $report) {
                if ($report->isSuccess()) {
                    $successCount++;
                } else {
                    $reason = $report->getReason();
                    error_log('Push notification failed: ' . $reason . ' (Endpoint: ' . substr($report->getEndpoint(), 0, 50) . '...)');
                    $errorMessages[] = $reason;
                }
            }
            
            if ($successCount > 0) {
                return ['success' => true, 'message' => 'Push-Benachrichtigung gesendet', 'sent' => $successCount];
            } else {
                return ['success' => false, 'message' => implode('; ', $errorMessages)];
            }
            
        } catch (\Exception $e) {
            error_log('Web Push Error: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    } else {
        // Fallback: Library nicht installiert
        error_log('Web Push Library nicht installiert. Installiere mit: composer require minishlink/web-push');
        return [
            'success' => false, 
            'message' => 'Web Push Library nicht installiert. Bitte installieren Sie: composer require minishlink/web-push'
        ];
    }
}

// Wenn direkt aufgerufen (für Tests)
if (php_sapi_name() === 'cli' || (isset($_GET['test']) && $_GET['test'] === '1')) {
    // Nur für Tests - sollte in Produktion nicht direkt aufgerufen werden
    header('Content-Type: application/json');
    
    if (!isset($_GET['user_id']) || !isset($_GET['title']) || !isset($_GET['message'])) {
        echo json_encode(['success' => false, 'message' => 'Parameter fehlen']);
        exit;
    }
    
    $result = sendPushNotificationToUser(
        $_GET['user_id'],
        $_GET['title'],
        $_GET['message'],
        $_GET['url'] ?? '/profile.php',
        $_GET['notification_id'] ?? null,
        $_GET['type'] ?? 'info'
    );
    
    echo json_encode($result);
}
?>
