<?php
/**
 * API-Endpoint zum Speichern von Feedback
 */

header('Content-Type: application/json');

session_start();
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/auth.php';

// Prüfe ob User eingeloggt ist
if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Nicht eingeloggt']);
    exit;
}

$user = getCurrentUser();
if (!$user) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Benutzer nicht gefunden']);
    exit;
}

// Input lesen
$input = json_decode(file_get_contents('php://input'), true);
if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Ungültige Eingabe']);
    exit;
}

// Validierung
$rating = isset($input['rating']) ? (int)$input['rating'] : 0;
$feedback_text = isset($input['feedback_text']) ? trim($input['feedback_text']) : '';
$feedback_type = isset($input['feedback_type']) ? $input['feedback_type'] : 'general';
$page_url = isset($input['page_url']) ? trim($input['page_url']) : '';
$is_anonymous = isset($input['is_anonymous']) ? (bool)$input['is_anonymous'] : false;

// Validierung
if ($rating < 1 || $rating > 5) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Bewertung muss zwischen 1 und 5 liegen']);
    exit;
}

if (empty($feedback_text) || strlen($feedback_text) < 10) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Feedback-Text muss mindestens 10 Zeichen lang sein']);
    exit;
}

if (!in_array($feedback_type, ['service', 'booking', 'general'])) {
    $feedback_type = 'general';
}

// Autohaus-ID ermitteln (aus Session oder aus dem aktuellen Kontext)
$autohaus_id = null;
if (isset($_SESSION['selected_autohaus_id'])) {
    $autohaus_id = (int)$_SESSION['selected_autohaus_id'];
} else {
    // Versuche Autohaus aus User-Präferenzen zu ermitteln
    if (isset($user['preferred_dealership_id']) && !empty($user['preferred_dealership_id'])) {
        $autohaus_id = (int)$user['preferred_dealership_id'];
    }
}

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Prüfe ob Benutzer bereits Feedback gegeben hat
    if (!$is_anonymous && isset($user['id'])) {
        $checkStmt = $conn->prepare("SELECT id FROM feedback WHERE user_id = ? LIMIT 1");
        $checkStmt->execute([$user['id']]);
        if ($checkStmt->rowCount() > 0) {
            http_response_code(400);
            echo json_encode([
                'success' => false, 
                'error' => 'Sie haben bereits Feedback abgegeben. Jeder Benutzer kann nur einmal Feedback geben.'
            ]);
            exit;
        }
    }
    
    // Prüfe ob Feedback-Tabelle existiert, falls nicht erstelle sie
    try {
        // Prüfe ob Tabelle existiert
        $checkTable = $conn->query("SHOW TABLES LIKE 'feedback'");
        if ($checkTable->rowCount() == 0) {
            // Tabelle erstellen ohne FOREIGN KEY Constraints (werden später hinzugefügt falls nötig)
            $createTableSQL = "
                CREATE TABLE feedback (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT NULL,
                    autohaus_id INT NULL,
                    rating INT NOT NULL,
                    feedback_text TEXT NOT NULL,
                    feedback_type ENUM('service', 'booking', 'general') DEFAULT 'general',
                    page_url VARCHAR(500) NULL,
                    is_anonymous BOOLEAN DEFAULT FALSE,
                    status ENUM('new', 'read', 'archived') DEFAULT 'new',
                    admin_response TEXT NULL,
                    admin_response_by INT NULL,
                    admin_response_at DATETIME NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    INDEX idx_user_id (user_id),
                    INDEX idx_autohaus_id (autohaus_id),
                    INDEX idx_status (status),
                    INDEX idx_created_at (created_at),
                    INDEX idx_feedback_type (feedback_type)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ";
            
            $conn->exec($createTableSQL);
            error_log("Feedback table created successfully");
        }
    } catch (PDOException $tableError) {
        // Tabelle existiert möglicherweise bereits oder Fehler beim Erstellen
        error_log("Feedback Table Check/Create Error: " . $tableError->getMessage());
        error_log("PDO Error Code: " . $tableError->getCode());
        error_log("PDO Error Info: " . print_r($tableError->errorInfo, true));
        // Weiter versuchen, Tabelle könnte bereits existieren
    }
    
    // Validierung der Bewertung (da CHECK Constraint möglicherweise nicht unterstützt wird)
    if ($rating < 1 || $rating > 5) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Bewertung muss zwischen 1 und 5 liegen']);
        exit;
    }
    
    // Feedback speichern
    $stmt = $conn->prepare("
        INSERT INTO feedback 
        (user_id, autohaus_id, rating, feedback_text, feedback_type, page_url, is_anonymous, status)
        VALUES (?, ?, ?, ?, ?, ?, ?, 'new')
    ");
    
    $result = $stmt->execute([
        $is_anonymous ? null : $user['id'],
        $autohaus_id,
        $rating,
        $feedback_text,
        $feedback_type,
        $page_url ?: null,
        $is_anonymous ? 1 : 0
    ]);
    
    if (!$result) {
        throw new Exception('Fehler beim Ausführen des INSERT-Statements');
    }
    
    $feedback_id = $conn->lastInsertId();
    
    echo json_encode([
        'success' => true,
        'message' => 'Feedback erfolgreich gespeichert',
        'feedback_id' => $feedback_id
    ]);
    
} catch (PDOException $e) {
    error_log("Feedback Save PDO Error: " . $e->getMessage());
    error_log("PDO Error Code: " . $e->getCode());
    error_log("PDO Error Info: " . print_r($e->errorInfo(), true));
    http_response_code(500);
    
    // Für Entwicklung: Detaillierte Fehlermeldung
    $errorMessage = 'Fehler beim Speichern des Feedbacks';
    $debugInfo = null;
    
    // In Entwicklungsumgebung mehr Details anzeigen
    if (isset($_SERVER['HTTP_HOST']) && (strpos($_SERVER['HTTP_HOST'], 'localhost') !== false || strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false)) {
        $errorMessage = $e->getMessage();
        $debugInfo = [
            'code' => $e->getCode(),
            'errorInfo' => $e->errorInfo(),
            'sqlState' => $e->getCode()
        ];
    }
    
    echo json_encode([
        'success' => false, 
        'error' => $errorMessage,
        'debug' => $debugInfo
    ]);
} catch (Exception $e) {
    error_log("Feedback Save Error: " . $e->getMessage());
    error_log("Stack Trace: " . $e->getTraceAsString());
    http_response_code(500);
    
    // Für Entwicklung: Detaillierte Fehlermeldung
    $errorMessage = $e->getMessage();
    $debugInfo = null;
    
    // In Entwicklungsumgebung mehr Details anzeigen
    if (isset($_SERVER['HTTP_HOST']) && (strpos($_SERVER['HTTP_HOST'], 'localhost') !== false || strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false)) {
        $debugInfo = [
            'message' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ];
    }
    
    echo json_encode([
        'success' => false, 
        'error' => $errorMessage,
        'debug' => $debugInfo
    ]);
}

