<?php
/**
 * resy_scrape.php
 * Verwendung:
 *   CLI:  php resy_scrape.php --autohaus_id=1 [--kennzeichen="KI-BE 123"]
 *   HTTP: resy_scrape.php?autohaus_id=1&kennzeichen=KI-BE+123
 *
 * DB-Zugang via ENV: DB_HOST, DB_NAME, DB_USER, DB_PASS
 */

declare(strict_types=1);

// ---------- Eingaben ----------
$cli = (php_sapi_name() === 'cli');
parse_str(implode('&', array_slice($argv ?? [], 1)), $cliArgs);

$autohausId   = (int)($cli ? ($cliArgs['--autohaus_id'] ?? $cliArgs['autohaus_id'] ?? 0) : ($_GET['autohaus_id'] ?? 0));
$kennzeichenF = (string)($cli ? ($cliArgs['--kennzeichen']   ?? $cliArgs['kennzeichen']   ?? '') : ($_GET['kennzeichen'] ?? ''));

if ($autohausId <= 0) {
  http_response_code(400);
  exit(json_encode(['error'=>'autohaus_id fehlt oder ist ungültig'], JSON_UNESCAPED_UNICODE));
}

// ---------- DB: Credentials laden ----------
function db(): PDO {
  $dsn = sprintf('mysql:host=%s;dbname=%s;charset=utf8mb4', getenv('DB_HOST'), getenv('DB_NAME'));
  $pdo = new PDO($dsn, getenv('DB_USER'), getenv('DB_PASS'), [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

try {
  $pdo = db();
  $stmt = $pdo->prepare("SELECT resy_username, resy_password, resy_base_url FROM autohaus WHERE id = ?");
  $stmt->execute([$autohausId]);
  $row = $stmt->fetch();
  if (!$row) throw new RuntimeException("Autohaus-ID nicht gefunden");

  $RESY_USER = $row['resy_username'];
  $RESY_PASS = $row['resy_password'];
  $BASE_URL  = rtrim($row['resy_base_url'] ?: 'https://resy.dtm-reifen.de', '/');
} catch (Throwable $e) {
  http_response_code(500);
  exit(json_encode(['error'=>'DB-Fehler','detail'=>$e->getMessage()], JSON_UNESCAPED_UNICODE));
}

// ---------- HTTP-Helper mit cURL ----------
$cookieFile = tempnam(sys_get_temp_dir(), 'resy_cookie_');
$UA = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/126.0 Safari/537.36";

function http(array $opt): array {
  global $cookieFile, $UA;
  $ch = curl_init();
  $base = [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_SSL_VERIFYPEER => false,
    CURLOPT_SSL_VERIFYHOST => false,
    CURLOPT_USERAGENT      => $UA,
    CURLOPT_CONNECTTIMEOUT => 30,
    CURLOPT_TIMEOUT        => 60,
    CURLOPT_COOKIEJAR      => $cookieFile,
    CURLOPT_COOKIEFILE     => $cookieFile,
    CURLOPT_HEADER         => false,
  ];
  curl_setopt_array($ch, $base + $opt);
  $body = curl_exec($ch);
  if ($body === false) {
    $err = curl_error($ch);
    curl_close($ch);
    throw new RuntimeException("cURL: $err");
  }
  $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);
  return [$status, $body];
}

function absUrl(string $base, string $path): string {
  if (preg_match('~^https?://~i', $path)) return $path;
  return rtrim($base, '/') . '/' . ltrim($path, '/');
}

function xp(string $html): DOMXPath {
  libxml_use_internal_errors(true);
  $doc = new DOMDocument();
  $doc->loadHTML($html, LIBXML_NOWARNING|LIBXML_NOERROR|LIBXML_NONET|LIBXML_COMPACT);
  libxml_clear_errors();
  return new DOMXPath($doc);
}
function nodeText(?DOMNode $n): string { return $n ? trim($n->textContent) : ''; }
function q1(DOMXPath $xp, string $xpath, ?DOMNode $ctx=null): ?DOMNode {
  $nl = $xp->query($xpath, $ctx);
  return ($nl && $nl->length) ? $nl->item(0) : null;
}
function deFloat(string $s): float {
  if ($s === '') return 0.0;
  if (preg_match('/[-+]?\d{1,3}(?:[.,]\d+)?/', $s, $m)) {
    return (float)str_replace([',','.'], ['.',''], str_replace('.', '', $m[0]));
  }
  return 0.0;
}

// ---------- 1) Login-Seite holen ----------
const LOGIN_PATH        = '/index.php?m=login';       // ggf. anpassen
const APPOINTMENTS_PATH = '/index.php?m=appointment'; // ggf. anpassen

try {
  [$st, $loginHtml] = http([CURLOPT_URL => absUrl($BASE_URL, LOGIN_PATH), CURLOPT_HTTPGET => true]);
  if ($st >= 400) throw new RuntimeException("Login-Seite HTTP $st");

  $x = xp($loginHtml);
  $form = q1($x, "//form[.//input[@type='password']]");
  if (!$form) throw new RuntimeException("Login-Formular nicht gefunden");

  $action = $form->attributes->getNamedItem('action')?->nodeValue ?: LOGIN_PATH;
  $method = strtolower($form->attributes->getNamedItem('method')?->nodeValue ?: 'post');

  // Hidden + Text Inputs sammeln
  $fields = [];
  foreach ($x->query(".//input", $form) as $in) {
    /** @var DOMElement $in */
    $type = strtolower($in->getAttribute('type'));
    $name = $in->getAttribute('name');
    if ($name === '' || in_array($type, ['submit','button','image'])) continue;
    $fields[$name] = $in->getAttribute('value');
  }
  // Username/Password-Feldnamen heuristisch
  $userField = null;
  foreach ($x->query(".//input[@type='text' or @type='email' or @type='username']", $form) as $in) {
    $n = strtolower($in->getAttribute('name'));
    if (str_contains($n,'user') || str_contains($n,'login') || str_contains($n,'email')) { $userField=$in->getAttribute('name'); break; }
  }
  $passField = q1($x, ".//input[@type='password']", $form)?->getAttribute('name');
  if (!$userField || !$passField) throw new RuntimeException("Login-Feldnamen nicht erkannt");

  $fields[$userField] = $RESY_USER;
  $fields[$passField] = $RESY_PASS;

  [$st, $afterLogin] = http([
    CURLOPT_URL => absUrl($BASE_URL, $action),
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => http_build_query($fields),
    CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
  ]);
  if ($st >= 400) throw new RuntimeException("Login-POST HTTP $st");

  if (!str_contains($afterLogin, 'Alle Termine')) {
    [$st, $afterLogin] = http([CURLOPT_URL => absUrl($BASE_URL, APPOINTMENTS_PATH), CURLOPT_HTTPGET => true]);
    if ($st >= 400) throw new RuntimeException("Terminliste HTTP $st");
  }

  // ---------- 2) Terminliste → Detail-Link ----------
  $x = xp($afterLogin);
  $rows = $x->query("//table//tbody//tr");
  if (!$rows || !$rows->length) throw new RuntimeException("Keine Terminzeilen gefunden");

  $detailHref = null;
  foreach ($rows as $tr) {
    $rowTxt = strtolower(nodeText($tr));
    if ($kennzeichenF && !str_contains($rowTxt, strtolower($kennzeichenF))) continue;

    $view = q1($x, ".//a[@title='Ansehen']", $tr) ?: q1($x, ".//td[last()]//a[1]", $tr);
    if ($view) { $detailHref = $view->getAttribute('href'); break; }
  }
  if (!$detailHref) throw new RuntimeException("Kein Detail-Link gefunden");

  // ---------- 3) Detailseite ----------
  [$st, $detailHtml] = http([CURLOPT_URL => absUrl($BASE_URL, $detailHref), CURLOPT_HTTPGET => true]);
  if ($st >= 400) throw new RuntimeException("Detailseite HTTP $st");
  $x = xp($detailHtml);

  // ---------- 4) Felder scrapen ----------
  $einlagerung = [
    'satznummer'    => nodeText(q1($x, "//p[@id='WheelLGS_ID']/b")),
    'eingangsdatum' => nodeText(q1($x, "//p[@id='WheelEINGANG']/b")),
    'serviceart'    => nodeText(q1($x, "//p[@id='WheelSERV_ID']/b")),
    'ausgangsdatum' => nodeText(q1($x, "//p[@id='WheelAUSGANG']/b")),
    'autohaus'      => nodeText(q1($x, "//p[@id='WheelKD_ID']/b")),
  ];
  $fahrzeug = [
    'kennzeichen' => nodeText(q1($x, "//p[@id='WheelKENNZEICHEN']/b")),
    'fin'         => nodeText(q1($x, "//p[@id='WheelFAHRGESTNR']/b")),
  ];

  // Tabelle
  $tabMeta = [];
  foreach ($x->query("//table[@id='WheelsetViewWheels']//tbody//tr") as $tr) {
    $tds = $x->query("./td", $tr); if ($tds->length < 14) continue;
    $pos   = trim($tds->item(0)->textContent ?? '');
    $size  = trim($tds->item(1)->textContent ?? '');
    $brand = trim($tds->item(5)->textContent ?? '');
    $model = trim($tds->item(6)->textContent ?? '');
    $dot   = trim($tds->item(8)->textContent ?? '');
    $type  = trim($tds->item(11)->textContent ?? '');
    $empf  = trim($tds->item(13)->textContent ?? '');
    if ($pos) $tabMeta[$pos] = compact('size','brand','model','dot','type','empf');
  }

  // Profiltiefen je Position
  $posList = ['VL','VR','HL','HR'];
  $reifen  = [];
  foreach ($posList as $p) {
    $box = q1($x, "//div[@id='{$p}']");
    $i = $m = $a = 0.0;
    if ($box) {
      $i = deFloat(nodeText(q1($x, ".//div[@id='Profilinnen']//p/b", $box)));
      $m = deFloat(nodeText(q1($x, ".//div[@id='Profilmitte']//p/b", $box)));
      $a = deFloat(nodeText(q1($x, ".//div[@id='Profilaussen']//p/b", $box)));
    }
    $avg = round((($i+$m+$a)/3), 1);
    $state = $avg >= 6.0 ? 'sehr_gut' : ($avg >= 4.0 ? 'gut' : ($avg >= 2.0 ? 'befriedigend' : 'kritisch'));
    $meta  = $tabMeta[$p] ?? ['size'=>'','brand'=>'','model'=>'','dot'=>'','type'=>'','empf'=>''];
    $s = mb_strtolower($meta['empf'] ?? '');
    $status = str_contains($s,'austausch') ? 'austausch_notwendig' : (str_contains($s,'prüf') ? 'pruefen' : 'keine_fehler');

    $reifen[] = [
      'pos' => $p,
      'size'=>$meta['size'] ?: '—',
      'brand'=>$meta['brand'] ?: '—',
      'model'=>$meta['model'] ?: '—',
      'dot'=>$meta['dot'] ?: '—',
      'type'=>$meta['type'] ?: '—',
      'profile'=>['innen'=>$i,'mitte'=>$m,'aussen'=>$a],
      'avg'=>$avg, 'state'=>$state, 'status'=>$status,
      'empfehlung'=>$meta['empf'] ?: '—',
    ];
  }

  // ---------- Ausgabe ----------
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode([
    'satznummer'    => $einlagerung['satznummer'],
    'eingangsdatum' => $einlagerung['eingangsdatum'],
    'serviceart'    => $einlagerung['serviceart'],
    'ausgangsdatum' => $einlagerung['ausgangsdatum'],
    'autohaus'      => $einlagerung['autohaus'],
    'fahrzeug'      => $fahrzeug,
    'reifen'        => $reifen,
  ], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode(['error'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
} finally {
  @unlink($cookieFile ?? '');
}
