<?php
/**
 * RESY Image Proxy - Vereinfachte Version
 * Lädt Reifenbilder mit korrekten Session-Cookies
 */

header('Content-Type: image/jpeg');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_image_proxy.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

try {
    $lgfotoId = $_GET['lgfoto_id'] ?? null;
    $appointmentId = $_GET['appointment_id'] ?? null;
    
    debugLog("Image Proxy Request - LGFOTO_ID: $lgfotoId, Appointment ID: $appointmentId");
    
    if (!$lgfotoId) {
        throw new Exception('LGFOTO_ID fehlt');
    }
    
    // Prüfe zuerst lokalen Cache
    require_once __DIR__ . '/image_cache_manager.php';
    $imageCache = new ImageCacheManager();
    
    if ($imageCache->isImageCached($lgfotoId)) {
        debugLog("Bild im lokalen Cache gefunden: LGFOTO_ID $lgfotoId");
        
        $cachedImage = $imageCache->getCachedImage($lgfotoId);
        if ($cachedImage['success']) {
            // Setze korrekten Content-Type
            header('Content-Type: ' . $cachedImage['content_type']);
            header('Content-Length: ' . $cachedImage['size']);
            header('Cache-Control: max-age=86400'); // 24 Stunden Cache für lokale Bilder
            
            echo $cachedImage['data'];
            exit;
        }
    }
    
    debugLog("Bild nicht im lokalen Cache, lade von RESY: LGFOTO_ID $lgfotoId");
    
    // Hole Autohaus-Daten
    require_once __DIR__ . '/../config/database.php';
    $pdo = getDatabaseConnection();
    
    $stmt = $pdo->prepare("SELECT resy_base_url, resy_username, resy_password FROM autohaus WHERE id = ?");
    $stmt->execute([1]); // Verwende Autohaus ID 1
    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$autohaus) {
        throw new Exception('Autohaus nicht gefunden');
    }
    
    debugLog("Autohaus gefunden: " . $autohaus['resy_base_url']);
    
    // Verwende direkt die Fallback-Cookies (automatischer Login funktioniert nicht)
    debugLog("--- Verwende Fallback-Cookies direkt ---");
    debugLog("⚠️ Automatischer Login deaktiviert - verwende Fallback-Cookies");
    
    $realCookies = [
        "PHPSESSID=tujfce9g5sesps0hdod1o3dqio",
        "Ladeplatz_Usage[]=1",
        "Ladeplatz_Usage[0]=0",
        "Autohaus_Usage[453]=14",
        "Ladeplatz_Usage[1]=1",
        "Ladeplatz_LastChange=1758873482",
        "Autohaus_LastChange=1758873482",
        "HideWheelsetSearchResultSTATUS_ID=1",
        "HideWheelsetSearchResultRFFEHL_FLG=1",
        "HideWheelsetSearchResultFLGFEHL_FLG=1",
        "HideWheelsetSearchResultRADKAPFEHL_FLG=1",
        "HideWheelsetSearchResultMINPROFIL=0",
        "HideWheelsetSearchResultPRUFDOC_ID=1",
        "SortConfirmTableCol=2",
        "SortConfirmTableDesc=1",
        "SortWheelsetSearchResultCol=11",
        "SortWheelsetSearchResultDesc=-1",
        "HideWheelsetSearchResultBEZEICHNUNG=0",
        "HideWheelsetSearchResultKDNR=0",
        "HideWheelsetSearchResultORT=0",
        "HideWheelsetSearchResultPLZ=0",
        "HideWheelsetSearchResultSTRASSE=0",
        "HideWheelsetSearchResultTELEFON=0",
        "HideWheelsetSearchResultEMAIL=0",
        "HideWheelsetSearchResultLGS_ID=0"
    ];
    
    $cookieString = implode("; ", $realCookies);
    debugLog("Verwende aktuelle Session-Cookies für Image Proxy (PHPSESSID: tujfce9g5sesps0hdod1o3dqio)");
    
    // Teste verschiedene Bild-URL-Formate
    $possibleUrls = [
        $autohaus['resy_base_url'] . '/resyweb/index.php?m=wheelphoto&a=show&LGFOTO_ID=' . $lgfotoId . '&FN=ShowPruefDoc',
        $autohaus['resy_base_url'] . '/resyweb/index.php?FN=ShowPruefDoc&a=show&m=wheelphoto&LGFOTO_ID=' . $lgfotoId,
        $autohaus['resy_base_url'] . '/index.php?m=wheelphoto&a=show&LGFOTO_ID=' . $lgfotoId . '&FN=ShowPruefDoc',
        $autohaus['resy_base_url'] . '/wheelphoto/show/' . $lgfotoId,
        $autohaus['resy_base_url'] . '/images/wheelphoto/' . $lgfotoId . '.jpg',
        $autohaus['resy_base_url'] . '/uploads/wheelphoto/' . $lgfotoId . '.jpg'
    ];
    
    debugLog("Teste " . count($possibleUrls) . " mögliche Bild-URLs");
    
    foreach ($possibleUrls as $index => $imageUrl) {
        debugLog("Teste URL " . ($index + 1) . ": " . substr($imageUrl, 0, 80) . "...");
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $imageUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_COOKIE, $cookieString);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: image/webp,image/apng,image/*,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br, zstd',
            'Referer: ' . $autohaus['resy_base_url'] . '/resyweb/index.php?m=wheelset&a=search'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $imageResponse = curl_exec($ch);
        $imageHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $imageContentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        curl_close($ch);
        
        debugLog("URL " . ($index + 1) . ": HTTP $imageHttpCode, Content-Type: $imageContentType, Length: " . strlen($imageResponse));
        
        // Dekodiere gzip falls nötig
        if (substr($imageResponse, 0, 2) === "\x1f\x8b") {
            $imageResponse = gzdecode($imageResponse);
            debugLog("Gzip-Dekodierung durchgeführt für URL " . ($index + 1));
        }
        
        // Prüfe ob es ein Bild ist
        $isImage = strpos($imageContentType, 'image/') === 0 &&
                   strpos($imageResponse, '<html') === false &&
                   strpos($imageResponse, '<!DOCTYPE') === false;
        
        if ($isImage) {
            debugLog("🎉 BILD GEFUNDEN mit URL " . ($index + 1) . "! 🎉");
            debugLog("Content-Type: $imageContentType, Größe: " . strlen($imageResponse) . " Bytes");
            
            // Speichere das Bild im Cache
            $imageCache->cacheImage($lgfotoId, $imageResponse, $imageContentType);
            
            // Setze korrekten Content-Type
            header('Content-Type: ' . $imageContentType);
            header('Content-Length: ' . strlen($imageResponse));
            header('Cache-Control: max-age=86400'); // 24 Stunden Cache
            
            echo $imageResponse;
            exit;
        } else {
            debugLog("URL " . ($index + 1) . ": Kein Bild - wahrscheinlich HTML (Login-Seite)");
            
            if (strpos($imageResponse, 'login') !== false || strpos($imageResponse, 'Login') !== false) {
                debugLog("⚠️ Login-Seite erkannt für URL " . ($index + 1) . " - Session abgelaufen");
            }
        }
    }
    
    // Kein Bild gefunden - generiere Fehlerbild
    debugLog("❌ Kein Bild gefunden für LGFOTO_ID $lgfotoId");
    
    // Generiere SVG-Fehlermeldung
    $errorSvg = '<?xml version="1.0" encoding="UTF-8"?>
<svg width="200" height="150" xmlns="http://www.w3.org/2000/svg">
  <rect width="200" height="150" fill="#f0f0f0" stroke="#ccc" stroke-width="1"/>
  <text x="100" y="50" text-anchor="middle" font-family="Arial, sans-serif" font-size="14" fill="#666">
    Bild nicht verfügbar
  </text>
  <text x="100" y="70" text-anchor="middle" font-family="Arial, sans-serif" font-size="12" fill="#999">
    LGFOTO_ID: ' . htmlspecialchars($lgfotoId) . '
  </text>
  <text x="100" y="90" text-anchor="middle" font-family="Arial, sans-serif" font-size="10" fill="#999">
    Session abgelaufen
  </text>
  <text x="100" y="110" text-anchor="middle" font-family="Arial, sans-serif" font-size="10" fill="#999">
    ' . date('Y-m-d H:i:s') . '
  </text>
</svg>';
    
    header('Content-Type: image/svg+xml');
    header('Content-Length: ' . strlen($errorSvg));
    header('Cache-Control: no-cache, no-store, must-revalidate');
    
    echo $errorSvg;
    
} catch (Exception $e) {
    debugLog("❌ Fehler in Image Proxy: " . $e->getMessage());
    
    // Generiere SVG-Fehlermeldung
    $errorSvg = '<?xml version="1.0" encoding="UTF-8"?>
<svg width="200" height="150" xmlns="http://www.w3.org/2000/svg">
  <rect width="200" height="150" fill="#ffe6e6" stroke="#ff0000" stroke-width="1"/>
  <text x="100" y="50" text-anchor="middle" font-family="Arial, sans-serif" font-size="14" fill="#cc0000">
    Fehler beim Laden
  </text>
  <text x="100" y="70" text-anchor="middle" font-family="Arial, sans-serif" font-size="12" fill="#999">
    ' . htmlspecialchars($e->getMessage()) . '
  </text>
  <text x="100" y="90" text-anchor="middle" font-family="Arial, sans-serif" font-size="10" fill="#999">
    ' . date('Y-m-d H:i:s') . '
  </text>
</svg>';
    
    header('Content-Type: image/svg+xml');
    header('Content-Length: ' . strlen($errorSvg));
    header('Cache-Control: no-cache, no-store, must-revalidate');
    
    echo $errorSvg;
}
?>