<?php
/**
 * RESY Data Extraction API - Echte HTML-Parsing Version
 * Extrahiert Daten direkt aus dem RESY-HTML basierend auf LGS_ID
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'cookies' => [],
        'follow_redirects' => true,
        'timeout' => 30
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    if (!empty($options['cookies'])) {
        $cookieString = '';
        foreach ($options['cookies'] as $name => $value) {
            $cookieString .= $name . '=' . $value . '; ';
        }
        curl_setopt($ch, CURLOPT_COOKIE, rtrim($cookieString, '; '));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// HTML-Parsing-Funktionen
function extractFormData($html) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    debugLog("Suche nach Login-Formular...");
    
    // Suche nach Login-Formular
    $forms = $xpath->query('//form');
    debugLog("Gefundene Formulare: " . $forms->length);
    
    $form = null;
    foreach ($forms as $f) {
        $action = $f->getAttribute('action');
        debugLog("Formular-Action: '$action'");
        
        // Suche nach Login-Formular (enthält UserName und Password Felder)
        $userNameInput = $xpath->query('.//input[@name="UserName"]', $f);
        $passwordInput = $xpath->query('.//input[@name="Password"]', $f);
        
        if ($userNameInput->length > 0 && $passwordInput->length > 0) {
            debugLog("Login-Formular gefunden mit Action: '$action'");
            $form = $f;
            break;
        }
    }
    
    if (!$form) {
        debugLog("FEHLER: Kein Login-Formular gefunden");
        debugLog("HTML Preview: " . substr($html, 0, 1000));
        throw new Exception("Login-Formular nicht gefunden");
    }
    
    $formData = [];
    $action = $form->getAttribute('action');
    debugLog("Verwende Formular-Action: '$action'");
    
    // Wenn Action leer ist, verwende die aktuelle Seite
    if (empty($action)) {
        $action = 'index.php?m=login&a=login';
        debugLog("Action war leer, verwende Standard: '$action'");
    }
    
    $formData['action'] = $action;
    
    debugLog("Extrahiere Input-Felder...");
    $inputs = $xpath->query('.//input[@name]', $form);
    debugLog("Gefundene Input-Felder: " . $inputs->length);
    
    foreach ($inputs as $input) {
        $name = $input->getAttribute('name');
        $value = $input->getAttribute('value');
        $type = $input->getAttribute('type');
        debugLog("Input: name='$name', type='$type', value='$value'");
        $formData['data'][$name] = $value;
    }
    
    debugLog("Formular-Daten extrahiert: " . json_encode($formData));
    return $formData;
}

function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    // Mapping der Feldnamen zu HTML-IDs
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'wheelsRUECK.LADESTELLE',
        'bemerkung' => 'WheelKDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    // Suche nach dem Element mit der entsprechenden ID
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        debugLog("Element mit ID '$htmlId' nicht gefunden für Feld '$field'");
        return '';
    }
    
    // Extrahiere den Textinhalt
    $value = trim($element->textContent);
    
    // Für Bemerkung-Felder: Suche nach textarea
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
        }
    }
    
    debugLog("Feld '$field' (ID: $htmlId) extrahiert: '$value'");
    return $value;
}

// Echter RESY-Login
function resyLoginReal($username, $password, $baseUrl) {
    debugLog("=== ECHTER RESY LOGIN START ===");
    debugLog("Username: $username");
    debugLog("Base URL: $baseUrl");
    
    try {
        // Schritt 1: Login-Seite laden
        $loginUrl = $baseUrl . '/index.php?m=login&a=login';
        debugLog("Lade Login-Seite: $loginUrl");
        
        $response = httpRequest($loginUrl);
        debugLog("Login-Seite Response Code: " . $response['http_code']);
        
        if ($response['http_code'] !== 200) {
            throw new Exception("Login-Seite konnte nicht geladen werden. HTTP Code: " . $response['http_code']);
        }
        
        // Schritt 2: Formular-Daten extrahieren
        debugLog("Extrahiere Formular-Daten...");
        $formData = extractFormData($response['body']);
        
        // Schritt 3: Login-Daten setzen
        $formData['data']['UserName'] = $username;
        $formData['data']['Password'] = $password;
        $formData['data']['button'] = 'submit';
        
        debugLog("Login-Daten gesetzt, sende Login-Request");
        
        // Schritt 4: Login-Request senden
        $loginAction = $baseUrl . '/' . $formData['action'];
        $loginResponse = httpRequest($loginAction, [
            'method' => 'POST',
            'data' => http_build_query($formData['data']),
            'headers' => [
                'Content-Type: application/x-www-form-urlencoded',
                'Referer: ' . $loginUrl
            ]
        ]);
        
        debugLog("Login-Response Code: " . $loginResponse['http_code']);
        
        // Schritt 5: Erfolg prüfen
        if (strpos($loginResponse['body'], 'Anmeldename') !== false) {
            throw new Exception("Login fehlgeschlagen - Anmeldeformular noch sichtbar");
        }
        
        debugLog("Login erfolgreich!");
        
        return [
            'success' => true,
            'sessionData' => [
                'baseUrl' => $baseUrl,
                'loggedIn' => true,
                'timestamp' => time(),
                'username' => $username,
                'cookies' => [] // Hier könnten Cookies gespeichert werden
            ]
        ];
        
    } catch (Exception $e) {
        debugLog("Login-Fehler: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

// Navigation zur Suchseite
function navigateToSearch($sessionData) {
    debugLog("Navigation zur Suchseite...");
    
    $searchUrl = $sessionData['baseUrl'] . '/index.php?m=wheelset&a=search';
    $response = httpRequest($searchUrl);
    
    if ($response['http_code'] !== 200) {
        throw new Exception("Suchseite konnte nicht geladen werden");
    }
    
    debugLog("Suchseite erfolgreich geladen");
    return [
        'success' => true,
        'message' => 'Navigation zur Suchseite erfolgreich',
        'sessionData' => $sessionData
    ];
}

// Suche nach LGS_ID
function searchLgsId($lgsId, $sessionData) {
    debugLog("Suche nach LGS_ID: $lgsId");
    
    $searchUrl = $sessionData['baseUrl'] . '/index.php?m=wheelset&a=search';
    
    // Lade Suchseite
    $response = httpRequest($searchUrl);
    if ($response['http_code'] !== 200) {
        throw new Exception("Suchseite konnte nicht geladen werden");
    }
    
    // Extrahiere Formular-Daten
    $formData = extractFormData($response['body']);
    
    // Setze LGS_ID
    $formData['data']['LGS_ID'] = $lgsId;
    $formData['data']['button'] = 'submit';
    
    // Sende Such-Request
    $searchAction = $sessionData['baseUrl'] . '/' . $formData['action'];
    $searchResponse = httpRequest($searchAction, [
        'method' => 'POST',
        'data' => http_build_query($formData['data']),
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'Referer: ' . $searchUrl
        ]
    ]);
    
    debugLog("Suche abgeschlossen, Response Code: " . $searchResponse['http_code']);
    
    return [
        'success' => true,
        'message' => "Suche nach LGS_ID $lgsId erfolgreich",
        'sessionData' => $sessionData,
        'searchHtml' => $searchResponse['body']
    ];
}

// Navigation zur Detailseite
function navigateToDetail($lgsId, $sessionData) {
    debugLog("Navigation zur Detailseite für LGS_ID: $lgsId");
    
    $detailUrl = $sessionData['baseUrl'] . "/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult";
    $response = httpRequest($detailUrl);
    
    if ($response['http_code'] !== 200) {
        throw new Exception("Detailseite konnte nicht geladen werden");
    }
    
    debugLog("Detailseite erfolgreich geladen");
    
    return [
        'success' => true,
        'message' => "Navigation zur Detailseite für LGS_ID $lgsId erfolgreich",
        'sessionData' => $sessionData,
        'detailHtml' => $response['body']
    ];
}

// Extrahiere Feld aus HTML
function extractFieldFromHtml($field, $html, $sessionData) {
    debugLog("Extrahiere Feld '$field' aus HTML...");
    
    try {
        $value = extractDataFromHtml($html, $field);
        
        return [
            'success' => true,
            'value' => $value,
            'field' => $field,
            'message' => "Feld '$field' erfolgreich extrahiert"
        ];
        
    } catch (Exception $e) {
        debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'field' => $field
        ];
    }
}

// Hauptverarbeitung
try {
    debugLog("=== ECHTE HTML RESY API Request Started ===");
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'login':
            $appointmentId = $input['appointment_id'] ?? null;
            debugLog("Echter Login für Appointment ID: $appointmentId");
            
            if (!$appointmentId) {
                throw new Exception('Appointment ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_username, a.resy_password, a.resy_base_url 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            $result = resyLoginReal($autohaus['resy_username'], $autohaus['resy_password'], $autohaus['resy_base_url']);
            echo json_encode($result);
            break;
            
        case 'navigate_to_search':
            debugLog("Navigate to search");
            $sessionData = $input['sessionData'] ?? [];
            $result = navigateToSearch($sessionData);
            echo json_encode($result);
            break;
            
        case 'search_lgs_id':
            $lgsId = $input['lgsId'] ?? null;
            $sessionData = $input['sessionData'] ?? [];
            debugLog("Search LGS ID: $lgsId");
            
            if (!$lgsId) {
                throw new Exception('LGS_ID fehlt');
            }
            
            $result = searchLgsId($lgsId, $sessionData);
            echo json_encode($result);
            break;
            
        case 'navigate_to_detail':
            $lgsId = $input['lgsId'] ?? null;
            $sessionData = $input['sessionData'] ?? [];
            debugLog("Navigate to detail for LGS_ID: $lgsId");
            
            if (!$lgsId) {
                throw new Exception('LGS_ID fehlt');
            }
            
            $result = navigateToDetail($lgsId, $sessionData);
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            $sessionData = $input['sessionData'] ?? [];
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            $result = extractFieldFromHtml($field, $html, $sessionData);
            echo json_encode($result);
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== ECHTE HTML RESY API Request Ended ===");
?>
