<?php
/**
 * RESY Data Extraction API - Hybrid Final Version
 * Verwendet den erfolgreichen Prozess aus debug_search.php mit der Extraktion aus final.php
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'timeout' => 30,
        'cookies' => null,
        'follow_redirects' => true,
        'cookie_file' => sys_get_temp_dir() . '/resy_cookies.txt'
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    curl_setopt($ch, CURLOPT_COOKIEJAR, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $options['cookie_file']);
    
    if ($options['cookies']) {
        curl_setopt($ch, CURLOPT_COOKIE, $options['cookies']);
    }
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// RESY Login (exakt wie in debug_search.php)
function performResyLogin($baseUrl, $username, $password) {
    debugLog("=== HYBRID FINAL LOGIN START ===");
    debugLog("Base URL: $baseUrl");
    debugLog("Username: $username");
    
    // Cookie-Datei löschen für sauberen Start
    $cookieFile = sys_get_temp_dir() . '/resy_cookies.txt';
    if (file_exists($cookieFile)) {
        unlink($cookieFile);
    }
    
    // Login-Seite laden
    $loginUrl = "$baseUrl/index.php?m=login&a=login";
    debugLog("Lade Login-Seite: $loginUrl");
    
    $response = httpRequest($loginUrl, [
        'cookie_file' => $cookieFile,
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1'
        ]
    ]);
    
    if ($response['http_code'] !== 200) {
        throw new Exception("Login-Seite nicht erreichbar: HTTP " . $response['http_code']);
    }
    
    debugLog("Login-Seite erfolgreich geladen");
    
    // Login-Daten senden
    $postData = [
        'FN' => 'login',
        'UserName' => $username,
        'Password' => $password,
        'button' => 'submit'
    ];
    
    $loginDataString = http_build_query($postData);
    debugLog("Login-Daten: " . str_replace($password, '***', $loginDataString));
    
    debugLog("Sende Login-Daten...");
    
    $loginResponse = httpRequest($loginUrl, [
        'method' => 'POST',
        'data' => $loginDataString,
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $loginUrl
        ],
        'cookie_file' => $cookieFile
    ]);
    
    debugLog("Login-Response Code: " . $loginResponse['http_code']);
    
    // Login-Erfolg prüfen
    $responseBody = $loginResponse['body'];
    
    if (strpos($responseBody, 'Anmeldename') === false && 
        strpos($responseBody, 'Kennwort') === false &&
        strpos($responseBody, 'Anmeldung') === false) {
        debugLog("✅ Login erfolgreich!");
        debugLog("=== HYBRID FINAL LOGIN END ===");
        return $loginResponse;
    }
    
    throw new Exception("Login fehlgeschlagen - alle Feldkombinationen getestet");
}

// Hybrid-Suche: Verwende den erfolgreichen Prozess aus debug_search.php
function hybridSearch($baseUrl, $searchTerm) {
    debugLog("=== HYBRID SEARCH START ===");
    debugLog("Search Term: '$searchTerm'");
    
    // Verwende die erfolgreiche Suchstrategie aus debug_search.php
    $searchUrl = $baseUrl . '/index.php?m=wheelset&a=search&LGS_ID=' . urlencode($searchTerm) . '&button=submit&FN=WheelsetSearch';
    debugLog("Search URL: $searchUrl");
    
    $response = httpRequest($searchUrl, [
        'method' => 'GET',
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $baseUrl . '/index.php?m=wheelset&a=search'
        ],
        'cookie_file' => sys_get_temp_dir() . '/resy_cookies.txt'
    ]);
    
    debugLog("Search Response Code: " . $response['http_code']);
    debugLog("Search Response Length: " . strlen($response['body']));
    
    // Analysiere die Antwort (wie in debug_search.php)
    $analysis = analyzeSearchResponse($response['body'], 'Hybrid Search');
    
    debugLog("Analysis: " . json_encode($analysis));
    
    // Prüfe ob LGS_IDs gefunden wurden
    if (!empty($analysis['found_lgs_ids']) && is_array($analysis['found_lgs_ids'])) {
        $lgsId = $analysis['found_lgs_ids'][0]; // Verwende die erste gefundene LGS_ID
        debugLog("✅ LGS_ID gefunden: $lgsId");
        debugLog("=== HYBRID SEARCH END ===");
        return ['success' => true, 'lgs_id' => $lgsId, 'search_response' => $response['body']];
    }
    
    throw new Exception("Keine LGS_ID in Antwort gefunden für Search Term: $searchTerm");
}

// Analysiere Suchantwort (exakt wie in debug_search.php)
function analyzeSearchResponse($html, $strategyName) {
    debugLog("Analysiere Antwort für Strategie: $strategyName");
    
    $analysis = [
        'contains_login_form' => false,
        'contains_search_form' => false,
        'contains_results_table' => false,
        'contains_error_message' => false,
        'found_lgs_ids' => [],
        'found_links' => [],
        'error_messages' => [],
        'preview' => ''
    ];
    
    // Prüfe auf Login-Formular
    if (strpos($html, 'UserName') !== false || strpos($html, 'Password') !== false) {
        $analysis['contains_login_form'] = true;
        debugLog("⚠️ Login-Formular erkannt");
    }
    
    // Prüfe auf Suchformular
    if (strpos($html, 'WheelLGS_ID') !== false || strpos($html, 'LGS_ID') !== false) {
        $analysis['contains_search_form'] = true;
        debugLog("✅ Suchformular erkannt");
    }
    
    // Prüfe auf Ergebnisse-Tabelle
    if (strpos($html, 'ResultTable') !== false || strpos($html, 'table') !== false) {
        $analysis['contains_results_table'] = true;
        debugLog("✅ Ergebnisse-Tabelle erkannt");
    }
    
    // Prüfe auf Fehlermeldungen
    $errorPatterns = [
        'Keine Ergebnisse gefunden',
        'Nicht gefunden',
        'keine Treffer',
        'Keine Treffer',
        'Keine Sätze gefunden',
        'keine Sätze gefunden',
        'Fehler',
        'Error'
    ];
    
    foreach ($errorPatterns as $pattern) {
        if (strpos($html, $pattern) !== false) {
            $analysis['contains_error_message'] = true;
            $analysis['error_messages'][] = $pattern;
            debugLog("⚠️ Fehlermeldung erkannt: $pattern");
        }
    }
    
    // Suche nach LGS_IDs in der Antwort
    if (preg_match_all('/LGS_ID=(\d+)/', $html, $matches)) {
        $analysis['found_lgs_ids'] = array_unique($matches[1]);
        debugLog("✅ LGS_IDs gefunden: " . implode(', ', $analysis['found_lgs_ids']));
    }
    
    // Suche nach Links
    if (preg_match_all('/href="([^"]*LGS_ID[^"]*)"/', $html, $matches)) {
        $analysis['found_links'] = array_unique($matches[1]);
        debugLog("✅ Links gefunden: " . count($analysis['found_links']));
    }
    
    // Erstelle Preview
    $analysis['preview'] = substr($html, 0, 500);
    
    return $analysis;
}

// Zur Detailseite navigieren
function navigateToDetailPage($baseUrl, $lgsId) {
    debugLog("=== HYBRID NAVIGATE TO DETAIL PAGE START ===");
    debugLog("LGS_ID: $lgsId");
    
    // Verwende die exakte URL-Struktur
    $detailUrl = "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult";
    debugLog("Detail URL: $detailUrl");
    
    $response = httpRequest($detailUrl, [
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $baseUrl . '/index.php?m=wheelset&a=search'
        ],
        'cookie_file' => sys_get_temp_dir() . '/resy_cookies.txt'
    ]);
    
    debugLog("Detail Response Code: " . $response['http_code']);
    debugLog("Detail Response Length: " . strlen($response['body']));
    
    if ($response['http_code'] === 200) {
        // Prüfe ob es eine Detailseite ist
        if (strpos($response['body'], 'WheelHALTERNAME') !== false) {
            debugLog("✅ Detailseite gefunden!");
            debugLog("=== HYBRID NAVIGATE TO DETAIL PAGE END ===");
            return $response;
        }
        
        // Prüfe ob es eine Login-Seite ist
        if (strpos($response['body'], 'Anmeldename') !== false) {
            debugLog("⚠️ Login-Seite erkannt");
        }
        
        // Prüfe ob es eine Fehlerseite ist
        if (strpos($response['body'], 'Fehler') !== false || strpos($response['body'], 'Error') !== false) {
            debugLog("⚠️ Fehlerseite erkannt");
        }
        
        debugLog("Response Preview: " . substr($response['body'], 0, 500));
    }
    
    throw new Exception("Keine Detailseite für LGS_ID $lgsId gefunden");
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    debugLog("=== HTML-ANALYSE FÜR FELD: $field ===");
    
    // Mapping der Feldnamen zu HTML-IDs
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID', // Autohaus ist die Ladestelle
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    debugLog("Suche nach Element mit ID: $htmlId");
    
    // Suche nach dem Element mit der entsprechenden ID
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        debugLog("Element mit ID '$htmlId' nicht gefunden für Feld '$field'");
        return '';
    }
    
    // Extrahiere den Textinhalt
    $value = trim($element->textContent);
    
    // Für Bemerkung-Felder: Suche nach textarea
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
            debugLog("Bemerkung aus textarea extrahiert: '$value'");
        }
    }
    
    debugLog("Feld '$field' (ID: $htmlId) extrahiert: '$value'");
    
    // Wenn Wert leer ist, suche nach <b> Tags innerhalb des Elements
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
            debugLog("Wert aus <b> Tag extrahiert: '$value'");
        }
    }
    
    debugLog("=== ENDE HTML-ANALYSE FÜR FELD: $field ===");
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== HYBRID FINAL RESY API Request Started ===");
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'hybrid_final':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Hybrid Final für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url, a.resy_username, a.resy_password 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            // Schritt 1: Login durchführen
            performResyLogin($autohaus['resy_base_url'], $autohaus['resy_username'], $autohaus['resy_password']);
            
            // Schritt 2: Hybrid-Suche durchführen (verwendet erfolgreichen Prozess aus debug_search.php)
            $searchResult = hybridSearch($autohaus['resy_base_url'], $lgsId);
            
            // Schritt 3: Zur Detailseite navigieren
            $detailResponse = navigateToDetailPage($autohaus['resy_base_url'], $searchResult['lgs_id']);
            
            $result = [
                'success' => true,
                'detailHtml' => $detailResponse['body'],
                'url' => $detailResponse['url'],
                'approach' => 'hybrid_final',
                'message' => 'Erfolgreich eingeloggt, gesucht und zur Detailseite navigiert',
                'found_lgs_id' => $searchResult['lgs_id'],
                'search_response' => $searchResult['search_response']
            ];
            
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== HYBRID FINAL RESY API Request Ended ===");
?>
