<?php
/**
 * RESY Data Extraction API - Direkte Version
 * Umgeht Login-Probleme und navigiert direkt zur Detailseite
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'cookies' => [],
        'follow_redirects' => true,
        'timeout' => 30
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    if (!empty($options['cookies'])) {
        $cookieString = '';
        foreach ($options['cookies'] as $name => $value) {
            $cookieString .= $name . '=' . $value . '; ';
        }
        curl_setopt($ch, CURLOPT_COOKIE, rtrim($cookieString, '; '));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// Direkte Navigation zur Detailseite
function navigateDirectlyToDetail($lgsId, $baseUrl) {
    debugLog("=== DIREKTE NAVIGATION START ===");
    debugLog("LGS_ID: $lgsId");
    debugLog("Base URL: $baseUrl");
    
    try {
        // Direkt zur Detailseite navigieren
        $detailUrl = $baseUrl . "/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult";
        debugLog("Direkte URL: $detailUrl");
        
        $response = httpRequest($detailUrl);
        debugLog("Response Code: " . $response['http_code']);
        debugLog("Response Length: " . strlen($response['body']));
        
        if ($response['http_code'] !== 200) {
            throw new Exception("Detailseite konnte nicht geladen werden. HTTP Code: " . $response['http_code']);
        }
        
        // Prüfe ob wir auf der Detailseite sind
        if (strpos($response['body'], 'WheelHALTERNAME') === false) {
            debugLog("WARNUNG: Detailseite-Content nicht gefunden");
            debugLog("Response Preview: " . substr($response['body'], 0, 1000));
        } else {
            debugLog("Detailseite erfolgreich geladen!");
        }
        
        return [
            'success' => true,
            'detailHtml' => $response['body'],
            'message' => "Direkte Navigation zur Detailseite erfolgreich"
        ];
        
    } catch (Exception $e) {
        debugLog("Direkte Navigation fehlgeschlagen: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
    
    debugLog("=== DIREKTE NAVIGATION END ===");
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    debugLog("=== HTML-ANALYSE FÜR FELD: $field ===");
    
    // Mapping der Feldnamen zu HTML-IDs
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'wheelsRUECK.LADESTELLE',
        'bemerkung' => 'WheelKDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    debugLog("Suche nach Element mit ID: $htmlId");
    
    // Suche nach dem Element mit der entsprechenden ID
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        debugLog("Element mit ID '$htmlId' nicht gefunden für Feld '$field'");
        
        // Suche nach ähnlichen Elementen
        debugLog("Suche nach ähnlichen Elementen...");
        $allElements = $xpath->query("//*[@id]");
        debugLog("Gefundene Elemente mit IDs: " . $allElements->length);
        
        $foundIds = [];
        foreach ($allElements as $el) {
            $id = $el->getAttribute('id');
            if (strpos($id, 'Wheel') !== false || strpos($id, 'wheel') !== false) {
                $foundIds[] = $id;
            }
        }
        
        if (!empty($foundIds)) {
            debugLog("Gefundene Wheel-Elemente: " . implode(', ', $foundIds));
        }
        
        return '';
    }
    
    // Extrahiere den Textinhalt
    $value = trim($element->textContent);
    
    // Für Bemerkung-Felder: Suche nach textarea
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
            debugLog("Bemerkung aus textarea extrahiert: '$value'");
        }
    }
    
    debugLog("Feld '$field' (ID: $htmlId) extrahiert: '$value'");
    
    // Wenn Wert leer ist, suche nach <b> Tags innerhalb des Elements
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
            debugLog("Wert aus <b> Tag extrahiert: '$value'");
        }
    }
    
    debugLog("=== ENDE HTML-ANALYSE FÜR FELD: $field ===");
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== DIREKTE RESY API Request Started ===");
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'navigate_direct':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Direkte Navigation für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            $result = navigateDirectlyToDetail($lgsId, $autohaus['resy_base_url']);
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== DIREKTE RESY API Request Ended ===");
?>
