<?php
/**
 * RESY Data Extraction API
 * Extrahiert Daten aus dem RESY-System Schritt für Schritt
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Error reporting für Debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Session starten
session_start();

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'cookies' => [],
        'follow_redirects' => true,
        'timeout' => 30
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    if (!empty($options['cookies'])) {
        $cookieString = '';
        foreach ($options['cookies'] as $name => $value) {
            $cookieString .= $name . '=' . $value . '; ';
        }
        curl_setopt($ch, CURLOPT_COOKIE, rtrim($cookieString, '; '));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// HTML-Parsing-Helper-Funktionen
function extractFormData($html, $formAction = null) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $forms = $xpath->query('//form');
    foreach ($forms as $form) {
        if ($formAction && $form->getAttribute('action') !== $formAction) {
            continue;
        }
        
        $formData = [];
        $inputs = $xpath->query('.//input', $form);
        foreach ($inputs as $input) {
            $name = $input->getAttribute('name');
            $value = $input->getAttribute('value');
            $type = $input->getAttribute('type');
            
            if ($name && $type !== 'submit') {
                $formData[$name] = $value;
            }
        }
        
        $selects = $xpath->query('.//select', $form);
        foreach ($selects as $select) {
            $name = $select->getAttribute('name');
            $options = $xpath->query('.//option[@selected]', $select);
            if ($options->length > 0) {
                $formData[$name] = $options->item(0)->getAttribute('value');
            }
        }
        
        return [
            'action' => $form->getAttribute('action'),
            'method' => $form->getAttribute('method') ?: 'POST',
            'data' => $formData
        ];
    }
    
    return null;
}

function extractTextContent($html, $selector) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $elements = $xpath->query($selector);
    if ($elements->length > 0) {
        return trim($elements->item(0)->textContent);
    }
    
    return null;
}

function extractAllTextContent($html, $selector) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $elements = $xpath->query($selector);
    $results = [];
    
    foreach ($elements as $element) {
        $results[] = trim($element->textContent);
    }
    
    return $results;
}

// Hauptfunktionen für RESY-Operationen
function resyLogin($username, $password, $baseUrl) {
    debugLog("Starte RESY Login für Benutzer: $username");
    
    try {
        // Schritt 1: Login-Seite laden
        $loginUrl = $baseUrl . '/index.php?m=login&a=login';
        $response = httpRequest($loginUrl);
        
        if ($response['http_code'] !== 200) {
            throw new Exception("Login-Seite konnte nicht geladen werden. HTTP Code: " . $response['http_code']);
        }
        
        debugLog("Login-Seite erfolgreich geladen");
        
        // Schritt 2: Formular-Daten extrahieren
        $formData = extractFormData($response['body']);
        if (!$formData) {
            throw new Exception("Login-Formular nicht gefunden");
        }
        
        // Schritt 3: Login-Daten setzen
        $formData['data']['UserName'] = $username;
        $formData['data']['Password'] = $password;
        $formData['data']['button'] = 'submit';
        
        debugLog("Login-Daten gesetzt, sende Login-Request");
        
        // Schritt 4: Login-Request senden
        $loginAction = $baseUrl . '/' . $formData['action'];
        $loginResponse = httpRequest($loginAction, [
            'method' => 'POST',
            'data' => http_build_query($formData['data']),
            'headers' => [
                'Content-Type: application/x-www-form-urlencoded',
                'Referer: ' . $loginUrl
            ]
        ]);
        
        debugLog("Login-Response erhalten. HTTP Code: " . $loginResponse['http_code']);
        
        // Schritt 5: Erfolg prüfen
        if (strpos($loginResponse['body'], 'Anmeldename') !== false) {
            throw new Exception("Login fehlgeschlagen - Anmeldeformular noch sichtbar");
        }
        
        debugLog("Login erfolgreich!");
        
        return [
            'success' => true,
            'sessionData' => [
                'baseUrl' => $baseUrl,
                'loggedIn' => true,
                'timestamp' => time()
            ]
        ];
        
    } catch (Exception $e) {
        debugLog("Login-Fehler: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function resyNavigateToSearch($sessionData) {
    debugLog("Navigiere zu Sätze/Rücklieferungen bestellen");
    
    try {
        $baseUrl = $sessionData['baseUrl'];
        $searchUrl = $baseUrl . '/index.php?m=wheelset&a=search';
        
        $response = httpRequest($searchUrl);
        
        if ($response['http_code'] !== 200) {
            throw new Exception("Suchseite konnte nicht geladen werden. HTTP Code: " . $response['http_code']);
        }
        
        debugLog("Suchseite erfolgreich geladen");
        
        return [
            'success' => true,
            'sessionData' => $sessionData,
            'searchPageHtml' => $response['body']
        ];
        
    } catch (Exception $e) {
        debugLog("Navigation-Fehler: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function resySearchLgsId($lgsId, $sessionData) {
    debugLog("Suche nach LGS_ID: $lgsId");
    
    try {
        $baseUrl = $sessionData['baseUrl'];
        $searchUrl = $baseUrl . '/index.php?m=wheelset&a=search';
        
        // Suchformular laden
        $response = httpRequest($searchUrl);
        if ($response['http_code'] !== 200) {
            throw new Exception("Suchseite konnte nicht geladen werden");
        }
        
        // Formular-Daten extrahieren
        $formData = extractFormData($response['body']);
        if (!$formData) {
            throw new Exception("Suchformular nicht gefunden");
        }
        
        // LGS_ID in das Formular eintragen
        $formData['data']['LGS_ID'] = $lgsId;
        $formData['data']['button'] = 'submit';
        
        debugLog("Sende Suchanfrage für LGS_ID: $lgsId");
        
        // Suchanfrage senden
        $searchAction = $baseUrl . '/' . $formData['action'];
        $searchResponse = httpRequest($searchAction, [
            'method' => 'POST',
            'data' => http_build_query($formData['data']),
            'headers' => [
                'Content-Type: application/x-www-form-urlencoded',
                'Referer: ' . $searchUrl
            ]
        ]);
        
        debugLog("Such-Response erhalten. HTTP Code: " . $searchResponse['http_code']);
        
        // Prüfen ob Ergebnisse gefunden wurden
        if (strpos($searchResponse['body'], $lgsId) === false) {
            throw new Exception("Keine Ergebnisse für LGS_ID $lgsId gefunden");
        }
        
        debugLog("LGS_ID $lgsId erfolgreich gefunden");
        
        return [
            'success' => true,
            'sessionData' => $sessionData,
            'searchResultsHtml' => $searchResponse['body']
        ];
        
    } catch (Exception $e) {
        debugLog("Suche-Fehler: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function resyNavigateToDetail($lgsId, $sessionData) {
    debugLog("Navigiere zur Detailseite für LGS_ID: $lgsId");
    
    try {
        $baseUrl = $sessionData['baseUrl'];
        $detailUrl = $baseUrl . "/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult";
        
        $response = httpRequest($detailUrl);
        
        if ($response['http_code'] !== 200) {
            throw new Exception("Detailseite konnte nicht geladen werden. HTTP Code: " . $response['http_code']);
        }
        
        debugLog("Detailseite erfolgreich geladen");
        
        return [
            'success' => true,
            'sessionData' => $sessionData,
            'detailPageHtml' => $response['body']
        ];
        
    } catch (Exception $e) {
        debugLog("Detail-Navigation-Fehler: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function extractFieldData($field, $html, $sessionData) {
    debugLog("Extrahiere Feld: $field");
    
    try {
        $value = null;
        
        switch ($field) {
            case 'fahrzeughalter':
                $value = extractTextContent($html, '//td[contains(text(), "Fahrzeughalter")]/following-sibling::td');
                break;
                
            case 'kennzeichen':
                $value = extractTextContent($html, '//td[contains(text(), "Kennzeichen")]/following-sibling::td');
                break;
                
            case 'fin':
                $value = extractTextContent($html, '//td[contains(text(), "FIN")]/following-sibling::td');
                break;
                
            case 'eingangsnummer':
                $value = extractTextContent($html, '//td[contains(text(), "Eingangsnummer")]/following-sibling::td');
                break;
                
            case 'ladestelle':
                $value = extractTextContent($html, '//td[contains(text(), "Ladestelle")]/following-sibling::td');
                break;
                
            case 'bemerkung':
                $value = extractTextContent($html, '//td[contains(text(), "Bemerkung")]/following-sibling::td');
                break;
                
            case 'ruecklieferungsdatum':
                $value = extractTextContent($html, '//td[contains(text(), "Rücklieferungsdatum")]/following-sibling::td');
                break;
                
            case 'rueckgabe_bemerkung':
                $value = extractTextContent($html, '//td[contains(text(), "Rückgabe")]/following-sibling::td');
                break;
                
            default:
                throw new Exception("Unbekanntes Feld: $field");
        }
        
        if ($value === null || $value === '') {
            throw new Exception("Feld '$field' nicht gefunden oder leer");
        }
        
        debugLog("Feld '$field' erfolgreich extrahiert: '$value'");
        
        return [
            'success' => true,
            'value' => $value
        ];
        
    } catch (Exception $e) {
        debugLog("Feld-Extraktion-Fehler für '$field': " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

// Hauptverarbeitung
try {
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input: " . $rawInput);
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    require_once __DIR__ . '/../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    switch ($action) {
        case 'login':
            $appointmentId = $input['appointment_id'] ?? null;
            if (!$appointmentId) {
                throw new Exception('Appointment ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_username, a.resy_password, a.resy_base_url 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden');
            }
            
            $result = resyLogin($autohaus['resy_username'], $autohaus['resy_password'], $autohaus['resy_base_url']);
            echo json_encode($result);
            break;
            
        case 'navigate_to_search':
            $sessionData = $input['sessionData'] ?? [];
            $result = resyNavigateToSearch($sessionData);
            echo json_encode($result);
            break;
            
        case 'search_lgs_id':
            $lgsId = $input['lgsId'] ?? null;
            $sessionData = $input['sessionData'] ?? [];
            
            if (!$lgsId) {
                throw new Exception('LGS_ID fehlt');
            }
            
            $result = resySearchLgsId($lgsId, $sessionData);
            echo json_encode($result);
            break;
            
        case 'navigate_to_detail':
            $lgsId = $input['lgsId'] ?? null;
            $sessionData = $input['sessionData'] ?? [];
            
            if (!$lgsId) {
                throw new Exception('LGS_ID fehlt');
            }
            
            $result = resyNavigateToDetail($lgsId, $sessionData);
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $sessionData = $input['sessionData'] ?? [];
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            // HTML aus der letzten Detailseite verwenden
            $html = $sessionData['detailPageHtml'] ?? '';
            if (!$html) {
                throw new Exception('Detailseiten-HTML nicht verfügbar');
            }
            
            $result = extractFieldData($field, $html, $sessionData);
            echo json_encode($result);
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]
    ]);
}
?>
