<?php
/**
 * RESY REST API Routes - Alle verfügbaren Endpunkte
 * Zentrale API für alle RESY-Operationen
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';
require_once __DIR__ . '/resy_rest_api_complete.php';
require_once __DIR__ . '/cache_manager.php';

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? 1;

try {
    $api = new ResyRestAPIComplete($autohausId);
    $db = new Database();
    $cacheManager = new CacheManager($db->getConnection());
    
    switch ($action) {
        // ============ WHEELSETS ============
        case 'get_wheelsets':
            $wheelsets = $api->getAllWheelsetsSmart();
            echo json_encode([
                'success' => true,
                'wheelsets' => $wheelsets,
                'count' => count($wheelsets),
                'autohaus_id' => $autohausId
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_wheelset_details':
            $wheelsetNumber = $_GET['wheelset_number'] ?? $_POST['wheelset_number'] ?? null;
            if (!$wheelsetNumber) {
                throw new Exception("wheelset_number Parameter erforderlich");
            }
            $details = $api->getWheelsetDetails($wheelsetNumber);
            echo json_encode([
                'success' => true,
                'wheelset' => $details
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_wheelsets_by_vehicle':
            $vin = $_GET['vin'] ?? $_POST['vin'] ?? null;
            $licenseNo = $_GET['license_no'] ?? $_POST['license_no'] ?? null;
            $wheelsets = $api->getWheelsetsByVehicle($vin, $licenseNo);
            echo json_encode([
                'success' => true,
                'wheelsets' => $wheelsets
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
        
        // ============ PICKUP ORDERS ============
        case 'get_pickup_orders':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $orders = $api->getPickupOrders($customerNo);
            echo json_encode([
                'success' => true,
                'pickupOrders' => $orders,
                'count' => count($orders)
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'create_pickup_order':
            // Lese JSON aus dem Body falls vorhanden
            $jsonInput = file_get_contents('php://input');
            $jsonData = [];
            if (!empty($jsonInput)) {
                $jsonData = json_decode($jsonInput, true) ?? [];
            }
            
            $customerNo = $jsonData['customer_no'] ?? $_POST['customer_no'] ?? null;
            $orders = $jsonData['pickupOrders'] ?? $_POST['pickupOrders'] ?? [];
            if (empty($orders)) {
                throw new Exception("pickupOrders Array erforderlich");
            }
            $result = $api->createPickupOrder($customerNo ?? $api->customerNo, $orders);
            
            // Cache invalidieren nach Statusänderung
            try {
                $cacheManager->invalidateCache($autohausId);
            } catch (Exception $cacheError) {
                error_log("Cache-Invalidierung fehlgeschlagen: " . $cacheError->getMessage());
                // Fehler ignorieren, da die Hauptoperation erfolgreich war
            }
            
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'delete_pickup_order':
            // Lese JSON aus dem Body falls vorhanden
            $jsonInput = file_get_contents('php://input');
            $jsonData = [];
            if (!empty($jsonInput)) {
                $jsonData = json_decode($jsonInput, true) ?? [];
            }
            
            $wheelsetNumber = $jsonData['wheelset_number'] ?? $_POST['wheelset_number'] ?? $_GET['wheelset_number'] ?? null;
            if (!$wheelsetNumber) {
                throw new Exception("wheelset_number Parameter erforderlich");
            }
            $result = $api->deletePickupOrder($wheelsetNumber);
            
            // Cache invalidieren nach Statusänderung
            try {
                $cacheManager->invalidateCache($autohausId);
            } catch (Exception $cacheError) {
                error_log("Cache-Invalidierung fehlgeschlagen: " . $cacheError->getMessage());
                // Fehler ignorieren, da die Hauptoperation erfolgreich war
            }
            
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
        
        // ============ RETURN ORDERS ============
        case 'get_return_orders':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $orders = $api->getReturnOrders($customerNo);
            echo json_encode([
                'success' => true,
                'returnOrders' => $orders,
                'count' => count($orders)
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'create_return_order':
            // Lese JSON aus dem Body falls vorhanden
            $jsonInput = file_get_contents('php://input');
            $jsonData = [];
            if (!empty($jsonInput)) {
                $jsonData = json_decode($jsonInput, true) ?? [];
            }
            
            $customerNo = $jsonData['customer_no'] ?? $_POST['customer_no'] ?? null;
            $orders = $jsonData['returnOrders'] ?? $_POST['returnOrders'] ?? [];
            if (empty($orders)) {
                throw new Exception("returnOrders Array erforderlich");
            }
            
            // Transformiere appointment zu date und time falls nötig
            foreach ($orders as &$order) {
                if (!is_array($order)) {
                    error_log("RESY createReturnOrder: Ungültiger Order-Eintrag (kein Array): " . var_export($order, true));
                    continue; // Überspringe ungültige Einträge
                }
                
                // Logge ursprüngliche Order-Daten
                error_log("RESY createReturnOrder: Original Order: " . json_encode($order));
                
                // Prüfe ob wheelSetNumbers ein Array ist und konvertiere zu Integer
                if (isset($order['wheelSetNumbers'])) {
                    if (!is_array($order['wheelSetNumbers'])) {
                        error_log("RESY createReturnOrder: wheelSetNumbers ist kein Array, konvertiere...");
                        $order['wheelSetNumbers'] = [$order['wheelSetNumbers']];
                    }
                    // Stelle sicher, dass alle Werte Integer sind
                    $order['wheelSetNumbers'] = array_map('intval', $order['wheelSetNumbers']);
                    error_log("RESY createReturnOrder: wheelSetNumbers nach Konvertierung: " . json_encode($order['wheelSetNumbers']));
                } else {
                    error_log("RESY createReturnOrder: WARNUNG - wheelSetNumbers fehlt in Order!");
                }
                
                // Transformiere appointment zu date und time
                if (isset($order['appointment']) && !isset($order['date'])) {
                    // Parse ISO 8601 Datum/Zeit Format: YYYY-MM-DDTHH:MM:SS+01:00
                    $appointment = $order['appointment'];
                    error_log("RESY createReturnOrder: Transformiere appointment: {$appointment}");
                    
                    if (is_string($appointment) && preg_match('/^(\d{4}-\d{2}-\d{2})T(\d{2}):(\d{2}):\d{2}([+-]\d{2}:\d{2})?/', $appointment, $matches)) {
                        // Prüfe ob alle Matches vorhanden sind
                        if (isset($matches[1]) && isset($matches[2]) && isset($matches[3])) {
                            // API erwartet date als ISO 8601 Datum
                            // Laut Dokumentation: "date: date of the delivery (ISO 8601)"
                            // Das könnte bedeuten: nur Datum (YYYY-MM-DD) oder vollständiges ISO 8601 Datum
                            // Versuche zuerst nur das Datum (wie in der Dokumentation beschrieben)
                            $dateOnly = $matches[1]; // YYYY-MM-DD
                            $timeOnly = $matches[2] . ':' . $matches[3]; // HH:MM
                            
                            // Prüfe ob timezone vorhanden ist
                            $timezone = isset($matches[4]) ? $matches[4] : '+01:00';
                            
                            // API-Dokumentation sagt "date: date of the delivery (ISO 8601)"
                            // und "time: time of the delivery"
                            // Versuche mit HH:MM Format (ohne Sekunden)
                            $order['date'] = $dateOnly; // YYYY-MM-DD
                            $order['time'] = $timeOnly; // HH:MM (ohne Sekunden)
                            
                            // Entferne alle optionalen Felder, die null oder leer sind
                            foreach (['offset', 'comment'] as $optionalField) {
                                if (isset($order[$optionalField]) && ($order[$optionalField] === null || $order[$optionalField] === '')) {
                                    unset($order[$optionalField]);
                                }
                            }
                            
                            // TEST: Versuche ohne time, falls die API Probleme damit hat
                            // time ist optional laut API-Dokumentation
                            // Entferne time temporär, um zu testen, ob das das Problem löst
                            // unset($order['time']); // AUSKOMMENTIERT - wieder aktivieren falls nötig
                            
                            unset($order['appointment']);
                            error_log("RESY createReturnOrder: Transformiert zu date={$order['date']}, time={$order['time']}");
                        } else {
                            error_log("RESY createReturnOrder: Regex-Matches fehlen für appointment: {$appointment}");
                        }
                    } else {
                        error_log("RESY createReturnOrder: appointment Format nicht erkannt: {$appointment}");
                    }
                } elseif (isset($order['date']) && !isset($order['time'])) {
                    // Falls nur date vorhanden, aber keine time, setze default time
                    error_log("RESY createReturnOrder: Nur date vorhanden, keine time. Date: {$order['date']}");
                }
                
                // Stelle sicher, dass date vorhanden ist (erforderlich laut API-Dokumentation)
                // time ist optional laut API-Dokumentation
                if (!isset($order['date'])) {
                    error_log("RESY createReturnOrder: FEHLER - date fehlt nach Transformation!");
                    error_log("RESY createReturnOrder: Order nach Transformation: " . json_encode($order));
                }
                
                // Prüfe ob time vorhanden ist (optional, aber empfohlen)
                if (!isset($order['time'])) {
                    error_log("RESY createReturnOrder: WARNUNG - time fehlt nach Transformation (optional)");
                }
                
                // Stelle sicher, dass nur die erwarteten Felder vorhanden sind
                // Erlaubte Felder laut API-Dokumentation:
                // - wheelSetNumbers (erforderlich)
                // - date (erforderlich)
                // - time (optional, aber empfohlen)
                // - offset (optional)
                // - deliveryOption (optional)
                // - comment (optional)
                $allowedFields = ['wheelSetNumbers', 'date', 'time', 'offset', 'deliveryOption', 'comment'];
                $cleanedOrder = [];
                foreach ($allowedFields as $field) {
                    if (isset($order[$field])) {
                        $cleanedOrder[$field] = $order[$field];
                    }
                }
                $order = $cleanedOrder;
                
                // Logge transformierte Order-Daten
                error_log("RESY createReturnOrder: Transformierte Order: " . json_encode($order));
            }
            unset($order); // Wichtig: Referenz löschen
            
            // Validiere finales Order-Array vor dem Senden
            foreach ($orders as $index => $order) {
                if (!isset($order['wheelSetNumbers']) || !is_array($order['wheelSetNumbers']) || empty($order['wheelSetNumbers'])) {
                    throw new Exception("Order {$index}: wheelSetNumbers fehlt oder ist leer");
                }
                if (!isset($order['date']) || empty($order['date'])) {
                    throw new Exception("Order {$index}: date fehlt oder ist leer");
                }
            }
            
            error_log("RESY createReturnOrder: Finale Orders vor API-Call: " . json_encode($orders));
            
            try {
                $result = $api->createReturnOrder($customerNo ?? $api->customerNo, $orders);
            } catch (Exception $e) {
                // Wenn die API einen Fehler zurückgibt, werfe eine Exception
                error_log("RESY createReturnOrder Exception: " . $e->getMessage());
                throw new Exception("API-Fehler: " . $e->getMessage());
            }
            
            // Prüfe ob result ein Array ist
            if (!is_array($result)) {
                error_log("RESY createReturnOrder: Ungültige Antwort-Struktur. Response: " . var_export($result, true));
                throw new Exception("Ungültige API-Antwort erhalten");
            }
            
            // Validiere API-Antwort: Prüfe ob orderResult vorhanden ist und erfolgreich
            $success = true;
            $errorMessages = [];
            
            if (isset($result['orderResult']) && is_array($result['orderResult'])) {
                foreach ($result['orderResult'] as $orderGroup) {
                    if (is_array($orderGroup)) {
                        foreach ($orderGroup as $orderResult) {
                            if (is_array($orderResult) && isset($orderResult['statusCode'])) {
                                // Prüfe ob statusCode auf Erfolg hinweist
                                // StatusCodes können verschiedene Werte haben, prüfe auf Fehler
                                $statusCode = $orderResult['statusCode'];
                                $status = isset($orderResult['status']) ? $orderResult['status'] : '';
                                
                                // Wenn statusCode nicht erfolgreich ist oder status eine Fehlermeldung enthält
                                if (is_string($statusCode) && (stripos($statusCode, 'error') !== false || stripos($statusCode, 'fail') !== false)) {
                                    $success = false;
                                    $errorMessages[] = $status ?: "Status Code: {$statusCode}";
                                } elseif (is_string($status) && (stripos($status, 'fehler') !== false || stripos($status, 'error') !== false)) {
                                    $success = false;
                                    $errorMessages[] = $status;
                                }
                            }
                        }
                    }
                }
            } else {
                // Falls keine orderResult vorhanden, aber HTTP 200, könnte es trotzdem erfolgreich sein
                // Logge zur Debugging
                error_log("RESY createReturnOrder: Keine orderResult in Antwort, aber HTTP 200. Response: " . json_encode($result));
            }
            
            if (!$success) {
                throw new Exception("Rücklieferung fehlgeschlagen: " . implode(', ', $errorMessages));
            }
            
            // Cache invalidieren nach Statusänderung
            try {
                $cacheManager->invalidateCache($autohausId);
            } catch (Exception $cacheError) {
                error_log("Cache-Invalidierung fehlgeschlagen: " . $cacheError->getMessage());
                // Fehler ignorieren, da die Hauptoperation erfolgreich war
            }
            
            echo json_encode([
                'success' => true,
                'result' => $result,
                'orderResult' => isset($result['orderResult']) ? $result['orderResult'] : null
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'delete_return_order':
            // Lese JSON aus dem Body falls vorhanden
            $jsonInput = file_get_contents('php://input');
            $jsonData = [];
            if (!empty($jsonInput)) {
                $jsonData = json_decode($jsonInput, true) ?? [];
            }
            
            $wheelsetNumber = $jsonData['wheelset_number'] ?? $_POST['wheelset_number'] ?? $_GET['wheelset_number'] ?? null;
            if (!$wheelsetNumber) {
                throw new Exception("wheelset_number Parameter erforderlich");
            }
            $result = $api->deleteReturnOrder($wheelsetNumber);
            
            // Cache invalidieren nach Statusänderung
            try {
                $cacheManager->invalidateCache($autohausId);
            } catch (Exception $cacheError) {
                error_log("Cache-Invalidierung fehlgeschlagen: " . $cacheError->getMessage());
                // Fehler ignorieren, da die Hauptoperation erfolgreich war
            }
            
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
        
        // ============ CUSTOMER INFO ============
        case 'get_customer_details':
            $startTime = microtime(true);
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            if (!$customerNo) {
                throw new Exception("customer_no Parameter erforderlich");
            }
            // Konvertiere customerNo zu Integer falls nötig
            $customerNo = is_numeric($customerNo) ? (int)$customerNo : $customerNo;
            error_log("RESY get_customer_details: customerNo = " . $customerNo);
            
            try {
                $details = $api->getCustomerDetails($customerNo);
                $duration = round((microtime(true) - $startTime) * 1000, 2);
                error_log("RESY get_customer_details: API-Call dauerte {$duration}ms");
                
                if ($details === null || (is_array($details) && empty($details))) {
                    error_log("RESY get_customer_details: Kunde nicht gefunden für customerNo = " . $customerNo);
                    throw new Exception("Kunde nicht gefunden");
                }
                
                error_log("RESY get_customer_details: details gefunden für customerNo = " . $customerNo);
                echo json_encode([
                    'success' => true,
                    'customer' => $details
                ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            } catch (Exception $e) {
                $duration = round((microtime(true) - $startTime) * 1000, 2);
                error_log("RESY get_customer_details: Fehler nach {$duration}ms - " . $e->getMessage());
                throw $e;
            }
            break;
            
        case 'get_allowed_customers':
            $customers = $api->getAllowedCustomers();
            echo json_encode([
                'success' => true,
                'customers' => $customers
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_service_packages':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $packages = $api->getServicePackages($customerNo);
            echo json_encode([
                'success' => true,
                'servicePackages' => $packages
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_inventory':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $inventory = $api->getInventory($customerNo);
            echo json_encode([
                'success' => true,
                'inventory' => $inventory
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_global_inventory':
            $inventory = $api->getGlobalInventory();
            echo json_encode([
                'success' => true,
                'inventory' => $inventory
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_pickup_order_dates':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $dates = $api->getPickupOrderDates($customerNo);
            echo json_encode([
                'success' => true,
                'orderDays' => $dates
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_return_order_dates':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $dates = $api->getReturnOrderDates($customerNo);
            echo json_encode([
                'success' => true,
                'orderDays' => $dates
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_delivery_options':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $options = $api->getDeliveryOptions($customerNo);
            echo json_encode([
                'success' => true,
                'deliveryOptions' => $options
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_loading_points':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $points = $api->getLoadingPoints($customerNo);
            echo json_encode([
                'success' => true,
                'loadingPoints' => $points
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_order_times':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $times = $api->getOrderTimes($customerNo);
            echo json_encode([
                'success' => true,
                'orderTimes' => $times
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_pickup_order_dates_custom':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $date = $_GET['date'] ?? $_POST['date'] ?? null;
            $forwardDays = $_GET['forwardDays'] ?? $_POST['forwardDays'] ?? 30;
            if (!$date) {
                throw new Exception("date Parameter erforderlich");
            }
            $dates = $api->getPickupOrderDatesCustom($customerNo ?? $api->customerNo, $date, intval($forwardDays));
            echo json_encode([
                'success' => true,
                'orderDays' => $dates
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_return_order_dates_custom':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            $date = $_GET['date'] ?? $_POST['date'] ?? null;
            $forwardDays = $_GET['forwardDays'] ?? $_POST['forwardDays'] ?? 30;
            if (!$date) {
                throw new Exception("date Parameter erforderlich");
            }
            $dates = $api->getReturnOrderDatesCustom($customerNo ?? $api->customerNo, $date, intval($forwardDays));
            echo json_encode([
                'success' => true,
                'orderDays' => $dates
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
        
        // ============ TOOLS ============
        case 'check_wheelset':
            $wheelsetNumber = $_GET['wheelset_number'] ?? $_POST['wheelset_number'] ?? null;
            if (!$wheelsetNumber) {
                throw new Exception("wheelset_number Parameter erforderlich");
            }
            $result = $api->checkWheelsetNumber($wheelsetNumber);
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'check_license_plate':
            $plate = $_GET['license_plate'] ?? $_POST['license_plate'] ?? null;
            if (!$plate) {
                throw new Exception("license_plate Parameter erforderlich");
            }
            $result = $api->checkLicencePlate($plate);
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'check_vin':
            $vin = $_GET['vin'] ?? $_POST['vin'] ?? null;
            if (!$vin) {
                throw new Exception("vin Parameter erforderlich");
            }
            $result = $api->checkVIN($vin);
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
        
        // ============ EXTRA ORDERS ============
        case 'get_all_extra_orders':
            $orders = $api->getAllExtraOrders();
            echo json_encode([
                'success' => true,
                'extraOrders' => $orders,
                'count' => count($orders)
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'get_customer_extra_orders':
            $customerNo = $_GET['customer_no'] ?? $_POST['customer_no'] ?? null;
            // getExtraOrders gibt mögliche Extra Orders für einen Kunden zurück
            $orders = $api->getExtraOrders($customerNo ?? $api->customerNo);
            echo json_encode([
                'success' => true,
                'extraOrders' => $orders
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'create_extra_order':
            $data = json_decode(file_get_contents('php://input'), true);
            if (!$data || !isset($data['name']) || !isset($data['quantity'])) {
                throw new Exception("name und quantity sind erforderlich");
            }
            // Setze customerNo automatisch falls nicht gesetzt
            if (!isset($data['customerNo']) || empty($data['customerNo'])) {
                $data['customerNo'] = $api->customerNo;
            }
            $result = $api->createExtraOrder($data);
            
            // Cache invalidieren nach Statusänderung
            try {
                $cacheManager->invalidateCache($autohausId);
            } catch (Exception $cacheError) {
                error_log("Cache-Invalidierung fehlgeschlagen: " . $cacheError->getMessage());
            }
            
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
            
        case 'delete_extra_order':
            // Lese JSON aus dem Body falls vorhanden
            $jsonInput = file_get_contents('php://input');
            $jsonData = [];
            if (!empty($jsonInput)) {
                $jsonData = json_decode($jsonInput, true) ?? [];
            }
            
            $extraOrderId = $jsonData['extra_order_id'] ?? $_POST['extra_order_id'] ?? $_GET['extra_order_id'] ?? null;
            $orderName = $jsonData['name'] ?? $_POST['name'] ?? $_GET['name'] ?? null;
            $customerNo = $jsonData['customerNo'] ?? $_POST['customerNo'] ?? $_GET['customerNo'] ?? null;
            
            if (!$extraOrderId && (!$orderName || !$customerNo)) {
                throw new Exception("extra_order_id oder (name und customerNo) Parameter erforderlich");
            }
            
            // Wenn keine ID vorhanden, versuche über Name + CustomerNo zu finden
            if (!$extraOrderId && $orderName && $customerNo) {
                // Lade alle Extra Orders und suche nach passender Order
                $allExtraOrders = $api->getAllExtraOrders();
                foreach ($allExtraOrders as $order) {
                    if (isset($order['name']) && $order['name'] === $orderName && 
                        isset($order['customerNo']) && $order['customerNo'] == $customerNo) {
                        $extraOrderId = $order['id'] ?? $order['extraOrderId'] ?? $order['extra_order_id'] ?? null;
                        break;
                    }
                }
            }
            
            if (!$extraOrderId) {
                throw new Exception("Extra Order konnte nicht gefunden werden");
            }
            
            $result = $api->deleteExtraOrder($extraOrderId);
            
            // Cache invalidieren nach Statusänderung
            try {
                $cacheManager->invalidateCache($autohausId);
            } catch (Exception $cacheError) {
                error_log("Cache-Invalidierung fehlgeschlagen: " . $cacheError->getMessage());
            }
            
            echo json_encode([
                'success' => true,
                'result' => $result
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            break;
        
        default:
            throw new Exception("Unbekannte Aktion: " . $action);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}

