<?php
// register_user.php - Benutzerregistrierung
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

require_once 'config.php';

// Helper Funktionen
function getDeviceInfo() {
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    if (strpos($user_agent, 'Mobile') !== false) {
        if (preg_match('/iPhone|iPad|iPod/i', $user_agent)) return 'iOS';
        if (preg_match('/Android/i', $user_agent)) return 'Android';
        return 'Mobile';
    }
    if (preg_match('/Mac|Macintosh/i', $user_agent)) return 'Mac';
    if (preg_match('/Windows/i', $user_agent)) return 'Windows';
    if (preg_match('/Linux/i', $user_agent)) return 'Linux';
    return 'Unknown';
}

function getClientIP() {
    $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP) !== false) {
                    return $ip;
                }
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Validierung der Eingabedaten
    $required_fields = ['first_name', 'last_name', 'email', 'phone_number', 'license_plate'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            throw new Exception("Field '$field' is required");
        }
    }
    
    // E-Mail-Validierung
    if (!filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Invalid email format');
    }
    
    // Prüfen ob E-Mail bereits existiert
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$input['email']]);
    if ($stmt->fetch()) {
        throw new Exception('Email already exists');
    }
    
    // Prüfen ob autohaus_id oder preferred_dealership_id existiert (falls angegeben)
    $preferred_dealership_id = $input['autohaus_id'] ?? $input['preferred_dealership_id'] ?? null;
    
    if (!empty($preferred_dealership_id)) {
        // Prüfen ob die Autohaus-Tabelle existiert und ob die ID aktiv ist
        $stmt = $pdo->prepare("SELECT id FROM autohaus WHERE id = ? AND is_active = 1");
        $stmt->execute([$preferred_dealership_id]);
        if (!$stmt->fetch()) {
            // Autohaus-ID existiert nicht oder ist nicht aktiv, setze auf NULL
            $preferred_dealership_id = null;
        }
    } else {
        $preferred_dealership_id = null;
    }
    
    // Passwort-Validierung
    if (empty($input['password'])) {
        throw new Exception('Password is required');
    }
    
    // Passwort-Stärke prüfen
    $password = $input['password'];
    if (strlen($password) < 8) {
        throw new Exception('Password must be at least 8 characters long');
    }
    if (!preg_match('/[A-Z]/', $password)) {
        throw new Exception('Password must contain at least one uppercase letter');
    }
    if (!preg_match('/[0-9]/', $password)) {
        throw new Exception('Password must contain at least one number');
    }
    
    // Passwort-Bestätigung prüfen
    if (empty($input['password_confirm'])) {
        throw new Exception('Password confirmation is required');
    }
    if ($input['password'] !== $input['password_confirm']) {
        throw new Exception('Passwords do not match');
    }
    
    // Passwort-Hash erstellen
    $password_hash = password_hash($password, PASSWORD_DEFAULT);
    
    // Auth-Token generieren
    $auth_token = bin2hex(random_bytes(32));
    $token_expires = date('Y-m-d H:i:s', strtotime('+30 days'));
    
    // Registrierungsquelle bestimmen
    $registration_source = $input['registration_source'] ?? 'web';
    $registration_device = $input['registration_device'] ?? getDeviceInfo();
    $registration_ip = $input['registration_ip'] ?? getClientIP();
    
    // Benutzer in Datenbank einfügen
    $sql = "INSERT INTO users (
        first_name, last_name, email, phone_number, license_plate,
        preferred_dealership_id, profile_image_url, is_leasing_vehicle,
        is_contract_vehicle, password_hash, auth_token, token_expires_at,
        email_verified, verification_status, registration_source, registration_device, registration_ip
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute([
        $input['first_name'],
        $input['last_name'],
        $input['email'],
        $input['phone_number'],
        $input['license_plate'],
        $preferred_dealership_id,
        $input['profile_image_url'] ?? null,
        $input['is_leasing_vehicle'] ?? false,
        $input['is_contract_vehicle'] ?? false,
        $password_hash,
        $auth_token,
        $token_expires,
        false, // Email muss noch verifiziert werden
        'pending', // Neue User warten auf Verifizierung
        $registration_source,
        $registration_device,
        $registration_ip
    ]);
    
    if ($result) {
        $user_id = $pdo->lastInsertId();
        
        // Benutzerdaten zurückgeben
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Passwort-Hash aus Antwort entfernen
        unset($user['password_hash']);
        
        echo json_encode([
            'success' => true,
            'user' => $user,
            'token' => $auth_token,
            'message' => 'User registered successfully'
        ]);
    } else {
        throw new Exception('Failed to create user');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
