<?php
/**
 * Performance-Monitor für die API
 * 
 * Features:
 * - Ladezeit-Messung
 * - Memory-Usage-Tracking
 * - Performance-Metriken
 * - Optimierungsvorschläge
 */

class PerformanceMonitor {
    private $startTime;
    private $endTime;
    private $startMemory;
    private $endMemory;
    private $metrics = [];
    
    public function start() {
        $this->startTime = microtime(true);
        $this->startMemory = memory_get_usage(true);
        $this->metrics = [];
    }
    
    public function end() {
        $this->endTime = microtime(true);
        $this->endMemory = memory_get_usage(true);
    }
    
    /**
     * Gibt die Ladezeit in Sekunden zurück
     */
    public function getDuration() {
        if ($this->startTime && $this->endTime) {
            return round($this->endTime - $this->startTime, 2);
        }
        return 0;
    }
    
    /**
     * Gibt den Memory-Verbrauch in MB zurück
     */
    public function getMemoryUsage() {
        if ($this->startMemory && $this->endMemory) {
            return round(($this->endMemory - $this->startMemory) / 1024 / 1024, 2);
        }
        return 0;
    }
    
    /**
     * Fügt eine Metrik hinzu
     */
    public function addMetric($name, $value) {
        $this->metrics[$name] = $value;
    }
    
    /**
     * Gibt alle Metriken zurück
     */
    public function getMetrics() {
        return array_merge([
            'duration' => $this->getDuration(),
            'memory_usage' => $this->getMemoryUsage(),
            'peak_memory' => round(memory_get_peak_usage(true) / 1024 / 1024, 2)
        ], $this->metrics);
    }
    
    /**
     * Generiert Performance-Report
     */
    public function generateReport() {
        $duration = $this->getDuration();
        $memory = $this->getMemoryUsage();
        
        $report = [
            'duration' => $duration . 's',
            'memory_usage' => $memory . 'MB',
            'performance_rating' => $this->getPerformanceRating($duration, $memory),
            'recommendations' => $this->getRecommendations($duration, $memory)
        ];
        
        return $report;
    }
    
    /**
     * Bewertet die Performance
     */
    private function getPerformanceRating($duration, $memory) {
        if ($duration < 5 && $memory < 50) {
            return 'excellent';
        } elseif ($duration < 15 && $memory < 100) {
            return 'good';
        } elseif ($duration < 30 && $memory < 200) {
            return 'acceptable';
        } else {
            return 'poor';
        }
    }
    
    /**
     * Generiert Optimierungsvorschläge
     */
    private function getRecommendations($duration, $memory) {
        $recommendations = [];
        
        if ($duration > 30) {
            $recommendations[] = 'Ladezeit zu hoch - verwende Chunked-Loading';
        }
        
        if ($memory > 200) {
            $recommendations[] = 'Memory-Verbrauch zu hoch - optimiere Parsing';
        }
        
        if ($duration > 60) {
            $recommendations[] = 'Timeout-Risiko - reduziere Seitenanzahl';
        }
        
        return $recommendations;
    }
}
?>

