<?php
/**
 * P2 Reifenangebote API
 * CRUD-Operationen für P2 Radsatz-Angebote
 */

// Unterdrücke Warnings/Notices für saubere JSON-Antworten
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Output buffering für saubere JSON-Antworten
ob_start();

header('Content-Type: application/json; charset=UTF-8');
session_start();

// Prüfe Authentifizierung nur für bestimmte Aktionen
$action = $_GET['action'] ?? '';
$publicActions = ['get_by_token', 'save_customer_selection'];

// Für öffentliche Aktionen keine Admin-Authentifizierung erforderlich
if (!in_array($action, $publicActions)) {
    if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
        exit;
    }
}

require_once dirname(__DIR__) . '/config/database.php';
require_once dirname(__DIR__) . '/admin/includes/p2_api.php';

$db = new Database();
$conn = $db->getConnection();
$current_user_id = $_SESSION['admin_user_id'] ?? null;

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'get':
            // GET - Angebote für einen Kunden laden ODER einzelnes Angebot per offer_id
            $offerId = $_GET['offer_id'] ?? '';
            $kundenAdrId = $_GET['kunden_adr_id'] ?? '';
            $autohausId = $_GET['autohaus_id'] ?? '';
            
            // Wenn offer_id vorhanden, lade einzelnes Angebot
            if ($offerId) {
                if (!$autohausId) {
                    throw new Exception('autohaus_id ist erforderlich');
                }
                
                $stmt = $conn->prepare("
                    SELECT 
                        pto.*,
                        ah.name as autohaus_name,
                        ah.address as autohaus_address,
                        ah.phone as autohaus_phone,
                        ah.email as autohaus_email,
                        ah.email_from_name,
                        ah.email_smtp_host,
                        ah.email_smtp_port,
                        ah.email_smtp_username,
                        ah.email_smtp_password,
                        ah.email_smtp_encryption
                    FROM p2_tire_offers pto
                    JOIN autohaus ah ON pto.autohaus_id = ah.id
                    WHERE pto.id = ? AND pto.autohaus_id = ?
                ");
                $stmt->execute([$offerId, $autohausId]);
                $offer = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$offer) {
                    throw new Exception('Angebot nicht gefunden');
                }
                
                // Decode JSON fields
                if ($offer['offer_data']) {
                    $offer['offer_data'] = json_decode($offer['offer_data'], true);
                }
                if ($offer['customer_selection']) {
                    $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
                }
                
                ob_clean();
                echo json_encode([
                    'success' => true,
                    'offer' => $offer
                ]);
            } else {
                // Lade alle Angebote für einen Kunden
                if (!$kundenAdrId || !$autohausId) {
                    throw new Exception('kunden_adr_id und autohaus_id sind erforderlich');
                }
                
                $stmt = $conn->prepare("
                    SELECT 
                        id, kunden_adr_id, autohaus_id, customer_name, customer_vorname, license_plate, vin, auftrag_nr,
                        offer_data, letter_content, employee_name, token, token_expires_at, token_used_at,
                        email_sent, email_sent_at, pdf_filename, status, customer_selection, total_amount,
                        created_by, created_at, updated_at
                    FROM p2_tire_offers
                    WHERE kunden_adr_id = ? AND autohaus_id = ?
                    ORDER BY created_at DESC
                ");
                $stmt->execute([$kundenAdrId, $autohausId]);
                $offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Decode JSON fields
                foreach ($offers as &$offer) {
                    if ($offer['offer_data']) {
                        $offer['offer_data'] = json_decode($offer['offer_data'], true);
                    }
                    if ($offer['customer_selection']) {
                        $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
                    }
                }
                
                ob_clean();
                echo json_encode([
                    'success' => true,
                    'offers' => $offers,
                    'count' => count($offers)
                ]);
            }
            break;
            
        case 'get_all':
            // GET - Alle Angebote für ein Autohaus laden
            $autohausId = $_GET['autohaus_id'] ?? '';
            $filter = $_GET['filter'] ?? '';
            $search = $_GET['search'] ?? '';
            
            if (!$autohausId) {
                throw new Exception('autohaus_id ist erforderlich');
            }
            
            $sql = "
                SELECT 
                    id, kunden_adr_id, autohaus_id, customer_name, customer_vorname, license_plate, vin, auftrag_nr,
                    offer_data, letter_content, employee_name, token, token_expires_at, token_used_at,
                    email_sent, email_sent_at, pdf_filename, status, customer_selection, total_amount,
                    created_by, created_at, updated_at
                FROM p2_tire_offers
                WHERE autohaus_id = ?
            ";
            
            $params = [$autohausId];
            
            // Filter basierend auf Parameter
            if ($filter === 'pending') {
                $sql .= " AND status IN ('sent', 'viewed')";
            } elseif ($filter === 'accepted') {
                $sql .= " AND status = 'accepted'";
            } elseif ($filter === 'declined') {
                $sql .= " AND status = 'declined'";
            } elseif ($filter === 'new_request') {
                $sql .= " AND status = 'new_request'";
            } elseif ($filter === 'all') {
                $sql .= " AND status IN ('accepted', 'declined', 'new_request', 'sent', 'viewed')";
            } else {
                $sql .= " AND status IN ('accepted', 'declined', 'new_request')";
            }
            
            // Suchfilter
            if (!empty($search)) {
                $sql .= " AND (customer_name LIKE ? OR license_plate LIKE ?)";
                $searchParam = '%' . $search . '%';
                $params[] = $searchParam;
                $params[] = $searchParam;
            }
            
            $sql .= " ORDER BY updated_at DESC, created_at DESC";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Decode JSON fields
            foreach ($offers as &$offer) {
                if ($offer['offer_data']) {
                    $offer['offer_data'] = json_decode($offer['offer_data'], true);
                }
                if ($offer['customer_selection']) {
                    $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
                }
            }
            
            ob_clean();
            echo json_encode([
                'success' => true,
                'offers' => $offers,
                'count' => count($offers)
            ]);
            break;
            
        case 'create':
        case 'save':
            // POST - Angebot erstellen
            $input = file_get_contents('php://input');
            $data = json_decode($input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Ungültige JSON-Daten: ' . json_last_error_msg());
            }
            
            $kundenAdrId = $data['kunden_adr_id'] ?? '';
            $autohausId = $data['autohaus_id'] ?? '';
            $customerName = $data['customer_name'] ?? '';
            $customerVorname = $data['customer_vorname'] ?? '';
            $licensePlate = $data['license_plate'] ?? '';
            $vin = $data['vin'] ?? '';
            $auftragNr = $data['auftrag_nr'] ?? null;
            $newTires = $data['new_tires'] ?? [];
            $selectedServices = $data['selected_services'] ?? [];
            $totalAmount = $data['total_amount'] ?? 0;
            $letterContent = $data['letter_content'] ?? '';
            $employeeName = $data['employee_name'] ?? '';
            
            if (!$kundenAdrId || !$autohausId) {
                throw new Exception('kunden_adr_id und autohaus_id sind erforderlich');
            }
            
            // Generiere Token
            $token = bin2hex(random_bytes(32));
            $tokenExpiresAt = date('Y-m-d H:i:s', strtotime('+30 days'));
            
            // Speichere Angebot
            $offerDataJson = json_encode([
                'new_tires' => $newTires,
                'selected_services' => $selectedServices,
                'total_amount' => $totalAmount
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            
            $stmt = $conn->prepare("
                INSERT INTO p2_tire_offers 
                (kunden_adr_id, autohaus_id, customer_name, customer_vorname, license_plate, vin, auftrag_nr,
                 offer_data, letter_content, employee_name, token, token_expires_at,
                 total_amount, created_by, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
            ");
            
            $stmt->execute([
                $kundenAdrId,
                $autohausId,
                $customerName,
                $customerVorname,
                $licensePlate,
                $vin,
                $auftragNr,
                $offerDataJson,
                $letterContent,
                $employeeName,
                $token,
                $tokenExpiresAt,
                $totalAmount,
                $current_user_id
            ]);
            
            $offerId = $conn->lastInsertId();
            
            ob_clean();
            echo json_encode([
                'success' => true,
                'message' => 'Angebot erstellt',
                'offer_id' => $offerId,
                'token' => $token
            ]);
            break;
            
        case 'update':
            // PUT - Angebot aktualisieren
            $data = json_decode(file_get_contents('php://input'), true);
            
            $offerId = $data['id'] ?? '';
            if (!$offerId) {
                throw new Exception('id ist erforderlich');
            }
            
            $updates = [];
            $params = [];
            
            $allowedFields = [
                'customer_name', 'customer_vorname', 'license_plate', 'vin', 'auftrag_nr',
                'offer_data', 'letter_content', 'employee_name', 'status', 'customer_selection', 'total_amount'
            ];
            
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    if ($field === 'offer_data' || $field === 'customer_selection') {
                        $updates[] = "$field = ?";
                        $params[] = is_array($data[$field]) ? json_encode($data[$field], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) : $data[$field];
                    } else {
                        $updates[] = "$field = ?";
                        $params[] = $data[$field];
                    }
                }
            }
            
            if (empty($updates)) {
                throw new Exception('Keine Felder zum Aktualisieren');
            }
            
            $params[] = $offerId;
            $sql = "UPDATE p2_tire_offers SET " . implode(', ', $updates) . " WHERE id = ?";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            ob_clean();
            echo json_encode([
                'success' => true,
                'message' => 'Angebot aktualisiert'
            ]);
            break;
            
        case 'get_by_token':
            // GET - Angebot per Token laden (für Kundenauswahl)
            $token = $_GET['token'] ?? '';
            
            if (empty($token)) {
                throw new Exception('Token ist erforderlich');
            }
            
            $stmt = $conn->prepare("
                SELECT * FROM p2_tire_offers
                WHERE token = ? AND token_expires_at > NOW() AND token_used_at IS NULL
            ");
            $stmt->execute([$token]);
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$offer) {
                $stmt = $conn->prepare("SELECT * FROM p2_tire_offers WHERE token = ?");
                $stmt->execute([$token]);
                $expiredOffer = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($expiredOffer) {
                    if ($expiredOffer['token_used_at']) {
                        throw new Exception('Token bereits verwendet');
                    } else {
                        throw new Exception('Token abgelaufen');
                    }
                } else {
                    throw new Exception('Ungültiger Token');
                }
            }
            
            // Decode JSON fields
            if ($offer['offer_data']) {
                $offer['offer_data'] = json_decode($offer['offer_data'], true);
            }
            if ($offer['customer_selection']) {
                $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
            }
            
            ob_clean();
            echo json_encode([
                'success' => true,
                'offer' => $offer
            ]);
            break;
            
        case 'save_customer_selection':
            // POST - Kundenauswahl speichern
            $data = json_decode(file_get_contents('php://input'), true);
            
            $token = $data['token'] ?? '';
            $customerSelection = $data['customer_selection'] ?? [];
            
            if (empty($token)) {
                throw new Exception('Token ist erforderlich');
            }
            
            $stmt = $conn->prepare("
                UPDATE p2_tire_offers 
                SET customer_selection = ?, 
                    token_used_at = NOW(),
                    status = CASE 
                        WHEN ? = 'accepted' THEN 'accepted'
                        WHEN ? = 'declined' THEN 'declined'
                        ELSE 'new_request'
                    END
                WHERE token = ? AND token_expires_at > NOW() AND token_used_at IS NULL
            ");
            
            $selectionJson = json_encode($customerSelection, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            $status = $customerSelection['status'] ?? 'new_request';
            
            $stmt->execute([$selectionJson, $status, $status, $token]);
            
            if ($stmt->rowCount() === 0) {
                throw new Exception('Token nicht gefunden oder bereits verwendet');
            }
            
            ob_clean();
            echo json_encode([
                'success' => true,
                'message' => 'Auswahl gespeichert'
            ]);
            break;
            
        case 'send_email':
            // POST - Angebot per E-Mail versenden
            $input = file_get_contents('php://input');
            $data = json_decode($input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Ungültige JSON-Daten: ' . json_last_error_msg());
            }
            
            $offerId = $data['offer_id'] ?? '';
            $recipientEmail = $data['recipient_email'] ?? '';
            $emailSubject = $data['email_subject'] ?? 'Reifenangebot - Boxenstop';
            $emailMessage = $data['email_message'] ?? '';
            
            if (!$offerId || !$recipientEmail) {
                throw new Exception('offer_id und recipient_email sind erforderlich');
            }
            
            // Lade Angebot
            $stmt = $conn->prepare("
                SELECT 
                    pto.*,
                    ah.name as autohaus_name,
                    ah.address as autohaus_address,
                    ah.phone as autohaus_phone,
                    ah.email as autohaus_email,
                    ah.email_from_name,
                    ah.email_smtp_host,
                    ah.email_smtp_port,
                    ah.email_smtp_username,
                    ah.email_smtp_password,
                    ah.email_smtp_encryption
                FROM p2_tire_offers pto
                JOIN autohaus ah ON pto.autohaus_id = ah.id
                WHERE pto.id = ?
            ");
            $stmt->execute([$offerId]);
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$offer) {
                throw new Exception('Angebot nicht gefunden');
            }
            
            // Decode offer_data
            $offerData = json_decode($offer['offer_data'], true);
            
            // Generiere PDF
            $pdfContent = generatePDFForP2Offer($offer, $offerData, $conn);
            $filename = 'p2_angebot_' . $offerId . '_' . date('Y-m-d_H-i-s') . '.pdf';
            $filepath = '../temp/' . $filename;
            
            if (!is_dir('../temp')) {
                mkdir('../temp', 0755, true);
            }
            
            file_put_contents($filepath, $pdfContent);
            
            // Sende E-Mail
            $emailResult = sendP2OfferEmail($offer, $recipientEmail, $filepath, $filename, $emailSubject, $emailMessage);
            
            // Clean up temp file
            if (file_exists($filepath)) {
                unlink($filepath);
            }
            
            if ($emailResult['success']) {
                // Update offer status
                $stmt = $conn->prepare("
                    UPDATE p2_tire_offers 
                    SET email_sent = 1, 
                        email_sent_at = NOW(),
                        status = 'sent'
                    WHERE id = ?
                ");
                $stmt->execute([$offerId]);
                
                ob_clean();
                echo json_encode([
                    'success' => true,
                    'message' => 'E-Mail erfolgreich versendet'
                ]);
            } else {
                throw new Exception($emailResult['error']);
            }
            break;
            
        default:
            throw new Exception('Unbekannte Aktion: ' . $action);
    }
    
} catch (Exception $e) {
    // Clean output buffer
    ob_clean();
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
} catch (Error $e) {
    // Clean output buffer
    ob_clean();
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Interner Serverfehler: ' . $e->getMessage()
    ]);
}

// Clean output buffer before sending response
ob_end_flush();

/**
 * Sendet P2 Angebot per E-Mail
 */
function sendP2OfferEmail($offer, $recipientEmail, $pdfPath, $filename, $emailSubject = null, $emailMessage = '') {
    try {
        if (empty($offer['email_smtp_host']) || empty($offer['email_smtp_username'])) {
            throw new Exception('SMTP-Konfiguration für dieses Autohaus nicht vollständig. Bitte SMTP-Daten in der Datenbank konfigurieren.');
        }
        
        // PHPMailer laden - verwende gleiche Logik wie send_wheelset_offer_email.php
        // Verwende vendor-Version (richtige PHPMailer-Implementierung)
        if (file_exists(dirname(__DIR__) . '/vendor/phpmailer/PHPMailer.php')) {
            require_once dirname(__DIR__) . '/vendor/phpmailer/PHPMailer.php';
            require_once dirname(__DIR__) . '/vendor/phpmailer/SMTP.php';
            require_once dirname(__DIR__) . '/vendor/phpmailer/Exception.php';
        } else if (file_exists(dirname(__DIR__) . '/vendor/autoload.php')) {
            require_once dirname(__DIR__) . '/vendor/autoload.php';
        } else {
            require_once dirname(__DIR__) . '/api/phpmailer/PHPMailer.php';
            require_once dirname(__DIR__) . '/api/phpmailer/SMTP.php';
            require_once dirname(__DIR__) . '/api/phpmailer/Exception.php';
        }
        
        // PHPMailer instanziieren - prüfe Namespace
        $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
        
        try {
            $mail->isSMTP();
            $mail->Host = $offer['email_smtp_host'];
            $mail->SMTPAuth = true;
            $mail->Username = $offer['email_smtp_username'];
            $mail->Password = $offer['email_smtp_password'];
            $mail->SMTPSecure = $offer['email_smtp_encryption'] ?? 'tls';
            $mail->Port = $offer['email_smtp_port'] ?? 587;
            
            $mail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            
            $mail->SMTPDebug = 0;
            $mail->Debugoutput = 'error_log';
            
            $mail->setFrom($offer['email_smtp_username'], $offer['email_from_name'] ?? $offer['autohaus_name']);
            $mail->addAddress($recipientEmail);
            $mail->addReplyTo($offer['email_smtp_username'], $offer['email_from_name'] ?? $offer['autohaus_name']);
            
            $mail->isHTML(true);
            $mail->Subject = $emailSubject ?: 'Reifenangebot - ' . ($offer['customer_name'] ?? 'Kunde');
            $mail->Body = generateP2EmailBody($offer, $emailMessage);
            $mail->CharSet = 'UTF-8';
            
            $mail->addAttachment($pdfPath, $filename);
            
            $mail->send();
            
            return [
                'success' => true,
                'message' => 'E-Mail erfolgreich versendet'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'E-Mail konnte nicht versendet werden: ' . $e->getMessage()
            ];
        }
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Generiert E-Mail-Body für P2 Angebot (1:1 wie appointments.php)
 */
function generateP2EmailBody($offer, $customMessage = '') {
    $token = $offer['token'] ?? '';
    $baseUrl = getP2BaseUrl();
    $offerData = $offer['offer_data'] ?? [];
    $newTires = $offerData['new_tires'] ?? [];
    $licensePlate = $offer['license_plate'] ?? 'Nicht angegeben';
    
    // Generiere Reifenliste
    $tiresList = '';
    if (!empty($newTires)) {
        foreach ($newTires as $tire) {
            $brandModel = ($tire['brand'] ?? '') . ' ' . ($tire['model'] ?? '');
            $dimension = $tire['dimension'] ?? '';
            $price = number_format($tire['price'] ?? 0, 2, ',', '.') . ' €';
            $tiresList .= '<tr><td style="padding: 10px; border-bottom: 1px solid #e9ecef;">' . htmlspecialchars($brandModel) . ' (' . htmlspecialchars($dimension) . ')</td><td style="padding: 10px; border-bottom: 1px solid #e9ecef; text-align: right; font-weight: 600;">' . $price . '</td></tr>';
        }
    }
    
    return '
    <!DOCTYPE html>
    <html lang="de">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Reifenangebot - ' . htmlspecialchars($offer['autohaus_name']) . '</title>
        <style>
            @import url("https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap");
            
            body {
                font-family: "Inter", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
                line-height: 1.6;
                color: #2c3e50;
                background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
                margin: 0;
                padding: 20px;
            }
            
            .email-container {
                max-width: 600px;
                margin: 0 auto;
                background: white;
                border-radius: 16px;
                box-shadow: 0 10px 40px rgba(0,0,0,0.1);
                overflow: hidden;
            }
            
            .header {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 40px 30px;
                text-align: center;
                position: relative;
                overflow: hidden;
            }
            
            .header::before {
                content: "";
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: url("data:image/svg+xml,<svg xmlns=\'http://www.w3.org/2000/svg\' viewBox=\'0 0 100 100\'><defs><pattern id=\'grain\' width=\'100\' height=\'100\' patternUnits=\'userSpaceOnUse\'><circle cx=\'25\' cy=\'25\' r=\'1\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'75\' cy=\'75\' r=\'1\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'50\' cy=\'10\' r=\'0.5\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'10\' cy=\'60\' r=\'0.5\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'90\' cy=\'40\' r=\'0.5\' fill=\'white\' opacity=\'0.1\'/></pattern></defs><rect width=\'100\' height=\'100\' fill=\'url(%23grain)\'/></svg>");
                opacity: 0.3;
            }
            
            .header-content {
                position: relative;
                z-index: 1;
            }
            
            .header h1 {
                font-size: 28px;
                font-weight: 700;
                margin: 0 0 10px 0;
                text-shadow: 0 2px 4px rgba(0,0,0,0.1);
            }
            
            .header p {
                font-size: 16px;
                opacity: 0.95;
                margin: 0;
                font-weight: 400;
            }
            
            .content {
                padding: 40px 30px;
            }
            
            .greeting {
                font-size: 18px;
                color: #2c3e50;
                margin-bottom: 20px;
                font-weight: 500;
            }
            
            .custom-message {
                background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
                padding: 20px;
                border-radius: 12px;
                margin: 20px 0;
                border-left: 4px solid #1976d2;
                box-shadow: 0 4px 12px rgba(25, 118, 210, 0.1);
            }
            
            .custom-message p {
                margin: 0;
                color: #1976d2;
                font-weight: 500;
            }
            
            .offer-info {
                background: #f8f9fa;
                padding: 20px;
                border-radius: 12px;
                margin: 20px 0;
                border: 1px solid #e9ecef;
            }
            
            .offer-info h3 {
                color: #2c3e50;
                margin: 0 0 15px 0;
                font-size: 18px;
                font-weight: 600;
            }
            
            .license-plate {
                background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
                color: white;
                padding: 15px 20px;
                border-radius: 8px;
                text-align: center;
                font-size: 20px;
                font-weight: 700;
                margin: 15px 0;
                box-shadow: 0 4px 12px rgba(40, 167, 69, 0.2);
            }
            
            .price-warning {
                background: #fff3cd;
                border: 1px solid #ffeaa7;
                border-radius: 8px;
                padding: 15px;
                margin: 20px 0;
                color: #856404;
                font-size: 14px;
            }
            
            .tires-table {
                width: 100%;
                border-collapse: collapse;
                margin: 20px 0;
            }
            
            .tires-table th {
                background: #f8f9fa;
                padding: 12px;
                text-align: left;
                font-weight: 600;
                color: #2c3e50;
                border-bottom: 2px solid #dee2e6;
            }
            
            .tires-table td {
                padding: 10px;
                border-bottom: 1px solid #e9ecef;
            }
            
            .selection-card {
                background: linear-gradient(135deg, #fff 0%, #f8f9fa 100%);
                padding: 30px;
                border-radius: 16px;
                margin: 30px 0;
                border: 2px solid #007bff;
                box-shadow: 0 8px 25px rgba(0, 123, 255, 0.1);
                position: relative;
                overflow: hidden;
            }
            
            .selection-card::before {
                content: "";
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                height: 4px;
                background: linear-gradient(90deg, #007bff, #0056b3);
            }
            
            .selection-card h3 {
                color: #2c3e50;
                margin: 0 0 20px 0;
                font-size: 22px;
                font-weight: 700;
                display: flex;
                align-items: center;
                gap: 10px;
            }
            
            .selection-card h3::before {
                content: "🎯";
                font-size: 24px;
            }
            
            .selection-steps {
                display: grid;
                gap: 20px;
                margin: 20px 0;
            }
            
            .step {
                display: flex;
                align-items: flex-start;
                gap: 15px;
                padding: 15px;
                background: white;
                border-radius: 8px;
                border: 1px solid #e9ecef;
                transition: all 0.3s ease;
            }
            
            .step:hover {
                box-shadow: 0 4px 12px rgba(0,0,0,0.1);
                transform: translateY(-2px);
            }
            
            .step-icon {
                width: 40px;
                height: 40px;
                background: linear-gradient(135deg, #007bff, #0056b3);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
                font-size: 18px;
                font-weight: 700;
                flex-shrink: 0;
            }
            
            .step-content h4 {
                margin: 0 0 8px 0;
                color: #2c3e50;
                font-size: 16px;
                font-weight: 600;
            }
            
            .step-content p {
                margin: 0;
                color: #6c757d;
                font-size: 14px;
            }
            
            .cta-button {
                text-align: center;
                margin: 30px 0;
            }
            
            .cta-button a {
                background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
                color: white;
                padding: 18px 40px;
                text-decoration: none;
                border-radius: 12px;
                display: inline-block;
                font-weight: 700;
                font-size: 18px;
                box-shadow: 0 8px 25px rgba(0, 123, 255, 0.3);
                transition: all 0.3s ease;
                position: relative;
                overflow: hidden;
            }
            
            .cta-button a::before {
                content: "";
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
                transition: left 0.5s;
            }
            
            .cta-button a:hover {
                transform: translateY(-3px);
                box-shadow: 0 12px 35px rgba(0, 123, 255, 0.4);
            }
            
            .cta-button a:hover::before {
                left: 100%;
            }
            
            .disclaimer {
                background: #fff3cd;
                border: 1px solid #ffeaa7;
                border-radius: 8px;
                padding: 15px;
                margin: 20px 0;
                color: #856404;
                font-size: 14px;
            }
            
            .link-fallback {
                background: #f8f9fa;
                border: 1px solid #dee2e6;
                border-radius: 8px;
                padding: 15px;
                margin: 20px 0;
                font-size: 12px;
                word-break: break-all;
                color: #6c757d;
            }
            
            .footer {
                background: #f8f9fa;
                padding: 30px;
                text-align: center;
                border-top: 1px solid #e9ecef;
            }
            
            .footer p {
                margin: 0 0 10px 0;
                color: #6c757d;
            }
            
            .footer .signature {
                font-weight: 600;
                color: #2c3e50;
                margin: 15px 0;
            }
            
            .footer .autohaus-name {
                color: #007bff;
                font-weight: 700;
                font-size: 16px;
            }
            
            .footer .disclaimer-text {
                font-size: 12px;
                color: #6c757d;
                margin-top: 20px;
                padding-top: 20px;
                border-top: 1px solid #e9ecef;
            }
            
            @media (max-width: 600px) {
                body {
                    padding: 10px;
                }
                
                .content {
                    padding: 20px;
                }
                
                .header {
                    padding: 30px 20px;
                }
                
                .header h1 {
                    font-size: 24px;
                }
                
                .cta-button a {
                    padding: 15px 30px;
                    font-size: 16px;
                }
            }
        </style>
    </head>
    <body>
        <div class="email-container">
            <div class="header">
                <div class="header-content">
                    <h1>🚗 Reifenangebot</h1>
                    <p>Professioneller Service für Ihr Fahrzeug</p>
                </div>
            </div>
            
            <div class="content">
                <div class="greeting">
                    Sehr geehrte Damen und Herren,
                </div>
                
                ' . ($customMessage ? '<div class="custom-message"><p>' . nl2br(htmlspecialchars($customMessage)) . '</p></div>' : '<p>anbei erhalten Sie Ihr Reifenangebot für den kommenden Boxenstop.</p>') . '
                
                <div class="price-warning">
                    <strong>WICHTIGER HINWEIS ZUM PREIS:</strong><br>
                    Dieses Angebot stellt keinen verbindlichen Preis dar. Der Endpreis kann je nach aktuellen Marktpreisen variieren. Zusätzliche Arbeiten, die während der Montage erforderlich werden, können weitere Kosten verursachen. Der finale Preis wird Ihnen bei der Abholung oder nach Abschluss der Arbeiten mitgeteilt.
                </div>
                
                <div class="offer-info">
                    <h3>📋 Ihr Reifenangebot</h3>
                    <p><strong>Kennzeichen:</strong> ' . htmlspecialchars($licensePlate) . '</p>
                    ' . (!empty($offer['auftrag_nr']) ? '<p><strong>Satznummer:</strong> ' . htmlspecialchars($offer['auftrag_nr']) . '</p>' : '') . '
                    
                    ' . (!empty($tiresList) ? '
                    <h4 style="margin-top: 20px; margin-bottom: 10px;">Empfohlene Reifen:</h4>
                    <table class="tires-table">
                        <thead>
                            <tr>
                                <th>Reifen</th>
                                <th style="text-align: right;">Preis</th>
                            </tr>
                        </thead>
                        <tbody>
                            ' . $tiresList . '
                        </tbody>
                    </table>
                    ' : '') . '
                </div>
                
                ' . ($token ? '
                <div class="selection-card">
                    <h3>Kundenauswahl</h3>
                    <p style="margin-bottom: 20px; color: #6c757d;">Bitte wählen Sie aus den folgenden Optionen aus:</p>
                    
                    <div class="selection-steps">
                        <div class="step">
                            <div class="step-icon">🔧</div>
                            <div class="step-content">
                                <h4>Reifen auswählen</h4>
                                <p>Klicken Sie auf die gewünschten Reifen, um Ihre Auswahl zu treffen.</p>
                            </div>
                        </div>
                        
                        <div class="step">
                            <div class="step-icon">⚙️</div>
                            <div class="step-content">
                                <h4>Zusatzleistungen auswählen</h4>
                                <p>Wählen Sie die gewünschten Zusatzleistungen aus.</p>
                            </div>
                        </div>
                        
                        <div class="step">
                            <div class="step-icon">📝</div>
                            <div class="step-content">
                                <h4>Anmerkungen hinzufügen</h4>
                                <p>Fügen Sie optional besondere Wünsche oder Anmerkungen hinzu.</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="cta-button">
                        <a href="' . $baseUrl . '/p2_offer_selection.php?token=' . $token . '">
                            🚗 Auswahl treffen
                        </a>
                    </div>
                    
                    <div class="link-fallback">
                        <strong>Falls der Button nicht angezeigt wird, kopieren Sie bitte diesen Link:</strong><br>
                        ' . $baseUrl . '/p2_offer_selection.php?token=' . $token . '
                    </div>
                    
                    <div class="disclaimer">
                        <strong>⏰ Wichtiger Hinweis:</strong> Der Link ist 30 Tage gültig. Ihre Auswahl wird automatisch an uns übermittelt und Sie erhalten eine Bestätigung.
                    </div>
                </div>
                ' : '') . '
                
                <p style="margin-top: 30px; color: #6c757d;">Bei Fragen stehen wir Ihnen gerne zur Verfügung.</p>
            </div>
            
            <div class="footer">
                <p class="signature">Mit freundlichen Grüßen</p>
                <p class="autohaus-name">' . htmlspecialchars($offer['email_from_name'] ?? $offer['autohaus_name']) . '</p>
                <p>' . htmlspecialchars($offer['autohaus_name']) . '</p>
                <p class="disclaimer-text">
                    Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.
                </p>
            </div>
        </div>
    </body>
    </html>';
}

function getP2BaseUrl() {
    // Prüfe ob HTTPS verwendet wird
    $isHttps = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') || 
               (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') ||
               (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443);
    
    // Prüfe ob wir auf be-automobileapps.de sind
    $host = $_SERVER['HTTP_HOST'] ?? '';
    
    if (strpos($host, 'be-automobileapps.de') !== false) {
        return 'https://be-automobileapps.de';
    }
    
    // Prüfe ob wir auf 176.96.137.16 sind
    if (strpos($host, '176.96.137.16') !== false) {
        return 'http://176.96.137.16';
    }
    
    // For development, try to detect automatically
    if (strpos($host, 'localhost') !== false || 
        strpos($host, '127.0.0.1') !== false) {
        $protocol = $isHttps ? 'https' : 'http';
        
        // Check if port is already included in HTTP_HOST
        if (strpos($host, ':') !== false) {
            return $protocol . '://' . $host;
        }
        
        $port = $_SERVER['SERVER_PORT'] ?? '';
        if ($port && $port != 80 && $port != 443) {
            return $protocol . '://' . $host . ':' . $port;
        }
        
        return $protocol . '://' . $host;
    }
    
    // Default: be-automobileapps.de
    return 'https://be-automobileapps.de';
}

/**
 * Generiert PDF für P2 Angebot
 */
function generatePDFForP2Offer($offer, $offerData, $conn) {
    $pdf = "%PDF-1.4\n";
    
    // Font definition
    $pdf .= "5 0 obj\n<<\n/Type /Font\n/Subtype /Type1\n/BaseFont /Helvetica\n>>\nendobj\n\n";
    
    // Catalog
    $pdf .= "1 0 obj\n<<\n/Type /Catalog\n/Pages 2 0 R\n>>\nendobj\n\n";
    
    // Pages
    $pdf .= "2 0 obj\n<<\n/Type /Pages\n/Kids [3 0 R]\n/Count 1\n>>\nendobj\n\n";
    
    $pdfContent = "";
    $yPosition = 750;
    
    // Header
    $pdfContent .= "BT\n/F1 16 Tf\n50 " . $yPosition . " Td\n(B&E Autocentrum GmbH) Tj\nET\n";
    $yPosition -= 20;
    $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(Kiel, " . date('d.m.Y') . ") Tj\nET\n";
    $yPosition -= 30;
    
    // Greeting
    $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(Sehr geehrte Damen und Herren,) Tj\nET\n";
    $yPosition -= 30;
    
    // Default message if no letter content
    if (empty($letterContent)) {
        $defaultMessage = "die Vorbereitungen für unseren nächsten Boxenstopp laufen bereits und dabei ist uns leider aufgefallen, dass die bei uns eingelagerten Reifen ersetzt werden müssten.\n\nDaher erhalten Sie mit diesem Schreiben das beigefügte Angebot.";
        $letterContent = $defaultMessage;
    }
    
    // Letter content
    $letterContent = strip_tags($offer['letter_content'] ?? '');
    $lines = explode("\n", $letterContent);
    foreach ($lines as $line) {
        if (trim($line)) {
            $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(" . escapePDFText($line) . ") Tj\nET\n";
            $yPosition -= 20;
        } else {
            $yPosition -= 10;
        }
    }
    
    $yPosition -= 20;
    
    // New tires section
    if (!empty($offerData['new_tires'])) {
        $pdfContent .= "BT\n/F1 14 Tf\n50 " . $yPosition . " Td\n(Neue Reifen:) Tj\nET\n";
        $yPosition -= 25;
        
        foreach ($offerData['new_tires'] as $tire) {
            $tireText = "- " . ($tire['brand_model'] ?? '') . " (" . ($tire['dimension'] ?? '') . ") - " . 
                       ($tire['quantity'] ?? 1) . "x " . number_format($tire['total'] ?? 0, 2, ',', '.') . "€";
            $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(" . escapePDFText($tireText) . ") Tj\nET\n";
            $yPosition -= 20;
        }
    }
    
    $yPosition -= 20;
    
    // Services section
    if (!empty($offerData['selected_services'])) {
        $pdfContent .= "BT\n/F1 14 Tf\n50 " . $yPosition . " Td\n(Zusatzleistungen:) Tj\nET\n";
        $yPosition -= 25;
        
        foreach ($offerData['selected_services'] as $serviceId) {
            if ($serviceId === 'mounting') {
                $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(Rädermontage                    49,99€) Tj\nET\n";
            } elseif ($serviceId === 'storage') {
                $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(Einlagerung der Sommerreifen     85,00€) Tj\nET\n";
            }
            $yPosition -= 20;
        }
    }
    
    $yPosition -= 30;
    
    // Signature
    $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(Mit freundlichem Gruß) Tj\nET\n";
    $yPosition -= 20;
    $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(" . escapePDFText($offer['employee_name'] ?? '') . ") Tj\nET\n";
    $yPosition -= 20;
    $pdfContent .= "BT\n/F1 12 Tf\n50 " . $yPosition . " Td\n(B&E Autocentrum GmbH) Tj\nET\n";
    $yPosition -= 40;
    $pdfContent .= "BT\n/F1 10 Tf\n50 " . $yPosition . " Td\n(Dieses Dokument wurde maschinell erstellt und ist ohne Unterschrift gültig) Tj\nET\n";
    
    $pdf .= "3 0 obj\n<<\n/Type /Page\n/Parent 2 0 R\n/MediaBox [0 0 612 792]\n/Contents 4 0 R\n/Resources <<\n/Font <<\n/F1 5 0 R\n>>\n>>\n>>\nendobj\n\n";
    $pdf .= "4 0 obj\n<<\n/Length " . strlen($pdfContent) . "\n>>\nstream\n" . $pdfContent . "endstream\nendobj\n\n";
    $pdf .= "xref\n0 6\n0000000000 65535 f \n0000000009 00000 n \n0000000058 00000 n \n0000000115 00000 n \n0000000204 00000 n \n0000000300 00000 n \n";
    $pdf .= "trailer\n<<\n/Size 6\n/Root 1 0 R\n>>\nstartxref\n" . strlen($pdf) . "\n%%EOF\n";
    
    return $pdf;
}

/**
 * Escaped Text für PDF
 */
function escapePDFText($text) {
    $text = mb_convert_encoding($text, 'UTF-8', 'UTF-8');
    $replacements = [
        'ä' => chr(228), 'ö' => chr(246), 'ü' => chr(252),
        'Ä' => chr(196), 'Ö' => chr(214), 'Ü' => chr(220),
        'ß' => chr(223), '€' => chr(128)
    ];
    foreach ($replacements as $char => $replacement) {
        $text = str_replace($char, $replacement, $text);
    }
    $text = str_replace(['(', ')', '\\'], ['\\(', '\\)', '\\\\'], $text);
    return $text;
}

