<?php
/**
 * API zum Speichern und Abrufen von P2 Teilenummer-Mappings
 * Speichert die Zuordnung zwischen Reifen-Identifikatoren und P2 Teilenummern
 * 
 * Sicherheitsfeatures:
 * - Transaktionen für atomare Operationen
 * - Normalisierung aller Teilenummern (Großbuchstaben, keine Leerzeichen)
 * - Unique Constraints in der Datenbank
 * - Prepared Statements gegen SQL-Injection
 * - Validierung aller Eingabedaten
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';

/**
 * Normalisiert eine Teilenummer (Großbuchstaben, keine Leerzeichen)
 */
function normalizePartNumber($partNumber) {
    if (empty($partNumber)) {
        return null;
    }
    return strtoupper(trim(str_replace(' ', '', $partNumber)));
}

/**
 * Generiert alle Suchvarianten für eine Teilenummer (mit/ohne RE-Präfix)
 */
function generateSearchVariants($partNumber) {
    if (empty($partNumber)) {
        return [];
    }
    
    $normalized = normalizePartNumber($partNumber);
    $variants = [$normalized];
    
    // Variante mit RE-Präfix
    if (substr($normalized, 0, 2) !== 'RE') {
        $variants[] = 'RE' . $normalized;
    }
    
    // Variante ohne RE-Präfix
    if (substr($normalized, 0, 2) === 'RE' && strlen($normalized) > 2) {
        $variants[] = substr($normalized, 2);
    }
    
    return array_unique($variants);
}

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    // Erstelle Tabelle falls nicht vorhanden mit Unique Constraints
    $createTableSql = "
        CREATE TABLE IF NOT EXISTS `p2_teilenr_mapping` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `ean` varchar(50) DEFAULT NULL,
            `oem_part_number` varchar(100) DEFAULT NULL,
            `manufacturer_number` varchar(100) DEFAULT NULL,
            `part_number` varchar(100) DEFAULT NULL,
            `p2_teilenr` varchar(100) NOT NULL,
            `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
            `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
            PRIMARY KEY (`id`),
            UNIQUE KEY `unique_ean` (`ean`),
            UNIQUE KEY `unique_oem_part_number` (`oem_part_number`),
            UNIQUE KEY `unique_manufacturer_number` (`manufacturer_number`),
            UNIQUE KEY `unique_part_number` (`part_number`),
            KEY `idx_p2_teilenr` (`p2_teilenr`),
            KEY `idx_ean` (`ean`),
            KEY `idx_oem_part_number` (`oem_part_number`),
            KEY `idx_manufacturer_number` (`manufacturer_number`),
            KEY `idx_part_number` (`part_number`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    $conn->exec($createTableSql);
    
    $method = $_SERVER['REQUEST_METHOD'];
    
    // GET: Hole P2 Teilenummer
    if ($method === 'GET') {
        $ean = isset($_GET['ean']) ? normalizePartNumber($_GET['ean']) : null;
        $oemPartNumber = isset($_GET['oem_part_number']) ? normalizePartNumber($_GET['oem_part_number']) : null;
        $manufacturerNumber = isset($_GET['manufacturer_number']) ? normalizePartNumber($_GET['manufacturer_number']) : null;
        $partNumber = isset($_GET['part_number']) ? normalizePartNumber($_GET['part_number']) : null;
        
        if (!$ean && !$oemPartNumber && !$manufacturerNumber && !$partNumber) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => 'Mindestens ein Identifikator (ean, oem_part_number, manufacturer_number, part_number) erforderlich'
            ]);
            exit;
        }
        
        // Generiere alle Suchvarianten
        $searchVariants = [];
        
        if ($ean) {
            $searchVariants[] = ['field' => 'ean', 'value' => $ean];
        }
        
        if ($oemPartNumber) {
            $variants = generateSearchVariants($oemPartNumber);
            foreach ($variants as $variant) {
                $searchVariants[] = ['field' => 'oem_part_number', 'value' => $variant];
            }
        }
        
        if ($manufacturerNumber) {
            $variants = generateSearchVariants($manufacturerNumber);
            foreach ($variants as $variant) {
                $searchVariants[] = ['field' => 'manufacturer_number', 'value' => $variant];
            }
        }
        
        if ($partNumber) {
            $variants = generateSearchVariants($partNumber);
            foreach ($variants as $variant) {
                $searchVariants[] = ['field' => 'part_number', 'value' => $variant];
            }
        }
        
        // Entferne Duplikate
        $uniqueVariants = [];
        $seen = [];
        foreach ($searchVariants as $variant) {
            $key = $variant['field'] . '|' . $variant['value'];
            if (!isset($seen[$key]) && !empty($variant['value'])) {
                $seen[$key] = true;
                $uniqueVariants[] = $variant;
            }
        }
        
        if (empty($uniqueVariants)) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => 'Keine gültigen Suchkriterien'
            ]);
            exit;
        }
        
        // Baue SQL-Query mit allen Varianten
        $conditions = [];
        $params = [];
        foreach ($uniqueVariants as $variant) {
            $conditions[] = $variant['field'] . ' = ?';
            $params[] = $variant['value'];
        }
        
        $sql = "SELECT p2_teilenr, created_at, updated_at FROM p2_teilenr_mapping WHERE " . implode(' OR ', $conditions) . " LIMIT 1";
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            echo json_encode([
                'success' => true,
                'p2_teilenr' => $result['p2_teilenr'],
                'created_at' => $result['created_at'],
                'updated_at' => $result['updated_at']
            ]);
        } else {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'error' => 'Keine P2 Teilenummer gefunden'
            ]);
        }
        exit;
    }
    
    // POST: Speichere P2 Teilenummer
    if ($method === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input || !isset($input['p2_teilenr'])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => 'p2_teilenr ist erforderlich'
            ]);
            exit;
        }
        
        // Normalisiere und validiere P2 Teilenummer
        $p2Teilenr = normalizePartNumber($input['p2_teilenr']);
        if (empty($p2Teilenr) || strlen($p2Teilenr) < 3) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => 'p2_teilenr muss mindestens 3 Zeichen lang sein'
            ]);
            exit;
        }
        
        // Normalisiere alle Eingabewerte
        $ean = isset($input['ean']) ? normalizePartNumber($input['ean']) : null;
        $oemPartNumber = isset($input['oem_part_number']) ? normalizePartNumber($input['oem_part_number']) : null;
        $manufacturerNumber = isset($input['manufacturer_number']) ? normalizePartNumber($input['manufacturer_number']) : null;
        $partNumber = isset($input['part_number']) ? normalizePartNumber($input['part_number']) : null;
        
        if (!$ean && !$oemPartNumber && !$manufacturerNumber && !$partNumber) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => 'Mindestens ein Identifikator (ean, oem_part_number, manufacturer_number, part_number) erforderlich'
            ]);
            exit;
        }
        
        // Beginne Transaktion für atomare Operation
        $conn->beginTransaction();
        
        try {
            // Prüfe ob bereits ein Mapping existiert (mit allen Varianten)
            $checkVariants = [];
            if ($ean) {
                $checkVariants[] = ['field' => 'ean', 'value' => $ean];
            }
            if ($oemPartNumber) {
                $variants = generateSearchVariants($oemPartNumber);
                foreach ($variants as $variant) {
                    $checkVariants[] = ['field' => 'oem_part_number', 'value' => $variant];
                }
            }
            if ($manufacturerNumber) {
                $variants = generateSearchVariants($manufacturerNumber);
                foreach ($variants as $variant) {
                    $checkVariants[] = ['field' => 'manufacturer_number', 'value' => $variant];
                }
            }
            if ($partNumber) {
                $variants = generateSearchVariants($partNumber);
                foreach ($variants as $variant) {
                    $checkVariants[] = ['field' => 'part_number', 'value' => $variant];
                }
            }
            
            $conditions = [];
            $params = [];
            foreach ($checkVariants as $variant) {
                $conditions[] = $variant['field'] . ' = ?';
                $params[] = $variant['value'];
            }
            
            $checkSql = "SELECT id, p2_teilenr FROM p2_teilenr_mapping WHERE " . implode(' OR ', $conditions) . " LIMIT 1";
            $checkStmt = $conn->prepare($checkSql);
            $checkStmt->execute($params);
            $existing = $checkStmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existing) {
                // Update vorhandenes Mapping, wenn P2 Teilenummer sich geändert hat
                if ($existing['p2_teilenr'] !== $p2Teilenr) {
                    $updateSql = "UPDATE p2_teilenr_mapping SET p2_teilenr = ?, updated_at = NOW() WHERE id = ?";
                    $updateStmt = $conn->prepare($updateSql);
                    $updateStmt->execute([$p2Teilenr, $existing['id']]);
                }
                
                $conn->commit();
                
                echo json_encode([
                    'success' => true,
                    'message' => 'P2 Teilenummer aktualisiert',
                    'p2_teilenr' => $p2Teilenr,
                    'updated' => true
                ]);
            } else {
                // Neues Mapping erstellen - speichere alle Varianten
                $insertSql = "INSERT INTO p2_teilenr_mapping (ean, oem_part_number, manufacturer_number, part_number, p2_teilenr) VALUES (?, ?, ?, ?, ?)";
                $insertStmt = $conn->prepare($insertSql);
                
                // Speichere Haupt-Mapping
                $insertStmt->execute([$ean, $oemPartNumber, $manufacturerNumber, $partNumber, $p2Teilenr]);
                
                // Speichere zusätzliche Varianten (mit/ohne RE-Präfix) für bessere Suche
                $p2TeilenrUpper = strtoupper($p2Teilenr);
                $p2TeilenrWithoutRE = null;
                
                if (substr($p2TeilenrUpper, 0, 2) === 'RE' && strlen($p2TeilenrUpper) > 2) {
                    $p2TeilenrWithoutRE = substr($p2TeilenrUpper, 2);
                }
                
                // Speichere Varianten für oem_part_number
                if ($oemPartNumber) {
                    $variants = generateSearchVariants($oemPartNumber);
                    foreach ($variants as $variant) {
                        if ($variant !== $oemPartNumber) {
                            try {
                                $insertStmt->execute([$ean, $variant, null, null, $p2Teilenr]);
                            } catch (PDOException $e) {
                                // Ignoriere Duplikat-Fehler (Unique Constraint)
                                if ($e->getCode() !== '23000') {
                                    throw $e;
                                }
                            }
                        }
                    }
                }
                
                // Speichere Varianten für manufacturer_number
                if ($manufacturerNumber) {
                    $variants = generateSearchVariants($manufacturerNumber);
                    foreach ($variants as $variant) {
                        if ($variant !== $manufacturerNumber) {
                            try {
                                $insertStmt->execute([$ean, null, $variant, null, $p2Teilenr]);
                            } catch (PDOException $e) {
                                // Ignoriere Duplikat-Fehler (Unique Constraint)
                                if ($e->getCode() !== '23000') {
                                    throw $e;
                                }
                            }
                        }
                    }
                }
                
                // Speichere Varianten für part_number
                if ($partNumber) {
                    $variants = generateSearchVariants($partNumber);
                    foreach ($variants as $variant) {
                        if ($variant !== $partNumber) {
                            try {
                                $insertStmt->execute([$ean, null, null, $variant, $p2Teilenr]);
                            } catch (PDOException $e) {
                                // Ignoriere Duplikat-Fehler (Unique Constraint)
                                if ($e->getCode() !== '23000') {
                                    throw $e;
                                }
                            }
                        }
                    }
                }
                
                $conn->commit();
                
                echo json_encode([
                    'success' => true,
                    'message' => 'P2 Teilenummer gespeichert',
                    'p2_teilenr' => $p2Teilenr,
                    'created' => true
                ]);
            }
        } catch (PDOException $e) {
            $conn->rollBack();
            error_log("P2 Teilenummer Mapping PDO Error: " . $e->getMessage());
            
            // Prüfe auf Duplikat-Fehler
            if ($e->getCode() === '23000') {
                http_response_code(409);
                echo json_encode([
                    'success' => false,
                    'error' => 'Mapping existiert bereits'
                ]);
            } else {
                throw $e;
            }
        }
        exit;
    }
    
    // Unbekannte Methode
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Unbekannte Anfrage'
    ]);
    
} catch (PDOException $e) {
    error_log("P2 Teilenummer Mapping PDO Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Datenbankfehler'
    ]);
} catch (Exception $e) {
    error_log("P2 Teilenummer Mapping Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Fehler: ' . $e->getMessage()
    ]);
}
