<?php
/**
 * Proxy-Endpoint für P2 Teile-API
 * Umgeht CORS-Probleme durch serverseitige Anfragen
 */

// Unterdrücke alle Ausgaben außer JSON
ob_start();
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Teile-Token, X-Teile-Edit-Token');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ob_end_clean();
    exit(0);
}

// Error Handler
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    if ($errno === E_WARNING || $errno === E_NOTICE) {
        error_log("P2 Teile Proxy Warning: $errstr in $errfile:$errline");
        return true;
    }
    ob_clean();
    echo json_encode([
        'success' => false,
        'error' => 'PHP Fehler: ' . $errstr
    ]);
    exit;
});

try {
    $baseUrl = 'http://100.82.207.72:5001/api/teile';
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    
    // GET: Hole Teil
    if ($method === 'GET' && isset($_GET['teilenr'])) {
        $teilenr = $_GET['teilenr'];
        $url = $baseUrl . '/' . urlencode($teilenr);
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json'
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        ob_clean();
        
        if ($error) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Verbindungsfehler: ' . $error
            ]);
            exit;
        }
        
        // WICHTIG: Setze HTTP-Status-Code BEVOR JSON ausgegeben wird
        http_response_code($httpCode);
        
        $data = json_decode($response, true);
        
        // Wenn 404 und keine Daten, erstelle Standard-404-Response
        if ($httpCode === 404 && $data === null) {
            $data = [
                'success' => false,
                'error' => "Teil {$teilenr} nicht gefunden"
            ];
        } elseif ($data === null && $httpCode !== 404) {
            // Nur bei nicht-404 Fehlern einen Parse-Fehler melden
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Ungültige API-Antwort'
            ]);
            exit;
        }
        
        echo json_encode($data);
        exit;
    }
    
    // POST: Neues Teil anlegen
    if ($method === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            ob_clean();
            echo json_encode([
                'success' => false,
                'error' => 'Keine Daten übermittelt'
            ]);
            exit;
        }
        
        // Token aus Header holen
        $token = $_SERVER['HTTP_X_TEILE_TOKEN'] ?? '';
        if (empty($token)) {
            $token = 'bungert-teile-api-token-2024-default'; // Fallback
        }
        
        $url = $baseUrl;
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($input),
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-Teile-Token: ' . $token
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        ob_clean();
        
        if ($error) {
            echo json_encode([
                'success' => false,
                'error' => 'Verbindungsfehler: ' . $error
            ]);
            exit;
        }
        
        $data = json_decode($response, true);
        if ($data === null) {
            echo json_encode([
                'success' => false,
                'error' => 'Ungültige API-Antwort'
            ]);
            exit;
        }
        
        http_response_code($httpCode);
        echo json_encode($data);
        exit;
    }
    
    // PUT: Teil ändern
    if ($method === 'PUT' && isset($_GET['teilenr'])) {
        $teilenr = $_GET['teilenr'];
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            ob_clean();
            echo json_encode([
                'success' => false,
                'error' => 'Keine Daten übermittelt'
            ]);
            exit;
        }
        
        // Edit-Token aus Header holen
        $token = $_SERVER['HTTP_X_TEILE_EDIT_TOKEN'] ?? '';
        if (empty($token)) {
            $token = 'bungert-teile-edit-token-2024-default'; // Fallback
        }
        
        $url = $baseUrl . '/' . urlencode($teilenr);
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'PUT',
            CURLOPT_POSTFIELDS => json_encode($input),
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-Teile-Edit-Token: ' . $token
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        ob_clean();
        
        if ($error) {
            echo json_encode([
                'success' => false,
                'error' => 'Verbindungsfehler: ' . $error
            ]);
            exit;
        }
        
        $data = json_decode($response, true);
        if ($data === null) {
            echo json_encode([
                'success' => false,
                'error' => 'Ungültige API-Antwort'
            ]);
            exit;
        }
        
        http_response_code($httpCode);
        echo json_encode($data);
        exit;
    }
    
    // Unbekannte Methode
    ob_clean();
    echo json_encode([
        'success' => false,
        'error' => 'Unbekannte Anfrage'
    ]);
    http_response_code(400);
    
} catch (Exception $e) {
    ob_clean();
    error_log("P2 Teile Proxy Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Fehler: ' . $e->getMessage()
    ]);
    http_response_code(500);
} catch (Throwable $e) {
    ob_clean();
    error_log("P2 Teile Proxy Fatal Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Kritischer Fehler: ' . $e->getMessage()
    ]);
    http_response_code(500);
}

