<?php
/**
 * P2 API Proxy
 * Proxy für P2 API-Aufrufe von PHP
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';
require_once dirname(__DIR__) . '/admin/includes/p2_api.php';

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? null;

try {
    if (!$autohausId) {
        throw new Exception('autohaus_id ist erforderlich');
    }

    $api = new P2API($autohausId);

    switch ($action) {
        case 'search_customers':
            $searchTerm = $_GET['search'] ?? '';
            $limit = $_GET['limit'] ?? 1000; // Erhöhtes Limit für vollständige Ladung
            $page = $_GET['page'] ?? 1;
            $customers = $api->searchCustomers($searchTerm, $limit);
            
            // Keine Pagination auf API-Ebene - Frontend macht das
            echo json_encode([
                'success' => true,
                'data' => $customers,
                'count' => count($customers),
                'total' => count($customers)
            ]);
            break;

        case 'get_customer':
            $kundenAdrId = $_GET['kunden_adr_id'] ?? null;
            if (!$kundenAdrId) {
                throw new Exception('kunden_adr_id ist erforderlich');
            }
            $customer = $api->getCustomerById($kundenAdrId);
            if ($customer) {
                echo json_encode([
                    'success' => true,
                    'data' => $customer
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'error' => 'Kunde nicht gefunden'
                ]);
            }
            break;

        case 'get_orders_by_customer':
            $kundenAdrId = $_GET['kunden_adr_id'] ?? null;
            if (!$kundenAdrId) {
                throw new Exception('kunden_adr_id ist erforderlich');
            }
            $days = $_GET['days'] ?? 2000;
            $limit = $_GET['limit'] ?? 500;
            $orders = $api->getOrdersByCustomer($kundenAdrId, $days, $limit);
            echo json_encode([
                'success' => true,
                'data' => $orders,
                'count' => count($orders)
            ]);
            break;

        case 'get_orders_by_license':
            $licensePlate = $_GET['license_plate'] ?? null;
            if (!$licensePlate) {
                throw new Exception('license_plate ist erforderlich');
            }
            $days = $_GET['days'] ?? 2000;
            $limit = $_GET['limit'] ?? 500;
            
            // Debug: Logge die Suche
            error_log("P2_API_PROXY: Suche nach Kennzeichen: '$licensePlate', days: $days, limit: $limit");
            
            $orders = $api->getOrdersByLicensePlate($licensePlate, $days, $limit);
            
            // Debug: Logge das Ergebnis
            error_log("P2_API_PROXY: Kennzeichen-Suche Ergebnis: " . count($orders) . " Aufträge gefunden");
            if (count($orders) === 0) {
                error_log("P2_API_PROXY: Keine Aufträge für Kennzeichen '$licensePlate' gefunden");
            }
            
            echo json_encode([
                'success' => true,
                'data' => $orders,
                'count' => count($orders),
                'debug' => [
                    'license_plate' => $licensePlate,
                    'days' => $days,
                    'limit' => $limit
                ]
            ]);
            break;

        case 'get_orders_by_vin':
            $vin = $_GET['vin'] ?? null;
            if (!$vin) {
                throw new Exception('vin ist erforderlich');
            }
            $days = $_GET['days'] ?? 2000;
            $limit = $_GET['limit'] ?? 500;
            $orders = $api->getOrdersByVin($vin, $days, $limit);
            echo json_encode([
                'success' => true,
                'data' => $orders,
                'count' => count($orders)
            ]);
            break;

        case 'get_orders_by_vin_last6':
            $vinLast6 = $_GET['vin_last6'] ?? null;
            if (!$vinLast6) {
                throw new Exception('vin_last6 ist erforderlich');
            }
            $days = $_GET['days'] ?? 2000;
            $limit = $_GET['limit'] ?? 500;
            $orders = $api->getOrdersByVinLast6($vinLast6, $days, $limit);
            echo json_encode([
                'success' => true,
                'data' => $orders,
                'count' => count($orders)
            ]);
            break;

        case 'get_customer_by_id_with_orders':
            $kundenAdrId = $_GET['kunden_adr_id'] ?? null;
            if (!$kundenAdrId) {
                throw new Exception('kunden_adr_id ist erforderlich');
            }
            $customer = $api->getCustomerById($kundenAdrId);
            if ($customer) {
                $days = $_GET['days'] ?? 2000;
                $limit = $_GET['limit'] ?? 500;
                $orders = $api->getOrdersByCustomer($kundenAdrId, $days, $limit);
                echo json_encode([
                    'success' => true,
                    'data' => [
                        'customer' => $customer,
                        'orders' => $orders
                    ],
                    'count' => count($orders)
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'error' => 'Kunde nicht gefunden',
                    'data' => [
                        'customer' => null,
                        'orders' => []
                    ],
                    'count' => 0
                ]);
            }
            break;

        case 'get_orders_by_order_number':
            $orderNumber = $_GET['order_number'] ?? null;
            if (!$orderNumber) {
                throw new Exception('order_number ist erforderlich');
            }
            $limit = $_GET['limit'] ?? 10;
            $orders = $api->getOrdersByOrderNumber($orderNumber, $limit);
            echo json_encode([
                'success' => true,
                'data' => $orders,
                'count' => count($orders)
            ]);
            break;

        case 'get_order':
            $auftragNr = $_GET['auftrag_nr'] ?? null;
            if (!$auftragNr) {
                throw new Exception('auftrag_nr ist erforderlich');
            }
            $order = $api->getOrderById($auftragNr);
            if ($order) {
                echo json_encode([
                    'success' => true,
                    'data' => $order
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'error' => 'Auftrag nicht gefunden'
                ]);
            }
            break;

        case 'get_order_positions':
            $auftragNr = $_GET['auftrag_nr'] ?? null;
            if (!$auftragNr) {
                throw new Exception('auftrag_nr ist erforderlich');
            }
            $positions = $api->getOrderPositions($auftragNr);
            echo json_encode([
                'success' => true,
                'data' => $positions,
                'count' => count($positions)
            ]);
            break;

        case 'get_all_orders':
            $days = $_GET['days'] ?? 2000;
            $limit = $_GET['limit'] ?? 2000;
            $result = $api->apiCall('/auftraege', 'GET', [
                'days' => $days,
                'limit' => $limit
            ]);
            if (isset($result['success']) && $result['success']) {
                echo json_encode([
                    'success' => true,
                    'data' => $result['data'] ?? [],
                    'count' => count($result['data'] ?? [])
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'error' => 'Fehler beim Laden der Aufträge',
                    'data' => []
                ]);
            }
            break;

        case 'health_check':
            $isHealthy = $api->healthCheck();
            echo json_encode([
                'success' => true,
                'healthy' => $isHealthy
            ]);
            break;

        default:
            throw new Exception('Unbekannte Aktion: ' . $action);
    }

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

