<?php
/**
 * GET /api/get_events.php
 * Liefert alle aktiven Events für Client-Apps (iOS/Web)
 * 
 * Query-Parameter (optional):
 * - autohaus_id: Filter nach Autohaus-ID
 * - date_from: Filter ab Datum (YYYY-MM-DD)
 * - date_to: Filter bis Datum (YYYY-MM-DD)
 * - limit: Anzahl der Ergebnisse (default: 100, max: 1000)
 * - offset: Pagination-Offset (default: 0)
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Api-Key');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();

    // Query-Parameter auslesen und validieren
    $autohaus_id = !empty($_GET['autohaus_id']) ? (int)$_GET['autohaus_id'] : null;
    $date_from = $_GET['date_from'] ?? null;
    $date_to = $_GET['date_to'] ?? null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 100;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;

    // Limit validieren (max 1000)
    if ($limit > 1000) {
        $limit = 1000;
    }
    if ($limit < 1) {
        $limit = 1;
    }
    if ($offset < 0) {
        $offset = 0;
    }

    // Datum-Format validieren
    if ($date_from && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_from)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Ungültiges Datumsformat für date_from. Erwartet: YYYY-MM-DD'
        ]);
        exit;
    }

    if ($date_to && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_to)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Ungültiges Datumsformat für date_to. Erwartet: YYYY-MM-DD'
        ]);
        exit;
    }

    // SQL Query bauen
    $sql = "
        SELECT 
            e.*, 
            a.name as autohaus_name, 
            a.short_name as autohaus_short_name,
            a.address as autohaus_address,
            a.phone as autohaus_phone,
            a.email as autohaus_email,
            (SELECT COUNT(*) FROM appointments ap 
             WHERE ap.event_id=e.id 
             AND ap.status IN ('booked','vorbereitet')) AS booked_count
        FROM events e 
        LEFT JOIN autohaus a ON e.autohaus_id = a.id
        WHERE e.is_active = 1 
          AND e.event_date >= CURDATE()
    ";

    $params = [];

    // Filter hinzufügen
    if ($autohaus_id !== null) {
        $sql .= " AND e.autohaus_id = ?";
        $params[] = $autohaus_id;
    }

    if ($date_from) {
        $sql .= " AND e.event_date >= ?";
        $params[] = $date_from;
    }

    if ($date_to) {
        $sql .= " AND e.event_date <= ?";
        $params[] = $date_to;
    }

    // Sortierung
    $sql .= " ORDER BY e.event_date ASC, e.start_time ASC";

    // Pagination (LIMIT und OFFSET müssen direkt im SQL stehen, können nicht als Platzhalter verwendet werden)
    $sql .= " LIMIT " . (int)$limit . " OFFSET " . (int)$offset;

    // Query ausführen
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Events verarbeiten und erweiterte Felder berechnen
    $processedEvents = [];
    
    foreach ($events as $event) {
        // Total Slots berechnen (basierend auf index.php Zeile 1003)
        $start_time = strtotime($event['start_time']);
        $end_time = strtotime($event['end_time']);
        $slot_duration_minutes = (int)$event['slot_duration'];
        $max_monteure = (int)$event['max_monteure'];
        
        if ($start_time && $end_time && $slot_duration_minutes > 0) {
            $time_diff_seconds = $end_time - $start_time;
            $slots_per_time = floor($time_diff_seconds / ($slot_duration_minutes * 60));
            $total_slots = $slots_per_time * $max_monteure;
        } else {
            $total_slots = 0;
        }

        $booked_count = (int)($event['booked_count'] ?? 0);
        $available_slots = max(0, $total_slots - $booked_count);
        $utilization_percent = $total_slots > 0 ? round(($booked_count / $total_slots) * 100) : 0;

        // Nächsten freien Slot ermitteln (basierend auf index.php Zeilen 1007-1018)
        $next_free_slot = null;
        if ($event['event_date'] && $start_time && $end_time) {
            $countsStmt = $conn->prepare("
                SELECT slot_time, COUNT(*) as c 
                FROM appointments 
                WHERE event_id = ? 
                  AND slot_date = ? 
                  AND status IN ('booked','vorbereitet') 
                GROUP BY slot_time
            ");
            $countsStmt->execute([$event['id'], $event['event_date']]);
            $counts = [];
            foreach ($countsStmt->fetchAll() as $row) {
                $counts[$row['slot_time']] = (int)$row['c'];
            }
            
            $slotDur = $slot_duration_minutes * 60;
            for ($t = $start_time; $t < $end_time; $t += $slotDur) {
                $tStr = date('H:i:s', $t);
                $booked_at_slot = $counts[$tStr] ?? 0;
                if ($booked_at_slot < $max_monteure) {
                    $next_free_slot = date('H:i', $t);
                    break;
                }
            }
        }

        // Event-Objekt zusammenstellen
        $processedEvents[] = [
            'id' => (int)$event['id'],
            'name' => $event['name'],
            'description' => $event['description'] ?? null,
            'event_date' => $event['event_date'],
            'event_time' => $event['event_time'] ?? $event['start_time'],
            'start_time' => $event['start_time'],
            'end_time' => $event['end_time'],
            'slot_duration' => (int)$event['slot_duration'],
            'max_monteure' => (int)$event['max_monteure'],
            'pause_start' => $event['pause_start'] ?? null,
            'pause_end' => $event['pause_end'] ?? null,
            'blocked_start' => $event['blocked_start'] ?? null,
            'blocked_end' => $event['blocked_end'] ?? null,
            'autohaus_id' => $event['autohaus_id'] ? (int)$event['autohaus_id'] : null,
            'autohaus_name' => $event['autohaus_name'],
            'autohaus_short_name' => $event['autohaus_short_name'],
            'autohaus_address' => $event['autohaus_address'],
            'autohaus_phone' => $event['autohaus_phone'],
            'autohaus_email' => $event['autohaus_email'],
            'is_active' => (bool)$event['is_active'],
            'booked_count' => $booked_count,
            'total_slots' => $total_slots,
            'available_slots' => $available_slots,
            'utilization_percent' => $utilization_percent,
            'next_free_slot' => $next_free_slot,
            'created_at' => $event['created_at'] ?? null,
            'updated_at' => $event['updated_at'] ?? null
        ];
    }

    // Gesamtanzahl für Pagination ermitteln (ohne LIMIT)
    $countSql = "
        SELECT COUNT(*) 
        FROM events e 
        WHERE e.is_active = 1 
          AND e.event_date >= CURDATE()
    ";
    $countParams = [];

    if ($autohaus_id !== null) {
        $countSql .= " AND e.autohaus_id = ?";
        $countParams[] = $autohaus_id;
    }

    if ($date_from) {
        $countSql .= " AND e.event_date >= ?";
        $countParams[] = $date_from;
    }

    if ($date_to) {
        $countSql .= " AND e.event_date <= ?";
        $countParams[] = $date_to;
    }

    $countStmt = $conn->prepare($countSql);
    $countStmt->execute($countParams);
    $total_count = (int)$countStmt->fetchColumn();

    // Response zusammenstellen
    $has_more = ($offset + $limit) < $total_count;

    echo json_encode([
        'success' => true,
        'events' => $processedEvents,
        'total_count' => $total_count,
        'meta' => [
            'limit' => $limit,
            'offset' => $offset,
            'has_more' => $has_more,
            'returned_count' => count($processedEvents)
        ]
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Exception $e) {
    error_log("Error in get_events.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Interner Serverfehler beim Laden der Events',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}
?>

