<?php
/**
 * Kurze Kundenübersicht mit Aktivitäten für Admin Dashboard
 */

// Debug-Logging aktivieren
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
$log_file = __DIR__ . '/../admin/api/php_errors.log';
ini_set('error_log', $log_file);

session_start();

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

error_log("🔍 [get_customer_profile] Request gestartet - Method: " . $_SERVER['REQUEST_METHOD']);
error_log("🔍 [get_customer_profile] GET-Parameter: " . print_r($_GET, true));
error_log("🔍 [get_customer_profile] Session admin_logged_in: " . (isset($_SESSION['admin_logged_in']) ? ($_SESSION['admin_logged_in'] ? 'true' : 'false') : 'nicht gesetzt'));

// Admin-Check
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    error_log("❌ [get_customer_profile] Admin-Check fehlgeschlagen");
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Nicht autorisiert']);
    exit;
}

require_once __DIR__ . '/../config/database.php';
error_log("✅ [get_customer_profile] Database-Datei geladen");

try {
    error_log("🔍 [get_customer_profile] Starte Datenbankverbindung");
    $db = new Database();
    $conn = $db->getConnection();
    error_log("✅ [get_customer_profile] Datenbankverbindung erfolgreich");
    
    $user_id = intval($_GET['user_id'] ?? 0);
    error_log("🔍 [get_customer_profile] User-ID: " . $user_id);
    
    if ($user_id <= 0) {
        error_log("❌ [get_customer_profile] Ungültige User-ID: " . $user_id);
        throw new Exception('Ungültige Benutzer-ID');
    }
    
    // Benutzer-Daten laden
    error_log("🔍 [get_customer_profile] Lade Benutzer-Daten für ID: " . $user_id);
    $stmt = $conn->prepare("
        SELECT 
            id, first_name, last_name, email, phone_number, 
            verification_status, created_at, profile_image,
            registration_source
        FROM users 
        WHERE id = ?
    ");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        error_log("❌ [get_customer_profile] Benutzer nicht gefunden für ID: " . $user_id);
        throw new Exception('Benutzer nicht gefunden');
    }
    error_log("✅ [get_customer_profile] Benutzer gefunden: " . $user['first_name'] . " " . $user['last_name']);
    
    // Fahrzeuge laden
    error_log("🔍 [get_customer_profile] Lade Fahrzeuge für User-ID: " . $user_id);
    $stmt = $conn->prepare("
        SELECT * FROM user_vehicles 
        WHERE user_id = ? AND is_active = 1 
        ORDER BY is_default DESC, created_at DESC
    ");
    $stmt->execute([$user_id]);
    $vehicles = $stmt->fetchAll(PDO::FETCH_ASSOC);
    error_log("✅ [get_customer_profile] " . count($vehicles) . " Fahrzeuge gefunden");
    
    // Termine laden
    error_log("🔍 [get_customer_profile] Lade Termine für User-ID: " . $user_id);
    $stmt = $conn->prepare("
        SELECT 
            a.*,
            e.name as event_name,
            e.event_date,
            COALESCE(e.event_time, e.start_time) as event_time,
            ah.name as autohaus_name
        FROM appointments a
        LEFT JOIN events e ON a.event_id = e.id
        LEFT JOIN autohaus ah ON e.autohaus_id = ah.id
        WHERE a.user_id = ? AND a.is_active = 1
        ORDER BY e.event_date DESC, COALESCE(e.event_time, e.start_time) DESC
        LIMIT 10
    ");
    $stmt->execute([$user_id]);
    $appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    error_log("✅ [get_customer_profile] " . count($appointments) . " Termine gefunden");
    
    // Activity Log laden
    error_log("🔍 [get_customer_profile] Lade Aktivitäten für User-ID: " . $user_id);
    $stmt = $conn->prepare("
        SELECT * FROM user_activity_log 
        WHERE user_id = ? 
        ORDER BY created_at DESC 
        LIMIT 20
    ");
    $stmt->execute([$user_id]);
    $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    error_log("✅ [get_customer_profile] " . count($activities) . " Aktivitäten gefunden");
    
    // Statistiken
    $stats = [
        'total_appointments' => count($appointments),
        'total_vehicles' => count($vehicles),
        'total_activities' => count($activities),
        'verification_status' => $user['verification_status']
    ];
    
    $response = [
        'success' => true,
        'user' => $user,
        'vehicles' => $vehicles,
        'appointments' => $appointments,
        'activities' => $activities,
        'stats' => $stats
    ];
    
    error_log("✅ [get_customer_profile] Response vorbereitet - Success: true");
    error_log("🔍 [get_customer_profile] Response-Daten: " . json_encode([
        'user_id' => $user['id'],
        'vehicles_count' => count($vehicles),
        'appointments_count' => count($appointments),
        'activities_count' => count($activities)
    ]));
    
    echo json_encode($response);
    
} catch (\Exception $e) {
    error_log("❌ [get_customer_profile] EXCEPTION: " . $e->getMessage());
    error_log("❌ [get_customer_profile] Stack Trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
