<?php
require_once __DIR__ . '/_guard_bootstrap.php';
/**
 * API Endpoint: Get Available Time Slots
 * 
 * This API provides available time slots for an event on a specific date.
 * It can be used by mobile apps, web apps, or other services.
 * 
 * Parameters:
 * - event_id (required): The event ID
 * - date (required): The date in YYYY-MM-DD format
 * 
 * Returns:
 * - success: boolean
 * - event: object with event details
 * - available_slots: array of available time slots
 * - date: the requested date
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

require_once __DIR__ . '/../config/database.php';

try {
    $event_id = $_GET['event_id'] ?? null;
    $date = $_GET['date'] ?? null;
    
    if (!$event_id || !$date) {
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'error' => 'Event ID and date are required',
            'message' => 'Please provide both event_id and date parameters'
        ]);
        exit;
    }
    
    // Validate date format
    $dateObj = DateTime::createFromFormat('Y-m-d', $date);
    if (!$dateObj || $dateObj->format('Y-m-d') !== $date) {
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'error' => 'Invalid date format',
            'message' => 'Date must be in YYYY-MM-DD format'
        ]);
        exit;
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get event details
    $stmt = $conn->prepare("
        SELECT 
            e.id as event_id,
            e.name as event_name,
            e.event_date,
            e.start_time,
            e.end_time,
            e.slot_duration,
            e.max_monteure,
            e.pause_start,
            e.pause_end,
            e.blocked_start,
            e.blocked_end
        FROM events e 
        WHERE e.id = ? AND e.is_active = 1
    ");
    $stmt->execute([$event_id]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event) {
        http_response_code(404);
        echo json_encode([
            'success' => false, 
            'error' => 'Event not found',
            'message' => 'The requested event does not exist or is inactive'
        ]);
        exit;
    }
    
    // WICHTIG: Verwende das angeforderte Datum ($date), nicht das Event-Datum!
    // Der Admin kann Termine für verschiedene Tage desselben Events bearbeiten
    $eventDate = $event['event_date'];
    
    // Für die Slot-Berechnung verwenden wir das angeforderte Datum
    $requestedDate = $date;
    
    // Get booked appointments - ZUERST: Alle Termine für das Event (wie in index.php Zeile 285)
    // WICHTIG: Alle aktiven Status zählen: pending, approved, booked, vorbereitet, completed
    // Nur cancelled und rejected zählen NICHT als belegt
    $totalBookedStmt = $conn->prepare("
        SELECT COUNT(*) as total_booked
        FROM appointments 
        WHERE event_id = ? AND status IN ('pending', 'approved', 'booked', 'vorbereitet', 'completed', 'rescheduled')
    ");
    $totalBookedStmt->execute([$event_id]);
    $totalBookedResult = $totalBookedStmt->fetch(PDO::FETCH_ASSOC);
    $totalBookedForEvent = (int)($totalBookedResult['total_booked'] ?? 0);
    
    // Get booked appointments for the REQUESTED date - ALLE aktiven Status
    // pending = wartend auf Genehmigung (belegt!)
    // approved = genehmigt (belegt!)
    // booked = gebucht (belegt!)
    // vorbereitet = vorbereitet (belegt!)
    // completed = abgeschlossen (belegt!)
    // rescheduled = verschoben (belegt, bis neuer Termin bestätigt)
    $stmt = $conn->prepare("
        SELECT slot_time, COUNT(*) as booked_count
        FROM appointments 
        WHERE event_id = ? AND slot_date = ? AND status IN ('pending', 'approved', 'booked', 'vorbereitet', 'completed', 'rescheduled')
        GROUP BY slot_time
    ");
    $stmt->execute([$event_id, $requestedDate]);
    $bookedSlotsRaw = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Debug: Log raw data
    error_log("get_available_slots.php - Event ID: $event_id, Event Date: $eventDate, Requested Date: $requestedDate");
    error_log("get_available_slots.php - Found " . count($bookedSlotsRaw) . " booked slot times");
    
    // Debug: Prüfe auch ohne Datum-Filter
    $debugStmt = $conn->prepare("
        SELECT COUNT(*) as total_count, 
               COUNT(CASE WHEN slot_date = ? THEN 1 END) as count_for_event_date,
               COUNT(CASE WHEN slot_date = ? THEN 1 END) as count_for_request_date,
               COUNT(CASE WHEN status IN ('pending', 'approved', 'booked', 'vorbereitet', 'completed', 'rescheduled') THEN 1 END) as count_with_active_status
        FROM appointments 
        WHERE event_id = ?
    ");
    $debugStmt->execute([$eventDate, $requestedDate, $event_id]);
    $debugInfo = $debugStmt->fetch(PDO::FETCH_ASSOC);
    error_log("get_available_slots.php - Debug Info: " . json_encode($debugInfo));
    
    if (count($bookedSlotsRaw) > 0) {
        error_log("get_available_slots.php - First booked slot: " . json_encode($bookedSlotsRaw[0]));
        error_log("get_available_slots.php - First slot_time type: " . gettype($bookedSlotsRaw[0]['slot_time']));
        error_log("get_available_slots.php - First slot_time value: " . var_export($bookedSlotsRaw[0]['slot_time'], true));
    } else {
        // Prüfe ob es überhaupt Termine für dieses Event gibt
        $checkStmt = $conn->prepare("
            SELECT id, slot_time, slot_date, status 
            FROM appointments 
            WHERE event_id = ? 
            LIMIT 5
        ");
        $checkStmt->execute([$event_id]);
        $sampleAppointments = $checkStmt->fetchAll(PDO::FETCH_ASSOC);
        error_log("get_available_slots.php - Sample appointments for event: " . json_encode($sampleAppointments));
    }
    
    // Normalize slot_time format - verwende die gleiche Logik wie in index.php
    // In index.php wird slot_time direkt als Key verwendet, daher müssen wir es im gleichen Format speichern
    $bookedSlots = [];
    foreach ($bookedSlotsRaw as $row) {
        $slotTimeRaw = $row['slot_time'];
        $bookedCount = (int)$row['booked_count'];
        
        // Konvertiere zu String falls es ein Objekt ist
        if (is_object($slotTimeRaw)) {
            $slotTimeRaw = (string)$slotTimeRaw;
        }
        
        // Normalisiere das Zeitformat - MySQL TIME gibt normalerweise "HH:MM:SS" zurück
        // Aber es kann auch "HH:MM" sein, daher normalisieren wir es zu "HH:MM:SS"
        $slotTime = trim($slotTimeRaw);
        
        // Versuche verschiedene Parsing-Methoden
        $slotTimeFull = null;
        $slotTimeShort = null;
        
        // Methode 1: DateTime-Objekt (am zuverlässigsten)
        try {
            // Versuche verschiedene Formate
            $timeObj = false;
            $formats = ['H:i:s', 'H:i', 'G:i:s', 'G:i'];
            foreach ($formats as $format) {
                $timeObj = DateTime::createFromFormat($format, $slotTime);
                if ($timeObj !== false) {
                    break;
                }
            }
            
            if ($timeObj === false) {
                // Versuche direktes Parsing
                $timeObj = new DateTime($slotTime);
            }
            
            $slotTimeFull = $timeObj->format('H:i:s');
            $slotTimeShort = $timeObj->format('H:i');
        } catch (Exception $e) {
            // Methode 2: Regex-Parsing
            if (preg_match('/^(\d{1,2}):(\d{2})(?::(\d{2}))?$/', $slotTime, $matches)) {
                $hours = str_pad((int)$matches[1], 2, '0', STR_PAD_LEFT);
                $minutes = str_pad((int)$matches[2], 2, '0', STR_PAD_LEFT);
                $seconds = isset($matches[3]) ? str_pad((int)$matches[3], 2, '0', STR_PAD_LEFT) : '00';
                $slotTimeFull = "$hours:$minutes:$seconds";
                $slotTimeShort = "$hours:$minutes";
            } else {
                error_log("get_available_slots.php - Could not parse time: " . $slotTime . " (Error: " . $e->getMessage() . ")");
                continue; // Überspringe diesen Eintrag
            }
        }
        
        // Speichere in beiden Formaten (wie in index.php verwendet)
        if ($slotTimeFull && $slotTimeShort) {
            $bookedSlots[$slotTimeFull] = $bookedCount;
            $bookedSlots[$slotTimeShort] = $bookedCount;
            // Speichere auch das Original-Format falls es anders ist
            if ($slotTime !== $slotTimeFull && $slotTime !== $slotTimeShort) {
                $bookedSlots[$slotTime] = $bookedCount;
            }
        }
    }
    
    // Debug: Log normalized booked slots
    error_log("get_available_slots.php - Normalized booked slots count: " . count($bookedSlots));
    if (count($bookedSlots) > 0) {
        $sampleKeys = array_slice(array_keys($bookedSlots), 0, 3);
        error_log("get_available_slots.php - Sample booked slot keys: " . implode(', ', $sampleKeys));
    }
    
    // Generate available time slots
    $availableSlots = generateTimeSlots($event, $bookedSlots);
    
    // Calculate total booked places
    // WICHTIG: Verwende totalBookedForEvent für die Gesamtauslastung (wie in index.php)
    // Aber für die Slot-Anzeige verwenden wir die gefilterten Slots
    $totalBookedPlaces = 0;
    $totalAvailablePlaces = 0;
    foreach ($availableSlots as $slot) {
        $totalBookedPlaces += $slot['booked'] ?? 0;
        $totalAvailablePlaces += $slot['available_count'] ?? 0;
    }
    
    // Für die Gesamtauslastung verwenden wir totalBookedForEvent (alle Termine des Events)
    // Das entspricht der Logik in index.php Zeile 285
    $totalBookedPlaces = $totalBookedForEvent;
    
    // Return success response
    $response = [
        'success' => true,
        'event' => [
            'id' => $event['event_id'],
            'name' => $event['event_name'],
            'event_date' => $event['event_date'],
            'start_time' => $event['start_time'],
            'end_time' => $event['end_time'],
            'slot_duration' => $event['slot_duration'],
            'max_monteure' => $event['max_monteure']
        ],
        'available_slots' => $availableSlots,
        'date' => $date,
        'total_slots' => count($availableSlots),
        'total_available' => array_sum(array_column($availableSlots, 'available_count')),
        'total_booked' => $totalBookedPlaces, // Alle Termine des Events (wie in index.php)
        'total_booked_for_date' => array_sum(array_column($availableSlots, 'booked')), // Nur für event_date
        'total_capacity' => count($availableSlots) * ($event['max_monteure'] ?? 5)
    ];
    
    // Debug-Informationen (immer aktiviert für Troubleshooting)
    $response['debug'] = [
        'booked_slots_raw_count' => count($bookedSlotsRaw),
        'booked_slots_raw' => $bookedSlotsRaw,
        'booked_slots_normalized_count' => count($bookedSlots),
        'booked_slots_normalized' => $bookedSlots,
        'event_id' => $event_id,
        'request_date' => $date,
        'event_date' => $eventDate,
        'date_match' => ($date === $eventDate),
        'total_booked_for_event' => $totalBookedForEvent,
        'total_booked_for_date' => array_sum(array_column($availableSlots, 'booked')),
        'debug_info' => $debugInfo ?? null,
        'sample_appointments' => $sampleAppointments ?? null
    ];
    
    echo json_encode($response);
    
} catch (Exception $e) {
    error_log("Error in get_available_slots.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => 'Internal server error',
        'message' => $e->getMessage()
    ]);
}

/**
 * Generate time slots for an event
 */
function generateTimeSlots($event, $bookedSlots) {
    $slots = [];
    $startTime = new DateTime($event['start_time']);
    $endTime = new DateTime($event['end_time']);
    
    $slotDuration = $event['slot_duration'] ?? 30; // minutes
    $maxCount = $event['max_monteure'] ?? 5;
    
    // Pause times
    $pauseStart = $event['pause_start'] ? new DateTime($event['pause_start']) : null;
    $pauseEnd = $event['pause_end'] ? new DateTime($event['pause_end']) : null;
    $blockedStart = $event['blocked_start'] ? new DateTime($event['blocked_start']) : null;
    $blockedEnd = $event['blocked_end'] ? new DateTime($event['blocked_end']) : null;
    
    $currentTime = clone $startTime;
    
    while ($currentTime < $endTime) {
        $timeStr = $currentTime->format('H:i:s');
        $timeStrShort = $currentTime->format('H:i');
        
        // Try both formats for backward compatibility (MySQL TIME can return H:i or H:i:s)
        $booked = $bookedSlots[$timeStr] ?? $bookedSlots[$timeStrShort] ?? 0;
        $availableCount = max(0, $maxCount - $booked);
        
        // Skip if in pause time
        if ($pauseStart && $pauseEnd) {
            if ($currentTime >= $pauseStart && $currentTime < $pauseEnd) {
                $currentTime->modify("+{$slotDuration} minutes");
                continue;
            }
        }
        
        // Skip if in blocked time
        if ($blockedStart && $blockedEnd) {
            if ($currentTime >= $blockedStart && $currentTime < $blockedEnd) {
                $currentTime->modify("+{$slotDuration} minutes");
                continue;
            }
        }
        
        $slots[] = [
            'time' => $timeStr,
            'formatted_time' => $currentTime->format('H:i'),
            'available' => $availableCount > 0,
            'available_count' => $availableCount,
            'booked' => $booked,
            'max_count' => $maxCount
        ];
        
        $currentTime->modify("+{$slotDuration} minutes");
    }
    
    return $slots;
}
?>

