<?php
/**
 * API: Hole E-Mail-Adresse aus appointments basierend auf Einlagerungsnummer oder Kennzeichen
 */

header('Content-Type: application/json; charset=UTF-8');
session_start();

require_once dirname(__DIR__) . '/config/database.php';

try {
    $wheelset_number = $_GET['wheelset_number'] ?? '';
    $license_plate = $_GET['license_plate'] ?? '';
    $autohaus_id = $_GET['autohaus_id'] ?? '';
    
    if (empty($wheelset_number) && empty($license_plate)) {
        throw new Exception('wheelset_number oder license_plate ist erforderlich');
    }
    
    if (empty($autohaus_id)) {
        throw new Exception('autohaus_id ist erforderlich');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    /**
     * Normalisiert Einlagerungsnummern für Vergleich
     */
    function normalizeStorageNumber($number) {
        if (empty($number)) {
            return '';
        }
        $normalized = preg_replace('/[^\w.]/', '', $number);
        $normalized = strtoupper($normalized);
        if (preg_match('/^[\d.]+$/', $normalized)) {
            $normalized = preg_replace('/\.+/', '.', trim($normalized, '.'));
        }
        return $normalized;
    }
    
    /**
     * Normalisiert Kennzeichen für Vergleich
     */
    function normalizeLicensePlate($plate) {
        if (empty($plate)) {
            return '';
        }
        return strtoupper(preg_replace('/[\s-]+/', '', $plate));
    }
    
    // Suche zuerst über Einlagerungsnummer
    if (!empty($wheelset_number)) {
        $normalizedStorage = normalizeStorageNumber($wheelset_number);
        
        $stmt = $conn->prepare("
            SELECT email, customer_name, license_plate, storage_number
            FROM appointments
            WHERE autohaus_id = ?
            AND (
                REPLACE(REPLACE(REPLACE(UPPER(storage_number), ' ', ''), '-', ''), '.', '') = ?
                OR storage_number LIKE ?
                OR storage_number = ?
            )
            AND email IS NOT NULL
            AND email != ''
            ORDER BY created_at DESC
            LIMIT 1
        ");
        
        $normalizedForLike = '%' . $normalizedStorage . '%';
        $stmt->execute([$autohaus_id, $normalizedStorage, $normalizedForLike, $wheelset_number]);
        $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($appointment && !empty($appointment['email'])) {
            echo json_encode([
                'success' => true,
                'email' => $appointment['email'],
                'customer_name' => $appointment['customer_name'],
                'source' => 'storage_number',
                'matched_storage_number' => $appointment['storage_number']
            ]);
            exit;
        }
    }
    
    // Falls nicht gefunden, suche über Kennzeichen
    if (!empty($license_plate)) {
        $normalizedLicense = normalizeLicensePlate($license_plate);
        
        $stmt = $conn->prepare("
            SELECT email, customer_name, license_plate, storage_number
            FROM appointments
            WHERE autohaus_id = ?
            AND REPLACE(REPLACE(UPPER(license_plate), ' ', ''), '-', '') = ?
            AND email IS NOT NULL
            AND email != ''
            ORDER BY created_at DESC
            LIMIT 1
        ");
        
        $stmt->execute([$autohaus_id, $normalizedLicense]);
        $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($appointment && !empty($appointment['email'])) {
            echo json_encode([
                'success' => true,
                'email' => $appointment['email'],
                'customer_name' => $appointment['customer_name'],
                'source' => 'license_plate',
                'matched_license_plate' => $appointment['license_plate']
            ]);
            exit;
        }
    }
    
    // Keine E-Mail gefunden
    echo json_encode([
        'success' => false,
        'email' => null,
        'message' => 'Keine E-Mail-Adresse für diese Einlagerungsnummer/Kennzeichen gefunden'
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

