<?php
/**
 * API-Endpunkt zum Finden des nächsten Autohauses basierend auf Koordinaten
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config/database.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['latitude']) || !isset($input['longitude'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Koordinaten erforderlich']);
        exit;
    }
    
    $userLat = floatval($input['latitude']);
    $userLng = floatval($input['longitude']);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Hole alle aktiven Autohäuser
    $stmt = $conn->prepare("SELECT id, name, short_name, address FROM autohaus WHERE is_active = 1");
    $stmt->execute();
    $autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($autohauser)) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Keine Autohäuser gefunden']);
        exit;
    }
    
    // Geocode alle Autohaus-Adressen und berechne Distanzen
    $autohauserWithDistance = [];
    
    foreach ($autohauser as $autohaus) {
        // Geocode Adresse zu Koordinaten
        $address = urlencode($autohaus['address']);
        $geocodeUrl = "https://nominatim.openstreetmap.org/search?format=json&q={$address}&limit=1";
        
        $geocodeContext = stream_context_create([
            'http' => [
                'timeout' => 2,
                'method' => 'GET',
                'header' => "User-Agent: Boxenstop-App/1.0\r\n"
            ]
        ]);
        
        $geocodeResponse = @file_get_contents($geocodeUrl, false, $geocodeContext);
        
        if ($geocodeResponse) {
            $geocodeData = json_decode($geocodeResponse, true);
            if (!empty($geocodeData) && isset($geocodeData[0]['lat']) && isset($geocodeData[0]['lon'])) {
                $autohausLat = floatval($geocodeData[0]['lat']);
                $autohausLng = floatval($geocodeData[0]['lon']);
                
                // Berechne Distanz (Haversine-Formel)
                $distance = calculateDistance($userLat, $userLng, $autohausLat, $autohausLng);
                
                $autohauserWithDistance[] = [
                    'id' => $autohaus['id'],
                    'name' => $autohaus['name'],
                    'short_name' => $autohaus['short_name'],
                    'address' => $autohaus['address'],
                    'latitude' => $autohausLat,
                    'longitude' => $autohausLng,
                    'distance_km' => round($distance, 2)
                ];
            }
        }
        
        // Rate-Limiting: Kurze Pause zwischen Requests
        usleep(200000); // 0.2 Sekunden
    }
    
    if (empty($autohauserWithDistance)) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Konnte Autohaus-Koordinaten nicht ermitteln']);
        exit;
    }
    
    // Sortiere nach Distanz
    usort($autohauserWithDistance, function($a, $b) {
        return $a['distance_km'] <=> $b['distance_km'];
    });
    
    $nearest = $autohauserWithDistance[0];
    
    echo json_encode([
        'success' => true,
        'nearest_autohaus' => $nearest,
        'all_autohauser' => $autohauserWithDistance
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Fehler beim Finden des nächsten Autohauses: ' . $e->getMessage()
    ]);
}

/**
 * Berechnet die Distanz zwischen zwei Koordinaten (Haversine-Formel)
 * @param float $lat1 Latitude Punkt 1
 * @param float $lon1 Longitude Punkt 1
 * @param float $lat2 Latitude Punkt 2
 * @param float $lon2 Longitude Punkt 2
 * @return float Distanz in Kilometern
 */
function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371; // Radius der Erde in Kilometern
    
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    
    $a = sin($dLat/2) * sin($dLat/2) +
         cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
         sin($dLon/2) * sin($dLon/2);
    
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    
    return $earthRadius * $c;
}

