<?php
/**
 * E-Mail-System - Vollständiges E-Mail-System für Boxenstop
 * Verwaltet E-Mail-Templates, Versand und Logging
 */

class EmailSystem {
    private $conn;
    private $autohausId;
    private $smtpHost;
    private $smtpPort;
    private $smtpUsername;
    private $smtpPassword;
    private $fromEmail;
    private $fromName;
    private $useTLS;
    
    /**
     * @param PDO $database Datenbankverbindung
     * @param int|null $autohausId Autohaus-ID für Multi-Autohaus-System (optional)
     */
    public function __construct($database, $autohausId = null) {
        $this->conn = $database;
        $this->autohausId = $autohausId;
        $this->loadSettings();
    }
    
    /**
     * Lädt SMTP-Einstellungen aus der Datenbank
     * Priorität: 1. Autohaus-spezifische Einstellungen, 2. Globale Einstellungen, 3. Fallback
     */
    private function loadSettings() {
        try {
            // Versuche zuerst Autohaus-spezifische Einstellungen zu laden
            if ($this->autohausId) {
                $stmt = $this->conn->prepare("
                    SELECT 
                        email_smtp_host, email_smtp_port, email_smtp_username, 
                        email_smtp_password, email_smtp_encryption, email_from_name, email
                    FROM autohaus 
                    WHERE id = ? AND is_active = 1
                ");
                $stmt->execute([$this->autohausId]);
                $autohausSettings = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($autohausSettings && !empty($autohausSettings['email_smtp_host'])) {
                    $this->smtpHost = $autohausSettings['email_smtp_host'];
                    $this->smtpPort = $autohausSettings['email_smtp_port'] ?? 587;
                    $this->smtpUsername = $autohausSettings['email_smtp_username'] ?? '';
                    $this->smtpPassword = $autohausSettings['email_smtp_password'] ?? '';
                    $this->fromEmail = $autohausSettings['email'] ?? '';
                    $this->fromName = $autohausSettings['email_from_name'] ?? ($autohausSettings['name'] ?? 'Boxenstop');
                    $this->useTLS = ($autohausSettings['email_smtp_encryption'] ?? 'tls') === 'tls';
                    return; // Autohaus-Einstellungen gefunden, verwende diese
                }
            }
            
            // Fallback: Globale Einstellungen aus settings-Tabelle
            $stmt = $this->conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key LIKE 'email_%'");
            $stmt->execute();
            $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
            
            $this->smtpHost = $settings['email_smtp_host'] ?? 'smtp-mail.outlook.com';
            $this->smtpPort = $settings['email_smtp_port'] ?? '587';
            $this->smtpUsername = $settings['email_smtp_username'] ?? '';
            $this->smtpPassword = $settings['email_smtp_password'] ?? '';
            $this->fromEmail = $settings['email_from'] ?? '';
            $this->fromName = $settings['email_from_name'] ?? 'Boxenstop';
            $this->useTLS = ($settings['email_use_tls'] ?? '1') === '1';
        } catch (Exception $e) {
            // Fallback-Werte
            $this->smtpHost = 'smtp-mail.outlook.com';
            $this->smtpPort = '587';
            $this->smtpUsername = '';
            $this->smtpPassword = '';
            $this->fromEmail = '';
            $this->fromName = 'Boxenstop';
            $this->useTLS = true;
        }
    }
    
    /**
     * Setzt die Autohaus-ID und lädt die entsprechenden Einstellungen neu
     */
    public function setAutohausId($autohausId) {
        $this->autohausId = $autohausId;
        $this->loadSettings();
    }
    
    /**
     * Prüft ob PHPMailer verfügbar ist und lädt es falls nötig
     */
    private function isPHPMailerAvailable() {
        // Prüfe zuerst ob Klasse bereits geladen ist
        if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
            return true;
        }
        
        // Versuche über Autoloader zu laden
        $autoloadPath = __DIR__ . '/../vendor/autoload.php';
        if (file_exists($autoloadPath)) {
            require_once $autoloadPath;
            if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
                return true;
            }
        }
        
        // Versuche manuell aus vendor/phpmailer zu laden (wie in anderen Dateien)
        $vendorPhpmailerPath = __DIR__ . '/../vendor/phpmailer/PHPMailer.php';
        if (file_exists($vendorPhpmailerPath)) {
            require_once $vendorPhpmailerPath;
            require_once __DIR__ . '/../vendor/phpmailer/SMTP.php';
            require_once __DIR__ . '/../vendor/phpmailer/Exception.php';
            if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
                return true;
            }
        }
        
        // Versuche manuell aus api/phpmailer zu laden (Fallback)
        $apiPhpmailerPath = __DIR__ . '/phpmailer/PHPMailer.php';
        if (file_exists($apiPhpmailerPath)) {
            require_once $apiPhpmailerPath;
            require_once __DIR__ . '/phpmailer/SMTP.php';
            require_once __DIR__ . '/phpmailer/Exception.php';
            if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Sendet eine E-Mail
     */
    public function sendEmail($to, $subject, $body, $templateKey = null, $appointmentId = null) {
        try {
            // Prüfe ob SMTP-Einstellungen vorhanden sind
            $hasSmtpConfig = !empty($this->smtpHost) && !empty($this->smtpUsername) && !empty($this->smtpPassword);
            
            // Verwende PHPMailer falls verfügbar und SMTP konfiguriert
            if ($hasSmtpConfig && $this->isPHPMailerAvailable()) {
                return $this->sendWithPHPMailer($to, $subject, $body, $templateKey, $appointmentId);
            } elseif ($hasSmtpConfig) {
                // SMTP konfiguriert, aber PHPMailer nicht verfügbar
                $this->logEmail($to, $subject, 'failed', 'PHPMailer nicht verfügbar, aber SMTP konfiguriert', $templateKey, $appointmentId);
                return ['success' => false, 'error' => 'PHPMailer ist nicht installiert. Bitte installieren Sie PHPMailer: composer require phpmailer/phpmailer'];
            } else {
                // Keine SMTP-Konfiguration - verwende mail() nur wenn keine SMTP-Einstellungen vorhanden
                if (empty($this->smtpHost)) {
                    $this->logEmail($to, $subject, 'failed', 'Keine SMTP-Einstellungen konfiguriert', $templateKey, $appointmentId);
                    return ['success' => false, 'error' => 'Keine SMTP-Einstellungen konfiguriert. Bitte konfigurieren Sie SMTP in den Einstellungen.'];
                }
                return $this->sendWithMail($to, $subject, $body, $templateKey, $appointmentId);
            }
        } catch (Exception $e) {
            $this->logEmail($to, $subject, 'failed', $e->getMessage(), $templateKey, $appointmentId);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Sendet E-Mail mit PHPMailer
     */
    private function sendWithPHPMailer($to, $subject, $body, $templateKey, $appointmentId) {
        // PHPMailer sollte bereits in isPHPMailerAvailable() geladen worden sein
        // Aber zur Sicherheit nochmal prüfen und laden falls nötig
        if (!class_exists('PHPMailer\PHPMailer\PHPMailer')) {
            $this->isPHPMailerAvailable();
        }
        
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        
        try {
            // SMTP-Konfiguration
            $mail->isSMTP();
            $mail->Host = $this->smtpHost;
            $mail->SMTPAuth = true;
            $mail->Username = $this->smtpUsername;
            $mail->Password = $this->smtpPassword;
            $mail->SMTPSecure = $this->useTLS ? PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS : PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
            $mail->Port = $this->smtpPort;
            
            // Outlook/Office 365 spezifische Einstellungen
            $mail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            
            // WICHTIG: Microsoft 365/Outlook erlaubt nur das Senden von der authentifizierten E-Mail-Adresse
            // Verwende SMTP-Username als From-Adresse (wie in send_offer_email.php)
            $fromEmail = !empty($this->smtpUsername) ? $this->smtpUsername : $this->fromEmail;
            $fromName = $this->fromName;
            
            // Absender und Empfänger
            $mail->setFrom($fromEmail, $fromName);
            $mail->addAddress($to);
            
            // Wenn eine andere From-Adresse konfiguriert ist, setze sie als Reply-To
            if (!empty($this->fromEmail) && $this->fromEmail !== $fromEmail) {
                $mail->addReplyTo($this->fromEmail, $fromName);
            } else {
                $mail->addReplyTo($fromEmail, $fromName);
            }
            
            // Inhalt
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $body;
            $mail->AltBody = strip_tags($body);
            $mail->CharSet = 'UTF-8';
            
            $mail->send();
            
            $this->logEmail($to, $subject, 'sent', null, $templateKey, $appointmentId);
            
            return ['success' => true, 'message' => 'E-Mail erfolgreich gesendet'];
        } catch (Exception $e) {
            $errorInfo = $mail->ErrorInfo ?? $e->getMessage();
            
            // Verbesserte Fehlermeldung für SendAsDenied-Fehler
            if (strpos($errorInfo, 'SendAsDenied') !== false || strpos($errorInfo, 'not allowed to send as') !== false) {
                $errorMessage = "SMTP-Fehler: Die E-Mail-Adresse '{$this->fromEmail}' darf nicht als Absender verwendet werden. ";
                $errorMessage .= "Microsoft 365/Outlook erlaubt nur das Senden von der authentifizierten E-Mail-Adresse '{$this->smtpUsername}'. ";
                $errorMessage .= "Bitte verwenden Sie die SMTP-Username-Adresse als From-Adresse oder kontaktieren Sie Ihren Administrator, um Send-As-Berechtigungen einzurichten.";
            } else {
                $errorMessage = $errorInfo;
            }
            
            $this->logEmail($to, $subject, 'failed', $errorMessage, $templateKey, $appointmentId);
            return ['success' => false, 'error' => $errorMessage];
        }
    }
    
    /**
     * Sendet E-Mail mit mail() - NUR als letzter Fallback wenn keine SMTP-Einstellungen vorhanden
     * WARNUNG: mail() verwendet standardmäßig localhost:25 und funktioniert meist nicht ohne lokalen Mail-Server
     * Diese Funktion sollte NUR verwendet werden, wenn PHPMailer nicht verfügbar ist UND keine SMTP-Einstellungen vorhanden sind
     */
    private function sendWithMail($to, $subject, $body, $templateKey, $appointmentId) {
        // Wenn SMTP konfiguriert ist, sollte PHPMailer verwendet werden
        if (!empty($this->smtpHost) && !empty($this->smtpUsername)) {
            $error = 'SMTP ist konfiguriert, aber PHPMailer ist nicht verfügbar. Bitte installieren Sie PHPMailer: composer require phpmailer/phpmailer';
            $this->logEmail($to, $subject, 'failed', $error, $templateKey, $appointmentId);
            return ['success' => false, 'error' => $error];
        }
        
        // Versuche SMTP über ini_set zu konfigurieren (funktioniert nur auf Windows)
        if (!empty($this->smtpHost)) {
            ini_set('SMTP', $this->smtpHost);
            if (!empty($this->smtpPort)) {
                ini_set('smtp_port', $this->smtpPort);
            }
        }
        
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type: text/html; charset=UTF-8\r\n";
        $headers .= "From: {$this->fromName} <{$this->fromEmail}>\r\n";
        $headers .= "Reply-To: {$this->fromEmail}\r\n";
        
        $result = @mail($to, $subject, $body, $headers);
        
        if ($result) {
            $this->logEmail($to, $subject, 'sent', null, $templateKey, $appointmentId);
            return ['success' => true, 'message' => 'E-Mail erfolgreich gesendet'];
        } else {
            $error = 'mail() Funktion fehlgeschlagen. Bitte konfigurieren Sie SMTP-Einstellungen und installieren Sie PHPMailer (composer require phpmailer/phpmailer)';
            $this->logEmail($to, $subject, 'failed', $error, $templateKey, $appointmentId);
            return ['success' => false, 'error' => $error];
        }
    }
    
    /**
     * Sendet E-Mail mit Template
     */
    public function sendTemplateEmail($to, $templateKey, $variables = [], $appointmentId = null) {
        try {
            $template = $this->getTemplate($templateKey);
            
            if (!$template) {
                return ['success' => false, 'error' => 'Template nicht gefunden'];
            }
            
            // Ersetze Variablen
            $subject = $this->replaceVariables($template['subject'], $variables);
            $body = $this->replaceVariables($template['body'], $variables);
            
            return $this->sendEmail($to, $subject, $body, $templateKey, $appointmentId);
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Holt ein E-Mail-Template
     */
    public function getTemplate($templateKey) {
        try {
            $stmt = $this->conn->prepare("SELECT * FROM email_templates WHERE template_key = ? AND is_active = 1");
            $stmt->execute([$templateKey]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            return null;
        }
    }
    
    /**
     * Ersetzt Variablen in einem String
     */
    private function replaceVariables($text, $variables) {
        foreach ($variables as $key => $value) {
            $text = str_replace('{{' . $key . '}}', $value, $text);
        }
        return $text;
    }
    
    /**
     * Loggt eine E-Mail
     */
    private function logEmail($to, $subject, $status, $errorMessage = null, $templateKey = null, $appointmentId = null) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO email_logs (appointment_id, template_key, recipient_email, subject, status, sent_at, error_message, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $appointmentId,
                $templateKey,
                $to,
                $subject,
                $status,
                $status === 'sent' ? date('Y-m-d H:i:s') : null,
                $errorMessage
            ]);
        } catch (Exception $e) {
            // Logging-Fehler ignorieren
            error_log("E-Mail-Logging fehlgeschlagen: " . $e->getMessage());
        }
    }
    
    /**
     * Prüft ob E-Mail-System verfügbar ist
     */
    public function isAvailable() {
        try {
            $templatesTable = false;
            $logsTable = false;
            
            try {
                $stmt = $this->conn->query("SELECT COUNT(*) FROM email_templates LIMIT 1");
                $templatesTable = true;
            } catch (Exception $e) {}
            
            try {
                $stmt = $this->conn->query("SELECT COUNT(*) FROM email_logs LIMIT 1");
                $logsTable = true;
            } catch (Exception $e) {}
            
            return $templatesTable && $logsTable;
        } catch (Exception $e) {
            return false;
        }
    }
}
?>

