<?php
/**
 * check_tire_status.php
 * Vereinfachte Version für direkten Aufruf mit Einlagerungsnummer
 * 
 * Verwendung:
 *   POST: {"storage_number": "9877670", "appointment_date": "2025-10-22", "event_name": "Boxenstop"}
 *   GET:  ?storage_number=9877670&appointment_date=2025-10-22&event_name=Boxenstop
 */

declare(strict_types=1);

// Debug-Modus aktivieren
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Log-Funktion
function debugLog($message) {
    error_log("RESY_API_DEBUG: " . $message);
}

// ---------- Eingaben ----------
debugLog("API gestartet");

$jsonInput = json_decode(file_get_contents('php://input'), true);
debugLog("JSON Input: " . json_encode($jsonInput));

$storageNumber = $jsonInput['storage_number'] ?? $_GET['storage_number'] ?? $_POST['storage_number'] ?? '';
$appointmentDate = $jsonInput['appointment_date'] ?? $_GET['appointment_date'] ?? $_POST['appointment_date'] ?? '';
$eventName = $jsonInput['event_name'] ?? $_GET['event_name'] ?? $_POST['event_name'] ?? '';

debugLog("Storage Number: " . $storageNumber);
debugLog("Appointment Date: " . $appointmentDate);
debugLog("Event Name: " . $eventName);

if (empty($storageNumber)) {
    debugLog("ERROR: storage_number fehlt");
    http_response_code(400);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['error' => 'storage_number fehlt'], JSON_UNESCAPED_UNICODE);
    exit;
}

// ---------- RESY-Konfiguration ----------
// DB-Verbindung für Autohaus-Credentials
function getDbConnection() {
    $host = getenv('DB_HOST') ?: 'localhost';
    $dbname = getenv('DB_NAME') ?: 'boxxenstopp';
    $username = getenv('DB_USER') ?: 'root';
    $password = getenv('DB_PASS') ?: '';
    
    $dsn = "mysql:host=$host;dbname=$dbname;charset=utf8mb4";
    return new PDO($dsn, $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
}

// Autohaus-ID aus appointment ermitteln
$autohausId = 1; // Standard: Autohaus 1 (B & E Volvo Kiel)

// Falls autohaus_id im Request mitgesendet wird, verwende diese
if (isset($jsonInput['autohaus_id'])) {
    $autohausId = (int)$jsonInput['autohaus_id'];
} elseif (isset($_GET['autohaus_id'])) {
    $autohausId = (int)$_GET['autohaus_id'];
} elseif (isset($_POST['autohaus_id'])) {
    $autohausId = (int)$_POST['autohaus_id'];
}

try {
    debugLog("Versuche DB-Verbindung für Autohaus-ID: " . $autohausId);
    $pdo = getDbConnection();
    debugLog("DB-Verbindung erfolgreich");
    
    $stmt = $pdo->prepare("SELECT resy_username, resy_password, resy_base_url FROM autohaus WHERE id = ?");
    $stmt->execute([$autohausId]);
    $autohaus = $stmt->fetch();
    
    debugLog("Autohaus-Daten: " . json_encode($autohaus));
    
    if (!$autohaus) {
        throw new Exception("Autohaus-ID $autohausId nicht gefunden");
    }
    
    $RESY_USER = $autohaus['resy_username'];
    $RESY_PASS = $autohaus['resy_password'];
    $BASE_URL = rtrim($autohaus['resy_base_url'] ?: 'https://resy.dtm-reifen.de', '/');
    
    debugLog("RESY_USER: " . $RESY_USER);
    debugLog("BASE_URL: " . $BASE_URL);
    
} catch (Exception $e) {
    debugLog("DB-Fehler: " . $e->getMessage());
    http_response_code(500);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['error' => 'DB-Fehler: ' . $e->getMessage()], JSON_UNESCAPED_UNICODE);
    exit;
}

// ---------- HTTP-Helper mit cURL ----------
$cookieFile = tempnam(sys_get_temp_dir(), 'resy_cookie_');
$UA = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/126.0 Safari/537.36";

function http(array $opt): array {
    global $cookieFile, $UA;
    $ch = curl_init();
    $base = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT      => $UA,
        CURLOPT_CONNECTTIMEOUT => 30,
        CURLOPT_TIMEOUT        => 60,
        CURLOPT_COOKIEJAR      => $cookieFile,
        CURLOPT_COOKIEFILE     => $cookieFile,
        CURLOPT_HEADER         => false,
        CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
        CURLOPT_ENCODING       => 'gzip,deflate',
        CURLOPT_REFERER        => 'https://resy.dtm-reifen.de/',
    ];
    curl_setopt_array($ch, $base + $opt);
    $body = curl_exec($ch);
    
    if ($body === false) {
        $err = curl_error($ch);
        $errno = curl_errno($ch);
        curl_close($ch);
        debugLog("cURL Fehler: $err (Code: $errno)");
        throw new RuntimeException("cURL: $err");
    }
    
    $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    $info = curl_getinfo($ch);
    debugLog("cURL Info: HTTP $status, URL: " . $info['url']);
    curl_close($ch);
    return [$status, $body];
}

function absUrl(string $base, string $path): string {
    if (preg_match('~^https?://~i', $path)) return $path;
    
    // Handle relative paths starting with ./
    if (strpos($path, './') === 0) {
        $path = substr($path, 2); // Remove ./
    }
    
    return rtrim($base, '/') . '/' . ltrim($path, '/');
}

function xp(string $html): DOMXPath {
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html, LIBXML_NOWARNING|LIBXML_NOERROR|LIBXML_NONET|LIBXML_COMPACT);
    libxml_clear_errors();
    return new DOMXPath($doc);
}

function nodeText(?DOMNode $n): string { 
    return $n ? trim($n->textContent) : ''; 
}

function q1(DOMXPath $xp, string $xpath, ?DOMNode $ctx=null): ?DOMNode {
    $nl = $xp->query($xpath, $ctx);
    return ($nl && $nl->length) ? $nl->item(0) : null;
}

function deFloat(string $s): float {
    if ($s === '') return 0.0;
    if (preg_match('/[-+]?\d{1,3}(?:[.,]\d+)?/', $s, $m)) {
        return (float)str_replace([',','.'], ['.',''], str_replace('.', '', $m[0]));
    }
    return 0.0;
}

// ---------- Hauptfunktion ----------
try {
// 1) Prüfe zuerst RESY (Live-Daten haben Priorität)
debugLog("Versuche RESY-Login für Einlagerungsnummer: $storageNumber");

// 1) Login
debugLog("Versuche Login-Seite zu laden: " . absUrl($BASE_URL, '/index.php?m=login&a=login'));
[$st, $loginHtml] = http([CURLOPT_URL => absUrl($BASE_URL, '/index.php?m=login&a=login'), CURLOPT_HTTPGET => true]);
    debugLog("Login-Seite Response: HTTP $st, Länge: " . strlen($loginHtml));
    
    if ($st >= 400) {
        debugLog("Login-Seite Fehler: HTTP $st");
        throw new RuntimeException("Login-Seite HTTP $st - Möglicherweise ist RESY nicht erreichbar");
    }

    $x = xp($loginHtml);
    $form = q1($x, "//form[.//input[@type='password']]");
    if (!$form) {
        debugLog("Login-Formular nicht gefunden");
        throw new RuntimeException("Login-Formular nicht gefunden");
    }

    $action = $form->getAttribute('action') ?: '/index.php?m=login';
    debugLog("Login-Formular gefunden: action=$action");
    
    $fields = [];
    
    // Alle Input-Felder sammeln
    foreach ($x->query(".//input", $form) as $in) {
        $type = strtolower($in->getAttribute('type'));
        $name = $in->getAttribute('name');
        if ($name === '' || in_array($type, ['submit','button','image'])) continue;
        $fields[$name] = $in->getAttribute('value');
    }
    
    debugLog("Gefundene Felder: " . json_encode($fields));
    
    // Username/Password-Feldnamen finden - RESY-spezifisch
    $userField = 'UserName';  // RESY verwendet 'UserName'
    $passField = 'Password';  // RESY verwendet 'Password'
    
    debugLog("RESY-Feldnamen: UserField=$userField, PassField=$passField");
    
    debugLog("UserField: $userField, PassField: $passField");
    
    if (!$userField || !$passField) {
        debugLog("Login-Feldnamen nicht erkannt");
        throw new RuntimeException("Login-Feldnamen nicht erkannt");
    }

    $fields[$userField] = $RESY_USER;
    $fields[$passField] = $RESY_PASS;
    
    debugLog("Finale Login-Daten: " . json_encode($fields));

    [$st, $afterLogin] = http([
        CURLOPT_URL => absUrl($BASE_URL, $action),
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($fields),
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
    ]);
    
    debugLog("Login-POST Response: HTTP $st, Länge: " . strlen($afterLogin));
    
    if ($st >= 400) {
        debugLog("Login-POST Fehler: HTTP $st");
        throw new RuntimeException("Login-POST HTTP $st");
    }
    
    // Prüfe ob Login erfolgreich war
    if (strpos($afterLogin, 'Anmeldename') !== false) {
        debugLog("Login fehlgeschlagen - Login-Formular noch vorhanden");
        throw new RuntimeException("Login fehlgeschlagen - Anmeldedaten möglicherweise falsch");
    }
    
    debugLog("Login erfolgreich - Weiterleitung zur Suche");

    // 2) Zur Suche navigieren
    debugLog("Versuche Suche-Seite zu laden: " . absUrl($BASE_URL, '/index.php?m=wheelset&a=search'));
    [$st, $searchHtml] = http([CURLOPT_URL => absUrl($BASE_URL, '/index.php?m=wheelset&a=search'), CURLOPT_HTTPGET => true]);
    debugLog("Suche-Seite Response: HTTP $st, Länge: " . strlen($searchHtml));
    
    if ($st >= 400) {
        debugLog("Suche-Seite Fehler: HTTP $st");
        debugLog("Response Content: " . substr($searchHtml, 0, 500));
        throw new RuntimeException("Suche-Seite HTTP $st - Login möglicherweise fehlgeschlagen");
    }

    // 3) Suche nach Einlagerungsnummer durchführen
    $x = xp($searchHtml);
    
    // Debug: Alle verfügbaren Formulare anzeigen
    $allForms = $x->query("//form");
    debugLog("Verfügbare Formulare: " . $allForms->length);
    for ($i = 0; $i < $allForms->length; $i++) {
        $f = $allForms->item($i);
        debugLog("Formular $i: action=" . $f->getAttribute('action') . ", method=" . $f->getAttribute('method'));
        
        // Alle Input-Felder in diesem Formular
        $inputs = $x->query(".//input", $f);
        debugLog("  Input-Felder: " . $inputs->length);
        for ($j = 0; $j < $inputs->length; $j++) {
            $in = $inputs->item($j);
            debugLog("    - Name: " . $in->getAttribute('name') . ", Type: " . $in->getAttribute('type'));
        }
    }
    
    $searchForm = q1($x, "//form[.//input[@name='LGS_ID']]");
    if (!$searchForm) {
        debugLog("Suchformular mit LGS_ID nicht gefunden");
        debugLog("Suche nach alternativen Formularen...");
        
        // Fallback: Erstes Formular verwenden
        $searchForm = $allForms->item(0);
        if (!$searchForm) {
            debugLog("Kein Formular gefunden");
            throw new RuntimeException("Kein Formular gefunden");
        }
        debugLog("Verwende erstes verfügbares Formular");
    }
    
    $searchAction = $searchForm->getAttribute('action') ?: '/index.php?m=wheelset&a=search';
    debugLog("Suchformular gefunden: action=$searchAction");
    
    // Suchfelder sammeln
    $searchFields = [];
    foreach ($x->query(".//input", $searchForm) as $in) {
        $name = $in->getAttribute('name');
        if ($name === '') continue;
        $searchFields[$name] = $in->getAttribute('value');
    }
    
    // Einlagerungsnummer und Autohaus setzen
    $searchFields['LGS_ID'] = $storageNumber;
    $searchFields['KD_ID'] = '453'; // 113 B & E / Kiel (basierend auf der HTML)
    
    debugLog("Suchfelder: " . json_encode($searchFields));
    
    // Suche durchführen
    [$st, $searchResult] = http([
        CURLOPT_URL => absUrl($BASE_URL, $searchAction),
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($searchFields),
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
    ]);
    
    debugLog("Suche-Ergebnis: HTTP $st, Länge: " . strlen($searchResult));
    
    if ($st >= 400) {
        debugLog("Suche fehlgeschlagen: HTTP $st");
        throw new RuntimeException("Suche fehlgeschlagen: HTTP $st");
    }
    
    // 4) Detail-Link aus Suchergebnissen finden
    $x = xp($searchResult);
    $detailHref = null;
    
    // Suche nach Link mit LGS_ID Parameter
    $links = $x->query("//a[contains(@href, 'LGS_ID=')]");
    foreach ($links as $link) {
        $href = $link->getAttribute('href');
        if (strpos($href, $storageNumber) !== false) {
            $detailHref = $href;
            break;
        }
    }
    
    if (!$detailHref) {
        debugLog("Detail-Link für Einlagerungsnummer $storageNumber nicht gefunden");
        throw new RuntimeException("Detail-Link für Einlagerungsnummer $storageNumber nicht gefunden");
    }
    
    debugLog("Detail-Link gefunden: $detailHref");

    // 4) Detailseite laden
    [$st, $detailHtml] = http([CURLOPT_URL => absUrl($BASE_URL, $detailHref), CURLOPT_HTTPGET => true]);
    if ($st >= 400) throw new RuntimeException("Detailseite HTTP $st");
    $x = xp($detailHtml);

    // 5) Daten scrapen - basierend auf der tatsächlichen HTML-Struktur
    $einlagerung = [
        'satznummer'    => nodeText(q1($x, "//p[@id='WheelLGS_ID']/b")),
        'eingangsdatum' => nodeText(q1($x, "//p[@id='WheelEINGANG']/b")),
        'serviceart'    => nodeText(q1($x, "//p[@id='WheelSERV_ID']/b")),
        'ausgangsdatum' => nodeText(q1($x, "//p[@id='WheelAUSGANG']/b")),
        'autohaus'      => nodeText(q1($x, "//p[@id='WheelKD_ID']/b")),
    ];
    
    $fahrzeug = [
        'kennzeichen' => nodeText(q1($x, "//p[@id='WheelKENNZEICHEN']/b")),
        'fin'         => nodeText(q1($x, "//p[@id='WheelFAHRGESTNR']/b")),
    ];
    
    debugLog("Einlagerungsdaten: " . json_encode($einlagerung));
    debugLog("Fahrzeugdaten: " . json_encode($fahrzeug));

    // Reifen-Tabelle
    $tabMeta = [];
    foreach ($x->query("//table[@id='WheelsetViewWheels']//tbody//tr") as $tr) {
        $tds = $x->query("./td", $tr); 
        if ($tds->length < 14) continue;
        $pos   = trim($tds->item(0)->textContent ?? '');
        $size  = trim($tds->item(1)->textContent ?? '');
        $brand = trim($tds->item(5)->textContent ?? '');
        $model = trim($tds->item(6)->textContent ?? '');
        $dot   = trim($tds->item(8)->textContent ?? '');
        $type  = trim($tds->item(11)->textContent ?? '');
        $empf  = trim($tds->item(13)->textContent ?? '');
        if ($pos) $tabMeta[$pos] = compact('size','brand','model','dot','type','empf');
    }

    // Profiltiefen je Position - basierend auf der tatsächlichen HTML-Struktur
    $posList = ['VL','VR','HL','HR'];
    $reifen  = [];
    foreach ($posList as $p) {
        $box = q1($x, "//div[@id='{$p}']");
        $i = $m = $a = 0.0;
        if ($box) {
            // Profiltiefen aus den korrekten Selektoren extrahieren
            $i = deFloat(nodeText(q1($x, ".//div[@id='Profilinnen']//p/b", $box)));
            $m = deFloat(nodeText(q1($x, ".//div[@id='Profilmitte']//p/b", $box)));
            $a = deFloat(nodeText(q1($x, ".//div[@id='Profilaussen']//p/b", $box)));
            
            debugLog("Position $p: I=$i, M=$m, A=$a");
        }
        $avg = round((($i+$m+$a)/3), 1);
        $state = $avg >= 6.0 ? 'sehr_gut' : ($avg >= 4.0 ? 'gut' : ($avg >= 2.0 ? 'befriedigend' : 'kritisch'));
        $meta  = $tabMeta[$p] ?? ['size'=>'','brand'=>'','model'=>'','dot'=>'','type'=>'','empf'=>''];
        $s = strtolower($meta['empf'] ?? '');
        $status = (strpos($s,'austausch') !== false) ? 'austausch_notwendig' : ((strpos($s,'prüf') !== false) ? 'pruefen' : 'keine_fehler');

        $reifen[] = [
            'pos' => $p,
            'size'=>$meta['size'] ?: '—',
            'brand'=>$meta['brand'] ?: '—',
            'model'=>$meta['model'] ?: '—',
            'dot'=>$meta['dot'] ?: '—',
            'type'=>$meta['type'] ?: '—',
            'profile'=>['innen'=>$i,'mitte'=>$m,'aussen'=>$a],
            'avg'=>$avg, 'state'=>$state, 'status'=>$status,
            'empfehlung'=>$meta['empf'] ?: '—',
        ];
    }

    // 6) JSON-Ausgabe
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'success' => true,
        'data' => [
            'satznummer'    => $einlagerung['satznummer'],
            'eingangsdatum' => $einlagerung['eingangsdatum'],
            'serviceart'    => $einlagerung['serviceart'],
            'ausgangsdatum' => $einlagerung['ausgangsdatum'],
            'autohaus'      => $einlagerung['autohaus'],
            'fahrzeug'      => $fahrzeug,
            'reifen'        => $reifen,
        ],
        'metadata' => [
            'timestamp' => date('Y-m-d H:i:s'),
            'storage_number' => $storageNumber,
            'source' => 'RESY_REAL_DATA'
        ]
    ], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
    http_response_code(500);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'metadata' => [
            'timestamp' => date('Y-m-d H:i:s'),
            'storage_number' => $storageNumber,
            'source' => 'RESY_ERROR'
        ]
    ], JSON_UNESCAPED_UNICODE);
} finally {
    @unlink($cookieFile ?? '');
}
