<?php
/**
 * RESY Tire Status API - Hybrid-Lösung
 * 
 * 1. Versuche RESY-Login (für aktuelle Daten)
 * 2. Falls RESY fehlschlägt, verwende Datenbank-Fallback
 * 3. Speichere Daten in Datenbank für zukünftige Aufrufe
 */

// Debug-Logging aktivieren
function debugLog($message) {
    error_log("RESY_API_DEBUG: $message");
}

// Setze Content-Type
header('Content-Type: application/json; charset=utf-8');

// Debug-Logging starten
debugLog("API gestartet");

// ---------- Input-Verarbeitung ----------
$input = file_get_contents('php://input');
$requestData = json_decode($input, true) ?: [];

debugLog("JSON Input: " . ($input ?: 'null'));

// Parameter extrahieren
$storageNumber = $requestData['storage_number'] ?? $_POST['storage_number'] ?? $_GET['storage_number'] ?? '';
$appointmentDate = $requestData['appointment_date'] ?? $_POST['appointment_date'] ?? $_GET['appointment_date'] ?? '';
$eventName = $requestData['event_name'] ?? $_POST['event_name'] ?? $_GET['event_name'] ?? '';
$autohausId = $requestData['autohaus_id'] ?? $_POST['autohaus_id'] ?? $_GET['autohaus_id'] ?? 1;

debugLog("Storage Number: $storageNumber");
debugLog("Appointment Date: $appointmentDate");
debugLog("Event Name: $eventName");

if (!$storageNumber) {
    debugLog("ERROR: storage_number fehlt");
    echo json_encode(['success' => false, 'error' => 'storage_number fehlt']);
    exit;
}

// ---------- Datenbank-Verbindung ----------
try {
    require_once __DIR__ . '/../config/database.php';
    $db = new Database();
    $pdo = $db->getConnection();
    
    debugLog("Versuche DB-Verbindung für Autohaus-ID: $autohausId");
    
    // Autohaus-Daten laden
    $stmt = $pdo->prepare("SELECT resy_username, resy_password, resy_base_url FROM autohaus WHERE id = ?");
    $stmt->execute([$autohausId]);
    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$autohaus) {
        throw new RuntimeException("Autohaus mit ID $autohausId nicht gefunden");
    }
    
    debugLog("DB-Verbindung erfolgreich");
    debugLog("Autohaus-Daten: " . json_encode($autohaus));
    
    $RESY_USER = $autohaus['resy_username'];
    $RESY_PASS = $autohaus['resy_password'];
    $BASE_URL = $autohaus['resy_base_url'];
    
    debugLog("RESY_USER: $RESY_USER");
    debugLog("BASE_URL: $BASE_URL");
    
} catch (Exception $e) {
    debugLog("DB-Fehler: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'Datenbankfehler: ' . $e->getMessage()]);
    exit;
}

// ---------- HTTP-Helper-Funktionen ----------
function http(array $opts = []): array {
    $UA = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
    $cookieFile = tempnam(sys_get_temp_dir(), 'res') . '.tmp';
    
    $base = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT      => $UA,
        CURLOPT_CONNECTTIMEOUT => 30,
        CURLOPT_TIMEOUT        => 60,
        CURLOPT_COOKIEJAR      => $cookieFile,
        CURLOPT_COOKIEFILE     => $cookieFile,
        CURLOPT_HEADER         => false,
        CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
        CURLOPT_ENCODING       => 'gzip,deflate',
        CURLOPT_REFERER        => 'https://resy.dtm-reifen.de/',
    ];
    
    $ch = curl_init();
    curl_setopt_array($ch, $base + $opts);
    $body = curl_exec($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $info = curl_getinfo($ch);
    debugLog("cURL Info: HTTP $status, URL: " . $info['url']);
    curl_close($ch);
    return [$status, $body];
}

function absUrl(string $base, string $path): string {
    if (preg_match('~^https?://~i', $path)) return $path;
    
    if (strpos($path, './') === 0) {
        $path = substr($path, 2);
    }
    
    return rtrim($base, '/') . '/' . ltrim($path, '/');
}

function xp(string $html): DOMXPath {
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html, LIBXML_NOWARNING|LIBXML_NOERROR|LIBXML_NONET|LIBXML_COMPACT);
    return new DOMXPath($doc);
}

function q1(DOMXPath $x, string $query, $context = null) {
    $nodes = $x->query($query, $context);
    return $nodes->length > 0 ? $nodes->item(0) : null;
}

function nodeText($node): string {
    return $node ? trim($node->textContent) : '';
}

function deFloat(string $str): float {
    $str = str_replace(',', '.', $str);
    $str = preg_replace('/[^0-9.-]/', '', $str);
    return floatval($str);
}

// ---------- Datenbank-Fallback-Funktion ----------
function getDataFromDatabase($pdo, $storageNumber) {
    debugLog("Prüfe Datenbank auf vorhandene Daten für Einlagerungsnummer: $storageNumber");
    $stmt = $pdo->prepare("SELECT * FROM resy_tire_data WHERE einlagerungsnummer = ? OR satznummer = ? ORDER BY updated_at DESC LIMIT 1");
    $stmt->execute([$storageNumber, $storageNumber]);
    $existingData = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$existingData) {
        debugLog("❌ Keine Daten in Datenbank gefunden");
        return null;
    }

    debugLog("✅ Daten in Datenbank gefunden! Verwende vorhandene Daten.");
    
    // Konvertiere Datenbank-Daten in das erwartete Format
    $einlagerung = [
        'satznummer'    => $existingData['satznummer'],
        'eingangsdatum' => $existingData['eingangsdatum'] !== '0000-00-00' ? $existingData['eingangsdatum'] : '',
        'serviceart'    => $existingData['serviceart'],
        'ausgangsdatum' => '', // Nicht in der DB gespeichert
        'autohaus'      => $existingData['autohaus_id'] . ' B & E / Kiel', // Vereinfacht
    ];
    
    $fahrzeug = [
        'kennzeichen' => $existingData['kennzeichen'],
        'fin'         => $existingData['fin'],
    ];
    
    // Parse Reifendaten aus JSON
    $reifen = [];
    $positions = ['VL', 'VR', 'HL', 'HR'];
    foreach ($positions as $pos) {
        $tireData = json_decode($existingData['reifen_' . strtolower($pos)], true);
        if ($tireData) {
            // Berechne Durchschnitt der Profiltiefen
            $innen = floatval($tireData['profiltiefe']['i'] ?? 0);
            $mitte = floatval($tireData['profiltiefe']['m'] ?? 0);
            $aussen = floatval($tireData['profiltiefe']['a'] ?? 0);
            $avg = round(($innen + $mitte + $aussen) / 3, 1);
            
            // Bestimme Zustand basierend auf Durchschnitt
            $state = 'unbekannt';
            if ($avg >= 6.0) {
                $state = 'sehr_gut';
            } elseif ($avg >= 4.0) {
                $state = 'gut';
            } elseif ($avg >= 2.0) {
                $state = 'befriedigend';
            } else {
                $state = 'kritisch';
            }
            
            $reifen[] = [
                'pos' => $pos,
                'size' => $tireData['groesse'] ?? '',
                'brand' => $tireData['marke'] ?? '',
                'model' => $tireData['modell'] ?? '',
                'dot' => $tireData['dot'] ?? '',
                'type' => $tireData['art'] ?? '',
                'profile' => [
                    'innen' => $innen,
                    'mitte' => $mitte,
                    'aussen' => $aussen,
                ],
                'avg' => $avg,
                'state' => $state,
                'status' => $tireData['zustand'] ?? 'unbekannt',
                'empfehlung' => $tireData['empfehlung'] ?? '',
            ];
        }
    }
    
    return [
        'einlagerung' => $einlagerung,
        'fahrzeug' => $fahrzeug,
        'reifen' => $reifen,
        'source' => 'DATABASE_CACHE',
        'cached_at' => $existingData['updated_at'],
    ];
}

// ---------- Hauptfunktion ----------
$resySuccess = false;
$scrapedData = null;

// 1. Versuche zuerst RESY-Login
try {
    debugLog("Versuche RESY-Login für Einlagerungsnummer: $storageNumber");

    // Login-Seite laden
    debugLog("Versuche Login-Seite zu laden: " . absUrl($BASE_URL, '/index.php?m=login&a=login'));
    [$st, $loginHtml] = http([CURLOPT_URL => absUrl($BASE_URL, '/index.php?m=login&a=login'), CURLOPT_HTTPGET => true]);
    debugLog("Login-Seite Response: HTTP $st, Länge: " . strlen($loginHtml));
    
    if ($st >= 400) {
        throw new RuntimeException("Login-Seite HTTP $st");
    }

    $x = xp($loginHtml);
    $form = q1($x, "//form[.//input[@type='password']]");
    if (!$form) {
        throw new RuntimeException("Login-Formular nicht gefunden");
    }

    $action = $form->getAttribute('action') ?: '/index.php?m=login';
    debugLog("Login-Formular gefunden: action=$action");
    
    $fields = [];
    foreach ($x->query(".//input", $form) as $in) {
        $type = strtolower($in->getAttribute('type'));
        $name = $in->getAttribute('name');
        if ($name === '' || in_array($type, ['submit','button','image'])) continue;
        $fields[$name] = $in->getAttribute('value');
    }
    
    $fields['UserName'] = $RESY_USER;
    $fields['Password'] = $RESY_PASS;
    
    debugLog("Finale Login-Daten: " . json_encode($fields));

    [$st, $afterLogin] = http([
        CURLOPT_URL => absUrl($BASE_URL, $action),
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($fields),
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
    ]);
    
    debugLog("Login-POST Response: HTTP $st, Länge: " . strlen($afterLogin));
    
    if ($st >= 400) {
        throw new RuntimeException("Login-POST HTTP $st");
    }
    
    // Prüfe ob Login erfolgreich war
    if (strpos($afterLogin, 'Anmeldename') !== false) {
        throw new RuntimeException("Login fehlgeschlagen - Anmeldedaten möglicherweise falsch");
    }
    
    debugLog("Login erfolgreich - Weiterleitung zur Suche");

    // Suche durchführen
    debugLog("Versuche Suche-Seite zu laden: " . absUrl($BASE_URL, '/index.php?m=wheelset&a=search'));
    [$st, $searchHtml] = http([CURLOPT_URL => absUrl($BASE_URL, '/index.php?m=wheelset&a=search'), CURLOPT_HTTPGET => true]);
    debugLog("Suche-Seite Response: HTTP $st, Länge: " . strlen($searchHtml));
    
    if ($st >= 400) {
        throw new RuntimeException("Suche-Seite HTTP $st");
    }

    $x = xp($searchHtml);
    $searchForm = q1($x, "//form[.//input[@name='LGS_ID']]");
    
    if (!$searchForm) {
        throw new RuntimeException("Suchformular nicht gefunden");
    }
    
    $searchAction = $searchForm->getAttribute('action') ?: '/index.php?m=wheelset&a=search';
    debugLog("Suchformular gefunden: action=$searchAction");
    
    $searchFields = [];
    foreach ($x->query(".//input", $searchForm) as $in) {
        $name = $in->getAttribute('name');
        if ($name) $searchFields[$name] = $in->getAttribute('value');
    }
    
    $searchFields['LGS_ID'] = $storageNumber;
    $searchFields['KD_ID'] = '453'; // 113 B & E / Kiel
    
    debugLog("Suchfelder: " . json_encode($searchFields));
    
    [$st, $searchResult] = http([
        CURLOPT_URL => absUrl($BASE_URL, $searchAction),
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($searchFields),
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
    ]);
    
    debugLog("Suche-Response: HTTP $st, Länge: " . strlen($searchResult));
    
    if ($st >= 400) {
        throw new RuntimeException("Suche HTTP $st");
    }
    
    // Detail-Link finden
    $x = xp($searchResult);
    $detailLink = null;
    
    $links = $x->query("//a[contains(@href, 'LGS_ID')]");
    foreach ($links as $link) {
        $href = $link->getAttribute('href');
        if (strpos($href, $storageNumber) !== false) {
            $detailLink = $href;
            break;
        }
    }
    
    if (!$detailLink) {
        throw new RuntimeException("Einlagerungsnummer $storageNumber nicht im RESY-System gefunden");
    }
    
    debugLog("Detail-Link gefunden: $detailLink");
    
    // Detail-Seite laden
    [$st, $detailHtml] = http([CURLOPT_URL => absUrl($BASE_URL, $detailLink), CURLOPT_HTTPGET => true]);
    debugLog("Detail-Seite Response: HTTP $st, Länge: " . strlen($detailHtml));
    
    if ($st >= 400) {
        throw new RuntimeException("Detail-Seite HTTP $st");
    }
    
    // Daten scrapen
    $x = xp($detailHtml);
    
    $einlagerung = [
        'satznummer'    => nodeText(q1($x, "//p[@id='WheelLGS_ID']/b")),
        'eingangsdatum' => nodeText(q1($x, "//p[@id='WheelEINGANG']/b")),
        'serviceart'    => nodeText(q1($x, "//p[@id='WheelSERV_ID']/b")),
        'ausgangsdatum' => nodeText(q1($x, "//p[@id='WheelAUSGANG']/b")),
        'autohaus'      => nodeText(q1($x, "//p[@id='WheelKD_ID']/b")),
    ];
    
    $fahrzeug = [
        'kennzeichen' => nodeText(q1($x, "//p[@id='WheelKENNZEICHEN']/b")),
        'fin'         => nodeText(q1($x, "//p[@id='WheelFAHRGESTNR']/b")),
    ];
    
    $reifen = [];
    $rows = $x->query("//table[@id='WheelsetViewWheels']//tbody//tr");
    
    foreach ($rows as $row) {
        $cells = $x->query(".//td", $row);
        if ($cells->length < 14) continue;
        
        $pos = nodeText($cells->item(0));
        $size = nodeText($cells->item(1));
        $brand = nodeText($cells->item(5));
        $model = nodeText($cells->item(6));
        $dot = nodeText($cells->item(8));
        $type = nodeText($cells->item(11));
        $empfehlung = nodeText($cells->item(13));
        
        $box = q1($x, "//div[@id='$pos']");
        if ($box) {
            $innen = deFloat(nodeText(q1($x, ".//div[@id='Profilinnen']//p/b", $box)));
            $mitte = deFloat(nodeText(q1($x, ".//div[@id='Profilmitte']//p/b", $box)));
            $aussen = deFloat(nodeText(q1($x, ".//div[@id='Profilaussen']//p/b", $box)));
            
            $avg = round(($innen + $mitte + $aussen) / 3, 1);
            
            $state = 'unbekannt';
            if ($avg >= 6.0) {
                $state = 'sehr_gut';
            } elseif ($avg >= 4.0) {
                $state = 'gut';
            } elseif ($avg >= 2.0) {
                $state = 'befriedigend';
            } else {
                $state = 'kritisch';
            }
            
            $status = 'keine_fehler';
            if (strpos(strtolower($empfehlung), 'austausch') !== false) {
                $status = 'austausch_notwendig';
            } elseif (strpos(strtolower($empfehlung), 'prüf') !== false) {
                $status = 'pruefen';
            }
            
            $reifen[] = [
                'pos' => $pos,
                'size' => $size,
                'brand' => $brand,
                'model' => $model,
                'dot' => $dot,
                'type' => $type,
                'profile' => [
                    'innen' => $innen,
                    'mitte' => $mitte,
                    'aussen' => $aussen,
                ],
                'avg' => $avg,
                'state' => $state,
                'status' => $status,
                'empfehlung' => $empfehlung,
            ];
        }
    }
    
    $resySuccess = true;
    $scrapedData = [
        'einlagerung' => $einlagerung,
        'fahrzeug' => $fahrzeug,
        'reifen' => $reifen,
        'source' => 'RESY_LIVE_DATA',
    ];
    
    debugLog("✅ RESY-Scraping erfolgreich!");
    
} catch (Exception $e) {
    debugLog("❌ RESY-Login fehlgeschlagen: " . $e->getMessage());
    $resySuccess = false;
}

// 2. Falls RESY fehlgeschlagen, verwende Datenbank-Fallback
if (!$resySuccess) {
    debugLog("Verwende Datenbank-Fallback...");
    $scrapedData = getDataFromDatabase($pdo, $storageNumber);
    
    if (!$scrapedData) {
        debugLog("❌ Keine Daten in Datenbank gefunden");
        $response = [
            'success' => false,
            'error' => "Einlagerungsnummer $storageNumber nicht im RESY-System gefunden",
            'metadata' => [
                'timestamp' => date('Y-m-d H:i:s'),
                'storage_number' => $storageNumber,
                'source' => 'NOT_FOUND',
                'suggestion' => 'Bitte prüfen Sie die Einlagerungsnummer oder kontaktieren Sie den Administrator'
            ]
        ];
        echo json_encode($response);
        exit;
    }
}

// 3. Speichere Daten in Datenbank (falls RESY erfolgreich war)
if ($resySuccess) {
    debugLog("Speichere RESY-Daten in Datenbank...");
    
    // Bereite Reifendaten für JSON-Speicherung vor
    $reifenData = [];
    foreach ($scrapedData['reifen'] as $tire) {
        $pos = strtolower($tire['pos']);
        $reifenData["reifen_$pos"] = json_encode([
            'position' => $tire['pos'],
            'marke' => $tire['brand'],
            'modell' => $tire['model'],
            'groesse' => $tire['size'],
            'art' => $tire['type'],
            'dot' => $tire['dot'],
            'profiltiefe' => [
                'i' => $tire['profile']['innen'],
                'm' => $tire['profile']['mitte'],
                'a' => $tire['profile']['aussen'],
            ],
            'empfehlung' => $tire['empfehlung'],
            'zustand' => $tire['status'],
        ]);
    }
    
    // Prüfe ob Datensatz bereits existiert
    $stmt = $pdo->prepare("SELECT id FROM resy_tire_data WHERE einlagerungsnummer = ? OR satznummer = ?");
    $stmt->execute([$storageNumber, $storageNumber]);
    $existingId = $stmt->fetchColumn();
    
    if ($existingId) {
        // Update vorhandenen Datensatz
        $stmt = $pdo->prepare("
            UPDATE resy_tire_data SET 
                satznummer = ?, eingangsdatum = ?, serviceart = ?, 
                autohaus_id = ?, kennzeichen = ?, fin = ?, 
                reifen_vl = ?, reifen_vr = ?, reifen_hl = ?, reifen_hr = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $scrapedData['einlagerung']['satznummer'],
            $scrapedData['einlagerung']['eingangsdatum'],
            $scrapedData['einlagerung']['serviceart'],
            $autohausId,
            $scrapedData['fahrzeug']['kennzeichen'],
            $scrapedData['fahrzeug']['fin'],
            $reifenData['reifen_vl'] ?? '',
            $reifenData['reifen_vr'] ?? '',
            $reifenData['reifen_hl'] ?? '',
            $reifenData['reifen_hr'] ?? '',
            $existingId
        ]);
        debugLog("Datensatz aktualisiert: ID $existingId");
    } else {
        // Neuen Datensatz erstellen
        $stmt = $pdo->prepare("
            INSERT INTO resy_tire_data 
            (einlagerungsnummer, satznummer, eingangsdatum, serviceart, autohaus_id, 
             kennzeichen, fin, reifen_vl, reifen_vr, reifen_hl, reifen_hr, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        $stmt->execute([
            $storageNumber,
            $scrapedData['einlagerung']['satznummer'],
            $scrapedData['einlagerung']['eingangsdatum'],
            $scrapedData['einlagerung']['serviceart'],
            $autohausId,
            $scrapedData['fahrzeug']['kennzeichen'],
            $scrapedData['fahrzeug']['fin'],
            $reifenData['reifen_vl'] ?? '',
            $reifenData['reifen_vr'] ?? '',
            $reifenData['reifen_hl'] ?? '',
            $reifenData['reifen_hr'] ?? '',
        ]);
        debugLog("Neuer Datensatz erstellt");
    }
}

// 4. Response erstellen
$response = [
    'success' => true,
    'data' => [
        'einlagerung' => $scrapedData['einlagerung'],
        'fahrzeug' => $scrapedData['fahrzeug'],
        'reifen' => $scrapedData['reifen'],
    ],
    'metadata' => [
        'timestamp' => date('Y-m-d H:i:s'),
        'storage_number' => $storageNumber,
        'source' => $scrapedData['source'],
        'cached' => true,
    ]
];

if (isset($scrapedData['cached_at'])) {
    $response['metadata']['cached_at'] = $scrapedData['cached_at'];
}

debugLog("Finale Response erstellt: " . json_encode($response));
echo json_encode($response);
?>
