<?php
/**
 * Autohaus-spezifische WhatsApp-Benachrichtigungen mit HelloMateo API
 * Erweiterte Version mit Multi-Autohaus-Unterstützung
 */

// Ermittle das Root-Verzeichnis des Projekts (unabhängig vom Ausführungsverzeichnis)
// Diese Datei kann von verschiedenen Stellen aufgerufen werden (cron, api, admin)
$rootDir = dirname(__DIR__);

// Lade benötigte Dateien mit absolutem Pfad
require_once $rootDir . '/config/database.php';
require_once $rootDir . '/admin/includes/hellomateo_api.php';
require_once $rootDir . '/admin/includes/template_config_helper.php';

class AutohausWhatsAppNotifications {
    private $db;
    private $conn;
    private $autohausId;
    private $autohausSettings;
    
    public function __construct($autohausId = null) {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
        $this->autohausId = $autohausId;
        $this->autohausSettings = $this->getAutohausSettings();
    }
    
    /**
     * Sendet eine WhatsApp-Benachrichtigung für Terminstatusänderungen
     */
    public function sendAppointmentNotification($appointmentId, $action) {
        try {
            // Hole Termindetails
            $appointment = $this->getAppointmentDetails($appointmentId);
            if (!$appointment) {
                return ['success' => false, 'error' => 'Termin nicht gefunden'];
            }
            
            // Hole Telefonnummer (aus Termin ODER Kundenprofil)
            $phoneNumber = $this->getCustomerPhoneNumber($appointment);
            if (empty($phoneNumber)) {
                return ['success' => false, 'error' => 'Keine gültige Mobiltelefonnummer für Termin vorhanden'];
            }
            
            // Prüfe Opt-out
            if ($this->isCustomerOptedOut($appointment)) {
                return ['success' => false, 'error' => 'Kunde hat WhatsApp-Benachrichtigungen deaktiviert'];
            }
            
            // Prüfe ob WhatsApp-Benachrichtigungen aktiviert sind
            if (!$this->isWhatsAppNotificationsEnabled()) {
                return ['success' => true, 'message' => 'WhatsApp-Benachrichtigungen deaktiviert'];
            }
            
            // Prüfe ob Template für diese Aktion verwendet werden soll
            $useTemplate = false;
            $templateId = null;
            $placeholderValues = null;
            
            // Mappe Aktion zu action_type
            $actionTypeMap = [
                'reminder' => 'reminder_24h', // Standard für reminder (kann über Timing konfiguriert werden)
                'reminder_24h' => 'reminder_24h',
                'reminder_48h' => 'reminder_48h',
                'confirmation' => 'confirmation',
                'approved' => 'approved',
                'rejected' => 'rejected',
                'cancellation' => 'cancellation',
                'rescheduled' => 'rescheduled',
                'review_request' => 'review_request'
            ];
            
            $actionType = $actionTypeMap[$action] ?? $action; // Falls Aktion bereits action_type Format hat
            
            // Hole Template-Konfiguration aus Datenbank
            $templateConfig = null;
            if ($actionType) {
                $templateConfig = $this->getTemplateConfig($actionType);
            }
            
            // Prüfe ob is_enabled Spalte existiert
            $stmt = $this->conn->query("SHOW COLUMNS FROM whatsapp_template_configs LIKE 'is_enabled'");
            $hasIsEnabled = $stmt->rowCount() > 0;
            
            // Verwende Template-Konfiguration wenn vorhanden und aktiv
            $isConfigEnabled = $templateConfig && $templateConfig['is_active'] == 1;
            if ($hasIsEnabled && $templateConfig) {
                $isConfigEnabled = $isConfigEnabled && ($templateConfig['is_enabled'] == 1 || $templateConfig['is_enabled'] === null);
            }
            
            if ($isConfigEnabled) {
                // Verwende konfigurierte Template-Konfiguration
                $useTemplate = true;
                $templateId = $templateConfig['template_id'];
                
                // Parse custom_fields_mapping
                $customFieldsMapping = [];
                if (!empty($templateConfig['custom_fields_mapping'])) {
                    if (is_string($templateConfig['custom_fields_mapping'])) {
                        $customFieldsMapping = json_decode($templateConfig['custom_fields_mapping'], true) ?? [];
                    } else {
                        $customFieldsMapping = $templateConfig['custom_fields_mapping'];
                    }
                }
                
                // Falls kein Mapping vorhanden, verwende Standard
                if (empty($customFieldsMapping)) {
                    $customFieldsMapping = getDefaultCustomFieldsMapping($actionType);
                }
                
                // Extrahiere Custom Fields mit erweiterten Feldern
                $placeholderValues = extractCustomerFieldsForTemplate($appointment, $customFieldsMapping);
                
                error_log("WhatsApp (Autohaus) - ✅ Verwende Template-Konfiguration: " . $actionType);
                error_log("WhatsApp (Autohaus) - Template-ID: " . $templateId);
                error_log("WhatsApp (Autohaus) - Custom Fields Mapping: " . json_encode($customFieldsMapping));
                error_log("WhatsApp (Autohaus) - Platzhalter-Werte: " . json_encode($placeholderValues));
            } else {
                // KEINE Template-Konfiguration gefunden - Logge Warnung
                error_log("WhatsApp (Autohaus) - ⚠️ WARNUNG: Keine aktive Template-Konfiguration für Aktion '{$actionType}' gefunden.");
                error_log("WhatsApp (Autohaus) - Bitte konfigurieren Sie ein Template in admin/whatsapp_reminders_admin.php");
                
                // Für confirmation ist es kritisch - Fehler zurückgeben
                if ($actionType === 'confirmation') {
                    return ['success' => false, 'error' => "Keine Template-Konfiguration für 'Bestätigung' gefunden. Bitte konfigurieren Sie ein Template in der WhatsApp-Verwaltung."];
                }
                
                // Für andere Aktionen: Warnung aber nicht blockieren (kann optional sein)
                return ['success' => false, 'error' => "Keine Template-Konfiguration für '{$actionType}' gefunden. Bitte konfigurieren Sie ein Template in der WhatsApp-Verwaltung."];
            }
            
            // Erstelle WhatsApp-Nachricht basierend auf Aktion (nur wenn kein Template verwendet wird)
            $message = $useTemplate ? null : $this->createWhatsAppMessage($appointment, $action);
            
            // Debug: Log WhatsApp-Details
            error_log("WhatsApp (Autohaus) - Sende Nachricht an: " . $phoneNumber);
            if ($useTemplate) {
                error_log("WhatsApp (Autohaus) - Verwende Template: " . $templateId);
            } else {
                error_log("WhatsApp (Autohaus) - Nachricht: " . $message);
            }
            error_log("WhatsApp (Autohaus) - Aktion: " . $action);
            error_log("WhatsApp (Autohaus) - Autohaus: " . $this->autohausSettings['name']);
            
            // Sende WhatsApp-Nachricht
            $result = $this->sendWhatsAppMessage($phoneNumber, $message, $templateId, null, $placeholderValues, $appointmentId, $action);
            
            error_log("WhatsApp (Autohaus) - Nachricht-Ergebnis: " . json_encode($result));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("WhatsApp (Autohaus) - Nachricht-Fehler: " . $e->getMessage());
            return ['success' => false, 'error' => 'WhatsApp-Nachricht konnte nicht gesendet werden: ' . $e->getMessage()];
        }
    }
    
    /**
     * Holt Telefonnummer für Termin (aus Termin ODER Kundenprofil)
     * Priorität: 1. appointments.phone, 2. users.phone_number
     * Validiert ob Mobiltelefonnummer
     */
    private function getCustomerPhoneNumber($appointment) {
        // 1. Versuche Telefonnummer aus Termin
        $phone = $appointment['phone'] ?? null;
        
        // 2. Falls nicht vorhanden und user_id vorhanden, hole aus Profil
        if (empty($phone) && !empty($appointment['user_id'])) {
            $stmt = $this->conn->prepare("SELECT phone_number FROM users WHERE id = ?");
            $stmt->execute([$appointment['user_id']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($user && !empty($user['phone_number'])) {
                $phone = $user['phone_number'];
            }
        }
        
        // 3. Validiere und formatiere Telefonnummer
        if (empty($phone)) {
            return null;
        }
        
        return $this->validateAndFormatPhoneNumber($phone);
    }
    
    /**
     * Validiert und formatiert Telefonnummer für WhatsApp
     * Prüft ob Mobiltelefonnummer (nicht Festnetz)
     */
    private function validateAndFormatPhoneNumber($phone) {
        // Entferne alle Leerzeichen und Sonderzeichen außer +
        $phone = preg_replace('/[^0-9+]/', '', trim($phone));
        
        if (empty($phone)) {
            return null;
        }
        
        // Formatierung für deutsche Nummern
        // Wenn mit 0 beginnt, ersetze durch +49
        if (str_starts_with($phone, '0')) {
            $phone = '+49' . substr($phone, 1);
        } elseif (!str_starts_with($phone, '+')) {
            // Wenn keine + am Anfang, füge +49 hinzu (für deutsche Nummern)
            $phone = '+49' . $phone;
        }
        
        // Prüfe ob Mobiltelefonnummer (deutsche Mobilfunk-Präfixe: 015, 016, 017)
        // Format: +49 15..., +49 16..., +49 17...
        $mobilePrefixes = ['+4915', '+4916', '+4917'];
        $isMobile = false;
        foreach ($mobilePrefixes as $prefix) {
            if (str_starts_with($phone, $prefix)) {
                $isMobile = true;
                break;
            }
        }
        
        // Falls nicht als Mobilnummer erkannt, aber mit +49 beginnt und 10-13 Ziffern hat, akzeptiere trotzdem
        // (kann auch ausländische Nummer sein)
        if (!$isMobile && str_starts_with($phone, '+49')) {
            $digitsOnly = preg_replace('/[^0-9]/', '', $phone);
            if (strlen($digitsOnly) >= 10 && strlen($digitsOnly) <= 13) {
                // Akzeptiere als potenzielle Mobilnummer
                $isMobile = true;
            }
        }
        
        // Falls nicht +49, könnte es internationale Nummer sein - akzeptiere wenn gültig
        if (!$isMobile && str_starts_with($phone, '+')) {
            $digitsOnly = preg_replace('/[^0-9]/', '', $phone);
            if (strlen($digitsOnly) >= 10 && strlen($digitsOnly) <= 15) {
                // Internationale Nummer - akzeptiere
                return $phone;
            }
        }
        
        return $isMobile ? $phone : null;
    }
    
    /**
     * Prüft ob Kunde Opt-out hat
     */
    private function isCustomerOptedOut($appointment) {
        if (empty($appointment['user_id'])) {
            return false; // Kein user_id = kein Opt-out möglich
        }
        
        $stmt = $this->conn->prepare("SELECT whatsapp_opt_out FROM users WHERE id = ?");
        $stmt->execute([$appointment['user_id']]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $user && ($user['whatsapp_opt_out'] == 1 || $user['whatsapp_opt_out'] == true);
    }
    
    /**
     * Holt Autohaus-spezifische WhatsApp-Einstellungen
     */
    private function getAutohausSettings() {
        if (!$this->autohausId) {
            // Fallback auf globale Einstellungen
            return $this->getGlobalWhatsAppSettings();
        }
        
        // Prüfe welche WhatsApp-Spalten existieren
        $stmt = $this->conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp%'");
        $whatsappColumns = $stmt->fetchAll(PDO::FETCH_COLUMN);
        $hasChannelId = in_array('whatsapp_channel_id', $whatsappColumns);
        
        $selectFields = "a.*, 
                   a.whatsapp_notifications_enabled,
                   a.whatsapp_api_key,
                   a.whatsapp_phone_number,
                   a.whatsapp_business_name,
                   a.whatsapp_template_id,
                   a.whatsapp_test_mode";
        
        // Prüfe ob reminder_template_id Spalte existiert
        $stmt = $this->conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_reminder_template_id'");
        $hasReminderTemplateId = $stmt->rowCount() > 0;
        
        if ($hasChannelId) {
            $selectFields .= ", a.whatsapp_channel_id";
        }
        
        if ($hasReminderTemplateId) {
            $selectFields .= ", a.whatsapp_reminder_template_id";
        }
        
        $stmt = $this->conn->prepare("
            SELECT {$selectFields}
            FROM autohaus a 
            WHERE a.id = ?
        ");
        $stmt->execute([$this->autohausId]);
        $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$autohaus) {
            return $this->getGlobalWhatsAppSettings();
        }
        
        // Verwende Autohaus-spezifische Einstellungen oder Fallback auf globale
        $globalSettings = $this->getGlobalWhatsAppSettings();
        
        $channelId = null;
        if ($hasChannelId && isset($autohaus['whatsapp_channel_id']) && !empty($autohaus['whatsapp_channel_id'])) {
            $channelId = $autohaus['whatsapp_channel_id'];
        } elseif (isset($globalSettings['channel_id']) && !empty($globalSettings['channel_id'])) {
            $channelId = $globalSettings['channel_id'];
        }
        
        return [
            'id' => $autohaus['id'],
            'name' => $autohaus['name'],
            'short_name' => $autohaus['short_name'],
            'notifications_enabled' => $autohaus['whatsapp_notifications_enabled'] ?? $globalSettings['notifications_enabled'],
            'api_key' => $autohaus['whatsapp_api_key'] ?? $globalSettings['api_key'],
            'phone_number' => $autohaus['whatsapp_phone_number'] ?? $globalSettings['phone_number'],
            'business_name' => $autohaus['whatsapp_business_name'] ?? $autohaus['name'],
            'template_id' => $autohaus['whatsapp_template_id'] ?? $globalSettings['template_id'],
            'reminder_template_id' => ($hasReminderTemplateId && isset($autohaus['whatsapp_reminder_template_id'])) 
                ? $autohaus['whatsapp_reminder_template_id'] 
                : null,
            'test_mode' => $autohaus['whatsapp_test_mode'] ?? $globalSettings['test_mode'],
            'channel_id' => $channelId
        ];
    }
    
    /**
     * Holt Template-Konfiguration für eine Aktion
     */
    private function getTemplateConfig($actionType) {
        if (!$this->autohausId) {
            return null;
        }
        
        try {
            // Prüfe ob is_enabled Spalte existiert
            $stmt = $this->conn->query("SHOW COLUMNS FROM whatsapp_template_configs LIKE 'is_enabled'");
            $hasIsEnabled = $stmt->rowCount() > 0;
            
            if ($hasIsEnabled) {
                $stmt = $this->conn->prepare("
                    SELECT * FROM whatsapp_template_configs 
                    WHERE autohaus_id = ? AND action_type = ? AND is_active = 1 AND is_enabled = 1
                    LIMIT 1
                ");
            } else {
                $stmt = $this->conn->prepare("
                    SELECT * FROM whatsapp_template_configs 
                    WHERE autohaus_id = ? AND action_type = ? AND is_active = 1
                    LIMIT 1
                ");
            }
            $stmt->execute([$this->autohausId, $actionType]);
            $config = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($config && !empty($config['custom_fields_mapping'])) {
                // Parse JSON falls String
                if (is_string($config['custom_fields_mapping'])) {
                    $config['custom_fields_mapping'] = json_decode($config['custom_fields_mapping'], true) ?? [];
                }
            }
            
            return $config ?: null;
        } catch (Exception $e) {
            error_log("WhatsApp (Autohaus) - Fehler beim Laden der Template-Konfiguration: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Holt globale WhatsApp-Einstellungen als Fallback
     */
    private function getGlobalWhatsAppSettings() {
        $stmt = $this->conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key LIKE 'whatsapp_%'");
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        return [
            'id' => null,
            'name' => 'Global',
            'short_name' => 'Global',
            'notifications_enabled' => $results['whatsapp_notifications_enabled'] ?? '0',
            'api_key' => $results['whatsapp_api_key'] ?? '',
            'phone_number' => $results['whatsapp_phone_number'] ?? '',
            'business_name' => $results['whatsapp_business_name'] ?? 'Boxenstop',
            'template_id' => $results['whatsapp_template_id'] ?? '',
            'test_mode' => $results['whatsapp_test_mode'] ?? '1'
        ];
    }
    
    /**
     * Holt Termindetails aus der Datenbank (inkl. user_id für Profil-Zugriff)
     */
    private function getAppointmentDetails($appointmentId) {
        $stmt = $this->conn->prepare("
            SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                   ah.name as autohaus_name, ah.short_name as autohaus_short_name,
                   u.phone_number as user_phone_number, u.whatsapp_opt_out
            FROM appointments a 
            LEFT JOIN events e ON a.event_id = e.id 
            LEFT JOIN autohaus ah ON a.autohaus_id = ah.id
            LEFT JOIN users u ON a.user_id = u.id
            WHERE a.id = ?
        ");
        $stmt->execute([$appointmentId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Prüft ob WhatsApp-Benachrichtigungen aktiviert sind
     */
    private function isWhatsAppNotificationsEnabled() {
        return $this->autohausSettings['notifications_enabled'] == '1' || $this->autohausSettings['notifications_enabled'] == 1;
    }
    
    /**
     * Erstellt WhatsApp-Nachricht basierend auf Terminaktion
     */
    private function createWhatsAppMessage($appointment, $action) {
        $eventDate = date('d.m.Y', strtotime($appointment['event_date']));
        $eventTime = date('H:i', strtotime($appointment['slot_time']));
        $autohausName = $appointment['autohaus_name'] ?? $this->autohausSettings['name'];
        
        switch ($action) {
            case 'approved':
                return "✅ *Termin bestätigt!*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Ihr Termin wurde erfolgreich bestätigt:\n\n" .
                       "🏢 *Autohaus:* " . $autohausName . "\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Datum:* " . $eventDate . "\n" .
                       "🕐 *Zeit:* " . $eventTime . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n" .
                       "🔢 *Einlagerungsnummer:* " . $appointment['storage_number'] . "\n\n" .
                       "Wir freuen uns auf Ihren Besuch!\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $autohausName . " Team";
                       
            case 'rejected':
                return "❌ *Termin abgelehnt*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Leider können wir Ihren Termin nicht bestätigen:\n\n" .
                       "🏢 *Autohaus:* " . $autohausName . "\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Datum:* " . $eventDate . "\n" .
                       "🕐 *Zeit:* " . $eventTime . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n\n" .
                       "Bitte kontaktieren Sie uns für alternative Termine.\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $autohausName . " Team";
                       
            case 'rescheduled':
                return "📅 *Termin verschoben*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Ihr Termin wurde verschoben:\n\n" .
                       "🏢 *Autohaus:* " . $autohausName . "\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Neues Datum:* " . date('d.m.Y', strtotime($appointment['slot_date'])) . "\n" .
                       "🕐 *Neue Zeit:* " . date('H:i', strtotime($appointment['slot_time'])) . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n\n" .
                       "Grund: " . ($appointment['reschedule_reason'] ?? 'Termin verschoben') . "\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $autohausName . " Team";
                       
            case 'reminder':
                $slotDate = date('d.m.Y', strtotime($appointment['slot_date']));
                $slotTime = date('H:i', strtotime($appointment['slot_time']));
                return "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "wir erinnern Sie gerne an Ihren Termin bei " . $autohausName . ":\n\n" .
                       "📅 Morgen, " . $slotDate . " um " . $slotTime . " Uhr\n" .
                       "🚗 Kennzeichen: " . $appointment['license_plate'] . "\n" .
                       ($appointment['storage_number'] ? "🔢 Einlagerungsnummer: " . $appointment['storage_number'] . "\n" : "") .
                       "\nBitte bringen Sie folgendes mit:\n" .
                       ($appointment['brings_wheels'] ? "• Reifen (falls vorhanden)\n" : "") .
                       ($appointment['has_wheel_lock'] ? "• Radmutternschlüssel\n" : "") .
                       ($appointment['wiper_blades'] ? "• Scheibenwischerblätter (falls gewünscht)\n" : "") .
                       "\nFalls Sie den Termin verschieben möchten, kontaktieren Sie uns bitte.\n\n" .
                       "Wir freuen uns auf Ihren Besuch!\n\n" .
                       "Ihr Team von " . $autohausName;
                       
            case 'confirmation':
                $slotDate = date('d.m.Y', strtotime($appointment['slot_date']));
                $slotTime = date('H:i', strtotime($appointment['slot_time']));
                return "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "vielen Dank für Ihre Terminbuchung bei " . $autohausName . "!\n\n" .
                       "📅 Termin: " . $slotDate . " um " . $slotTime . " Uhr\n" .
                       "🚗 Kennzeichen: " . $appointment['license_plate'] . "\n" .
                       ($appointment['storage_number'] ? "🔢 Einlagerungsnummer: " . $appointment['storage_number'] . "\n" : "") .
                       "\nWir freuen uns auf Ihren Besuch!\n\n" .
                       "Bei Fragen stehen wir Ihnen gerne zur Verfügung.\n\n" .
                       "Ihr Team von " . $autohausName;
                       
            default:
                return "📱 *Termin-Update*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Update zu Ihrem Termin:\n\n" .
                       "🏢 *Autohaus:* " . $autohausName . "\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Datum:* " . $eventDate . "\n" .
                       "🕐 *Zeit:* " . $eventTime . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n\n" .
                       "Status: " . ucfirst($action) . "\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $autohausName . " Team";
        }
    }
    
    /**
     * Sendet WhatsApp-Nachricht über HelloMateo API
     * Verwendet die HelloMateoAPI Wrapper-Klasse für korrekte API-Aufrufe
     * 
     * @param string $phoneNumber Telefonnummer
     * @param string|null $message Text-Nachricht (wird ignoriert wenn Template verwendet wird)
     * @param string|null $templateId Template-ID (optional)
     * @param string|null $channelId Channel-ID (optional, wird automatisch geholt wenn nicht vorhanden)
     * @param array|null $placeholderValues Platzhalter-Werte für Template (optional)
     */
    private function sendWhatsAppMessage($phoneNumber, $message = null, $templateId = null, $channelId = null, $placeholderValues = null, $appointmentId = null, $action = null) {
        try {
            $settings = $this->autohausSettings;
            
            // Test-Modus oder kein API Key
            if ($settings['test_mode'] || empty($settings['api_key'])) {
                return $this->sendWhatsAppTest($phoneNumber, $message, $settings);
            }
            
            // Verwende HelloMateoAPI Wrapper-Klasse
            // Diese Klasse lädt automatisch den korrekten API-Key pro Autohaus
            $hellomateo = new HelloMateoAPI(null, $this->autohausId);
            
            // Hole Channel-ID aus Einstellungen (falls vorhanden)
            $channelId = $settings['channel_id'] ?? null;
            
            // Falls keine Channel-ID in Einstellungen, hole den ersten verfügbaren Channel
            if (empty($channelId)) {
                $channelsResult = $hellomateo->listChannels();
                if ($channelsResult['success'] && !empty($channelsResult['data'])) {
                    $channels = $channelsResult['data'];
                    // Wenn es ein Array von Channels ist, nimm den ersten
                    if (is_array($channels) && isset($channels[0])) {
                        $firstChannel = is_array($channels[0]) ? $channels[0] : $channels;
                        $channelId = $firstChannel['id'] ?? $firstChannel['handle'] ?? null;
                        error_log("WhatsApp (Autohaus) - Channel-ID automatisch geholt: " . $channelId);
                    } elseif (is_array($channels) && isset($channels['id'])) {
                        // Direktes Channel-Objekt
                        $channelId = $channels['id'] ?? $channels['handle'] ?? null;
                        error_log("WhatsApp (Autohaus) - Channel-ID automatisch geholt: " . $channelId);
                    }
                }
            }
            
            // Template-ID wird als Parameter übergeben (falls verwendet)
            // Wenn Template verwendet wird, wird $message ignoriert
            
            // Sende Nachricht über HelloMateo API
            // Die Wrapper-Klasse formatiert die Nachricht korrekt
            // placeholderValues wird direkt übergeben (kann Array oder Objekt sein)
            $result = $hellomateo->sendMessage($phoneNumber, $message, $templateId, $channelId, $placeholderValues);
            
            // Log in Datenbank
            $this->logWhatsAppMessage($appointmentId, $phoneNumber, $action ?? 'unknown', $result);
            
            if ($result['success']) {
                error_log("WhatsApp (Autohaus) - Nachricht erfolgreich gesendet an: " . $phoneNumber);
                return ['success' => true, 'message' => 'WhatsApp-Nachricht erfolgreich gesendet an ' . $phoneNumber];
            } else {
                $errorMsg = $result['error'] ?? 'Unbekannter Fehler';
                error_log("WhatsApp (Autohaus) - Nachricht-Fehler für: " . $phoneNumber . " - " . $errorMsg);
                return ['success' => false, 'error' => 'WhatsApp API Fehler: ' . $errorMsg];
            }
            
        } catch (Exception $e) {
            $errorMsg = 'WhatsApp-Fehler: ' . $e->getMessage();
            error_log("WhatsApp (Autohaus) - Nachricht-Fehler für " . $phoneNumber . ": " . $errorMsg);
            return ['success' => false, 'error' => $errorMsg];
        }
    }
    
    /**
     * Test-Modus für WhatsApp (ohne API)
     */
    private function sendWhatsAppTest($phoneNumber, $message, $settings = []) {
        try {
            // Simuliere WhatsApp-Versendung
            error_log("WhatsApp Test (Autohaus) - Nachricht an " . $phoneNumber . ": " . $message);
            error_log("WhatsApp Test (Autohaus) - Einstellungen: " . json_encode($settings));
            
            // Hier würde normalerweise die HelloMateo API aufgerufen werden
            // Für den Test simulieren wir eine erfolgreiche Versendung
            
            $mode = $settings['test_mode'] ? 'Test-Modus' : 'Kein API-Key';
            return ['success' => true, 'message' => 'WhatsApp-Nachricht erfolgreich gesendet an ' . $phoneNumber . ' (' . $mode . ')'];
            
        } catch (Exception $e) {
            error_log("WhatsApp Test (Autohaus) - Fehler für " . $phoneNumber . ": " . $e->getMessage());
            return ['success' => false, 'error' => 'WhatsApp Test-Fehler: ' . $e->getMessage()];
        }
    }
    
    /**
     * Sendet Terminerinnerungen für alle anstehenden Termine
     */
    public function sendAppointmentReminders($hoursBefore = 24) {
        try {
            $reminderTime = date('Y-m-d H:i:s', strtotime("+{$hoursBefore} hours"));
            
            $stmt = $this->conn->prepare("
                SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                       ah.name as autohaus_name, ah.short_name as autohaus_short_name
                FROM appointments a 
                LEFT JOIN events e ON a.event_id = e.id 
                LEFT JOIN autohaus ah ON a.autohaus_id = ah.id
                WHERE a.status = 'approved' 
                AND a.slot_date = CURDATE()
                AND a.slot_time BETWEEN TIME_SUB(CURTIME(), INTERVAL 1 HOUR) AND TIME_ADD(CURTIME(), INTERVAL 1 HOUR)
                AND a.autohaus_id = ?
            ");
            $stmt->execute([$this->autohausId]);
            $appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $results = [];
            foreach ($appointments as $appointment) {
                $result = $this->sendAppointmentNotification($appointment['id'], 'reminder');
                $results[] = [
                    'appointment_id' => $appointment['id'],
                    'customer_name' => $appointment['customer_name'],
                    'result' => $result
                ];
            }
            
            return ['success' => true, 'reminders_sent' => count($results), 'results' => $results];
            
        } catch (Exception $e) {
            error_log("WhatsApp (Autohaus) - Erinnerungen-Fehler: " . $e->getMessage());
            return ['success' => false, 'error' => 'Erinnerungen konnten nicht gesendet werden: ' . $e->getMessage()];
        }
    }
    
    /**
     * Loggt WhatsApp-Versand in Datenbank
     */
    private function logWhatsAppMessage($appointmentId, $phoneNumber, $actionType, $result) {
        try {
            // Hole Kundenname aus Appointment falls vorhanden
            $customerName = null;
            if ($appointmentId) {
                $stmt = $this->conn->prepare("SELECT customer_name FROM appointments WHERE id = ?");
                $stmt->execute([$appointmentId]);
                $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
                $customerName = $appointment['customer_name'] ?? null;
            }
            
            // Prüfe ob Tabelle existiert
            $checkTable = $this->conn->query("SHOW TABLES LIKE 'whatsapp_logs'");
            if ($checkTable->rowCount() === 0) {
                // Tabelle existiert nicht, erstelle sie
                $this->conn->exec("
                    CREATE TABLE whatsapp_logs (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        appointment_id INT,
                        autohaus_id INT,
                        phone_number VARCHAR(20),
                        customer_name VARCHAR(255),
                        action_type VARCHAR(50),
                        status VARCHAR(20),
                        message TEXT,
                        error_message TEXT,
                        created_at DATETIME,
                        updated_at DATETIME,
                        INDEX idx_appointment (appointment_id),
                        INDEX idx_autohaus (autohaus_id),
                        INDEX idx_status (status),
                        INDEX idx_created (created_at)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
                ");
            } else {
                // Tabelle existiert - füge fehlende Spalten hinzu
                $columns = $this->conn->query("SHOW COLUMNS FROM whatsapp_logs")->fetchAll(PDO::FETCH_COLUMN);
                
                if (!in_array('customer_name', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN customer_name VARCHAR(255) AFTER phone_number");
                }
                if (!in_array('action_type', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN action_type VARCHAR(50) AFTER customer_name");
                }
                if (!in_array('status', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN status VARCHAR(20) AFTER action_type");
                }
                if (!in_array('message', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN message TEXT AFTER status");
                }
                if (!in_array('error_message', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN error_message TEXT AFTER message");
                }
                if (!in_array('updated_at', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN updated_at DATETIME AFTER created_at");
                }
            }
            
            $status = $result['success'] ? 'success' : 'error';
            $message = $result['message'] ?? ($result['success'] ? 'Nachricht gesendet' : null);
            $errorMessage = $result['success'] ? null : ($result['error'] ?? 'Unbekannter Fehler');
            
            $logStmt = $this->conn->prepare("
                INSERT INTO whatsapp_logs (
                    appointment_id, autohaus_id, phone_number, customer_name,
                    action_type, status, message, error_message, created_at, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
            ");
            
            $logStmt->execute([
                $appointmentId,
                $this->autohausId,
                $phoneNumber,
                $customerName,
                $actionType,
                $status,
                $message,
                $errorMessage
            ]);
        } catch (Exception $e) {
            // Fehler beim Loggen ignorieren, um Hauptfunktion nicht zu beeinträchtigen
            error_log("Fehler beim Loggen von WhatsApp-Nachricht: " . $e->getMessage());
        }
    }
    
    /**
     * Stellt sicher, dass die whatsapp_logs Tabelle und alle benötigten Spalten existieren
     * Kann vor Tests oder anderen Operationen aufgerufen werden
     */
    public function ensureWhatsAppLogsTableAndColumnsExist() {
        try {
            // Prüfe ob Tabelle existiert
            $checkTable = $this->conn->query("SHOW TABLES LIKE 'whatsapp_logs'");
            if ($checkTable->rowCount() === 0) {
                // Tabelle existiert nicht, erstelle sie
                $this->conn->exec("
                    CREATE TABLE whatsapp_logs (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        appointment_id INT,
                        autohaus_id INT,
                        phone_number VARCHAR(20),
                        customer_name VARCHAR(255),
                        action_type VARCHAR(50),
                        status VARCHAR(20),
                        message TEXT,
                        error_message TEXT,
                        created_at DATETIME,
                        updated_at DATETIME,
                        INDEX idx_appointment (appointment_id),
                        INDEX idx_autohaus (autohaus_id),
                        INDEX idx_status (status),
                        INDEX idx_created (created_at)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
                ");
            } else {
                // Tabelle existiert - füge fehlende Spalten hinzu
                $columns = $this->conn->query("SHOW COLUMNS FROM whatsapp_logs")->fetchAll(PDO::FETCH_COLUMN);
                
                if (!in_array('customer_name', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN customer_name VARCHAR(255) AFTER phone_number");
                }
                if (!in_array('action_type', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN action_type VARCHAR(50) AFTER customer_name");
                }
                if (!in_array('status', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN status VARCHAR(20) AFTER action_type");
                }
                if (!in_array('message', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN message TEXT AFTER status");
                }
                if (!in_array('error_message', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN error_message TEXT AFTER message");
                }
                if (!in_array('updated_at', $columns)) {
                    $this->conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN updated_at DATETIME AFTER created_at");
                }
            }
            
            return true;
        } catch (Exception $e) {
            error_log("Fehler beim Erstellen/Prüfen der whatsapp_logs Tabelle: " . $e->getMessage());
            return false;
        }
    }
}
?>
