<?php
/**
 * WhatsApp-Erinnerungen Verwaltung
 * Moderne, visuelle Template-Zuweisung
 */

session_start();

ini_set('display_errors', 1);
ini_set('log_errors', 1);
error_reporting(E_ALL);

try {
    require_once '../config/database.php';
    require_once 'includes/helpers.php';
    require_once 'includes/hellomateo_api.php';
    require_once '../api/autohaus_whatsapp_notifications.php';
    require_once 'includes/template_config_helper.php';
} catch (Exception $e) {
    die('Fehler beim Laden der Abhängigkeiten: ' . $e->getMessage());
}

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$can_switch_autohaus = false;

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.role, u.autohaus_id, u.can_switch_autohaus, a.name as autohaus_name, u.username
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($user_data) {
        $current_user_role = $user_data['role'] ?? 'admin';
        $current_user_autohaus_id = !empty($user_data['autohaus_id']) ? intval($user_data['autohaus_id']) : null;
        $can_switch_autohaus = ($user_data['can_switch_autohaus'] == 1) || ($current_user_role === 'admin');
    }
}

// Autohaus-Auswahl
$selected_autohaus_id = null;
if ($can_switch_autohaus || $current_user_role === 'admin') {
    if (isset($_GET['autohaus_id']) && !empty($_GET['autohaus_id'])) {
        $_SESSION['selected_autohaus_id'] = intval($_GET['autohaus_id']);
        $selected_autohaus_id = $_SESSION['selected_autohaus_id'];
    } elseif (isset($_SESSION['selected_autohaus_id'])) {
        $selected_autohaus_id = $_SESSION['selected_autohaus_id'];
    }
}

$active_autohaus_id = $selected_autohaus_id ?? $current_user_autohaus_id;

// Lade alle Autohäuser
$all_autohauser = [];
if ($can_switch_autohaus || $current_user_role === 'admin') {
    $stmt = $conn->prepare("SELECT id, name, short_name FROM autohaus WHERE is_active = 1 ORDER BY name ASC");
    $stmt->execute();
    $all_autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
    header('Content-Type: application/json');
    ob_clean();
    
    $action = $_POST['ajax_action'];
    $response = ['success' => false, 'error' => 'Unbekannte Aktion'];
    
    try {
        switch ($action) {
            case 'save_template_assignment':
                $autohausId = intval($_POST['autohaus_id'] ?? 0);
                $actionType = $_POST['action_type'] ?? '';
                $templateId = trim($_POST['template_id'] ?? '');
                $timingHours = intval($_POST['timing_hours'] ?? 0);
                $isEnabled = isset($_POST['is_enabled']) ? intval($_POST['is_enabled']) : 1;
                $customFieldsMapping = json_decode($_POST['custom_fields_mapping'] ?? '{}', true);
                
                if (!$autohausId || !$actionType || !$templateId) {
                    $response = ['success' => false, 'error' => 'Autohaus-ID, Aktion und Template-ID sind erforderlich'];
                    break;
                }
                
                // Prüfe ob Konfiguration existiert
                $stmt = $conn->prepare("SELECT id FROM whatsapp_template_configs WHERE autohaus_id = ? AND action_type = ?");
                $stmt->execute([$autohausId, $actionType]);
                $existing = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Prüfe ob is_enabled Spalte existiert
                $stmt = $conn->query("SHOW COLUMNS FROM whatsapp_template_configs LIKE 'is_enabled'");
                $hasIsEnabled = $stmt->rowCount() > 0;
                
                if ($existing) {
                    // Update
                    if ($hasIsEnabled) {
                        $stmt = $conn->prepare("
                            UPDATE whatsapp_template_configs 
                            SET template_id = ?, send_timing_hours = ?,
                                custom_fields_mapping = ?, is_active = 1, is_enabled = ?, updated_at = NOW()
                            WHERE id = ?
                        ");
                        $stmt->execute([
                            $templateId,
                            $timingHours ?? null,
                            json_encode($customFieldsMapping),
                            $isEnabled,
                            $existing['id']
                        ]);
                    } else {
                        $stmt = $conn->prepare("
                            UPDATE whatsapp_template_configs 
                            SET template_id = ?, send_timing_hours = ?,
                                custom_fields_mapping = ?, is_active = 1, updated_at = NOW()
                            WHERE id = ?
                        ");
                        $stmt->execute([
                            $templateId,
                            $timingHours ?? null,
                            json_encode($customFieldsMapping),
                            $existing['id']
                        ]);
                    }
                } else {
                    // Insert
                    if ($hasIsEnabled) {
                        $stmt = $conn->prepare("
                            INSERT INTO whatsapp_template_configs 
                            (autohaus_id, action_type, template_id, send_timing_hours, 
                             custom_fields_mapping, is_active, is_enabled, created_at, updated_at)
                            VALUES (?, ?, ?, ?, ?, 1, ?, NOW(), NOW())
                        ");
                        $stmt->execute([
                            $autohausId,
                            $actionType,
                            $templateId,
                            $timingHours ?? null,
                            json_encode($customFieldsMapping),
                            $isEnabled
                        ]);
                    } else {
                        $stmt = $conn->prepare("
                            INSERT INTO whatsapp_template_configs 
                            (autohaus_id, action_type, template_id, send_timing_hours, 
                             custom_fields_mapping, is_active, created_at, updated_at)
                            VALUES (?, ?, ?, ?, ?, 1, NOW(), NOW())
                        ");
                        $stmt->execute([
                            $autohausId,
                            $actionType,
                            $templateId,
                            $timingHours ?? null,
                            json_encode($customFieldsMapping)
                        ]);
                    }
                }
                
                $response = ['success' => true, 'message' => 'Template-Zuweisung gespeichert'];
                break;
                
            case 'toggle_enabled':
                $configId = intval($_POST['config_id'] ?? 0);
                $isEnabled = isset($_POST['is_enabled']) ? intval($_POST['is_enabled']) : 0;
                
                if ($configId) {
                    // Prüfe ob is_enabled Spalte existiert
                    $stmt = $conn->query("SHOW COLUMNS FROM whatsapp_template_configs LIKE 'is_enabled'");
                    $hasIsEnabled = $stmt->rowCount() > 0;
                    
                    if ($hasIsEnabled) {
                        $stmt = $conn->prepare("UPDATE whatsapp_template_configs SET is_enabled = ? WHERE id = ?");
                        $stmt->execute([$isEnabled, $configId]);
                    } else {
                        // Fallback: Verwende is_active
                        $stmt = $conn->prepare("UPDATE whatsapp_template_configs SET is_active = ? WHERE id = ?");
                        $stmt->execute([$isEnabled, $configId]);
                    }
                    $response = ['success' => true, 'message' => 'Status aktualisiert'];
                }
                break;
                
            case 'test_send':
                $configId = intval($_POST['config_id'] ?? 0);
                $testPhone = trim($_POST['test_phone'] ?? '');
                $testAppointmentId = intval($_POST['test_appointment_id'] ?? 0);
                
                if (!$configId) {
                    $response = ['success' => false, 'error' => 'Konfigurations-ID fehlt'];
                    break;
                }
                
                // Hole Konfiguration
                $stmt = $conn->prepare("SELECT * FROM whatsapp_template_configs WHERE id = ?");
                $stmt->execute([$configId]);
                $config = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$config) {
                    $response = ['success' => false, 'error' => 'Konfiguration nicht gefunden'];
                    break;
                }
                
                // Prüfe ob Test-Telefonnummer oder Test-Termin-ID vorhanden
                if (empty($testPhone) && !$testAppointmentId) {
                    $response = ['success' => false, 'error' => 'Bitte geben Sie eine Test-Telefonnummer oder wählen Sie einen Test-Termin'];
                    break;
                }
                
                // Lade Autohaus-ID
                $autohausId = $config['autohaus_id'];
                
                // Erstelle Test-Termin falls nur Telefonnummer vorhanden
                if ($testAppointmentId) {
                    $appointmentId = $testAppointmentId;
                } else {
                    // Erstelle temporären Test-Termin
                    $stmt = $conn->prepare("
                        INSERT INTO appointments 
                        (autohaus_id, customer_name, phone, email, license_plate, slot_date, slot_time, status, source)
                        VALUES (?, 'Test Kunde', ?, 'test@example.com', 'TEST-123', CURDATE(), '10:00:00', 'booked', 'admin_test')
                    ");
                    $stmt->execute([$autohausId, $testPhone]);
                    $appointmentId = $conn->lastInsertId();
                }
                
                // Sende Test-Nachricht
                require_once '../api/autohaus_whatsapp_notifications.php';
                $whatsapp = new AutohausWhatsAppNotifications($autohausId);
                $result = $whatsapp->sendAppointmentNotification($appointmentId, $config['action_type']);
                
                // Lösche temporären Test-Termin falls erstellt
                if (!$testAppointmentId) {
                    $stmt = $conn->prepare("DELETE FROM appointments WHERE id = ? AND source = 'admin_test'");
                    $stmt->execute([$appointmentId]);
                }
                
                if ($result['success']) {
                    $response = ['success' => true, 'message' => 'Test-Nachricht erfolgreich gesendet', 'details' => $result];
                } else {
                    $response = ['success' => false, 'error' => $result['error'] ?? 'Fehler beim Senden', 'details' => $result];
                }
                break;
                
            case 'delete_template_assignment':
                $configId = intval($_POST['config_id'] ?? 0);
                if ($configId) {
                    $stmt = $conn->prepare("DELETE FROM whatsapp_template_configs WHERE id = ?");
                    $stmt->execute([$configId]);
                    $response = ['success' => true, 'message' => 'Zuweisung gelöscht'];
                }
                break;
        }
    } catch (Exception $e) {
        $response = ['success' => false, 'error' => $e->getMessage()];
    }
    
    echo json_encode($response);
    exit;
}

// Lade Template-Konfigurationen
$template_configs = [];
if ($active_autohaus_id) {
    $stmt = $conn->prepare("
        SELECT * FROM whatsapp_template_configs 
        WHERE autohaus_id = ? AND is_active = 1
    ");
    $stmt->execute([$active_autohaus_id]);
    $template_configs = $stmt->fetchAll(PDO::FETCH_ASSOC);
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WhatsApp Template-Verwaltung - Boxenstop</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/unified-sidebar.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --whatsapp-green: #25D366;
            --primary: #0056b3;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --bg-light: #f8f9fa;
            --border: #dee2e6;
            --text-muted: #6c757d;
        }
        
        .whatsapp-admin-container {
            padding: 2rem;
            max-width: 1800px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
            color: white;
            padding: 2rem;
            border-radius: 16px;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 211, 102, 0.3);
        }
        
        .header-section h1 {
            margin: 0 0 0.5rem 0;
            font-size: 2rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .autohaus-selector {
            background: rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            border-radius: 12px;
            padding: 1rem;
            margin-top: 1rem;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .autohaus-selector select {
            flex: 1;
            padding: 0.75rem 1rem;
            border: none;
            border-radius: 8px;
            background: white;
            font-size: 1rem;
            font-weight: 500;
            cursor: pointer;
        }
        
        .visual-assignment-area {
            display: grid;
            grid-template-columns: 1fr 2fr;
            gap: 2rem;
            margin-top: 2rem;
        }
        
        .actions-panel, .templates-panel {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            box-shadow: 0 2px 12px rgba(0,0,0,0.1);
        }
        
        .panel-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--bg-light);
        }
        
        .panel-header h2 {
            margin: 0;
            font-size: 1.5rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .action-card {
            background: var(--bg-light);
            border: 2px solid var(--border);
            border-radius: 12px;
            padding: 1.25rem;
            margin-bottom: 1rem;
            cursor: move;
            transition: all 0.3s;
            position: relative;
        }
        
        .action-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .action-card.connected {
            border-color: var(--whatsapp-green);
            background: linear-gradient(135deg, rgba(37, 211, 102, 0.1) 0%, rgba(18, 140, 126, 0.1) 100%);
        }
        
        .action-card-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 0.5rem;
        }
        
        .action-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: white;
        }
        
        .action-icon.reminder { background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%); }
        .action-icon.confirmation { background: linear-gradient(135deg, #28a745 0%, #20c997 100%); }
        .action-icon.approved { background: linear-gradient(135deg, #17a2b8 0%, #138496 100%); }
        .action-icon.rejected { background: linear-gradient(135deg, #dc3545 0%, #c82333 100%); }
        .action-icon.cancellation { background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%); }
        .action-icon.rescheduled { background: linear-gradient(135deg, #6f42c1 0%, #5a32a3 100%); }
        .action-icon.review { background: linear-gradient(135deg, #ffd700 0%, #ffa500 100%); }
        
        .action-title {
            font-weight: 600;
            font-size: 1.1rem;
            color: #212529;
        }
        
        .action-description {
            font-size: 0.9rem;
            color: var(--text-muted);
            margin-top: 0.5rem;
        }
        
        .connected-template {
            margin-top: 0.75rem;
            padding: 0.75rem;
            background: white;
            border-radius: 8px;
            border-left: 4px solid var(--whatsapp-green);
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .template-name {
            font-weight: 500;
            color: #212529;
        }
        
        .template-status {
            font-size: 0.8rem;
            padding: 0.4rem 0.9rem;
            border-radius: 20px;
            font-weight: 600;
            display: inline-flex;
            align-items: center;
            gap: 0.4rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.3);
            transition: all 0.2s ease;
        }
        
        .template-status:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.15);
        }
        
        .status-approved { 
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: #ffffff;
            border-color: #047857;
        }
        
        .status-approved::before {
            content: '✓';
            font-weight: bold;
            font-size: 0.9rem;
        }
        
        .status-pending { 
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
            color: #ffffff;
            border-color: #b45309;
        }
        
        .status-pending::before {
            content: '⏱';
            font-size: 0.9rem;
        }
        
        .status-rejected { 
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            color: #ffffff;
            border-color: #b91c1c;
        }
        
        .status-rejected::before {
            content: '✕';
            font-weight: bold;
            font-size: 0.9rem;
        }
        
        .status-unknown {
            background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
            color: #ffffff;
            border-color: #374151;
        }
        
        .status-unknown::before {
            content: '?';
            font-weight: bold;
            font-size: 0.9rem;
        }
        
        .templates-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 1rem;
            height: calc(100vh - 280px); /* Nutze komplette verfügbare Höhe */
            overflow-y: auto;
            padding: 0.5rem;
            padding-right: 0.5rem;
        }
        
        /* Scrollbar Styling */
        .templates-grid::-webkit-scrollbar {
            width: 8px;
        }
        
        .templates-grid::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 4px;
        }
        
        .templates-grid::-webkit-scrollbar-thumb {
            background: #888;
            border-radius: 4px;
        }
        
        .templates-grid::-webkit-scrollbar-thumb:hover {
            background: #555;
        }
        
        .template-card {
            background: white;
            border: 2px solid var(--border);
            border-radius: 12px;
            padding: 1rem;
            cursor: pointer;
            transition: all 0.3s;
            position: relative;
        }
        
        .template-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .template-card.selected {
            border-color: var(--whatsapp-green);
            background: linear-gradient(135deg, rgba(37, 211, 102, 0.1) 0%, rgba(18, 140, 126, 0.1) 100%);
        }
        
        .template-card-header {
            display: flex;
            align-items: flex-start;
            justify-content: space-between;
            margin-bottom: 0.75rem;
        }
        
        .template-name-large {
            font-weight: 600;
            font-size: 1rem;
            color: #212529;
            flex: 1;
        }
        
        .template-id {
            font-size: 0.75rem;
            color: var(--text-muted);
            font-family: monospace;
            margin-top: 0.25rem;
        }
        
        .template-preview {
            margin-top: 0.75rem;
        }
        
        /* WhatsApp-ähnliche Vorschau Styles */
        .whatsapp-preview-container {
            background: #e5ddd5;
            background-image: 
                repeating-linear-gradient(45deg, transparent, transparent 10px, rgba(0,0,0,.03) 10px, rgba(0,0,0,.03) 20px);
            padding: 1rem;
            border-radius: 8px;
            margin-top: 0.5rem;
        }
        
        .whatsapp-message-bubble {
            background: #dcf8c6;
            border-radius: 7.5px;
            padding: 0.5rem 0.75rem;
            max-width: 85%;
            margin-left: auto;
            margin-right: 0;
            box-shadow: 0 1px 0.5px rgba(0,0,0,0.13);
            position: relative;
        }
        
        .whatsapp-message-bubble::after {
            content: '';
            position: absolute;
            right: -8px;
            bottom: 0;
            width: 0;
            height: 0;
            border-style: solid;
            border-width: 0 0 13px 13px;
            border-color: transparent transparent #dcf8c6 transparent;
        }
        
        .whatsapp-header-text {
            font-weight: 600;
            font-size: 0.9rem;
            margin-bottom: 0.5rem;
            color: #000;
        }
        
        .whatsapp-header-media {
            margin-bottom: 0.5rem;
        }
        
        .whatsapp-media-placeholder {
            background: rgba(0,0,0,0.1);
            border-radius: 4px;
            padding: 2rem;
            text-align: center;
            color: #666;
        }
        
        .whatsapp-media-placeholder i {
            font-size: 2rem;
            display: block;
            margin-bottom: 0.5rem;
        }
        
        .whatsapp-body {
            color: #000;
            font-size: 0.9rem;
            line-height: 1.4;
            margin-bottom: 0.5rem;
        }
        
        .whatsapp-placeholder {
            background: rgba(0,0,0,0.1);
            padding: 0.1rem 0.3rem;
            border-radius: 3px;
            font-weight: 600;
        }
        
        .whatsapp-footer {
            font-size: 0.75rem;
            color: #666;
            margin-top: 0.5rem;
            padding-top: 0.5rem;
            border-top: 1px solid rgba(0,0,0,0.1);
        }
        
        .whatsapp-buttons {
            margin-top: 0.75rem;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .whatsapp-button-quick,
        .whatsapp-button-url,
        .whatsapp-button-phone {
            background: white;
            border: 1px solid rgba(0,0,0,0.1);
            border-radius: 4px;
            padding: 0.5rem;
            text-align: center;
            font-size: 0.85rem;
            color: #075e54;
            font-weight: 500;
        }
        
        .whatsapp-time {
            text-align: right;
            font-size: 0.65rem;
            color: #666;
            margin-top: 0.25rem;
            padding-right: 0.5rem;
        }
        
        .whatsapp-preview-empty {
            text-align: center;
            color: var(--text-muted);
            font-style: italic;
            padding: 1rem;
        }
        
        .template-preview {
            margin-top: 0.75rem;
            cursor: pointer;
            transition: opacity 0.2s;
        }
        
        .template-preview:hover {
            opacity: 0.8;
        }
        
        /* Modal für vollständige Vorschau */
        .preview-modal {
            display: none;
            position: fixed;
            z-index: 10000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.7);
            backdrop-filter: blur(5px);
        }
        
        .preview-modal.show {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .preview-modal-content {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            max-width: 500px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            position: relative;
        }
        
        .preview-modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--bg-light);
        }
        
        .preview-modal-header h3 {
            margin: 0;
            font-size: 1.5rem;
            color: #212529;
        }
        
        .preview-modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: var(--text-muted);
            padding: 0.5rem;
            line-height: 1;
        }
        
        .preview-modal-close:hover {
            color: #212529;
        }
        
        .preview-modal-body {
            margin-bottom: 1rem;
        }
        
        .preview-modal-footer {
            display: flex;
            gap: 0.5rem;
            justify-content: flex-end;
            margin-top: 1.5rem;
            padding-top: 1rem;
            border-top: 2px solid var(--bg-light);
        }
        
        .whatsapp-preview-container-full {
            background: #e5ddd5;
            background-image: 
                repeating-linear-gradient(45deg, transparent, transparent 10px, rgba(0,0,0,.03) 10px, rgba(0,0,0,.03) 20px);
            padding: 2rem;
            border-radius: 12px;
            margin-top: 1rem;
        }
        
        .whatsapp-message-bubble-full {
            background: #dcf8c6;
            border-radius: 7.5px;
            padding: 1rem 1.25rem;
            max-width: 85%;
            margin-left: auto;
            margin-right: 0;
            box-shadow: 0 1px 0.5px rgba(0,0,0,0.13);
            position: relative;
            font-size: 1rem;
        }
        
        .whatsapp-message-bubble-full::after {
            content: '';
            position: absolute;
            right: -8px;
            bottom: 0;
            width: 0;
            height: 0;
            border-style: solid;
            border-width: 0 0 13px 13px;
            border-color: transparent transparent #dcf8c6 transparent;
        }
        
        .template-select-dropdown {
            width: 100%;
            padding: 0.5rem;
            border: 1px solid var(--border);
            border-radius: 6px;
            font-size: 0.9rem;
            background: white;
            cursor: pointer;
        }
        
        .template-select-dropdown:hover {
            border-color: var(--primary);
        }
        
        /* WhatsApp-ähnliche Vorschau Styles */
        .whatsapp-preview-container {
            background: #e5ddd5;
            background-image: 
                repeating-linear-gradient(45deg, transparent, transparent 10px, rgba(0,0,0,.03) 10px, rgba(0,0,0,.03) 20px);
            padding: 1rem;
            border-radius: 8px;
            margin-top: 0.5rem;
        }
        
        .whatsapp-message-bubble {
            background: #dcf8c6;
            border-radius: 7.5px;
            padding: 0.5rem 0.75rem;
            max-width: 85%;
            margin-left: auto;
            margin-right: 0;
            box-shadow: 0 1px 0.5px rgba(0,0,0,0.13);
            position: relative;
        }
        
        .whatsapp-message-bubble::after {
            content: '';
            position: absolute;
            right: -8px;
            bottom: 0;
            width: 0;
            height: 0;
            border-style: solid;
            border-width: 0 0 13px 13px;
            border-color: transparent transparent #dcf8c6 transparent;
        }
        
        .whatsapp-header-text {
            font-weight: 600;
            font-size: 0.9rem;
            margin-bottom: 0.5rem;
            color: #000;
        }
        
        .whatsapp-header-media {
            margin-bottom: 0.5rem;
        }
        
        .whatsapp-media-placeholder {
            background: rgba(0,0,0,0.1);
            border-radius: 4px;
            padding: 2rem;
            text-align: center;
            color: #666;
        }
        
        .whatsapp-media-placeholder i {
            font-size: 2rem;
            display: block;
            margin-bottom: 0.5rem;
        }
        
        .whatsapp-body {
            color: #000;
            font-size: 0.9rem;
            line-height: 1.4;
            margin-bottom: 0.5rem;
        }
        
        .whatsapp-placeholder {
            background: rgba(0,0,0,0.1);
            padding: 0.1rem 0.3rem;
            border-radius: 3px;
            font-weight: 600;
        }
        
        .whatsapp-footer {
            font-size: 0.75rem;
            color: #666;
            margin-top: 0.5rem;
            padding-top: 0.5rem;
            border-top: 1px solid rgba(0,0,0,0.1);
        }
        
        .whatsapp-buttons {
            margin-top: 0.75rem;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .whatsapp-button-quick,
        .whatsapp-button-url,
        .whatsapp-button-phone {
            background: white;
            border: 1px solid rgba(0,0,0,0.1);
            border-radius: 4px;
            padding: 0.5rem;
            text-align: center;
            font-size: 0.85rem;
            color: #075e54;
            font-weight: 500;
        }
        
        .whatsapp-time {
            text-align: right;
            font-size: 0.65rem;
            color: #666;
            margin-top: 0.25rem;
            padding-right: 0.5rem;
        }
        
        .whatsapp-preview-empty {
            text-align: center;
            color: var(--text-muted);
            font-style: italic;
            padding: 1rem;
        }
        
        .connection-line {
            position: absolute;
            pointer-events: none;
            z-index: 1;
        }
        
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 9999;
        }
        
        .loading-spinner {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 1rem;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-primary:hover {
            background: #004085;
        }
        
        .btn-success {
            background: var(--whatsapp-green);
            color: white;
        }
        
        .btn-danger {
            background: var(--danger);
            color: white;
        }
        
        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
        }
        
        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1rem;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        #alert-container {
            position: fixed;
            top: 2rem;
            right: 2rem;
            z-index: 10000;
            max-width: 400px;
        }
        
        /* Toggle Switch */
        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 50px;
            height: 26px;
            cursor: pointer;
        }
        
        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        
        .toggle-slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: .4s;
            border-radius: 26px;
        }
        
        .toggle-slider:before {
            position: absolute;
            content: "";
            height: 20px;
            width: 20px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: .4s;
            border-radius: 50%;
        }
        
        .toggle-switch input:checked + .toggle-slider {
            background-color: var(--whatsapp-green);
        }
        
        .toggle-switch input:checked + .toggle-slider:before {
            transform: translateX(24px);
        }
        
        /* Test Send Modal */
        .test-send-modal {
            display: none;
            position: fixed;
            z-index: 10001;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.7);
            backdrop-filter: blur(5px);
        }
        
        .test-send-modal.show {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .test-send-modal-content {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            max-width: 500px;
            width: 90%;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
        }
        
        .test-send-modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--bg-light);
        }
        
        .test-send-modal-header h3 {
            margin: 0;
            font-size: 1.5rem;
            color: #212529;
        }
        
        .test-send-modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: var(--text-muted);
            padding: 0.5rem;
            line-height: 1;
        }
        
        .test-send-modal-close:hover {
            color: #212529;
        }
        
        .test-send-form-group {
            margin-bottom: 1.5rem;
        }
        
        .test-send-form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: #212529;
        }
        
        .test-send-form-group input,
        .test-send-form-group select {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 0.9rem;
        }
        
        .test-send-form-group small {
            display: block;
            margin-top: 0.25rem;
            color: var(--text-muted);
            font-size: 0.85rem;
        }
        
        .test-send-modal-footer {
            display: flex;
            gap: 0.5rem;
            justify-content: flex-end;
            margin-top: 1.5rem;
            padding-top: 1rem;
            border-top: 2px solid var(--bg-light);
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>
        <main class="admin-main">
            <div class="whatsapp-admin-container">
                <div class="header-section">
                    <h1>
                        <i class="fab fa-whatsapp"></i>
                        WhatsApp Template-Verwaltung
                    </h1>
                    <p style="margin: 0; opacity: 0.9;">Verwalten Sie Templates und weisen Sie sie Aktionen zu</p>
                    
                    <?php if ($can_switch_autohaus && !empty($all_autohauser)): ?>
                    <div class="autohaus-selector">
                        <label style="font-weight: 500; white-space: nowrap;">Autohaus:</label>
                        <select id="autohaus-select" onchange="switchAutohaus(this.value)">
                            <option value="">-- Autohaus wählen --</option>
                            <?php foreach ($all_autohauser as $ah): ?>
                                <option value="<?php echo $ah['id']; ?>" <?php echo ($active_autohaus_id == $ah['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($ah['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php endif; ?>
                </div>
                
                <div id="alert-container"></div>
                
                <?php if ($active_autohaus_id): ?>
                    <div class="visual-assignment-area">
                        <!-- Aktionen Panel -->
                        <div class="actions-panel">
                            <div class="panel-header">
                                <h2><i class="fas fa-bolt"></i> Aktionen</h2>
                            </div>
                            <div id="actions-container">
                                <!-- Wird per JavaScript gefüllt -->
                            </div>
                        </div>
                        
                        <!-- Templates Panel -->
                        <div class="templates-panel">
                            <div class="panel-header">
                                <h2><i class="fas fa-file-alt"></i> Templates</h2>
                                <button class="btn btn-primary btn-sm" onclick="loadTemplates()">
                                    <i class="fas fa-sync-alt"></i> Neu laden
                                </button>
                            </div>
                            <div id="templates-container" class="templates-grid">
                                <div style="text-align: center; padding: 2rem; color: var(--text-muted);">
                                    <i class="fas fa-spinner fa-spin"></i> Lade Templates...
                                </div>
                            </div>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-triangle"></i>
                        Bitte wählen Sie ein Autohaus aus.
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
    
    <script>
        // Cache-Busting: Version für JavaScript-Updates
        const APP_VERSION = '<?php echo time(); ?>';
        
        let autohausId = <?php echo $active_autohaus_id ?? 'null'; ?>;
        let templateConfigs = <?php echo json_encode($template_configs); ?>;
        let allTemplates = [];
        let selectedAction = null;
        let selectedTemplate = null;
        
        const actionTypes = {
            'confirmation': {
                label: 'Termin-Bestätigung',
                icon: 'check-circle',
                description: 'Wird sofort nach erfolgreicher Buchung gesendet',
                timing: 0,
                color: 'confirmation'
            },
            'reminder_24h': {
                label: 'Erinnerung 24h',
                icon: 'bell',
                description: 'Automatische Erinnerung 24 Stunden vor dem Termin',
                timing: -24,
                color: 'reminder'
            },
            'reminder_48h': {
                label: 'Erinnerung 48h',
                icon: 'bell',
                description: 'Automatische Erinnerung 48 Stunden vor dem Termin',
                timing: -48,
                color: 'reminder'
            },
            'approved': {
                label: 'Termin genehmigt',
                icon: 'thumbs-up',
                description: 'Wird gesendet, wenn ein Termin manuell genehmigt wird',
                timing: 0,
                color: 'approved'
            },
            'rejected': {
                label: 'Termin abgelehnt',
                icon: 'thumbs-down',
                description: 'Wird gesendet, wenn ein Termin abgelehnt wird',
                timing: 0,
                color: 'rejected'
            },
            'cancellation': {
                label: 'Termin storniert',
                icon: 'times-circle',
                description: 'Wird gesendet, wenn ein Termin storniert wird',
                timing: 0,
                color: 'cancellation'
            },
            'rescheduled': {
                label: 'Termin verschoben',
                icon: 'calendar-alt',
                description: 'Wird gesendet, wenn ein Termin verschoben wird',
                timing: 0,
                color: 'rescheduled'
            },
            'review_request': {
                label: 'Bewertungsanfrage',
                icon: 'star',
                description: 'Wird 24 Stunden nach dem Termin gesendet',
                timing: 24,
                color: 'review'
            }
        };
        
        function switchAutohaus(newAutohausId) {
            if (!newAutohausId) {
                showAlert('warning', 'Bitte wählen Sie ein Autohaus aus');
                return;
            }
            
            // Update globale Variable
            autohausId = newAutohausId;
            
            // Reset State
            selectedAction = null;
            selectedTemplate = null;
            allTemplates = [];
            templateConfigs = [];
            
            // Lade Templates automatisch beim Autohaus-Wechsel
            loadTemplates();
            if (!autohausId) return;
            // Zeige Loading
            showLoading();
            window.location.href = '?autohaus_id=' + autohausId;
        }
        
        function showLoading() {
            const overlay = document.createElement('div');
            overlay.className = 'loading-overlay';
            overlay.id = 'loading-overlay';
            overlay.innerHTML = `
                <div class="loading-spinner">
                    <i class="fas fa-spinner fa-spin" style="font-size: 2rem; color: var(--primary);"></i>
                    <div>Lade Templates...</div>
                </div>
            `;
            document.body.appendChild(overlay);
        }
        
        function hideLoading() {
            const overlay = document.getElementById('loading-overlay');
            if (overlay) overlay.remove();
        }
        
        function loadTemplates() {
            if (!autohausId) {
                document.getElementById('templates-container').innerHTML = '<div class="alert alert-error">Bitte wählen Sie ein Autohaus aus</div>';
                return;
            }
            
            const container = document.getElementById('templates-container');
            container.innerHTML = '<div style="text-align: center; padding: 2rem;"><i class="fas fa-spinner fa-spin"></i> Lade Templates...</div>';
            
            // Lade auch aktuelle Konfigurationen neu
            Promise.all([
                fetch(`api/get_template_configs.php?autohaus_id=${autohausId}&v=${APP_VERSION}`).then(r => r.json()),
                fetch(`api/load_hellomateo_templates.php?autohaus_id=${autohausId}&v=${APP_VERSION}`).then(r => r.json())
            ])
            .then(([configData, templateData]) => {
                if (configData.success) {
                    templateConfigs = configData.data || [];
                }
                
                // Detailliertes Logging für Debugging
                console.log('Template Data Response:', templateData);
                
                if (templateData.success && templateData.data) {
                    allTemplates = templateData.data;
                    renderTemplates();
                    renderActions(); // Neu rendern mit aktualisierten Konfigurationen
                    showAlert('success', `${templateData.count || allTemplates.length} Templates geladen`);
                } else {
                    // Detaillierte Fehlermeldung
                    const errorMsg = templateData.error || 'Keine Templates gefunden';
                    console.error('Template Loading Error:', errorMsg, templateData);
                    
                    // Prüfe ob es ein API-Key Problem ist
                    if (errorMsg.includes('API-Key') || errorMsg.includes('Invalid token') || errorMsg.includes('ungültig')) {
                        container.innerHTML = `
                            <div class="alert alert-error">
                                <strong>API-Key Problem:</strong><br>
                                ${errorMsg}<br>
                                <small>Bitte überprüfen Sie den API-Key in den Einstellungen für dieses Autohaus.</small>
                            </div>
                        `;
                    } else {
                        container.innerHTML = `<div class="alert alert-error">${errorMsg}</div>`;
                    }
                    showAlert('error', errorMsg);
                }
            })
                .catch(error => {
                    container.innerHTML = '<div class="alert alert-error">Fehler beim Laden: ' + error.message + '</div>';
                    showAlert('error', 'Fehler beim Laden der Templates: ' + error.message);
                })
                .finally(() => {
                    hideLoading();
                });
        }
        
        function renderTemplates() {
            const container = document.getElementById('templates-container');
            
            if (allTemplates.length === 0) {
                container.innerHTML = '<div class="alert alert-error">Keine Templates verfügbar</div>';
                return;
            }
            
            let html = '';
            allTemplates.forEach(template => {
                const isSelected = selectedTemplate && selectedTemplate.id === template.id;
                // Status normalisieren (case-insensitive)
                const statusUpper = (template.status || '').toUpperCase();
                const statusClass = statusUpper === 'APPROVED' ? 'status-approved' : 
                                   statusUpper === 'PENDING' ? 'status-pending' : 
                                   statusUpper === 'REJECTED' ? 'status-rejected' : 'status-unknown';
                const statusText = statusUpper === 'APPROVED' ? 'Genehmigt' : 
                                  statusUpper === 'PENDING' ? '⏳ Ausstehend' : 
                                  statusUpper === 'REJECTED' ? '❌ Abgelehnt' : '❓ Unbekannt';
                
                html += `
                    <div class="template-card ${isSelected ? 'selected' : ''}" 
                         data-template-id="${template.id}">
                        <div class="template-card-header">
                            <div>
                                <div class="template-name-large">${escapeHtml(template.name)}</div>
                                <div class="template-id">${template.id}</div>
                            </div>
                            <span class="template-status ${statusClass}">${statusText}</span>
                        </div>
                        <div class="template-preview" onclick="event.stopPropagation(); showFullPreview(${JSON.stringify(template).replace(/"/g, '&quot;')})" style="cursor: pointer;" title="Klicken für vollständige Vorschau" data-template-id="${template.id}">
                            ${(() => {
                                try {
                                    return renderWhatsAppPreview(template);
                                } catch (e) {
                                    console.error('Fehler beim Rendern der Vorschau:', e);
                                    return '<div class="whatsapp-preview-empty">Fehler beim Rendern</div>';
                                }
                            })()}
                        </div>
                    </div>
                `;
            });
            
            container.innerHTML = html;
            
            // Lade Details für Templates, die sie benötigen (lazy loading)
            // WICHTIG: Dies passiert asynchron nach dem Rendern
            allTemplates.forEach(async (template) => {
                // Prüfe ob Body fehlt (dann Details nachladen)
                const hasBody = template.components && template.components.some(c => c && c.type === 'BODY');
                if (!hasBody && template.needs_details !== false) {
                    const previewElement = document.querySelector(`[data-template-id="${template.id}"]`);
                    if (previewElement) {
                        // Zeige Loading-Indikator
                        previewElement.innerHTML = '<div class="preview-loading"><i class="fas fa-spinner fa-spin"></i> Lade Vorschau...</div>';
                        
                        try {
                            const fullTemplate = await loadTemplateDetailsIfNeeded(template);
                            // Aktualisiere die Vorschau
                            previewElement.innerHTML = renderWhatsAppPreview(fullTemplate);
                            // Aktualisiere auch im allTemplates Array
                            const index = allTemplates.findIndex(t => t.id === template.id);
                            if (index !== -1) {
                                allTemplates[index] = fullTemplate;
                            }
                        } catch (error) {
                            console.error('Fehler beim Laden der Template-Details:', error);
                            previewElement.innerHTML = '<div class="whatsapp-preview-empty">Fehler beim Laden der Vorschau</div>';
                        }
                    }
                }
            });
        }
        
        function renderActions() {
            const container = document.getElementById('actions-container');
            let html = '';
            
            Object.entries(actionTypes).forEach(([key, action]) => {
                const config = templateConfigs.find(c => c.action_type === key);
                const connectedTemplate = config ? allTemplates.find(t => t.id === config.template_id) : null;
                const isEnabled = config ? (config.is_enabled !== undefined ? config.is_enabled : config.is_active) : false;
                
                html += `
                    <div class="action-card ${config ? 'connected' : ''}" 
                         onclick="selectAction('${key}')"
                         data-action-type="${key}">
                        <div class="action-card-header">
                            <div class="action-icon ${action.color}">
                                <i class="fas fa-${action.icon}"></i>
                            </div>
                            <div style="flex: 1; margin-left: 1rem;">
                                <div class="action-title">${action.label}</div>
                                <div class="action-description">${action.description}</div>
                            </div>
                        </div>
                        ${connectedTemplate ? `
                            <div class="connected-template">
                                <div style="flex: 1;">
                                    <div class="template-name">${escapeHtml(connectedTemplate.name)}</div>
                                    <div class="template-id" style="font-size: 0.75rem; margin-top: 0.25rem;">${connectedTemplate.id}</div>
                                </div>
                                <div style="display: flex; gap: 0.5rem; align-items: center;">
                                    <label class="toggle-switch" onclick="event.stopPropagation();">
                                        <input type="checkbox" ${isEnabled ? 'checked' : ''} 
                                               onchange="toggleEnabled(${config.id}, this.checked)">
                                        <span class="toggle-slider"></span>
                                    </label>
                                    <button class="btn btn-primary btn-sm" onclick="event.stopPropagation(); showTestSendModal(${config.id}, '${key}', '${escapeHtml(connectedTemplate.name)}')" title="Test senden">
                                        <i class="fas fa-paper-plane"></i>
                                    </button>
                                    <button class="btn btn-danger btn-sm" onclick="event.stopPropagation(); disconnectTemplate('${key}', ${config.id})" title="Zuweisung entfernen">
                                        <i class="fas fa-unlink"></i>
                                    </button>
                                </div>
                            </div>
                        ` : `
                            <div style="margin-top: 0.75rem;">
                                <select class="template-select-dropdown" 
                                        onchange="assignTemplateFromDropdown('${key}', this.value)"
                                        style="width: 100%; padding: 0.5rem; border: 1px solid var(--border); border-radius: 6px; font-size: 0.9rem;">
                                    <option value="">-- Template auswählen --</option>
                                    ${allTemplates.map(t => `
                                        <option value="${t.id}">${escapeHtml(t.name)}</option>
                                    `).join('')}
                                </select>
                            </div>
                        `}
                    </div>
                `;
            });
            
            container.innerHTML = html;
            
            // Lade Details für Templates, die sie benötigen (lazy loading)
            // WICHTIG: Dies passiert asynchron nach dem Rendern
            allTemplates.forEach(async (template) => {
                // Prüfe ob Body fehlt (dann Details nachladen)
                const hasBody = template.components && template.components.some(c => c && c.type === 'BODY');
                if (!hasBody && template.needs_details !== false) {
                    const previewElement = document.querySelector(`[data-template-id="${template.id}"]`);
                    if (previewElement) {
                        // Zeige Loading-Indikator
                        previewElement.innerHTML = '<div class="preview-loading"><i class="fas fa-spinner fa-spin"></i> Lade Vorschau...</div>';
                        
                        try {
                            const fullTemplate = await loadTemplateDetailsIfNeeded(template);
                            // Aktualisiere die Vorschau
                            previewElement.innerHTML = renderWhatsAppPreview(fullTemplate);
                            // Aktualisiere auch im allTemplates Array
                            const index = allTemplates.findIndex(t => t.id === template.id);
                            if (index !== -1) {
                                allTemplates[index] = fullTemplate;
                            }
                        } catch (error) {
                            console.error('Fehler beim Laden der Template-Details:', error);
                            previewElement.innerHTML = '<div class="whatsapp-preview-empty">Fehler beim Laden der Vorschau</div>';
                        }
                    }
                }
            });
        }
        
        function selectAction(actionType) {
            selectedAction = actionType;
            document.querySelectorAll('.action-card').forEach(card => {
                card.classList.remove('selected');
            });
            document.querySelector(`[data-action-type="${actionType}"]`)?.classList.add('selected');
            
            // Markiere zugewiesenes Template
            const config = templateConfigs.find(c => c.action_type === actionType);
            if (config) {
                document.querySelectorAll('.template-card').forEach(card => {
                    card.classList.remove('selected');
                    if (card.dataset.templateId === config.template_id) {
                        card.classList.add('selected');
                    }
                });
            }
        }
        
        // Alte Funktion entfernt - Zuweisung nur über Dropdown
        
        // Zuweisung über Dropdown
        function assignTemplateFromDropdown(actionType, templateId) {
            if (!templateId || templateId === '') {
                return;
            }
            
            const template = allTemplates.find(t => t.id === templateId);
            if (!template) {
                showAlert('error', 'Template nicht gefunden');
                return;
            }
            
            // Speichere Zuweisung direkt
            saveAssignment(actionType, template);
        }
        
        function saveAssignment(actionType, template) {
            const action = actionTypes[actionType];
            const defaultMapping = {
                'custom_fields.last_name': 'last_name',
                'custom_fields.first_name': 'first_name'
            };
            
            const formData = new FormData();
            formData.append('ajax_action', 'save_template_assignment');
            formData.append('autohaus_id', autohausId);
            formData.append('action_type', actionType);
            formData.append('template_id', template.id);
            formData.append('timing_hours', action.timing);
            formData.append('custom_fields_mapping', JSON.stringify(defaultMapping));
            
            fetch('whatsapp_reminders_admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showAlert('success', 'Template erfolgreich zugewiesen');
                    // Reload page to refresh configs
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    showAlert('error', data.error || 'Fehler beim Speichern');
                }
            })
            .catch(error => {
                showAlert('error', 'Fehler: ' + error.message);
            });
        }
        
        function toggleEnabled(configId, enabled) {
            const formData = new FormData();
            formData.append('ajax_action', 'toggle_enabled');
            formData.append('config_id', configId);
            formData.append('is_enabled', enabled ? 1 : 0);
            
            fetch('whatsapp_reminders_admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showAlert('success', enabled ? 'Nachricht aktiviert' : 'Nachricht deaktiviert');
                } else {
                    showAlert('error', data.error || 'Fehler beim Aktualisieren');
                    // Revert toggle
                    const toggle = document.querySelector(`input[onchange*="toggleEnabled(${configId}"]`);
                    if (toggle) toggle.checked = !enabled;
                }
            })
            .catch(error => {
                showAlert('error', 'Fehler: ' + error.message);
                // Revert toggle
                const toggle = document.querySelector(`input[onchange*="toggleEnabled(${configId}"]`);
                if (toggle) toggle.checked = !enabled;
            });
        }
        
        function showTestSendModal(configId, actionType, templateName) {
            // Erstelle Modal falls nicht vorhanden
            let modal = document.getElementById('test-send-modal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'test-send-modal';
                modal.className = 'test-send-modal';
                document.body.appendChild(modal);
                
                // Schließe Modal bei Klick außerhalb
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        closeTestSendModal();
                    }
                });
            }
            
            // Lade Termine für Dropdown
            fetch(`api/get_appointments_for_test.php?autohaus_id=${autohausId}`)
                .then(r => r.json())
                .then(appointmentsData => {
                    const appointments = appointmentsData.success ? appointmentsData.data : [];
                    
                    modal.innerHTML = `
                        <div class="test-send-modal-content">
                            <div class="test-send-modal-header">
                                <h3><i class="fas fa-paper-plane"></i> Test-Nachricht senden</h3>
                                <button class="test-send-modal-close" onclick="closeTestSendModal()">&times;</button>
                            </div>
                            <div class="test-send-modal-body">
                                <div style="margin-bottom: 1rem; padding: 1rem; background: #e3f2fd; border-radius: 8px;">
                                    <strong>Template:</strong> ${escapeHtml(templateName)}<br>
                                    <strong>Aktion:</strong> ${actionTypes[actionType]?.label || actionType}
                                </div>
                                <div class="test-send-form-group">
                                    <label>Test-Telefonnummer *</label>
                                    <input type="tel" id="test-phone" placeholder="+49 123 456 789" required>
                                    <small>Telefonnummer im internationalen Format (z.B. +49123456789)</small>
                                </div>
                                <div class="test-send-form-group">
                                    <label>ODER wählen Sie einen Test-Termin</label>
                                    <select id="test-appointment-id" onchange="document.getElementById('test-phone').required = !this.value">
                                        <option value="">-- Kein Termin (nur Telefonnummer) --</option>
                                        ${appointments.map(apt => `
                                            <option value="${apt.id}" data-phone="${escapeHtml(apt.phone || '')}">
                                                ${escapeHtml(apt.customer_name || 'Unbekannt')} - ${escapeHtml(apt.phone || 'Keine Telefonnummer')} (${apt.slot_date} ${apt.slot_time})
                                            </option>
                                        `).join('')}
                                    </select>
                                    <small>Wenn ein Termin ausgewählt wird, wird die Telefonnummer des Termins verwendet</small>
                                </div>
                            </div>
                            <div class="test-send-modal-footer">
                                <button class="btn btn-secondary" onclick="closeTestSendModal()">Abbrechen</button>
                                <button class="btn btn-primary" onclick="sendTestMessage(${configId})">
                                    <i class="fas fa-paper-plane"></i> Test senden
                                </button>
                            </div>
                        </div>
                    `;
                    
                    // Wenn Termin ausgewählt wird, fülle Telefonnummer automatisch
                    document.getElementById('test-appointment-id').addEventListener('change', function() {
                        const selectedOption = this.options[this.selectedIndex];
                        if (selectedOption.value && selectedOption.dataset.phone) {
                            document.getElementById('test-phone').value = selectedOption.dataset.phone;
                        }
                    });
                    
                    modal.classList.add('show');
                })
                .catch(error => {
                    console.error('Fehler beim Laden der Termine:', error);
                    // Zeige Modal trotzdem ohne Termine
                    modal.innerHTML = `
                        <div class="test-send-modal-content">
                            <div class="test-send-modal-header">
                                <h3><i class="fas fa-paper-plane"></i> Test-Nachricht senden</h3>
                                <button class="test-send-modal-close" onclick="closeTestSendModal()">&times;</button>
                            </div>
                            <div class="test-send-modal-body">
                                <div style="margin-bottom: 1rem; padding: 1rem; background: #e3f2fd; border-radius: 8px;">
                                    <strong>Template:</strong> ${escapeHtml(templateName)}<br>
                                    <strong>Aktion:</strong> ${actionTypes[actionType]?.label || actionType}
                                </div>
                                <div class="test-send-form-group">
                                    <label>Test-Telefonnummer *</label>
                                    <input type="tel" id="test-phone" placeholder="+49 123 456 789" required>
                                    <small>Telefonnummer im internationalen Format (z.B. +49123456789)</small>
                                </div>
                            </div>
                            <div class="test-send-modal-footer">
                                <button class="btn btn-secondary" onclick="closeTestSendModal()">Abbrechen</button>
                                <button class="btn btn-primary" onclick="sendTestMessage(${configId})">
                                    <i class="fas fa-paper-plane"></i> Test senden
                                </button>
                            </div>
                        </div>
                    `;
                    modal.classList.add('show');
                });
        }
        
        function closeTestSendModal() {
            const modal = document.getElementById('test-send-modal');
            if (modal) {
                modal.classList.remove('show');
            }
        }
        
        function sendTestMessage(configId) {
            const testPhone = document.getElementById('test-phone')?.value.trim();
            const testAppointmentId = document.getElementById('test-appointment-id')?.value || '';
            
            if (!testPhone && !testAppointmentId) {
                showAlert('error', 'Bitte geben Sie eine Telefonnummer ein oder wählen Sie einen Termin');
                return;
            }
            
            const formData = new FormData();
            formData.append('ajax_action', 'test_send');
            formData.append('config_id', configId);
            if (testPhone) formData.append('test_phone', testPhone);
            if (testAppointmentId) formData.append('test_appointment_id', testAppointmentId);
            
            // Zeige Loading
            const sendButton = document.querySelector('#test-send-modal .btn-primary');
            const originalText = sendButton.innerHTML;
            sendButton.disabled = true;
            sendButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sende...';
            
            fetch('whatsapp_reminders_admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showAlert('success', 'Test-Nachricht erfolgreich gesendet!');
                    closeTestSendModal();
                } else {
                    showAlert('error', data.error || 'Fehler beim Senden der Test-Nachricht');
                    if (data.details) {
                        console.error('Test-Versand Details:', data.details);
                    }
                }
            })
            .catch(error => {
                showAlert('error', 'Fehler: ' + error.message);
            })
            .finally(() => {
                sendButton.disabled = false;
                sendButton.innerHTML = originalText;
            });
        }
        
        function disconnectTemplate(actionType, configId) {
            if (!confirm('Möchten Sie diese Zuweisung wirklich entfernen?')) return;
            
            const formData = new FormData();
            formData.append('ajax_action', 'delete_template_assignment');
            formData.append('config_id', configId);
            
            fetch('whatsapp_reminders_admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showAlert('success', 'Zuweisung entfernt');
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    showAlert('error', data.error || 'Fehler beim Löschen');
                }
            });
        }
        
        function showAlert(type, message) {
            const container = document.getElementById('alert-container');
            const alert = document.createElement('div');
            alert.className = `alert alert-${type}`;
            alert.innerHTML = `<i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}"></i> ${escapeHtml(message)}`;
            container.innerHTML = '';
            container.appendChild(alert);
            
            setTimeout(() => alert.remove(), 5000);
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Extrahiere HTML aus Slate-Content-Objekt (HelloMateo verwendet Slate-Format)
        // Gibt HTML zurück, um Struktur und Formatierung zu erhalten
        function extractHtmlFromSlate(content, preserveStructure = true) {
            if (!content || typeof content !== 'object') return '';
            
            let html = '';
            
            function processNode(node) {
                if (!node) return '';
                
                let nodeHtml = '';
                
                // Text-Node
                if (node.type === 'text' && node.content) {
                    // Prüfe auf Formatierung
                    let text = escapeHtml(node.content);
                    if (node.bold) text = '<strong>' + text + '</strong>';
                    if (node.italic) text = '<em>' + text + '</em>';
                    if (node.underline) text = '<u>' + text + '</u>';
                    nodeHtml = text;
                }
                // Handlebar-Node (Platzhalter)
                else if (node.type === 'handlebar' && node.key) {
                    nodeHtml = '<span class="whatsapp-placeholder">{{' + node.key + '}}</span>';
                }
                // Newline
                else if (node.type === 'newline') {
                    nodeHtml = '<br>';
                }
                // Paragraph
                else if (node.type === 'paragraph') {
                    let paraContent = '';
                    if (node.children && Array.isArray(node.children)) {
                        node.children.forEach(child => {
                            paraContent += processNode(child);
                        });
                    }
                    nodeHtml = '<p>' + paraContent + '</p>';
                }
                // Heading
                else if (node.type === 'heading') {
                    let headingContent = '';
                    if (node.children && Array.isArray(node.children)) {
                        node.children.forEach(child => {
                            headingContent += processNode(child);
                        });
                    }
                    const level = node.level || 1;
                    nodeHtml = '<h' + level + '>' + headingContent + '</h' + level + '>';
                }
                // Liste
                else if (node.type === 'list') {
                    let listContent = '';
                    if (node.children && Array.isArray(node.children)) {
                        node.children.forEach(child => {
                            listContent += '<li>' + processNode(child) + '</li>';
                        });
                    }
                    const listTag = node.listType === 'ordered' ? 'ol' : 'ul';
                    nodeHtml = '<' + listTag + '>' + listContent + '</' + listTag + '>';
                }
                // Andere Container mit children
                else if (node.children && Array.isArray(node.children)) {
                    node.children.forEach(child => {
                        nodeHtml += processNode(child);
                    });
                }
                
                return nodeHtml;
            }
            
            // Starte mit root oder direkt mit content
            if (content.type === 'root' && content.children) {
                content.children.forEach(child => {
                    html += processNode(child);
                });
            } else {
                html = processNode(content);
            }
            
            return html.trim();
        }
        
        // Extrahiere Text aus Slate-Content-Objekt (für einfache Text-Extraktion)
        function extractTextFromSlate(content) {
            if (!content || typeof content !== 'object') return '';
            
            let text = '';
            
            function processNode(node) {
                if (!node) return;
                
                // Text-Node
                if (node.type === 'text' && node.content) {
                    text += node.content;
                }
                // Handlebar-Node (Platzhalter)
                else if (node.type === 'handlebar' && node.key) {
                    text += '{{' + node.key + '}}';
                }
                // Newline
                else if (node.type === 'newline') {
                    text += '\n';
                }
                // Paragraph oder andere Container
                else if (node.children && Array.isArray(node.children)) {
                    node.children.forEach(child => {
                        processNode(child);
                    });
                    // Füge Zeilenumbruch nach Paragraph hinzu
                    if (node.type === 'paragraph') {
                        text += '\n';
                    }
                }
            }
            
            // Starte mit root oder direkt mit content
            if (content.type === 'root' && content.children) {
                content.children.forEach(child => {
                    processNode(child);
                });
            } else {
                processNode(content);
            }
            
            return text.trim();
        }
        
        // Lade Template-Details bei Bedarf (lazy loading)
        async function loadTemplateDetailsIfNeeded(template) {
            // Prüfe ob Details bereits vorhanden sind
            const hasBody = template.components && template.components.some(c => c.type === 'BODY');
            if (hasBody || !template.needs_details) {
                return template; // Details bereits vorhanden
            }
            
            // Lade Details von API
            try {
                const response = await fetch(`api/get_template_details.php?template_id=${template.id}&autohaus_id=${autohausId}&v=${APP_VERSION}`);
                const data = await response.json();
                
                if (data.success && data.data) {
                    // Merge die Details mit dem Template
                    if (data.data.components && data.data.components.length > 0) {
                        template.components = data.data.components;
                    }
                    if (data.data.body) template.body = data.data.body;
                    if (data.data.header) template.header = data.data.header;
                    if (data.data.footer) template.footer = data.data.footer;
                    template.needs_details = false; // Markiere als geladen
                }
            } catch (error) {
                console.error('Fehler beim Laden der Template-Details:', error);
            }
            
            return template;
        }
        
        // WhatsApp-ähnliche Vorschau rendern
        function renderWhatsAppPreview(template, fullSize = false) {
            // Prüfe ob Components vorhanden sind
            let components = template.components;
            
            // Falls components nicht als Array kommt, versuche es zu konvertieren
            if (!components) {
                components = [];
            } else if (!Array.isArray(components)) {
                // Falls es ein Objekt ist, versuche es zu konvertieren
                if (typeof components === 'object') {
                    components = Object.values(components);
                } else {
                    components = [];
                }
            }
            
            // Prüfe ob Body direkt im Template ist (nicht in Components)
            // HelloMateo API kann Body-Text direkt im Template-Objekt haben
            if (template.body && !components.find(c => c.type === 'BODY')) {
                console.log('Body direkt im Template gefunden, füge zu Components hinzu:', template.body);
                const bodyText = typeof template.body === 'string' ? template.body : (template.body.text || template.body.content || '');
                if (bodyText) {
                    components.push({type: 'BODY', text: bodyText});
                }
            }
            
            // Prüfe ob Header direkt im Template ist
            if (template.header && !components.find(c => c.type === 'HEADER')) {
                console.log('Header direkt im Template gefunden, füge zu Components hinzu:', template.header);
                const headerText = typeof template.header === 'string' ? template.header : (template.header.text || template.header.content || '');
                if (headerText) {
                    components.push({type: 'HEADER', text: headerText, format: template.header.format || null});
                }
            }
            
            if (components.length === 0) {
                console.log('Keine Components gefunden - Template Keys:', Object.keys(template));
                // Versuche alternative Felder
                if (template.body || template.header || template.footer) {
                    // Baue Components aus separaten Feldern
                    components = [];
                    if (template.header) {
                        const headerText = typeof template.header === 'string' ? template.header : (template.header.text || template.header.content || '');
                        components.push({type: 'HEADER', text: headerText});
                    }
                    if (template.body) {
                        const bodyText = typeof template.body === 'string' ? template.body : (template.body.text || template.body.content || '');
                        components.push({type: 'BODY', text: bodyText});
                    }
                    if (template.footer) {
                        const footerText = typeof template.footer === 'string' ? template.footer : (template.footer.text || template.footer.content || '');
                        components.push({type: 'FOOTER', text: footerText});
                    }
                }
                
                if (components.length === 0) {
                    return '<div class="whatsapp-preview-empty">Keine Vorschau verfügbar (keine Components)</div>';
                }
            }
            
            // Finde Header, Body und Footer
            const header = components.find(c => c.type === 'HEADER');
            const body = components.find(c => c.type === 'BODY');
            const footer = components.find(c => c.type === 'FOOTER');
            const buttons = components.filter(c => c.type === 'BUTTONS');
            
            // Debug
            console.log('Components gefunden:', {header, body, footer, buttons: buttons.length, componentsCount: components.length});
            
            // Größere Vorschau für Modal
            const containerClass = fullSize ? 'whatsapp-preview-container-full' : 'whatsapp-preview-container';
            const bubbleClass = fullSize ? 'whatsapp-message-bubble-full' : 'whatsapp-message-bubble';
            
            let html = `<div class="${containerClass}">`;
            html += `<div class="${bubbleClass}">`;
            
            // Header (falls vorhanden)
            if (header) {
                if (header.format === 'IMAGE' || header.format === 'VIDEO' || header.format === 'DOCUMENT') {
                    // Prüfe ob URL vorhanden ist
                    if (header.url && header.format === 'IMAGE') {
                        html += `<div class="whatsapp-header-media">
                            <img src="${escapeHtml(header.url)}" alt="${escapeHtml(header.filename || 'Header')}" style="max-width: 100%; border-radius: 8px 8px 0 0;" onerror="this.parentElement.innerHTML='<div class=\\'whatsapp-media-placeholder\\'><i class=\\'fas fa-image\\'></i><span>Bild konnte nicht geladen werden</span></div>'">
                        </div>`;
                    } else {
                        html += `<div class="whatsapp-header-media">
                            <div class="whatsapp-media-placeholder">
                                <i class="fas fa-${header.format === 'IMAGE' ? 'image' : header.format === 'VIDEO' ? 'video' : 'file'}"></i>
                                <span>${header.format}</span>
                                ${header.filename ? '<small>' + escapeHtml(header.filename) + '</small>' : ''}
                            </div>
                        </div>`;
                    }
                } else if (header.text) {
                    html += `<div class="whatsapp-header-text">${escapeHtml(header.text)}</div>`;
                }
            }
            
            // Body - Prüfe verschiedene mögliche Felder
            if (body) {
                // Debug: Logge Body-Struktur
                console.log('Body Component:', body);
                console.log('Body Type:', typeof body);
                console.log('Body Keys:', typeof body === 'object' ? Object.keys(body) : 'N/A');
                
                // Prüfe verschiedene mögliche Text-Felder im Body
                let bodyText = '';
                let bodyIsHtml = false;
                
                // Falls body ein String ist, verwende es direkt
                if (typeof body === 'string') {
                    bodyText = body;
                } else if (typeof body === 'object' && body !== null) {
                    // Prüfe verschiedene Felder im Body-Objekt
                    // WICHTIG: Prüfe zuerst 'text', dann 'content', dann andere Felder
                    bodyText = body.text || body.content || body.message || body.format || body.value || '';
                    
                    // Falls body.content ein Slate-Objekt ist, extrahiere HTML (mit Struktur)
                    if (!bodyText && body.content && typeof body.content === 'object') {
                        // Verwende HTML-Extraktion für bessere Struktur
                        const bodyHtml = extractHtmlFromSlate(body.content, true);
                        if (bodyHtml) {
                            // Verwende HTML direkt statt Text
                            bodyText = bodyHtml;
                            bodyIsHtml = true; // Flag für HTML-Inhalt
                        } else {
                            // Fallback zu Text
                            bodyText = extractTextFromSlate(body.content);
                        }
                    }
                    
                    // Falls body selbst ein Array ist (unwahrscheinlich, aber möglich)
                    if (!bodyText && Array.isArray(body)) {
                        bodyText = body.map(item => {
                            if (typeof item === 'string') return item;
                            if (typeof item === 'object' && item !== null) {
                                return item.text || item.content || item.message || '';
                            }
                            return '';
                        }).filter(t => t).join(' ');
                    }
                }
                
                // Falls immer noch kein Text, versuche alle Werte des Objekts zu durchsuchen
                if (!bodyText && typeof body === 'object' && body !== null) {
                    const allValues = Object.values(body).filter(v => typeof v === 'string' && v.length > 0);
                    if (allValues.length > 0) {
                        bodyText = allValues[0]; // Nimm den ersten String-Wert
                        console.log('Body Text aus Objekt-Werten extrahiert:', bodyText);
                    }
                }
                
                if (bodyText) {
                    // Ersetze Platzhalter mit Beispielwerten (verschiedene Formate)
                    // {{1}}, {{2}}, etc.
                    bodyText = bodyText.replace(/\{\{(\d+)\}\}/g, (match, num) => {
                        const examples = ['Max', 'Mustermann', '01.01.2024', '10:00', 'Kiel'];
                        return `<span class="whatsapp-placeholder">${examples[parseInt(num) - 1] || 'Wert'}</span>`;
                    });
                    // Ersetze auch {{custom_fields.last_name}} Format
                    bodyText = bodyText.replace(/\{\{custom_fields\.(\w+)\}\}/g, (match, field) => {
                        const examples = {last_name: 'Mustermann', first_name: 'Max', date: '01.01.2024', time: '10:00'};
                        return `<span class="whatsapp-placeholder">${examples[field] || field}</span>`;
                    });
                    // Ersetze auch {{placeholder}}, {{full_name}}, etc.
                    bodyText = bodyText.replace(/\{\{(\w+)\}\}/g, (match, key) => {
                        const examples = {
                            placeholder: 'Max Mustermann',
                            full_name: 'Max Mustermann',
                            organisation_name: 'B&E Autocentrum',
                            salutation: 'Herr',
                            review_app_link: 'https://example.com/review'
                        };
                        return `<span class="whatsapp-placeholder">${examples[key] || key}</span>`;
                    });
                    // Ersetze Zeilenumbrüche nur wenn es kein HTML ist
                    if (!bodyIsHtml) {
                        bodyText = bodyText.replace(/\n/g, '<br>');
                    }
                    
                    html += `<div class="whatsapp-body">${bodyText}</div>`;
                } else {
                    // Debug: Zeige welche Felder im Body vorhanden sind
                    console.log('Body gefunden, aber kein Text extrahiert. Body-Objekt:', JSON.stringify(body, null, 2));
                    html += '<div class="whatsapp-body" style="color: #999; font-style: italic;">(Kein Text im Body gefunden)</div>';
                }
            } else {
                html += '<div class="whatsapp-body" style="color: #999; font-style: italic;">(Kein Body-Component)</div>';
            }
            
            // Footer
            if (footer && footer.text) {
                html += `<div class="whatsapp-footer">${escapeHtml(footer.text)}</div>`;
            }
            
            // Buttons
            if (buttons && buttons.length > 0) {
                html += '<div class="whatsapp-buttons">';
                buttons.forEach(buttonGroup => {
                    if (buttonGroup.buttons) {
                        buttonGroup.buttons.forEach(btn => {
                            if (btn.type === 'QUICK_REPLY') {
                                html += `<div class="whatsapp-button-quick">${escapeHtml(btn.text || btn.payload || 'Button')}</div>`;
                            } else if (btn.type === 'URL') {
                                html += `<div class="whatsapp-button-url">🔗 ${escapeHtml(btn.text || 'Link')}</div>`;
                            } else if (btn.type === 'PHONE_NUMBER') {
                                html += `<div class="whatsapp-button-phone">📞 ${escapeHtml(btn.text || 'Anrufen')}</div>`;
                            }
                        });
                    }
                });
                html += '</div>';
            }
            
            html += '</div>'; // whatsapp-message-bubble
            const now = new Date();
            html += `<div class="whatsapp-time">${now.toLocaleTimeString('de-DE', {hour: '2-digit', minute: '2-digit'})}</div>`;
            html += '</div>'; // whatsapp-preview-container
            
            return html;
        }
        
        // Vollständige Vorschau in Modal anzeigen
        async function showFullPreview(template) {
            // Erstelle Modal falls nicht vorhanden
            let modal = document.getElementById('preview-modal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'preview-modal';
                modal.className = 'preview-modal';
                modal.innerHTML = `
                    <div class="preview-modal-content">
                        <div class="preview-modal-header">
                            <h3><i class="fab fa-whatsapp"></i> Template-Vorschau</h3>
                            <button class="preview-modal-close" onclick="closeFullPreview()">&times;</button>
                        </div>
                        <div class="preview-modal-body" id="preview-modal-body">
                            <!-- Wird gefüllt -->
                        </div>
                        <div class="preview-modal-footer">
                            <button class="btn btn-secondary" onclick="closeFullPreview()">Schließen</button>
                        </div>
                    </div>
                `;
                document.body.appendChild(modal);
                
                // Schließe Modal bei Klick außerhalb
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        closeFullPreview();
                    }
                });
                
                // Schließe Modal bei ESC-Taste
                document.addEventListener('keydown', function(e) {
                    if (e.key === 'Escape') {
                        if (modal.classList.contains('show')) {
                            closeFullPreview();
                        }
                        const testModal = document.getElementById('test-send-modal');
                        if (testModal && testModal.classList.contains('show')) {
                            closeTestSendModal();
                        }
                    }
                });
            }
            
            // Fülle Modal mit vollständiger Vorschau
            const modalBody = document.getElementById('preview-modal-body');
            modalBody.innerHTML = `
                <div style="margin-bottom: 1rem;">
                    <strong>Template-Name:</strong> ${escapeHtml(template.name)}<br>
                    <strong>Template-ID:</strong> <code>${escapeHtml(template.id)}</code><br>
                    <strong>Status:</strong> ${template.status || 'Unbekannt'}<br>
                    <strong>Sprache:</strong> ${template.language || 'de'}<br>
                    <strong>Kategorie:</strong> ${template.category || 'UTILITY'}
                </div>
                <div style="margin-top: 1.5rem;">
                    <strong>Vorschau:</strong>
                </div>
                <div id="modal-preview-content">Lade Vorschau...</div>
            `;
            
            // Lade Details und rendere Vorschau
            (async function() {
                const fullTemplate = await loadTemplateDetailsIfNeeded(template);
                document.getElementById('modal-preview-content').innerHTML = renderWhatsAppPreview(fullTemplate, true);
            })();
            
            // Zeige Modal
            modal.classList.add('show');
        }
        
        function closeFullPreview() {
            const modal = document.getElementById('preview-modal');
            if (modal) {
                modal.classList.remove('show');
            }
        }
        
        // Initialisierung - Lade Templates automatisch beim ersten Laden
        if (autohausId) {
            showLoading();
            // Lade Templates automatisch
            loadTemplates();
            renderActions();
        } else {
            hideLoading();
        }
        
        // Lade Templates automatisch beim Seitenaufruf
        window.addEventListener('DOMContentLoaded', function() {
            if (autohausId) {
                loadTemplates();
            }
        });
    </script>
</body>
</html>
