<?php
/**
 * WhatsApp Logs Export
 * Exportiert WhatsApp-Logs als CSV oder Excel
 */

session_start();

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

require_once __DIR__ . '/../config/database.php';
require_once 'includes/helpers.php';

$db = new Database();
$conn = $db->getConnection();

$exportFormat = $_GET['export'] ?? 'csv';
$filterStatus = $_GET['status'] ?? 'all';
$filterAutohaus = $_GET['autohaus_id'] ?? null;
$filterActionType = $_GET['action_type'] ?? 'all';
$filterSearch = $_GET['search'] ?? '';
$filterDateFrom = $_GET['date_from'] ?? null;
$filterDateTo = $_GET['date_to'] ?? null;
$timeRange = $_GET['time_range'] ?? '7d';

// Zeitraum-Berechnung
$dateFrom = null;
$dateTo = date('Y-m-d');
switch ($timeRange) {
    case 'today':
        $dateFrom = date('Y-m-d');
        break;
    case '7d':
        $dateFrom = date('Y-m-d', strtotime('-7 days'));
        break;
    case '30d':
        $dateFrom = date('Y-m-d', strtotime('-30 days'));
        break;
    case '90d':
        $dateFrom = date('Y-m-d', strtotime('-90 days'));
        break;
    case 'custom':
        if ($filterDateFrom) $dateFrom = $filterDateFrom;
        if ($filterDateTo) $dateTo = $filterDateTo;
        break;
}

// Baue Query
$where = ['1=1'];
$params = [];

if ($dateFrom) {
    $where[] = 'DATE(wl.created_at) >= ?';
    $params[] = $dateFrom;
}

if ($dateTo) {
    $where[] = 'DATE(wl.created_at) <= ?';
    $params[] = $dateTo;
}

if ($filterStatus !== 'all') {
    $where[] = 'wl.status = ?';
    $params[] = $filterStatus;
}

if ($filterAutohaus) {
    $where[] = 'wl.autohaus_id = ?';
    $params[] = $filterAutohaus;
}

if ($filterActionType !== 'all') {
    $where[] = 'wl.action_type = ?';
    $params[] = $filterActionType;
}

if ($filterSearch) {
    $where[] = '(wl.phone_number LIKE ? OR wl.customer_name LIKE ? OR wl.message LIKE ?)';
    $searchTerm = '%' . $filterSearch . '%';
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$whereClause = implode(' AND ', $where);

// Lade Logs (ohne Limit für Export)
$logsStmt = $conn->prepare("
    SELECT 
        wl.id,
        wl.created_at,
        wl.customer_name,
        wl.phone_number,
        a.name as autohaus_name,
        wl.action_type,
        wl.status,
        wl.message,
        wl.error_message,
        ap.slot_date,
        ap.slot_time
    FROM whatsapp_logs wl
    LEFT JOIN autohaus a ON wl.autohaus_id = a.id
    LEFT JOIN appointments ap ON wl.appointment_id = ap.id
    WHERE $whereClause
    ORDER BY wl.created_at DESC
");
$logsStmt->execute($params);
$logs = $logsStmt->fetchAll(PDO::FETCH_ASSOC);

// CSV Export
if ($exportFormat === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="whatsapp_logs_' . date('Y-m-d_His') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // UTF-8 BOM für Excel-Kompatibilität
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Header
    fputcsv($output, [
        'ID',
        'Datum/Zeit',
        'Kunde',
        'Telefon',
        'Autohaus',
        'Aktionstyp',
        'Status',
        'Nachricht',
        'Fehler',
        'Termin Datum',
        'Termin Zeit'
    ], ';');
    
    // Daten
    foreach ($logs as $log) {
        fputcsv($output, [
            $log['id'],
            $log['created_at'],
            $log['customer_name'] ?? '',
            $log['phone_number'],
            $log['autohaus_name'] ?? '',
            $log['action_type'] ?? '',
            $log['status'],
            $log['message'] ?? '',
            $log['error_message'] ?? '',
            $log['slot_date'] ?? '',
            $log['slot_time'] ?? ''
        ], ';');
    }
    
    fclose($output);
    exit;
}

// Excel Export (als CSV mit Excel-Kompatibilität)
header('Content-Type: application/vnd.ms-excel; charset=utf-8');
header('Content-Disposition: attachment; filename="whatsapp_logs_' . date('Y-m-d_His') . '.xls"');

echo '<html><head><meta charset="UTF-8"></head><body>';
echo '<table border="1">';
echo '<tr>';
echo '<th>ID</th>';
echo '<th>Datum/Zeit</th>';
echo '<th>Kunde</th>';
echo '<th>Telefon</th>';
echo '<th>Autohaus</th>';
echo '<th>Aktionstyp</th>';
echo '<th>Status</th>';
echo '<th>Nachricht</th>';
echo '<th>Fehler</th>';
echo '<th>Termin Datum</th>';
echo '<th>Termin Zeit</th>';
echo '</tr>';

foreach ($logs as $log) {
    echo '<tr>';
    echo '<td>' . htmlspecialchars($log['id']) . '</td>';
    echo '<td>' . htmlspecialchars($log['created_at']) . '</td>';
    echo '<td>' . htmlspecialchars($log['customer_name'] ?? '') . '</td>';
    echo '<td>' . htmlspecialchars($log['phone_number']) . '</td>';
    echo '<td>' . htmlspecialchars($log['autohaus_name'] ?? '') . '</td>';
    echo '<td>' . htmlspecialchars($log['action_type'] ?? '') . '</td>';
    echo '<td>' . htmlspecialchars($log['status']) . '</td>';
    echo '<td>' . htmlspecialchars($log['message'] ?? '') . '</td>';
    echo '<td>' . htmlspecialchars($log['error_message'] ?? '') . '</td>';
    echo '<td>' . htmlspecialchars($log['slot_date'] ?? '') . '</td>';
    echo '<td>' . htmlspecialchars($log['slot_time'] ?? '') . '</td>';
    echo '</tr>';
}

echo '</table></body></html>';
exit;
?>




