<?php
session_start();
require_once '../config/database.php';
require_once 'includes/helpers.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT full_name, username, role FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
    $current_user_role = $user_data['role'] ?? 'admin';
}

// Get user theme preference
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Handle customer linking
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['link_customer_to_admin'])) {
            $customer_user_id = $_POST['customer_user_id'];
            $admin_user_id = $_POST['admin_user_id'];
            
            $stmt = $conn->prepare("UPDATE users SET admin_user_id = ? WHERE id = ?");
            $stmt->execute([$admin_user_id, $customer_user_id]);
            
            $_SESSION['success_message'] = 'Konto erfolgreich mit Admin verknüpft';
            header('Location: users.php');
            exit;
        }
        
        if (isset($_POST['unlink_customer'])) {
            $customer_user_id = $_POST['customer_user_id'];
            
            $stmt = $conn->prepare("UPDATE users SET admin_user_id = NULL WHERE id = ?");
            $stmt->execute([$customer_user_id]);
            
            $_SESSION['success_message'] = 'Verknüpfung erfolgreich entfernt';
            header('Location: users.php');
            exit;
        }
    } catch (Exception $e) {
        $_SESSION['error_message'] = 'Fehler: ' . $e->getMessage();
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['action']) && $_POST['action'] === 'create_user') {
            $username = $_POST['username'];
            $email = $_POST['email'];
            $password = password_hash($_POST['password'], PASSWORD_DEFAULT);
            $role = $_POST['role'];
            $full_name = $_POST['full_name'];
            $department = $_POST['department'];
            $autohaus_id = $_POST['autohaus_id'] ?: null;
            $can_switch_autohaus = isset($_POST['can_switch_autohaus']) ? 1 : 0;
            $phone = $_POST['phone'] ?? '';
            
            // Prüfe ob phone Spalte existiert
            try {
                $checkPhone = $conn->query("SHOW COLUMNS FROM admin_users LIKE 'phone'")->rowCount();
                if ($checkPhone > 0) {
                    $stmt = $conn->prepare("
                        INSERT INTO admin_users (username, email, password_hash, role, full_name, department, autohaus_id, can_switch_autohaus, phone, is_active, needs_dashboard_setup, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 1, 1, NOW())
                    ");
                    $stmt->execute([$username, $email, $password, $role, $full_name, $department, $autohaus_id, $can_switch_autohaus, $phone]);
                } else {
            $stmt = $conn->prepare("
                INSERT INTO admin_users (username, email, password_hash, role, full_name, department, autohaus_id, can_switch_autohaus, is_active, needs_dashboard_setup, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1, 1, NOW())
            ");
            $stmt->execute([$username, $email, $password, $role, $full_name, $department, $autohaus_id, $can_switch_autohaus]);
                }
            } catch (Exception $e) {
                // Fallback ohne phone
                $stmt = $conn->prepare("
                    INSERT INTO admin_users (username, email, password_hash, role, full_name, department, autohaus_id, can_switch_autohaus, is_active, needs_dashboard_setup, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1, 1, NOW())
                ");
                $stmt->execute([$username, $email, $password, $role, $full_name, $department, $autohaus_id, $can_switch_autohaus]);
            }
            
            $new_user_id = $conn->lastInsertId();
            
            // Aktiviere 2FA und sende Verifizierungscode wenn Telefonnummer vorhanden
            if (!empty($phone)) {
                require_once 'includes/whatsapp_2fa.php';
                $verification_result = setWhatsApp2FA($new_user_id, $phone, true, $autohaus_id);
                
                if ($verification_result['success']) {
                    $_SESSION['success_message'] = 'Benutzer erfolgreich erstellt. Ein Verifizierungscode wurde per WhatsApp gesendet.';
                    $_SESSION['pending_verification_user_id'] = $new_user_id;
                } else {
                    $_SESSION['warning_message'] = 'Benutzer erfolgreich erstellt, aber Verifizierungscode konnte nicht gesendet werden: ' . ($verification_result['error'] ?? 'Unbekannter Fehler');
                }
            } else {
            $_SESSION['success_message'] = 'Benutzer erfolgreich erstellt';
            }
            header('Location: users.php');
            exit;
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'update_user') {
            $user_id = $_POST['user_id'];
            $username = $_POST['username'];
            $email = $_POST['email'];
            $role = $_POST['role'];
            $full_name = $_POST['full_name'];
            $department = $_POST['department'];
            $status = $_POST['status'];
            $autohaus_id = $_POST['autohaus_id'] ?: null;
            $can_switch_autohaus = isset($_POST['can_switch_autohaus']) ? 1 : 0;
            $phone = $_POST['phone'] ?? '';
            
            // Prüfe ob phone Spalte existiert
            try {
                $checkPhone = $conn->query("SHOW COLUMNS FROM admin_users LIKE 'phone'")->rowCount();
                if ($checkPhone > 0) {
                    $stmt = $conn->prepare("
                        UPDATE admin_users 
                        SET username = ?, email = ?, role = ?, full_name = ?, department = ?, autohaus_id = ?, can_switch_autohaus = ?, phone = ?, is_active = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $is_active = $status === 'active' ? 1 : 0;
                    $stmt->execute([$username, $email, $role, $full_name, $department, $autohaus_id, $can_switch_autohaus, $phone, $is_active, $user_id]);
                } else {
            $stmt = $conn->prepare("
                UPDATE admin_users 
                SET username = ?, email = ?, role = ?, full_name = ?, department = ?, autohaus_id = ?, can_switch_autohaus = ?, is_active = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $is_active = $status === 'active' ? 1 : 0;
            $stmt->execute([$username, $email, $role, $full_name, $department, $autohaus_id, $can_switch_autohaus, $is_active, $user_id]);
                }
            } catch (Exception $e) {
                // Fallback ohne phone
                $stmt = $conn->prepare("
                    UPDATE admin_users 
                    SET username = ?, email = ?, role = ?, full_name = ?, department = ?, autohaus_id = ?, can_switch_autohaus = ?, is_active = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $is_active = $status === 'active' ? 1 : 0;
                $stmt->execute([$username, $email, $role, $full_name, $department, $autohaus_id, $can_switch_autohaus, $is_active, $user_id]);
            }
            
            // Aktualisiere 2FA-Einstellungen und sende Verifizierungscode wenn Telefonnummer geändert
            require_once 'includes/whatsapp_2fa.php';
            if (!empty($phone)) {
                // Prüfe ob Telefonnummer geändert wurde
                $stmt_check = $conn->prepare("SELECT phone FROM admin_user_2fa WHERE user_id = ?");
                $stmt_check->execute([$user_id]);
                $old_phone = $stmt_check->fetchColumn();
                
                // Sende Code nur wenn Telefonnummer neu oder geändert
                $sendCode = true;
                if ($old_phone && trim($old_phone) === trim($phone)) {
                    $sendCode = false; // Telefonnummer unverändert, kein neuer Code nötig
                }
                
                if ($sendCode) {
                    $verification_result = setWhatsApp2FA($user_id, $phone, true, $autohaus_id);
                    
                    if ($verification_result['success']) {
                        $_SESSION['success_message'] = 'Benutzer erfolgreich aktualisiert. Ein Verifizierungscode wurde per WhatsApp gesendet.';
                        $_SESSION['pending_verification_user_id'] = $user_id;
                    } else {
                        $_SESSION['warning_message'] = 'Benutzer erfolgreich aktualisiert, aber Verifizierungscode konnte nicht gesendet werden: ' . ($verification_result['error'] ?? 'Unbekannter Fehler');
                    }
                } else {
                    $_SESSION['success_message'] = 'Benutzer erfolgreich aktualisiert.';
                }
            } else {
                // Deaktiviere 2FA wenn keine Telefonnummer
                $stmt = $conn->prepare("UPDATE admin_user_2fa SET two_factor_enabled = 0, phone_verified = 0 WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $_SESSION['success_message'] = 'Benutzer erfolgreich aktualisiert.';
            }
            header('Location: users.php');
            exit;
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'delete_user') {
            $user_id = $_POST['user_id'];
            
            $stmt = $conn->prepare("DELETE FROM admin_users WHERE id = ?");
            $stmt->execute([$user_id]);
            
            $_SESSION['success_message'] = 'Benutzer erfolgreich gelöscht';
            header('Location: users.php');
            exit;
        }
    } catch (Exception $e) {
        $_SESSION['error_message'] = 'Fehler: ' . $e->getMessage();
    }
}

// Stelle sicher, dass die Tabelle und Verifizierungs-Spalten existieren
try {
    // Prüfe ob Tabelle existiert
    $tableExists = $conn->query("SHOW TABLES LIKE 'admin_user_2fa'")->rowCount() > 0;
    
    if ($tableExists) {
        $columns = $conn->query("SHOW COLUMNS FROM admin_user_2fa")->fetchAll(PDO::FETCH_COLUMN);
        
        if (!in_array('phone_verified', $columns)) {
            $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verified TINYINT(1) DEFAULT 0");
        }
        if (!in_array('phone_verified_at', $columns)) {
            $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verified_at DATETIME NULL");
        }
        if (!in_array('phone_verification_error', $columns)) {
            $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verification_error VARCHAR(255) NULL");
        }
    } else {
        // Erstelle Tabelle mit allen Spalten
        $conn->exec("
            CREATE TABLE IF NOT EXISTS admin_user_2fa (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL UNIQUE,
                two_factor_secret VARCHAR(32),
                two_factor_method VARCHAR(20) DEFAULT 'whatsapp',
                two_factor_enabled TINYINT(1) DEFAULT 1,
                phone VARCHAR(20) DEFAULT NULL,
                phone_verified TINYINT(1) DEFAULT 0,
                phone_verified_at DATETIME NULL,
                phone_verification_error VARCHAR(255) NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_user_id (user_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
    }
} catch (PDOException $e) {
    error_log("Phone verification setup error: " . $e->getMessage());
}

// Get all admin users with autohaus information, linked customer count and phone verification status
$stmt = $conn->prepare("
    SELECT u.*, 
           a.name as autohaus_name, 
           a.short_name as autohaus_short_name,
           (SELECT COUNT(*) FROM users WHERE admin_user_id = u.id) as linked_customer_count,
           COALESCE(tfa.phone_verified, 0) as phone_verified,
           tfa.phone_verified_at,
           tfa.phone_verification_error
    FROM admin_users u
    LEFT JOIN autohaus a ON u.autohaus_id = a.id
    LEFT JOIN admin_user_2fa tfa ON u.id = tfa.user_id
    ORDER BY u.created_at DESC
");
$stmt->execute();
$admin_users = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get all autohauser for forms
$stmt = $conn->prepare("SELECT * FROM autohaus WHERE is_active = 1 ORDER BY name ASC");
$stmt->execute();
$autohauser = $stmt->fetchAll();

// Get all customer accounts for linking
$stmt = $conn->prepare("SELECT id, email, first_name, last_name, admin_user_id FROM users ORDER BY email ASC");
$stmt->execute();
$all_customers = $stmt->fetchAll();

?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Benutzerverwaltung - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="admin-body">
    <div class="admin-container">
        <!-- Include Unified Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <main class="admin-main">
            <!-- Header -->
            <header class="admin-header">
                <div class="header-content">
                    <h1><i class="fas fa-users"></i> Benutzerverwaltung</h1>
                </div>
            </header>

            <!-- Content -->
            <div class="admin-content">
                <!-- Action Bar mit Suche und Filter -->
                <div class="action-bar">
                    <div class="action-bar-top">
                        <div class="action-bar-left">
                            <h2><i class="fas fa-users"></i> Benutzerverwaltung</h2>
                            <span class="user-count"><?php echo count($admin_users); ?> Benutzer insgesamt</span>
                        </div>
                        <div class="action-bar-right">
                            <button class="btn btn-primary btn-large" onclick="openCreateUserPanel()">
                                <i class="fas fa-plus-circle"></i>
                                <span>Neuer Benutzer</span>
                            </button>
                        </div>
                    </div>
                    <div class="action-bar-bottom">
                        <div class="search-filter-container">
                            <div class="search-box">
                                <i class="fas fa-search"></i>
                                <input type="text" id="searchInput" placeholder="Benutzer suchen..." onkeyup="filterUsers()">
                            </div>
                            <div class="filter-group">
                                <select id="roleFilter" onchange="filterUsers()">
                                    <option value="">Alle Rollen</option>
                                    <option value="admin">Administrator</option>
                                    <option value="manager">Manager</option>
                                    <option value="employee">Mitarbeiter</option>
                                </select>
                                <select id="statusFilter" onchange="filterUsers()">
                                    <option value="">Alle Status</option>
                                    <option value="active">Aktiv</option>
                                    <option value="inactive">Inaktiv</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <?php if (isset($_SESSION['success_message'])): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <span><?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?></span>
                    </div>
                <?php endif; ?>

                <?php if (isset($_SESSION['error_message'])): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <span><?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?></span>
                    </div>
                <?php endif; ?>

                <?php if (isset($_SESSION['warning_message'])): ?>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i>
                        <span><?php echo $_SESSION['warning_message']; unset($_SESSION['warning_message']); ?></span>
                    </div>
                <?php endif; ?>

                <?php 
                // Prüfe ob Verifizierung ausstehend ist
                $pending_verification_user_id = null;
                if (isset($_SESSION['pending_verification_user_id'])) {
                    $pending_verification_user_id = $_SESSION['pending_verification_user_id'];
                    unset($_SESSION['pending_verification_user_id']);
                }
                ?>

                <!-- Modern Table Layout -->
                <div class="users-table-container">
                    <div class="table-wrapper">
                        <table class="users-table" id="usersTable">
                            <thead>
                                <tr>
                                    <th class="col-user">
                                        <i class="fas fa-user"></i> Benutzer
                                    </th>
                                    <th class="col-role">
                                        <i class="fas fa-user-tag"></i> Rolle
                                    </th>
                                    <th class="col-autohaus">
                                        <i class="fas fa-building"></i> Autohaus
                                    </th>
                                    <th class="col-department">
                                        <i class="fas fa-briefcase"></i> Abteilung
                                    </th>
                                    <th class="col-customers">
                                        <i class="fas fa-users"></i> Kunden
                                    </th>
                                    <th class="col-status">
                                        <i class="fas fa-toggle-on"></i> Status
                                    </th>
                                    <th class="col-verification">
                                        <i class="fas fa-shield-alt"></i> Verifizierung
                                    </th>
                                    <th class="col-actions">
                                        <i class="fas fa-cog"></i> Aktionen
                                    </th>
                                </tr>
                            </thead>
                            <tbody>
                    <?php foreach ($admin_users as $user): ?>
                    <?php
                    // Lade verknüpfte Kunden für diesen Admin
                    $stmt = $conn->prepare("
                        SELECT id, email, first_name, last_name 
                        FROM users 
                        WHERE admin_user_id = ? 
                        ORDER BY created_at DESC
                    ");
                    $stmt->execute([$user['id']]);
                    $linked_customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    ?>
                                <tr class="user-row" 
                                    data-username="<?php echo strtolower(htmlspecialchars($user['username'])); ?>"
                                    data-name="<?php echo strtolower(htmlspecialchars($user['full_name'])); ?>"
                                    data-email="<?php echo strtolower(htmlspecialchars($user['email'])); ?>"
                                    data-role="<?php echo $user['role']; ?>"
                                    data-status="<?php echo $user['is_active'] ? 'active' : 'inactive'; ?>"
                                    onclick="expandRow(this)">
                                    <td class="col-user">
                                        <div class="user-cell">
                                            <div class="user-avatar">
                                                <span class="avatar-initials"><?php echo strtoupper(substr($user['full_name'], 0, 2)); ?></span>
                                </div>
                                            <div class="user-info">
                                    <div class="user-name"><?php echo htmlspecialchars($user['full_name']); ?></div>
                                                <div class="user-meta">
                                                    <span class="user-username">@<?php echo htmlspecialchars($user['username']); ?></span>
                                                    <span class="user-separator">•</span>
                                                    <span class="user-email"><?php echo htmlspecialchars($user['email']); ?></span>
                                </div>
                            </div>
                            </div>
                                    </td>
                                    <td class="col-role">
                                        <span class="role-badge role-<?php echo $user['role']; ?>">
                                            <?php 
                                            $roleNames = ['admin' => 'Admin', 'manager' => 'Manager', 'employee' => 'Mitarbeiter'];
                                            echo $roleNames[$user['role']] ?? ucfirst($user['role']); 
                                            ?>
                                    </span>
                                    </td>
                                    <td class="col-autohaus">
                            <?php if ($user['autohaus_name']): ?>
                                            <span class="autohaus-name">
                                    <i class="fas fa-building"></i>
                                                <?php echo htmlspecialchars($user['autohaus_name']); ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-muted">—</span>
                            <?php endif; ?>
                                    </td>
                                    <td class="col-department">
                                        <?php if ($user['department']): ?>
                                            <span class="department-name"><?php echo htmlspecialchars($user['department']); ?></span>
                                        <?php else: ?>
                                            <span class="text-muted">—</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="col-customers">
                            <?php if ($user['linked_customer_count'] > 0): ?>
                                            <span class="customer-count highlight">
                                    <i class="fas fa-users"></i>
                                                <?php echo $user['linked_customer_count']; ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-muted">0</span>
                            <?php endif; ?>
                                    </td>
                                    <td class="col-status">
                                        <span class="status-indicator status-<?php echo $user['is_active'] ? 'active' : 'inactive'; ?>">
                                            <span class="status-dot"></span>
                                            <?php echo $user['is_active'] ? 'Aktiv' : 'Inaktiv'; ?>
                            </span>
                                    </td>
                                    <td class="col-verification" onclick="event.stopPropagation()">
                                        <?php if (!empty($user['phone'])): ?>
                                            <?php if ($user['phone_verified'] == 1): ?>
                                                <span class="verification-badge verified" title="Verifiziert am <?php echo $user['phone_verified_at'] ? date('d.m.Y H:i', strtotime($user['phone_verified_at'])) : ''; ?>">
                                                    <i class="fas fa-check-circle"></i>
                                                    Verifiziert
                                                </span>
                                            <?php else: ?>
                                                <div class="verification-container">
                                                    <span class="verification-badge not-verified" title="<?php echo htmlspecialchars($user['phone_verification_error'] ?? 'Nicht verifiziert'); ?>">
                                                        <i class="fas fa-times-circle"></i>
                                                        Nicht verifiziert
                                                    </span>
                                                    <button class="verify-btn" onclick="verifyPhoneNumber(<?php echo $user['id']; ?>)" title="Telefonnummer verifizieren">
                                                        <i class="fas fa-sync-alt"></i>
                                                    </button>
                        </div>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="verification-badge no-phone">
                                                <i class="fas fa-minus-circle"></i>
                                                Keine Nummer
                            </span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="col-actions" onclick="event.stopPropagation()">
                                        <div class="action-buttons">
                                            <button class="action-btn edit-btn" onclick="editUser(<?php echo $user['id']; ?>)" title="Bearbeiten">
                                    <i class="fas fa-edit"></i>
                                </button>
                                            <button class="action-btn link-btn" onclick="showLinkPanel(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['full_name']); ?>')" title="Kunden verknüpfen">
                                    <i class="fas fa-link"></i>
                                </button>
                                            <button class="action-btn delete-btn" onclick="deleteUser(<?php echo $user['id']; ?>)" title="Löschen">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                                    </td>
                                </tr>
                                <tr class="user-details-row" style="display: none;">
                                    <td colspan="8">
                                        <div class="user-details-content">
                                            <div class="details-grid">
                                                <div class="detail-section">
                                                    <h4><i class="fas fa-info-circle"></i> Informationen</h4>
                                                    <div class="detail-item">
                                                        <span class="detail-label">Erstellt am:</span>
                                                        <span class="detail-value"><?php echo date('d.m.Y H:i', strtotime($user['created_at'])); ?></span>
                        </div>
                                                    <?php if ($user['updated_at']): ?>
                                                    <div class="detail-item">
                                                        <span class="detail-label">Aktualisiert am:</span>
                                                        <span class="detail-value"><?php echo date('d.m.Y H:i', strtotime($user['updated_at'])); ?></span>
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if ($user['can_switch_autohaus']): ?>
                                                    <div class="detail-item">
                                                        <span class="detail-label">Berechtigung:</span>
                                                        <span class="detail-value"><i class="fas fa-check-circle text-success"></i> Kann zwischen Autohäusern wechseln</span>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                                <?php if ($user['linked_customer_count'] > 0): ?>
                                                <div class="detail-section">
                                                    <h4><i class="fas fa-users"></i> Verknüpfte Kunden (<?php echo $user['linked_customer_count']; ?>)</h4>
                                                    <div class="linked-customers-list">
                                                        <?php foreach (array_slice($linked_customers, 0, 5) as $customer): ?>
                                                        <div class="linked-customer-item">
                                                            <i class="fas fa-user-circle"></i>
                                                            <span><?php echo htmlspecialchars($customer['first_name'] . ' ' . $customer['last_name']); ?></span>
                                                            <span class="customer-email"><?php echo htmlspecialchars($customer['email']); ?></span>
                    </div>
                    <?php endforeach; ?>
                                                        <?php if (count($linked_customers) > 5): ?>
                                                        <div class="more-customers">
                                                            <i class="fas fa-ellipsis-h"></i>
                                                            <?php echo count($linked_customers) - 5; ?> weitere Kunden
                                                        </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Slide-in Panel Overlay -->
    <div id="panelOverlay" class="panel-overlay" onclick="closeAllPanels()"></div>

    <!-- Create User Panel -->
    <div id="createUserPanel" class="slide-panel">
        <div class="modal-content modal-large">
            <div class="panel-header">
                <div class="panel-header-content">
                    <h3><i class="fas fa-user-plus"></i> Neuen Benutzer erstellen</h3>
                    <p class="panel-subtitle">Füllen Sie die folgenden Felder aus, um einen neuen Benutzer anzulegen</p>
            </div>
                <button class="panel-close" onclick="closeCreateUserPanel()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form method="POST" class="panel-form" id="createUserForm">
                <input type="hidden" name="action" value="create_user">
                
                <!-- Form Steps Indicator -->
                <div class="form-steps">
                    <div class="step active" data-step="1">
                        <div class="step-number">1</div>
                        <div class="step-label">Grunddaten</div>
                    </div>
                    <div class="step" data-step="2">
                        <div class="step-number">2</div>
                        <div class="step-label">Zugriff & Rechte</div>
                    </div>
                    <div class="step" data-step="3">
                        <div class="step-number">3</div>
                        <div class="step-label">Sicherheit</div>
                    </div>
                </div>
                
                <!-- Step 1: Grunddaten -->
                <div class="form-step-content active" data-step="1">
                    <div class="form-section">
                        <h4 class="section-title">
                            <i class="fas fa-user"></i>
                            Persönliche Informationen
                        </h4>
                <div class="form-grid">
                    <div class="form-group">
                                <label for="full_name">
                                    <i class="fas fa-id-card"></i>
                                    Vollständiger Name <span class="required">*</span>
                                </label>
                                <input type="text" id="full_name" name="full_name" required placeholder="Max Mustermann">
                    </div>
                    <div class="form-group">
                                <label for="username">
                                    <i class="fas fa-at"></i>
                                    Benutzername <span class="required">*</span>
                                </label>
                                <input type="text" id="username" name="username" required placeholder="max.mustermann">
                    </div>
                    <div class="form-group">
                                <label for="email">
                                    <i class="fas fa-envelope"></i>
                                    E-Mail-Adresse <span class="required">*</span>
                                </label>
                                <input type="email" id="email" name="email" required placeholder="max.mustermann@example.com">
                    </div>
                    <div class="form-group">
                                <label for="department">
                                    <i class="fas fa-briefcase"></i>
                                    Abteilung
                                </label>
                                <input type="text" id="department" name="department" placeholder="z.B. Verkauf, Service">
                    </div>
                        </div>
                    </div>
                </div>
                
                <!-- Step 2: Zugriff & Rechte -->
                <div class="form-step-content" data-step="2">
                    <div class="form-section">
                        <h4 class="section-title">
                            <i class="fas fa-shield-alt"></i>
                            Zugriffsrechte & Zugehörigkeit
                        </h4>
                        <div class="form-grid">
                    <div class="form-group">
                                <label for="role">
                                    <i class="fas fa-user-tag"></i>
                                    Rolle <span class="required">*</span>
                                </label>
                        <select id="role" name="role" required>
                                    <option value="">-- Rolle auswählen --</option>
                            <option value="admin">Administrator</option>
                            <option value="manager">Manager</option>
                            <option value="employee">Mitarbeiter</option>
                        </select>
                                <small class="form-hint">Bestimmt die Zugriffsrechte des Benutzers</small>
                    </div>
                    <div class="form-group">
                                <label for="autohaus_id">
                                    <i class="fas fa-building"></i>
                                    Autohaus
                                </label>
                        <select id="autohaus_id" name="autohaus_id">
                            <option value="">Kein Autohaus zugeordnet</option>
                            <?php foreach ($autohauser as $autohaus): ?>
                                <option value="<?php echo $autohaus['id']; ?>">
                                    <?php echo htmlspecialchars($autohaus['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group full-width">
                        <label class="checkbox-label">
                                    <input type="checkbox" name="can_switch_autohaus" value="1" id="can_switch_autohaus">
                                    <span>
                                        <i class="fas fa-exchange-alt"></i>
                                        Kann zwischen Autohäusern wechseln (nur für Administratoren)
                                    </span>
                        </label>
                    </div>
                </div>
                    </div>
                </div>
                
                <!-- Step 3: Sicherheit -->
                <div class="form-step-content" data-step="3">
                    <div class="form-section">
                        <h4 class="section-title">
                            <i class="fas fa-lock"></i>
                            Sicherheit & Authentifizierung
                        </h4>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="password">
                                    <i class="fas fa-key"></i>
                                    Passwort <span class="required">*</span>
                                </label>
                                <input type="password" id="password" name="password" required minlength="8" placeholder="Mindestens 8 Zeichen">
                                <small class="form-hint">Mindestens 8 Zeichen</small>
                            </div>
                            <div class="form-group">
                                <label for="phone">
                                    <i class="fas fa-mobile-alt"></i>
                                    Mobilnummer (für 2FA) <span class="required">*</span>
                                </label>
                                <input type="tel" id="phone" name="phone" required 
                                       placeholder="+49123456789 oder 0123456789"
                                       pattern="[\+]?[0-9\s\-\(\)]+">
                                <small class="form-hint">
                                    <i class="fas fa-info-circle"></i>
                                    2FA per WhatsApp ist für Admin-Benutzer erforderlich
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="panel-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeCreateUserPanel()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button type="button" class="btn btn-secondary" id="prevStepBtn" onclick="prevStep()" style="display: none;">
                        <i class="fas fa-arrow-left"></i> Zurück
                    </button>
                    <button type="button" class="btn btn-primary" id="nextStepBtn" onclick="nextStep()">
                        Weiter <i class="fas fa-arrow-right"></i>
                    </button>
                    <button type="submit" class="btn btn-primary" id="submitBtn" style="display: none;">
                        <i class="fas fa-check"></i> Benutzer erstellen
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Edit User Panel -->
    <div id="editUserPanel" class="slide-panel">
        <div class="modal-content modal-large">
            <div class="panel-header">
                <div class="panel-header-content">
                    <h3><i class="fas fa-user-edit"></i> Benutzer bearbeiten</h3>
                    <p class="panel-subtitle">Aktualisieren Sie die Benutzerinformationen</p>
            </div>
                <button class="panel-close" onclick="closeEditUserPanel()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form method="POST" class="panel-form" id="editUserForm">
                <input type="hidden" name="action" value="update_user">
                <input type="hidden" name="user_id" id="edit_user_id">
                
                <div class="form-sections">
                    <div class="form-section">
                        <h4 class="section-title">
                            <i class="fas fa-user"></i>
                            Persönliche Informationen
                        </h4>
                <div class="form-grid">
                    <div class="form-group">
                                <label for="edit_full_name">
                                    <i class="fas fa-id-card"></i>
                                    Vollständiger Name <span class="required">*</span>
                                </label>
                                <input type="text" id="edit_full_name" name="full_name" required>
                            </div>
                            <div class="form-group">
                                <label for="edit_username">
                                    <i class="fas fa-at"></i>
                                    Benutzername <span class="required">*</span>
                                </label>
                        <input type="text" id="edit_username" name="username" required>
                    </div>
                    <div class="form-group">
                                <label for="edit_email">
                                    <i class="fas fa-envelope"></i>
                                    E-Mail-Adresse <span class="required">*</span>
                                </label>
                        <input type="email" id="edit_email" name="email" required>
                    </div>
                    <div class="form-group">
                                <label for="edit_department">
                                    <i class="fas fa-briefcase"></i>
                                    Abteilung
                                </label>
                                <input type="text" id="edit_department" name="department">
                    </div>
                        </div>
                    </div>
                    
                    <div class="form-section">
                        <h4 class="section-title">
                            <i class="fas fa-shield-alt"></i>
                            Zugriffsrechte & Status
                        </h4>
                        <div class="form-grid">
                    <div class="form-group">
                                <label for="edit_role">
                                    <i class="fas fa-user-tag"></i>
                                    Rolle <span class="required">*</span>
                                </label>
                        <select id="edit_role" name="role" required>
                            <option value="admin">Administrator</option>
                            <option value="manager">Manager</option>
                            <option value="employee">Mitarbeiter</option>
                        </select>
                    </div>
                    <div class="form-group">
                                <label for="edit_status">
                                    <i class="fas fa-toggle-on"></i>
                                    Status <span class="required">*</span>
                                </label>
                                <select id="edit_status" name="status" required>
                                    <option value="active">Aktiv</option>
                                    <option value="inactive">Inaktiv</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="edit_autohaus_id">
                                    <i class="fas fa-building"></i>
                                    Autohaus
                                </label>
                        <select id="edit_autohaus_id" name="autohaus_id">
                            <option value="">Kein Autohaus zugeordnet</option>
                            <?php foreach ($autohauser as $autohaus): ?>
                                <option value="<?php echo $autohaus['id']; ?>">
                                    <?php echo htmlspecialchars($autohaus['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group full-width">
                        <label class="checkbox-label">
                            <input type="checkbox" name="can_switch_autohaus" id="edit_can_switch_autohaus" value="1">
                                    <span>
                                        <i class="fas fa-exchange-alt"></i>
                                        Kann zwischen Autohäusern wechseln (nur für Administratoren)
                                    </span>
                        </label>
                    </div>
                </div>
                    </div>
                    
                    <div class="form-section">
                        <h4 class="section-title">
                            <i class="fas fa-lock"></i>
                            Sicherheit
                        </h4>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="edit_phone">
                                    <i class="fas fa-mobile-alt"></i>
                                    Mobilnummer (für 2FA) <span class="required">*</span>
                                </label>
                                <input type="tel" id="edit_phone" name="phone" required 
                                       placeholder="+49123456789 oder 0123456789"
                                       pattern="[\+]?[0-9\s\-\(\)]+">
                                <small class="form-hint">
                                    <i class="fas fa-info-circle"></i>
                                    2FA per WhatsApp ist für Admin-Benutzer erforderlich
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="panel-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeEditUserPanel()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Änderungen speichern
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Link Customer Panel -->
    <div id="linkCustomerPanel" class="slide-panel">

    <!-- Verification Code Modal -->
    <div id="verificationCodeModal" class="modern-modal">
        <div class="modal-overlay" onclick="closeVerificationCodeModal()"></div>
        <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                    <div class="modal-icon confirm-icon">
                        <i class="fas fa-shield-alt"></i>
            </div>
                    <h3>Telefonnummer verifizieren</h3>
                </div>
                <div class="modal-body">
                    <p style="margin-bottom: var(--spacing-lg);">
                        Ein 6-stelliger Verifizierungscode wurde per WhatsApp gesendet. 
                        Bitte geben Sie den Code ein, um die Telefonnummer zu verifizieren.
                    </p>
                    <input type="hidden" id="verification_user_id">
                    <div class="form-group" style="margin-bottom: var(--spacing-md);">
                        <label for="verification_code" style="display: block; margin-bottom: var(--spacing-sm); font-weight: var(--font-weight-semibold);">
                            <i class="fas fa-key"></i> Verifizierungscode
                        </label>
                        <input 
                            type="text" 
                            id="verification_code" 
                            maxlength="6" 
                            pattern="[0-9]{6}" 
                            placeholder="000000"
                            style="width: 100%; padding: var(--spacing-md); font-size: var(--font-size-xl); text-align: center; letter-spacing: 0.5em; border: 2px solid var(--border-color); border-radius: var(--border-radius); font-family: monospace;"
                            onkeypress="return /[0-9]/i.test(event.key)"
                            oninput="this.value = this.value.replace(/[^0-9]/g, '')"
                            onkeydown="if(event.key === 'Enter') submitVerificationCode()"
                        >
                        <div id="verification_error" class="error-message" style="display: none; margin-top: var(--spacing-sm); color: var(--color-error); font-size: var(--font-size-sm);"></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-secondary" onclick="closeVerificationCodeModal()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button class="btn btn-primary" id="verify_code_btn" onclick="submitVerificationCode()">
                        <i class="fas fa-check"></i> Verifizieren
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Alert Modal -->
    <div id="alertModal" class="modern-modal">
        <div class="modal-overlay" onclick="closeAlertModal()"></div>
        <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                    <div class="modal-icon alert-icon">
                        <i class="fas fa-exclamation-circle"></i>
            </div>
                    <h3 id="alertTitle">Hinweis</h3>
                </div>
                <div class="modal-body">
                    <p id="alertMessage"></p>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" onclick="closeAlertModal()">
                        <i class="fas fa-check"></i> OK
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Confirm Modal -->
    <div id="confirmModal" class="modern-modal">
        <div class="modal-overlay" onclick="closeConfirmModal()"></div>
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <div class="modal-icon confirm-icon">
                        <i class="fas fa-question-circle"></i>
                    </div>
                    <h3 id="confirmTitle">Bestätigung erforderlich</h3>
                </div>
                <div class="modal-body">
                    <p id="confirmMessage"></p>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-secondary" onclick="closeConfirmModal()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button class="btn btn-danger" id="confirmButton" onclick="confirmAction()">
                        <i class="fas fa-check"></i> Bestätigen
                    </button>
                </div>
            </div>
        </div>
    </div>
        <div class="modal-content">
            <div class="panel-header">
                <div class="panel-header-content">
                    <h3><i class="fas fa-link"></i> Kundenkonto verknüpfen</h3>
                    <p class="panel-subtitle" id="linkPanelSubtitle">Wählen Sie ein Kundenkonto aus, das mit diesem Admin verknüpft werden soll</p>
                </div>
                <button class="panel-close" onclick="closeLinkCustomerPanel()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form method="POST" class="panel-form">
                <input type="hidden" name="link_customer_to_admin" value="1">
                <input type="hidden" name="admin_user_id" id="link_admin_user_id">
                <div class="form-group">
                    <label for="link_customer_user_id">Kundenkonto auswählen</label>
                    <select id="link_customer_user_id" name="customer_user_id" required>
                        <option value="">-- Kundenkonto auswählen --</option>
                        <?php foreach ($all_customers as $customer): ?>
                            <?php 
                            // Zeige nur wenn nicht bereits verknüpft oder wenn es bereits mit diesem Admin verknüpft ist
                            if (!$customer['admin_user_id'] || $customer['admin_user_id'] == '0'):
                            ?>
                                <option value="<?php echo $customer['id']; ?>">
                                    <?php echo htmlspecialchars($customer['first_name'] . ' ' . $customer['last_name'] . ' (' . $customer['email'] . ')'); ?>
                                </option>
                            <?php 
                            endif;
                            endforeach; 
                            ?>
                    </select>
                    <small style="margin-top: 0.5rem; display: block; color: #6c757d;">
                        Nur nicht verknüpfte Konten werden angezeigt
                    </small>
                </div>
                <div class="panel-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeLinkCustomerPanel()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-link"></i> Verknüpfen
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <style>
        /* Volvo-Apple Hybrid Design System */
        :root {
            /* Volvo Brand Colors */
            --volvo-blue: #0066CC;
            --volvo-blue-dark: #004C99;
            --volvo-blue-light: #E6F2FF;
            --volvo-gray: #2C3E50;
            --volvo-gray-light: #ECF0F1;
            --volvo-accent: #F39C12;
            
            /* Apple-inspired Neutrals */
            --apple-white: #FFFFFF;
            --apple-gray-1: #F2F2F7;
            --apple-gray-2: #E5E5EA;
            --apple-gray-3: #C7C7CC;
            --apple-gray-4: #8E8E93;
            --apple-gray-5: #636366;
            --apple-gray-6: #48484A;
            --apple-black: #1C1C1E;
            
            /* Design System */
            --color-primary: var(--volvo-blue);
            --color-primary-hover: var(--volvo-blue-dark);
            --color-primary-light: var(--volvo-blue-light);
            --color-secondary: var(--apple-gray-5);
            --color-accent: var(--volvo-accent);
            --color-success: #34C759;
            --color-warning: #FF9500;
            --color-error: #FF3B30;
            
            /* Surfaces */
            --surface-primary: var(--apple-white);
            --surface-secondary: var(--apple-gray-1);
            --surface-tertiary: var(--apple-gray-2);
            --surface-elevated: var(--apple-white);
            
            /* Text */
            --text-primary: var(--apple-black);
            --text-secondary: var(--apple-gray-5);
            --text-tertiary: var(--apple-gray-4);
            
            /* Borders & Radii */
            --border-color: var(--apple-gray-3);
            --border-radius: 12px;
            --border-radius-large: 20px;
            
            /* Spacing (Apple's 8pt grid system) */
            --spacing-xs: 4px;
            --spacing-sm: 8px;
            --spacing-md: 16px;
            --spacing-lg: 24px;
            --spacing-xl: 32px;
            --spacing-xxl: 48px;
            
            /* Typography (SF Pro Display inspired) */
            --font-family: 'SF Pro Display', 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            --font-size-xs: 12px;
            --font-size-sm: 14px;
            --font-size-md: 16px;
            --font-size-lg: 18px;
            --font-size-xl: 24px;
            --font-size-xxl: 32px;
            --font-weight-regular: 400;
            --font-weight-medium: 500;
            --font-weight-semibold: 600;
            --font-weight-bold: 700;
            --font-weight-heavy: 800;
            
            /* Shadows (Subtle Apple-style) */
            --shadow-sm: 0 1px 3px rgba(0, 0, 0, 0.04), 0 1px 2px rgba(0, 0, 0, 0.06);
            --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.04), 0 2px 4px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.04), 0 4px 6px rgba(0, 0, 0, 0.05);
            --shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.04), 0 10px 10px rgba(0, 0, 0, 0.04);
            
            /* Transitions (Apple's ease curves) */
            --transition-fast: 0.15s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-normal: 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-slow: 0.5s cubic-bezier(0.4, 0, 0.2, 1);
        }

        /* Dark Theme Variables */
        [data-theme="dark"] {
            --surface-primary: #1C1C1E;
            --surface-secondary: #2C2C2E;
            --surface-tertiary: #3A3A3C;
            --surface-elevated: #2C2C2E;
            --text-primary: #FFFFFF;
            --text-secondary: #8E8E93;
            --text-tertiary: #636366;
            --border-color: #48484A;
            --color-primary: #0A84FF;
            --color-primary-hover: #0051D5;
            --color-primary-light: rgba(10, 132, 255, 0.1);
        }

        /* Global Reset & Base Styles */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: var(--font-family);
            background: var(--surface-secondary);
            color: var(--text-primary);
            line-height: 1.6;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        /* Layout */
        .admin-main {
            margin-left: 280px;
            transition: margin-left var(--transition-normal);
            min-height: 100vh;
            background: var(--surface-secondary);
        }

        .modern-sidebar.collapsed + .admin-main {
            margin-left: 80px;
        }

        /* Header */
        .admin-header {
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color);
            padding: var(--spacing-xl);
            width: 100%;
        }

        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            width: 100%;
            max-width: none;
            margin: 0;
            padding: 0 var(--spacing-xl);
        }

        .admin-header h1 {
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-bold);
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }

        .admin-header h1 i {
            color: var(--color-primary);
        }

        .header-actions {
            display: flex;
            gap: var(--spacing-md);
        }

        /* Content */
        .admin-content {
            padding: var(--spacing-xl);
            width: 100%;
            max-width: none;
            margin: 0;
        }

        /* Action Bar */
        .action-bar {
            background: var(--surface-primary);
            border-radius: var(--border-radius-large);
            border: 1px solid var(--border-color);
            box-shadow: var(--shadow-sm);
            margin-bottom: var(--spacing-xl);
            transition: all var(--transition-normal);
            overflow: hidden;
            width: 100%;
        }

        .action-bar:hover {
            box-shadow: var(--shadow-md);
        }

        .action-bar-top {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-xl);
            border-bottom: 1px solid var(--border-color);
        }

        .action-bar-bottom {
            padding: var(--spacing-lg) var(--spacing-xl);
            background: var(--surface-secondary);
        }

        .action-bar-left {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
        }

        .action-bar-left h2 {
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-bold);
            color: var(--text-primary);
            margin: 0;
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
        }

        .user-count {
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
            font-weight: var(--font-weight-medium);
        }

        .action-bar-right {
            display: flex;
            gap: var(--spacing-md);
        }

        /* Search & Filter */
        .search-filter-container {
            display: flex;
            gap: var(--spacing-md);
            align-items: center;
            flex-wrap: wrap;
        }

        .search-box {
            position: relative;
            flex: 1;
            min-width: 250px;
        }

        .search-box i {
            position: absolute;
            left: var(--spacing-md);
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-tertiary);
        }

        .search-box input {
            width: 100%;
            padding: var(--spacing-md) var(--spacing-md) var(--spacing-md) 40px;
            border: 2px solid var(--border-color);
            border-radius: var(--border-radius);
            font-family: var(--font-family);
            font-size: var(--font-size-md);
            background: var(--surface-primary);
            color: var(--text-primary);
            transition: all var(--transition-normal);
        }

        .search-box input:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px var(--color-primary-light);
        }

        .filter-group {
            display: flex;
            gap: var(--spacing-sm);
        }

        .filter-group select {
            padding: var(--spacing-md);
            border: 2px solid var(--border-color);
            border-radius: var(--border-radius);
            font-family: var(--font-family);
            font-size: var(--font-size-sm);
            background: var(--surface-primary);
            color: var(--text-primary);
            cursor: pointer;
            transition: all var(--transition-normal);
        }

        .filter-group select:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px var(--color-primary-light);
        }

        .btn-large {
            padding: var(--spacing-md) var(--spacing-xl);
            font-size: var(--font-size-md);
            font-weight: var(--font-weight-semibold);
            border-radius: var(--border-radius);
            box-shadow: var(--shadow-md);
        }

        .btn-large:hover {
            transform: translateY(-3px);
            box-shadow: var(--shadow-lg);
        }

        /* Alert System */
        .alert {
            display: flex;
            align-items: flex-start;
            gap: var(--spacing-md);
            padding: var(--spacing-md) var(--spacing-lg);
            margin-bottom: var(--spacing-lg);
            border-radius: var(--border-radius);
            border: 1px solid;
            position: relative;
            backdrop-filter: blur(10px);
        }

        .alert-success {
            background: rgba(52, 199, 89, 0.1);
            border-color: var(--color-success);
            color: var(--color-success);
        }

        .alert-error {
            background: rgba(255, 59, 48, 0.1);
            border-color: var(--color-error);
            color: var(--color-error);
        }

        .alert-warning {
            background: rgba(255, 149, 0, 0.1);
            border-color: var(--color-warning);
            color: var(--color-warning);
        }

        .alert-icon {
            flex-shrink: 0;
            margin-top: 2px;
        }

        .alert-content {
            flex: 1;
        }

        .alert-content p {
            margin: 0;
            font-weight: var(--font-weight-medium);
        }

        /* Users Table Container */
        .users-table-container {
            background: var(--surface-primary);
            border-radius: var(--border-radius-large);
            border: 1px solid var(--border-color);
            box-shadow: var(--shadow-sm);
            overflow: hidden;
            margin-top: var(--spacing-lg);
            width: 100%;
        }

        .table-wrapper {
            overflow-x: auto;
            width: 100%;
        }

        .users-table {
            width: 100%;
            border-collapse: collapse;
            background: var(--surface-primary);
            table-layout: auto;
        }

        .users-table th,
        .users-table td {
            padding: var(--spacing-lg) var(--spacing-md);
        }

        .users-table .col-user {
            min-width: 280px;
            width: 25%;
        }

        .users-table .col-role {
            min-width: 120px;
            width: 10%;
        }

        .users-table .col-autohaus {
            min-width: 180px;
            width: 15%;
        }

        .users-table .col-department {
            min-width: 150px;
            width: 12%;
        }

        .users-table .col-customers {
            min-width: 100px;
            width: 8%;
            text-align: center;
        }

        .users-table .col-status {
            min-width: 120px;
            width: 10%;
        }

        .users-table .col-verification {
            min-width: 150px;
            width: 12%;
            text-align: center;
        }

        .users-table .col-actions {
            min-width: 140px;
            width: 10%;
            text-align: center;
        }

        .verification-badge {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-semibold);
            white-space: nowrap;
        }

        .verification-badge.verified {
            background: rgba(52, 199, 89, 0.1);
            color: var(--color-success);
            border: 1px solid var(--color-success);
        }

        .verification-badge.not-verified {
            background: rgba(255, 59, 48, 0.1);
            color: var(--color-error);
            border: 1px solid var(--color-error);
        }

        .verification-badge.no-phone {
            background: var(--surface-secondary);
            color: var(--text-tertiary);
            border: 1px solid var(--border-color);
        }

        .verification-container {
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
            justify-content: center;
        }

        .verify-btn {
            width: 28px;
            height: 28px;
            border: none;
            border-radius: var(--border-radius);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all var(--transition-fast);
            color: var(--color-primary);
            background: var(--color-primary-light);
            border: 1px solid var(--color-primary);
            font-size: var(--font-size-xs);
        }

        .verify-btn:hover {
            background: var(--color-primary);
            color: white;
            transform: rotate(180deg);
        }

        .verify-btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none;
        }

        .users-table thead {
            background: var(--surface-secondary);
            border-bottom: 2px solid var(--border-color);
        }

        .users-table th {
            padding: var(--spacing-lg) var(--spacing-md);
            text-align: left;
            font-weight: var(--font-weight-semibold);
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.05em;
            white-space: nowrap;
        }

        .users-table th.col-customers,
        .users-table th.col-actions {
            text-align: center;
        }

        .users-table th i {
            margin-right: var(--spacing-xs);
            color: var(--color-primary);
        }

        .users-table tbody tr {
            border-bottom: 1px solid var(--border-color);
            transition: all var(--transition-fast);
            cursor: pointer;
        }

        .users-table tbody tr:hover {
            background: var(--surface-secondary);
        }

        .users-table tbody tr.expanded {
            background: var(--surface-secondary);
        }

        .users-table tbody tr.expanded + .user-details-row {
            display: table-row;
        }

        .users-table td {
            padding: var(--spacing-lg) var(--spacing-md);
            vertical-align: middle;
        }

        .users-table td.col-customers,
        .users-table td.col-actions {
            text-align: center;
        }

        .user-cell {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }

        .user-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--color-primary), var(--color-primary-hover));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: var(--font-weight-bold);
            font-size: var(--font-size-sm);
            flex-shrink: 0;
            box-shadow: var(--shadow-sm);
        }

        .user-info {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
            min-width: 0;
        }

        .user-name {
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            font-size: var(--font-size-md);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .user-meta {
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
            font-size: var(--font-size-xs);
            color: var(--text-tertiary);
            flex-wrap: wrap;
        }

        .user-separator {
            color: var(--text-tertiary);
        }

        .user-email {
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 100%;
        }

        .role-badge {
            display: inline-block;
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-semibold);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .autohaus-name {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            color: var(--text-primary);
            font-size: var(--font-size-sm);
        }

        .department-name {
            color: var(--text-primary);
            font-size: var(--font-size-sm);
        }

        .customer-count {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            padding: var(--spacing-xs) var(--spacing-sm);
            background: var(--color-primary-light);
            color: var(--color-primary);
            border-radius: var(--border-radius);
            font-weight: var(--font-weight-semibold);
            font-size: var(--font-size-sm);
        }

        .customer-count.highlight {
            background: var(--color-primary);
            color: white;
        }

        .status-indicator {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            font-size: var(--font-size-sm);
            font-weight: var(--font-weight-medium);
        }

        .status-dot {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            display: inline-block;
        }

        .status-indicator.status-active .status-dot {
            background: var(--color-success);
            box-shadow: 0 0 0 2px rgba(52, 199, 89, 0.2);
        }

        .status-indicator.status-inactive .status-dot {
            background: var(--color-error);
            box-shadow: 0 0 0 2px rgba(255, 59, 48, 0.2);
        }

        .action-buttons {
            display: flex;
            gap: var(--spacing-xs);
            justify-content: center;
        }

        .action-btn {
            width: 32px;
            height: 32px;
            border: none;
            border-radius: var(--border-radius);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all var(--transition-fast);
            color: var(--text-secondary);
            background: var(--surface-secondary);
            border: 1px solid var(--border-color);
            font-size: var(--font-size-xs);
        }

        .action-btn:hover {
            transform: translateY(-1px);
            box-shadow: var(--shadow-sm);
        }

        .action-btn.edit-btn:hover {
            background: var(--color-primary);
            color: white;
            border-color: var(--color-primary);
        }

        .action-btn.link-btn:hover {
            background: var(--color-success);
            color: white;
            border-color: var(--color-success);
        }

        .action-btn.delete-btn:hover {
            background: var(--color-error);
            color: white;
            border-color: var(--color-error);
        }

        .user-details-row {
            background: var(--surface-secondary);
        }

        .user-details-content {
            padding: var(--spacing-xl);
        }

        .details-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: var(--spacing-xl);
        }

        .detail-section {
            background: var(--surface-primary);
            padding: var(--spacing-lg);
            border-radius: var(--border-radius);
            border: 1px solid var(--border-color);
        }

        .detail-section h4 {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-size: var(--font-size-md);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            margin-bottom: var(--spacing-md);
            padding-bottom: var(--spacing-sm);
            border-bottom: 1px solid var(--border-color);
        }

        .detail-section h4 i {
            color: var(--color-primary);
        }

        .detail-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-sm) 0;
            gap: var(--spacing-md);
        }

        .detail-label {
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
            font-weight: var(--font-weight-medium);
        }

        .detail-value {
            font-size: var(--font-size-sm);
            color: var(--text-primary);
            font-weight: var(--font-weight-semibold);
            text-align: right;
        }

        .linked-customers-list {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-sm);
        }

        .linked-customer-item {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            padding: var(--spacing-sm);
            background: var(--surface-secondary);
            border-radius: var(--border-radius);
            font-size: var(--font-size-sm);
        }

        .linked-customer-item i {
            color: var(--color-primary);
        }

        .customer-email {
            color: var(--text-tertiary);
            font-size: var(--font-size-xs);
            margin-left: auto;
        }

        .more-customers {
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
            padding: var(--spacing-sm);
            color: var(--text-tertiary);
            font-size: var(--font-size-xs);
            font-style: italic;
        }

        /* Slide-in Panel System */
        .panel-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 9998;
            backdrop-filter: blur(4px);
            animation: fadeIn 0.2s ease-out;
        }

        .panel-overlay.active {
            display: block;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        .slide-panel {
            position: fixed;
            top: 0;
            right: -600px;
            width: 600px;
            max-width: 90vw;
            height: 100vh;
            background: var(--surface-primary);
            box-shadow: -4px 0 20px rgba(0, 0, 0, 0.15);
            z-index: 9999;
            display: flex;
            flex-direction: column;
            transition: right var(--transition-normal) cubic-bezier(0.4, 0, 0.2, 1);
            overflow: hidden;
        }

        .slide-panel.active {
            right: 0;
        }

        .panel-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            padding: var(--spacing-xl);
            border-bottom: 2px solid var(--border-color);
            background: linear-gradient(135deg, var(--color-primary-light), var(--surface-primary));
            flex-shrink: 0;
        }

        .panel-header-content {
            flex: 1;
        }

        .panel-header h3 {
            margin: 0 0 var(--spacing-xs) 0;
            color: var(--text-primary);
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-bold);
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
        }

        .panel-header h3 i {
            color: var(--color-primary);
        }

        .panel-subtitle {
            margin: 0;
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
        }

        .panel-close {
            background: none;
            border: none;
            font-size: var(--font-size-lg);
            cursor: pointer;
            color: var(--text-tertiary);
            transition: all var(--transition-fast);
            padding: var(--spacing-sm);
            border-radius: var(--border-radius);
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .panel-close:hover {
            color: var(--text-primary);
            background: var(--surface-tertiary);
        }

        .panel-form {
            flex: 1;
            overflow-y: auto;
            padding: var(--spacing-xl);
        }

        .panel-actions {
            display: flex;
            justify-content: flex-end;
            gap: var(--spacing-md);
            padding: var(--spacing-xl);
            border-top: 1px solid var(--border-color);
            background: var(--surface-secondary);
            flex-shrink: 0;
        }

        /* Modern Modal System */
        .modern-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 10000;
        }

        .modern-modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
            animation: fadeIn 0.2s ease-out;
        }

        .modern-modal .modal-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.6);
            backdrop-filter: blur(4px);
        }

        .modern-modal .modal-dialog {
            position: relative;
            width: 90%;
            max-width: 500px;
            z-index: 10001;
            animation: slideDown 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-30px) scale(0.95);
            }
            to {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }

        .modern-modal .modal-content {
            background: var(--surface-primary);
            border-radius: var(--border-radius-large);
            box-shadow: var(--shadow-xl);
            overflow: hidden;
            border: 1px solid var(--border-color);
        }

        .modern-modal .modal-header {
            padding: var(--spacing-xl);
            text-align: center;
            border-bottom: 1px solid var(--border-color);
            background: linear-gradient(135deg, var(--color-primary-light), var(--surface-primary));
        }

        .modern-modal .modal-icon {
            width: 64px;
            height: 64px;
            border-radius: 50%;
            margin: 0 auto var(--spacing-md);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: var(--font-size-xxl);
            box-shadow: var(--shadow-md);
        }

        .modal-icon.alert-icon {
            background: linear-gradient(135deg, #0A84FF, #0051D5);
            color: white;
        }

        .modal-icon.success-icon {
            background: linear-gradient(135deg, var(--color-success), #28A745);
            color: white;
        }

        .modal-icon.error-icon {
            background: linear-gradient(135deg, var(--color-error), #DC3545);
            color: white;
        }

        .modal-icon.warning-icon {
            background: linear-gradient(135deg, var(--color-warning), #FF6B00);
            color: white;
        }

        .modal-icon.confirm-icon {
            background: linear-gradient(135deg, var(--color-warning), #FF6B00);
            color: white;
        }

        .modern-modal .modal-header h3 {
            margin: 0;
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-bold);
            color: var(--text-primary);
        }

        .modern-modal .modal-body {
            padding: var(--spacing-xl);
            text-align: center;
        }

        .modern-modal .modal-body p {
            margin: 0;
            font-size: var(--font-size-md);
            color: var(--text-primary);
            line-height: 1.6;
        }

        .modern-modal .modal-footer {
            padding: var(--spacing-lg) var(--spacing-xl);
            border-top: 1px solid var(--border-color);
            background: var(--surface-secondary);
            display: flex;
            justify-content: center;
            gap: var(--spacing-md);
        }

        .error-message {
            color: var(--color-error);
            font-size: var(--font-size-sm);
            margin-top: var(--spacing-xs);
        }

        .user-card {
            background: var(--surface-primary);
            border-radius: var(--border-radius-large);
            border: 1px solid var(--border-color);
            box-shadow: var(--shadow-sm);
            overflow: hidden;
            transition: all var(--transition-normal);
            display: flex;
            flex-direction: column;
        }

        .user-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
            border-color: var(--color-primary);
        }

        .user-card-header {
            padding: var(--spacing-xl);
            background: linear-gradient(135deg, var(--color-primary-light), var(--surface-primary));
            border-bottom: 1px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            position: relative;
        }

        .user-info-main {
            display: flex;
            gap: var(--spacing-md);
            flex: 1;
        }

        .user-avatar-large {
            width: 72px;
            height: 72px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--color-primary), var(--color-primary-hover));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-bold);
            box-shadow: var(--shadow-md);
            flex-shrink: 0;
        }

        .avatar-initials {
            font-size: var(--font-size-xl);
        }

        .user-details-main {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
        }

        .user-name-row {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            flex-wrap: wrap;
        }

        .user-name-row h3 {
            margin: 0;
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-bold);
            color: var(--text-primary);
        }

        .user-card-menu {
            position: relative;
        }

        .card-menu-btn {
            background: none;
            border: none;
            padding: var(--spacing-sm);
            cursor: pointer;
            color: var(--text-tertiary);
            border-radius: var(--border-radius);
            transition: all var(--transition-fast);
        }

        .card-menu-btn:hover {
            background: var(--surface-tertiary);
            color: var(--text-primary);
        }

        .card-menu-dropdown {
            display: none;
            position: absolute;
            top: 100%;
            right: 0;
            background: var(--surface-primary);
            border: 1px solid var(--border-color);
            border-radius: var(--border-radius);
            box-shadow: var(--shadow-lg);
            min-width: 200px;
            z-index: 100;
            margin-top: var(--spacing-xs);
        }

        .card-menu-dropdown.show {
            display: block;
        }

        .card-menu-dropdown button {
            width: 100%;
            padding: var(--spacing-md);
            border: none;
            background: none;
            text-align: left;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            color: var(--text-primary);
            transition: all var(--transition-fast);
        }

        .card-menu-dropdown button:hover {
            background: var(--surface-secondary);
        }

        .card-menu-dropdown button.danger {
            color: var(--color-error);
        }

        .card-menu-dropdown button.danger:hover {
            background: rgba(255, 59, 48, 0.1);
        }

        .user-card-body {
            padding: var(--spacing-xl);
            flex: 1;
        }

        .info-section {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-md);
        }

        .info-item {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            padding: var(--spacing-md);
            background: var(--surface-secondary);
            border-radius: var(--border-radius);
            transition: all var(--transition-normal);
        }

        .info-item:hover {
            background: var(--surface-tertiary);
        }

        .info-item.highlight {
            background: var(--color-primary-light);
            border: 1px solid var(--color-primary);
        }

        .info-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--color-primary), var(--color-primary-hover));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: var(--font-size-sm);
            flex-shrink: 0;
        }

        .info-icon.highlight {
            background: var(--color-primary);
        }

        .info-content {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
            flex: 1;
        }

        .info-label {
            font-size: var(--font-size-xs);
            color: var(--text-tertiary);
            font-weight: var(--font-weight-medium);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .info-value {
            font-size: var(--font-size-md);
            color: var(--text-primary);
            font-weight: var(--font-weight-semibold);
        }

        .highlight-value {
            color: var(--color-primary);
        }

        .user-card-footer {
            padding: var(--spacing-lg) var(--spacing-xl);
            background: var(--surface-secondary);
            border-top: 1px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: var(--spacing-md);
        }

        .footer-info {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
        }

        .footer-actions {
            display: flex;
            gap: var(--spacing-sm);
        }

        .action-btn {
            width: 36px;
            height: 36px;
            border: none;
            border-radius: var(--border-radius);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all var(--transition-normal);
            color: var(--text-secondary);
            background: var(--surface-primary);
            border: 1px solid var(--border-color);
        }

        .action-btn:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-sm);
        }

        .action-btn.edit-btn:hover {
            background: var(--color-primary);
            color: white;
            border-color: var(--color-primary);
        }

        .action-btn.link-btn:hover {
            background: var(--color-success);
            color: white;
            border-color: var(--color-success);
        }

        .action-btn.delete-btn:hover {
            background: var(--color-error);
            color: white;
            border-color: var(--color-error);
        }

        .user-email {
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
            margin-top: var(--spacing-xs);
        }

        .user-stat {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            padding: var(--spacing-sm);
            background: var(--surface-secondary);
            border-radius: var(--border-radius);
            transition: all var(--transition-normal);
        }

        .user-stat.highlight {
            background: var(--color-primary-light);
            border: 1px solid var(--color-primary);
        }

        .user-stat:hover {
            background: var(--surface-tertiary);
        }

        .stat-icon {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--color-primary), var(--color-primary-hover));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: var(--font-size-sm);
            flex-shrink: 0;
        }

        .stat-content {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
            flex: 1;
        }

        .stat-label {
            font-size: var(--font-size-xs);
            color: var(--text-tertiary);
            font-weight: var(--font-weight-medium);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .stat-value {
            font-size: var(--font-size-md);
            color: var(--text-primary);
            font-weight: var(--font-weight-semibold);
        }

        .created-date {
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
        }

        .card-actions {
            display: flex;
            gap: var(--spacing-sm);
        }

        /* User Info */
        .user-info {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }

        .user-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--color-primary), var(--color-primary-hover));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-semibold);
            box-shadow: var(--shadow-sm);
        }

        .user-details {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
        }

        .user-name {
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            font-size: var(--font-size-md);
        }

        .user-username {
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
            font-weight: var(--font-weight-medium);
        }

        /* Badges */
        .role-badge {
            padding: var(--spacing-sm) var(--spacing-md);
            border-radius: var(--border-radius-large);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-bold);
            text-transform: uppercase;
            letter-spacing: 0.05em;
            box-shadow: var(--shadow-sm);
            transition: all var(--transition-normal);
        }

        .role-badge:hover {
            transform: translateY(-1px);
            box-shadow: var(--shadow-md);
        }

        .role-admin {
            background: linear-gradient(135deg, #FF3B30, #D70015);
            color: white;
        }

        .role-manager {
            background: linear-gradient(135deg, #FF9500, #FF6B00);
            color: white;
        }

        .role-employee {
            background: linear-gradient(135deg, var(--apple-gray-5), var(--apple-gray-6));
            color: white;
        }

        .autohaus-badge {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-sm);
            padding: var(--spacing-sm) var(--spacing-md);
            background: var(--color-primary-light);
            color: var(--color-primary);
            border-radius: var(--border-radius-large);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-semibold);
            box-shadow: var(--shadow-sm);
            transition: all var(--transition-normal);
        }

        .autohaus-badge:hover {
            transform: translateY(-1px);
            box-shadow: var(--shadow-md);
        }

        .autohaus-badge i {
            font-size: var(--font-size-xs);
        }

        .text-muted {
            color: var(--text-tertiary);
            font-style: italic;
            font-size: var(--font-size-sm);
        }

        .status-badge {
            padding: var(--spacing-sm) var(--spacing-md);
            border-radius: var(--border-radius-large);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-bold);
            text-transform: uppercase;
            letter-spacing: 0.05em;
            box-shadow: var(--shadow-sm);
            transition: all var(--transition-normal);
        }

        .status-badge:hover {
            transform: translateY(-1px);
            box-shadow: var(--shadow-md);
        }

        .status-active {
            background: linear-gradient(135deg, var(--color-success), #28A745);
            color: white;
        }

        .status-inactive {
            background: linear-gradient(135deg, var(--color-error), #DC3545);
            color: white;
        }

        /* Action Buttons */
        .action-buttons {
            display: flex;
            gap: var(--spacing-sm);
        }

        .btn {
            padding: var(--spacing-sm) var(--spacing-md);
            border: none;
            border-radius: var(--border-radius);
            font-size: var(--font-size-sm);
            font-weight: var(--font-weight-semibold);
            cursor: pointer;
            transition: all var(--transition-normal);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-sm);
            position: relative;
            overflow: hidden;
        }

        .btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
            transition: left 0.5s;
        }

        .btn:hover::before {
            left: 100%;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--color-primary), var(--color-primary-hover));
            color: white;
            box-shadow: var(--shadow-sm);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .btn-secondary {
            background: linear-gradient(135deg, var(--apple-gray-5), var(--apple-gray-6));
            color: white;
            box-shadow: var(--shadow-sm);
        }

        .btn-secondary:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .btn-danger {
            background: linear-gradient(135deg, var(--color-error), #DC3545);
            color: white;
            box-shadow: var(--shadow-sm);
        }

        .btn-danger:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .btn-sm {
            padding: var(--spacing-xs) var(--spacing-sm);
            font-size: var(--font-size-xs);
        }

        /* Modal System */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            backdrop-filter: blur(10px);
        }

        .modal-content {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: var(--surface-primary);
            border-radius: var(--border-radius-large);
            border: 1px solid var(--border-color);
            width: 90%;
            max-width: 600px;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-xl);
            animation: modalSlideIn 0.3s ease-out;
        }

        .modal-large {
            max-width: 800px;
        }

        .modal-header-content {
            flex: 1;
        }

        .modal-subtitle {
            margin: var(--spacing-xs) 0 0 0;
            font-size: var(--font-size-sm);
            color: var(--text-tertiary);
        }

        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translate(-50%, -60%);
            }
            to {
                opacity: 1;
                transform: translate(-50%, -50%);
            }
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-xl);
            border-bottom: 1px solid var(--border-color);
            background: var(--surface-secondary);
        }

        .modal-header h3 {
            margin: 0;
            color: var(--text-primary);
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-semibold);
        }

        .modal-close {
            background: none;
            border: none;
            font-size: var(--font-size-xl);
            cursor: pointer;
            color: var(--text-tertiary);
            transition: color var(--transition-fast);
            padding: var(--spacing-sm);
            border-radius: var(--border-radius);
        }

        .modal-close:hover {
            color: var(--text-primary);
            background: var(--surface-tertiary);
        }

        .modal-form {
            padding: var(--spacing-xl);
        }

        .form-steps {
            display: flex;
            justify-content: space-between;
            margin-bottom: var(--spacing-xl);
            padding-bottom: var(--spacing-lg);
            border-bottom: 2px solid var(--border-color);
        }

        .step {
            flex: 1;
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: var(--spacing-sm);
            position: relative;
            opacity: 0.5;
            transition: all var(--transition-normal);
        }

        .step::after {
            content: '';
            position: absolute;
            top: 20px;
            left: 60%;
            width: 100%;
            height: 2px;
            background: var(--border-color);
            z-index: -1;
        }

        .step:last-child::after {
            display: none;
        }

        .step.active {
            opacity: 1;
        }

        .step.active .step-number {
            background: var(--color-primary);
            color: white;
            border-color: var(--color-primary);
        }

        .step-number {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: var(--surface-secondary);
            border: 2px solid var(--border-color);
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: var(--font-weight-bold);
            transition: all var(--transition-normal);
        }

        .step-label {
            font-size: var(--font-size-xs);
            color: var(--text-tertiary);
            font-weight: var(--font-weight-medium);
            text-align: center;
        }

        .step.active .step-label {
            color: var(--text-primary);
            font-weight: var(--font-weight-semibold);
        }

        .form-step-content {
            display: none;
        }

        .form-step-content.active {
            display: block;
            animation: fadeIn 0.3s ease-out;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .form-sections {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xl);
        }

        .form-section {
            padding: var(--spacing-lg);
            background: var(--surface-secondary);
            border-radius: var(--border-radius);
            border: 1px solid var(--border-color);
        }

        .section-title {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-size: var(--font-size-md);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            margin-bottom: var(--spacing-lg);
            padding-bottom: var(--spacing-md);
            border-bottom: 1px solid var(--border-color);
        }

        .section-title i {
            color: var(--color-primary);
        }

        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: var(--spacing-lg);
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-sm);
        }

        .form-group label {
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            font-size: var(--font-size-sm);
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
        }

        .form-group label i {
            color: var(--text-tertiary);
            font-size: var(--font-size-xs);
        }

        .required {
            color: var(--color-error);
        }

        .form-group input,
        .form-group select {
            padding: var(--spacing-md);
            border: 2px solid var(--border-color);
            border-radius: var(--border-radius);
            font-family: var(--font-family);
            font-size: var(--font-size-md);
            background: var(--surface-primary);
            color: var(--text-primary);
            transition: all var(--transition-normal);
        }

        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px var(--color-primary-light);
        }

        .form-hint {
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
            font-size: var(--font-size-xs);
            color: var(--text-tertiary);
            margin-top: var(--spacing-xs);
        }

        .form-hint i {
            font-size: var(--font-size-xs);
        }

        .checkbox-label {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            cursor: pointer;
            font-weight: var(--font-weight-medium);
        }

        .checkbox-label input[type="checkbox"] {
            margin: 0;
            transform: scale(1.2);
        }

        .modal-actions {
            display: flex;
            justify-content: flex-end;
            gap: var(--spacing-md);
        }

        /* Responsive Design */
        @media (max-width: 1024px) {
            .admin-main {
                margin-left: 0;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }

            .search-filter-container {
                flex-direction: column;
            }

            .search-box {
                width: 100%;
            }

            .filter-group {
                width: 100%;
            }

            .filter-group select {
                flex: 1;
            }
        }

        @media (max-width: 768px) {
            .admin-content {
                padding: var(--spacing-md);
            }

            .users-table-container {
                overflow-x: auto;
                border-radius: var(--border-radius);
            }

            .users-table {
                min-width: 1000px;
            }

            .users-table th,
            .users-table td {
                padding: var(--spacing-md) var(--spacing-sm);
            }

            .slide-panel {
                width: 100%;
                max-width: 100vw;
            }
            
            .user-card-footer {
                flex-direction: column;
                gap: var(--spacing-md);
                align-items: flex-start;
            }
            
            .footer-actions {
                width: 100%;
                justify-content: flex-end;
            }
            
            .admin-header {
                padding: var(--spacing-lg);
            }
            
            .header-content {
                flex-direction: column;
                gap: var(--spacing-lg);
                text-align: center;
            }
            
            .admin-content {
                padding: var(--spacing-lg);
            }
            
            .action-bar-top {
                flex-direction: column;
                gap: var(--spacing-lg);
                text-align: center;
            }
            
            .action-bar-right {
                width: 100%;
                justify-content: center;
            }
            
            .btn-large {
                width: 100%;
                justify-content: center;
            }
            
            .modal-content {
                width: 95%;
            }
            
            .modal-form {
                padding: var(--spacing-lg);
            }

            .form-steps {
                flex-direction: column;
                gap: var(--spacing-md);
            }

            .step::after {
                display: none;
            }

            .user-info-main {
                flex-direction: column;
                text-align: center;
            }

            .user-name-row {
                justify-content: center;
            }

            .info-item {
                flex-direction: column;
                text-align: center;
            }
        }
    </style>
    
    <script>
        let currentStep = 1;
        const totalSteps = 3;

        function openCreateUserPanel() {
            document.getElementById('panelOverlay').classList.add('active');
            document.getElementById('createUserPanel').classList.add('active');
            resetFormSteps();
        }
        
        function closeCreateUserPanel() {
            document.getElementById('createUserPanel').classList.remove('active');
            document.getElementById('panelOverlay').classList.remove('active');
            resetFormSteps();
            document.getElementById('createUserForm').reset();
        }

        function closeAllPanels() {
            document.querySelectorAll('.slide-panel').forEach(panel => panel.classList.remove('active'));
            document.getElementById('panelOverlay').classList.remove('active');
        }

        function resetFormSteps() {
            currentStep = 1;
            updateFormSteps();
        }

        function nextStep() {
            if (currentStep < totalSteps) {
                // Validiere aktuellen Schritt
                const currentStepContent = document.querySelector(`.form-step-content[data-step="${currentStep}"]`);
                const requiredFields = currentStepContent.querySelectorAll('[required]');
                let isValid = true;

                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        isValid = false;
                        field.style.borderColor = 'var(--color-error)';
                    } else {
                        field.style.borderColor = '';
                    }
                });

                if (isValid) {
                    currentStep++;
                    updateFormSteps();
                } else {
                    showAlert('Validierungsfehler', 'Bitte füllen Sie alle erforderlichen Felder aus.');
                }
            }
        }

        function prevStep() {
            if (currentStep > 1) {
                currentStep--;
                updateFormSteps();
            }
        }

        function updateFormSteps() {
            // Update step indicators
            document.querySelectorAll('.step').forEach((step, index) => {
                const stepNum = index + 1;
                if (stepNum <= currentStep) {
                    step.classList.add('active');
                } else {
                    step.classList.remove('active');
                }
            });

            // Update step content
            document.querySelectorAll('.form-step-content').forEach((content, index) => {
                const stepNum = index + 1;
                if (stepNum === currentStep) {
                    content.classList.add('active');
                } else {
                    content.classList.remove('active');
                }
            });

            // Update buttons
            const prevBtn = document.getElementById('prevStepBtn');
            const nextBtn = document.getElementById('nextStepBtn');
            const submitBtn = document.getElementById('submitBtn');

            if (currentStep === 1) {
                prevBtn.style.display = 'none';
            } else {
                prevBtn.style.display = 'inline-flex';
            }

            if (currentStep === totalSteps) {
                nextBtn.style.display = 'none';
                submitBtn.style.display = 'inline-flex';
            } else {
                nextBtn.style.display = 'inline-flex';
                submitBtn.style.display = 'none';
            }
        }
        
        function editUser(userId) {
            // Lade Benutzerdaten
            fetch(`api/get_user.php?id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const user = data.user;
                        
                        // Fülle das Edit-Formular
                        document.getElementById('edit_user_id').value = user.id;
                        document.getElementById('edit_username').value = user.username;
                        document.getElementById('edit_email').value = user.email;
                        document.getElementById('edit_full_name').value = user.full_name;
                        document.getElementById('edit_role').value = user.role;
                        document.getElementById('edit_autohaus_id').value = user.autohaus_id || '';
                        document.getElementById('edit_department').value = user.department || '';
                        document.getElementById('edit_status').value = user.is_active ? 'active' : 'inactive';
                        document.getElementById('edit_can_switch_autohaus').checked = user.can_switch_autohaus == 1;
                        document.getElementById('edit_phone').value = user.phone || '';
                        
                        // Zeige das Edit-Panel
                        document.getElementById('panelOverlay').classList.add('active');
                        document.getElementById('editUserPanel').classList.add('active');
                    } else {
                        showAlert('Fehler', 'Fehler beim Laden der Benutzerdaten: ' + data.error);
                    }
                })
                .catch(error => {
                    console.error('Fehler:', error);
                    showAlert('Fehler', 'Fehler beim Laden der Benutzerdaten. Bitte versuchen Sie es erneut.');
                });
        }
        
        function closeEditUserPanel() {
            document.getElementById('editUserPanel').classList.remove('active');
            document.getElementById('panelOverlay').classList.remove('active');
        }
        
        let pendingDeleteUserId = null;
        
        function deleteUser(userId) {
            pendingDeleteUserId = userId;
            showConfirm(
                'Benutzer löschen',
                'Möchten Sie diesen Benutzer wirklich löschen? Diese Aktion kann nicht rückgängig gemacht werden.',
                'Löschen',
                function() {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete_user">
                        <input type="hidden" name="user_id" value="${pendingDeleteUserId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
            );
        }

        function showAlert(title, message, type = 'info') {
            document.getElementById('alertTitle').textContent = title;
            document.getElementById('alertMessage').textContent = message;
            
            const modal = document.getElementById('alertModal');
            const icon = modal.querySelector('.modal-icon');
            const iconElement = icon.querySelector('i');
            
            // Setze Icon basierend auf Typ
            icon.className = 'modal-icon';
            if (type === 'error') {
                icon.classList.add('error-icon');
                iconElement.className = 'fas fa-exclamation-triangle';
            } else if (type === 'success') {
                icon.classList.add('success-icon');
                iconElement.className = 'fas fa-check-circle';
            } else if (type === 'warning') {
                icon.classList.add('warning-icon');
                iconElement.className = 'fas fa-exclamation-circle';
            } else {
                icon.classList.add('alert-icon');
                iconElement.className = 'fas fa-info-circle';
            }
            
            modal.classList.add('active');
        }

        function closeAlertModal() {
            document.getElementById('alertModal').classList.remove('active');
        }

        let confirmCallback = null;

        function showConfirm(title, message, confirmText = 'Bestätigen', callback) {
            document.getElementById('confirmTitle').textContent = title;
            document.getElementById('confirmMessage').textContent = message;
            document.getElementById('confirmButton').innerHTML = '<i class="fas fa-check"></i> ' + confirmText;
            confirmCallback = callback;
            document.getElementById('confirmModal').classList.add('active');
        }

        function confirmAction() {
            if (confirmCallback) {
                confirmCallback();
                confirmCallback = null;
            }
            closeConfirmModal();
        }

        function closeConfirmModal() {
            document.getElementById('confirmModal').classList.remove('active');
            confirmCallback = null;
        }

        // ESC-Taste schließt auch Verifizierungs-Modal
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeAlertModal();
                closeConfirmModal();
                closeVerificationCodeModal();
            }
        });
        
        function showLinkPanel(adminUserId, adminUserName) {
            document.getElementById('link_admin_user_id').value = adminUserId;
            document.getElementById('panelOverlay').classList.add('active');
            document.getElementById('linkCustomerPanel').classList.add('active');
            
            // Update panel title
            const panelTitle = document.querySelector('#linkCustomerPanel h3');
            panelTitle.innerHTML = '<i class="fas fa-link"></i> Kundenkonto mit ' + adminUserName + ' verknüpfen';
            const panelSubtitle = document.getElementById('linkPanelSubtitle');
            if (panelSubtitle) {
                panelSubtitle.textContent = 'Wählen Sie ein Kundenkonto aus, das mit ' + adminUserName + ' verknüpft werden soll';
            }
        }
        
        function closeLinkCustomerPanel() {
            document.getElementById('linkCustomerPanel').classList.remove('active');
            document.getElementById('panelOverlay').classList.remove('active');
        }

        function expandRow(row) {
            const detailsRow = row.nextElementSibling;
            if (detailsRow && detailsRow.classList.contains('user-details-row')) {
                row.classList.toggle('expanded');
            }
        }

        function verifyPhoneNumber(userId) {
            const btn = event.target.closest('.verify-btn');
            if (btn.disabled) return;
            
            btn.disabled = true;
            const icon = btn.querySelector('i');
            icon.className = 'fas fa-spinner fa-spin';
            
            const formData = new FormData();
            formData.append('user_id', userId);
            
            fetch('api/verify_phone.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                btn.disabled = false;
                icon.className = 'fas fa-sync-alt';
                
                if (data.success && data.code_sent) {
                    // Zeige Code-Eingabedialog
                    showVerificationCodeModal(userId);
                } else {
                    showAlert('Fehler', data.error || 'Fehler beim Senden des Codes', 'error');
                }
            })
            .catch(error => {
                console.error('Fehler:', error);
                showAlert('Fehler', 'Fehler beim Senden des Codes. Bitte versuchen Sie es erneut.', 'error');
                btn.disabled = false;
                icon.className = 'fas fa-sync-alt';
            });
        }

        function showVerificationCodeModal(userId) {
            const modal = document.getElementById('verificationCodeModal');
            document.getElementById('verification_user_id').value = userId;
            document.getElementById('verification_code').value = '';
            document.getElementById('verification_error').textContent = '';
            document.getElementById('verification_error').style.display = 'none';
            modal.classList.add('active');
            document.getElementById('panelOverlay').classList.add('active');
            
            // Fokus auf Code-Eingabefeld
            setTimeout(() => {
                document.getElementById('verification_code').focus();
            }, 100);
        }

        function closeVerificationCodeModal() {
            document.getElementById('verificationCodeModal').classList.remove('active');
            document.getElementById('panelOverlay').classList.remove('active');
        }

        function submitVerificationCode() {
            const userId = document.getElementById('verification_user_id').value;
            const code = document.getElementById('verification_code').value.trim();
            const errorDiv = document.getElementById('verification_error');
            const submitBtn = document.getElementById('verify_code_btn');
            
            if (!code || code.length !== 6) {
                errorDiv.textContent = 'Bitte geben Sie einen 6-stelligen Code ein';
                errorDiv.style.display = 'block';
                return;
            }
            
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Verifiziere...';
            errorDiv.style.display = 'none';
            
            const formData = new FormData();
            formData.append('action', 'verify_code');
            formData.append('user_id', userId);
            formData.append('code', code);
            
            fetch('api/verify_phone.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showAlert('Erfolg', data.message, 'success');
                    closeVerificationCodeModal();
                    // Seite neu laden nach kurzer Verzögerung
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    errorDiv.textContent = data.error || 'Ungültiger Code';
                    errorDiv.style.display = 'block';
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = '<i class="fas fa-check"></i> Verifizieren';
                    // Code-Feld leeren und Fokus setzen
                    document.getElementById('verification_code').value = '';
                    document.getElementById('verification_code').focus();
                }
            })
            .catch(error => {
                console.error('Fehler:', error);
                errorDiv.textContent = 'Fehler bei der Verifizierung. Bitte versuchen Sie es erneut.';
                errorDiv.style.display = 'block';
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fas fa-check"></i> Verifizieren';
            });
        }

        function toggleCardMenu(btn) {
            const dropdown = btn.nextElementSibling;
            document.querySelectorAll('.card-menu-dropdown').forEach(menu => {
                if (menu !== dropdown) {
                    menu.classList.remove('show');
                }
            });
            dropdown.classList.toggle('show');
        }

        function filterUsers() {
            const searchTerm = document.getElementById('searchInput').value.toLowerCase();
            const roleFilter = document.getElementById('roleFilter').value;
            const statusFilter = document.getElementById('statusFilter').value;
            const rows = document.querySelectorAll('.user-row');

            rows.forEach(row => {
                const username = row.dataset.username || '';
                const name = row.dataset.name || '';
                const email = row.dataset.email || '';
                const role = row.dataset.role || '';
                const status = row.dataset.status || '';

                const matchesSearch = !searchTerm || 
                    username.includes(searchTerm) || 
                    name.includes(searchTerm) || 
                    email.includes(searchTerm);

                const matchesRole = !roleFilter || role === roleFilter;
                const matchesStatus = !statusFilter || status === statusFilter;

                if (matchesSearch && matchesRole && matchesStatus) {
                    row.style.display = '';
                    const detailsRow = row.nextElementSibling;
                    if (detailsRow && detailsRow.classList.contains('user-details-row')) {
                        detailsRow.style.display = row.classList.contains('expanded') ? 'table-row' : 'none';
                    }
                } else {
                    row.style.display = 'none';
                    const detailsRow = row.nextElementSibling;
                    if (detailsRow && detailsRow.classList.contains('user-details-row')) {
                        detailsRow.style.display = 'none';
                    }
                }
            });
        }
        
        // Close dropdown menus when clicking outside
        window.onclick = function(event) {
            // Close dropdown menus
            if (!event.target.closest('.card-menu-btn') && !event.target.closest('.card-menu-dropdown')) {
                document.querySelectorAll('.card-menu-dropdown').forEach(menu => {
                    menu.classList.remove('show');
                });
            }
        }

        // ESC-Taste schließt Modals
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeAlertModal();
                closeConfirmModal();
            }
        });
        
        // User menu toggle
        document.addEventListener('DOMContentLoaded', function() {
            const userMenuToggle = document.querySelector('.user-menu-toggle');
            const userMenu = document.querySelector('.user-menu');
            
            if (userMenuToggle && userMenu) {
                userMenuToggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    userMenu.classList.toggle('active');
                });
                
                // Close menu when clicking outside
                document.addEventListener('click', function(e) {
                    if (!userMenu.contains(e.target)) {
                        userMenu.classList.remove('active');
                    }
                });
            }

            // Öffne Verifizierungs-Modal automatisch wenn Code gesendet wurde
            <?php if ($pending_verification_user_id): ?>
            setTimeout(function() {
                showVerificationCodeModal(<?php echo $pending_verification_user_id; ?>);
            }, 500);
            <?php endif; ?>
        });
    </script>
    
    <script src="../assets/js/main.js"></script>
    <script src="assets/js/admin.js"></script>
    <script src="assets/js/unified-sidebar.js"></script>
    
</body>
</html>
