<?php
/**
 * UI für WhatsApp-Aktionen Test
 */

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'includes/helpers.php';

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Lade Autohäuser
$autohausStmt = $conn->query("SELECT id, name FROM autohaus WHERE is_active = 1 ORDER BY name");
$autohauser = $autohausStmt->fetchAll(PDO::FETCH_ASSOC);

$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_autohaus_id = null;

if ($current_user_id) {
    $stmt = $conn->prepare("SELECT autohaus_id FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_autohaus_id = $user_data['autohaus_id'] ?? null;
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WhatsApp Aktionen Test - Boxenstop</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/unified-sidebar.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .test-container {
            padding: 2rem;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .test-header {
            background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
            color: white;
            padding: 2rem;
            border-radius: 16px;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 211, 102, 0.3);
        }
        
        .test-header h1 {
            margin: 0 0 0.5rem 0;
            font-size: 2rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .test-form {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 2px 12px rgba(0,0,0,0.1);
            margin-bottom: 2rem;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: #212529;
        }
        
        .form-group input,
        .form-group select {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            font-size: 0.9rem;
        }
        
        .action-checkboxes {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }
        
        .action-checkbox {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem;
            background: #f8f9fa;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .action-checkbox:hover {
            background: #e9ecef;
        }
        
        .action-checkbox input[type="checkbox"] {
            width: auto;
            cursor: pointer;
        }
        
        .test-results {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 2px 12px rgba(0,0,0,0.1);
            display: none;
        }
        
        .test-results.show {
            display: block;
        }
        
        .result-item {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1rem;
            border-left: 4px solid;
        }
        
        .result-item.success {
            background: #d4edda;
            border-color: #28a745;
            color: #155724;
        }
        
        .result-item.error {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        
        .result-item-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.5rem;
        }
        
        .result-item-title {
            font-weight: 600;
            font-size: 1.1rem;
        }
        
        .result-item-details {
            font-size: 0.9rem;
            margin-top: 0.5rem;
        }
        
        .summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .summary-card {
            background: white;
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            text-align: center;
        }
        
        .summary-card-value {
            font-size: 2rem;
            font-weight: 700;
            color: #0056b3;
        }
        
        .summary-card-label {
            font-size: 0.9rem;
            color: #6c757d;
            margin-top: 0.5rem;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-primary {
            background: #0056b3;
            color: white;
        }
        
        .btn-primary:hover {
            background: #004085;
        }
        
        .btn-primary:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .loading {
            text-align: center;
            padding: 2rem;
        }
        
        .loading i {
            font-size: 2rem;
            color: #0056b3;
        }
        
        .info-box {
            background: #e3f2fd;
            border-left: 4px solid #1976d2;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }
        
        .info-box h3 {
            margin: 0 0 0.5rem 0;
            color: #1565c0;
            font-size: 1rem;
        }
        
        .info-box ul {
            margin: 0.5rem 0;
            padding-left: 1.5rem;
        }
        
        .info-box li {
            margin: 0.25rem 0;
            color: #0d47a1;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>
        <main class="admin-main">
            <div class="test-container">
                <div class="test-header">
                    <h1>
                        <i class="fab fa-whatsapp"></i>
                        WhatsApp Aktionen Test
                    </h1>
                    <p style="margin: 0; opacity: 0.9;">Testen Sie alle WhatsApp-Aktionen mit einer Test-Telefonnummer</p>
                </div>
                
                <div class="info-box">
                    <h3><i class="fas fa-info-circle"></i> So testen Sie WhatsApp-Aktionen:</h3>
                    <ul>
                        <li><strong>24h-Erinnerung:</strong> Erstellt einen Test-Termin für morgen und sendet die Erinnerung</li>
                        <li><strong>48h-Erinnerung:</strong> Erstellt einen Test-Termin für übermorgen und sendet die Erinnerung</li>
                        <li><strong>Bestätigung:</strong> Erstellt einen Test-Termin und sendet die Bestätigung</li>
                        <li><strong>Genehmigt/Abgelehnt/Storniert/Verschoben:</strong> Erstellt einen Test-Termin mit entsprechendem Status</li>
                        <li><strong>Bewertungsanfrage:</strong> Erstellt einen Test-Termin für gestern und sendet die Bewertungsanfrage (24h nach Termin)</li>
                    </ul>
                    <p style="margin: 0.5rem 0 0 0; color: #0d47a1; font-size: 0.9rem;">
                        <strong>Hinweis:</strong> Alle Test-Termine werden automatisch nach dem Versand gelöscht (außer Stornierung). 
                        Die Logs können Sie in <a href="whatsapp_logs.php" style="color: #1976d2; text-decoration: underline;">WhatsApp Logs</a> einsehen.
                    </p>
                </div>
                
                <div class="test-form">
                    <form id="test-form" onsubmit="runTests(event)">
                        <div class="form-group">
                            <label for="autohaus_id">Autohaus *</label>
                            <select id="autohaus_id" name="autohaus_id" required>
                                <?php foreach ($autohauser as $ah): ?>
                                    <option value="<?php echo $ah['id']; ?>" <?php echo ($current_user_autohaus_id == $ah['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($ah['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="test_phone">Test-Telefonnummer *</label>
                            <input type="tel" id="test_phone" name="phone" value="01782465658" required 
                                   placeholder="+49 123 456 789">
                            <small style="color: #6c757d; display: block; margin-top: 0.25rem;">
                                Telefonnummer im internationalen Format (z.B. +491782465658)
                            </small>
                        </div>
                        
                        <div class="form-group">
                            <label>Aktionen zum Testen</label>
                            <div class="action-checkboxes">
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="confirmation" checked>
                                    <span>Termin-Bestätigung</span>
                                </label>
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="reminder_24h" checked>
                                    <span>Erinnerung 24h</span>
                                </label>
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="reminder_48h" checked>
                                    <span>Erinnerung 48h</span>
                                </label>
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="approved" checked>
                                    <span>Termin genehmigt</span>
                                </label>
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="rejected" checked>
                                    <span>Termin abgelehnt</span>
                                </label>
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="cancellation" checked>
                                    <span>Termin storniert</span>
                                </label>
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="rescheduled" checked>
                                    <span>Termin verschoben</span>
                                </label>
                                <label class="action-checkbox">
                                    <input type="checkbox" name="actions[]" value="review_request" checked>
                                    <span>Bewertungsanfrage</span>
                                </label>
                            </div>
                            <div style="margin-top: 1rem;">
                                <button type="button" class="btn btn-secondary" onclick="selectAllActions()" style="background: #6c757d; color: white; padding: 0.5rem 1rem; font-size: 0.875rem;">
                                    Alle auswählen
                                </button>
                                <button type="button" class="btn btn-secondary" onclick="deselectAllActions()" style="background: #6c757d; color: white; padding: 0.5rem 1rem; font-size: 0.875rem; margin-left: 0.5rem;">
                                    Alle abwählen
                                </button>
                            </div>
                        </div>
                        
                        <div style="display: flex; gap: 1rem; justify-content: flex-end;">
                            <button type="button" class="btn btn-secondary" onclick="window.location.href='whatsapp_reminders_admin.php'" style="background: #6c757d; color: white;">
                                <i class="fas fa-arrow-left"></i> Zurück
                            </button>
                            <button type="submit" class="btn btn-primary" id="test-btn">
                                <i class="fas fa-play"></i> Tests starten
                            </button>
                        </div>
                    </form>
                </div>
                
                <div id="test-results" class="test-results">
                    <h2 style="margin-top: 0;">Test-Ergebnisse</h2>
                    <div id="summary" class="summary"></div>
                    <div id="results-container"></div>
                    <div id="logs-container" style="margin-top: 2rem;">
                        <h3>Logs</h3>
                        <div id="logs-list"></div>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script>
        function selectAllActions() {
            document.querySelectorAll('input[name="actions[]"]').forEach(cb => cb.checked = true);
        }
        
        function deselectAllActions() {
            document.querySelectorAll('input[name="actions[]"]').forEach(cb => cb.checked = false);
        }
        
        async function runTests(event) {
            event.preventDefault();
            
            const form = document.getElementById('test-form');
            const resultsDiv = document.getElementById('test-results');
            const resultsContainer = document.getElementById('results-container');
            const summaryDiv = document.getElementById('summary');
            const logsList = document.getElementById('logs-list');
            const testBtn = document.getElementById('test-btn');
            
            // Hole ausgewählte Aktionen
            const selectedActions = Array.from(document.querySelectorAll('input[name="actions[]"]:checked'))
                .map(cb => cb.value);
            
            if (selectedActions.length === 0) {
                alert('Bitte wählen Sie mindestens eine Aktion aus');
                return;
            }
            
            // Zeige Loading
            testBtn.disabled = true;
            testBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Tests laufen...';
            resultsDiv.classList.add('show');
            resultsContainer.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i><p>Tests werden durchgeführt...</p></div>';
            summaryDiv.innerHTML = '';
            logsList.innerHTML = '';
            
            // Führe Tests für jede Aktion durch
            const allResults = [];
            
            for (const action of selectedActions) {
                try {
                    const formData = new FormData();
                    formData.append('autohaus_id', document.getElementById('autohaus_id').value);
                    formData.append('phone', document.getElementById('test_phone').value);
                    formData.append('action', action);
                    
                    const response = await fetch('test_whatsapp_actions.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    const data = await response.json();
                    
                    if (data.success && data.results && data.results[action]) {
                        allResults.push(data.results[action]);
                    } else {
                        allResults.push({
                            action: action,
                            label: action,
                            success: false,
                            error: data.error || 'Unbekannter Fehler'
                        });
                    }
                    
                    // Kurze Pause zwischen Tests
                    await new Promise(resolve => setTimeout(resolve, 1000));
                    
                } catch (error) {
                    allResults.push({
                        action: action,
                        label: action,
                        success: false,
                        error: error.message
                    });
                }
            }
            
            // Hole Logs
            try {
                const logsResponse = await fetch(`whatsapp_logs.php?autohaus_id=${document.getElementById('autohaus_id').value}&phone=${encodeURIComponent(document.getElementById('test_phone').value)}&time_range=today&format=json`);
                const logsData = await logsResponse.json();
                // Logs werden separat geladen
            } catch (error) {
                console.error('Fehler beim Laden der Logs:', error);
            }
            
            // Zeige Ergebnisse
            displayResults(allResults);
            
            // Button wieder aktivieren
            testBtn.disabled = false;
            testBtn.innerHTML = '<i class="fas fa-play"></i> Tests starten';
        }
        
        function displayResults(results) {
            const summaryDiv = document.getElementById('summary');
            const resultsContainer = document.getElementById('results-container');
            
            const successful = results.filter(r => r.success).length;
            const failed = results.filter(r => !r.success).length;
            const total = results.length;
            
            // Summary
            summaryDiv.innerHTML = `
                <div class="summary-card">
                    <div class="summary-card-value">${total}</div>
                    <div class="summary-card-label">Getestet</div>
                </div>
                <div class="summary-card">
                    <div class="summary-card-value" style="color: #28a745;">${successful}</div>
                    <div class="summary-card-label">Erfolgreich</div>
                </div>
                <div class="summary-card">
                    <div class="summary-card-value" style="color: #dc3545;">${failed}</div>
                    <div class="summary-card-label">Fehlgeschlagen</div>
                </div>
            `;
            
            // Ergebnisse
            let html = '';
            results.forEach(result => {
                const isSuccess = result.success;
                html += `
                    <div class="result-item ${isSuccess ? 'success' : 'error'}">
                        <div class="result-item-header">
                            <div class="result-item-title">
                                <i class="fas fa-${isSuccess ? 'check-circle' : 'exclamation-circle'}"></i>
                                ${result.label || result.action}
                            </div>
                            <span style="font-size: 0.85rem; opacity: 0.8;">
                                ${isSuccess ? '✓ Erfolgreich' : '✗ Fehler'}
                            </span>
                        </div>
                        <div class="result-item-details">
                            ${result.message ? `<div><strong>Nachricht:</strong> ${escapeHtml(result.message)}</div>` : ''}
                            ${result.error ? `<div><strong>Fehler:</strong> ${escapeHtml(result.error)}</div>` : ''}
                            ${result.appointment_id ? `<div><strong>Termin-ID:</strong> ${result.appointment_id}</div>` : ''}
                            ${result.slot_date ? `<div><strong>Datum:</strong> ${result.slot_date} ${result.slot_time || ''}</div>` : ''}
                        </div>
                    </div>
                `;
            });
            
            resultsContainer.innerHTML = html;
            
            // Lade Logs
            loadLogs();
        }
        
        async function loadLogs() {
            const autohausId = document.getElementById('autohaus_id').value;
            const phone = document.getElementById('test_phone').value;
            const logsList = document.getElementById('logs-list');
            
            try {
                // Lade Logs von der API
                const response = await fetch(`api/get_whatsapp_logs.php?autohaus_id=${autohausId}&phone=${encodeURIComponent(phone)}&limit=20`);
                const data = await response.json();
                
                if (data.success && data.logs && data.logs.length > 0) {
                    let html = '<div style="max-height: 400px; overflow-y: auto;">';
                    data.logs.forEach(log => {
                        const statusClass = log.status === 'success' ? 'success' : 'error';
                        html += `
                            <div class="result-item ${statusClass}" style="margin-bottom: 0.5rem;">
                                <div class="result-item-header">
                                    <div class="result-item-title" style="font-size: 0.95rem;">
                                        ${log.action_type || 'Unbekannt'} - ${log.status || 'Unbekannt'}
                                    </div>
                                    <span style="font-size: 0.75rem; opacity: 0.8;">
                                        ${new Date(log.created_at).toLocaleString('de-DE')}
                                    </span>
                                </div>
                                <div class="result-item-details" style="font-size: 0.85rem;">
                                    ${log.message ? `<div>${escapeHtml(log.message)}</div>` : ''}
                                    ${log.error_message ? `<div style="color: #dc3545;"><strong>Fehler:</strong> ${escapeHtml(log.error_message)}</div>` : ''}
                                </div>
                            </div>
                        `;
                    });
                    html += '</div>';
                    logsList.innerHTML = html;
                } else {
                    logsList.innerHTML = '<p style="color: #6c757d;">Keine Logs gefunden</p>';
                }
            } catch (error) {
                logsList.innerHTML = '<p style="color: #dc3545;">Fehler beim Laden der Logs: ' + error.message + '</p>';
            }
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    </script>
</body>
</html>

