<?php
/**
 * API-Endpoint für Etikett-Druck aus Appointments
 * API-Endpoint für Etikett-Druck
 */

// Session nur starten, wenn noch nicht gestartet
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Database config
if (file_exists(__DIR__ . '/../config/database.php')) {
    require_once __DIR__ . '/../config/database.php';
}
require_once __DIR__ . '/../config/printer.php';  // Lade Drucker-Konfiguration
require_once __DIR__ . '/includes/smart_printer_detection.php';  // Intelligente Drucker-Erkennung

// WICHTIG: Session-Prüfung wird später im API-Handler durchgeführt
// Hier nur die notwendigen Includes laden

// Drucker-Server-Credentials werden jetzt aus config/printer.php geladen

// Output-Buffering starten
ob_start();

// JSON-Header setzen
header('Content-Type: application/json; charset=utf-8');

// Timeout erhöhen für Druck-Operationen (können länger dauern)
set_time_limit(120); // 2 Minuten Timeout
ini_set('max_execution_time', 120);

// Fehlerbehandlung für sauberes JSON
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Bereinige alle String-Werte im Response-Array für JSON
function cleanForJson($data) {
    if (is_array($data)) {
        foreach ($data as $key => $value) {
            $data[$key] = cleanForJson($value);
        }
    } elseif (is_string($data)) {
        // Versuche verschiedene Windows-Codepages zu UTF-8 zu konvertieren
        $encodings = ['Windows-1252', 'CP850', 'ISO-8859-1', 'UTF-8'];
        foreach ($encodings as $enc) {
            $converted = @mb_convert_encoding($data, 'UTF-8', $enc);
            if ($converted !== false && mb_check_encoding($converted, 'UTF-8')) {
                $data = $converted;
                break;
            }
        }
        // Entferne ungültige UTF-8 Zeichen
        $data = mb_convert_encoding($data, 'UTF-8', 'UTF-8');
        // Entferne problematische Zeichen, behalte aber Newlines und Tabs
        $data = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $data);
        // Ersetze problematische Zeichen durch Platzhalter
        $data = mb_convert_encoding($data, 'UTF-8', 'UTF-8');
        if (!mb_check_encoding($data, 'UTF-8')) {
            $data = utf8_encode($data); // Fallback
        }
    }
    return $data;
}

// Druck-Funktion
function printEtikett($name, $kennzeichen, $copies = 1, $autohaus_id = null) {
    try {
        $docx_template = __DIR__ . '/../Reifeneinlagerungsetikett.docx';
        
        // WICHTIG: Lade Drucker-Einstellungen aus der Datenbank (pro Autohaus)
        // Falls keine autohaus_id übergeben wurde, versuche sie aus der Session zu holen
        if (!$autohaus_id && isset($_SESSION['admin_user_id'])) {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            $stmt = $conn->prepare("SELECT autohaus_id FROM admin_users WHERE id = ?");
            $stmt->execute([$_SESSION['admin_user_id']]);
            $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
            $autohaus_id = $user_data['autohaus_id'] ?? null;
        }
        
        // Lade Drucker-Einstellungen aus der Datenbank
        $printer_config = null;
        if ($autohaus_id) {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            $stmt = $conn->prepare("
                SELECT 
                    printer_name, printer_server_ip, printer_server_hostname, 
                    printer_server_user, printer_server_pass, printer_name_server, 
                    printer_name_local, printer_enabled
                FROM autohaus 
                WHERE id = ? AND is_active = 1 AND printer_enabled = 1
            ");
            $stmt->execute([$autohaus_id]);
            $printer_config = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        // Falls keine Drucker-Einstellungen in der DB gefunden, verwende Standard-Werte aus config
        if (!$printer_config || !$printer_config['printer_name_local']) {
            // Fallback auf config/printer.php
            $server_tailscale_ip = PRINTER_SERVER_TAILSCALE_IP;
            $server_hostname = PRINTER_SERVER_HOSTNAME;
            $printer_name_server = defined('PRINTER_NAME_SERVER') ? PRINTER_NAME_SERVER : 'BUEK-PRT17';
            $printer_name_local = defined('PRINTER_NAME_LOCAL') ? PRINTER_NAME_LOCAL : 'PRT17_BUEK-Server';
            $printer_server_user = defined('PRINTER_SERVER_USER') ? PRINTER_SERVER_USER : 'VOLVOKIE\Projekt';
            $printer_server_pass = defined('PRINTER_SERVER_PASS') ? PRINTER_SERVER_PASS : 'start#2025';
        } else {
            // Verwende Drucker-Einstellungen aus der Datenbank
            $server_tailscale_ip = $printer_config['printer_server_ip'] ?: PRINTER_SERVER_TAILSCALE_IP;
            $server_hostname = $printer_config['printer_server_hostname'] ?: PRINTER_SERVER_HOSTNAME;
            $printer_name_server = $printer_config['printer_name_server'] ?: 'BUEK-PRT17';
            $printer_name_local = $printer_config['printer_name_local'] ?: 'PRT17_BUEK-Server';
            $printer_server_user = $printer_config['printer_server_user'] ?: (defined('PRINTER_SERVER_USER') ? PRINTER_SERVER_USER : 'VOLVOKIE\Projekt');
            $printer_server_pass = $printer_config['printer_server_pass'] ?: (defined('PRINTER_SERVER_PASS') ? PRINTER_SERVER_PASS : 'start#2025');
        }
        
        $printer_name_api = $printer_name_local;
        
        $printer_options = [
            // WICHTIG: PRT17_BUEK-Server ist der primäre Drucker
            // Lokaler Name (nach Installation) - höchste Priorität
            'local_name' => $printer_name_local,  // PRT17_BUEK-Server
            
            // Server-Name (wie in Windows Explorer) - für Installation
            'server_name' => $printer_name_server,  // BUEK-PRT17
            'server_tailscale' => "\\\\$server_tailscale_ip\\$printer_name_server",  // \\100.82.207.72\BUEK-PRT17
            'server_hostname' => "\\\\$server_hostname\\$printer_name_server",  // \\buek-s01\BUEK-PRT17
            
            // API-Server UNC-Pfade (primär)
            'api_server_unc' => "\\\\$server_hostname\\$printer_name_api",  // buek-s01\BUEK-PRT17
            'api_server_tailscale' => "\\\\$server_tailscale_ip\\$printer_name_api",  // 100.82.207.72\BUEK-PRT17
            
            // Fallback: labelprinter service (falls BUEK-PRT17 nicht verfügbar)
            'fallback_name' => defined('PRINTER_NAME_FALLBACK') ? PRINTER_NAME_FALLBACK : 'labelprinter service',
            'fallback_server' => "\\\\$server_hostname\\" . (defined('PRINTER_NAME_FALLBACK') ? PRINTER_NAME_FALLBACK : 'labelprinter service'),
            'fallback_tailscale' => "\\\\$server_tailscale_ip\\" . (defined('PRINTER_NAME_FALLBACK') ? PRINTER_NAME_FALLBACK : 'labelprinter service'),
            
            // Lokale IP (falls verfügbar)
            'ip_local' => '172.24.10.239',  // Aktualisiert auf die vom Benutzer angegebene IP
            'ip_local_server' => "\\\\172.24.10.239\\$printer_name_server"
        ];
        
        // Standard-Drucker: Verwende lokalen Namen (PRT17_BUEK-Server)
        $default_printer = $printer_options['local_name'];
        
        // Standard: Verwende lokalen Namen (PRT17_BUEK-Server)
        $printer = $printer_options['local_name'];
        
        if (!file_exists($docx_template)) {
            return ['success' => false, 'message' => 'Word-Dokument nicht gefunden: ' . $docx_template];
        }
        
        // 1. Word-Dokument verarbeiten
        $zip = new ZipArchive();
        $temp_file = sys_get_temp_dir() . '\\etikett_' . uniqid() . '.docx';
        copy($docx_template, $temp_file);
        
        if ($zip->open($temp_file) !== TRUE) {
            return ['success' => false, 'message' => 'Kann DOCX-Datei nicht öffnen'];
        }
        
        $content = $zip->getFromName('word/document.xml');
        if ($content === false) {
            $zip->close();
            return ['success' => false, 'message' => 'Kann document.xml nicht lesen'];
        }
        
        // Platzhalter ersetzen
        $content = str_replace('[name1]', htmlspecialchars($name, ENT_XML1, 'UTF-8'), $content);
        $content = str_replace('[kennzeichen]', htmlspecialchars($kennzeichen, ENT_XML1, 'UTF-8'), $content);
        
        $zip->addFromString('word/document.xml', $content);
        $zip->close();
        
        // 2. PDF generieren (falls PhpOffice verfügbar)
        $pdf_path = null;
        if (class_exists('PhpOffice\PhpWord\IOFactory')) {
            try {
                $phpWord = \PhpOffice\PhpWord\IOFactory::load($temp_file);
                $writer = \PhpOffice\PhpWord\IOFactory::createWriter($phpWord, 'PDF');
                $pdf_path = sys_get_temp_dir() . '\\etikett_' . uniqid() . '.pdf';
                $writer->save($pdf_path);
            } catch (Exception $e) {
                // PDF-Generierung fehlgeschlagen, verwende DOCX
            }
        }
        
        // 3. Prüfe Drucker-Erreichbarkeit (mehrere Methoden)
        $printer_found = false;
        $printer_check_output = [];
        $printer_check_details = [];
        $best_printer = $printer; // Welcher Drucker-Pfad funktioniert am besten
        
        // Methode 1: Prüfe Tailscale IP-Adresse (100.82.207.72) - VPN-Verbindung
        $printer_ip_tailscale = '100.82.207.72';
        $ping_tailscale_cmd = sprintf('ping -n 1 -w 2000 %s 2>&1', escapeshellarg($printer_ip_tailscale));
        exec($ping_tailscale_cmd, $ping_tailscale_output, $ping_tailscale_code);
        $tailscale_reachable = ($ping_tailscale_code === 0);
        $printer_check_details['printer_ip_tailscale_ping'] = [
            'reachable' => $tailscale_reachable,
            'ip' => $printer_ip_tailscale,
            'output' => implode("\n", $ping_tailscale_output)
        ];
        
        // Methode 1b: Prüfe lokale IP-Adresse (172.24.10.229)
        $printer_ip_local = '172.24.10.229';
        $ping_ip_cmd = sprintf('ping -n 1 -w 2000 %s 2>&1', escapeshellarg($printer_ip_local));
        exec($ping_ip_cmd, $ping_ip_output, $ping_ip_code);
        $ip_reachable = ($ping_ip_code === 0);
        $printer_check_details['printer_ip_local_ping'] = [
            'reachable' => $ip_reachable,
            'ip' => $printer_ip_local,
            'output' => implode("\n", $ping_ip_output)
        ];
        
        // Methode 2: Prüfe Server-Erreichbarkeit (buek-s01) - VPN-Verbindung
        $server_name = 'buek-s01';
        $ping_cmd = sprintf('ping -n 1 -w 2000 %s 2>&1', escapeshellarg($server_name));
        exec($ping_cmd, $ping_output, $ping_code);
        $server_reachable = ($ping_code === 0);
        $printer_check_details['server_ping'] = [
            'reachable' => $server_reachable,
            'output' => implode("\n", $ping_output)
        ];
        
        // Methode 3: Prüfe UNC-Pfad-Zugriff (Server)
        $unc_path = "\\\\$server_name";
        $unc_accessible = @file_exists($unc_path) || @is_dir($unc_path);
        $printer_check_details['unc_path'] = [
            'accessible' => $unc_accessible,
            'path' => $unc_path
        ];
        
        // Methode 4: Prüfe UNC-Pfad mit lokaler IP
        $unc_path_ip_local = "\\\\$printer_ip_local";
        $unc_ip_local_accessible = @file_exists($unc_path_ip_local) || @is_dir($unc_path_ip_local);
        $printer_check_details['unc_path_ip_local'] = [
            'accessible' => $unc_ip_local_accessible,
            'path' => $unc_path_ip_local
        ];
        
        // Methode 4b: Prüfe UNC-Pfad mit Tailscale IP
        $unc_path_ip_tailscale = "\\\\$printer_ip_tailscale";
        $unc_ip_tailscale_accessible = @file_exists($unc_path_ip_tailscale) || @is_dir($unc_path_ip_tailscale);
        $printer_check_details['unc_path_ip_tailscale'] = [
            'accessible' => $unc_ip_tailscale_accessible,
            'path' => $unc_path_ip_tailscale
        ];
        
        // Methode 5: Net View - Liste Freigaben (Server)
        $net_cmd = sprintf('net view "%s" 2>&1', $unc_path);
        exec($net_cmd, $net_output, $net_code);
        $printer_in_shares = false;
        if ($net_code === 0 && !empty($net_output)) {
            foreach ($net_output as $line) {
                if (stripos($line, 'BUEK-PRT17') !== false || stripos($line, 'PRT17') !== false) {
                    $printer_in_shares = true;
                    break;
                }
            }
        }
        $printer_check_details['net_view'] = [
            'found' => $printer_in_shares,
            'output' => implode("\n", $net_output)
        ];
        
        // Beste Drucker-Methode wählen
        // Priorität: BUEK-PRT17 über buek-s01 > BUEK-PRT17 über Tailscale IP > Fallback
        // WICHTIG: BUEK-PRT17 hat höchste Priorität, da er im Windows Explorer sichtbar ist
        if ($server_reachable || $tailscale_reachable) {
            // Server ist erreichbar - versuche zuerst BUEK-PRT17 über Hostname
            $best_printer = $printer_options['server_hostname'];  // \\buek-s01\BUEK-PRT17
            $printer_found = true;
        } elseif ($tailscale_reachable || $unc_ip_tailscale_accessible) {
            // Tailscale IP ist erreichbar - versuche BUEK-PRT17 über Tailscale IP
            $best_printer = $printer_options['server_tailscale'];  // \\100.82.207.72\BUEK-PRT17
            $printer_found = true;
        } elseif ($ip_reachable || $unc_ip_local_accessible) {
            // Lokale IP ist erreichbar - versuche BUEK-PRT17 über lokale IP
            $best_printer = $printer_options['ip_local_server'];  // \\172.24.10.239\BUEK-PRT17
            $printer_found = true;
        }
        
        // Verwende die beste verfügbare Methode
        $printer = $best_printer;
        
        // 2.5: Verbinde mit Server über net use (für Authentifizierung)
        // Das ist notwendig, da der Server Admin-Credentials benötigt
        // WICHTIG: net use MUSS erfolgreich sein, damit der Drucker-Port funktioniert!
        $net_use_success = false;
        if ($server_reachable || $tailscale_reachable) {
            // Versuche Verbindung zum Server herzustellen
            // WICHTIG: Passwort OHNE /p: direkt angeben (wie im Test-Script)
            $net_use_cmd = sprintf(
                'net use "\\\\%s" %s /user:%s 2>&1',
                $server_hostname,
                PRINTER_SERVER_PASS ? escapeshellarg(PRINTER_SERVER_PASS) : '',
                escapeshellarg(PRINTER_SERVER_USER)
            );
            exec($net_use_cmd, $net_use_output, $net_use_code);
            
            // Auch mit Tailscale IP versuchen (höhere Priorität)
            $net_use_tailscale_cmd = sprintf(
                'net use "\\\\%s" %s /user:%s 2>&1',
                $server_tailscale_ip,
                PRINTER_SERVER_PASS ? escapeshellarg(PRINTER_SERVER_PASS) : '',
                escapeshellarg(PRINTER_SERVER_USER)
            );
            exec($net_use_tailscale_cmd, $net_use_tailscale_output, $net_use_tailscale_code);
            
            // Prüfe, ob Verbindung erfolgreich war
            $net_use_success = (
                ($net_use_code === 0 && (stripos(implode("\n", $net_use_output), 'erfolgreich') !== false || stripos(implode("\n", $net_use_output), 'bereits verbunden') !== false)) ||
                ($net_use_tailscale_code === 0 && (stripos(implode("\n", $net_use_tailscale_output), 'erfolgreich') !== false || stripos(implode("\n", $net_use_tailscale_output), 'bereits verbunden') !== false))
            );
            
            // WICHTIG: Wenn net use fehlschlägt, versuche es erneut mit expliziter Authentifizierung
            if (!$net_use_success) {
                // Versuche mit expliziter Authentifizierung
                $net_use_retry_cmd = sprintf(
                    'net use "\\\\%s" /user:%s %s 2>&1',
                    $server_tailscale_ip,
                    escapeshellarg(PRINTER_SERVER_USER),
                    PRINTER_SERVER_PASS ? escapeshellarg(PRINTER_SERVER_PASS) : ''
                );
                exec($net_use_retry_cmd, $net_use_retry_output, $net_use_retry_code);
                if ($net_use_retry_code === 0) {
                    $net_use_success = true;
                }
            }
            
            $printer_check_details['net_use'] = [
                'success' => $net_use_success,
                'hostname_code' => $net_use_code,
                'hostname_output' => implode("\n", $net_use_output),
                'tailscale_code' => $net_use_tailscale_code,
                'tailscale_output' => implode("\n", $net_use_tailscale_output),
                'hint' => $net_use_success 
                    ? 'Server-Verbindung erfolgreich hergestellt - Drucker-Port sollte jetzt funktionieren' 
                    : 'WICHTIG: net use fehlgeschlagen! Der Drucker-Port kann ohne Authentifizierung nicht initialisiert werden. Prüfe PRINTER_SERVER_USER und PRINTER_SERVER_PASS in config/printer.php'
            ];
        }
        
        // 2.6: Versuche Drucker lokal zu installieren, falls nicht verfügbar
        // Das ist notwendig, da UNC-Pfade nicht funktionieren, wenn der Drucker nicht lokal installiert ist
        $printer_locally_installed = false;
        $printer_install_attempted = false;
        $local_printer_name = null;  // Speichere den tatsächlichen lokalen Druckernamen
        
        // WICHTIG: Verwende intelligente Drucker-Erkennung
        // Diese Methode verwendet mehrere Erkennungsmethoden und ist zuverlässiger
        $alternative_names = defined('PRINTER_LOCAL_NAMES') ? PRINTER_LOCAL_NAMES : [];
        $detection_result = smartDetectPrinter($printer_name_local, $printer_name_server, $alternative_names);
        
        if ($detection_result['found']) {
            $printer_locally_installed = true;
            $local_printer_name = $detection_result['name'];
            
            // Prüfe Port (muss Netzwerkport sein)
            $port_check = checkPrinterPort($local_printer_name);
            if (!$port_check['valid']) {
                // Port ist ungültig - versuche zu reparieren
                $printer_check_details['port_warning'] = $port_check['error'];
            } else {
                $printer_check_details['port'] = $port_check['port'];
            }
            
            // Prüfe Status
            $status_check = checkPrinterStatus($local_printer_name);
            $printer_check_details['status'] = $status_check;
            
            $printer_check_details['detection_method'] = $detection_result['method'];
            $printer_check_details['detection_details'] = $detection_result['details'];
        } else {
            // Fallback: Alte Methode (für Kompatibilität)
            $check_ps_cmd = 'powershell -NoProfile -ExecutionPolicy Bypass -Command "Get-Printer | Select-Object -ExpandProperty Name" 2>&1';
            exec($check_ps_cmd, $check_ps_output, $check_ps_code);
            
            foreach ($check_ps_output as $line) {
                $line_trimmed = trim($line);
                if (empty($line_trimmed)) continue;
                
                if (stripos($line_trimmed, $printer_name_local) !== false) {
                    $printer_locally_installed = true;
                    $local_printer_name = $line_trimmed;
                    break;
                }
            }
        }
        
        // Wenn immer noch nicht gefunden, versuche wmic (falls PowerShell nicht verfügbar)
        if (!$printer_locally_installed) {
            $check_local_cmd = 'wmic printer get name 2>&1';
            exec($check_local_cmd, $check_local_output, $check_local_code);
            
            // Suche NUR nach PRT17_BUEK-Server - keine anderen Drucker!
            foreach ($check_local_output as $line) {
                $line_trimmed = trim($line);
                // Überspringe Header-Zeilen
                if (empty($line_trimmed) || stripos($line_trimmed, 'Name') !== false || 
                    stripos($line_trimmed, 'Default') !== false || stripos($line_trimmed, 'Status') !== false) {
                    continue;
                }
                
                // NUR PRT17_BUEK-Server akzeptieren
                if (stripos($line_trimmed, 'PRT17_BUEK-Server') !== false) {
                    $printer_locally_installed = true;
                    $local_printer_name = $line_trimmed;
                    break;
                }
            }
        }
        
        // Wenn nicht lokal installiert, versuche Installation
        if (!$printer_locally_installed && ($server_reachable || $tailscale_reachable || $net_use_success)) {
                $printer_install_attempted = true;
                $install_outputs = [];
                $install_methods = [];
                
                // WICHTIG: Auf dem Server heißt der Drucker "BUEK-PRT17" (wie in Windows Explorer)
                // Nach Installation heißt er lokal "PRT17_BUEK-Server" (tatsächlicher Name!)
                $printers_to_install = [];
                if ($server_reachable || $tailscale_reachable) {
                    // Priorität 1: BUEK-PRT17 über Tailscale IP (Server-Name, wie in Windows Explorer)
                    $printers_to_install[] = [
                        'name' => $printer_name_local,  // PRT17_BUEK-Server (lokaler Name nach Installation)
                        'unc' => $printer_options['server_tailscale'],  // \\100.82.207.72\BUEK-PRT17
                        'priority' => 1
                    ];
                    // Priorität 2: BUEK-PRT17 über Hostname
                    $printers_to_install[] = [
                        'name' => $printer_name_local,  // PRT17_BUEK-Server
                        'unc' => $printer_options['server_hostname'],  // \\buek-s01\BUEK-PRT17
                        'priority' => 2
                    ];
                }
                
                // Sortiere nach Priorität
                usort($printers_to_install, function($a, $b) {
                    return $a['priority'] <=> $b['priority'];
                });
                
                foreach ($printers_to_install as $printer_info) {
                    if ($printer_locally_installed) break;
                    
                    $printer_to_install = $printer_info['unc'];
                    $printer_display_name = $printer_info['name'];
                    
                    // Methode 1: PowerShell Add-Printer mit explizitem Port (wie im Test-Script)
                    // Erstelle zuerst einen Netzwerkport, dann installiere Drucker
                    $port_name = $printer_to_install;
                    $printer_unc_escaped = str_replace("'", "''", $port_name);
                    
                    // Versuche Installation mit PowerShell (wie im Test-Script erfolgreich)
                    $install_cmd = sprintf(
                        'powershell -NoProfile -ExecutionPolicy Bypass -Command "try { Add-Printer -Name \'%s\' -DriverName \'Bar Code Printer TTP068-50\' -PortName \'%s\' -ErrorAction Stop; Write-Host \'SUCCESS\' } catch { Write-Host $_.Exception.Message }" 2>&1',
                        str_replace("'", "''", $printer_display_name),
                        $printer_unc_escaped
                    );
                    exec($install_cmd, $install_output, $install_code);
                    $install_outputs[] = "Methode 1 (PowerShell Add-Printer mit Port): " . implode("\n", $install_output);
                    $install_methods[] = "PowerShell Add-Printer: $printer_display_name → $printer_to_install";
                    
                    // Prüfe sofort, ob erfolgreich
                    sleep(2);
                    exec($check_local_cmd, $check_local_output2, $check_local_code2);
                    foreach ($check_local_output2 as $line) {
                        $line_trimmed = trim($line);
                        if (empty($line_trimmed) || stripos($line_trimmed, 'Name') !== false || 
                            stripos($line_trimmed, 'Default') !== false || stripos($line_trimmed, 'Status') !== false) {
                            continue;
                        }
                        // Prüfe auf Druckernamen (PRT17_BUEK-Server hat Priorität!)
                        if (stripos($line_trimmed, 'PRT17_BUEK-Server') !== false ||
                            stripos($line_trimmed, 'PRT17') !== false ||
                            stripos($line_trimmed, 'BUEK-PRT17') !== false ||
                            stripos($line_trimmed, $printer_display_name) !== false ||
                            stripos($line_trimmed, 'LABELPRINTER') !== false ||
                            stripos($line_trimmed, 'LABEL') !== false) {
                            // Prüfe Port - muss Netzwerkport sein, nicht LPT1
                            $port_check_cmd = sprintf('powershell -NoProfile -ExecutionPolicy Bypass -Command "try { $p = Get-Printer -Name \'%s\' -ErrorAction Stop; Write-Host $p.PortName } catch { }" 2>&1', str_replace("'", "''", $line_trimmed));
                            exec($port_check_cmd, $port_check_output, $port_check_code);
                            $port_name_check = implode("", $port_check_output);
                            
                            if (stripos($port_name_check, 'LPT') === false && !empty(trim($port_name_check))) {
                                // Port ist korrekt (nicht LPT1)
                                $printer_locally_installed = true;
                                $local_printer_name = $line_trimmed;
                                break 2; // Beide Schleifen verlassen
                            } else {
                                // Port ist falsch (LPT1) - lösche und versuche erneut
                                $delete_cmd = sprintf('rundll32 printui.dll,PrintUIEntry /dl /n "%s" 2>&1', $line_trimmed);
                                exec($delete_cmd, $delete_output, $delete_code);
                                $install_outputs[] = "Drucker gelöscht (falscher Port LPT1): $line_trimmed";
                            }
                        }
                    }
                    
                    // Methode 2: PowerShell Add-Printer mit ConnectionName (Fallback)
                    if (!$printer_locally_installed) {
                        $install_cmd2 = sprintf(
                            'powershell -NoProfile -ExecutionPolicy Bypass -Command "$ErrorActionPreference = \'Stop\'; try { Add-Printer -ConnectionName \'%s\' -ErrorAction Stop; Write-Host \'SUCCESS\' } catch { Write-Host $_.Exception.Message }" 2>&1',
                            $printer_unc_escaped
                        );
                        exec($install_cmd2, $install_output2, $install_code2);
                        $install_outputs[] = "Methode 2 (PowerShell Add-Printer ConnectionName): " . implode("\n", $install_output2);
                        $install_methods[] = "PowerShell Add-Printer ConnectionName: $printer_to_install";
                        sleep(2);
                        
                        // Prüfe erneut
                        exec($check_local_cmd, $check_local_output3, $check_local_code3);
                        foreach ($check_local_output3 as $line) {
                            $line_trimmed = trim($line);
                            if (empty($line_trimmed) || stripos($line_trimmed, 'Name') !== false || 
                                stripos($line_trimmed, 'Default') !== false || stripos($line_trimmed, 'Status') !== false) {
                                continue;
                            }
                            if (stripos($line_trimmed, $printer_display_name) !== false ||
                                stripos($line_trimmed, 'LABELPRINTER') !== false ||
                                stripos($line_trimmed, 'LABEL') !== false) {
                                // Prüfe Port erneut
                                $port_check_cmd2 = sprintf('powershell -NoProfile -ExecutionPolicy Bypass -Command "try { $p = Get-Printer -Name \'%s\' -ErrorAction Stop; Write-Host $p.PortName } catch { }" 2>&1', str_replace("'", "''", $line_trimmed));
                                exec($port_check_cmd2, $port_check_output2, $port_check_code2);
                                $port_name_check2 = implode("", $port_check_output2);
                                
                                if (stripos($port_name_check2, 'LPT') === false && !empty(trim($port_name_check2))) {
                                    $printer_locally_installed = true;
                                    $local_printer_name = $line_trimmed;
                                    break 2;
                                }
                            }
                        }
                    }
                    
                    // Methode 3: Rundll32 (Windows Drucker-Installation) - Fallback
                    if (!$printer_locally_installed) {
                        $install_cmd3 = sprintf(
                            'rundll32 printui.dll,PrintUIEntry /in /n "%s" 2>&1',
                            $printer_to_install
                        );
                        exec($install_cmd3, $install_output3, $install_code3);
                        $install_outputs[] = "Methode 3 (Rundll32): " . implode("\n", $install_output3);
                        $install_methods[] = "Rundll32: $printer_to_install";
                        
                        sleep(2);
                        exec($check_local_cmd, $check_local_output4, $check_local_code4);
                        foreach ($check_local_output4 as $line) {
                            $line_trimmed = trim($line);
                            if (empty($line_trimmed) || stripos($line_trimmed, 'Name') !== false || 
                                stripos($line_trimmed, 'Default') !== false || stripos($line_trimmed, 'Status') !== false) {
                                continue;
                            }
                            if (stripos($line_trimmed, 'PRT17_BUEK-Server') !== false ||
                                stripos($line_trimmed, 'PRT17') !== false ||
                                stripos($line_trimmed, 'BUEK-PRT17') !== false ||
                                stripos($line_trimmed, $printer_display_name) !== false ||
                                stripos($line_trimmed, 'LABELPRINTER') !== false ||
                                stripos($line_trimmed, 'LABEL') !== false) {
                                $printer_locally_installed = true;
                                $local_printer_name = $line_trimmed;
                                break 2;
                            }
                        }
                    }
                    
                    // Methode 3: PowerShell mit expliziter UNC-Prüfung
                    if (!$printer_locally_installed && $net_use_success) {
                        $install_cmd3 = sprintf(
                            'powershell -NoProfile -ExecutionPolicy Bypass -Command "$ErrorActionPreference = \'Stop\'; $printer = \'%s\'; try { if (Test-Path $printer) { Add-Printer -ConnectionName $printer -ErrorAction Stop; Write-Host \'SUCCESS\' } else { Write-Host \'UNC path not accessible\' } } catch { Write-Host $_.Exception.Message }" 2>&1',
                            $printer_unc_escaped
                        );
                        exec($install_cmd3, $install_output3, $install_code3);
                        $install_outputs[] = "Methode 3 (PowerShell mit UNC-Prüfung): " . implode("\n", $install_output3);
                        $install_methods[] = "PowerShell UNC-Prüfung: $printer_to_install";
                        
                        sleep(1);
                        exec($check_local_cmd, $check_local_output4, $check_local_code4);
                        foreach ($check_local_output4 as $line) {
                            $line_trimmed = trim($line);
                            if (empty($line_trimmed) || stripos($line_trimmed, 'Name') !== false || 
                                stripos($line_trimmed, 'Default') !== false || stripos($line_trimmed, 'Status') !== false) {
                                continue;
                            }
                            // Priorität: PRT17_BUEK-Server zuerst
                            if (stripos($line_trimmed, 'PRT17_BUEK-Server') !== false ||
                                stripos($line_trimmed, 'PRT17') !== false ||
                                stripos($line_trimmed, 'BUEK-PRT17') !== false) {
                                $printer_locally_installed = true;
                                $local_printer_name = $line_trimmed;
                                break 2;
                            }
                            foreach (PRINTER_LOCAL_NAMES as $search_name) {
                                if (stripos($line_trimmed, $search_name) !== false) {
                                    $printer_locally_installed = true;
                                    $local_printer_name = $line_trimmed;
                                    break 2;
                                }
                            }
                        }
                    }
                }
                
                // Finale Prüfung
                if (!$printer_locally_installed) {
                    sleep(2); // Etwas länger warten
                    exec($check_local_cmd, $check_local_output_final, $check_local_code_final);
                    foreach ($check_local_output_final as $line) {
                        $line_trimmed = trim($line);
                        if (empty($line_trimmed) || stripos($line_trimmed, 'Name') !== false || 
                            stripos($line_trimmed, 'Default') !== false || stripos($line_trimmed, 'Status') !== false) {
                            continue;
                        }
                        // Priorität: PRT17_BUEK-Server zuerst
                        if (stripos($line_trimmed, 'PRT17_BUEK-Server') !== false ||
                            stripos($line_trimmed, 'PRT17') !== false ||
                            stripos($line_trimmed, 'BUEK-PRT17') !== false) {
                            $printer_locally_installed = true;
                            $local_printer_name = $line_trimmed;
                            break;
                        }
                        foreach (PRINTER_LOCAL_NAMES as $search_name) {
                            if (stripos($line_trimmed, $search_name) !== false) {
                                $printer_locally_installed = true;
                                $local_printer_name = $line_trimmed;
                                break 2;
                            }
                        }
                    }
                }
                
                // Manuelle Installationsanleitung generieren
                $printer_name_server_install = defined('PRINTER_NAME_SERVER') ? PRINTER_NAME_SERVER : 'labelprinter service';
                $manual_install_cmd = sprintf(
                    'rundll32 printui.dll,PrintUIEntry /in /n "\\\\%s\\%s"',
                    $server_tailscale_ip,
                    $printer_name_server_install
                );
                $manual_install_cmd2 = sprintf(
                    'rundll32 printui.dll,PrintUIEntry /in /n "\\\\%s\\%s"',
                    $server_tailscale_ip,
                    'LABEL_Printer_TTP068-50'
                );
                
                $printer_check_details['printer_installed'] = [
                    'success' => $printer_locally_installed,
                    'attempted' => true,
                    'methods_tried' => $install_methods,
                    'output' => implode("\n\n", $install_outputs),
                    'local_printer_name' => $local_printer_name,
                    'manual_install_commands' => [
                        'labelprinter service' => $manual_install_cmd,
                        'LABEL_Printer_TTP068-50' => $manual_install_cmd2,
                        'windows_explorer' => "Windows Explorer → \\\\$server_tailscale_ip → Rechtsklick auf '$printer_name_server_install' → Drucker verbinden"
                    ],
                    'hint' => $printer_locally_installed 
                        ? 'Drucker erfolgreich installiert! (Name: ' . ($local_printer_name ?: 'unbekannt') . ')' 
                        : 'Drucker konnte nicht automatisch installiert werden. Bitte manuell installieren über Windows Explorer oder mit den bereitgestellten Befehlen.'
                ];
        } else {
            // Drucker ist bereits lokal installiert
            $printer_check_details['printer_installed'] = [
                'success' => true,
                'attempted' => false,
                'local_printer_name' => $local_printer_name,
                'message' => 'Drucker ist bereits lokal installiert' . ($local_printer_name ? ' (Name: ' . $local_printer_name . ')' : '')
            ];
        }
        
        // 3. Drucken (DOCX oder PDF)
        $file_to_print = $pdf_path ?? $temp_file;
        
        // Prüfe, ob Datei existiert
        if (!file_exists($file_to_print)) {
            @unlink($temp_file);
            if ($pdf_path) @unlink($pdf_path);
            return [
                'success' => false,
                'message' => 'Datei zum Drucken nicht gefunden: ' . $file_to_print
            ];
        }
        
        // Prüfe Dateigröße
        $file_size = filesize($file_to_print);
        if ($file_size === false || $file_size === 0) {
            @unlink($temp_file);
            if ($pdf_path) @unlink($pdf_path);
            return [
                'success' => false,
                'message' => 'Datei ist leer oder kann nicht gelesen werden'
            ];
        }
        
        // Versuche verschiedene Druck-Methoden
        $print_success = false;
        $print_error = '';
        $print_output = '';
        $print_command = '';
        $print_method = '';
        
        // WICHTIG: Verwende PDF, da DOCX möglicherweise nicht direkt gedruckt werden kann
        // Falls kein PDF verfügbar, versuche trotzdem DOCX
        $file_to_print_final = ($pdf_path && file_exists($pdf_path)) ? $pdf_path : $file_to_print;
        
        // WICHTIG: NUR PRT17_BUEK-Server verwenden - KEINE Fallbacks!
        if (!$printer_locally_installed || !$local_printer_name) {
            @unlink($temp_file);
            if ($pdf_path) @unlink($pdf_path);
            return [
                'success' => false,
                'message' => 'FEHLER: PRT17_BUEK-Server nicht gefunden!',
                'hint' => 'Der Drucker PRT17_BUEK-Server MUSS lokal installiert sein. Bitte installiere ihn manuell über Windows Explorer: \\\\100.82.207.72\\BUEK-PRT17',
                'debug' => [
                    'printer_locally_installed' => $printer_locally_installed,
                    'local_printer_name' => $local_printer_name,
                    'available_printers' => $check_ps_output
                ]
            ];
        }
        
        // Verwende NUR den lokalen Namen von PRT17_BUEK-Server
        $printer_for_print = $local_printer_name;
        $printer_check_details['printer_used'] = [
            'type' => 'local',
            'name' => $local_printer_name,
            'note' => 'Verwende NUR PRT17_BUEK-Server - keine Fallbacks!'
        ];
        
        // Methode 1: PowerShell Out-Printer (funktioniert besser als Windows print)
        // WICHTIG: Windows print gibt "Gerät konnte nicht initialisiert werden" zurück,
        // aber PowerShell Out-Printer funktioniert!
        $all_outputs = [];
        $all_success = true;
        $first_output = '';
        
        // Escaped Dateipfad für PowerShell
        $ps_file_escaped = str_replace("'", "''", $file_to_print_final);
        $ps_file_escaped = str_replace('\\', '/', $ps_file_escaped);
        $ps_printer_escaped = str_replace("'", "''", $printer_for_print);
        
        // WICHTIG: Für PDF/DOCX muss die Datei mit dem Standard-Drucker-Handler gedruckt werden
        // Out-Printer sendet die Datei als Text/Rohdaten, was zu "wirrem Zeug" führt
        // Verwende stattdessen Windows print-Befehl, der PDF/DOCX richtig verarbeitet
        
        for ($i = 0; $i < $copies; $i++) {
            // WICHTIG: Verwende PowerShell PrintTo, um Drucker-Einstellungen zu berücksichtigen
            // PrintTo verwendet die konfigurierten Drucker-Einstellungen (Papiergröße, Ausrichtung, etc.)
            $command = sprintf(
                'powershell -NoProfile -ExecutionPolicy Bypass -Command "$file = \'%s\'; $printer = \'%s\'; try { $psi = New-Object System.Diagnostics.ProcessStartInfo; $psi.FileName = $file; $psi.Verb = \'PrintTo\'; $psi.Arguments = \'\"\' + $printer + \'\"\'; $psi.UseShellExecute = $true; $psi.CreateNoWindow = $true; $process = [System.Diagnostics.Process]::Start($psi); if ($process) { Start-Sleep -Milliseconds 800; Write-Host \"Druck gestartet\" } else { throw \"Prozess konnte nicht gestartet werden\" } } catch { Write-Host \"Fehler: $_\"; exit 1 }" 2>&1',
                $ps_file_escaped,
                $ps_printer_escaped
            );
            
            exec($command, $output, $return_code);
            $output_text = implode("\n", $output);
            if (function_exists('mb_convert_encoding')) {
                $output_text = @mb_convert_encoding($output_text, 'UTF-8', 'Windows-1252');
            }
            $output_text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $output_text);
            
            // Falls PrintTo nicht funktioniert, versuche Windows print als Fallback
            if ($return_code !== 0 || stripos($output_text, 'Fehler') !== false || stripos($output_text, 'error') !== false) {
                // Fallback: Windows print-Befehl (verwendet Standard-Drucker-Einstellungen)
                $win_print_cmd = sprintf(
                    'print /D:"%s" "%s" 2>&1',
                    $printer_for_print,
                    $file_to_print_final
                );
                exec($win_print_cmd, $win_output, $win_return_code);
                
                if ($win_return_code === 0 && stripos(implode("\n", $win_output), 'konnte nicht initialisiert werden') === false) {
                    $output = $win_output;
                    $return_code = 0;
                    $output_text = implode("\n", $win_output);
                    if (function_exists('mb_convert_encoding')) {
                        $output_text = @mb_convert_encoding($output_text, 'UTF-8', 'Windows-1252');
                    }
                    $print_method = 'Windows print-Befehl (Fallback)';
                } else {
                    // Beide Methoden fehlgeschlagen
                    $all_success = false;
                }
            } else {
                $print_method = 'PowerShell PrintTo (mit Drucker-Einstellungen)';
            }
            
            if ($i === 0) {
                $first_output = $output_text;
            }
            $all_outputs[] = $output_text;
            
            // Prüfe auf Fehler
            $has_error = (
                stripos($output_text, 'konnte nicht initialisiert werden') !== false ||
                stripos($output_text, 'could not be initialized') !== false ||
                stripos($output_text, 'error') !== false ||
                stripos($output_text, 'fehler') !== false ||
                $return_code !== 0
            );
            
            if ($has_error) {
                $all_success = false;
                break; // Stoppe bei Fehler
            }
            
            // Kurze Pause zwischen Kopien
            if ($i < $copies - 1) {
                usleep(1000000); // 1 Sekunde zwischen Kopien
            }
        }
        
        $output = $all_outputs;
        $return_code = $all_success ? 0 : 1;
        
        $output_text = implode("\n", $all_outputs);
        if (function_exists('mb_convert_encoding')) {
            $output_text = @mb_convert_encoding($output_text, 'UTF-8', 'Windows-1252');
        }
        $output_text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $output_text);
        
        if (empty($print_method)) {
            $print_method = 'PowerShell PrintTo (wiederholt für ' . $copies . ' Kopien)';
        }
        $print_command = sprintf('PowerShell PrintTo -Printer "%s" "%s" (x%d Kopien)', $printer_for_print, $file_to_print_final, $copies);
        $print_output = $output_text;
        
        // Windows print gibt 0 zurück, auch wenn der Drucker nicht erreichbar ist
        // Prüfe daher auch die Ausgabe auf Fehler
        $has_error_in_output = (
            stripos($output_text, 'konnte nicht initialisiert werden') !== false ||
            stripos($output_text, 'could not be initialized') !== false ||
            stripos($output_text, 'error') !== false ||
            stripos($output_text, 'fehler') !== false ||
            stripos($output_text, 'nicht gefunden') !== false ||
            stripos($output_text, 'not found') !== false ||
            stripos($output_text, 'zugriff verweigert') !== false ||
            stripos($output_text, 'access denied') !== false
        );
        
        // WICHTIG: Wenn "konnte nicht initialisiert werden" und Drucker nicht lokal installiert ist,
        // dann ist das ein kritisches Problem
        if ($has_error_in_output && stripos($output_text, 'konnte nicht initialisiert werden') !== false) {
            if (!$printer_locally_installed) {
                $manual_install_hint = "KRITISCH: Drucker konnte nicht initialisiert werden, da er nicht lokal installiert ist.\n\n";
                $manual_install_hint .= "MANUELLE INSTALLATION ERFORDERLICH:\n";
                $manual_install_hint .= "1. Windows Explorer öffnen\n";
                $manual_install_hint .= "2. In Adresszeile eingeben: \\\\$server_tailscale_ip\n";
                $manual_install_hint .= "3. Rechtsklick auf 'BUEK-PRT17' → 'Drucker verbinden'\n\n";
                $manual_install_hint .= "ODER per Kommandozeile (als Administrator):\n";
                $manual_install_hint .= "rundll32 printui.dll,PrintUIEntry /in /n \"\\\\$server_tailscale_ip\\BUEK-PRT17\"\n\n";
                $manual_install_hint .= "Nach der Installation bitte erneut versuchen.";
                $print_error = $manual_install_hint;
            } else {
                // Drucker ist lokal installiert, aber Port kann nicht initialisiert werden
                // Mögliche Ursachen:
                // 1. net use fehlgeschlagen (keine Authentifizierung für Netzwerkport)
                // 2. Drucker offline trotz "Normal" Status
                // 3. Port nicht erreichbar
                
                $error_details = [];
                if (!$net_use_success) {
                    $error_details[] = "net use fehlgeschlagen - Netzwerkport benötigt Authentifizierung";
                }
                $error_details[] = "Drucker lokal installiert: $local_printer_name";
                $error_details[] = "Port: " . ($printer_check_details['printer_used']['path'] ?? 'unbekannt');
                
                $print_error = "Drucker konnte nicht initialisiert werden, obwohl er lokal installiert ist.\n";
                $print_error .= "Mögliche Ursachen:\n";
                $print_error .= "- " . implode("\n- ", $error_details) . "\n";
                $print_error .= "- Drucker offline oder nicht erreichbar\n";
                $print_error .= "- Netzwerkproblem oder Berechtigungsproblem";
                
                // Versuche, den Drucker online zu setzen
                if ($local_printer_name) {
                    $set_online_cmd = sprintf(
                        'powershell -NoProfile -ExecutionPolicy Bypass -Command "try { Set-Printer -Name \'%s\' -WorkOffline $false -ErrorAction Stop; Write-Host \"Drucker online gesetzt\" } catch { Write-Host \"Fehler: $_\" }" 2>&1',
                        str_replace("'", "''", $local_printer_name)
                    );
                    exec($set_online_cmd, $set_online_output, $set_online_code);
                    if ($set_online_code === 0) {
                        $print_error .= "\n\nVersucht, Drucker online zu setzen: " . implode(" ", $set_online_output);
                    }
                }
            }
            $print_success = false;
        }
        
        // Prüfe, ob der Druckauftrag tatsächlich in der Warteschlange gelandet ist
        $job_in_queue = false;
        if ($return_code === 0 && !$has_error_in_output) {
            // Warte kurz, damit der Auftrag in der Warteschlange erscheint
            sleep(2);
            
            // Prüfe Druckerwarteschlange mit PowerShell (genauer als wmic)
            $queue_check_cmd = sprintf(
                'powershell -NoProfile -ExecutionPolicy Bypass -Command "try { $jobs = Get-PrintJob -PrinterName \'%s\' -ErrorAction SilentlyContinue | Sort-Object SubmittedTime -Descending | Select-Object -First 5; if ($jobs) { Write-Host \"GEFUNDEN: $($jobs.Count) Auftrag(e)\"; foreach ($job in $jobs) { Write-Host \"ID $($job.Id): Status=$($job.JobStatus), Zeit=$($job.SubmittedTime)\" } } else { Write-Host \"KEINE_AUFTRAGE\" } } catch { Write-Host \"FEHLER: $_\" }" 2>&1',
                str_replace("'", "''", $printer_for_print)
            );
            exec($queue_check_cmd, $queue_output, $queue_code);
            
            $queue_text = implode("\n", $queue_output);
            if (function_exists('mb_convert_encoding')) {
                $queue_text = @mb_convert_encoding($queue_text, 'UTF-8', 'Windows-1252');
            }
            
            // Prüfe, ob Aufträge gefunden wurden
            if (stripos($queue_text, 'KEINE_AUFTRAGE') === false && stripos($queue_text, 'FEHLER:') === false && stripos($queue_text, 'GEFUNDEN:') !== false) {
                // Es wurden Aufträge gefunden
                $job_in_queue = true;
                $print_success = true;
            } else {
                // Kein Auftrag in Warteschlange
                // WICHTIG: Bei PowerShell Out-Printer kann es sein, dass der Auftrag sofort verarbeitet wird
                // oder dass der Drucker den Auftrag verwirft, wenn er das Format nicht unterstützt
                $print_success = true; // Befehl war erfolgreich (Return Code 0)
                $print_error = "WARNUNG: PowerShell Out-Printer erfolgreich (Code: 0), aber kein Auftrag in Warteschlange gefunden. Mögliche Ursachen:\n- Drucker verarbeitet Auftrag sofort (keine Warteschlange)\n- Drucker unterstützt das Dateiformat nicht (PDF/DOCX)\n- Drucker ist nicht erreichbar oder offline\n\nBitte prüfen Sie manuell, ob der Drucker gedruckt hat.";
            }
        } else {
            $print_error = "Print-Befehl fehlgeschlagen (Code: $return_code): $output_text";
            $print_success = false;
        }
        
        // KEINE Methode 2 mehr - NUR PRT17_BUEK-Server verwenden!
        // Wenn der erste Versuch fehlgeschlagen ist, gibt es einen Fehler zurück
        // Keine Fallback-Drucker mehr!
        
        // Methode 3: PowerShell mit Start-Process (nur wenn Methode 1+2 fehlschlagen)
        // ACHTUNG: Diese Methode funktioniert nur mit lokal installierten Druckern!
        if (!$print_success && $printer_found) {
            // Versuche mit Start-Process (öffnet Standard-Druckdialog)
            $ps_file_escaped = str_replace("'", "''", $file_to_print_final);
            $ps_file_escaped = str_replace('\\', '/', $ps_file_escaped);
            
            $ps_command = sprintf(
                'powershell -NoProfile -ExecutionPolicy Bypass -Command "Start-Process -FilePath \'%s\' -Verb Print -WindowStyle Hidden" 2>&1',
                $ps_file_escaped
            );
            
            exec($ps_command, $ps_output, $ps_return_code);
            
            $ps_output_text = implode("\n", $ps_output);
            if (function_exists('mb_convert_encoding')) {
                $ps_output_text = @mb_convert_encoding($ps_output_text, 'UTF-8', 'Windows-1252');
            }
            $ps_output_text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $ps_output_text);
            
            if ($ps_return_code === 0 && stripos($ps_output_text, 'error') === false) {
                $print_success = true;
                $print_command = $ps_command;
                $print_output = $ps_output_text;
                $print_method = 'PowerShell Start-Process';
            } else {
                $print_error .= " | PowerShell fehlgeschlagen (Code: $ps_return_code): $ps_output_text";
            }
        }
        
        // Methode 3: Direktes Senden an Drucker mit copy-Befehl (für Textdateien, hier als Fallback)
        if (!$print_success && $pdf_path) {
            // Versuche mit copy-Befehl (funktioniert nur bei bestimmten Druckern)
            $copy_command = sprintf(
                'copy /B "%s" "%s" 2>&1',
                $pdf_path,
                $printer
            );
            
            exec($copy_command, $copy_output, $copy_return_code);
            
            $copy_output_text = implode("\n", $copy_output);
            if (function_exists('mb_convert_encoding')) {
                $copy_output_text = @mb_convert_encoding($copy_output_text, 'UTF-8', 'Windows-1252');
            }
            $copy_output_text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $copy_output_text);
            
            if ($copy_return_code === 0) {
                $print_success = true;
                $print_command = $copy_command;
                $print_output = $copy_output_text;
            } else {
                $print_error .= " | Copy-Befehl fehlgeschlagen (Code: $copy_return_code): $copy_output_text";
            }
        }
        
        // Aufräumen (Dateien behalten für Debugging, falls Druck fehlgeschlagen)
        if ($print_success) {
            @unlink($temp_file);
            if ($pdf_path) @unlink($pdf_path);
        } else {
            // Bei Fehler: Dateien behalten für manuelle Prüfung
            // @unlink($temp_file);
            // if ($pdf_path) @unlink($pdf_path);
        }
        
        if ($print_success) {
            $message = "Etikett erfolgreich gedruckt ($copies Exemplar(e))";
            if ($job_in_queue) {
                $message .= " - Auftrag in Warteschlange bestätigt";
            } else {
                $message .= " - WICHTIG: Bitte prüfen Sie, ob der Drucker tatsächlich gedruckt hat!";
            }
            
            return [
                'success' => true,
                'message' => $message,
                'name' => $name,
                'kennzeichen' => $kennzeichen,
                'copies' => $copies,
                'command' => $print_command,
                'output' => $print_output,
                'file_printed' => $file_to_print_final,
                'file_size' => $file_size,
                'printer' => $printer,
                'printer_found' => $printer_found,
                'printer_check_details' => $printer_check_details,
                'file_type' => $pdf_path ? 'PDF' : 'DOCX',
                'method_used' => $print_method,
                'job_in_queue' => $job_in_queue,
                'warning' => !$job_in_queue ? 'Druckauftrag möglicherweise nicht beim Drucker angekommen - bitte manuell prüfen!' : null
            ];
        } else {
            // Erweitere Fehlermeldung mit Installationshinweisen
            $error_message = 'Druck fehlgeschlagen - Bitte prüfen Sie die Drucker-Verbindung';
            $hint_message = 'Prüfen Sie: 1) Ist der Server erreichbar? (' . ($printer_check_details['server_ping']['reachable'] ? 'Ja' : 'Nein') . ') 2) Ist der UNC-Pfad zugänglich? (' . ($printer_check_details['unc_path']['accessible'] ? 'Ja' : 'Nein') . ') 3) Wurde der Drucker in net view gefunden? (' . ($printer_check_details['net_view']['found'] ? 'Ja' : 'Nein') . ') 4) Unterstützt der Drucker PDF/DOCX? 5) Ist der Drucker online und bereit?';
            
            // Wenn Drucker nicht lokal installiert ist, füge Installationshinweise hinzu
            if (isset($printer_check_details['printer_installed']) && 
                !$printer_check_details['printer_installed']['success'] &&
                isset($printer_check_details['printer_installed']['manual_install_commands'])) {
                $error_message = 'Druck fehlgeschlagen - Drucker muss lokal installiert werden';
                $hint_message = "MANUELLE INSTALLATION ERFORDERLICH:\n\n";
                $hint_message .= "Option 1 - Windows Explorer:\n";
                $hint_message .= $printer_check_details['printer_installed']['manual_install_commands']['windows_explorer'] . "\n\n";
                $hint_message .= "Option 2 - Kommandozeile (als Administrator):\n";
                $hint_message .= "BUEK-PRT17: " . $printer_check_details['printer_installed']['manual_install_commands']['BUEK-PRT17'] . "\n";
                $hint_message .= "LABEL_Printer_TTP068-50: " . $printer_check_details['printer_installed']['manual_install_commands']['LABEL_Printer_TTP068-50'] . "\n\n";
                $hint_message .= "Nach der Installation bitte erneut versuchen.";
            }
            
            return [
                'success' => false,
                'message' => $error_message,
                'error' => $print_error,
                'command' => $print_command,
                'output' => $print_output,
                'file_path' => $file_to_print,
                'file_size' => $file_size,
                'printer' => $printer,
                'printer_found' => $printer_found,
                'printer_check_details' => $printer_check_details,
                'file_type' => $pdf_path ? 'PDF' : 'DOCX',
                'method_used' => $print_method,
                'hint' => $hint_message
            ];
        }
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Fehler beim Drucken: ' . $e->getMessage(),
            'error' => $e->getTraceAsString()
        ];
    }
}

// API-Handler
try {
    // Prüfe Request-Methode und Parameter
    $request_method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    $action = $_POST['action'] ?? null;
    
    if ($request_method === 'POST' && $action === 'print_etikett') {
        // JETZT Session-Prüfung durchführen (nur bei POST-Requests mit action)
        if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
            http_response_code(401);
            ob_clean();
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Nicht autorisiert - Bitte anmelden',
                'debug' => [
                    'session_status' => session_status(),
                    'session_active' => isset($_SESSION),
                    'admin_logged_in' => isset($_SESSION['admin_logged_in']) ? $_SESSION['admin_logged_in'] : 'nicht gesetzt'
                ]
            ]);
            ob_end_flush();
            exit;
        }
        
        // Alle bisherigen Ausgaben löschen
        ob_clean();
        
        $name = $_POST['name'] ?? '';
        $kennzeichen = $_POST['kennzeichen'] ?? '';
        $copies = (int)($_POST['copies'] ?? 6);
        $autohaus_id = isset($_POST['autohaus_id']) ? (int)$_POST['autohaus_id'] : null;
        
        // Falls keine autohaus_id übergeben wurde, versuche sie aus dem Appointment zu holen
        if (!$autohaus_id && isset($_POST['appointment_id'])) {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            $stmt = $conn->prepare("SELECT autohaus_id FROM appointments WHERE id = ?");
            $stmt->execute([$_POST['appointment_id']]);
            $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
            $autohaus_id = $appointment['autohaus_id'] ?? null;
        }
        
        if (empty($name) || empty($kennzeichen)) {
            $response = [
                'success' => false,
                'message' => 'Name und Kennzeichen sind erforderlich'
            ];
        } else {
            $response = printEtikett($name, $kennzeichen, $copies, $autohaus_id);
        }
        
        // Bereinige Response für JSON
        $response = cleanForJson($response);
        
        $json_options = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_INVALID_UTF8_IGNORE;
        $json_output = json_encode($response, $json_options);
        
        if ($json_output === false) {
            throw new Exception('JSON-Encoding fehlgeschlagen: ' . json_last_error_msg());
        }
        
        ob_clean();
        echo $json_output;
        ob_end_flush();
        exit;
    } else {
        // Debug-Informationen für ungültige Anfrage
        $debug_info = [
            'request_method' => $request_method,
            'action_received' => $action,
            'post_data' => array_keys($_POST),
            'get_data' => array_keys($_GET),
            'expected_action' => 'print_etikett',
            'session_active' => isset($_SESSION['admin_logged_in']),
            'session_logged_in' => isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']
        ];
        
        // Wenn es ein GET-Request ist, zeige hilfreiche Informationen
        if ($request_method === 'GET') {
            http_response_code(400);
            ob_clean();
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Diese API erfordert einen POST-Request',
                'info' => 'Diese API-Endpoint muss über POST mit folgenden Parametern aufgerufen werden:',
                'required_params' => [
                    'action' => 'print_etikett',
                    'name' => 'Name des Kunden',
                    'kennzeichen' => 'Kennzeichen',
                    'copies' => 'Anzahl Exemplare (optional, Standard: 6)'
                ],
                'example' => 'Verwende fetch() oder XMLHttpRequest mit POST-Methode',
                'debug' => $debug_info
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            ob_end_flush();
            exit;
        }
        
        http_response_code(400);
        ob_clean();
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => 'Ungültige Anfrage',
            'debug' => $debug_info
        ]);
        ob_end_flush();
        exit;
    }
} catch (Exception $e) {
    ob_clean();
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => 'Server-Fehler: ' . $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    ob_end_flush();
    exit;
} catch (Error $e) {
    ob_clean();
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => 'PHP-Fehler: ' . $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    ob_end_flush();
    exit;
}

