<?php
/**
 * P2 Kundenverwaltung - Innovatives Split-View Design
 * P2 API Integration für Kunden, Aufträge und Reifenangebote
 * Master-Detail Layout mit Sidebar-Filters und Slide-in Panel
 */

session_start();

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

require_once dirname(__DIR__) . '/config/database.php';
require_once 'includes/helpers.php';
require_once 'includes/p2_api.php';

$db = new Database();
$conn = $db->getConnection();

// Lade Autohäuser
try {
    $stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE is_active = 1 ORDER BY name");
    $stmt->execute();
    $autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $autohauser = [];
}

// User-Daten
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.full_name, u.username, u.role, u.autohaus_id, a.name as autohaus_name
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
    $current_user_role = $user_data['role'] ?? 'admin';
}

$js_current_user_name = json_encode($current_user_name, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT);

// Theme
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>P2 Kundenverwaltung</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #3b82f6;
            --primary-dark: #2563eb;
            --secondary: #64748b;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #06b6d4;
            
            --bg-main: #f1f5f9;
            --bg-panel: #ffffff;
            --bg-sidebar: #ffffff;
            --bg-hover: #f8fafc;
            --text-main: #0f172a;
            --text-secondary: #475569;
            --text-muted: #94a3b8;
            --border: #e2e8f0;
            --border-light: #f1f5f9;
            
            --shadow-sm: 0 1px 2px 0 rgba(0,0,0,0.05);
            --shadow: 0 1px 3px 0 rgba(0,0,0,0.1), 0 1px 2px 0 rgba(0,0,0,0.06);
            --shadow-md: 0 4px 6px -1px rgba(0,0,0,0.1), 0 2px 4px -1px rgba(0,0,0,0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0,0,0,0.1), 0 4px 6px -2px rgba(0,0,0,0.05);
            
            --radius: 8px;
            --radius-lg: 12px;
        }

        * {
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, sans-serif;
            margin: 0;
            padding: 0;
            background: var(--bg-main);
            color: var(--text-main);
        }

        /* Main Layout - Split View */
        .p2-layout {
            display: flex;
            height: 100vh;
            overflow: hidden;
            width: 100%;
        }

        .admin-main {
            margin-left: 300px !important;
            transition: margin-left 0.3s ease;
            width: calc(100% - 300px);
        }

        .smart-sidebar.collapsed + .admin-main {
            margin-left: 80px !important;
            width: calc(100% - 80px);
        }

        /* Left Sidebar */
        .filter-sidebar {
            width: 320px;
            background: var(--bg-sidebar);
            border-right: 1px solid var(--border);
            display: flex;
            flex-direction: column;
            overflow-y: auto;
        }

        .sidebar-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            background: var(--bg-hover);
        }

        .sidebar-title {
            font-size: 1.125rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .autohaus-select {
            width: 100%;
            padding: 0.625rem;
            border: 1px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            color: var(--text-main);
            font-size: 0.875rem;
            margin-bottom: 1rem;
        }

        .filter-section {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
        }

        .filter-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-secondary);
            margin-bottom: 1rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .filter-input, .filter-select {
            width: 100%;
            padding: 0.5rem 0.75rem;
            border: 1px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            color: var(--text-main);
            font-size: 0.875rem;
        }

        .filter-input:focus, .filter-select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .stats-compact {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 0.75rem;
            margin-top: 1rem;
        }

        .stat-mini {
            background: var(--bg-hover);
            padding: 0.75rem;
            border-radius: var(--radius);
            text-align: center;
        }

        .stat-mini-value {
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
        }

        .stat-mini-label {
            font-size: 0.75rem;
            color: var(--text-secondary);
            margin-top: 0.25rem;
        }

        /* Main Content Area */
        .content-area {
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }

        .content-header {
            padding: 1.25rem 1.5rem;
            background: var(--bg-panel);
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-shrink: 0;
        }

        .view-controls {
            display: flex;
            gap: 0.5rem;
        }

        .view-btn {
            padding: 0.5rem 1rem;
            border: 1px solid var(--border);
            background: var(--bg-panel);
            color: var(--text-secondary);
            border-radius: var(--radius);
            cursor: pointer;
            font-size: 0.875rem;
            transition: all 0.2s;
        }

        .view-btn:hover {
            background: var(--bg-hover);
        }

        .view-btn.active {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }

        .content-body {
            flex: 1;
            overflow-y: auto;
            padding: 1.5rem;
        }

        /* Customer Grid View */
        .customer-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1.25rem;
        }

        .customer-item {
            background: var(--bg-panel);
            border: 1px solid var(--border);
            border-radius: var(--radius-lg);
            padding: 1.25rem;
            cursor: pointer;
            transition: all 0.2s;
        }

        .customer-item:hover {
            box-shadow: var(--shadow-md);
            transform: translateY(-2px);
        }

        .customer-item.selected {
            border-color: var(--primary);
            background: rgba(59, 130, 246, 0.05);
        }

        .customer-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 1rem;
        }

        .customer-name {
            font-size: 1.125rem;
            font-weight: 600;
            color: var(--text-main);
        }

        .customer-info {
            display: grid;
            gap: 0.5rem;
            font-size: 0.875rem;
        }

        .info-row {
            display: flex;
            justify-content: space-between;
        }

        .info-label {
            color: var(--text-secondary);
        }

        .info-value {
            color: var(--text-main);
            font-weight: 500;
        }

        /* Detail Panel */
        .detail-panel {
            width: 0;
            background: var(--bg-panel);
            border-left: 1px solid var(--border);
            transition: width 0.3s ease;
            overflow: hidden;
            position: relative;
        }

        .detail-panel.open {
            width: 500px;
        }

        .panel-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .panel-title {
            font-size: 1.125rem;
            font-weight: 600;
        }

        .panel-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: var(--text-secondary);
        }

        .panel-body {
            padding: 1.5rem;
            overflow-y: auto;
            height: calc(100vh - 80px);
        }

        .detail-section {
            margin-bottom: 2rem;
        }

        .section-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-secondary);
            margin-bottom: 1rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .detail-grid {
            display: grid;
            gap: 0.75rem;
        }

        .detail-row {
            display: flex;
            justify-content: space-between;
            padding: 0.625rem 0;
            border-bottom: 1px solid var(--border-light);
        }

        .detail-label {
            color: var(--text-secondary);
            font-size: 0.875rem;
        }

        .detail-value {
            color: var(--text-main);
            font-weight: 500;
            font-size: 0.875rem;
        }

        /* Orders Table */
        .orders-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
        }

        .orders-table th,
        .orders-table td {
            padding: 0.75rem;
            text-align: left;
            border-bottom: 1px solid var(--border);
        }

        .orders-table th {
            background: var(--bg-hover);
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-secondary);
        }

        .orders-table tr:hover {
            background: var(--bg-hover);
        }

        .badge {
            padding: 0.25rem 0.5rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .badge-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .badge-warning {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning);
        }

        .badge-danger {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        .badge-info {
            background: rgba(6, 182, 212, 0.1);
            color: var(--info);
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 3rem;
            color: var(--text-secondary);
        }

        .empty-icon {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        /* Loading */
        .loading {
            text-align: center;
            padding: 2rem;
        }

        .spinner {
            border: 3px solid var(--border);
            border-top: 3px solid var(--primary);
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Toast Notifications */
        .toast-container {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 10000;
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            max-width: 400px;
        }

        .toast {
            background: var(--bg-panel);
            border: 1px solid var(--border);
            border-radius: var(--radius);
            padding: 1rem 1.25rem;
            box-shadow: var(--shadow-lg);
            display: flex;
            align-items: center;
            gap: 0.75rem;
            animation: slideInRight 0.3s ease-out;
            min-width: 300px;
        }

        .toast.success {
            border-left: 4px solid var(--success);
        }

        .toast.error {
            border-left: 4px solid var(--danger);
        }

        .toast.warning {
            border-left: 4px solid var(--warning);
        }

        .toast.info {
            border-left: 4px solid var(--info);
        }

        .toast-icon {
            font-size: 1.25rem;
            flex-shrink: 0;
        }

        .toast.success .toast-icon {
            color: var(--success);
        }

        .toast.error .toast-icon {
            color: var(--danger);
        }

        .toast.warning .toast-icon {
            color: var(--warning);
        }

        .toast.info .toast-icon {
            color: var(--info);
        }

        .toast-content {
            flex: 1;
        }

        .toast-title {
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-main);
            margin-bottom: 0.25rem;
        }

        .toast-message {
            font-size: 0.75rem;
            color: var(--text-secondary);
        }

        .toast-close {
            background: none;
            border: none;
            color: var(--text-secondary);
            cursor: pointer;
            font-size: 1.25rem;
            padding: 0;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--radius);
            transition: all 0.2s;
        }

        .toast-close:hover {
            background: var(--bg-hover);
            color: var(--text-main);
        }

        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        @keyframes slideOutRight {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(100%);
                opacity: 0;
            }
        }

        /* Buttons */
        .btn {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: var(--radius);
            cursor: pointer;
            font-size: 0.875rem;
            font-weight: 500;
            transition: all 0.2s;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
        }

        .btn-secondary {
            background: var(--bg-hover);
            color: var(--text-main);
        }

        .btn-secondary:hover {
            background: var(--border);
        }

        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        }

        .modal {
            background: var(--bg-panel);
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-lg);
            max-width: 800px;
            width: 90%;
            max-height: 90vh;
            display: flex;
            flex-direction: column;
        }

        .modal-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-title {
            font-size: 1.25rem;
            font-weight: 600;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: var(--text-secondary);
            padding: 0;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .modal-close:hover {
            color: var(--text-main);
        }

        /* Orders Panel Styles (für Angebots-Panel) */
        .orders-panel {
            position: fixed;
            top: 0;
            right: -800px;
            width: 800px;
            height: 100vh;
            background: var(--bg-panel);
            border-left: 1px solid var(--border);
            box-shadow: -4px 0 20px rgba(0, 0, 0, 0.1);
            z-index: 2000;
            transition: right 0.3s ease;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }

        .orders-panel.open {
            right: 0;
        }

        .orders-panel-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            background: var(--bg-hover);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-shrink: 0;
        }
        
        .orders-panel-body-wrapper {
            display: flex;
            flex-direction: row;
            flex: 1;
            overflow: hidden;
        }

        .orders-panel-title {
            font-size: 1.125rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .orders-panel-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: var(--text-secondary);
            cursor: pointer;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--radius);
            transition: all 0.2s;
        }

        .orders-panel-close:hover {
            background: var(--bg-hover);
            color: var(--text-main);
        }

        .orders-panel-sidebar {
            width: 200px;
            background: var(--bg-sidebar);
            border-right: 1px solid var(--border);
            display: flex;
            flex-direction: column;
            overflow-y: auto;
        }
        
        .orders-sidebar-header {
            padding: 1rem;
            border-bottom: 1px solid var(--border);
            background: var(--bg-hover);
        }
        
        .orders-sidebar-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .orders-sidebar-nav {
            display: flex;
            flex-direction: column;
            padding: 0.5rem;
        }

        .orders-sidebar-item {
            padding: 0.75rem 1rem;
            background: transparent;
            border: none;
            border-radius: var(--radius);
            cursor: pointer;
            text-align: left;
            display: flex;
            align-items: center;
            justify-content: space-between;
            color: var(--text-main);
            transition: all 0.2s;
            margin-bottom: 0.25rem;
        }

        .orders-sidebar-item:hover {
            background: var(--bg-hover);
        }

        .orders-sidebar-item.active {
            background: var(--primary);
            color: white;
        }

        .orders-sidebar-badge {
            background: #1E40AF;
            color: white;
            font-size: 0.6875rem;
            font-weight: 600;
            line-height: 1;
            padding: 0.25rem 0.5rem;
            border-radius: 12px;
            min-width: 20px;
            height: 20px;
            text-align: center;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
            border: 2px solid white;
            margin-left: auto;
        }

        .orders-sidebar-badge.large {
            padding: 0.25rem 0.375rem;
            font-size: 0.625rem;
        }

        .orders-panel-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }

        .orders-panel-body {
            flex: 1;
            overflow-y: auto;
            padding: 1rem;
        }

        /* Modern Loading Wrapper */
        .loading-wrapper {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(15, 23, 42, 0.75);
            backdrop-filter: blur(8px);
            z-index: 9999999 !important;
            display: none;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.3s ease, visibility 0.3s ease;
            visibility: hidden;
        }

        .loading-wrapper.active {
            display: flex;
            opacity: 1;
            visibility: visible;
        }

        .loading-content {
            background: var(--bg-panel);
            border-radius: var(--radius-lg);
            padding: 2.5rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.3), 0 10px 10px -5px rgba(0, 0, 0, 0.2);
            min-width: 280px;
            max-width: 400px;
            text-align: center;
            animation: fadeInScale 0.3s ease;
        }

        @keyframes fadeInScale {
            from {
                opacity: 0;
                transform: scale(0.9);
            }
            to {
                opacity: 1;
                transform: scale(1);
            }
        }

        .loading-spinner {
            width: 64px;
            height: 64px;
            margin: 0 auto 1.5rem;
            position: relative;
        }

        .loading-spinner-inner {
            width: 100%;
            height: 100%;
            border: 4px solid var(--border-light);
            border-top-color: var(--primary);
            border-right-color: var(--primary);
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }

        .loading-spinner-pulse {
            position: absolute;
            top: -4px;
            left: -4px;
            width: 100%;
            height: 100%;
            border: 4px solid var(--primary);
            border-radius: 50%;
            opacity: 0.3;
            animation: pulse 1.5s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        @keyframes pulse {
            0%, 100% {
                transform: scale(1);
                opacity: 0.3;
            }
            50% {
                transform: scale(1.2);
                opacity: 0;
            }
        }

        .loading-text {
            font-size: 1rem;
            font-weight: 500;
            color: var(--text-main);
            margin-bottom: 0.5rem;
        }

        .loading-subtext {
            font-size: 0.875rem;
            color: var(--text-secondary);
            margin-top: 0.5rem;
        }

        .loading-progress {
            width: 100%;
            height: 6px;
            background: var(--border-light);
            border-radius: 9999px;
            margin-top: 1.5rem;
            overflow: hidden;
        }

        .loading-progress-bar {
            height: 100%;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            border-radius: 9999px;
            animation: progress 1.5s ease-in-out infinite;
        }

        @keyframes progress {
            0% {
                width: 0%;
                transform: translateX(0);
            }
            50% {
                width: 70%;
                transform: translateX(0);
            }
            100% {
                width: 100%;
                transform: translateX(0);
            }
        }

        .loading-dots {
            display: inline-flex;
            gap: 0.5rem;
            margin-top: 1rem;
        }

        .loading-dot {
            width: 10px;
            height: 10px;
            background: var(--primary);
            border-radius: 50%;
            animation: dotBounce 1.4s ease-in-out infinite;
        }

        .loading-dot:nth-child(1) {
            animation-delay: 0s;
        }

        .loading-dot:nth-child(2) {
            animation-delay: 0.2s;
        }

        .loading-dot:nth-child(3) {
            animation-delay: 0.4s;
        }

        @keyframes dotBounce {
            0%, 80%, 100% {
                transform: scale(0.8);
                opacity: 0.5;
            }
            40% {
                transform: scale(1.2);
                opacity: 1;
            }
        }

        .modal-body {
            padding: 1.5rem;
            overflow-y: auto;
            flex: 1;
        }

        .modal-footer {
            padding: 1rem 1.5rem;
            border-top: 1px solid var(--border);
            display: flex;
            justify-content: flex-end;
            gap: 0.5rem;
        }

        .form-control {
            width: 100%;
            padding: 0.5rem 0.75rem;
            border: 1px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            color: var(--text-main);
            font-size: 0.875rem;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        /* Kompakte Auftragsliste */
        .orders-compact-list {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            margin-top: 1rem;
        }

        .order-compact-item {
            background: var(--bg-panel);
            border: 1px solid var(--border);
            border-radius: var(--radius-lg);
            padding: 1rem;
            cursor: pointer;
            transition: all 0.2s ease;
            box-shadow: var(--shadow-sm);
        }

        .order-compact-item:hover {
            background: var(--bg-hover);
            border-color: var(--primary);
            box-shadow: var(--shadow-md);
            transform: translateY(-1px);
        }

        .order-compact-main {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
        }

        .order-compact-header {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            flex-wrap: wrap;
        }

        .order-compact-number {
            font-size: 1rem;
            font-weight: 700;
            color: var(--primary);
            cursor: pointer;
            text-decoration: underline;
        }

        .order-compact-number:hover {
            color: var(--primary-dark);
        }

        .order-compact-date {
            font-size: 0.875rem;
            color: var(--text-secondary);
            margin-left: auto;
        }

        .order-compact-body {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            padding-top: 0.75rem;
            border-top: 1px solid var(--border);
        }

        .order-compact-info {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .order-compact-vehicle {
            display: inline-flex;
            align-items: center;
            padding: 0.375rem 0.75rem;
            background: var(--bg-hover);
            border-radius: var(--radius);
            font-size: 0.875rem;
            cursor: pointer;
            transition: background 0.2s;
            width: fit-content;
        }

        .order-compact-vehicle:hover {
            background: var(--primary);
            color: white;
        }

        .order-compact-vehicle:hover i {
            color: white !important;
        }

        .order-compact-financial {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
            align-items: flex-end;
            text-align: right;
        }

        .order-compact-amount,
        .order-compact-margin {
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
        }

        .order-compact-label {
            font-size: 0.75rem;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .order-compact-value {
            font-size: 0.9375rem;
            font-weight: 600;
            color: var(--text-main);
        }

        @media (max-width: 768px) {
            .order-compact-body {
                grid-template-columns: 1fr;
            }

            .order-compact-financial {
                align-items: flex-start;
                text-align: left;
            }
        }

        /* Fahrzeug-Gruppierung */
        .vehicle-orders-group {
            border: 2px solid var(--border);
            border-radius: var(--radius-lg);
            padding: 1.25rem;
            background: var(--bg-panel);
            box-shadow: var(--shadow-sm);
        }

        .vehicle-group-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-bottom: 1rem;
            margin-bottom: 1rem;
            border-bottom: 2px solid var(--border);
            flex-wrap: wrap;
            gap: 1rem;
        }

        .vehicle-group-title {
            display: flex;
            align-items: center;
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--text-main);
        }

        .vehicle-license {
            color: var(--primary);
            font-weight: 700;
        }

        .vehicle-vin {
            font-family: monospace;
        }

        .vehicle-group-stats {
            display: flex;
            gap: 1.5rem;
            flex-wrap: wrap;
        }

        .vehicle-stat {
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
            align-items: flex-end;
        }

        .vehicle-stat-label {
            font-size: 0.75rem;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .vehicle-stat-value {
            font-size: 0.9375rem;
            font-weight: 600;
            color: var(--text-main);
        }

        @media (max-width: 768px) {
            .vehicle-group-header {
                flex-direction: column;
                align-items: flex-start;
            }

            .vehicle-group-stats {
                width: 100%;
                justify-content: space-between;
            }

            .vehicle-stat {
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/sidebar.php'; ?>
    
    <div class="admin-main">
        <div class="p2-layout">
            <!-- Left Sidebar -->
            <div class="filter-sidebar">
                <div class="sidebar-header">
                    <div class="sidebar-title">
                        <i class="fas fa-filter"></i> Filter
                    </div>
                    <select id="autohausSelect" class="autohaus-select">
                        <option value="">Alle Autohäuser</option>
                        <?php foreach ($autohauser as $ah): ?>
                            <option value="<?php echo $ah['id']; ?>"><?php echo htmlspecialchars($ah['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-section">
                    <div class="filter-title">
                        <i class="fas fa-search"></i> Kunden-Suche
                    </div>
                    <input type="text" id="searchInput" class="filter-input" placeholder="Name, Kennzeichen oder Fahrgestellnummer...">
                    <div style="font-size: 0.75rem; color: var(--text-secondary); margin-top: 0.5rem; padding: 0.5rem; background: var(--bg-hover); border-radius: var(--radius);">
                        <i class="fas fa-info-circle"></i> Suche nach Name, Kennzeichen oder Fahrgestellnummer (VIN)
                    </div>
                </div>
                
                <div class="filter-section" style="border-top: 2px solid var(--primary); margin-top: 1rem; padding-top: 1rem;">
                    <div class="filter-title" style="color: var(--primary);">
                        <i class="fas fa-file-invoice"></i> Auftragsnummer-Suche
                    </div>
                    <input type="text" id="orderNumberInput" class="filter-input" placeholder="z.B. 1 - 123456 oder 123456" style="border: 2px solid var(--primary);">
                    <div style="font-size: 0.75rem; color: var(--text-secondary); margin-top: 0.5rem; padding: 0.5rem; background: var(--bg-hover); border-radius: var(--radius);">
                        <i class="fas fa-info-circle"></i> Format: "X - XXXXXX" oder nur "XXXXXX"
                    </div>
                    <button type="button" id="searchOrderNumberBtn" class="btn btn-primary" style="width: 100%; margin-top: 0.75rem; padding: 0.75rem;">
                        <i class="fas fa-search"></i> Auftrag suchen
                    </button>
                </div>
                
                <div class="filter-section">
                    <div class="filter-title">Statistiken</div>
                    <div class="stats-compact">
                        <div class="stat-mini">
                            <div class="stat-mini-value" id="statTotal">0</div>
                            <div class="stat-mini-label">Kunden</div>
                        </div>
                        <div class="stat-mini">
                            <div class="stat-mini-value" id="statOrders">0</div>
                            <div class="stat-mini-label">Aufträge</div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Main Content -->
            <div class="content-area">
                <div class="content-header">
                    <div>
                        <h1 style="margin: 0; font-size: 1.5rem; font-weight: 600;">P2 Kundenverwaltung</h1>
                        <p style="margin: 0.25rem 0 0 0; color: var(--text-secondary); font-size: 0.875rem;" id="resultsCount">0 Ergebnisse</p>
                    </div>
                    <div class="view-controls">
                        <button class="btn btn-primary" onclick="loadCustomers()">
                            <i class="fas fa-sync"></i> Aktualisieren
                        </button>
                        <button class="btn btn-secondary" onclick="window.showOffersNotificationsPanel()" style="margin-left: 0.5rem;">
                            <i class="fas fa-file-invoice-dollar"></i> Angebote
                            <span class="badge" id="offersNotificationsBadge" style="display: none; margin-left: 0.5rem; background: var(--danger); color: white; padding: 0.125rem 0.5rem; border-radius: 10px; font-size: 0.75rem;">0</span>
                        </button>
                    </div>
                </div>
                
                <div class="content-body" id="contentBody">
                    <div class="loading">
                        <div class="spinner"></div>
                        <p>Lade Kunden...</p>
                    </div>
                </div>
            </div>
            
            <!-- Detail Panel -->
            <div class="detail-panel" id="detailPanel">
                <div class="panel-header">
                    <div class="panel-title" id="panelTitle">Kundendetails</div>
                    <button class="panel-close" onclick="closeDetailPanel()">&times;</button>
                </div>
                <div class="panel-body" id="panelBody">
                    <p style="color: var(--text-secondary);">Wählen Sie einen Kunden aus</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Angebotsbenachrichtigungen Panel -->
    <div class="orders-panel" id="offersNotificationsPanel">
        <div class="orders-panel-header">
            <div class="orders-panel-title">
                <i class="fas fa-file-invoice-dollar"></i>
                Angebotsbenachrichtigungen
            </div>
            <button class="orders-panel-close" onclick="closeOffersNotificationsPanel()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="orders-panel-body-wrapper">
            <div class="orders-panel-sidebar">
                <div class="orders-sidebar-header">
                    <div class="orders-sidebar-title">Status</div>
                </div>
                <div class="orders-sidebar-nav">
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('accepted')" data-tab="accepted" id="offersTabAccepted">
                        <span>
                            <i class="fas fa-check-circle" style="margin-right: 0.5rem; color: #28a745;"></i>
                            Angenommen
                        </span>
                        <span class="orders-sidebar-badge" id="acceptedOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('declined')" data-tab="declined" id="offersTabDeclined">
                        <span>
                            <i class="fas fa-times-circle" style="margin-right: 0.5rem; color: #dc3545;"></i>
                            Abgelehnt
                        </span>
                        <span class="orders-sidebar-badge" id="declinedOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('new_request')" data-tab="new_request" id="offersTabNewRequest">
                        <span>
                            <i class="fas fa-redo" style="margin-right: 0.5rem; color: #17a2b8;"></i>
                            Neu angefragt
                        </span>
                        <span class="orders-sidebar-badge" id="newRequestOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('pending')" data-tab="pending" id="offersTabPending">
                        <span>
                            <i class="fas fa-clock" style="margin-right: 0.5rem; color: #f59e0b;"></i>
                            Noch Offen
                        </span>
                        <span class="orders-sidebar-badge" id="pendingOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item active" onclick="switchOffersNotificationsTab('all')" data-tab="all" id="offersTabAll">
                        <span>
                            <i class="fas fa-list" style="margin-right: 0.5rem;"></i>
                            Alle
                        </span>
                        <span class="orders-sidebar-badge" id="offersTabAllBadge" style="display: none;">0</span>
                    </button>
                </div>
            </div>
            <div class="orders-panel-content">
                <div style="padding: 0.75rem 1rem; border-bottom: 1px solid var(--border); background: var(--bg-hover); display: flex; justify-content: space-between; align-items: center;">
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">
                        <i class="fas fa-clock"></i> Angebotsbenachrichtigungen
                    </div>
                    <div style="font-size: 0.75rem; color: var(--text-secondary);" id="offersLastUpdateTime">
                        <i class="fas fa-sync"></i> Wird geladen...
                    </div>
                </div>
                <div class="orders-panel-body" id="offersNotificationsBody">
                    <div style="padding: 2rem; text-align: center; color: var(--text-secondary);">Bitte wählen Sie ein Autohaus aus</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Loading Wrapper -->
    <div class="loading-wrapper" id="loadingOverlay">
        <div class="loading-content">
            <div class="loading-spinner">
                <div class="loading-spinner-pulse"></div>
                <div class="loading-spinner-inner"></div>
            </div>
            <div class="loading-text" id="loadingText">Wird geladen...</div>
            <div class="loading-subtext" id="loadingSubtext">Bitte warten Sie einen Moment</div>
            <div class="loading-progress">
                <div class="loading-progress-bar"></div>
            </div>
            <div class="loading-dots">
                <div class="loading-dot"></div>
                <div class="loading-dot"></div>
                <div class="loading-dot"></div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        let customers = [];
        let filteredCustomers = [];
        let currentView = 'list'; // Nur Liste, kein Grid
        let selectedCustomer = null;
        let autohausId = null;
        let currentPage = 1;
        let itemsPerPage = 20;
        let totalPages = 1;

        $(document).ready(function() {
            // Initial: Zeige "Autohaus auswählen" wenn noch keins ausgewählt ist
            if (!$('#autohausSelect').val()) {
                $('#contentBody').html('<div class="empty-state"><div class="empty-icon"><i class="fas fa-exclamation-triangle"></i></div><h3>Autohaus auswählen</h3><p>Bitte wählen Sie ein Autohaus aus</p></div>');
            }
            
            $('#autohausSelect').on('change', function() {
                autohausId = $(this).val();
                currentPage = 1;
                customers = [];
                filteredCustomers = [];
                
                if (autohausId) {
                    // Zeige Toast-Notification zur Suche
                    showToast('info', 'Autohaus ausgewählt', 'Bitte suchen Sie nach einem Kunden (Name, Kennzeichen oder Fahrgestellnummer)');
                    $('#contentBody').html(`
                        <div class="empty-state">
                            <div class="empty-icon"><i class="fas fa-search"></i></div>
                            <h3>Suche starten</h3>
                            <p>Bitte geben Sie einen Suchbegriff ein:</p>
                            <ul style="text-align: left; display: inline-block; margin-top: 1rem;">
                                <li>Kundenname (Vor- oder Nachname)</li>
                                <li>Kennzeichen</li>
                                <li>Fahrgestellnummer (VIN)</li>
                            </ul>
                        </div>
                    `);
                    updateStats();
                } else {
                    $('#contentBody').html('<div class="empty-state"><div class="empty-icon"><i class="fas fa-exclamation-triangle"></i></div><h3>Autohaus auswählen</h3><p>Bitte wählen Sie ein Autohaus aus</p></div>');
                }
            });

            // Universelle Suche - reagiert auf Enter oder nach 500ms Pause
            let searchTimeout;
            $('#searchInput').on('input', function() {
                clearTimeout(searchTimeout);
                const searchTerm = $(this).val().trim();
                
                if (!autohausId) {
                    showToast('warning', 'Autohaus erforderlich', 'Bitte wählen Sie zuerst ein Autohaus aus');
                    return;
                }
                
                if (searchTerm.length >= 2) {
                    searchTimeout = setTimeout(() => {
                        performUniversalSearch(searchTerm);
                    }, 500);
                } else if (searchTerm.length === 0) {
                    customers = [];
                    filteredCustomers = [];
                    $('#contentBody').html(`
                        <div class="empty-state">
                            <div class="empty-icon"><i class="fas fa-search"></i></div>
                            <h3>Suche starten</h3>
                            <p>Bitte geben Sie einen Suchbegriff ein (mindestens 2 Zeichen)</p>
                        </div>
                    `);
                    updateStats();
                }
            });

            // Enter-Taste für sofortige Suche
            $('#searchInput').on('keypress', function(e) {
                if (e.which === 13) {
                    clearTimeout(searchTimeout);
                    const searchTerm = $(this).val().trim();
                    if (searchTerm.length >= 2 && autohausId) {
                        performUniversalSearch(searchTerm);
                    }
                }
            });

            // Auftragsnummer-Suche
            let orderNumberTimeout;
            $('#orderNumberInput').on('input', function() {
                clearTimeout(orderNumberTimeout);
                const orderNumber = $(this).val().trim();
                
                if (!autohausId) {
                    return; // Keine Warnung beim Tippen
                }
                
                // Validiere Format beim Tippen
                if (orderNumber.length > 0) {
                    const orderPattern = /^(\d+\s*-\s*)?\d{0,7}$/;
                    if (!orderPattern.test(orderNumber)) {
                        $(this).css('border-color', 'var(--danger)');
                    } else {
                        $(this).css('border-color', 'var(--primary)');
                    }
                } else {
                    $(this).css('border-color', 'var(--primary)');
                }
            });

            // Enter-Taste für Auftragsnummer-Suche
            $('#orderNumberInput').on('keypress', function(e) {
                if (e.which === 13) {
                    clearTimeout(orderNumberTimeout);
                    const orderNumber = $(this).val().trim();
                    if (orderNumber.length > 0 && autohausId) {
                        performOrderNumberSearch(orderNumber);
                    }
                }
            });

            // Button für Auftragsnummer-Suche
            $('#searchOrderNumberBtn').on('click', function() {
                const orderNumber = $('#orderNumberInput').val().trim();
                if (orderNumber.length > 0 && autohausId) {
                    performOrderNumberSearch(orderNumber);
                } else if (!autohausId) {
                    showToast('warning', 'Autohaus erforderlich', 'Bitte wählen Sie zuerst ein Autohaus aus');
                } else {
                    showToast('warning', 'Eingabe erforderlich', 'Bitte geben Sie eine Auftragsnummer ein');
                }
            });

            // Keine initiale Ladung mehr
        });

        /**
         * Normalisiert Kennzeichen für verschiedene Formate
         * Unterstützt: KI-SD 221, KI-SD221, KI-SD221E, KI-SD 221E
         */
        function normalizeLicensePlate(plate) {
            if (!plate) return '';
            // Entferne alle Leerzeichen und Bindestriche, konvertiere zu Großbuchstaben
            return plate.replace(/[\s-]/g, '').toUpperCase();
        }

        /**
         * Erkennt den Suchtyp basierend auf dem Suchbegriff
         */
        function detectSearchType(searchTerm) {
            const trimmed = searchTerm.trim();
            
            // 1. Auftragsnummer: Format "X - XXXXXX" oder nur "XXXXXX" (5-7 Ziffern)
            // Prüfe zuerst auf Format "X - XXXXXX"
            const orderNumberPattern = /^(\d+)\s*-\s*(\d{5,7})$/;
            const orderNumberMatch = trimmed.match(orderNumberPattern);
            if (orderNumberMatch) {
                return 'order_number';
            }
            
            // Prüfe auf reine Nummer (5-7 Ziffern) - könnte Auftragsnummer oder Kunden-ID sein
            // Wenn es 5 Ziffern hat, ist es wahrscheinlich eine Auftragsnummer (Kunden-IDs sind meist 5-stellig, aber Auftragsnummern auch)
            // Priorisiere Auftragsnummern-Suche für 5-7 Ziffern im separaten Feld
            // Aber in der normalen Suche: 5 Ziffern = Kunden-ID, 6-7 Ziffern = Auftragsnummer
            // Diese Funktion wird nur für die normale Suche verwendet, nicht für das Auftragsnummern-Feld
            
            // 3. Kennzeichen: Enthält Buchstaben und Zahlen, typisches Format
            // Format: 1-3 Buchstaben, optional Bindestrich/Leerzeichen, 1-3 Buchstaben, optional Bindestrich/Leerzeichen, 1-4 Ziffern, optional 1 Buchstabe
            // Beispiele: KI-SD 221, KI-SD221, KI-SD221E, KI-SD 221E, M-AB 123, B-AB 1234
            const licensePattern = /^[A-ZÄÖÜ]{1,3}[\s-]?[A-Z0-9]{1,4}[\s-]?[0-9]{1,4}[A-Z]?$/i;
            // Alternative: Einfacheres Pattern - enthält Buchstaben UND Zahlen, 4-12 Zeichen
            const hasLetters = /[A-ZÄÖÜ]/i.test(trimmed);
            const hasDigits = /[0-9]/.test(trimmed);
            const length = trimmed.replace(/[\s-]/g, '').length;
            
            if (hasLetters && hasDigits && length >= 4 && length <= 12) {
                // Prüfe ob es nicht eine reine VIN ist (zu lang oder zu viele Ziffern)
                const digitCount = (trimmed.match(/\d/g) || []).length;
                const letterCount = (trimmed.match(/[A-ZÄÖÜ]/gi) || []).length;
                
                // Wenn mehr als 6 Ziffern und weniger als 3 Buchstaben, könnte es VIN sein
                if (digitCount > 6 && letterCount < 3) {
                    // Weiter zur VIN-Prüfung
                } else {
                    return 'license';
                }
            }
            
            // 4. Fahrgestellnummer: 6-17 Zeichen alphanumerisch
            const vinPattern = /^[A-Z0-9]{6,17}$/i;
            const digitsOnly = trimmed.replace(/[^0-9]/g, '');
            const cleanedVin = trimmed.replace(/[\s-]/g, '').toUpperCase();
            
            // Wenn 6 oder mehr Ziffern und alphanumerisch, könnte es VIN sein
            if (digitsOnly.length >= 6 && vinPattern.test(cleanedVin)) {
                // Wenn genau 6 Ziffern oder letzte 6 Ziffern einer längeren Nummer
                if (digitsOnly.length === 6 || cleanedVin.length <= 10) {
                    return 'vin_last6';
                }
                // Wenn 10+ Zeichen, wahrscheinlich vollständige VIN
                if (cleanedVin.length >= 10) {
                    return 'vin_full';
                }
            }
            
            // 5. Standard: Name
            return 'name';
        }

        /**
         * Führt eine Suche nach Auftragsnummer durch
         */
        function performOrderNumberSearch(orderNumber) {
            const autohaus = $('#autohausSelect').val();
            
            if (!autohaus) {
                showToast('warning', 'Autohaus erforderlich', 'Bitte wählen Sie zuerst ein Autohaus aus');
                return;
            }
            
            // Trimme und validiere die Auftragsnummer
            const trimmedOrderNumber = orderNumber ? orderNumber.trim() : '';
            
            if (!trimmedOrderNumber || trimmedOrderNumber.length === 0) {
                showToast('warning', 'Eingabe erforderlich', 'Bitte geben Sie eine Auftragsnummer ein');
                return;
            }
            
            // Validiere Format: Muss mindestens 4 Ziffern haben
            const digitsOnly = trimmedOrderNumber.replace(/[^0-9]/g, '');
            if (digitsOnly.length < 4) {
                showToast('warning', 'Ungültiges Format', 'Bitte geben Sie eine gültige Auftragsnummer ein (mindestens 4 Ziffern)');
                return;
            }
            
            currentPage = 1;
            $('#contentBody').html('<div class="loading"><div class="spinner"></div><p>Suche nach Auftragsnummer...</p></div>');
            
            
            fetch(`../api/p2_api_proxy.php?action=get_orders_by_order_number&autohaus_id=${autohaus}&order_number=${encodeURIComponent(trimmedOrderNumber)}&limit=10`)
                .then(r => {
                    if (!r.ok) {
                        throw new Error(`HTTP ${r.status}: ${r.statusText}`);
                    }
                    return r.json();
                })
                .then(data => {
                    
                    // Konvertiere data.data zu einem Array, falls es ein einzelnes Objekt ist
                    let orders = [];
                    if (data.success && data.data) {
                        if (Array.isArray(data.data)) {
                            orders = data.data;
                        } else if (typeof data.data === 'object' && data.data.AuftragNr) {
                            // Es ist ein einzelnes Objekt, konvertiere zu Array
                            orders = [data.data];
                        }
                    }
                    
                    
                    if (orders.length > 0) {
                        // Verarbeite Aufträge und zeige Kunden an
                        const allCustomers = {};
                        
                        orders.forEach(order => {
                            const customerId = order.KundenAdrID;
                            if (customerId) {
                                if (!allCustomers[customerId]) {
                                    allCustomers[customerId] = {
                                        KundenAdrID: customerId,
                                    KundenName1: order.KundenName1 || '',
                                    KundenVorname: order.KundenVorname || '',
                                    KundenPlz: order.KundenPlz || '',
                                    KundenOrt: order.KundenOrt || '',
                                    KundenStrasse: order.KundenStrasse || '',
                                    Telefon: order.Telefon || '',
                                    Email: order.Email || '',
                                    AnzahlAuftraege: 1,
                                    GesamtUmsatz: parseFloat(order.Umsatz || 0),
                                        foundBy: ['order_number'],
                                    matchingVehicles: [],
                                        matchingLicenses: order.ZulassungsKz ? [order.ZulassungsKz] : [],
                                        allVehicles: [],
                                        matchingOrders: [order], // Speichere gefundene Aufträge
                                        searchedOrderNumber: trimmedOrderNumber // Speichere gesuchte Auftragsnummer
                                };
                            } else {
                                    allCustomers[customerId].AnzahlAuftraege++;
                                    allCustomers[customerId].GesamtUmsatz += parseFloat(order.Umsatz || 0);
                                    if (order.ZulassungsKz && !allCustomers[customerId].matchingLicenses.includes(order.ZulassungsKz)) {
                                        allCustomers[customerId].matchingLicenses.push(order.ZulassungsKz);
                                    }
                                    if (!allCustomers[customerId].matchingOrders) {
                                        allCustomers[customerId].matchingOrders = [];
                                    }
                                    allCustomers[customerId].matchingOrders.push(order);
                                    // Speichere auch die gesuchte Auftragsnummer
                                    if (!allCustomers[customerId].searchedOrderNumber) {
                                        allCustomers[customerId].searchedOrderNumber = trimmedOrderNumber;
                                    }
                                }
                            }
                        });
                        
                        customers = Object.values(allCustomers).filter(c => c && c.KundenAdrID);
                        
                        if (customers.length > 0) {
                            showToast('success', 'Auftrag gefunden', `${customers.length} Kunde${customers.length !== 1 ? 'n' : ''} mit ${orders.length} Auftrag${orders.length !== 1 ? 'en' : ''} gefunden`);
                            filterCustomers();
                        } else {
                            showToast('info', 'Keine Ergebnisse', `Keine Kunden für Auftragsnummer "${trimmedOrderNumber}" gefunden`);
                            $('#contentBody').html(`
                                <div class="empty-state">
                                    <div class="empty-icon"><i class="fas fa-file-invoice"></i></div>
                                    <h3>Keine Ergebnisse</h3>
                                    <p>Für die Auftragsnummer "${trimmedOrderNumber}" wurden keine Kunden gefunden.</p>
                                    <p style="font-size: 0.875rem; color: var(--text-secondary); margin-top: 1rem;">
                                        Format: "X - XXXXXX" oder nur "XXXXXX"
                                    </p>
                                </div>
                            `);
                        }
                    } else {
                        showToast('info', 'Keine Ergebnisse', `Keine Aufträge für "${trimmedOrderNumber}" gefunden`);
                        $('#contentBody').html(`
                            <div class="empty-state">
                                <div class="empty-icon"><i class="fas fa-file-invoice"></i></div>
                                <h3>Keine Ergebnisse</h3>
                                <p>Für die Auftragsnummer "${trimmedOrderNumber}" wurden keine Aufträge gefunden.</p>
                                <p style="font-size: 0.875rem; color: var(--text-secondary); margin-top: 1rem;">
                                    Format: "X - XXXXXX" oder nur "XXXXXX"
                                </p>
                            </div>
                        `);
                    }
                })
                .catch(error => {
                    showToast('error', 'Fehler', 'Fehler bei der Suche: ' + error.message);
                    $('#contentBody').html(`
                        <div class="empty-state">
                            <div class="empty-icon"><i class="fas fa-exclamation-circle"></i></div>
                            <h3>Fehler</h3>
                            <p>${error.message}</p>
                        </div>
                    `);
                });
        }

        function performUniversalSearch(searchTerm) {
            const autohaus = $('#autohausSelect').val();
            
            if (!autohaus) {
                showToast('warning', 'Autohaus erforderlich', 'Bitte wählen Sie zuerst ein Autohaus aus');
                return;
            }
            
            if (!searchTerm || searchTerm.length < 2) {
                showToast('warning', 'Suchbegriff zu kurz', 'Bitte geben Sie mindestens 2 Zeichen ein');
                return;
            }
            
            currentPage = 1;
            $('#contentBody').html('<div class="loading"><div class="spinner"></div><p>Suche läuft...</p></div>');
            
            const searchType = detectSearchType(searchTerm);
            const searchPromises = [];
            
            // Immer nach Namen suchen (außer bei eindeutiger Kunden-ID, Auftragsnummer oder Kennzeichen/VIN)
            if (searchType !== 'customer_id' && searchType !== 'order_number' && searchType !== 'license' && searchType !== 'vin_full' && searchType !== 'vin_last6') {
                // OPTIMIERT: Suche zuerst mit kleinerem Limit (50), dann erweitern wenn nötig
                searchPromises.push(
                    fetch(`../api/p2_api_proxy.php?action=search_customers&autohaus_id=${autohaus}&search=${encodeURIComponent(searchTerm)}&limit=50`)
                        .then(r => r.json())
                        .then(data => {
                            // Wenn viele Ergebnisse, zeige zuerst diese an und lade dann mehr im Hintergrund
                            if (data.success && data.data && data.data.length >= 50) {
                                // Zeige zuerst die ersten 50 Ergebnisse an
                                setTimeout(() => {
                                    // Lade weitere Ergebnisse im Hintergrund
                                    fetch(`../api/p2_api_proxy.php?action=search_customers&autohaus_id=${autohaus}&search=${encodeURIComponent(searchTerm)}&limit=500`)
                                        .then(r => r.json())
                                        .then(extendedData => {
                                            if (extendedData.success && extendedData.data && extendedData.data.length > data.data.length) {
                                                // Erweitere die Ergebnisse
                                                const extendedResult = processSearchResults([extendedData], searchType, searchTerm, autohaus);
                                                if (extendedResult && extendedResult.customers) {
                                                    customers = extendedResult.customers;
                                                    filteredCustomers = customers;
                                                    filterCustomers();
                                                    showToast('info', 'Ergebnisse erweitert', `${extendedResult.customers.length} Kunden gefunden`);
                                                }
                                            }
                                        })
                                        .catch(err => {});
                                }, 100);
                            }
                            return data;
                        })
                        .catch(err => {
                            return {success: false, data: []};
                        })
                );
            }
            
            // Suche basierend auf erkanntem Typ
            switch(searchType) {
                case 'order_number':
                    // Suche nach Auftragsnummer - unterstützt Format "X - XXXXXX" oder nur "XXXXXX"
                    searchPromises.push(
                        fetch(`../api/p2_api_proxy.php?action=get_orders_by_order_number&autohaus_id=${autohaus}&order_number=${encodeURIComponent(searchTerm)}&limit=10`)
                            .then(r => r.json())
                            .then(data => {
                                return data;
                            })
                            .catch(err => {
                                return {success: false, data: []};
                            })
                    );
                    break;
                    
                case 'customer_id':
                    // Suche nach Kunden-ID
                    searchPromises.push(
                        fetch(`../api/p2_api_proxy.php?action=get_customer_by_id_with_orders&autohaus_id=${autohaus}&kunden_adr_id=${encodeURIComponent(searchTerm)}&days=2000&limit=1000`)
                            .then(r => r.json())
                            .then(data => {
                                if (data.success && data.data && data.data.customer) {
                                    // Konvertiere Kunde in Array-Format für Konsistenz
                                    return {
                                        success: true,
                                        data: [{
                                            customer: data.data.customer,
                                            orders: data.data.orders || []
                                        }]
                                    };
                                }
                                return {success: false, data: []};
                            })
                            .catch(() => ({success: false, data: []}))
                    );
                    break;
                    
                case 'license':
                    // Suche nach Kennzeichen - Backend normalisiert automatisch verschiedene Formate
                    searchPromises.push(
                        fetch(`../api/p2_api_proxy.php?action=get_orders_by_license&autohaus_id=${autohaus}&license_plate=${encodeURIComponent(searchTerm)}&limit=1000`)
                            .then(r => r.json())
                            .then(data => {
                                return data;
                            })
                            .catch(err => {
                                return {success: false, data: []};
                            })
                    );
                    break;
                    
                case 'vin_full':
                    // Suche nach vollständiger Fahrgestellnummer
                    searchPromises.push(
                        fetch(`../api/p2_api_proxy.php?action=get_orders_by_vin&autohaus_id=${autohaus}&vin=${encodeURIComponent(searchTerm)}&limit=1000`)
                            .then(r => r.json())
                            .then(data => {
                                return data;
                            })
                            .catch(err => {
                                return {success: false, data: []};
                            })
                    );
                    break;
                    
                case 'vin_last6':
                    // Suche nach letzten 6 Ziffern der Fahrgestellnummer
                    const last6Digits = searchTerm.replace(/[^0-9]/g, '').slice(-6);
                    if (last6Digits.length >= 6) {
                        searchPromises.push(
                            fetch(`../api/p2_api_proxy.php?action=get_orders_by_vin_last6&autohaus_id=${autohaus}&vin_last6=${encodeURIComponent(last6Digits)}&limit=1000`)
                                .then(r => r.json())
                                .then(data => {
                                    return data;
                                })
                                .catch(err => {
                                    return {success: false, data: []};
                                })
                        );
                            } else {
                    }
                    break;
                    
                default:
                    // Bei Name oder unbekanntem Typ: auch nach Kennzeichen versuchen
                    searchPromises.push(
                        fetch(`../api/p2_api_proxy.php?action=get_orders_by_license&autohaus_id=${autohaus}&license_plate=${encodeURIComponent(searchTerm)}&limit=1000`)
                            .then(r => r.json())
                            .catch(() => ({success: false, data: []}))
                    );
            }
            
            Promise.all(searchPromises)
            .then((results) => {
                
                // Prüfe ob überhaupt Ergebnisse da sind
                let hasAnyResults = false;
                results.forEach(result => {
                    if (result && result.success && result.data && result.data.length > 0) {
                        hasAnyResults = true;
                    }
                });
                
                // Wenn keine Ergebnisse und nicht bereits alle Typen versucht wurden, versuche Fallback
                if (!hasAnyResults && searchType !== 'name') {
                    const fallbackPromises = [];
                    
                    // Versuche auch nach Kennzeichen (wenn nicht bereits versucht)
                    if (searchType !== 'license') {
                        fallbackPromises.push(
                            fetch(`../api/p2_api_proxy.php?action=get_orders_by_license&autohaus_id=${autohaus}&license_plate=${encodeURIComponent(searchTerm)}&limit=1000`)
                                .then(r => r.json())
                                .then(data => {
                                    return data;
                                })
                                .catch(() => ({success: false, data: []}))
                        );
                    }
                    
                    // Versuche auch nach VIN (wenn nicht bereits versucht)
                    if (searchType !== 'vin_full' && searchType !== 'vin_last6') {
                        const digitsOnly = searchTerm.replace(/[^0-9]/g, '');
                        if (digitsOnly.length >= 6) {
                            fallbackPromises.push(
                                fetch(`../api/p2_api_proxy.php?action=get_orders_by_vin_last6&autohaus_id=${autohaus}&vin_last6=${encodeURIComponent(digitsOnly.slice(-6))}&limit=1000`)
                                    .then(r => r.json())
                                    .then(data => {
                                        return data;
                                    })
                                    .catch(() => ({success: false, data: []}))
                            );
                        }
                    }
                    
                    // Wenn Fallback-Suchen vorhanden, warte auf diese und verarbeite dann alles zusammen
                    if (fallbackPromises.length > 0) {
                        return Promise.all(fallbackPromises).then(fallbackResults => {
                            // Kombiniere mit ursprünglichen Ergebnissen
                            const allResults = [...results, ...fallbackResults];
                            const result = processSearchResults(allResults, searchType, searchTerm, autohaus);
                            return result.customers || result || {};
                        });
                    }
                }
                
                // Verarbeite alle Ergebnisse
                return processSearchResults(results, searchType, searchTerm, autohaus);
            })
            .then((allCustomers) => {
                // Sammle alle Fahrzeuge für jeden Kunden
                const customerIds = Object.keys(allCustomers);
                if (customerIds.length === 0) {
                    return allCustomers;
                }
                return Promise.all(customerIds.map(id => 
                    fetch(`../api/p2_api_proxy.php?action=get_orders_by_customer&autohaus_id=${autohaus}&kunden_adr_id=${id}&days=2000&limit=2000`)
                        .then(r => r.json())
                        .then(data => {
                            if (data.success && data.data) {
                                const vehicles = {};
                                data.data.forEach(order => {
                                    const license = order.ZulassungsKz || '';
                                    const vin = order.Fahrgestellnr || order.VIN || order.vin || order.Fahrgestellnummer || '';
                                    if (license) {
                                        if (!vehicles[license]) {
                                            vehicles[license] = {
                                                license: license,
                                                vin: vin || '',
                                                auftraege: 1,
                                                umsatz: parseFloat(order.Umsatz || 0)
                                            };
                                        } else {
                                            vehicles[license].auftraege++;
                                            vehicles[license].umsatz += parseFloat(order.Umsatz || 0);
                                        }
                                    }
                                });
                                allCustomers[id].allVehicles = Object.values(vehicles);
                                // Initialisiere foundBy falls nicht vorhanden
                                if (!allCustomers[id].foundBy) allCustomers[id].foundBy = [];
                                if (!allCustomers[id].matchingVehicles) allCustomers[id].matchingVehicles = [];
                                if (!allCustomers[id].matchingLicenses) allCustomers[id].matchingLicenses = [];
                            }
                        })
                        .catch(() => {})
                )).then(() => allCustomers);
            })
            .then((allCustomers) => {
                // Filtere ungültige Einträge und stelle sicher, dass KundenAdrID vorhanden ist
                const allCustomersArray = Object.values(allCustomers).filter(c => c && c.KundenAdrID);
                
                // OPTIMIERT: Zeige zuerst die ersten Ergebnisse sofort an (progressive loading)
                if (allCustomersArray.length > 0) {
                    // Zeige zuerst die ersten 50 Kunden sofort an
                    customers = allCustomersArray.slice(0, 50);
                    filteredCustomers = customers;
                    filterCustomers();
                    
                    // Wenn mehr als 50 Ergebnisse, zeige Toast und lade weitere im Hintergrund
                    if (allCustomersArray.length > 50) {
                        showToast('info', 'Ergebnisse geladen', `Zeige ${customers.length} von ${allCustomersArray.length} Kunden. Weitere werden geladen...`);
                        
                        // Lade weitere Ergebnisse schrittweise
                        setTimeout(() => {
                            customers = allCustomersArray.slice(0, 100);
                            filteredCustomers = customers;
                            filterCustomers();
                            if (allCustomersArray.length > 100) {
                                setTimeout(() => {
                                    customers = allCustomersArray;
                                    filteredCustomers = customers;
                                    filterCustomers();
                                    showToast('success', 'Alle Ergebnisse geladen', `${customers.length} Kunden gefunden`);
                                }, 500);
                    } else {
                                showToast('success', 'Alle Ergebnisse geladen', `${customers.length} Kunden gefunden`);
                            }
                        }, 500);
                    } else {
                        customers = allCustomersArray;
                        filteredCustomers = customers;
                        filterCustomers();
                    }
                }
                
                // Debug: Zeige Details der gefundenen Kunden
                if (allCustomersArray.length > 0) {
                }
                
                // Speichere Suchbegriff für Suggestions-Anzeige
                window.currentSearchTerm = searchTerm;
                
                if (allCustomersArray.length === 0) {
                    const suggestions = window.currentSearchSuggestions || [];
                    
                    if (suggestions.length > 0) {
                        showToast('info', 'Keine exakten Ergebnisse', `Keine exakten Treffer für "${searchTerm}". ${suggestions.length} ähnliche Ergebnis${suggestions.length !== 1 ? 'se' : ''} gefunden.`);
                    } else {
                        showToast('info', 'Keine Ergebnisse', `Keine Kunden gefunden für "${searchTerm}". Versuchen Sie einen anderen Suchbegriff.`);
                    }
                    customers = [];
                    filteredCustomers = [];
                    filterCustomers();
                }
                // filterCustomers() wird bereits oben aufgerufen für progressive Anzeige
            })
            .catch(error => {
                showToast('error', 'Fehler', 'Fehler bei der Suche: ' + error.message);
                $('#contentBody').html(`<div class="empty-state"><div class="empty-icon"><i class="fas fa-exclamation-circle"></i></div><h3>Fehler</h3><p>${error.message}</p><p style="font-size: 0.9em; color: var(--text-muted); margin-top: 0.5rem;">Bitte öffnen Sie die Browser-Konsole (F12) für weitere Details.</p></div>`);
            });
        }

        /**
         * Verarbeitet Suchergebnisse und erstellt Kunden-Objekte
         */
        function processSearchResults(results, searchType, searchTerm, autohausId) {
            const allCustomers = {};
            const allSuggestions = []; // Sammle alle Suggestions
            const searchTermUpper = searchTerm.toUpperCase();
            
            // Verarbeite alle Ergebnisse
            results.forEach((result, index) => {
                // Sammle Suggestions aus den Ergebnissen
                if (result && result.suggestions && result.suggestions.length > 0) {
                    result.suggestions.forEach(suggestion => {
                        // Prüfe ob Suggestion bereits vorhanden
                        const exists = allSuggestions.some(s => {
                            if (suggestion.ZulassungsKz) {
                                return s.ZulassungsKz === suggestion.ZulassungsKz;
                            } else if (suggestion.name) {
                                return s.name === suggestion.name && s.KundenAdrID === suggestion.KundenAdrID;
                            }
                            return false;
                        });
                        if (!exists) {
                            allSuggestions.push(suggestion);
                        }
                    });
                }
                if (!result || !result.success) {
                    return;
                }
                
                // Spezialbehandlung für Kunden-ID-Suche
                if (searchType === 'customer_id' && result.data && result.data.length > 0) {
                    const customerData = result.data[0];
                    if (customerData.customer) {
                        const customer = customerData.customer;
                        const id = customer.KundenAdrID;
                        if (id) {
                            allCustomers[id] = {
                                ...customer,
                                foundBy: ['customer_id'],
                                matchingVehicles: [],
                                matchingLicenses: [],
                                allVehicles: [],
                                initialOrders: customerData.orders || []
                            };
                        }
                    }
                    return;
                }
                
                // Standard: Array von Kunden oder Aufträgen
                const data = result.data || [];
                
                if (data.length === 0) {
                    return;
                }
                
                data.forEach((item, itemIndex) => {
                    // Prüfe ob es ein Kunde oder ein Auftrag ist
                    if (item.KundenAdrID && !item.AuftragNr) {
                        // Es ist ein Kunde
                        const id = item.KundenAdrID;
                        if (id && !allCustomers[id]) {
                            allCustomers[id] = {
                                ...item,
                                foundBy: ['name'],
                                matchingVehicles: [],
                                matchingLicenses: [],
                                allVehicles: []
                            };
                        } else if (allCustomers[id]) {
                            if (!allCustomers[id].foundBy) allCustomers[id].foundBy = [];
                            if (!allCustomers[id].foundBy.includes('name')) {
                                allCustomers[id].foundBy.push('name');
                            }
                        }
                    } else if (item.AuftragNr) {
                        // Es ist ein Auftrag
                        const id = item.KundenAdrID;
                        const licensePlate = item.ZulassungsKz || '';
                        const vin = item.Fahrgestellnr || item.VIN || item.vin || item.Fahrgestellnummer || '';
                        
                        
                        if (id) {
                            // Bestimme foundBy basierend auf Suchtyp
                            let foundByType = 'order';
                            if (searchType === 'license') foundByType = 'license';
                            else if (searchType === 'vin_full' || searchType === 'vin_last6') foundByType = 'vin';
                            
                            if (!allCustomers[id]) {
                                allCustomers[id] = {
                                    KundenAdrID: id,
                                    KundenName1: item.KundenName1 || '',
                                    KundenVorname: item.KundenVorname || '',
                                    KundenPlz: item.KundenPlz || '',
                                    KundenOrt: item.KundenOrt || '',
                                    KundenStrasse: item.KundenStrasse || '',
                                    Telefon: item.Telefon || '',
                                    Email: item.Email || '',
                                    AnzahlAuftraege: 1,
                                    GesamtUmsatz: parseFloat(item.Umsatz || 0),
                                    foundBy: [foundByType],
                                    matchingVehicles: vin ? [{vin: vin, license: licensePlate}] : [],
                                    matchingLicenses: licensePlate ? [licensePlate] : [],
                                    allVehicles: []
                                };
                            } else {
                                allCustomers[id].AnzahlAuftraege = (allCustomers[id].AnzahlAuftraege || 0) + 1;
                                allCustomers[id].GesamtUmsatz = (allCustomers[id].GesamtUmsatz || 0) + parseFloat(item.Umsatz || 0);
                                if (!allCustomers[id].foundBy) allCustomers[id].foundBy = [];
                                if (!allCustomers[id].foundBy.includes(foundByType)) {
                                    allCustomers[id].foundBy.push(foundByType);
                                }
                                if (vin) {
                                    if (!allCustomers[id].matchingVehicles) allCustomers[id].matchingVehicles = [];
                                    const vehicleExists = allCustomers[id].matchingVehicles.some(v => v.vin === vin);
                                    if (!vehicleExists) {
                                        allCustomers[id].matchingVehicles.push({vin: vin, license: licensePlate});
                                    }
                                }
                                if (licensePlate) {
                                    if (!allCustomers[id].matchingLicenses) allCustomers[id].matchingLicenses = [];
                                    if (!allCustomers[id].matchingLicenses.includes(licensePlate)) {
                                        allCustomers[id].matchingLicenses.push(licensePlate);
                                    }
                                }
                            }
                        } else {
                        }
                    } else {
                    }
                });
            });
            
            
            // Speichere Suggestions global für spätere Verwendung
            window.currentSearchSuggestions = allSuggestions;
            
            // Gib nur allCustomers zurück (wie vorher erwartet)
            return allCustomers;
        }

        function filterCustomers() {
            // Alle gefundenen Kunden anzeigen (keine weitere Filterung nötig, da bereits gefiltert)
            filteredCustomers = customers;

            // Pagination anwenden
            const startIndex = (currentPage - 1) * itemsPerPage;
            const endIndex = startIndex + itemsPerPage;
            const paginatedCustomers = filteredCustomers.slice(startIndex, endIndex);
            
            updateStats();
            renderView(paginatedCustomers);
        }

        function updateStats() {
            const totalCustomers = filteredCustomers.length;
            const totalOrders = filteredCustomers.reduce((sum, c) => sum + (parseInt(c.AnzahlAuftraege) || 0), 0);
            $('#statTotal').text(totalCustomers);
            $('#statOrders').text(totalOrders);
            const startIndex = (currentPage - 1) * itemsPerPage + 1;
            const endIndex = Math.min(startIndex + itemsPerPage - 1, totalCustomers);
            $('#resultsCount').text(`${startIndex}-${endIndex} von ${totalCustomers} Ergebnis${totalCustomers !== 1 ? 'se' : ''}`);
        }

        function setView(view) {
            // Grid deaktiviert - nur Liste
            currentView = 'list';
            renderView();
        }

        function renderView(customersToRender = null) {
            const body = $('#contentBody');
            const customersToShow = customersToRender || filteredCustomers;
            
            if (filteredCustomers.length === 0) {
                // Zeige Suggestions wenn vorhanden
                const suggestions = window.currentSearchSuggestions || [];
                const searchTerm = window.currentSearchTerm || '';
                
                if (suggestions.length > 0) {
                    let suggestionsHtml = `
                        <div class="empty-state" style="max-width: 800px; margin: 0 auto;">
                            <div class="empty-icon"><i class="fas fa-question-circle" style="color: var(--warning);"></i></div>
                            <h3>Keine exakten Ergebnisse für "${searchTerm}"</h3>
                            <p style="margin-bottom: 1.5rem;">Meinten Sie vielleicht einen der folgenden?</p>
                            <div style="text-align: left; display: inline-block; width: 100%;">
                    `;
                    
                    suggestions.forEach((suggestion, index) => {
                        if (suggestion.ZulassungsKz) {
                            // Kennzeichen-Vorschlag
                            suggestionsHtml += `
                                <div class="suggestion-item" style="padding: 0.75rem; margin-bottom: 0.5rem; border: 1px solid var(--border); border-radius: 6px; cursor: pointer; transition: all 0.2s; background: var(--bg-secondary);"
                                     onmouseover="this.style.background='var(--bg-hover)'; this.style.borderColor='var(--primary)';"
                                     onmouseout="this.style.background='var(--bg-secondary)'; this.style.borderColor='var(--border)';"
                                     onclick="performUniversalSearch('${suggestion.ZulassungsKz.replace(/'/g, "\\'")}')">
                                    <div style="display: flex; align-items: center; gap: 0.75rem;">
                                        <i class="fas fa-car" style="color: var(--primary); font-size: 1.25rem;"></i>
                                        <div style="flex: 1;">
                                            <div style="font-weight: 600; color: var(--text-main);">${suggestion.ZulassungsKz}</div>
                                            <div style="font-size: 0.875rem; color: var(--text-secondary);">${suggestion.anzahl_auftraege || 0} Aufträge</div>
                                        </div>
                                        <i class="fas fa-chevron-right" style="color: var(--text-muted);"></i>
                                    </div>
                                </div>
                            `;
                        } else if (suggestion.name) {
                            // Kundenname-Vorschlag
                            suggestionsHtml += `
                                <div class="suggestion-item" style="padding: 0.75rem; margin-bottom: 0.5rem; border: 1px solid var(--border); border-radius: 6px; cursor: pointer; transition: all 0.2s; background: var(--bg-secondary);"
                                     onmouseover="this.style.background='var(--bg-hover)'; this.style.borderColor='var(--primary)';"
                                     onmouseout="this.style.background='var(--bg-secondary)'; this.style.borderColor='var(--border)';"
                                     onclick="performUniversalSearch('${suggestion.name.replace(/'/g, "\\'")}')">
                                    <div style="display: flex; align-items: center; gap: 0.75rem;">
                                        <i class="fas fa-user" style="color: var(--primary); font-size: 1.25rem;"></i>
                                        <div style="flex: 1;">
                                            <div style="font-weight: 600; color: var(--text-main);">${suggestion.name}</div>
                                            <div style="font-size: 0.875rem; color: var(--text-secondary);">ID: ${suggestion.KundenAdrID || 'N/A'} • ${suggestion.anzahl_auftraege || 0} Aufträge</div>
                                        </div>
                                        <i class="fas fa-chevron-right" style="color: var(--text-muted);"></i>
                                    </div>
                                </div>
                            `;
                        }
                    });
                    
                    suggestionsHtml += `
                            </div>
                            <p style="margin-top: 1.5rem; font-size: 0.875rem; color: var(--text-muted);">
                                Klicken Sie auf einen Vorschlag, um danach zu suchen.
                            </p>
                        </div>
                    `;
                    
                    body.html(suggestionsHtml);
                } else {
                body.html('<div class="empty-state"><div class="empty-icon"><i class="fas fa-search"></i></div><h3>Keine Ergebnisse</h3><p>Passen Sie Ihre Suche an</p></div>');
                }
                return;
            }

            // Nur Liste, kein Grid
            let contentHtml = renderListView(customersToShow);
            
            // Pagination Controls
            const paginationHtml = renderPagination();
            
            body.html(contentHtml + paginationHtml);
        }

        function renderGridView(customersToShow) {
            return `
                <div class="customer-grid">
                    ${customersToShow.map(customer => {
                        // Bestimme warum Kunde gefunden wurde
                        const foundByLabels = {
                            'name': {icon: 'fa-user', label: 'Name', color: '#0056b3'},
                            'license': {icon: 'fa-car', label: 'Kennzeichen', color: '#28a745'},
                            'vin': {icon: 'fa-key', label: 'Fahrgestellnr', color: '#ffc107'}
                        };
                        const foundByBadges = (customer.foundBy || []).map(fb => foundByLabels[fb] || {icon: 'fa-search', label: fb, color: '#6c757d'});
                        const vehicleCount = customer.allVehicles ? customer.allVehicles.length : 0;
                        
                        return `
                        <div class="customer-item ${selectedCustomer && selectedCustomer.KundenAdrID === customer.KundenAdrID ? 'selected' : ''}" 
                             onclick="selectCustomer(${customer.KundenAdrID})">
                            <div class="customer-header">
                                <div class="customer-name">
                                    ${customer.KundenName1 || ''} ${customer.KundenVorname || ''}
                                </div>
                                ${foundByBadges.length > 0 ? `
                                <div style="display: flex; gap: 0.25rem; margin-top: 0.5rem; flex-wrap: wrap;">
                                    ${foundByBadges.map(badge => `
                                        <span class="badge" style="background-color: ${badge.color}; color: white; font-size: 0.7rem; padding: 0.2rem 0.5rem; border-radius: 9999px; display: inline-flex; align-items: center; gap: 0.25rem;">
                                            <i class="fas ${badge.icon}" style="font-size: 0.65rem;"></i>
                                            ${badge.label}
                                        </span>
                                    `).join('')}
                                </div>
                                ` : ''}
                            </div>
                            <div class="customer-info">
                                <div class="info-row">
                                    <span class="info-label">Kunden-ID:</span>
                                    <span class="info-value">${customer.KundenAdrID || '-'}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Aufträge:</span>
                                    <span class="info-value">${customer.AnzahlAuftraege || 0}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Umsatz:</span>
                                    <span class="info-value">${formatCurrency(customer.GesamtUmsatz || 0)}</span>
                                </div>
                                ${vehicleCount > 0 ? `
                                <div class="info-row">
                                    <span class="info-label">Fahrzeuge:</span>
                                    <span class="info-value" style="display: flex; flex-direction: column; gap: 0.25rem; margin-top: 0.25rem;">
                                        ${customer.allVehicles.slice(0, 3).map(v => `
                                            <span style="font-size: 0.75rem; padding: 0.2rem 0.4rem; background: var(--bg-hover); border-radius: var(--radius); display: inline-block;">
                                                <i class="fas fa-car" style="margin-right: 0.25rem; color: var(--primary);"></i>
                                                ${v.license || '-'} ${v.vin ? `(${v.vin.slice(-6)})` : ''}
                                            </span>
                                        `).join('')}
                                        ${vehicleCount > 3 ? `<span style="font-size: 0.7rem; color: var(--text-secondary);">+${vehicleCount - 3} weitere</span>` : ''}
                                    </span>
                                </div>
                                ` : ''}
                                ${customer.matchingLicenses && customer.matchingLicenses.length > 0 ? `
                                <div class="info-row">
                                    <span class="info-label">Gefunden durch:</span>
                                    <span class="info-value" style="display: flex; gap: 0.25rem; flex-wrap: wrap; margin-top: 0.25rem;">
                                        ${customer.matchingLicenses.map(license => `
                                            <span class="badge badge-success" style="font-size: 0.7rem; padding: 0.2rem 0.5rem;">
                                                <i class="fas fa-car"></i> ${license}
                                            </span>
                                        `).join('')}
                                    </span>
                                </div>
                                ` : ''}
                                ${customer.KundenStrasse ? `
                                <div class="info-row">
                                    <span class="info-label">Adresse:</span>
                                    <span class="info-value">${customer.KundenStrasse || '-'}</span>
                                </div>
                                ` : ''}
                                <div class="info-row">
                                    <span class="info-label">Ort:</span>
                                    <span class="info-value">${(customer.KundenPlz || '') + ' ' + (customer.KundenOrt || '') || '-'}</span>
                                </div>
                                ${customer.Telefon ? `
                                <div class="info-row">
                                    <span class="info-label">Telefon:</span>
                                    <span class="info-value">${customer.Telefon || '-'}</span>
                                </div>
                                ` : ''}
                                ${customer.Email ? `
                                <div class="info-row">
                                    <span class="info-label">E-Mail:</span>
                                    <span class="info-value">${customer.Email || '-'}</span>
                                </div>
                                ` : ''}
                            </div>
                        </div>
                    `;
                    }).join('')}
                </div>
            `;
        }

        function renderListView(customersToShow) {
            const foundByLabels = {
                'name': {icon: 'fa-user', label: 'Name', color: '#0056b3'},
                'license': {icon: 'fa-car', label: 'Kennzeichen', color: '#28a745'},
                'vin': {icon: 'fa-key', label: 'Fahrgestellnr', color: '#ffc107'}
            };
            
            return `
                <div style="background: var(--bg-panel); border-radius: var(--radius-lg); overflow: hidden; box-shadow: var(--shadow-sm);">
                    <table class="orders-table" style="margin: 0;">
                        <thead>
                            <tr style="background: var(--bg-hover);">
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border);">Kunde</th>
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border);">Kunden-ID</th>
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border); text-align: center;">Aufträge</th>
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border); text-align: right;">Umsatz</th>
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border);">Fahrzeuge</th>
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border);">Gefunden durch</th>
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border);">Ort</th>
                                <th style="padding: 1rem; font-weight: 600; color: var(--text-main); border-bottom: 2px solid var(--border); text-align: center;">Aktionen</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${customersToShow.map(customer => {
                                const foundByBadges = (customer.foundBy || []).map(fb => foundByLabels[fb] || {icon: 'fa-search', label: fb, color: '#6c757d'});
                                const vehicleCount = customer.allVehicles ? customer.allVehicles.length : 0;
                                
                                return `
                                <tr onclick="selectCustomer(${customer.KundenAdrID})" 
                                    style="cursor: pointer; transition: background 0.2s; border-bottom: 1px solid var(--border);"
                                    onmouseover="this.style.background='var(--bg-hover)';"
                                    onmouseout="this.style.background='';">
                                    <td style="padding: 1rem; font-weight: 500; color: var(--text-main);">
                                        <div style="display: flex; flex-direction: column; gap: 0.25rem;">
                                            <div style="display: flex; align-items: center; gap: 0.5rem;">
                                                <i class="fas fa-user" style="color: var(--primary); font-size: 0.875rem;"></i>
                                                <span>${customer.KundenName1 || ''} ${customer.KundenVorname || ''}</span>
                                            </div>
                                            ${customer.KundenStrasse ? `
                                            <div style="font-size: 0.75rem; color: var(--text-secondary); padding-left: 1.5rem;">
                                                ${customer.KundenStrasse}
                                            </div>
                                            ` : ''}
                                        </div>
                                    </td>
                                    <td style="padding: 1rem; color: var(--text-secondary); font-size: 0.875rem;">${customer.KundenAdrID || '-'}</td>
                                    <td style="padding: 1rem; text-align: center;">
                                        <span class="badge badge-info" style="padding: 0.375rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 600;">
                                            ${customer.AnzahlAuftraege || 0}
                                        </span>
                                    </td>
                                    <td style="padding: 1rem; text-align: right; font-weight: 600; color: var(--success);">${formatCurrency(customer.GesamtUmsatz || 0)}</td>
                                    <td style="padding: 1rem;">
                                        ${vehicleCount > 0 ? `
                                            <div style="display: flex; flex-direction: column; gap: 0.25rem;">
                                                ${customer.allVehicles.slice(0, 2).map(v => `
                                                    <span style="font-size: 0.75rem; padding: 0.25rem 0.5rem; background: var(--bg-hover); border-radius: var(--radius); display: inline-block; width: fit-content;">
                                                        <i class="fas fa-car" style="margin-right: 0.25rem; color: var(--primary);"></i>
                                                        ${v.license || '-'}
                                                    </span>
                                                `).join('')}
                                                ${vehicleCount > 2 ? `<span style="font-size: 0.7rem; color: var(--text-secondary);">+${vehicleCount - 2} weitere</span>` : ''}
                                            </div>
                                        ` : '<span style="color: var(--text-secondary); font-size: 0.875rem;">-</span>'}
                                    </td>
                                    <td style="padding: 1rem;">
                                        ${foundByBadges.length > 0 ? `
                                            <div style="display: flex; gap: 0.25rem; flex-wrap: wrap;">
                                                ${foundByBadges.map(badge => `
                                                    <span class="badge" style="background-color: ${badge.color}; color: white; font-size: 0.7rem; padding: 0.25rem 0.5rem; border-radius: 9999px; display: inline-flex; align-items: center; gap: 0.25rem;">
                                                        <i class="fas ${badge.icon}" style="font-size: 0.65rem;"></i>
                                                        ${badge.label}
                                                    </span>
                                                `).join('')}
                                            </div>
                                        ` : '<span style="color: var(--text-secondary); font-size: 0.875rem;">-</span>'}
                                        ${customer.matchingLicenses && customer.matchingLicenses.length > 0 ? `
                                            <div style="margin-top: 0.25rem; display: flex; gap: 0.25rem; flex-wrap: wrap;">
                                                ${customer.matchingLicenses.map(license => `
                                                    <span class="badge badge-success" style="font-size: 0.65rem; padding: 0.2rem 0.4rem;">
                                                        ${license}
                                                    </span>
                                                `).join('')}
                                            </div>
                                        ` : ''}
                                    </td>
                                    <td style="padding: 1rem; color: var(--text-secondary); font-size: 0.875rem;">${(customer.KundenPlz || '') + ' ' + (customer.KundenOrt || '') || '-'}</td>
                                    <td style="padding: 1rem; text-align: center;">
                                        <button class="btn btn-primary" 
                                                onclick="event.stopPropagation(); selectCustomer(${customer.KundenAdrID})"
                                                style="padding: 0.5rem 1rem; font-size: 0.875rem; border-radius: var(--radius);">
                                            <i class="fas fa-eye"></i> Details
                                        </button>
                                    </td>
                                </tr>
                            `;
                            }).join('')}
                        </tbody>
                    </table>
                </div>
            `;
        }

        function renderPagination() {
            // Prüfe ob Pagination nötig ist
            if (!filteredCustomers || filteredCustomers.length <= itemsPerPage) {
                return ''; // Keine Pagination nötig
            }

            totalPages = Math.ceil(filteredCustomers.length / itemsPerPage);
            if (totalPages <= 1) {
                return ''; // Nur eine Seite, keine Pagination nötig
            }

            return `
                <div style="display: flex; justify-content: center; align-items: center; gap: 0.5rem; margin-top: 1.5rem; padding: 1rem; border-top: 1px solid var(--border);">
                    <button class="btn btn-secondary" onclick="loadCustomersPage(${currentPage - 1})" ${currentPage === 1 ? 'disabled style="opacity: 0.5; cursor: not-allowed;"' : ''}>
                        <i class="fas fa-chevron-left"></i> Zurück
                    </button>
                    <span style="padding: 0.5rem 1rem; background: var(--bg-hover); border-radius: var(--radius); font-size: 0.875rem;">
                        Seite ${currentPage} von ${totalPages} (${filteredCustomers.length} Kunden)
                    </span>
                    <button class="btn btn-secondary" onclick="loadCustomersPage(${currentPage + 1})" ${currentPage >= totalPages ? 'disabled style="opacity: 0.5; cursor: not-allowed;"' : ''}>
                        Weiter <i class="fas fa-chevron-right"></i>
                    </button>
                </div>
            `;
        }

        function loadCustomersPage(page) {
            if (page < 1 || page > totalPages) return;
            currentPage = page;
            filterCustomers();
        }

        // Toast Notification System
        function showToast(type, title, message) {
            // Erstelle Toast-Container falls nicht vorhanden
            let container = $('.toast-container');
            if (!container.length) {
                container = $('<div class="toast-container"></div>');
                $('body').append(container);
            }

            // Icons für verschiedene Typen
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };

            // Erstelle Toast
            const toast = $(`
                <div class="toast ${type}">
                    <i class="fas ${icons[type] || icons.info} toast-icon"></i>
                    <div class="toast-content">
                        <div class="toast-title">${title}</div>
                        <div class="toast-message">${message}</div>
                    </div>
                    <button class="toast-close" onclick="$(this).closest('.toast').fadeOut(300, function() { $(this).remove(); })">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `);

            container.append(toast);

            // Auto-Entfernen nach 5 Sekunden
            setTimeout(() => {
                toast.css('animation', 'slideOutRight 0.3s ease-out');
                setTimeout(() => toast.remove(), 300);
            }, 5000);
        }

        function selectCustomer(kundenAdrId) {
            selectedCustomer = filteredCustomers.find(c => c.KundenAdrID == kundenAdrId);
            if (!selectedCustomer) return;

            // Wenn Kunde über Auftragsnummer-Suche gefunden wurde, öffne direkt das Auftrags-Modal
            if (selectedCustomer.foundBy && selectedCustomer.foundBy.includes('order_number') && selectedCustomer.searchedOrderNumber) {
                const orderNumber = selectedCustomer.searchedOrderNumber;
                // Extrahiere die Nummer (falls Format "X - XXXXXX")
                const orderNumberClean = orderNumber.includes(' - ') ? orderNumber.split(' - ')[1].trim() : orderNumber.trim();
                // Konvertiere zu Integer für die viewOrder Funktion
                const auftragNr = parseInt(orderNumberClean);
                if (!isNaN(auftragNr)) {
                    window.viewOrder(auftragNr);
                    return;
                }
            }

            // Standard: Open detail panel
            $('#detailPanel').addClass('open');
            $('#panelTitle').text(`${selectedCustomer.KundenName1 || ''} ${selectedCustomer.KundenVorname || ''}`);
            
            // Load customer details
            loadCustomerDetails(kundenAdrId);
        }

        function loadCustomerDetails(kundenAdrId) {
            const autohaus = $('#autohausSelect').val();
            $('#panelBody').html('<div class="loading"><div class="spinner"></div><p>Lade Details...</p></div>');

            Promise.all([
                fetch(`../api/p2_api_proxy.php?action=get_customer&autohaus_id=${autohaus}&kunden_adr_id=${kundenAdrId}`).then(r => r.json()),
                fetch(`../api/p2_api_proxy.php?action=get_orders_by_customer&autohaus_id=${autohaus}&kunden_adr_id=${kundenAdrId}&days=2000&limit=2000`).then(r => r.json()),
                fetch(`../api/p2_tire_offers.php?action=get&kunden_adr_id=${kundenAdrId}&autohaus_id=${autohaus}`).then(r => r.json())
            ])
            .then(([customerData, ordersData, offersData]) => {
                const customer = customerData.success ? customerData.data : selectedCustomer;
                let orders = [];
                
                
                // Prüfe ob ordersData erfolgreich ist
                if (ordersData.success && ordersData.data && Array.isArray(ordersData.data)) {
                    orders = ordersData.data;
                    
                    // Debug: Zeige erste Aufträge und prüfe KundenAdrID
                    if (orders.length > 0) {
                    } else {
                        
                        // Alternative: Lade alle Aufträge und filtere clientseitig
                        fetch(`../api/p2_api_proxy.php?action=get_all_orders&autohaus_id=${autohaus}&days=2000&limit=2000`)
                            .then(r => r.json())
                            .then(allOrdersData => {
                                if (allOrdersData.success && allOrdersData.data) {
                                    const allOrders = allOrdersData.data;
                                    
                                    // Debug: Zeige erste Aufträge mit allen Feldern
                                    if (allOrders.length > 0) {
                                    }
                                    
                                    orders = allOrders.filter(o => {
                                        const orderKundenId = o.KundenAdrID || o.kunden_adr_id || o.KundenAdrId;
                                        if (!orderKundenId) {
                                            return false;
                                        }
                                        // Prüfe sowohl String- als auch Integer-Vergleich
                                        const match = parseInt(orderKundenId) === parseInt(kundenAdrId) || 
                                                     String(orderKundenId) === String(kundenAdrId);
                                        if (match) {
                                        }
                                        return match;
                                    });
                                    
                                    if (orders.length === 0 && allOrders.length > 0) {
                                    }
                                    
                                    renderCustomerDetails(customer, orders, offersData.success ? offersData.offers : []);
                                } else {
                                    renderCustomerDetails(customer, [], offersData.success ? offersData.offers : []);
                                }
                            })
                            .catch(err => {
                                renderCustomerDetails(customer, [], offersData.success ? offersData.offers : []);
                            });
                        return; // Warte auf alternative Methode
                    }
                } else {
                }
                
                renderCustomerDetails(customer, orders, offersData.success ? offersData.offers : []);
            })
            .catch(error => {
                $('#panelBody').html(`<div class="empty-state"><p>Fehler beim Laden: ${error.message}</p></div>`);
            });
        }

        function renderCustomerDetails(customer, orders, offers) {
            const kundenAdrId = customer.KundenAdrID || selectedCustomer?.KundenAdrID;
            
            // Gruppiere Aufträge nach Fahrzeugen (Kennzeichen)
            const ordersByVehicle = {};
            const ordersWithoutVehicle = [];
            
            orders.forEach(order => {
                const license = (order.ZulassungsKz || '').trim();
                if (license) {
                    if (!ordersByVehicle[license]) {
                        ordersByVehicle[license] = {
                            license: license,
                            vin: order.Fahrgestellnr || order.VIN || order.vin || order.Fahrgestellnummer || '',
                            orders: [],
                            totalUmsatz: 0,
                            totalMarge: 0,
                            totalMargeProzent: 0
                        };
                    }
                    ordersByVehicle[license].orders.push(order);
                    ordersByVehicle[license].totalUmsatz += parseFloat(order.Umsatz || 0);
                    ordersByVehicle[license].totalMarge += parseFloat(order.Marge || 0);
                } else {
                    ordersWithoutVehicle.push(order);
                }
            });
            
            // Berechne durchschnittliche Marge pro Fahrzeug
            Object.keys(ordersByVehicle).forEach(license => {
                const vehicle = ordersByVehicle[license];
                if (vehicle.orders.length > 0) {
                    const totalEK = vehicle.orders.reduce((sum, o) => sum + parseFloat(o.EK || 0), 0);
                    vehicle.totalMargeProzent = totalEK > 0 ? (vehicle.totalMarge / totalEK) * 100 : 0;
                }
            });
            
            // Sortiere Aufträge innerhalb jedes Fahrzeugs nach Datum (neueste zuerst)
            Object.keys(ordersByVehicle).forEach(license => {
                ordersByVehicle[license].orders.sort((a, b) => {
                    const dateA = new Date(a.DatumAuftrag || 0);
                    const dateB = new Date(b.DatumAuftrag || 0);
                    return dateB - dateA; // Neueste zuerst
                });
            });
            
            // Sortiere Aufträge ohne Fahrzeug nach Datum
            ordersWithoutVehicle.sort((a, b) => {
                const dateA = new Date(a.DatumAuftrag || 0);
                const dateB = new Date(b.DatumAuftrag || 0);
                return dateB - dateA; // Neueste zuerst
            });
            
            // Sortiere Fahrzeuge nach Anzahl Aufträge (absteigend), dann nach neuestem Auftrag
            const sortedVehicles = Object.values(ordersByVehicle).sort((a, b) => {
                if (b.orders.length !== a.orders.length) {
                    return b.orders.length - a.orders.length; // Mehr Aufträge zuerst
                }
                // Bei gleicher Anzahl: neuestes Datum zuerst
                const dateA = a.orders.length > 0 ? new Date(a.orders[0].DatumAuftrag || 0) : 0;
                const dateB = b.orders.length > 0 ? new Date(b.orders[0].DatumAuftrag || 0) : 0;
                return dateB - dateA;
            });
            
            let html = `
                <div class="detail-section">
                    <div class="section-title">Kundendaten</div>
                    <div class="detail-grid">
                        <div class="detail-row">
                            <span class="detail-label">Name:</span>
                            <span class="detail-value">${customer.KundenName1 || ''} ${customer.KundenVorname || ''}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Adresse:</span>
                            <span class="detail-value">${customer.KundenStrasse || '-'}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">PLZ / Ort:</span>
                            <span class="detail-value">${customer.KundenPlz || ''} ${customer.KundenOrt || ''}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Telefon:</span>
                            <span class="detail-value">${customer.Telefon || '-'}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">E-Mail:</span>
                            <span class="detail-value">${customer.Email || '-'}</span>
                        </div>
                    </div>
                </div>

                <div class="detail-section">
                    <div class="section-title">Aufträge (${orders.length}) - ${sortedVehicles.length} Fahrzeug${sortedVehicles.length !== 1 ? 'e' : ''}</div>
                        ${orders.length > 0 ? `
                            ${sortedVehicles.map((vehicle, vehicleIndex) => {
                                const vehicleMarginColor = vehicle.totalMargeProzent > 50 ? 'var(--success)' : vehicle.totalMargeProzent > 0 ? 'var(--warning)' : 'var(--danger)';
                                return `
                                    <div class="vehicle-orders-group" style="margin-bottom: 2rem;">
                                        <div class="vehicle-group-header">
                                            <div class="vehicle-group-title">
                                                <i class="fas fa-car" style="color: var(--primary); margin-right: 0.5rem;"></i>
                                                <span class="vehicle-license">${vehicle.license}</span>
                                                ${vehicle.vin ? `<span class="vehicle-vin" style="font-size: 0.75rem; color: var(--text-secondary); margin-left: 0.5rem;">(${vehicle.vin.slice(-6)})</span>` : ''}
                                            </div>
                                            <div class="vehicle-group-stats">
                                                <span class="vehicle-stat">
                                                    <span class="vehicle-stat-label">Aufträge:</span>
                                                    <span class="vehicle-stat-value">${vehicle.orders.length}</span>
                                                </span>
                                                <span class="vehicle-stat">
                                                    <span class="vehicle-stat-label">Umsatz:</span>
                                                    <span class="vehicle-stat-value" style="color: var(--success);">${formatCurrency(vehicle.totalUmsatz)}</span>
                                                </span>
                                                <span class="vehicle-stat">
                                                    <span class="vehicle-stat-label">Marge:</span>
                                                    <span class="vehicle-stat-value" style="color: ${vehicleMarginColor}; font-weight: 600;">${formatCurrency(vehicle.totalMarge)} (${vehicle.totalMargeProzent.toFixed(1)}%)</span>
                                                </span>
                                            </div>
                                        </div>
                                        <div class="orders-compact-list" style="margin-top: 1rem;">
                                            ${vehicle.orders.map(order => {
                                                const marginClass = getMarginBadgeClass(order.MargeProzent);
                                                const marginColor = order.MargeProzent > 50 ? 'var(--success)' : order.MargeProzent > 0 ? 'var(--warning)' : 'var(--danger)';
                                                return `
                                                    <div class="order-compact-item" onclick="event.stopPropagation(); window.viewOrder(${order.AuftragNr})">
                                                        <div class="order-compact-main">
                                                            <div class="order-compact-header">
                                                                <span class="order-compact-number" onclick="event.stopPropagation(); window.viewOrder(${order.AuftragNr})">#${order.AuftragNr || '-'}</span>
                                                                <span class="order-compact-date">${formatDate(order.DatumAuftrag)}</span>
                                                                <span class="badge badge-info" style="font-size: 0.7rem; padding: 0.2rem 0.5rem;">${order.Auftragstatus || '-'}</span>
                                                            </div>
                                                            <div class="order-compact-body">
                                                                <div class="order-compact-info">
                                                                    ${order.Auftragtext ? `
                                                                    <div class="order-compact-text" style="font-size: 0.8rem; color: var(--text-secondary); margin-top: 0.25rem;">
                                                                        ${order.Auftragtext.length > 60 ? order.Auftragtext.substring(0, 60) + '...' : order.Auftragtext}
                                                                    </div>
                                                                    ` : ''}
                                                                </div>
                                                                <div class="order-compact-financial">
                                                                    <div class="order-compact-amount">
                                                                        <span class="order-compact-label">Umsatz:</span>
                                                                        <span class="order-compact-value">${formatCurrency(order.Umsatz || 0)}</span>
                                                                    </div>
                                                                    <div class="order-compact-margin">
                                                                        <span class="order-compact-label">Marge:</span>
                                                                        <span class="order-compact-value" style="color: ${marginColor}; font-weight: 600;">
                                                                            ${formatCurrency(order.Marge || 0)} (${(order.MargeProzent || 0).toFixed(1)}%)
                                                                        </span>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                `;
                                            }).join('')}
                                        </div>
                                    </div>
                                `;
                            }).join('')}
                            ${ordersWithoutVehicle.length > 0 ? `
                                <div class="vehicle-orders-group" style="margin-bottom: 2rem; margin-top: 2rem;">
                                    <div class="vehicle-group-header">
                                        <div class="vehicle-group-title">
                                            <i class="fas fa-question-circle" style="color: var(--text-secondary); margin-right: 0.5rem;"></i>
                                            <span class="vehicle-license">Ohne Kennzeichen</span>
                                        </div>
                                        <div class="vehicle-group-stats">
                                            <span class="vehicle-stat">
                                                <span class="vehicle-stat-label">Aufträge:</span>
                                                <span class="vehicle-stat-value">${ordersWithoutVehicle.length}</span>
                                            </span>
                                        </div>
                                    </div>
                                    <div class="orders-compact-list" style="margin-top: 1rem;">
                                        ${ordersWithoutVehicle.map(order => {
                                            const marginClass = getMarginBadgeClass(order.MargeProzent);
                                            const marginColor = order.MargeProzent > 50 ? 'var(--success)' : order.MargeProzent > 0 ? 'var(--warning)' : 'var(--danger)';
                                            return `
                                                <div class="order-compact-item" onclick="event.stopPropagation(); window.viewOrder(${order.AuftragNr})">
                                                    <div class="order-compact-main">
                                                        <div class="order-compact-header">
                                                            <span class="order-compact-number" onclick="event.stopPropagation(); window.viewOrder(${order.AuftragNr})">#${order.AuftragNr || '-'}</span>
                                                            <span class="order-compact-date">${formatDate(order.DatumAuftrag)}</span>
                                                            <span class="badge badge-info" style="font-size: 0.7rem; padding: 0.2rem 0.5rem;">${order.Auftragstatus || '-'}</span>
                                                        </div>
                                                        <div class="order-compact-body">
                                                            <div class="order-compact-info">
                                                                ${order.Auftragtext ? `
                                                                <div class="order-compact-text" style="font-size: 0.8rem; color: var(--text-secondary); margin-top: 0.25rem;">
                                                                    ${order.Auftragtext.length > 60 ? order.Auftragtext.substring(0, 60) + '...' : order.Auftragtext}
                                                                </div>
                                                                ` : ''}
                                                            </div>
                                                            <div class="order-compact-financial">
                                                                <div class="order-compact-amount">
                                                                    <span class="order-compact-label">Umsatz:</span>
                                                                    <span class="order-compact-value">${formatCurrency(order.Umsatz || 0)}</span>
                                                                </div>
                                                                <div class="order-compact-margin">
                                                                    <span class="order-compact-label">Marge:</span>
                                                                    <span class="order-compact-value" style="color: ${marginColor}; font-weight: 600;">
                                                                        ${formatCurrency(order.Marge || 0)} (${(order.MargeProzent || 0).toFixed(1)}%)
                                                                    </span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            `;
                                        }).join('')}
                                    </div>
                                </div>
                            ` : ''}
                        ` : '<p style="color: var(--text-secondary);">Keine Aufträge gefunden</p>'}
                    </div>

                    <div class="detail-section">
                        <div class="section-title">Reifenangebote (${offers.length})</div>
                        ${offers.length > 0 ? `
                            <div style="display: flex; flex-direction: column; gap: 0.5rem;">
                                ${offers.map(offer => `
                                    <div style="padding: 1rem; background: var(--bg-hover); border-radius: var(--radius); cursor: pointer; transition: all 0.2s;" 
                                         onclick="viewP2OfferDetails(${offer.id})" 
                                         onmouseover="this.style.background='#e9ecef'; this.style.transform='translateY(-2px)'; this.style.boxShadow='0 2px 8px rgba(0,0,0,0.1)';" 
                                         onmouseout="this.style.background='var(--bg-hover)'; this.style.transform='none'; this.style.boxShadow='none';">
                                        <div style="display: flex; justify-content: space-between; align-items: center;">
                                            <div>
                                                <div style="font-weight: 600;">${offer.license_plate || '-'}</div>
                                                <div style="font-size: 0.875rem; color: var(--text-secondary);">${formatDate(offer.created_at)}</div>
                                            </div>
                                            <span class="badge ${getStatusBadgeClass(offer.status)}">${offer.status || 'draft'}</span>
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                        ` : '<p style="color: var(--text-secondary);">Keine Reifenangebote</p>'}
                        ${kundenAdrId ? `
                        <button class="btn btn-primary" style="margin-top: 1rem;" onclick="createTireOffer(${kundenAdrId})">
                            <i class="fas fa-plus"></i> Neues Reifenangebot
                        </button>
                        ` : ''}
                    </div>
            `;

            $('#panelBody').html(html);
        }

        function closeDetailPanel() {
            $('#detailPanel').removeClass('open');
            selectedCustomer = null;
        }

        function formatCurrency(value) {
            return new Intl.NumberFormat('de-DE', { style: 'currency', currency: 'EUR' }).format(value);
        }

        function formatDate(dateString) {
            if (!dateString) return '-';
            const date = new Date(dateString);
            return date.toLocaleDateString('de-DE');
        }

        function getMarginBadgeClass(marginPercent) {
            if (marginPercent >= 30) return 'badge-success';
            if (marginPercent >= 20) return 'badge-info';
            if (marginPercent >= 10) return 'badge-warning';
            return 'badge-danger';
        }

        function getStatusBadgeClass(status) {
            const statusMap = {
                'draft': 'badge-secondary',
                'sent': 'badge-info',
                'accepted': 'badge-success',
                'declined': 'badge-danger',
                'new_request': 'badge-warning'
            };
            return statusMap[status] || 'badge-secondary';
        }

        // ============ AUFTRÄGE-DETAILS ============
        // viewOrder Funktion - direkt als window.viewOrder definiert für globale Verfügbarkeit
        window.viewOrder = function(auftragNr) {
            const autohaus = $('#autohausSelect').val();
            if (!autohaus) {
                showToast('warning', 'Autohaus erforderlich', 'Bitte wählen Sie zuerst ein Autohaus aus');
                return;
            }

            // Erstelle Modal wie in appointments.php mit Overlay
            let modal = document.getElementById('p2OrderDetailsModal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'p2OrderDetailsModal';
                modal.className = 'modal-overlay';
                modal.style.cssText = 'position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; display: none; align-items: center; justify-content: center;';
                modal.onclick = function(e) {
                    if (e.target === modal) {
                        closeP2OrderModal();
                    }
                };
                modal.innerHTML = `
                    <div class="modal-content" style="max-width: 95vw; width: 95vw; max-height: 90vh; height: 90vh; display: flex; flex-direction: column; padding: 0; margin: auto; position: relative; background: #fff; border-radius: 8px; box-shadow: 0 10px 40px rgba(0,0,0,0.3);">
                        <div class="modal-header" style="flex-shrink: 0; border-bottom: 2px solid #dee2e6; padding: 1rem 1.5rem; background: #fff; position: sticky; top: 0; z-index: 100; border-radius: 8px 8px 0 0;">
                            <h3 class="modal-title" style="margin: 0;">Auftragsdetails</h3>
                            <button class="modal-close" onclick="closeP2OrderModal()" style="font-size: 1.5rem; padding: 0.25rem 0.5rem; position: absolute; right: 1rem; top: 50%; transform: translateY(-50%); background: none; border: none; cursor: pointer; color: #6c757d;">&times;</button>
                        </div>
                        <div id="p2OrderDetailsBody" style="flex: 1; overflow-y: auto; overflow-x: hidden; display: flex; flex-direction: column; min-height: 0;">
                            <div style="text-align: center; padding: 2rem;">
                                <div class="spinner" style="margin: 0 auto;"></div>
                                <p style="margin-top: 1rem;">Lade Auftragsdetails...</p>
                            </div>
                        </div>
                    </div>
                `;
                document.body.appendChild(modal);
            }
            
            // Zeige Modal
            modal.style.display = 'flex';
            const body = document.getElementById('p2OrderDetailsBody');
            if (body) {
                body.innerHTML = '<div style="text-align: center; padding: 2rem;"><div class="spinner" style="margin: 0 auto;"></div><p style="margin-top: 1rem;">Lade Auftragsdetails...</p></div>';
            }

            // Lade Auftragsdetails und Positionen
            Promise.all([
                fetch(`../api/p2_api_proxy.php?action=get_order&autohaus_id=${autohaus}&auftrag_nr=${auftragNr}`).then(r => {
                    if (!r.ok) {
                        throw new Error(`HTTP ${r.status}: ${r.statusText}`);
                    }
                    return r.json();
                }),
                fetch(`../api/p2_api_proxy.php?action=get_order_positions&autohaus_id=${autohaus}&auftrag_nr=${auftragNr}`).then(r => {
                    if (!r.ok) {
                        return {success: false, data: []};
                    }
                    return r.json();
                })
            ])
            .then(([orderData, positionsData]) => {
                
                const order = orderData.success ? orderData.data : null;
                const positions = positionsData.success ? positionsData.data : [];
                const body = document.getElementById('p2OrderDetailsBody');

                if (!order) {
                    body.innerHTML = '<div class="empty-state"><p>Auftrag nicht gefunden</p><p style="font-size: 0.875rem; color: var(--text-secondary); margin-top: 0.5rem;">Auftragsnummer: ' + auftragNr + '</p></div>';
                    return;
                }
                
                // Speichere Status global
                window.currentP2OrderStatus = order.Auftragstatus || '';
                window.currentP2OrderNumber = order.AuftragNr || null;
                
                // Prüfe ob Auftrag readonly ist
                const statusInfo = getStatusInfo(window.currentP2OrderStatus);
                const isReadOnly = statusInfo.readonly || window.currentP2OrderStatus === 'AB';

                // Berechne Gesamtwerte
                const totalVK = positions.reduce((sum, p) => sum + (parseFloat(p.VK || p.SumVK || p.VKGesamt || 0)), 0);
                const totalEK = positions.reduce((sum, p) => sum + (parseFloat(p.EK || p.SumEK || 0)), 0);
                const totalMarge = totalVK - totalEK;
                const totalMargePercent = totalEK > 0 ? (totalMarge / totalEK * 100) : 0;

                // Erstelle Tab-basiertes Layout wie in appointments.php
                let html = `
                    <!-- Header mit Auftragsinformationen -->
                    <div style="background: #f8f9fa; border-bottom: 2px solid #dee2e6; padding: 1rem 1.5rem; ${isReadOnly ? 'border-left: 4px solid #dc3545;' : ''}">
                        ${isReadOnly ? `
                            <div style="background: #fff3cd; border: 1px solid #ffc107; border-radius: 4px; padding: 0.75rem 1rem; margin-bottom: 1rem; display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-lock" style="color: #856404; font-size: 1.1rem;"></i>
                                <strong style="color: #856404; font-size: 0.9rem;">Auftrag hat Status "${statusInfo.name}" (${window.currentP2OrderStatus}) - Nur lesen erlaubt!</strong>
                            </div>
                        ` : ''}
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 2rem;">
                            <div>
                                <div style="display: grid; grid-template-columns: auto 1fr; gap: 0.5rem 1rem; font-size: 0.875rem;">
                                    <strong>Auftragart:</strong> <span>${order.Auftragart || '-'}</span>
                                    <strong>Auftragstatus:</strong> 
                                    <span style="display: flex; align-items: center; gap: 0.5rem;">
                                        <span class="badge" style="background-color: ${statusInfo.color}; color: white; font-size: 0.8rem; padding: 0.35rem 0.65rem; border-radius: 4px; font-weight: 600;" title="${statusInfo.description}">
                                            ${order.Auftragstatus || '-'} - ${statusInfo.name}
                                        </span>
                                    </span>
                                </div>
                            </div>
                            <div>
                                <div style="display: grid; grid-template-columns: auto 1fr; gap: 0.5rem 1rem; font-size: 0.875rem;">
                                    <strong>Auftragsnummer:</strong> <span>${order.AuftragNr || '-'}</span>
                                    <strong>Auftragdatum:</strong> <span>${formatDate(order.DatumAuftrag)}</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Tabs -->
                    <div style="background: #fff; border-bottom: 1px solid #dee2e6; display: flex; gap: 0;">
                        <button class="p2-tab-btn active" data-tab="uebersicht" onclick="switchP2Tab('uebersicht')" style="padding: 0.75rem 1.5rem; border: none; background: none; border-bottom: 3px solid #0056b3; color: #0056b3; font-weight: 600; cursor: pointer;">
                            Übersicht
                        </button>
                        <button class="p2-tab-btn" data-tab="positionen" onclick="switchP2Tab('positionen')" style="padding: 0.75rem 1.5rem; border: none; background: none; border-bottom: 3px solid transparent; color: #6c757d; cursor: pointer;">
                            Positionen ${positions.length > 0 ? `(${positions.length})` : ''}
                        </button>
                    </div>
                    
                    <!-- Tab Content -->
                    <div style="flex: 1; overflow-y: auto; overflow-x: hidden; display: flex; flex-direction: column; min-height: 0;">
                        <!-- Übersicht Tab -->
                        <div id="p2TabUebersicht" class="p2-tab-content" style="display: block; padding: 1.5rem; overflow-y: auto;">
                            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1rem;">
                                <!-- Grunddaten -->
                                <div style="background: #fff; padding: 1rem; border-radius: 6px; border: 1px solid #dee2e6; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                                    <h4 style="margin: 0 0 0.75rem 0; color: #0056b3; display: flex; align-items: center; gap: 0.5rem; font-size: 0.95rem; font-weight: 600; border-bottom: 1px solid #dee2e6; padding-bottom: 0.5rem;">
                                        <i class="fas fa-file-invoice" style="font-size: 0.875rem;"></i> Grunddaten
                                    </h4>
                                    <div style="display: grid; grid-template-columns: 1fr; gap: 0.5rem; font-size: 0.875rem;">
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Auftragsnummer:</strong> <span>${order.AuftragNr || '-'}</span></div>
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Auftragsart:</strong> <span>${order.Auftragart || '-'}</span></div>
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Datum:</strong> <span>${formatDate(order.DatumAuftrag)}</span></div>
                                    </div>
                                </div>
                                
                                <!-- Kunde -->
                                <div style="background: #fff; padding: 1rem; border-radius: 6px; border: 1px solid #dee2e6; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                                    <h4 style="margin: 0 0 0.75rem 0; color: #0056b3; display: flex; align-items: center; gap: 0.5rem; font-size: 0.95rem; font-weight: 600; border-bottom: 1px solid #dee2e6; padding-bottom: 0.5rem;">
                                        <i class="fas fa-user" style="font-size: 0.875rem;"></i> Kunde
                                    </h4>
                                    <div style="display: grid; grid-template-columns: 1fr; gap: 0.5rem; font-size: 0.875rem;">
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Name:</strong> <span>${order.KundenName1 || ''} ${order.KundenVorname || ''}</span></div>
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Kunden-ID:</strong> 
                                            ${order.KundenAdrID ? `
                                                <span onclick="openCustomerDetails(${order.KundenAdrID})" 
                                                      style="color: #0056b3; cursor: pointer; text-decoration: underline; font-weight: 600; transition: all 0.2s;"
                                                      onmouseover="this.style.color='#003d82'; this.style.textDecoration='none';"
                                                      onmouseout="this.style.color='#0056b3'; this.style.textDecoration='underline';"
                                                      title="Klicken für Kundendetails">${order.KundenAdrID}</span>
                                            ` : '<span>-</span>'}
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Fahrzeug -->
                                <div style="background: #fff; padding: 1rem; border-radius: 6px; border: 1px solid #dee2e6; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                                    <h4 style="margin: 0 0 0.75rem 0; color: #0056b3; display: flex; align-items: center; gap: 0.5rem; font-size: 0.95rem; font-weight: 600; border-bottom: 1px solid #dee2e6; padding-bottom: 0.5rem;">
                                        <i class="fas fa-car" style="font-size: 0.875rem;"></i> Fahrzeug
                                    </h4>
                                    <div style="display: grid; grid-template-columns: 1fr; gap: 0.5rem; font-size: 0.875rem;">
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Kennzeichen:</strong> 
                                            ${order.ZulassungsKz ? `
                                                <span onclick="openLicenseSearch('${order.ZulassungsKz.replace(/'/g, "\\'")}')" 
                                                      style="color: #0056b3; cursor: pointer; text-decoration: underline; font-weight: 600; transition: all 0.2s;"
                                                      onmouseover="this.style.color='#003d82'; this.style.textDecoration='none';"
                                                      onmouseout="this.style.color='#0056b3'; this.style.textDecoration='underline';"
                                                      title="Klicken für Suche nach diesem Kennzeichen">${order.ZulassungsKz}</span>
                                            ` : '<span>-</span>'}
                                        </div>
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Fahrgestellnr:</strong> <span>${order.Fahrgestellnr || '-'}</span></div>
                                    </div>
                                </div>
                                
                                <!-- Finanzen -->
                                <div style="background: #fff; padding: 1rem; border-radius: 6px; border: 1px solid #dee2e6; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                                    <h4 style="margin: 0 0 0.75rem 0; color: #0056b3; display: flex; align-items: center; gap: 0.5rem; font-size: 0.95rem; font-weight: 600; border-bottom: 1px solid #dee2e6; padding-bottom: 0.5rem;">
                                        <i class="fas fa-euro-sign" style="font-size: 0.875rem;"></i> Finanzen
                                    </h4>
                                    <div style="display: grid; grid-template-columns: 1fr; gap: 0.5rem; font-size: 0.875rem;">
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Umsatz:</strong> <span style="font-weight: 600; color: #28a745;">${formatCurrency(order.Umsatz || totalVK)}</span></div>
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Einkauf:</strong> <span>${formatCurrency(order.EK || totalEK)}</span></div>
                                        <div style="display: flex; justify-content: space-between;"><strong style="color: #6c757d;">Marge:</strong> <span style="font-weight: 600; color: ${totalMargePercent >= 20 ? '#28a745' : totalMargePercent >= 10 ? '#ffc107' : '#dc3545'};">${formatCurrency(order.Marge || totalMarge)} (${(order.MargeProzent || totalMargePercent).toFixed(1)}%)</span></div>
                                    </div>
                                </div>
                            </div>
                            
                            ${order.Auftragtext ? `
                            <div style="background: #fff; padding: 1rem; border-radius: 6px; border: 1px solid #dee2e6; box-shadow: 0 1px 3px rgba(0,0,0,0.1); margin-top: 1rem;">
                                <h4 style="margin: 0 0 0.75rem 0; color: #0056b3; display: flex; align-items: center; gap: 0.5rem; font-size: 0.95rem; font-weight: 600; border-bottom: 1px solid #dee2e6; padding-bottom: 0.5rem;">
                                    <i class="fas fa-align-left" style="font-size: 0.875rem;"></i> Auftragstext
                                </h4>
                                <div style="padding: 0.75rem; background: #f8f9fa; border-radius: 4px; line-height: 1.5; white-space: pre-wrap; border-left: 3px solid #0056b3; font-size: 0.875rem;">${order.Auftragtext}</div>
                            </div>
                            ` : ''}
                        </div>
                        
                        <!-- Positionen Tab -->
                        <div id="p2TabPositionen" class="p2-tab-content" style="display: none; flex: 1; overflow-y: auto; overflow-x: hidden; display: flex; flex-direction: column; min-height: 0;">
                        ${(() => {
                            // Positionen mit Job-Gruppierung - genau wie in appointments.php
                            if (positions && positions.length > 0) {
                                // Gruppiere Positionen nach Jobs
                                const jobs = {};
                                let currentJob = null;
                                let jobCounter = 0;
                                
                                positions.forEach((pos, index) => {
                                    // Prüfe ob es ein Job-Header ist (Art "JH" UND lange Bezeichnung)
                                    const isJobHeader = pos.Art === 'JH' && pos.Bezeichnung && pos.Bezeichnung.trim().length > 30;
                                    
                                    if (isJobHeader) {
                                        jobCounter++;
                                        const jobKey = `${jobCounter}/0`;
                                        currentJob = {
                                            jobNumber: jobKey,
                                            positions: [],
                                            isJobHeader: true,
                                            jobDescription: pos.Bezeichnung.trim()
                                        };
                                        jobs[jobKey] = currentJob;
                                        pos.formattedPosition = jobKey;
                                        currentJob.positions.push(pos);
                                    } else if (currentJob) {
                                        const posIndex = currentJob.positions.filter(p => !p.formattedPosition.endsWith('/0')).length + 1;
                                        const posNumber = `${jobCounter}/${posIndex}`;
                                        pos.formattedPosition = posNumber;
                                        currentJob.positions.push(pos);
                                    } else {
                                        jobCounter++;
                                        const jobKey = `${jobCounter}/0`;
                                        currentJob = {
                                            jobNumber: jobKey,
                                            positions: [],
                                            isJobHeader: false,
                                            jobDescription: null
                                        };
                                        jobs[jobKey] = currentJob;
                                        const posPart = pos.Position || index + 1;
                                        const posNumber = `${jobCounter}/${posPart}`;
                                        pos.formattedPosition = posNumber;
                                        currentJob.positions.push(pos);
                                    }
                                });
                                
                                const sortedJobKeys = Object.keys(jobs).sort((a, b) => {
                                    const aNum = parseInt(a.split('/')[0]) || 0;
                                    const bNum = parseInt(b.split('/')[0]) || 0;
                                    return aNum - bNum;
                                });
                                
                                const jobColors = [
                                    {primary: '#0056b3', light: '#e3f2fd'},
                                    {primary: '#28a745', light: '#e8f5e9'},
                                    {primary: '#dc3545', light: '#ffebee'},
                                    {primary: '#ffc107', light: '#fff8e1'},
                                    {primary: '#17a2b8', light: '#e0f7fa'},
                                    {primary: '#6f42c1', light: '#f3e5f5'},
                                    {primary: '#fd7e14', light: '#fff3e0'},
                                    {primary: '#20c997', light: '#e0f2f1'}
                                ];
                                
                                const getJobColor = (jobNum) => {
                                    const num = parseInt(jobNum.split('/')[0]) || 0;
                                    return jobColors[num % jobColors.length];
                                };
                                
                                const jobIcons = [
                                    'fa-folder-open', 'fa-tasks', 'fa-wrench', 'fa-cog',
                                    'fa-toolbox', 'fa-clipboard-list', 'fa-screwdriver', 'fa-hammer'
                                ];
                                
                                const getJobIcon = (jobNum) => {
                                    const num = parseInt(jobNum.split('/')[0]) || 0;
                                    return jobIcons[num % jobIcons.length];
                                };
                                
                                const getArtIcon = (art) => {
                                    const artIcons = {
                                        'OP': 'fa-cog', 'TS': 'fa-file-alt', 'JH': 'fa-folder',
                                        'CC': 'fa-check-circle', 'PA': 'fa-tools', 'AR': 'fa-wrench',
                                        'TE': 'fa-tire', 'RE': 'fa-redo', 'IN': 'fa-info-circle',
                                        'AN': 'fa-clipboard-check', 'ST': 'fa-stopwatch', 'MA': 'fa-user-cog',
                                        'FA': 'fa-car', 'KU': 'fa-user', 'LI': 'fa-list',
                                        'BE': 'fa-file-invoice', 'ZA': 'fa-calendar', 'AB': 'fa-check',
                                        'ER': 'fa-exclamation-triangle', 'AD': 'fa-clock'
                                    };
                                    return artIcons[art] || 'fa-circle';
                                };
                                
                                const getArtIconColor = (art) => {
                                    const artColors = {
                                        'OP': '#0056b3', 'TS': '#6c757d', 'JH': '#ffc107',
                                        'CC': '#28a745', 'PA': '#17a2b8', 'AR': '#dc3545',
                                        'TE': '#6f42c1', 'RE': '#fd7e14', 'IN': '#20c997',
                                        'AN': '#e83e8c', 'ST': '#6610f2', 'MA': '#0d6efd',
                                        'FA': '#198754', 'KU': '#0dcaf0', 'LI': '#6c757d',
                                        'BE': '#212529', 'ZA': '#ffc107', 'AB': '#28a745',
                                        'ER': '#ffc107', 'FA': '#198754', 'AD': '#6c757d'
                                    };
                                    return artColors[art] || '#6c757d';
                                };
                                
                                const columns = [
                                    {key: 'Position', label: 'Pos', width: '60px'},
                                    {key: 'Nummer', label: 'OP-Nr.', width: '100px'},
                                    {key: 'Bezeichnung', label: 'Bezeichnung', width: 'auto', flex: 1},
                                    {key: 'Anzahl', label: 'Anzahl', width: '80px', align: 'right'},
                                    {key: 'Zeit', label: 'Zeit', width: '80px', align: 'right'},
                                    {key: 'VKEinzel', label: 'Einzel', width: '100px', align: 'right'},
                                    {key: 'RabattProzent', label: 'Rab. %', width: '80px', align: 'right'},
                                    {key: 'VKGesamt', label: 'Extern Gesamt', width: '120px', align: 'right'},
                                    {key: 'VKGesamtInclMwSt', label: 'Extern incl. MwSt.', width: '150px', align: 'right'},
                                    {key: 'Art', label: 'Art', width: '80px'},
                                    {key: 'JobNummer', label: 'Job', width: '80px'},
                                    {key: 'OP_TXT', label: 'OP TXT', width: '200px'}
                                ];
                                
                                let positionsHtml = `
                                    <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 8px; flex: 1; overflow: hidden; display: flex; flex-direction: column;">
                                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem; flex-shrink: 0;">
                                            <h4 style="margin: 0; color: #0056b3; display: flex; align-items: center; gap: 0.5rem;">
                                                <i class="fas fa-list"></i> Positionen (${positions.length})
                                            </h4>
                                            <div style="display: flex; gap: 0.5rem;">
                                                <button class="btn btn-sm btn-secondary" onclick="exportP2OrderToCSV(${order.AuftragNr || 'null'})" title="Als CSV exportieren">
                                                    <i class="fas fa-download"></i> Export CSV
                                                </button>
                                                <button class="btn btn-sm btn-info" onclick="printP2OrderDetails()" title="Drucken">
                                                    <i class="fas fa-print"></i> Drucken
                                                </button>
                                            </div>
                                        </div>
                                        <div style="overflow-x: auto; overflow-y: auto; flex: 1; min-height: 0; border: 1px solid #dee2e6; border-radius: 4px; background: #fff;">
                                            <table style="width: 100%; border-collapse: collapse; font-size: 0.875rem; background: #fff;">
                                                <thead>
                                                    <tr style="background: #e9ecef; border-bottom: 2px solid #dee2e6; position: sticky; top: 0; z-index: 10;">
                                                        ${columns.map(col => `
                                                            <th style="padding: 0.5rem; text-align: ${col.align || 'left'}; font-weight: 600; color: #495057; white-space: nowrap; ${col.width !== 'auto' ? `width: ${col.width}; max-width: ${col.width};` : ''} ${col.flex ? 'min-width: 200px;' : ''}">
                                                                ${col.label}
                                                            </th>
                                                        `).join('')}
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    ${sortedJobKeys.map(jobKey => {
                                                        const job = jobs[jobKey];
                                                        const jobColor = getJobColor(job.jobNumber);
                                                        const jobIcon = getJobIcon(job.jobNumber);
                                                        let jobHtml = '';
                                                        
                                                        const jobRowId = `job-row-${job.jobNumber.replace(/\//g, '-')}`;
                                                        const jobContentId = `job-content-${job.jobNumber.replace(/\//g, '-')}`;
                                                        
                                                        if (job.isJobHeader && job.jobDescription) {
                                                            const jobHeaderPos = job.positions.find(p => p.formattedPosition === job.jobNumber);
                                                            const escapedJobDesc = (job.jobDescription || '').replace(/'/g, "\\'").replace(/\n/g, '\\n').replace(/\r/g, '\\r');
                                                            const jobHeaderPositionNr = jobHeaderPos ? (jobHeaderPos.Position || jobHeaderPos.PositionNr) : null;
                                                            const jobHeaderArt = jobHeaderPos ? (jobHeaderPos.Art || 'JH') : 'JH';
                                                            const canEditJobHeader = !isReadOnly && jobHeaderPositionNr;
                                                            const jobHeaderCursor = canEditJobHeader ? 'pointer' : 'default';
                                                            const jobHeaderTitle = canEditJobHeader ? 'Klicken zum Bearbeiten' : 'Auftrag ist abgeschlossen - nur lesen';
                                                            
                                                            jobHtml += `
                                                                <tr id="${jobRowId}" style="background: #e9ecef; border-bottom: 2px solid #dee2e6; font-weight: 600;" 
                                                                    onmouseover="this.style.backgroundColor='${canEditJobHeader ? '#d6d9dc' : '#e9ecef'}';" 
                                                                    onmouseout="this.style.backgroundColor='#e9ecef';">
                                                                    <td style="padding: 0.75rem; font-weight: 600; color: #495057; border-right: 1px solid #dee2e6;">
                                                                        <div style="display: flex; align-items: center; gap: 0.5rem;">
                                                                            <i class="fas fa-chevron-down job-toggle-icon" onclick="toggleJob('${jobContentId}', '${jobRowId}'); event.stopPropagation();" style="font-size: 0.75rem; color: ${jobColor.primary}; cursor: pointer; transition: transform 0.2s;" title="Job ein-/ausklappen"></i>
                                                                            <i class="fas ${jobIcon}" style="font-size: 1rem; color: ${jobColor.primary};"></i>
                                                                            <span>${job.jobNumber}</span>
                                                                            ${isReadOnly ? '<i class="fas fa-lock" style="font-size: 0.75rem; color: #6c757d; margin-left: 0.25rem;" title="Nur lesen"></i>' : ''}
                                                                        </div>
                                                                    </td>
                                                                    <td colspan="${columns.length - 1}" style="padding: 0.75rem; font-weight: 600; color: #495057; font-size: 0.9rem; line-height: 1.4; white-space: pre-wrap; cursor: ${jobHeaderCursor};" 
                                                                        ${canEditJobHeader ? `onclick="editP2Position(${order.AuftragNr || 'null'}, ${jobHeaderPositionNr}, '${escapedJobDesc}', '${jobHeaderArt}')"` : ''}
                                                                        title="${jobHeaderTitle}">
                                                                        ${job.jobDescription}
                                                                    </td>
                                                                </tr>
                                                            `;
                                                        } else {
                                                            jobHtml += `
                                                                <tr id="${jobRowId}" style="background: #e9ecef; border-bottom: 2px solid #dee2e6; font-weight: 600;">
                                                                    <td style="padding: 0.75rem; font-weight: 600; color: #495057; border-right: 1px solid #dee2e6;">
                                                                        <div style="display: flex; align-items: center; gap: 0.5rem;">
                                                                            <i class="fas fa-chevron-down job-toggle-icon" onclick="toggleJob('${jobContentId}', '${jobRowId}'); event.stopPropagation();" style="font-size: 0.75rem; color: ${jobColor.primary}; cursor: pointer; transition: transform 0.2s;" title="Job ein-/ausklappen"></i>
                                                                            <i class="fas ${jobIcon}" style="font-size: 1rem; color: ${jobColor.primary};"></i>
                                                                            <span>${job.jobNumber}</span>
                                                                        </div>
                                                                    </td>
                                                                    <td colspan="${columns.length - 1}" style="padding: 0.75rem; font-weight: 600; color: #495057;">
                                                                        Job ${job.jobNumber}
                                                                    </td>
                                                                </tr>
                                                            `;
                                                        }
                                                        
                                                        jobHtml += `<tbody id="${jobContentId}">`;
                                                        job.positions.forEach(pos => {
                                                            const posNum = pos.formattedPosition || pos.Position || '';
                                                            if (posNum.endsWith('/0') && pos.Bezeichnung && pos.Bezeichnung.length > 50) {
                                                                return;
                                                            }
                                                            
                                                            let rowBgColor;
                                                            if (pos.Art === 'OP') {
                                                                rowBgColor = '#f0f7ff';
                                                            } else if (pos.Art === 'JH') {
                                                                rowBgColor = '#faf9f7';
                                                            } else if (pos.Bezeichnung && pos.Bezeichnung.length > 100) {
                                                                rowBgColor = jobColor.light;
                                                            } else {
                                                                rowBgColor = '#ffffff';
                                                            }
                                                            const borderColor = jobColor.primary + '40';
                                                            
                                                            const canEditByArt = pos.Art === 'CC' || pos.Art === 'OP' || pos.Art === 'TS' || pos.Art === 'JH';
                                                            const isEditable = canEditByArt && !isReadOnly;
                                                            const editCursor = isEditable ? 'pointer' : 'default';
                                                            const editTitle = isReadOnly ? 'Auftrag ist abgeschlossen - nur lesen' : (isEditable ? 'Klicken zum Bearbeiten' : '');
                                                            
                                                            const escapedBezeichnung = (pos.Bezeichnung || '').replace(/'/g, "\\'").replace(/\n/g, '\\n').replace(/\r/g, '\\r');
                                                            
                                                            let hoverBgColor;
                                                            if (pos.Art === 'OP') {
                                                                hoverBgColor = '#e6f2ff';
                                                            } else if (pos.Art === 'JH') {
                                                                hoverBgColor = '#f5f3f0';
                                                            } else {
                                                                hoverBgColor = isEditable ? jobColor.light : rowBgColor;
                                                            }
                                                            
                                                            const rowStyle = `
                                                                border-bottom: 1px solid ${borderColor}; 
                                                                background: ${rowBgColor}; 
                                                                border-left: 3px solid ${jobColor.primary}; 
                                                                transition: all 0.2s ease; 
                                                                cursor: ${editCursor};
                                                                ${isEditable ? 'box-shadow: 0 1px 2px rgba(0,0,0,0.05);' : ''}
                                                            `;
                                                            
                                                            const lockIcon = isReadOnly && canEditByArt ? '<i class="fas fa-lock" style="font-size: 0.7rem; color: #6c757d; margin-left: 0.25rem; vertical-align: middle;" title="Nur lesen"></i>' : '';
                                                            
                                                            jobHtml += `
                                                                <tr style="${rowStyle}" 
                                                                    onmouseover="this.style.backgroundColor='${isEditable ? hoverBgColor : rowBgColor}'; this.style.boxShadow='${isEditable ? '0 2px 4px rgba(0,0,0,0.1)' : 'none'}';" 
                                                                    onmouseout="this.style.backgroundColor='${rowBgColor}'; this.style.boxShadow='${isEditable ? '0 1px 2px rgba(0,0,0,0.05)' : 'none'}';"
                                                                    ${isEditable ? `onclick="editP2Position(${order.AuftragNr || 'null'}, ${pos.Position || pos.PositionNr || 'null'}, '${escapedBezeichnung}', '${pos.Art || ''}')"` : ''}
                                                                    title="${editTitle}">
                                                                    ${columns.map(col => {
                                                                        let value = pos[col.key];
                                                                        let displayValue = '-';
                                                                        let cellStyle = 'padding: 0.5rem; text-align: ' + (col.align || 'left') + ';';
                                                                        
                                                                        const isTS = pos.Art === 'TS';
                                                                        const isPriceField = col.key.includes('VK') || col.key.includes('EK') || col.key.includes('Marge') || col.key.includes('Rabatt') || col.key === 'Einzel' || col.key === 'Gesamt' || col.key === 'Extern';
                                                                        if (isTS && isPriceField) {
                                                                            displayValue = '-';
                                                                        } else {
                                                                            if (col.key === 'Position') {
                                                                                value = pos.formattedPosition || pos.Position || '-';
                                                                                displayValue = value;
                                                                            } else if (col.key === 'Nummer') {
                                                                                value = pos.Nummer || pos.ServiceID || pos.RemedyCode || '-';
                                                                                if (!value || (typeof value === 'string' && value.trim() === '') || value === '-') {
                                                                                    displayValue = '-';
                                                                                } else {
                                                                                    const opNr = String(value).trim();
                                                                                    displayValue = `
                                                                                        <span style="white-space: nowrap;">${opNr}</span>
                                                                                        <button onclick="copyToClipboard('${opNr.replace(/'/g, "\\'")}', 'OP-Nr. kopiert!'); event.stopPropagation();" 
                                                                                            style="background: none; border: none; padding: 0.125rem 0.25rem; cursor: pointer; color: #6c757d; font-size: 0.7rem; opacity: 0.5; transition: opacity 0.2s; margin-left: 0.25rem; vertical-align: middle;" 
                                                                                            onmouseover="this.style.opacity='1';" 
                                                                                            onmouseout="this.style.opacity='0.5';"
                                                                                            title="OP-Nr. kopieren">
                                                                                            <i class="fas fa-copy"></i>
                                                                                        </button>
                                                                                    `;
                                                                                }
                                                                            } else if (col.key === 'Bezeichnung') {
                                                                                value = pos.Bezeichnung || '-';
                                                                                if (value && value !== '-') {
                                                                                    const escapedBezeichnung = String(value).replace(/'/g, "\\'").replace(/"/g, '&quot;').replace(/\n/g, '\\n').replace(/\r/g, '\\r');
                                                                                    displayValue = `
                                                                                        <div style="position: relative; width: 100%;">
                                                                                            <span style="display: inline-block; max-width: calc(100% - 30px);">${value}</span>
                                                                                            <button onclick="copyToClipboard('${escapedBezeichnung}', 'Bezeichnung kopiert!'); event.stopPropagation();" 
                                                                                                style="background: none; border: none; padding: 0.125rem 0.25rem; cursor: pointer; color: #6c757d; font-size: 0.7rem; opacity: 0.5; transition: opacity 0.2s; position: absolute; right: 0; top: 0;" 
                                                                                                onmouseover="this.style.opacity='1';" 
                                                                                                onmouseout="this.style.opacity='0.5';"
                                                                                                title="Bezeichnung kopieren">
                                                                                                <i class="fas fa-copy"></i>
                                                                                            </button>
                                                                                        </div>
                                                                                    `;
                                                                                } else {
                                                                                    displayValue = value;
                                                                                }
                                                                            } else if (col.key === 'Art') {
                                                                                const art = pos.Art || '';
                                                                                const artIcon = getArtIcon(art);
                                                                                const artColor = getArtIconColor(art);
                                                                                displayValue = art ? `<i class="fas ${artIcon}" style="color: ${artColor}; margin-right: 0.25rem; font-size: 0.875rem; vertical-align: middle;"></i><span style="vertical-align: middle;">${art}</span>${lockIcon}` : '-';
                                                                            } else if (value !== null && value !== undefined && value !== '') {
                                                                                if (typeof value === 'number') {
                                                                                    if (col.key.includes('Prozent') || col.key === 'MargeProzent') {
                                                                                        displayValue = parseFloat(value).toFixed(2).replace('.', ',') + ' %';
                                                                                    } else if (col.key.includes('VK') || col.key.includes('EK') || col.key.includes('Marge') || col.key.includes('Rabatt') || col.key.includes('Gesamt') || col.key.includes('Extern')) {
                                                                                        displayValue = parseFloat(value).toFixed(2).replace('.', ',') + ' €';
                                                                                    } else if (col.key === 'Anzahl' || col.key === 'Zeit') {
                                                                                        displayValue = parseFloat(value).toFixed(2).replace('.', ',');
                                                                                    } else {
                                                                                        displayValue = value.toString();
                                                                                    }
                                                                                } else {
                                                                                    displayValue = String(value);
                                                                                }
                                                                            }
                                                                        }
                                                                        
                                                                        if (col.key === 'Bezeichnung' && value && value.length > 100) {
                                                                            cellStyle += ' white-space: pre-wrap; line-height: 1.4; word-wrap: break-word;';
                                                                        } else if (col.key === 'Bezeichnung') {
                                                                            cellStyle += ' word-wrap: break-word; overflow-wrap: break-word;';
                                                                        }
                                                                        
                                                                        if (col.width && col.width !== 'auto' && col.key !== 'Bezeichnung') {
                                                                            cellStyle += ` max-width: ${col.width}; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;`;
                                                                        }
                                                                        
                                                                        return `<td style="${cellStyle}">${displayValue}</td>`;
                                                                    }).join('')}
                                                                </tr>
                                                            `;
                                                        });
                                                        
                                                        const jobSum = job.positions.reduce((sum, pos) => {
                                                            const posNum = pos.formattedPosition || pos.Position || '';
                                                            if (posNum.endsWith('/0') && pos.Bezeichnung && pos.Bezeichnung.length > 50) {
                                                                return sum;
                                                            }
                                                            if (pos.Art === 'TS') {
                                                                return sum;
                                                            }
                                                            return {
                                                                extern: (sum.extern || 0) + (parseFloat(pos.VKGesamt) || 0),
                                                                externMwSt: (sum.externMwSt || 0) + (parseFloat(pos.VKGesamtInclMwSt) || 0)
                                                            };
                                                        }, {extern: 0, externMwSt: 0});
                                                        
                                                        const sumColspan = columns.findIndex(col => col.key === 'VKGesamt') - 1;
                                                        jobHtml += `
                                                            <tr style="background: ${jobColor.light}; border-bottom: 2px solid ${jobColor.primary}; border-left: 3px solid ${jobColor.primary}; font-weight: 600;">
                                                                <td style="padding: 0.5rem; font-weight: 600; color: ${jobColor.primary};">
                                                                    <i class="fas fa-calculator" style="margin-right: 0.5rem;"></i>Jobsumme:
                                                                </td>
                                                                <td colspan="${sumColspan}" style="padding: 0.5rem;"></td>
                                                                <td style="padding: 0.5rem; text-align: right; font-weight: 600; color: ${jobColor.primary};">${jobSum.extern.toFixed(2).replace('.', ',')} €</td>
                                                                <td style="padding: 0.5rem; text-align: right; font-weight: 600; color: ${jobColor.primary};">${jobSum.externMwSt.toFixed(2).replace('.', ',')} €</td>
                                                                <td colspan="${columns.length - sumColspan - 2}" style="padding: 0.5rem;"></td>
                                                            </tr>
                                                        `;
                                                        
                                                        jobHtml += `</tbody>`;
                                                        return jobHtml;
                                                    }).join('')}
                                                    
                                                    ${(() => {
                                                        const totalSum = positions.reduce((sum, pos) => {
                                                            const posNum = pos.formattedPosition || pos.Position || '';
                                                            if (posNum.endsWith('/0') && pos.Bezeichnung && pos.Bezeichnung.length > 50) {
                                                                return sum;
                                                            }
                                                            if (pos.Art === 'TS') {
                                                                return sum;
                                                            }
                                                            return {
                                                                extern: (sum.extern || 0) + (parseFloat(pos.VKGesamt) || 0),
                                                                externMwSt: (sum.externMwSt || 0) + (parseFloat(pos.VKGesamtInclMwSt) || 0)
                                                            };
                                                        }, {extern: 0, externMwSt: 0});
                                                        
                                                        const sumColspanTotal = columns.findIndex(col => col.key === 'VKGesamt') - 1;
                                                        return `
                                                            <tr style="background: #0056b3; color: #fff; font-weight: 700; border-top: 3px solid #003d82;">
                                                                <td style="padding: 0.75rem; font-weight: 700;">Summe:</td>
                                                                <td colspan="${sumColspanTotal}" style="padding: 0.75rem;"></td>
                                                                <td style="padding: 0.75rem; text-align: right; font-weight: 700;">${totalSum.extern.toFixed(2).replace('.', ',')} €</td>
                                                                <td style="padding: 0.75rem; text-align: right; font-weight: 700;">${totalSum.externMwSt.toFixed(2).replace('.', ',')} €</td>
                                                                <td colspan="${columns.length - sumColspanTotal - 2}" style="padding: 0.75rem;"></td>
                                                            </tr>
                                                        `;
                                                    })()}
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                `;
                                return positionsHtml;
                            } else {
                                return `
                                    <div style="text-align: center; padding: 3rem; color: #6c757d;">
                                        <i class="fas fa-inbox" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.5;"></i>
                                        <p>Keine Positionen gefunden</p>
                                    </div>
                                `;
                            }
                        })()}
                        </div>
                    </div>
                `;

                body.innerHTML = html;
            })
            .catch(error => {
                const body = document.getElementById('p2OrderDetailsBody');
                if (body) {
                    body.innerHTML = `<div class="empty-state"><p>Fehler beim Laden: ${error.message}</p></div>`;
                }
            });
        }

        function closeModal(modalId) {
            $(`#${modalId}`).remove();
            // Wenn kein Modal mehr im Container ist, entferne den Container auch
            if ($('#modalContainer').length && $('#modalContainer').children().length === 0) {
                $('#modalContainer').remove();
            }
        }

        function createModal(id, title, content, hasSubmitButton = false, submitFunction = null, submitLabel = 'Speichern', size = 'normal') {
            let footerButtons = `<button class="btn btn-secondary" onclick="closeModal('${id}')">Schließen</button>`;
            
            if (hasSubmitButton && submitFunction) {
                footerButtons = `<button class="btn btn-secondary" onclick="closeModal('${id}')">Abbrechen</button>
                                 <button class="btn btn-primary" onclick="${submitFunction}()">${submitLabel}</button>`;
            }
            
            // Bestimme Modal-Größe (wie in storage.php)
            let modalWidth = 'max-width: 1700px; width: 90vw;';
            if (size === 'large') {
                modalWidth = 'max-width: 95vw; width: 95vw; height: 90vh;';
            } else if (size === 'medium') {
                modalWidth = 'max-width: 800px;';
            } else if (size === 'extra-large') {
                modalWidth = 'max-width: 1700px; width: 90vw;';
            } else if (size === 'normal') {
                modalWidth = 'max-width: 800px;';
            }
            
            const modal = $(`
                <div class="modal-overlay" id="${id}" onclick="if(event.target.id === '${id}') closeModal('${id}')">
                    <div class="modal" style="${modalWidth}">
                        <div class="modal-header">
                            <div class="modal-title">${title}</div>
                            <button class="modal-close" onclick="closeModal('${id}')">&times;</button>
                        </div>
                        <div class="modal-body" style="${size === 'large' ? 'height: calc(90vh - 120px); overflow-y: auto;' : ''}">${content}</div>
                        ${footerButtons ? `<div class="modal-footer">
                            ${footerButtons}
                        </div>` : ''}
                    </div>
                </div>
            `);
            
            if (!$('#modalContainer').length) {
                $('body').append('<div id="modalContainer"></div>');
            }
            
            // Entferne nur das spezifische Modal, nicht alle Modals
            $(`#${id}`).remove();
            
            // Füge das neue Modal hinzu
            $('#modalContainer').append(modal);
        }
        
        // P2 Order Modal Hilfsfunktionen (wie in appointments.php)
        function closeP2OrderModal() {
            const modal = document.getElementById('p2OrderDetailsModal');
            if (modal) {
                modal.style.display = 'none';
            }
        }
        
        // Öffne Kundendetails-Modal
        function openCustomerDetails(kundenAdrId) {
            // Schließe Auftrags-Modal
            closeP2OrderModal();
            // Öffne Kunden-Detail-Panel
            selectCustomer(kundenAdrId);
            // Scroll zum Kunden-Detail-Panel
            setTimeout(() => {
                const panel = document.getElementById('detailPanel');
                if (panel) {
                    panel.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    // Hervorhebung
                    panel.style.transition = 'box-shadow 0.3s ease';
                    panel.style.boxShadow = '0 0 20px rgba(0, 86, 179, 0.5)';
                    setTimeout(() => {
                        panel.style.boxShadow = '';
                    }, 2000);
                }
            }, 100);
        }
        
        // Öffne Kennzeichen-Suche
        function openLicenseSearch(licensePlate) {
            // Schließe Auftrags-Modal
            closeP2OrderModal();
            // Setze Suchfeld und führe Suche aus
            $('#searchInput').val(licensePlate);
            performUniversalSearch(licensePlate);
            // Scroll zum Suchfeld
            setTimeout(() => {
                const searchInput = document.getElementById('searchInput');
                if (searchInput) {
                    searchInput.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    // Hervorhebung
                    searchInput.style.transition = 'box-shadow 0.3s ease, border-color 0.3s ease';
                    searchInput.style.boxShadow = '0 0 15px rgba(0, 86, 179, 0.4)';
                    searchInput.style.borderColor = '#0056b3';
                    setTimeout(() => {
                        searchInput.style.boxShadow = '';
                        searchInput.style.borderColor = '';
                    }, 2000);
                }
            }, 100);
        }
        
        // Globale viewOrder Funktion für onclick-Handler
        // Die Funktion wird direkt als window.viewOrder definiert, um Endlosschleife zu vermeiden
        
        function switchP2Tab(tabName) {
            // Verstecke alle Tab-Contents
            document.querySelectorAll('.p2-tab-content').forEach(content => {
                content.style.display = 'none';
            });
            
            // Entferne active von allen Tab-Buttons
            document.querySelectorAll('.p2-tab-btn').forEach(btn => {
                btn.classList.remove('active');
                btn.style.borderBottom = '3px solid transparent';
                btn.style.color = '#6c757d';
                btn.style.fontWeight = 'normal';
            });
            
            // Zeige gewählten Tab
            const tabContent = document.getElementById('p2Tab' + tabName.charAt(0).toUpperCase() + tabName.slice(1));
            const tabBtn = document.querySelector(`.p2-tab-btn[data-tab="${tabName}"]`);
            
            if (tabContent) {
                if (tabName === 'positionen') {
                    tabContent.style.display = 'flex';
                    tabContent.style.flexDirection = 'column';
                    tabContent.style.overflow = 'hidden';
                } else {
                    tabContent.style.display = 'block';
                }
            }
            
            if (tabBtn) {
                tabBtn.classList.add('active');
                tabBtn.style.borderBottom = '3px solid #0056b3';
                tabBtn.style.color = '#0056b3';
                tabBtn.style.fontWeight = '600';
            }
        }
        
        // Status-Bedeutungen (wie in appointments.php)
        const statusMeanings = {
            'AB': {name: 'Abgeschlossen', description: 'Auftrag ist vollständig abgeschlossen', color: '#28a745', badge: 'success', readonly: true},
            'ER': {name: 'Erstellt', description: 'Auftrag wurde erstellt und wartet auf Bearbeitung', color: '#ffc107', badge: 'warning', readonly: false},
            'FA': {name: 'Fertig', description: 'Auftrag ist fertiggestellt, aber noch nicht abgeschlossen', color: '#17a2b8', badge: 'info', readonly: false},
            'AD': {name: 'Ausstehend', description: 'Auftrag wartet auf Bearbeitung', color: '#6c757d', badge: 'secondary', readonly: false},
            'IN': {name: 'In Bearbeitung', description: 'Auftrag wird aktuell bearbeitet', color: '#0056b3', badge: 'primary', readonly: false},
            'OF': {name: 'Offen', description: 'Auftrag ist offen und wartet', color: '#fd7e14', badge: 'warning', readonly: false},
            'ST': {name: 'Storniert', description: 'Auftrag wurde storniert', color: '#dc3545', badge: 'danger', readonly: true},
            'WA': {name: 'Wartend', description: 'Auftrag wartet auf Freigabe oder Material', color: '#6f42c1', badge: 'secondary', readonly: false},
            'RE': {name: 'Rechnung', description: 'Rechnung wurde erstellt', color: '#20c997', badge: 'success', readonly: false},
            'ZA': {name: 'Zahlung', description: 'Zahlung wurde erhalten', color: '#198754', badge: 'success', readonly: false},
            'QU': {name: 'Quittiert', description: 'Auftrag wurde quittiert', color: '#0dcaf0', badge: 'info', readonly: true}
        };
        
        function getStatusInfo(status) {
            return statusMeanings[status] || {
                name: status || 'Unbekannt',
                description: `Status: ${status || 'Unbekannt'}`,
                color: '#6c757d',
                badge: 'secondary',
                readonly: false
            };
        }
        
        function toggleJob(jobContentId, jobRowId) {
            const content = document.getElementById(jobContentId);
            const row = document.getElementById(jobRowId);
            if (!content || !row) return;
            
            const isVisible = content.style.display !== 'none';
            const icon = row.querySelector('.job-toggle-icon');
            
            if (isVisible) {
                content.style.display = 'none';
                if (icon) {
                    icon.classList.remove('fa-chevron-down');
                    icon.classList.add('fa-chevron-right');
                    icon.style.transform = 'rotate(0deg)';
                }
            } else {
                content.style.display = 'table-row-group';
                if (icon) {
                    icon.classList.remove('fa-chevron-right');
                    icon.classList.add('fa-chevron-down');
                    icon.style.transform = 'rotate(0deg)';
                }
            }
        }
        
        function copyToClipboard(text, successMessage) {
            const cleanText = text.replace(/\\n/g, '\n').replace(/\\r/g, '\r').replace(/\\'/g, "'").replace(/&quot;/g, '"');
            
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(cleanText).then(() => {
                    showCopyNotification(successMessage || 'In Zwischenablage kopiert!');
                }).catch(err => {
                    fallbackCopyToClipboard(cleanText, successMessage);
                });
            } else {
                fallbackCopyToClipboard(cleanText, successMessage);
            }
        }
        
        function fallbackCopyToClipboard(text, successMessage) {
            const textArea = document.createElement('textarea');
            textArea.value = text;
            textArea.style.position = 'fixed';
            textArea.style.left = '-999999px';
            textArea.style.top = '-999999px';
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            
            try {
                const successful = document.execCommand('copy');
                if (successful) {
                    showCopyNotification(successMessage || 'In Zwischenablage kopiert!');
                } else {
                    showCopyNotification('Fehler beim Kopieren', true);
                }
            } catch (err) {
                showCopyNotification('Fehler beim Kopieren', true);
            }
            
            document.body.removeChild(textArea);
        }
        
        function showCopyNotification(message, isError = false) {
            const existing = document.getElementById('copyNotification');
            if (existing) {
                existing.remove();
            }
            
            const notification = document.createElement('div');
            notification.id = 'copyNotification';
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${isError ? '#dc3545' : '#28a745'};
                color: white;
                padding: 0.75rem 1.5rem;
                border-radius: 4px;
                box-shadow: 0 4px 6px rgba(0,0,0,0.1);
                z-index: 10001;
                display: flex;
                align-items: center;
                gap: 0.5rem;
                font-size: 0.875rem;
                animation: slideIn 0.3s ease;
            `;
            notification.innerHTML = `
                <i class="fas ${isError ? 'fa-exclamation-circle' : 'fa-check-circle'}"></i>
                <span>${message}</span>
            `;
            
            const style = document.createElement('style');
            style.textContent = `
                @keyframes slideIn {
                    from {
                        transform: translateX(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }
            `;
            if (!document.head.querySelector('#copyNotificationStyle')) {
                style.id = 'copyNotificationStyle';
                document.head.appendChild(style);
            }
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.animation = 'slideIn 0.3s ease reverse';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.remove();
                    }
                }, 300);
            }, 3000);
        }
        
        function exportP2OrderToCSV(orderNumber) {
            const table = document.querySelector('#p2TabPositionen table');
            if (!table) {
                alert('Keine Tabelle zum Exportieren gefunden');
                return;
            }
            
            let csv = [];
            const rows = table.querySelectorAll('tr');
            
            rows.forEach(row => {
                const cols = row.querySelectorAll('td, th');
                const rowData = [];
                cols.forEach(col => {
                    let text = col.innerText || col.textContent || '';
                    text = text.replace(/"/g, '""');
                    rowData.push('"' + text + '"');
                });
                csv.push(rowData.join(','));
            });
            
            const csvContent = csv.join('\\n');
            const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
            const link = document.createElement('a');
            const url = URL.createObjectURL(blob);
            link.setAttribute('href', url);
            link.setAttribute('download', `Auftrag_${orderNumber}_Positionen.csv`);
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
        
        function printP2OrderDetails() {
            const modal = document.getElementById('p2OrderDetailsModal');
            if (!modal) return;
            
            const printWindow = window.open('', '_blank');
            const modalContent = modal.querySelector('.modal-content').cloneNode(true);
            
            modalContent.querySelectorAll('button, .btn').forEach(btn => btn.remove());
            
            printWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Auftragsdetails - Druckansicht</title>
                    <style>
                        body { font-family: Arial, sans-serif; margin: 20px; }
                        table { width: 100%; border-collapse: collapse; font-size: 0.75rem; }
                        th, td { border: 1px solid #ddd; padding: 0.5rem; text-align: left; }
                        th { background: #f8f9fa; font-weight: 600; }
                        .job-header { background: #e9ecef !important; font-weight: 600; }
                        .job-sum { background: #cfe2ff !important; font-weight: 600; }
                        .total-sum { background: #0056b3 !important; color: #fff !important; font-weight: 700; }
                        @media print {
                            @page { size: landscape; margin: 1cm; }
                        }
                    </style>
                </head>
                <body>
                    ${modalContent.innerHTML}
                </body>
                </html>
            `);
            printWindow.document.close();
            printWindow.focus();
            setTimeout(() => {
                printWindow.print();
                printWindow.close();
            }, 250);
        }
        
        function editP2Position(auftragNr, positionNr, currentBezeichnung, art) {
            if (!auftragNr || !positionNr) {
                alert('Fehler: Auftrags- oder Positionsnummer fehlt');
                return;
            }
            
            // WICHTIG: Prüfe ob Auftrag Status "AB" oder anderen readonly-Status hat - dann ist Bearbeitung NICHT erlaubt
            const auftragStatus = window.currentP2OrderStatus || '';
            const statusInfo = getStatusInfo(auftragStatus);
            if (statusInfo.readonly || auftragStatus === 'AB') {
                alert(`Auftrag mit Status "${auftragStatus}" (${statusInfo.name}) darf nicht bearbeitet werden. Nur lesen erlaubt.\n\n${statusInfo.description}`);
                return;
            }
            
            // Prüfe ob Art erlaubt ist
            if (art !== 'CC' && art !== 'OP' && art !== 'TS' && art !== 'JH') {
                alert(`Position mit Art "${art}" darf nicht bearbeitet werden. Nur CC, OP, TS und JH sind erlaubt.`);
                return;
            }
            
            // Erstelle Modal für Bearbeitung
            const modalId = 'p2EditPositionModal';
            let modal = document.getElementById(modalId);
            
            if (!modal) {
                modal = document.createElement('div');
                modal.id = modalId;
                modal.className = 'modal-overlay';
                modal.style.cssText = 'display: none; position: fixed; z-index: 10001; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.5);';
                modal.onclick = function(e) {
                    if (e.target === modal) {
                        closeP2EditPositionModal();
                    }
                };
                modal.innerHTML = `
                    <div class="modal-content" style="background-color: #fefefe; margin: 5% auto; padding: 2rem; border: 1px solid #888; width: 80%; max-width: 600px; border-radius: 8px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem;">
                            <h3 style="margin: 0; color: #0056b3;">
                                <i class="fas fa-edit"></i> Position bearbeiten
                            </h3>
                            <span class="close" onclick="closeP2EditPositionModal()" style="color: #aaa; font-size: 28px; font-weight: bold; cursor: pointer;">&times;</span>
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #495057;">
                                Auftragsnummer:
                            </label>
                            <input type="text" id="editPositionAuftragNr" readonly style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; background: #f8f9fa;">
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #495057;">
                                Positionsnummer:
                            </label>
                            <input type="text" id="editPositionPositionNr" readonly style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; background: #f8f9fa;">
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #495057;">
                                Art:
                            </label>
                            <input type="text" id="editPositionArt" readonly style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; background: #f8f9fa;">
                        </div>
                        <div style="margin-bottom: 1.5rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #495057;">
                                Bezeichnung (Text) <span style="color: #dc3545;">*</span>:
                            </label>
                            <textarea id="editPositionBezeichnung" rows="6" style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; font-family: Arial, sans-serif; resize: vertical;" required></textarea>
                        </div>
                        <div style="display: flex; gap: 1rem; justify-content: flex-end;">
                            <button onclick="closeP2EditPositionModal()" class="btn btn-secondary" style="padding: 0.5rem 1.5rem; border: none; border-radius: 4px; cursor: pointer; background: #6c757d; color: white;">
                                Abbrechen
                            </button>
                            <button onclick="saveP2Position()" class="btn btn-primary" style="padding: 0.5rem 1.5rem; border: none; border-radius: 4px; cursor: pointer; background: #0056b3; color: white;">
                                <i class="fas fa-save"></i> Speichern
                            </button>
                        </div>
                        <div id="editPositionError" style="margin-top: 1rem; padding: 0.75rem; border-radius: 4px; display: none;"></div>
                    </div>
                `;
                document.body.appendChild(modal);
            }
            
            // Fülle Modal mit Daten
            document.getElementById('editPositionAuftragNr').value = auftragNr;
            document.getElementById('editPositionPositionNr').value = positionNr;
            document.getElementById('editPositionArt').value = art;
            document.getElementById('editPositionBezeichnung').value = currentBezeichnung || '';
            document.getElementById('editPositionError').style.display = 'none';
            
            modal.style.display = 'block';
        }
        
        function closeP2EditPositionModal() {
            const modal = document.getElementById('p2EditPositionModal');
            if (modal) {
                modal.style.display = 'none';
            }
        }
        
        function saveP2Position() {
            const auftragNr = document.getElementById('editPositionAuftragNr').value;
            const positionNr = document.getElementById('editPositionPositionNr').value;
            const art = document.getElementById('editPositionArt').value;
            const bezeichnung = document.getElementById('editPositionBezeichnung').value.trim();
            const errorDiv = document.getElementById('editPositionError');
            
            // Validierung
            if (!bezeichnung) {
                errorDiv.style.display = 'block';
                errorDiv.style.background = '#f8d7da';
                errorDiv.style.color = '#721c24';
                errorDiv.style.border = '1px solid #f5c6cb';
                errorDiv.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Bezeichnung darf nicht leer sein!';
                return;
            }
            
            // Deaktiviere Button während des Speicherns
            const saveBtn = event.target;
            const originalText = saveBtn.innerHTML;
            saveBtn.disabled = true;
            saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Speichere...';
            
            // API-Aufruf über PHP-Proxy
            const autohaus = $('#autohausSelect').val();
            fetch(`../api/tire_offers_api.php?action=update_position&autohaus_id=${autohaus}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    auftrag_nr: auftragNr,
                    PositionNr: parseInt(positionNr),
                    Bezeichnung: bezeichnung
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Erfolg - schließe Modal und aktualisiere Anzeige
                    errorDiv.style.display = 'block';
                    errorDiv.style.background = '#d4edda';
                    errorDiv.style.color = '#155724';
                    errorDiv.style.border = '1px solid #c3e6cb';
                    errorDiv.innerHTML = '<i class="fas fa-check-circle"></i> Position erfolgreich aktualisiert!';
                    
                    // Nach 1 Sekunde Modal schließen und Seite neu laden
                    setTimeout(() => {
                        closeP2EditPositionModal();
                        // Aktualisiere die Positionen-Anzeige
                        const orderNumber = auftragNr;
                        if (window.currentP2OrderNumber === orderNumber) {
                            // Lade Auftragsdetails neu
                            window.viewOrder(orderNumber);
                        }
                    }, 1000);
                } else {
                    // Fehler
                    errorDiv.style.display = 'block';
                    errorDiv.style.background = '#f8d7da';
                    errorDiv.style.color = '#721c24';
                    errorDiv.style.border = '1px solid #f5c6cb';
                    errorDiv.innerHTML = `<i class="fas fa-exclamation-triangle"></i> Fehler: ${data.error || 'Unbekannter Fehler'}`;
                    saveBtn.disabled = false;
                    saveBtn.innerHTML = originalText;
                }
            })
            .catch(error => {
                errorDiv.style.display = 'block';
                errorDiv.style.background = '#f8d7da';
                errorDiv.style.color = '#721c24';
                errorDiv.style.border = '1px solid #f5c6cb';
                errorDiv.innerHTML = `<i class="fas fa-exclamation-triangle"></i> Netzwerkfehler: ${error.message}`;
                saveBtn.disabled = false;
                saveBtn.innerHTML = originalText;
            });
        }
        
        // Schließe Modal bei Klick außerhalb
        window.addEventListener('click', function(event) {
            const modal = document.getElementById('p2OrderDetailsModal');
            if (event.target === modal) {
                closeP2OrderModal();
            }
            const editModal = document.getElementById('p2EditPositionModal');
            if (event.target === editModal) {
                closeP2EditPositionModal();
            }
        });

        // ============ REIFENANGEBOTE-SYSTEM ============
        function createTireOffer(kundenAdrId) {
            const autohaus = $('#autohausSelect').val();
            if (!autohaus || !selectedCustomer) {
                alert('Bitte wählen Sie ein Autohaus und einen Kunden aus');
                return;
            }

            // Lade Kunden-Daten und Aufträge für Kennzeichen
            fetch(`../api/p2_api_proxy.php?action=get_customer&autohaus_id=${autohaus}&kunden_adr_id=${kundenAdrId}`)
                .then(r => r.json())
                .then(customerData => {
                    const customer = customerData.success ? customerData.data : selectedCustomer;
                    
                    // Lade neuesten Auftrag für Kennzeichen
                    return fetch(`../api/p2_api_proxy.php?action=get_orders_by_customer&autohaus_id=${autohaus}&kunden_adr_id=${kundenAdrId}&limit=1`)
                        .then(r => r.json())
                        .then(ordersData => {
                            const orders = ordersData.success ? ordersData.data : [];
                            const latestOrder = orders[0] || {};
                            
                            openTireOfferModal(customer, latestOrder, autohaus);
                        });
                })
                .catch(error => {
                    alert('Fehler beim Laden der Kundendaten: ' + error.message);
                });
        }

        function openTireOfferModal(customer, order, autohausId) {
            const modalId = 'tireOfferModal';
            const licensePlate = order.ZulassungsKz || '';
            const customerName = `${customer.KundenName1 || ''} ${customer.KundenVorname || ''}`.trim();
            
            // Standard-Brieftext
            const defaultLetterContent = `Sehr geehrte Damen und Herren,

anbei erhalten Sie Ihr Reifenangebot für den kommenden Boxenstop.

Kunde: ${customerName}
Kennzeichen: ${licensePlate}

Mit freundlichen Grüßen
B&E Autocentrum GmbH`;

            let modal = $(`#${modalId}`);
            if (!modal.length) {
                modal = $(`
                    <div class="modal-overlay" id="${modalId}" onclick="if(event.target.id === '${modalId}') closeModal('${modalId}')">
                        <div class="modal" style="max-width: 95vw; width: 95vw; max-height: 90vh;">
                            <div class="modal-header">
                                <div class="modal-title">Reifenangebot erstellen</div>
                                <button class="modal-close" onclick="closeModal('${modalId}')">&times;</button>
                            </div>
                            <div class="modal-body" id="${modalId}Body" style="max-height: calc(90vh - 120px); overflow-y: auto;">
                            </div>
                            <div class="modal-footer">
                                <button class="btn btn-secondary" onclick="closeModal('${modalId}')">Abbrechen</button>
                                <button class="btn btn-primary" onclick="saveTireOffer()">
                                    <i class="fas fa-save"></i> Angebot speichern
                                </button>
                            </div>
                        </div>
                    </div>
                `);
                $('body').append(modal);
            }

            // Modal-Inhalt
            let html = `
                <form id="tireOfferForm" style="display: grid; gap: 1.5rem;">
                    <!-- Kundendaten -->
                    <div class="detail-section" style="background: var(--bg-panel); padding: 1.5rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                        <div class="section-title" style="margin-bottom: 1rem; padding-bottom: 0.75rem; border-bottom: 2px solid var(--border); display: flex; align-items: center; gap: 0.5rem;">
                            <i class="fas fa-user" style="color: var(--primary);"></i>
                            <span>Kundendaten</span>
                        </div>
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1rem;">
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.875rem; font-weight: 600; color: var(--text-secondary);">Kunde</label>
                                <input type="text" class="form-control" value="${customerName}" readonly style="background: var(--bg-hover); font-size: 0.875rem;">
                            </div>
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.875rem; font-weight: 600; color: var(--text-secondary);">Kunden-ID</label>
                                <input type="text" class="form-control" value="${customer.KundenAdrID || '-'}" readonly style="background: var(--bg-hover); font-size: 0.875rem;">
                            </div>
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.875rem; font-weight: 600; color: var(--text-secondary);">Kennzeichen</label>
                                <input type="text" class="form-control" value="${licensePlate}" readonly style="background: var(--bg-hover); font-size: 0.875rem;">
                            </div>
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.875rem; font-weight: 600; color: var(--text-secondary);">
                                    E-Mail <span style="color: var(--danger);">*</span>
                                </label>
                                <input type="email" id="offerCustomerEmail" class="form-control" value="${customer.Email || ''}" placeholder="kunde@beispiel.de" required style="font-size: 0.875rem;">
                                ${!customer.Email ? '<small style="color: var(--text-secondary); font-size: 0.75rem; margin-top: 0.25rem; display: block;">Bitte E-Mail-Adresse eingeben</small>' : ''}
                            </div>
                            ${customer.KundenStrasse ? `
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.875rem; font-weight: 600; color: var(--text-secondary);">Adresse</label>
                                <input type="text" class="form-control" value="${customer.KundenStrasse}" readonly style="background: var(--bg-hover); font-size: 0.875rem;">
                            </div>
                            ` : ''}
                            ${customer.KundenPlz || customer.KundenOrt ? `
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.875rem; font-weight: 600; color: var(--text-secondary);">PLZ / Ort</label>
                                <input type="text" class="form-control" value="${(customer.KundenPlz || '') + ' ' + (customer.KundenOrt || '')}" readonly style="background: var(--bg-hover); font-size: 0.875rem;">
                            </div>
                            ` : ''}
                            ${customer.Telefon ? `
                            <div>
                                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.875rem; font-weight: 600; color: var(--text-secondary);">Telefon</label>
                                <input type="text" class="form-control" value="${customer.Telefon}" readonly style="background: var(--bg-hover); font-size: 0.875rem;">
                            </div>
                            ` : ''}
                        </div>
                    </div>

                    <!-- Neue Reifen -->
                    <div class="detail-section" style="background: var(--bg-panel); padding: 1.5rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                        <div class="section-title" style="margin-bottom: 1rem; padding-bottom: 0.75rem; border-bottom: 2px solid var(--border); display: flex; align-items: center; justify-content: space-between;">
                            <div style="display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-tire" style="color: var(--primary);"></i>
                                <span>Neue Reifen</span>
                            </div>
                            <div style="display: flex; gap: 0.5rem;">
                                <button type="button" class="btn btn-primary" onclick="openTireSelectionModal()" style="font-size: 0.875rem; padding: 0.5rem 1rem;">
                                    <i class="fas fa-search"></i> Reifen manuell auswählen
                                </button>
                                <button type="button" class="btn btn-secondary" onclick="addTireInput()" style="font-size: 0.875rem; padding: 0.5rem 1rem;">
                                    <i class="fas fa-plus"></i> Reifen manuell hinzufügen
                                </button>
                            </div>
                        </div>
                        <div id="tireInputsContainer" style="display: grid; gap: 1rem;">
                            <!-- Reifen werden hier dynamisch hinzugefügt -->
                        </div>
                    </div>

                    <!-- Zusatzleistungen -->
                    <div class="detail-section" style="background: var(--bg-panel); padding: 1.5rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                        <div class="section-title" style="margin-bottom: 1rem; padding-bottom: 0.75rem; border-bottom: 2px solid var(--border); display: flex; align-items: center; gap: 0.5rem;">
                            <i class="fas fa-tools" style="color: var(--primary);"></i>
                            <span>Zusatzleistungen</span>
                        </div>
                        <div id="servicesContainer" style="display: grid; gap: 0.75rem;">
                            <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem; background: var(--bg-hover); border-radius: var(--radius); cursor: pointer; transition: all 0.2s; border: 2px solid transparent;" 
                                   onmouseover="this.style.borderColor='var(--primary)'; this.style.background='var(--bg-panel)';" 
                                   onmouseout="this.style.borderColor='transparent'; this.style.background='var(--bg-hover)';">
                                <input type="checkbox" value="mounting" onchange="updateTireOfferTotal()" style="width: 1.25rem; height: 1.25rem; cursor: pointer;">
                                <div style="flex: 1;">
                                    <div style="font-weight: 600; font-size: 0.9375rem;">Rädermontage</div>
                                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Montage der Reifen auf die Felgen</div>
                                </div>
                                <span style="font-weight: 600; color: var(--primary); font-size: 1rem;">49,99 €</span>
                            </label>
                            <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem; background: var(--bg-hover); border-radius: var(--radius); cursor: pointer; transition: all 0.2s; border: 2px solid transparent;" 
                                   onmouseover="this.style.borderColor='var(--primary)'; this.style.background='var(--bg-panel)';" 
                                   onmouseout="this.style.borderColor='transparent'; this.style.background='var(--bg-hover)';">
                                <input type="checkbox" value="storage" onchange="updateTireOfferTotal()" style="width: 1.25rem; height: 1.25rem; cursor: pointer;">
                                <div style="flex: 1;">
                                    <div style="font-weight: 600; font-size: 0.9375rem;">Einlagerung der Sommerreifen</div>
                                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Saisonale Einlagerung der Reifen</div>
                                </div>
                                <span style="font-weight: 600; color: var(--primary); font-size: 1rem;">85,00 €</span>
                            </label>
                        </div>
                    </div>

                    <!-- Brieftext -->
                    <div class="detail-section" style="background: var(--bg-panel); padding: 1.5rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                        <div class="section-title" style="margin-bottom: 1rem; padding-bottom: 0.75rem; border-bottom: 2px solid var(--border); display: flex; align-items: center; gap: 0.5rem;">
                            <i class="fas fa-envelope" style="color: var(--primary);"></i>
                            <span>Brieftext</span>
                        </div>
                        <textarea id="offerLetterContent" class="form-control" rows="10" style="font-size: 0.875rem; min-height: 200px; font-family: Arial, sans-serif; line-height: 1.6;">${defaultLetterContent}</textarea>
                    </div>

                    <!-- Mitarbeiter -->
                    <div class="detail-section" style="background: var(--bg-panel); padding: 1.5rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                        <div class="section-title" style="margin-bottom: 1rem; padding-bottom: 0.75rem; border-bottom: 2px solid var(--border); display: flex; align-items: center; gap: 0.5rem;">
                            <i class="fas fa-user-tie" style="color: var(--primary);"></i>
                            <span>Mitarbeiter</span>
                        </div>
                        <input type="text" id="employeeName" class="form-control" placeholder="Name des Mitarbeiters" value="${<?php echo $js_current_user_name; ?>}" style="font-size: 0.875rem;">
                    </div>

                    <!-- Gesamtbetrag -->
                    <div class="detail-section" style="background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%); padding: 1.5rem; border-radius: var(--radius-lg); border: 2px solid var(--primary); box-shadow: var(--shadow-md);">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <div>
                                <div style="font-weight: 600; font-size: 1rem; color: rgba(255,255,255,0.9); margin-bottom: 0.25rem;">Gesamtbetrag</div>
                                <div style="font-size: 0.875rem; color: rgba(255,255,255,0.7);">inkl. aller Reifen und Zusatzleistungen</div>
                            </div>
                            <span id="totalAmount" style="font-weight: 700; font-size: 2rem; color: #fff; text-shadow: 0 2px 4px rgba(0,0,0,0.2);">0,00 €</span>
                        </div>
                    </div>
                </form>
            `;

            $(`#${modalId}Body`).html(html);
            modal.show();
            
            // Füge ersten Reifen hinzu
            addTireInput();
            
            // Speichere Daten global
            window.currentTireOffer = {
                customer: customer,
                order: order,
                autohausId: autohausId
            };
        }

        // ============ REIFEN MANUELL HINZUFÜGEN (1:1 aus storage.php) ============
        function addTireInput(containerId = 'tireInputsContainer') {
            const html = `
                <div class="form-group tire-input-group" style="border: 1px solid var(--border); padding: 1rem; border-radius: var(--radius); margin-bottom: 1rem;">
                    <div style="display: grid; grid-template-columns: 2fr 1fr 1fr 1fr 1fr; gap: 0.5rem;">
                        <input type="text" class="form-control tire-brand-model" placeholder="Hersteller, Profilbezeichnung" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-eu-label" placeholder="EU-Label" maxlength="10" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-dimension" placeholder="Dimension" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-rf" placeholder="RF" maxlength="5" style="font-size: 0.875rem;">
                        <input type="number" class="form-control tire-price" placeholder="Preis" step="0.01" min="0" style="font-size: 0.875rem;">
                    </div>
                    <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 0.5rem; margin-top: 0.5rem;">
                        <input type="number" class="form-control tire-mounting" placeholder="Montage Felge" step="0.01" min="0" style="font-size: 0.875rem;">
                        <input type="number" class="form-control tire-quantity" placeholder="Anzahl" min="1" value="1" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-amount" placeholder="Betrag" readonly style="font-size: 0.875rem; background: var(--bg-hover);">
                    </div>
                    <input type="hidden" class="tire-ean" value="">
                    <input type="hidden" class="tire-oem-part-number" value="">
                    <input type="hidden" class="tire-part-number" value="">
                    <input type="hidden" class="tire-manufacturer-number" value="">
                    <input type="hidden" class="tire-execution" value="">
                    <input type="hidden" class="tire-marking" value="">
                    <input type="hidden" class="tire-runflat" value="">
                    <input type="hidden" class="tire-vehicle-type" value="">
                    <input type="hidden" class="tire-application-area" value="">
                    <input type="hidden" class="tire-noise-emission" value="">
                    <input type="hidden" class="tire-noise-class" value="">
                    <input type="hidden" class="tire-wet-grip" value="">
                    <input type="hidden" class="tire-rolling-resistance" value="">
                    <input type="hidden" class="tire-price-net" value="">
                    <input type="hidden" class="tire-ek" value="">
                    <input type="hidden" class="tire-oem-ek" value="">
                    <input type="hidden" class="tire-rabatt" value="">
                    <input type="hidden" class="tire-kb" value="">
                    <input type="hidden" class="tire-kb-date" value="">
                    <input type="hidden" class="tire-stock" value="">
                    <input type="hidden" class="tire-warehouse" value="">
                    <input type="hidden" class="tire-notes" value="">
                    <button type="button" class="btn btn-sm btn-secondary" onclick="removeTireInput(this)" style="margin-top: 0.5rem;">
                        <i class="fas fa-trash"></i> Entfernen
                    </button>
                </div>
            `;
            $(`#${containerId}`).append(html);
            $(`#${containerId}`).on('input', '.tire-price, .tire-mounting, .tire-quantity', function() {
                calculateTireAmount($(this).closest('.tire-input-group'));
            });
        }
        
        function removeTireInput(button) {
            $(button).closest('.tire-input-group').remove();
            updateTireOfferTotal();
        }
        
        function calculateTireAmount(container) {
            const price = parseFloat($(container).find('.tire-price').val()) || 0;
            const mounting = parseFloat($(container).find('.tire-mounting').val()) || 0;
            const quantity = parseInt($(container).find('.tire-quantity').val()) || 1;
            const total = (price + mounting) * quantity;
            $(container).find('.tire-amount').val(total.toFixed(2).replace('.', ',') + ' €');
            updateTireOfferTotal();
        }

        function updateTireOfferTotal() {
            let total = 0;
            
            // Summiere alle Reifen (aus tire-amount Feld)
            $('#tireInputsContainer .tire-input-group').each(function() {
                const amountText = $(this).find('.tire-amount').val() || '0,00 €';
                const amount = parseFloat(amountText.replace(/[^\d,.-]/g, '').replace(',', '.')) || 0;
                total += amount;
            });
            
            // Addiere Services
            $('#servicesContainer input[type="checkbox"]:checked').each(function() {
                const value = $(this).val();
                if (value === 'mounting') total += 49.99;
                if (value === 'storage') total += 85.00;
            });
            
            $('#totalAmount').text(formatCurrency(total));
        }

        function saveTireOffer() {
            const offer = window.currentTireOffer;
            if (!offer) {
                alert('Fehler: Keine Angebotsdaten gefunden');
                return;
            }

            // Validiere E-Mail
            const customerEmail = $('#offerCustomerEmail').val().trim();
            if (!customerEmail) {
                alert('Bitte geben Sie eine E-Mail-Adresse ein');
                $('#offerCustomerEmail').focus();
                return;
            }
            
            // Email-Validierung
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(customerEmail)) {
                alert('Bitte geben Sie eine gültige E-Mail-Adresse ein');
                $('#offerCustomerEmail').focus();
                return;
            }
            
            // Sammle Reifen-Daten (1:1 aus storage.php)
            const newTires = [];
            $('#tireInputsContainer .tire-input-group').each(function() {
                const brandModel = $(this).find('.tire-brand-model').val();
                const dimension = $(this).find('.tire-dimension').val();
                const price = parseFloat($(this).find('.tire-price').val()) || 0;
                const mounting = parseFloat($(this).find('.tire-mounting').val()) || 0;
                const quantity = parseInt($(this).find('.tire-quantity').val()) || 1;
                const euLabel = $(this).find('.tire-eu-label').val();
                const rf = $(this).find('.tire-rf').val();
                
                // Zusätzliche Daten für Mitarbeiter (aus storage.php)
                const ean = $(this).find('.tire-ean').val();
                const oemPartNumber = $(this).find('.tire-oem-part-number').val();
                const partNumber = $(this).find('.tire-part-number').val();
                const manufacturerNumber = $(this).find('.tire-manufacturer-number').val();
                const execution = $(this).find('.tire-execution').val();
                const marking = $(this).find('.tire-marking').val();
                const runflat = $(this).find('.tire-runflat').val();
                const vehicleType = $(this).find('.tire-vehicle-type').val();
                const applicationArea = $(this).find('.tire-application-area').val();
                const noiseEmission = $(this).find('.tire-noise-emission').val();
                const noiseClass = $(this).find('.tire-noise-class').val();
                const wetGrip = $(this).find('.tire-wet-grip').val();
                const rollingResistance = $(this).find('.tire-rolling-resistance').val();
                const priceNet = parseFloat($(this).find('.tire-price-net').val()) || 0;
                const ek = parseFloat($(this).find('.tire-ek').val()) || 0;
                const oemEk = parseFloat($(this).find('.tire-oem-ek').val()) || 0;
                const rabatt = parseFloat($(this).find('.tire-rabatt').val()) || 0;
                const kb = parseFloat($(this).find('.tire-kb').val()) || 0;
                const kbDate = $(this).find('.tire-kb-date').val();
                const stock = $(this).find('.tire-stock').val();
                const warehouse = $(this).find('.tire-warehouse').val();
                const notes = $(this).find('.tire-notes').val();
                
                // Nur vollständig ausgefüllte Reifen hinzufügen
                if (brandModel && dimension && (price > 0 || mounting > 0)) {
                    newTires.push({
                        brand_model: brandModel,
                        dimension: dimension,
                        eu_label: euLabel,
                        rf: rf,
                        price: price,
                        mounting: mounting,
                        quantity: quantity,
                        total: (price + mounting) * quantity,
                        // Zusätzliche Felder
                        ean: ean,
                        oem_part_number: oemPartNumber,
                        part_number: partNumber,
                        manufacturer_number: manufacturerNumber,
                        execution: execution,
                        marking: marking,
                        runflat: runflat,
                        vehicle_type: vehicleType,
                        application_area: applicationArea,
                        noise_emission: noiseEmission,
                        noise_class: noiseClass,
                        wet_grip: wetGrip,
                        rolling_resistance: rollingResistance,
                        price_net: priceNet,
                        ek: ek,
                        oem_ek: oemEk,
                        rabatt: rabatt,
                        kb: kb,
                        kb_date: kbDate,
                        stock: stock,
                        warehouse: warehouse,
                        notes: notes
                    });
                }
            });

            // Sammle Services
            const selectedServices = [];
            $('#servicesContainer input[type="checkbox"]:checked').each(function() {
                selectedServices.push($(this).val());
            });

            if (newTires.length === 0 && selectedServices.length === 0) {
                alert('Bitte geben Sie mindestens einen Reifen ein oder wählen Sie eine Serviceleistung aus');
                return;
            }

            const letterContent = $('#offerLetterContent').val();
            const employeeName = $('#employeeName').val() || '<?php echo $current_user_name; ?>';
            const totalAmount = parseFloat($('#totalAmount').text().replace(/[^\d,.-]/g, '').replace(',', '.')) || 0;

            const requestData = {
                kunden_adr_id: offer.customer.KundenAdrID,
                autohaus_id: offer.autohausId,
                customer_name: offer.customer.KundenName1 || '',
                customer_vorname: offer.customer.KundenVorname || '',
                license_plate: offer.order.ZulassungsKz || '',
                auftrag_nr: offer.order.AuftragNr || null,
                customer_email: customerEmail,
                new_tires: newTires,
                selected_services: selectedServices,
                letter_content: letterContent,
                employee_name: employeeName,
                total_amount: totalAmount
            };

            // Zeige Loading
            $('#tireOfferModal .modal-footer button').prop('disabled', true);
            $('#tireOfferModal .modal-footer').append('<div class="loading" style="margin-top: 1rem;"><div class="spinner"></div><p>Speichere Angebot...</p></div>');

            // Zuerst Angebot erstellen
            fetch('../api/p2_tire_offers.php?action=create', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(requestData)
            })
            .then(async response => {
                const text = await response.text();
                
                // Prüfe ob JSON
                if (!text.trim()) {
                    throw new Error('Leere Antwort vom Server');
                }
                
                try {
                    return JSON.parse(text);
                } catch (e) {
                    throw new Error('Server-Antwort ist kein gültiges JSON. Möglicherweise ein PHP-Fehler: ' + text.substring(0, 200));
                }
            })
            .then(data => {
                if (data.success) {
                    // Speichere offer_id für E-Mail-Modal
                    window.currentP2OfferId = data.offer_id;
                    window.currentP2OfferToken = data.token;
                    
                    // Schließe Erstellungs-Modal
                    closeModal('tireOfferModal');
                    
                    // Öffne E-Mail-Modal
                    showP2EmailModal(customerEmail);
                } else {
                    throw new Error(data.error || 'Fehler beim Erstellen des Angebots');
                }
            })
            .catch(error => {
                showToast('error', 'Fehler', 'Fehler beim Speichern: ' + error.message);
            })
            .finally(() => {
                $('#tireOfferModal .modal-footer button').prop('disabled', false);
                $('#tireOfferModal .modal-footer .loading').remove();
            });
        }

        // ============ REIFEN AUSWAHL MODAL (1:1 aus storage.php) ============
        /**
         * Debounce-Funktion für Performance-Optimierung
         */
        function debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }

        /**
         * Öffnet das Modal zur Reifenauswahl
         */
        function openTireSelectionModal() {
            // Zeige Loading
            const loadingHtml = '<div style="position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 9999; display: flex; align-items: center; justify-content: center;"><div style="background: white; padding: 2rem; border-radius: 8px;"><i class="fas fa-spinner fa-spin"></i> Lade Reifendaten...</div></div>';
            $('body').append(loadingHtml);
            
            // Prüfe ob Daten bereits gecacht sind
            if (window.tireDataCache && window.tireDataCache.tires) {
                $('body > div:last-child').remove();
                renderTireSelectionModal(window.tireDataCache.tires);
                return;
            }
            
            // Lade alle Reifendaten aus der CSV
            fetch('../api/get_tire_recommendations.php?action=get_all_tires')
                .then(response => response.json())
                .then(data => {
                    $('body > div:last-child').remove();
                    if (data.success && data.tires) {
                        // Cache die Daten
                        window.tireDataCache = {
                            tires: data.tires,
                            timestamp: Date.now()
                        };
                        renderTireSelectionModal(data.tires);
                    } else {
                        showToast('error', 'Fehler', 'Fehler beim Laden der Reifendaten');
                    }
                })
                .catch(error => {
                    $('body > div:last-child').remove();
                    showToast('error', 'Fehler', 'Fehler beim Laden der Reifendaten: ' + error.message);
                });
        }

        /**
         * Rendert das Modal zur Reifenauswahl
         */
        function renderTireSelectionModal(tires) {
            // Extrahiere alle verfügbaren Marken
            const manufacturers = [...new Set(tires.map(t => t.manufacturer).filter(Boolean))].sort();
            
            const modalContent = `
                <div style="display: flex; flex-direction: column; height: 100%; max-height: 90vh;">
                    <!-- Header mit Suche -->
                    <div style="padding: 1.5rem; border-bottom: 2px solid var(--border); background: var(--bg-hover);">
                        <div style="display: flex; gap: 1rem; align-items: center; margin-bottom: 1rem;">
                            <div style="flex: 1; position: relative;">
                                <i class="fas fa-search" style="position: absolute; left: 1rem; top: 50%; transform: translateY(-50%); color: var(--text-secondary);"></i>
                                <input type="text" id="tireSearchInput" placeholder="Suche nach Dimension, Hersteller, Modell..." 
                                       class="form-control" 
                                       style="padding-left: 2.5rem; font-size: 0.9375rem; height: 2.5rem;">
                            </div>
                            <button type="button" class="btn btn-secondary" onclick="clearTireFilters()" style="white-space: nowrap;">
                                <i class="fas fa-eraser"></i> Filter zurücksetzen
                            </button>
                            <button type="button" class="btn btn-primary" onclick="applySelectedTires()" style="white-space: nowrap;">
                                <i class="fas fa-check"></i> Ausgewählte übernehmen (<span id="selectedTiresCount">0</span>)
                            </button>
                        </div>
                        <div style="display: flex; gap: 0.75rem; flex-wrap: wrap; align-items: center;">
                            <input type="text" id="tireDimensionFilter" placeholder="Dimension (z.B. 195/65 R 15)" 
                                   class="form-control" 
                                   style="width: 200px; font-size: 0.875rem; height: 2rem;">
                            <input type="text" id="tireKeywordFilter" placeholder="Stichwort" 
                                   class="form-control" 
                                   style="width: 200px; font-size: 0.875rem; height: 2rem;">
                            <label style="display: flex; align-items: center; gap: 0.5rem; cursor: pointer; font-size: 0.875rem;">
                                <input type="checkbox" id="tireStockFilter" style="cursor: pointer;">
                                <span>Nur mit Lagerbestand</span>
                            </label>
                        </div>
                    </div>
                    
                    <div style="display: flex; flex: 1; overflow: hidden;">
                        <!-- Sidebar mit Marken-Filter -->
                        <div style="width: 280px; border-right: 2px solid var(--border); background: var(--bg-hover); padding: 1.5rem; overflow-y: auto;">
                            <div style="margin-bottom: 1.5rem;">
                                <h3 style="font-size: 1rem; font-weight: 600; margin-bottom: 1rem; color: var(--text-main);">
                                    <i class="fas fa-filter"></i> Marken
                                </h3>
                                <div style="max-height: 300px; overflow-y: auto;">
                                    <div id="manufacturerFilters" style="display: grid; grid-template-columns: 1fr; gap: 0.5rem;">
                                        ${manufacturers.map(m => `
                                            <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.5rem; border-radius: 6px; cursor: pointer; transition: background 0.2s;">
                                                <input type="checkbox" class="manufacturer-filter" value="${m}" style="cursor: pointer;">
                                                <span style="font-size: 0.875rem; font-weight: 500;">${m}</span>
                                            </label>
                                        `).join('')}
                                    </div>
                                </div>
                            </div>
                            
                            <div style="margin-bottom: 1.5rem; padding-top: 1.5rem; border-top: 1px solid var(--border);">
                                <h3 style="font-size: 1rem; font-weight: 600; margin-bottom: 1rem; color: var(--text-main);">
                                    <i class="fas fa-tags"></i> Produkt-Segment
                                </h3>
                                <div style="display: flex; flex-direction: column; gap: 0.5rem;">
                                    <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.5rem; border-radius: 6px; cursor: pointer; transition: background 0.2s;">
                                        <input type="checkbox" class="segment-filter" value="Sommer" style="cursor: pointer;">
                                        <span style="font-size: 0.875rem;">🌞 Sommer</span>
                                    </label>
                                    <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.5rem; border-radius: 6px; cursor: pointer; transition: background 0.2s;">
                                        <input type="checkbox" class="segment-filter" value="Winter" style="cursor: pointer;">
                                        <span style="font-size: 0.875rem;">❄️ Winter</span>
                                    </label>
                                    <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.5rem; border-radius: 6px; cursor: pointer; transition: background 0.2s;">
                                        <input type="checkbox" class="segment-filter" value="All-Season" style="cursor: pointer;">
                                        <span style="font-size: 0.875rem;">🌍 Ganzjahres</span>
                                    </label>
                                </div>
                            </div>
                            
                            <div style="padding-top: 1.5rem; border-top: 1px solid var(--border);">
                                <div style="font-size: 0.875rem; color: var(--text-secondary);">
                                    <strong id="filteredTireCount">0</strong> von <strong>${tires.length}</strong> Reifen
                                </div>
                            </div>
                        </div>
                        
                        <!-- Hauptbereich mit Tabelle -->
                        <div style="flex: 1; overflow: hidden; display: flex; flex-direction: column;">
                            <div id="tireTableContainer" style="flex: 1; overflow-y: auto; padding: 1rem;">
                                <table id="tireSelectionTable" style="width: 100%; border-collapse: collapse; font-size: 0.875rem;">
                                    <thead style="position: sticky; top: 0; background: white; z-index: 10; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                                        <tr>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border); width: 40px;">
                                                <input type="checkbox" id="selectAllTires" style="cursor: pointer;">
                                            </th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border);">Dimension</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border);">EU-Label</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border);">Hersteller - Profil</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border);">Ausführung</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border);">Einsatzgebiet</th>
                                            <th style="padding: 0.75rem; text-align: center; border-bottom: 2px solid var(--border);">Bestand</th>
                                            <th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid var(--border);">VK brutto</th>
                                            <th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid var(--border);">VK netto</th>
                                            <th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid var(--border);">OT-EK</th>
                                        </tr>
                                    </thead>
                                    <tbody id="tireSelectionTableBody">
                                        <!-- Wird dynamisch gefüllt -->
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            // Erstelle Modal
            const modalId = 'tireSelectionModal';
            let modal = $(`#${modalId}`);
            if (!modal.length) {
                modal = $(`
                    <div class="modal-overlay" id="${modalId}" onclick="if(event.target.id === '${modalId}') closeModal('${modalId}')">
                        <div class="modal" style="max-width: 95vw; width: 95vw; max-height: 90vh;">
                            <div class="modal-header">
                                <div class="modal-title">Reifen auswählen</div>
                                <button class="modal-close" onclick="closeModal('${modalId}')">&times;</button>
                            </div>
                            <div class="modal-body" id="${modalId}Body" style="max-height: calc(90vh - 120px); overflow-y: auto; padding: 0;">
                            </div>
                            <div class="modal-footer">
                                <button class="btn btn-secondary" onclick="closeModal('${modalId}')">Abbrechen</button>
                                <button class="btn btn-primary" onclick="applySelectedTires()">
                                    <i class="fas fa-check"></i> Ausgewählte übernehmen (<span id="selectedTiresCountFooter">0</span>)
                                </button>
                            </div>
                        </div>
                    </div>
                `);
                $('body').append(modal);
            }
            
            $(`#${modalId}Body`).html(modalContent);
            modal.show();
            
            // Füge CSS für Hover-Effekte hinzu
            if (!$('#tireSelectionModalStyles').length) {
                $('head').append(`
                    <style id="tireSelectionModalStyles">
                        .tire-row {
                            transition: background-color 0.15s ease;
                        }
                        .tire-row:hover {
                            background-color: var(--bg-hover) !important;
                        }
                        .tire-row:active {
                            background-color: rgba(59, 130, 246, 0.1) !important;
                        }
                        #tireTableContainer {
                            scroll-behavior: smooth;
                        }
                        #tireTableContainer::-webkit-scrollbar {
                            width: 8px;
                        }
                        #tireTableContainer::-webkit-scrollbar-track {
                            background: var(--bg-hover);
                        }
                        #tireTableContainer::-webkit-scrollbar-thumb {
                            background: var(--border);
                            border-radius: 4px;
                        }
                        #tireTableContainer::-webkit-scrollbar-thumb:hover {
                            background: var(--text-secondary);
                        }
                    </style>
                `);
            }
            
            // Speichere alle Reifen global
            window.allTires = tires;
            window.filteredTires = tires;
            
            // Rendere initial die Tabelle
            renderTireTable(tires);
            
            // Event Handler
            setupTireSelectionHandlers();
        }

        /**
         * Rendert die Reifentabelle
         */
        let tireTableState = {
            currentPage: 0,
            itemsPerPage: 100,
            filteredTires: [],
            renderedIndices: new Set()
        };
        
        function renderTireTable(tires) {
            const tbody = $('#tireSelectionTableBody');
            const countSpan = $('#filteredTireCount');
            
            if (!tires || tires.length === 0) {
                tbody.html('<tr><td colspan="11" style="padding: 2rem; text-align: center; color: var(--text-secondary);">Keine Reifen gefunden</td></tr>');
                countSpan.text('0');
                tireTableState.filteredTires = [];
                tireTableState.currentPage = 0;
                tireTableState.renderedIndices.clear();
                return;
            }
            
            countSpan.text(tires.length);
            tireTableState.filteredTires = tires;
            tireTableState.currentPage = 0;
            tireTableState.renderedIndices.clear();
            
            // Leere Tabelle komplett
            tbody.html('');
            
            // Rendere nur die erste Seite
            renderTireTablePage(0);
            
            // Setup Scroll-Listener für Infinite Scroll
            const tableContainer = $('#tireTableContainer');
            if (tableContainer.length) {
                tableContainer.off('scroll.tireTable').on('scroll.tireTable', function() {
                    const container = $(this);
                    const scrollTop = container.scrollTop();
                    const scrollHeight = container[0].scrollHeight;
                    const clientHeight = container[0].clientHeight;
                    
                    // Lade nächste Seite wenn nahe am Ende
                    if (scrollTop + clientHeight >= scrollHeight - 200) {
                        const nextPage = tireTableState.currentPage + 1;
                        const totalPages = Math.ceil(tires.length / tireTableState.itemsPerPage);
                        if (nextPage < totalPages) {
                            renderTireTablePage(nextPage);
                        }
                    }
                });
            }
        }
        
        /**
         * Rendert eine Seite der Tabelle
         */
        function renderTireTablePage(page) {
            if (tireTableState.currentPage > page) return;
            
            const tbody = $('#tireSelectionTableBody');
            const startIndex = page * tireTableState.itemsPerPage;
            const endIndex = Math.min(startIndex + tireTableState.itemsPerPage, tireTableState.filteredTires.length);
            const tires = tireTableState.filteredTires.slice(startIndex, endIndex);
            
            if (tires.length === 0) return;
            
            let html = '';
            tires.forEach((tire, localIndex) => {
                const globalIndex = startIndex + localIndex;
                
                if (tireTableState.renderedIndices.has(globalIndex)) return;
                tireTableState.renderedIndices.add(globalIndex);
                
                const manufacturer = tire.manufacturer || '';
                const model = tire.model || '';
                const dimension = tire.dimension || '';
                const euLabel = tire.eu_label || '';
                const execution = tire.execution || '';
                const applicationArea = tire.application_area || '';
                const stock = tire.stock || '';
                const price = parseFloat(tire.price || 0);
                const priceNet = parseFloat(tire.price_net || 0);
                const ek = parseFloat(tire.ek || 0);
                
                // Bestimme Einsatzgebiet-Icon
                let areaIcon = '';
                if (applicationArea.includes('Winter')) areaIcon = '❄️';
                else if (applicationArea.includes('Sommer')) areaIcon = '🌞';
                else if (applicationArea.includes('Ganzjahres') || applicationArea.includes('All-Season')) areaIcon = '🌍';
                
                html += `
                    <tr class="tire-row" 
                        data-tire-index="${globalIndex}" 
                        data-tire-data='${JSON.stringify(tire).replace(/'/g, "&#39;")}'
                        style="border-bottom: 1px solid var(--border); cursor: pointer;">
                        <td style="padding: 0.75rem;">
                            <input type="checkbox" class="tire-select-checkbox" data-tire-index="${globalIndex}" style="cursor: pointer;">
                        </td>
                        <td style="padding: 0.75rem; font-weight: 600; color: var(--text-main);">${dimension}</td>
                        <td style="padding: 0.75rem;">
                            ${euLabel ? `<span style="padding: 0.25rem 0.5rem; background: var(--bg-hover); border-radius: 4px;">${euLabel}</span>` : '-'}
                        </td>
                        <td style="padding: 0.75rem;">
                            <div style="font-weight: 600; color: var(--text-main);">${manufacturer}</div>
                            <div style="font-size: 0.8125rem; color: var(--text-secondary);">${model}</div>
                        </td>
                        <td style="padding: 0.75rem; font-size: 0.8125rem; color: var(--text-secondary);">${execution}</td>
                        <td style="padding: 0.75rem;">
                            <div style="display: flex; align-items: center; gap: 0.5rem;">
                                ${areaIcon} <span style="font-size: 0.8125rem;">${applicationArea}</span>
                            </div>
                        </td>
                        <td style="padding: 0.75rem; text-align: center;">
                            ${stock ? `<span style="color: var(--success); font-weight: 600;">${stock}</span>` : '<span style="color: var(--text-secondary);">-</span>'}
                        </td>
                        <td style="padding: 0.75rem; text-align: right; font-weight: 600; color: var(--primary);">
                            ${price > 0 ? price.toFixed(2).replace('.', ',') + ' €' : '-'}
                        </td>
                        <td style="padding: 0.75rem; text-align: right; color: var(--text-secondary);">
                            ${priceNet > 0 ? priceNet.toFixed(2).replace('.', ',') + ' €' : '-'}
                        </td>
                        <td style="padding: 0.75rem; text-align: right; color: var(--text-secondary);">
                            ${ek > 0 ? ek.toFixed(2).replace('.', ',') + ' €' : '-'}
                        </td>
                        <td style="padding: 0.75rem; text-align: center;">
                            <button type="button" 
                                    class="btn btn-sm btn-secondary" 
                                    onclick="showTireDetailsModalFromTable(${globalIndex})"
                                    style="padding: 0.25rem 0.5rem; font-size: 0.75rem;"
                                    title="Alle Details anzeigen">
                                <i class="fas fa-info-circle"></i>
                            </button>
                        </td>
                    </tr>
                `;
            });
            
            if (html) {
                tbody.append(html);
                tireTableState.currentPage = page;
            }
        }

        /**
         * Setup Event Handler für Reifenauswahl
         */
        function setupTireSelectionHandlers() {
            // Suche mit Debounce
            $('#tireSearchInput').off('input.tireFilter').on('input.tireFilter', debounce(filterTires, 200));
            $('#tireDimensionFilter').off('input.tireFilter').on('input.tireFilter', debounce(filterTires, 200));
            $('#tireKeywordFilter').off('input.tireFilter').on('input.tireFilter', debounce(filterTires, 200));
            $('#tireStockFilter').off('change.tireFilter').on('change.tireFilter', filterTires);
            
            // Marken-Filter
            $('#manufacturerFilters').off('change.tireFilter').on('change.tireFilter', '.manufacturer-filter', filterTires);
            
            // Segment-Filter
            $(document).off('change.tireFilter', '.segment-filter').on('change.tireFilter', '.segment-filter', filterTires);
            
            // Select All
            $('#selectAllTires').off('change.tireFilter').on('change.tireFilter', function() {
                const checked = $(this).prop('checked');
                $('.tire-select-checkbox').prop('checked', checked);
                updateSelectedCount();
            });
            
            // Einzelne Checkboxes
            $(document).off('change.tireCheckbox', '.tire-select-checkbox').on('change.tireCheckbox', '.tire-select-checkbox', function() {
                updateSelectedCount();
                const total = $('.tire-select-checkbox').length;
                const checked = $('.tire-select-checkbox:checked').length;
                $('#selectAllTires').prop('checked', total > 0 && checked === total);
            });
            
            // Zeilen-Klick
            $(document).off('click.tireRow', '.tire-row').on('click.tireRow', '.tire-row', function(e) {
                if (!$(e.target).is('input[type="checkbox"]') && !$(e.target).closest('input[type="checkbox"]').length) {
                    const checkbox = $(this).find('.tire-select-checkbox');
                    checkbox.prop('checked', !checkbox.prop('checked')).trigger('change');
                }
            });
        }

        /**
         * Filtert Reifen basierend auf Filtern
         */
        let filterTiresTimeout = null;
        let filterTiresAnimationFrame = null;
        
        function filterTires() {
            if (filterTiresTimeout) {
                clearTimeout(filterTiresTimeout);
            }
            if (filterTiresAnimationFrame) {
                cancelAnimationFrame(filterTiresAnimationFrame);
            }
            
            const tbody = $('#tireSelectionTableBody');
            tbody.html('<tr><td colspan="10" style="padding: 2rem; text-align: center;"><i class="fas fa-spinner fa-spin"></i> Filtere Reifen...</td></tr>');
            
            filterTiresTimeout = setTimeout(() => {
                filterTiresAnimationFrame = requestAnimationFrame(() => {
                    const searchTerm = $('#tireSearchInput').val().toLowerCase();
                    const dimensionFilter = $('#tireDimensionFilter').val().toLowerCase();
                    const keywordFilter = $('#tireKeywordFilter').val().toLowerCase();
                    const stockOnly = $('#tireStockFilter').prop('checked');
                    const selectedManufacturers = $('.manufacturer-filter:checked').map(function() {
                        return $(this).val().toLowerCase();
                    }).get();
                    const selectedSegments = $('.segment-filter:checked').map(function() {
                        return $(this).val();
                    }).get();
                    
                    let filtered = [];
                    const allTires = window.allTires;
                    
                    allTires.forEach(tire => {
                        if (selectedManufacturers.length > 0) {
                            const tireManufacturer = (tire.manufacturer || '').toLowerCase();
                            if (!selectedManufacturers.includes(tireManufacturer)) return;
                        }
                        
                        if (stockOnly && !tire.stock) return;
                        
                        if (dimensionFilter && !(tire.dimension || '').toLowerCase().includes(dimensionFilter)) return;
                        
                        if (searchTerm) {
                            const searchable = (tire.dimension || '') + ' ' + 
                                             (tire.manufacturer || '') + ' ' + 
                                             (tire.model || '') + ' ' + 
                                             (tire.brand_model || '');
                            if (!searchable.toLowerCase().includes(searchTerm)) return;
                        }
                        
                        if (keywordFilter) {
                            const keywordSearchable = (tire.manufacturer || '') + ' ' + 
                                                     (tire.model || '') + ' ' + 
                                                     (tire.execution || '') + ' ' + 
                                                     (tire.application_area || '') + ' ' + 
                                                     (tire.oem_part_number || '');
                            if (!keywordSearchable.toLowerCase().includes(keywordFilter)) return;
                        }
                        
                        if (selectedSegments.length > 0) {
                            const applicationArea = tire.application_area || '';
                            let matches = false;
                            for (let j = 0; j < selectedSegments.length; j++) {
                                const segment = selectedSegments[j];
                                if (segment === 'Sommer' && applicationArea.includes('Sommer')) {
                                    matches = true;
                                    break;
                                }
                                if (segment === 'Winter' && applicationArea.includes('Winter')) {
                                    matches = true;
                                    break;
                                }
                                if (segment === 'All-Season' && (applicationArea.includes('Ganzjahres') || applicationArea.includes('All-Season'))) {
                                    matches = true;
                                    break;
                                }
                            }
                            if (!matches) return;
                        }
                        
                        filtered.push(tire);
                    });
                    
                    window.filteredTires = filtered;
                    renderTireTable(filtered);
                    updateSelectedCount();
                });
            }, 150);
        }

        /**
         * Aktualisiert die Anzahl ausgewählter Reifen
         */
        function updateSelectedCount() {
            const count = $('.tire-select-checkbox:checked').length;
            $('#selectedTiresCount').text(count);
            $('#selectedTiresCountFooter').text(count);
        }

        /**
         * Setzt Filter zurück
         */
        function clearTireFilters() {
            $('#tireSearchInput').val('');
            $('#tireDimensionFilter').val('');
            $('#tireKeywordFilter').val('');
            $('#tireStockFilter').prop('checked', false);
            $('.manufacturer-filter').prop('checked', false);
            $('.segment-filter').prop('checked', false);
            filterTires();
        }

        /**
         * Übernimmt ausgewählte Reifen
         */
        function applySelectedTires() {
            const selectedIndices = $('.tire-select-checkbox:checked').map(function() {
                return parseInt($(this).data('tire-index'));
            }).get();
            
            if (selectedIndices.length === 0) {
                showToast('warning', 'Keine Auswahl', 'Bitte wählen Sie mindestens einen Reifen aus');
                return;
            }
            
            const tireInputsContainer = $('#tireInputsContainer');
            if (!tireInputsContainer.length) {
                showToast('error', 'Fehler', 'Reifen-Eingabefelder nicht gefunden. Bitte öffnen Sie das Angebots-Modal erneut.');
                return;
            }
            
            const tireOfferModal = $('#tireOfferModal');
            
            const btn = $('button[onclick*="applySelectedTires"]');
            const originalHtml = btn.html();
            btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Verarbeite...');
            
            const selectedTires = selectedIndices.map(index => window.filteredTires[index]);
            
            // Füge ausgewählte Reifen zu den Eingabefeldern hinzu
            selectedTires.forEach(tire => {
                let tireGroup = tireInputsContainer.find('.tire-input-group').first();
                const brandModel = tireGroup.find('.tire-brand-model').val();
                const dimension = tireGroup.find('.tire-dimension').val();
                
                if (brandModel && dimension) {
                    addTireInput();
                    tireGroup = tireInputsContainer.find('.tire-input-group').last();
                }
                
                // Fülle Felder
                tireGroup.find('.tire-brand-model').val(tire.brand_model || `${tire.manufacturer || ''} ${tire.model || ''}`.trim());
                tireGroup.find('.tire-eu-label').val(tire.eu_label || '');
                tireGroup.find('.tire-dimension').val(tire.dimension || '');
                tireGroup.find('.tire-rf').val(tire.runflat || '');
                tireGroup.find('.tire-price').val(tire.price || 0);
                tireGroup.find('.tire-ean').val(tire.ean || '');
                tireGroup.find('.tire-oem-part-number').val(tire.oem_part_number || '');
                tireGroup.find('.tire-part-number').val(tire.part_number || '');
                tireGroup.find('.tire-manufacturer-number').val(tire.manufacturer_number || '');
                tireGroup.find('.tire-execution').val(tire.execution || '');
                tireGroup.find('.tire-marking').val(tire.marking || '');
                tireGroup.find('.tire-runflat').val(tire.runflat || '');
                tireGroup.find('.tire-vehicle-type').val(tire.vehicle_type || '');
                tireGroup.find('.tire-application-area').val(tire.application_area || '');
                tireGroup.find('.tire-noise-emission').val(tire.noise_emission || '');
                tireGroup.find('.tire-noise-class').val(tire.noise_class || '');
                tireGroup.find('.tire-wet-grip').val(tire.wet_grip || '');
                tireGroup.find('.tire-rolling-resistance').val(tire.rolling_resistance || '');
                tireGroup.find('.tire-price-net').val(tire.price_net || 0);
                tireGroup.find('.tire-ek').val(tire.ek || 0);
                tireGroup.find('.tire-oem-ek').val(tire.oem_ek || 0);
                tireGroup.find('.tire-rabatt').val(tire.rabatt || 0);
                tireGroup.find('.tire-kb').val(tire.kb || 0);
                tireGroup.find('.tire-kb-date').val(tire.kb_date || '');
                tireGroup.find('.tire-stock').val(tire.stock || '');
                tireGroup.find('.tire-warehouse').val(tire.warehouse || '');
                tireGroup.find('.tire-notes').val(tire.notes || '');
                
                calculateTireAmount(tireGroup);
            });
            
            // Schließe das Auswahl-Modal
            closeModal('tireSelectionModal');
            
            // Stelle sicher, dass das tireOfferModal noch sichtbar ist
            if (!tireOfferModal.is(':visible')) {
                tireOfferModal.show();
            }
            
            // Scroll zum ersten hinzugefügten Reifen
            setTimeout(() => {
                const firstTireGroup = tireInputsContainer.find('.tire-input-group').first();
                if (firstTireGroup.length && firstTireGroup.is(':visible')) {
                    const offset = firstTireGroup.offset();
                    if (offset) {
                        $('html, body').animate({
                            scrollTop: offset.top - 100
                        }, 300);
                    }
                }
            }, 100);
            
            btn.prop('disabled', false).html(originalHtml);
            showToast('success', 'Erfolg', `${selectedTires.length} Reifen erfolgreich hinzugefügt`);
        }

        // ============ REIFEN DETAILS MODAL (1:1 aus storage.php) ============
        /**
         * Kopiert Text in die Zwischenablage
         */
        function copyToClipboard(text, buttonElement) {
            if (!text || text === '-') {
                showToast('warning', 'Kein Wert', 'Kein Wert zum Kopieren');
                return;
            }
            
            const cleanText = text.toString().replace(/\s*€\s*/g, '').trim();
            
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(cleanText).then(() => {
                    const icon = $(buttonElement).find('i');
                    const originalClass = icon.attr('class');
                    icon.removeClass('fa-copy').addClass('fa-check');
                    $(buttonElement).css({
                        'background': 'var(--success)',
                        'color': 'white'
                    });
                    
                    setTimeout(() => {
                        icon.removeClass('fa-check').addClass('fa-copy');
                        $(buttonElement).css({
                            'background': 'var(--bg-hover)',
                            'color': 'var(--text-secondary)'
                        });
                    }, 1500);
                    
                    showToast('success', 'Kopiert', 'In Zwischenablage kopiert: ' + cleanText);
                }).catch(err => {
                    showToast('error', 'Fehler', 'Fehler beim Kopieren');
                });
            } else {
                const textArea = document.createElement('textarea');
                textArea.value = cleanText;
                textArea.style.position = 'fixed';
                textArea.style.opacity = '0';
                document.body.appendChild(textArea);
                textArea.select();
                try {
                    document.execCommand('copy');
                    showToast('success', 'Kopiert', 'In Zwischenablage kopiert: ' + cleanText);
                } catch (err) {
                    showToast('error', 'Fehler', 'Fehler beim Kopieren');
                }
                document.body.removeChild(textArea);
            }
        }

        /**
         * Erstellt ein kopierbares Feld mit Kopiersymbol
         */
        function createCopyableField(value, displayValue = null) {
            const display = displayValue !== null ? displayValue : (value || '-');
            if (!value || value === '-') {
                return `<span style="font-weight: 600; color: var(--text-main);">${display}</span>`;
            }
            const cleanValue = value.toString().replace(/\s*€\s*/g, '').trim();
            return `
                <div style="display: flex; align-items: center; gap: 0.5rem;">
                    <span style="font-weight: 600; color: var(--text-main); font-family: ${cleanValue.match(/^\d+$/) ? 'monospace' : 'inherit'}; font-size: ${cleanValue.match(/^\d+$/) ? '0.8125rem' : 'inherit'};">${display}</span>
                    <button type="button" 
                            onclick="copyToClipboard('${cleanValue.replace(/'/g, "\\'")}', this)"
                            style="padding: 0.25rem 0.5rem; background: var(--bg-hover); border: 1px solid var(--border); border-radius: 4px; cursor: pointer; color: var(--text-secondary); transition: all 0.2s; display: flex; align-items: center; justify-content: center; min-width: 28px; height: 28px;"
                            onmouseover="this.style.background='var(--primary)'; this.style.color='white'; this.style.borderColor='var(--primary)';"
                            onmouseout="this.style.background='var(--bg-hover)'; this.style.color='var(--text-secondary)'; this.style.borderColor='var(--border)';"
                            title="In Zwischenablage kopieren">
                        <i class="fas fa-copy" style="font-size: 0.75rem;"></i>
                    </button>
                </div>
            `;
        }

        /**
         * Zeigt Reifendetails aus der Tabelle
         */
        function showTireDetailsModalFromTable(tireIndex) {
            const row = $(`.tire-row[data-tire-index="${tireIndex}"]`);
            const tireDataStr = row.attr('data-tire-data');
            
            if (!tireDataStr) {
                if (window.allTires && window.allTires[tireIndex]) {
                    renderTireDetailsModal(window.allTires[tireIndex]);
                    return;
                }
                showToast('error', 'Fehler', 'Reifendaten nicht gefunden');
                return;
            }
            
            let tireData;
            try {
                tireData = JSON.parse(tireDataStr.replace(/&#39;/g, "'"));
            } catch (e) {
                if (window.allTires && window.allTires[tireIndex]) {
                    renderTireDetailsModal(window.allTires[tireIndex]);
                    return;
                }
                showToast('error', 'Fehler', 'Fehler beim Laden der Reifendaten');
                return;
            }
            
            renderTireDetailsModal(tireData);
        }

        /**
         * Rendert das Modal mit allen Reifendetails
         */
        function renderTireDetailsModal(tire) {
            const manufacturer = tire.manufacturer || 'Unbekannt';
            const model = tire.model || '';
            const dimension = tire.dimension || '-';
            const brandImageUrl = `../assets/images/brands/${manufacturer.toLowerCase().replace(/\s+/g, '-')}.png`;
            
            const formatPrice = (value) => {
                if (!value || value === 0) return '-';
                return parseFloat(value).toFixed(2).replace('.', ',') + ' €';
            };
            
            const formatDate = (value) => {
                if (!value || value === '') return '-';
                return value;
            };
            
            const calculateMargin = () => {
                const vkBrutto = parseFloat(tire.price) || 0;
                const ek = parseFloat(tire.oem_ek) || parseFloat(tire.ek) || 0;
                
                if (vkBrutto === 0 || ek === 0) {
                    return { absolute: null, percent: null, ekUsed: null };
                }
                
                const marginAbsolute = vkBrutto - ek;
                const marginPercent = (marginAbsolute / vkBrutto) * 100;
                const ekUsed = tire.oem_ek ? 'OEM-EK' : 'OT-EK';
                
                return {
                    absolute: marginAbsolute,
                    percent: marginPercent,
                    ekUsed: ekUsed
                };
            };
            
            const margin = calculateMargin();
            
            const modalContent = `
                <div style="max-height: 85vh; overflow-y: auto;">
                    <!-- Header mit Markenbild und Basis-Info -->
                    <div style="display: flex; align-items: center; gap: 1rem; padding: 1rem; background: linear-gradient(135deg, rgba(59, 130, 246, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%); border-radius: var(--radius-lg); margin-bottom: 1rem; border: 2px solid var(--primary);">
                        <div style="width: 80px; height: 80px; border-radius: 10px; background: white; display: flex; align-items: center; justify-content: center; border: 2px solid var(--border); flex-shrink: 0; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
                            <img src="${brandImageUrl}" 
                                 alt="${manufacturer}" 
                                 style="max-width: 70px; max-height: 70px; object-fit: contain;"
                                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                            <div style="display: none; width: 100%; height: 100%; align-items: center; justify-content: center; font-size: 1.25rem; color: var(--text-secondary); font-weight: 700;">
                                ${manufacturer.substring(0, 3).toUpperCase()}
                            </div>
                        </div>
                        <div style="flex: 1;">
                            <h3 style="margin: 0 0 0.25rem 0; font-size: 1.25rem; font-weight: 700; color: var(--text-main);">
                                ${manufacturer} ${model}
                            </h3>
                            <div style="font-size: 1rem; color: var(--text-secondary); font-weight: 600; margin-bottom: 0.5rem;">
                                ${dimension}
                            </div>
                            <div style="display: flex; gap: 1rem; flex-wrap: wrap; align-items: center;">
                                ${tire.eu_label ? `
                                <div style="display: flex; align-items: center; gap: 0.5rem; padding: 0.5rem 1rem; background: var(--bg-hover); border-radius: 6px; font-weight: 600; font-size: 0.9375rem;">
                                    <span>EU-Label: ${tire.eu_label}</span>
                                    <button type="button" 
                                            onclick="copyToClipboard('${tire.eu_label}', this)"
                                            style="padding: 0.25rem 0.5rem; background: transparent; border: 1px solid var(--border); border-radius: 4px; cursor: pointer; color: var(--text-secondary); transition: all 0.2s;"
                                            onmouseover="this.style.background='var(--primary)'; this.style.color='white'; this.style.borderColor='var(--primary)';"
                                            onmouseout="this.style.background='transparent'; this.style.color='var(--text-secondary)'; this.style.borderColor='var(--border)';"
                                            title="EU-Label kopieren">
                                        <i class="fas fa-copy" style="font-size: 0.75rem;"></i>
                                    </button>
                                </div>
                                ` : ''}
                                ${dimension ? `
                                <div style="display: flex; align-items: center; gap: 0.5rem; padding: 0.5rem 1rem; background: var(--bg-hover); border-radius: 6px; font-weight: 600; font-size: 0.9375rem;">
                                    <span>Dimension: ${dimension}</span>
                                    <button type="button" 
                                            onclick="copyToClipboard('${dimension}', this)"
                                            style="padding: 0.25rem 0.5rem; background: transparent; border: 1px solid var(--border); border-radius: 4px; cursor: pointer; color: var(--text-secondary); transition: all 0.2s;"
                                            onmouseover="this.style.background='var(--primary)'; this.style.color='white'; this.style.borderColor='var(--primary)';"
                                            onmouseout="this.style.background='transparent'; this.style.color='var(--text-secondary)'; this.style.borderColor='var(--border)';"
                                            title="Dimension kopieren">
                                        <i class="fas fa-copy" style="font-size: 0.75rem;"></i>
                                    </button>
                                </div>
                                ` : ''}
                                ${tire.price ? `<span style="padding: 0.5rem 1rem; background: var(--primary); color: white; border-radius: 6px; font-weight: 700; font-size: 1.125rem;">${formatPrice(tire.price)}</span>` : ''}
                            </div>
                        </div>
                    </div>
                    
                    <!-- Hauptinformationen in Grid-Layout -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
                        <!-- Basis-Informationen -->
                        <div style="background: var(--bg-panel); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-main); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-info-circle" style="color: var(--primary); font-size: 0.875rem;"></i>
                                Basis-Informationen
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Hersteller:</span>
                                    ${createCopyableField(manufacturer)}
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Modell:</span>
                                    ${createCopyableField(model)}
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Dimension:</span>
                                    ${createCopyableField(dimension)}
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Ausführung:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.execution || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Markierung:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.marking || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between;">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Fahrzeugart:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.vehicle_type || '-'}</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- EU-Label & Eigenschaften -->
                        <div style="background: var(--bg-panel); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-main); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-certificate" style="color: var(--primary); font-size: 0.875rem;"></i>
                                EU-Label & Eigenschaften
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">EU-Label:</span>
                                    ${tire.eu_label ? `
                                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                                        <span style="font-weight: 600; color: var(--text-main); padding: 0.25rem 0.75rem; background: var(--bg-hover); border-radius: 4px;">${tire.eu_label}</span>
                                        <button type="button" 
                                                onclick="copyToClipboard('${tire.eu_label}', this)"
                                                style="padding: 0.25rem 0.5rem; background: var(--bg-hover); border: 1px solid var(--border); border-radius: 4px; cursor: pointer; color: var(--text-secondary); transition: all 0.2s;"
                                                onmouseover="this.style.background='var(--primary)'; this.style.color='white'; this.style.borderColor='var(--primary)';"
                                                onmouseout="this.style.background='var(--bg-hover)'; this.style.color='var(--text-secondary)'; this.style.borderColor='var(--border)';"
                                                title="EU-Label kopieren">
                                            <i class="fas fa-copy" style="font-size: 0.75rem;"></i>
                                        </button>
                                    </div>
                                    ` : '<span style="font-weight: 600; color: var(--text-main);">-</span>'}
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Rollwiderstand:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.rolling_resistance || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Nasshaftung:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.wet_grip || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Geräuschemission:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.noise_emission || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Geräuschklasse:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.noise_class || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between;">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Einsatzgebiet:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.application_area || '-'}</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Technische Details -->
                        <div style="background: var(--bg-panel); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-main); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-cog" style="color: var(--primary); font-size: 0.875rem;"></i>
                                Technische Details
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">EAN:</span>
                                    ${createCopyableField(tire.ean)}
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Hersteller Artikel-Nr:</span>
                                    ${createCopyableField(tire.manufacturer_number)}
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Volvo-Ersatzteil-Nr:</span>
                                    ${createCopyableField(tire.oem_part_number)}
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Ersatzteil-Nr:</span>
                                    ${createCopyableField(tire.part_number)}
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Runflat:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.runflat || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between;">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Lagerbestand:</span>
                                    <span style="font-weight: 600; color: ${tire.stock ? 'var(--success)' : 'var(--text-secondary)'};">${tire.stock || '-'}</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Preisinformationen -->
                        <div style="background: var(--bg-panel); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-main); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-euro-sign" style="color: var(--primary); font-size: 0.875rem;"></i>
                                Preisinformationen
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">VK brutto:</span>
                                    <span style="font-weight: 700; color: var(--primary); font-size: 1.125rem;">${formatPrice(tire.price)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">VK netto:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${formatPrice(tire.price_net)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">OT-EK:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${formatPrice(tire.ek)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">OEM-EK:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${formatPrice(tire.oem_ek)}</span>
                                </div>
                                ${margin.absolute !== null ? `
                                <div style="padding: 0.75rem; background: ${margin.absolute >= 0 ? 'rgba(16, 185, 129, 0.1)' : 'rgba(239, 68, 68, 0.1)'}; border-radius: 6px; border: 1px solid ${margin.absolute >= 0 ? 'rgba(16, 185, 129, 0.3)' : 'rgba(239, 68, 68, 0.3)'}; margin: 0.25rem 0;">
                                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 0.25rem;">
                                        <span style="color: var(--text-secondary); font-size: 0.8125rem; font-weight: 600;">Marge (${margin.ekUsed}):</span>
                                        <span style="font-weight: 700; color: ${margin.absolute >= 0 ? 'var(--success)' : 'var(--danger)'}; font-size: 1rem;">
                                            ${formatPrice(margin.absolute)} (${margin.percent >= 0 ? '+' : ''}${margin.percent.toFixed(2).replace('.', ',')}%)
                                        </span>
                                    </div>
                                </div>
                                ` : ''}
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Rabatt:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${formatPrice(tire.rabatt)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between;">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">KB:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${formatPrice(tire.kb)}</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Lager & Sonstiges -->
                        <div style="background: var(--bg-panel); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-main); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-warehouse" style="color: var(--primary); font-size: 0.875rem;"></i>
                                Lager & Sonstiges
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Eigenes Lager:</span>
                                    <span style="font-weight: 600; color: ${tire.stock ? 'var(--success)' : 'var(--text-secondary)'};">${tire.stock || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Lager:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${tire.warehouse || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">KB-Datum:</span>
                                    <span style="font-weight: 600; color: var(--text-main);">${formatDate(tire.kb_date)}</span>
                                </div>
                                ${tire.notes ? `
                                <div style="padding-top: 0.5rem; border-top: 1px solid var(--border);">
                                    <div style="color: var(--text-secondary); font-size: 0.875rem; margin-bottom: 0.5rem;">Bemerkungen:</div>
                                    <div style="font-weight: 500; color: var(--text-main); line-height: 1.5; padding: 0.75rem; background: var(--bg-hover); border-radius: 6px; font-size: 0.875rem;">
                                        ${tire.notes}
                                    </div>
                                </div>
                                ` : ''}
                            </div>
                        </div>
                    </div>
                    
                    <!-- Aktions-Buttons -->
                    <div style="display: flex; gap: 1rem; justify-content: flex-end; padding-top: 1rem; border-top: 2px solid var(--border); margin-top: 1rem;">
                        <button type="button" 
                                class="btn btn-secondary" 
                                onclick="closeModal('tireDetailsModal')"
                                style="white-space: nowrap;">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                        <button type="button" 
                                class="btn btn-primary" 
                                onclick="selectTireFromDetailsModal()"
                                style="white-space: nowrap;">
                            <i class="fas fa-plus"></i> Reifen auswählen
                        </button>
                    </div>
                </div>
            `;
            
            window.currentTireDetails = tire;
            
            const modalId = 'tireDetailsModal';
            let modal = $(`#${modalId}`);
            if (!modal.length) {
                modal = $(`
                    <div class="modal-overlay" id="${modalId}" onclick="if(event.target.id === '${modalId}') closeModal('${modalId}')">
                        <div class="modal" style="max-width: 95vw; width: 95vw; max-height: 90vh;">
                            <div class="modal-header">
                                <div class="modal-title">Reifendetails - ${manufacturer} ${model}</div>
                                <button class="modal-close" onclick="closeModal('${modalId}')">&times;</button>
                            </div>
                            <div class="modal-body" id="${modalId}Body" style="max-height: calc(90vh - 120px); overflow-y: auto; padding: 1.5rem;">
                            </div>
                            <div class="modal-footer">
                                <button class="btn btn-secondary" onclick="closeModal('${modalId}')">Schließen</button>
                                <button class="btn btn-primary" onclick="selectTireFromDetailsModal()">
                                    <i class="fas fa-plus"></i> Reifen auswählen
                                </button>
                            </div>
                        </div>
                    </div>
                `);
                $('body').append(modal);
            }
            
            $(`#${modalId}Body`).html(modalContent);
            modal.show();
        }

        /**
         * Wählt einen Reifen aus dem Details-Modal aus
         */
        function selectTireFromDetailsModal() {
            const tire = window.currentTireDetails;
            if (!tire) {
                showToast('error', 'Fehler', 'Reifendaten nicht gefunden');
                return;
            }
            
            let tireGroup = $('#tireInputsContainer .tire-input-group').first();
            const brandModel = tireGroup.find('.tire-brand-model').val();
            const dimension = tireGroup.find('.tire-dimension').val();
            
            if (brandModel && dimension) {
                addTireInput();
                tireGroup = $('#tireInputsContainer .tire-input-group').last();
            }
            
            tireGroup.find('.tire-brand-model').val(tire.brand_model || `${tire.manufacturer || ''} ${tire.model || ''}`.trim());
            tireGroup.find('.tire-dimension').val(tire.dimension || '');
            tireGroup.find('.tire-eu-label').val(tire.eu_label || '');
            tireGroup.find('.tire-rf').val(tire.runflat || '');
            tireGroup.find('.tire-price').val(tire.price || 0);
            tireGroup.find('.tire-ean').val(tire.ean || '');
            tireGroup.find('.tire-oem-part-number').val(tire.oem_part_number || '');
            tireGroup.find('.tire-part-number').val(tire.part_number || '');
            tireGroup.find('.tire-manufacturer-number').val(tire.manufacturer_number || '');
            tireGroup.find('.tire-execution').val(tire.execution || '');
            tireGroup.find('.tire-marking').val(tire.marking || '');
            tireGroup.find('.tire-runflat').val(tire.runflat || '');
            tireGroup.find('.tire-vehicle-type').val(tire.vehicle_type || '');
            tireGroup.find('.tire-application-area').val(tire.application_area || '');
            tireGroup.find('.tire-noise-emission').val(tire.noise_emission || '');
            tireGroup.find('.tire-noise-class').val(tire.noise_class || '');
            tireGroup.find('.tire-wet-grip').val(tire.wet_grip || '');
            tireGroup.find('.tire-rolling-resistance').val(tire.rolling_resistance || '');
            tireGroup.find('.tire-price-net').val(tire.price_net || 0);
            tireGroup.find('.tire-ek').val(tire.ek || 0);
            tireGroup.find('.tire-oem-ek').val(tire.oem_ek || 0);
            tireGroup.find('.tire-rabatt').val(tire.rabatt || 0);
            tireGroup.find('.tire-kb').val(tire.kb || 0);
            tireGroup.find('.tire-kb-date').val(tire.kb_date || '');
            tireGroup.find('.tire-stock').val(tire.stock || '');
            tireGroup.find('.tire-warehouse').val(tire.warehouse || '');
            tireGroup.find('.tire-notes').val(tire.notes || '');
            
            calculateTireAmount(tireGroup);
            
            closeModal('tireDetailsModal');
            
            setTimeout(() => {
                const offset = tireGroup.offset();
                if (offset) {
                    $('html, body').animate({
                        scrollTop: offset.top - 100
                    }, 300);
                }
            }, 100);
            
            showToast('success', 'Erfolg', 'Reifen erfolgreich ausgewählt');
        }

        // ============ AUFTRÄGE PRO FAHRZEUG ============
        function viewOrdersByLicense(licensePlate) {
            const autohaus = $('#autohausSelect').val();
            if (!autohaus || !licensePlate) return;

            const modalId = 'ordersByLicenseModal';
            let modal = $(`#${modalId}`);
            if (!modal.length) {
                modal = $(`
                    <div class="modal-overlay" id="${modalId}" onclick="if(event.target.id === '${modalId}') closeModal('${modalId}')">
                        <div class="modal" style="max-width: 90vw; width: 90vw; max-height: 90vh;">
                            <div class="modal-header">
                                <div class="modal-title">Aufträge für Kennzeichen: ${licensePlate}</div>
                                <button class="modal-close" onclick="closeModal('${modalId}')">&times;</button>
                            </div>
                            <div class="modal-body" id="${modalId}Body" style="max-height: calc(90vh - 120px); overflow-y: auto;">
                                <div class="loading"><div class="spinner"></div><p>Lade Aufträge...</p></div>
                            </div>
                        </div>
                    </div>
                `);
                $('body').append(modal);
            }

            modal.show();

            fetch(`../api/p2_api_proxy.php?action=get_orders_by_license&autohaus_id=${autohaus}&license_plate=${encodeURIComponent(licensePlate)}`)
                .then(r => r.json())
                .then(data => {
                    const orders = data.success ? data.data : [];
                    
                    if (orders.length === 0) {
                        $(`#${modalId}Body`).html('<div class="empty-state"><p>Keine Aufträge für dieses Kennzeichen gefunden</p></div>');
                        return;
                    }

                    // Berechne Gesamtwerte
                    const totalUmsatz = orders.reduce((sum, o) => sum + (parseFloat(o.Umsatz || 0)), 0);
                    const totalMarge = orders.reduce((sum, o) => sum + (parseFloat(o.Marge || 0)), 0);
                    const avgMargePercent = orders.length > 0 ? (totalMarge / totalUmsatz * 100) : 0;

                    let html = `
                        <div style="display: grid; gap: 1.5rem;">
                            <div style="padding: 1rem; background: var(--bg-hover); border-radius: var(--radius); display: grid; grid-template-columns: repeat(3, 1fr); gap: 1rem;">
                                <div>
                                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Anzahl Aufträge</div>
                                    <div style="font-size: 1.5rem; font-weight: 700; color: var(--primary);">${orders.length}</div>
                                </div>
                                <div>
                                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Gesamtumsatz</div>
                                    <div style="font-size: 1.5rem; font-weight: 700; color: var(--success);">${formatCurrency(totalUmsatz)}</div>
                                </div>
                                <div>
                                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Durchschnittliche Marge</div>
                                    <div style="font-size: 1.5rem; font-weight: 700; color: var(--primary);">${avgMargePercent.toFixed(1)}%</div>
                                </div>
                            </div>

                            <table class="orders-table">
                                <thead>
                                    <tr>
                                        <th>Auftrag</th>
                                        <th>Datum</th>
                                        <th>Umsatz</th>
                                        <th>Marge</th>
                                        <th>Status</th>
                                        <th>Aktionen</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${orders.map(order => `
                                        <tr>
                                            <td>${order.AuftragNr || '-'}</td>
                                            <td>${formatDate(order.DatumAuftrag)}</td>
                                            <td>${formatCurrency(order.Umsatz || 0)}</td>
                                            <td>
                                                <span class="badge ${getMarginBadgeClass(order.MargeProzent)}">
                                                    ${formatCurrency(order.Marge || 0)} (${(order.MargeProzent || 0).toFixed(1)}%)
                                                </span>
                                            </td>
                                            <td><span class="badge badge-info">${order.Auftragstatus || '-'}</span></td>
                                            <td>
                                                <button class="btn btn-primary" onclick="viewOrder(${order.AuftragNr}); closeModal('${modalId}');">
                                                    <i class="fas fa-eye"></i> Details
                                                </button>
                                            </td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    `;

                    $(`#${modalId}Body`).html(html);
                })
                .catch(error => {
                    $(`#${modalId}Body`).html(`<div class="empty-state"><p>Fehler beim Laden: ${error.message}</p></div>`);
                });
        }

        // ============ E-MAIL-MODAL (Modern Design) ============
        function showP2EmailModal(customerEmail = '') {
            // Entferne vorhandenes Modal
            const existingModal = document.querySelector('.p2-email-modal');
            if (existingModal) {
                existingModal.remove();
            }
            
            const modal = document.createElement('div');
            modal.className = 'modal p2-email-modal';
            modal.style.cssText = 'display: flex; position: fixed; z-index: 10000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background: rgba(0,0,0,0.7); backdrop-filter: blur(10px); align-items: center; justify-content: center; padding: 2rem;';
            modal.innerHTML = `
                <div class="modal-content" style="background: var(--bg-panel); margin: auto; padding: 0; border: none; width: 100%; max-width: 650px; border-radius: var(--radius-lg); box-shadow: 0 20px 25px -5px rgba(0,0,0,0.3), 0 10px 10px -5px rgba(0,0,0,0.2); overflow: hidden;">
                    <div class="modal-header" style="padding: 1.75rem 2rem; border-bottom: 1px solid var(--border); background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%); display: flex; justify-content: space-between; align-items: center;">
                        <h3 style="margin: 0; color: white; font-size: 1.25rem; font-weight: 600; display: flex; align-items: center; gap: 0.75rem;">
                            <i class="fas fa-envelope" style="font-size: 1.25rem;"></i> Angebot per E-Mail versenden
                        </h3>
                        <button class="close-btn" onclick="closeP2EmailModal()" style="background: rgba(255,255,255,0.2); border: none; font-size: 1.5rem; cursor: pointer; color: white; width: 36px; height: 36px; border-radius: var(--radius); display: flex; align-items: center; justify-content: center; transition: all 0.2s ease;" onmouseover="this.style.background='rgba(255,255,255,0.3)'" onmouseout="this.style.background='rgba(255,255,255,0.2)'">&times;</button>
                    </div>
                    <div class="modal-body" style="padding: 2rem;">
                        <div class="form-group" style="margin-bottom: 1.5rem;">
                            <label for="p2RecipientEmail" style="display: block; margin-bottom: 0.75rem; font-weight: 600; color: var(--text-main); font-size: 0.9375rem;">
                                <i class="fas fa-at" style="margin-right: 0.5rem; color: var(--primary);"></i>E-Mail-Adresse des Kunden
                            </label>
                            <input type="email" id="p2RecipientEmail" class="form-control" 
                                   placeholder="kunde@beispiel.de" value="${customerEmail}" required 
                                   style="width: 100%; padding: 0.875rem 1rem; border: 2px solid var(--border); border-radius: var(--radius); font-size: 0.9375rem; transition: all 0.2s ease; background: var(--bg-panel); color: var(--text-main);" 
                                   onfocus="this.style.borderColor='var(--primary)'; this.style.boxShadow='0 0 0 3px rgba(59, 130, 246, 0.1)'" 
                                   onblur="this.style.borderColor='var(--border)'; this.style.boxShadow='none'">
                        </div>
                        
                        <div class="form-group" style="margin-bottom: 1.5rem;">
                            <label for="p2EmailSubject" style="display: block; margin-bottom: 0.75rem; font-weight: 600; color: var(--text-main); font-size: 0.9375rem;">
                                <i class="fas fa-tag" style="margin-right: 0.5rem; color: var(--primary);"></i>Betreff
                            </label>
                            <input type="text" id="p2EmailSubject" class="form-control" 
                                   value="Reifenangebot - Boxenstop" required
                                   style="width: 100%; padding: 0.875rem 1rem; border: 2px solid var(--border); border-radius: var(--radius); font-size: 0.9375rem; transition: all 0.2s ease; background: var(--bg-panel); color: var(--text-main);" 
                                   onfocus="this.style.borderColor='var(--primary)'; this.style.boxShadow='0 0 0 3px rgba(59, 130, 246, 0.1)'" 
                                   onblur="this.style.borderColor='var(--border)'; this.style.boxShadow='none'">
                        </div>
                        
                        <div class="form-group" style="margin-bottom: 0;">
                            <label for="p2EmailMessage" style="display: block; margin-bottom: 0.75rem; font-weight: 600; color: var(--text-main); font-size: 0.9375rem;">
                                <i class="fas fa-comment-alt" style="margin-right: 0.5rem; color: var(--primary);"></i>Nachricht
                            </label>
                            <textarea id="p2EmailMessage" class="form-control" rows="5" 
                                      placeholder="Ihre zusätzliche Nachricht an den Kunden..."
                                      style="width: 100%; padding: 0.875rem 1rem; border: 2px solid var(--border); border-radius: var(--radius); resize: vertical; font-size: 0.9375rem; transition: all 0.2s ease; background: var(--bg-panel); color: var(--text-main); font-family: inherit; line-height: 1.6;" 
                                      onfocus="this.style.borderColor='var(--primary)'; this.style.boxShadow='0 0 0 3px rgba(59, 130, 246, 0.1)'" 
                                      onblur="this.style.borderColor='var(--border)'; this.style.boxShadow='none'">Sehr geehrte Damen und Herren,

anbei erhalten Sie Ihr Reifenangebot für den kommenden Boxenstop.

Bei Fragen stehen wir Ihnen gerne zur Verfügung.

Mit freundlichen Grüßen
Ihr B&E Autocentrum Team</textarea>
                        </div>
                    </div>
                    <div class="modal-footer" style="padding: 1.5rem 2rem; border-top: 1px solid var(--border); background: var(--bg-hover); display: flex; justify-content: flex-end; gap: 0.75rem;">
                        <button class="btn btn-secondary" onclick="closeP2EmailModal()" style="padding: 0.75rem 1.75rem; border: 2px solid var(--border); border-radius: var(--radius); cursor: pointer; background: var(--bg-panel); color: var(--text-main); font-weight: 500; font-size: 0.9375rem; transition: all 0.2s ease;" onmouseover="this.style.background='var(--bg-hover)'; this.style.borderColor='var(--text-secondary)'" onmouseout="this.style.background='var(--bg-panel)'; this.style.borderColor='var(--border)'">
                            <i class="fas fa-times" style="margin-right: 0.5rem;"></i>Abbrechen
                        </button>
                        <button class="btn btn-primary" onclick="sendP2EmailWithModal()" style="padding: 0.75rem 1.75rem; border: none; border-radius: var(--radius); cursor: pointer; background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%); color: white; font-weight: 600; font-size: 0.9375rem; box-shadow: 0 4px 6px -1px rgba(59, 130, 246, 0.3); transition: all 0.2s ease;" onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 6px 12px -2px rgba(59, 130, 246, 0.4)'" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 6px -1px rgba(59, 130, 246, 0.3)'">
                            <i class="fas fa-paper-plane" style="margin-right: 0.5rem;"></i>E-Mail senden
                        </button>
                    </div>
                </div>
            `;
            
            modal.onclick = function(e) {
                if (e.target === modal) {
                    closeP2EmailModal();
                }
            };
            
            document.body.appendChild(modal);
        }
        
        function closeP2EmailModal() {
            const modal = document.querySelector('.p2-email-modal');
            if (modal) {
                modal.remove();
            }
        }
        
        async function sendP2EmailWithModal() {
            const recipientEmail = document.getElementById('p2RecipientEmail').value;
            const emailSubject = document.getElementById('p2EmailSubject').value;
            const emailMessage = document.getElementById('p2EmailMessage').value;
            
            // Validierung
            if (!recipientEmail) {
                alert('Bitte geben Sie eine E-Mail-Adresse ein.');
                return;
            }
            
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(recipientEmail)) {
                alert('Bitte geben Sie eine gültige E-Mail-Adresse ein.');
                return;
            }
            
            if (!emailSubject) {
                alert('Bitte geben Sie einen Betreff ein.');
                return;
            }
            
            if (!window.currentP2OfferId) {
                alert('Fehler: Keine Angebots-ID gefunden');
                return;
            }
            
            try {
                // Zeige Loading
                const sendButton = document.querySelector('.p2-email-modal .btn-primary');
                const originalText = sendButton.innerHTML;
                sendButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Wird gesendet...';
                sendButton.disabled = true;
                
                const response = await fetch('../api/p2_tire_offers.php?action=send_email', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        offer_id: window.currentP2OfferId,
                        recipient_email: recipientEmail,
                        email_subject: emailSubject,
                        email_message: emailMessage
                    })
                });
                
                const text = await response.text();
                let result;
                try {
                    result = JSON.parse(text);
                } catch (e) {
                    result = { success: false, error: 'Ungültige Server-Antwort' };
                }
                
                if (result.success) {
                    showToast('success', 'Erfolg', 'E-Mail erfolgreich versendet an: ' + recipientEmail);
                    closeP2EmailModal();
                    // Aktualisiere Kundendetails
                    if (selectedCustomer) {
                        loadCustomerDetails(selectedCustomer.KundenAdrID);
                    }
                } else {
                    alert('Fehler beim E-Mail-Versand: ' + (result.error || 'Unbekannter Fehler'));
                }
            } catch (error) {
                alert('Fehler beim E-Mail-Versand: ' + error.message);
            } finally {
                const sendButton = document.querySelector('.p2-email-modal .btn-primary');
                if (sendButton) {
                    sendButton.innerHTML = '<i class="fas fa-paper-plane"></i> E-Mail senden';
                    sendButton.disabled = false;
                }
            }
        }

        // ============ ANGEBOTS-DETAILS ============
        window.viewP2OfferDetails = function(offerId) {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId || autohausId === '') {
                showToast('warning', 'Autohaus erforderlich', 'Bitte wählen Sie zuerst ein Autohaus aus');
                return;
            }
            
            // Zeige Loading (falls showLoading nicht existiert, verwende einfache Variante)
            if (typeof showLoading === 'function') {
                showLoading(true);
            } else {
                const overlay = $('#loadingOverlay');
                if (overlay.length) {
                    overlay.addClass('active');
                }
            }
            
            fetch(`../api/p2_tire_offers.php?action=get&offer_id=${offerId}&autohaus_id=${autohausId}`)
                .then(async response => {
                    const text = await response.text();
                    try {
                        return JSON.parse(text);
                    } catch (e) {
                        throw new Error('Ungültige Server-Antwort');
                    }
                })
                .then(data => {
                    // Verstecke Loading
                    if (typeof showLoading === 'function') {
                        showLoading(false);
                    } else {
                        const overlay = $('#loadingOverlay');
                        if (overlay.length) {
                            overlay.removeClass('active');
                        }
                    }
                    
                    if (!data.success || !data.offer) {
                        showToast('error', 'Fehler', data.error || 'Angebot nicht gefunden');
                        return;
                    }
                    
                    const offer = data.offer;
                    const offerData = offer.offer_data || {};
                    const newTires = offerData.new_tires || [];
                    const selectedServices = offerData.selected_services || [];
                    const customerSelection = offer.customer_selection || {};
                    const selectedTireIndices = customerSelection.tires || [];
                    
                    const createdDate = new Date(offer.created_at).toLocaleDateString('de-DE', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                    });
                    
                    const statusConfig = {
                        'draft': { color: '#6c757d', label: 'Entwurf' },
                        'sent': { color: '#007bff', label: 'Gesendet' },
                        'viewed': { color: '#ffc107', label: 'Angesehen' },
                        'accepted': { color: '#28a745', label: 'Angenommen' },
                        'declined': { color: '#dc3545', label: 'Abgelehnt' },
                        'new_request': { color: '#17a2b8', label: 'Neu angefragt' }
                    }[offer.status] || { color: '#6c757d', label: offer.status };
                    
                    // Helper: Extrahiere Marke aus brand_model
                    const getBrandModel = (tire) => {
                        if (tire.brand_model) {
                            return tire.brand_model;
                        }
                        if (tire.brand && tire.model) {
                            return `${tire.brand} ${tire.model}`.trim();
                        }
                        if (tire.brand) {
                            return tire.brand;
                        }
                        if (tire.model) {
                            return tire.model;
                        }
                        return '-';
                    };
                    
                    let modalContent = `
                        <div style="max-width: 100%; margin: 0 auto;">
                            <div style="background: white; padding: 1.5rem; border-radius: 8px; margin-bottom: 1rem;">
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
                                    <h3 style="margin: 0;">Angebot Details</h3>
                                    <span class="badge" style="background: ${statusConfig.color}; color: white; padding: 0.5rem 1rem; border-radius: 4px;">${statusConfig.label}</span>
                                </div>
                                
                                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem;">
                                    <div>
                                        <strong>Kunde:</strong><br>
                                        ${offer.customer_name || '-'} ${offer.customer_vorname || ''}
                                    </div>
                                    <div>
                                        <strong>Kennzeichen:</strong><br>
                                        ${offer.license_plate || '-'}
                                    </div>
                                    <div>
                                        <strong>Erstellt am:</strong><br>
                                        ${createdDate}
                                    </div>
                                    <div>
                                        <strong>Gesamtbetrag:</strong><br>
                                        ${formatCurrency(offer.total_amount || 0)}
                                    </div>
                                </div>
                                
                                ${newTires.length > 0 ? `
                                <div style="margin-top: 1.5rem;">
                                    <h4 style="margin-bottom: 0.5rem;">Reifen:</h4>
                                    <div style="display: grid; gap: 0.75rem;">
                                        ${newTires.map((tire, index) => {
                                            const isSelected = selectedTireIndices.includes(index.toString()) || selectedTireIndices.includes(index);
                                            const borderColor = isSelected ? '#28a745' : (customerSelection.action === 'accepted' ? '#dc3545' : 'var(--border)');
                                            const bgColor = isSelected ? 'rgba(40, 167, 69, 0.1)' : (customerSelection.action === 'accepted' ? 'rgba(220, 53, 69, 0.1)' : 'var(--bg-hover)');
                                            const brandModel = getBrandModel(tire);
                                            
                                            // Escape tire data für JSON in onclick
                                            const tireDataEscaped = JSON.stringify(tire).replace(/'/g, "\\'").replace(/"/g, '&quot;');
                                            
                                            return `
                                                <div style="padding: 1rem; border: 2px solid ${borderColor}; border-radius: var(--radius); background: ${bgColor}; cursor: pointer; transition: all 0.2s;" 
                                                     onclick="showP2TireDetailsModal(${index}, '${tireDataEscaped}')"
                                                     onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 12px rgba(0,0,0,0.1)';"
                                                     onmouseout="this.style.transform='none'; this.style.boxShadow='none';">
                                                    <div style="display: flex; justify-content: space-between; align-items: start;">
                                                        <div style="flex: 1;">
                                                            <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                                                                <div style="font-weight: 700; font-size: 1.0625rem; color: var(--text-main);">${brandModel}</div>
                                                                ${isSelected ? '<span class="badge badge-success" style="font-size: 0.75rem;"><i class="fas fa-check"></i> Ausgewählt</span>' : ''}
                                                                ${customerSelection.action === 'accepted' && !isSelected ? '<span class="badge badge-danger" style="font-size: 0.75rem;"><i class="fas fa-times"></i> Nicht ausgewählt</span>' : ''}
                                                            </div>
                                                            <div style="display: flex; gap: 1rem; flex-wrap: wrap; font-size: 0.875rem; color: var(--text-secondary);">
                                                                ${tire.dimension ? `<span><i class="fas fa-ruler"></i> ${tire.dimension}</span>` : ''}
                                                                ${tire.eu_label ? `<span><i class="fas fa-tag"></i> ${tire.eu_label}</span>` : ''}
                                                                ${tire.rf ? `<span><i class="fas fa-info-circle"></i> RF: ${tire.rf}</span>` : ''}
                                                                <span><i class="fas fa-box"></i> ${tire.quantity || 1}x</span>
                                                            </div>
                                                        </div>
                                                        <div style="text-align: right; margin-left: 1rem;">
                                                            <div style="font-size: 0.875rem; color: var(--text-secondary); margin-bottom: 0.25rem;">Einzelpreis</div>
                                                            <div style="font-weight: 700; font-size: 1.25rem; color: var(--primary);">${formatCurrency(tire.price || 0)}</div>
                                                            <div style="font-size: 0.875rem; color: var(--text-secondary); margin-top: 0.25rem;">Gesamt: ${formatCurrency((tire.price || 0) * (tire.quantity || 1))}</div>
                                                        </div>
                                                    </div>
                                                    <div style="margin-top: 0.75rem; padding-top: 0.75rem; border-top: 1px solid ${borderColor}; display: flex; justify-content: flex-end;">
                                                        <button class="btn btn-sm btn-primary" style="font-size: 0.875rem; padding: 0.375rem 0.75rem;" onclick="event.stopPropagation(); showP2TireDetailsModal(${index}, '${tireDataEscaped}')">
                                                            <i class="fas fa-info-circle"></i> Details anzeigen
                                                        </button>
                                                    </div>
                                                </div>
                                            `;
                                        }).join('')}
                                    </div>
                                </div>
                                ` : ''}
                                
                                ${selectedServices.length > 0 ? `
                                <div style="margin-top: 1.5rem;">
                                    <h4 style="margin-bottom: 0.5rem;">Zusatzleistungen:</h4>
                                    <ul>
                                        ${selectedServices.map(service => `
                                            <li>${service.name || service.label || '-'} - ${formatCurrency(service.price || 0)}</li>
                                        `).join('')}
                                    </ul>
                                </div>
                                ` : ''}
                                
                                ${offer.letter_content ? `
                                <div style="margin-top: 1.5rem; padding: 1rem; background: #f8f9fa; border-radius: 4px;">
                                    <h4 style="margin-bottom: 0.5rem;">Anschreiben:</h4>
                                    <div style="white-space: pre-wrap;">${offer.letter_content}</div>
                                </div>
                                ` : ''}
                                
                                ${offer.email_sent ? `
                                <div style="background: #d1fae5; padding: 1rem; border-radius: 8px; margin-top: 1rem; border-left: 4px solid #10b981;">
                                    <strong style="color: #065f46;"><i class="fas fa-envelope"></i> E-Mail versendet</strong>
                                    ${offer.email_sent_at ? `<div style="font-size: 0.875rem; color: #065f46; margin-top: 0.25rem;">am ${new Date(offer.email_sent_at).toLocaleDateString('de-DE', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</div>` : ''}
                                </div>
                                ` : ''}
                                
                                <div style="display: flex; gap: 0.5rem; justify-content: flex-end; margin-top: 1.5rem;">
                                    ${offer.status === 'draft' || offer.status === 'sent' ? `
                                    <button class="btn btn-primary" onclick="resendP2Offer(${offer.id})">
                                        <i class="fas fa-envelope"></i> Erneut versenden
                                    </button>
                                    ` : ''}
                                    <button class="btn btn-secondary" onclick="closeModal('p2OfferDetailsModal')">
                                        <i class="fas fa-times"></i> Schließen
                                    </button>
                                </div>
                            </div>
                        </div>
                    `;
                    
                    createModal('p2OfferDetailsModal', 'Angebot Details', modalContent, false, null, null, 'extra-large');
                })
                .catch(error => {
                    // Verstecke Loading
                    if (typeof showLoading === 'function') {
                        showLoading(false);
                    } else {
                        const overlay = $('#loadingOverlay');
                        if (overlay.length) {
                            overlay.removeClass('active');
                        }
                    }
                    showToast('error', 'Fehler', 'Fehler: ' + error.message);
                });
        }
        
        function resendP2Offer(offerId) {
            window.currentP2OfferId = offerId;
            closeModal('p2OfferDetailsModal');
            showP2EmailModal();
        }
        
        // ============ REIFEN-DETAILS MODAL (wie storage.php) ============
        function showP2TireDetailsModal(tireIndex, tireData) {
            // Wenn tireData ein String ist, parse es
            let tire = tireData;
            if (typeof tireData === 'string') {
                try {
                    // Ersetze &quot; zurück zu " und parse JSON
                    tire = JSON.parse(tireData.replace(/&quot;/g, '"').replace(/\\'/g, "'"));
                } catch (e) {
                    tire = {};
                }
            }
            
            // Extrahiere Marke/Modell
            const brandModel = tire.brand_model || (tire.brand && tire.model ? `${tire.brand} ${tire.model}`.trim() : tire.brand || tire.model || 'Unbekannt');
            const manufacturer = tire.brand || brandModel.split(' ')[0] || 'Unbekannt';
            const model = tire.model || brandModel.replace(manufacturer, '').trim() || '';
            const dimension = tire.dimension || '-';
            const brandImageUrl = `../assets/images/brands/${manufacturer.toLowerCase().replace(/\s+/g, '-')}.png`;
            
            // Formatierung für Preise
            const formatPrice = (value) => {
                if (!value || value === 0) return '-';
                return parseFloat(value).toFixed(2).replace('.', ',') + ' €';
            };
            
            const modalContent = `
                <div style="max-height: 85vh; overflow-y: auto; max-width: 100%;">
                    <!-- Header mit Markenbild und Basis-Info -->
                    <div style="display: flex; align-items: center; gap: 1rem; padding: 1rem; background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%); border-radius: var(--radius-lg); margin-bottom: 1rem; border: 2px solid var(--primary);">
                        <div style="width: 80px; height: 80px; border-radius: 10px; background: white; display: flex; align-items: center; justify-content: center; border: 2px solid var(--border); flex-shrink: 0; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
                            <img src="${brandImageUrl}" 
                                 alt="${manufacturer}" 
                                 style="max-width: 70px; max-height: 70px; object-fit: contain;"
                                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                            <div style="display: none; width: 100%; height: 100%; align-items: center; justify-content: center; font-size: 1.25rem; color: var(--text-secondary); font-weight: 700;">
                                ${manufacturer.substring(0, 3).toUpperCase()}
                            </div>
                        </div>
                        <div style="flex: 1;">
                            <h3 style="margin: 0 0 0.25rem 0; font-size: 1.25rem; font-weight: 700; color: var(--text-primary);">
                                ${brandModel}
                            </h3>
                            <div style="font-size: 1rem; color: var(--text-secondary); font-weight: 600; margin-bottom: 0.5rem;">
                                ${dimension}
                            </div>
                            <div style="display: flex; gap: 1rem; flex-wrap: wrap; align-items: center;">
                                ${tire.eu_label ? `
                                <div style="display: flex; align-items: center; gap: 0.5rem; padding: 0.5rem 1rem; background: var(--bg-hover); border-radius: 6px; font-weight: 600; font-size: 0.9375rem;">
                                    <span>EU-Label: ${tire.eu_label}</span>
                                </div>
                                ` : ''}
                                ${dimension ? `
                                <div style="display: flex; align-items: center; gap: 0.5rem; padding: 0.5rem 1rem; background: var(--bg-hover); border-radius: 6px; font-weight: 600; font-size: 0.9375rem;">
                                    <span>Dimension: ${dimension}</span>
                                </div>
                                ` : ''}
                                ${tire.price ? `<span style="padding: 0.5rem 1rem; background: var(--primary); color: white; border-radius: 6px; font-weight: 700; font-size: 1.125rem;">${formatPrice(tire.price)}</span>` : ''}
                            </div>
                        </div>
                    </div>
                    
                    <!-- Hauptinformationen in Grid-Layout -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
                        <!-- Basis-Informationen -->
                        <div style="background: var(--bg-secondary); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-primary); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-info-circle" style="color: var(--primary); font-size: 0.875rem;"></i>
                                Basis-Informationen
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Marke/Modell:</span>
                                    <span style="font-weight: 600; color: var(--text-primary);">${brandModel}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Dimension:</span>
                                    <span style="font-weight: 600; color: var(--text-primary);">${dimension}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">EU-Label:</span>
                                    <span style="font-weight: 600; color: var(--text-primary);">${tire.eu_label || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">RF:</span>
                                    <span style="font-weight: 600; color: var(--text-primary);">${tire.rf || tire.runflat || '-'}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between;">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Anzahl:</span>
                                    <span style="font-weight: 600; color: var(--text-primary);">${tire.quantity || 1}</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Preisinformationen -->
                        <div style="background: var(--bg-secondary); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-primary); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-euro-sign" style="color: var(--primary); font-size: 0.875rem;"></i>
                                Preisinformationen
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Einzelpreis:</span>
                                    <span style="font-weight: 700; color: var(--primary); font-size: 1.125rem;">${formatPrice(tire.price)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between;">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Gesamtpreis:</span>
                                    <span style="font-weight: 700; color: var(--primary); font-size: 1.125rem;">${formatPrice((tire.price || 0) * (tire.quantity || 1))}</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Technische Details -->
                        ${(tire.ean || tire.oem_part_number || tire.part_number || tire.manufacturer_number) ? `
                        <div style="background: var(--bg-secondary); padding: 1rem; border-radius: var(--radius-lg); border: 1px solid var(--border);">
                            <h4 style="margin: 0 0 0.75rem 0; font-size: 0.9375rem; font-weight: 600; color: var(--text-primary); display: flex; align-items: center; gap: 0.5rem;">
                                <i class="fas fa-cog" style="color: var(--primary); font-size: 0.875rem;"></i>
                                Technische Details
                            </h4>
                            <div style="display: grid; gap: 0.5rem;">
                                ${tire.ean ? `
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">EAN:</span>
                                    <span style="font-weight: 600; color: var(--text-primary); font-family: monospace;">${tire.ean}</span>
                                </div>
                                ` : ''}
                                ${tire.oem_part_number ? `
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">OEM Teilenummer:</span>
                                    <span style="font-weight: 600; color: var(--text-primary); font-family: monospace;">${tire.oem_part_number}</span>
                                </div>
                                ` : ''}
                                ${tire.part_number ? `
                                <div style="display: flex; justify-content: space-between; padding-bottom: 0.5rem; border-bottom: 1px solid var(--border);">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Teilenummer:</span>
                                    <span style="font-weight: 600; color: var(--text-primary); font-family: monospace;">${tire.part_number}</span>
                                </div>
                                ` : ''}
                                ${tire.manufacturer_number ? `
                                <div style="display: flex; justify-content: space-between;">
                                    <span style="color: var(--text-secondary); font-size: 0.875rem;">Hersteller Artikel-Nr:</span>
                                    <span style="font-weight: 600; color: var(--text-primary); font-family: monospace;">${tire.manufacturer_number}</span>
                                </div>
                                ` : ''}
                            </div>
                        </div>
                        ` : ''}
                    </div>
                    
                    <!-- Aktions-Buttons -->
                    <div style="display: flex; gap: 1rem; justify-content: flex-end; padding-top: 1rem; border-top: 2px solid var(--border); margin-top: 1rem;">
                        <button type="button" 
                                class="btn btn-secondary" 
                                onclick="closeModal('p2TireDetailsModal')"
                                style="white-space: nowrap;">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                    </div>
                </div>
            `;
            
            createModal('p2TireDetailsModal', `Reifendetails - ${brandModel}`, modalContent, false, null, null, 'extra-large');
        }

        // ============ ANGEBOTS-PANEL (P2 + RESY) ============
        let currentOffersNotificationsTab = 'all';
        let offersNotificationsInterval = null;
        
        window.showOffersNotificationsPanel = function() {
            $('#offersNotificationsPanel').addClass('open');
            switchOffersNotificationsTab('all');
            
            // Starte Auto-Reload alle 3 Minuten
            if (offersNotificationsInterval) {
                clearInterval(offersNotificationsInterval);
            }
            offersNotificationsInterval = setInterval(() => {
                if ($('#offersNotificationsPanel').hasClass('open')) {
                    loadOffersNotifications(currentOffersNotificationsTab);
                }
            }, 180000);
        };
        
        function closeOffersNotificationsPanel() {
            $('#offersNotificationsPanel').removeClass('open');
            if (offersNotificationsInterval) {
                clearInterval(offersNotificationsInterval);
                offersNotificationsInterval = null;
            }
        }
        
        function switchOffersNotificationsTab(tab) {
            currentOffersNotificationsTab = tab;
            
            $('.orders-sidebar-item', '#offersNotificationsPanel').removeClass('active');
            $(`.orders-sidebar-item[data-tab="${tab}"]`, '#offersNotificationsPanel').addClass('active');
            
            loadOffersNotifications(tab);
        }
        
        function loadOffersNotifications(statusFilter = 'all') {
            const autohausSelect = $('#autohausSelect');
            let autohausId = autohausSelect.val();
            
            // Prüfe auch selectedIndex für Fallback
            if (!autohausId || autohausId === '' || autohausId === null || autohausId === undefined) {
                const selectedIndex = autohausSelect[0]?.selectedIndex;
                if (selectedIndex && selectedIndex > 0) {
                    autohausId = autohausSelect.find('option').eq(selectedIndex).val();
                }
            }
            
            // Debug: Logge den Wert
            
            if (!autohausId || autohausId === '' || autohausId === null || autohausId === undefined) {
                $('#offersNotificationsBody').html('<div style="padding: 2rem; text-align: center; color: var(--text-secondary);">Bitte wählen Sie ein Autohaus aus</div>');
                return;
            }
            
            // Zeige Loading
            showLoading(true);
            
            // Lade sowohl P2- als auch RESY-Angebote
            Promise.all([
                fetch(`../api/p2_tire_offers.php?action=get_all&autohaus_id=${autohausId}`).then(async r => {
                    const text = await r.text();
                    try {
                        return JSON.parse(text);
                    } catch (e) {
                        return { success: false, offers: [], error: 'JSON Parse Error' };
                    }
                }).catch(err => {
                    return { success: false, offers: [], error: err.message };
                }),
                fetch(`../api/wheelset_offers.php?action=get_all&autohaus_id=${autohausId}`).then(async r => {
                    const text = await r.text();
                    try {
                        return JSON.parse(text);
                    } catch (e) {
                        return { success: false, offers: [], error: 'JSON Parse Error' };
                    }
                }).catch(err => {
                    return { success: false, offers: [], error: err.message };
                })
            ])
            .then(([p2Data, resyData]) => {
                // Verstecke Loading
                showLoading(false);
                
                
                let allOffers = [];
                
                // P2-Angebote hinzufügen (mit Markierung)
                if (p2Data.success && p2Data.offers && Array.isArray(p2Data.offers)) {
                    allOffers = allOffers.concat(p2Data.offers.map(offer => ({
                        ...offer,
                        source: 'P2',
                        sourceLabel: 'P2 Kundenverwaltung'
                    })));
                } else {
                }
                
                // RESY-Angebote hinzufügen (mit Markierung)
                if (resyData.success && resyData.offers && Array.isArray(resyData.offers)) {
                    allOffers = allOffers.concat(resyData.offers.map(offer => ({
                        ...offer,
                        source: 'RESY',
                        sourceLabel: 'RESY Reifenlager'
                    })));
                } else {
                }
                
                
                // Filtere nach Status
                let filteredOffers = allOffers;
                if (statusFilter === 'pending') {
                    filteredOffers = allOffers.filter(o => o.status === 'sent' || o.status === 'viewed');
                } else if (statusFilter === 'accepted') {
                    filteredOffers = allOffers.filter(o => o.status === 'accepted');
                } else if (statusFilter === 'declined') {
                    filteredOffers = allOffers.filter(o => o.status === 'declined');
                } else if (statusFilter === 'new_request') {
                    filteredOffers = allOffers.filter(o => o.status === 'new_request');
                }
                
                // Sortiere nach Datum (neueste zuerst)
                filteredOffers.sort((a, b) => new Date(b.updated_at || b.created_at) - new Date(a.updated_at || a.created_at));
                
                // Update Badges
                updateOffersNotificationsBadges(allOffers);
                updateOffersLastUpdateTime();
                
                if (filteredOffers.length === 0) {
                    $('#offersNotificationsBody').html('<div style="padding: 2rem; text-align: center; color: var(--text-secondary);">Keine Angebote für diesen Status gefunden</div>');
                    return;
                }
                
                // Erstelle HTML
                let html = '';
                filteredOffers.forEach(offer => {
                    const statusConfig = {
                        'accepted': { color: '#28a745', icon: 'fa-check-circle', label: 'Angenommen' },
                        'declined': { color: '#dc3545', icon: 'fa-times-circle', label: 'Abgelehnt' },
                        'new_request': { color: '#17a2b8', icon: 'fa-redo', label: 'Neu angefragt' },
                        'sent': { color: '#007bff', icon: 'fa-envelope', label: 'Gesendet' },
                        'viewed': { color: '#ffc107', icon: 'fa-eye', label: 'Angesehen' },
                        'draft': { color: '#6c757d', icon: 'fa-file', label: 'Entwurf' }
                    }[offer.status] || { color: '#6c757d', icon: 'fa-question', label: 'Unbekannt' };
                    
                    const updatedDate = new Date(offer.updated_at || offer.created_at).toLocaleDateString('de-DE', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                    });
                    
                    const identifier = offer.source === 'RESY' ? (offer.wheelset_number || '-') : (offer.license_plate || '-');
                    const customerName = offer.customer_name || offer.customerName || '-';
                    
                    html += `
                        <div style="padding: 1rem; border: 1px solid var(--border); border-radius: var(--radius); margin-bottom: 0.75rem; background: white; cursor: pointer; transition: all 0.2s; ${offer.source === 'RESY' ? 'border-left: 4px solid #10b981;' : 'border-left: 4px solid #3b82f6;'}" 
                             onclick="openOfferFromNotification(${offer.id}, '${offer.source}')" 
                             onmouseover="this.style.background='var(--bg-hover)'; this.style.borderColor='${statusConfig.color}';" 
                             onmouseout="this.style.background='white'; this.style.borderColor='var(--border)';">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 0.5rem;">
                                <div style="flex: 1;">
                                    <div style="font-weight: 600; color: var(--text-primary); margin-bottom: 0.25rem;">
                                        <i class="fas ${statusConfig.icon}" style="color: ${statusConfig.color}; margin-right: 0.5rem;"></i>
                                        ${identifier} - ${statusConfig.label}
                                        <span class="badge" style="background: ${offer.source === 'RESY' ? '#10b981' : '#3b82f6'}; color: white; padding: 0.125rem 0.5rem; border-radius: 4px; font-size: 0.75rem; margin-left: 0.5rem;">${offer.sourceLabel || offer.source}</span>
                                    </div>
                                    <div style="font-size: 0.875rem; color: var(--text-secondary);">
                                        ${customerName} - ${identifier}
                                    </div>
                                </div>
                                <div style="text-align: right;">
                                    <div style="font-size: 0.75rem; color: var(--text-secondary);">${updatedDate}</div>
                                    ${offer.total_amount ? `<div style="font-weight: 600; color: var(--primary); margin-top: 0.25rem;">${formatCurrency(offer.total_amount)}</div>` : ''}
                                </div>
                            </div>
                            <div style="margin-top: 0.75rem; display: flex; gap: 0.5rem;">
                                <button type="button" class="btn btn-sm btn-primary" onclick="event.stopPropagation(); openOfferFromNotification(${offer.id}, '${offer.source}')" style="font-size: 0.75rem;">
                                    <i class="fas fa-eye"></i> Öffnen
                                </button>
                            </div>
                        </div>
                    `;
                });
                
                $('#offersNotificationsBody').html(html);
            })
            .catch(error => {
                // Verstecke Loading
                if (typeof showLoading === 'function') {
                    showLoading(false);
                } else {
                    const overlay = $('#loadingOverlay');
                    if (overlay.length) {
                        overlay.removeClass('active');
                    }
                }
                $('#offersNotificationsBody').html('<div style="padding: 2rem; text-align: center; color: var(--danger);">Fehler beim Laden der Angebote</div>');
            });
        }
        
        function updateOffersNotificationsBadges(offers) {
            const accepted = offers.filter(o => o.status === 'accepted').length;
            const declined = offers.filter(o => o.status === 'declined').length;
            const newRequest = offers.filter(o => o.status === 'new_request').length;
            const pending = offers.filter(o => o.status === 'sent' || o.status === 'viewed').length;
            const all = offers.filter(o => ['accepted', 'declined', 'new_request', 'sent', 'viewed', 'draft'].includes(o.status)).length;
            
            // Update Badges
            $('#acceptedOffersBadge').text(accepted > 99 ? '99+' : accepted).toggle(accepted > 0);
            $('#declinedOffersBadge').text(declined > 99 ? '99+' : declined).toggle(declined > 0);
            $('#newRequestOffersBadge').text(newRequest > 99 ? '99+' : newRequest).toggle(newRequest > 0);
            $('#pendingOffersBadge').text(pending > 99 ? '99+' : pending).toggle(pending > 0);
            $('#offersTabAllBadge').text(all > 99 ? '99+' : all).toggle(all > 0);
            
            // Haupt-Badge
            $('#offersNotificationsBadge').text(all > 99 ? '99+' : all).toggle(all > 0);
        }
        
        function updateOffersLastUpdateTime() {
            const now = new Date();
            const timeString = now.toLocaleTimeString('de-DE', {
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
            $('#offersLastUpdateTime').html(`<i class="fas fa-sync"></i> Zuletzt aktualisiert: ${timeString}`);
        }
        
        function openOfferFromNotification(offerId, source) {
            closeOffersNotificationsPanel();
            
            if (source === 'RESY') {
                // RESY-Angebot - hole zuerst wheelset_number und öffne dann storage.php mit beiden Parametern
                const autohausId = $('#autohausSelect').val();
                
                // Hole Angebotsdetails um wheelset_number zu bekommen
                fetch(`../api/wheelset_offers.php?action=get_by_id&offer_id=${offerId}&autohaus_id=${autohausId}`)
                    .then(r => r.json())
                    .then(data => {
                        if (data.success && data.offer) {
                            const wheelsetNumber = data.offer.wheelset_number;
                            if (wheelsetNumber) {
                                // Öffne storage.php mit wheelset_number und offer_id
                                window.open(`storage.php?wheelset=${wheelsetNumber}&offer_id=${offerId}`, '_blank');
                            } else {
                                // Fallback: nur mit offer_id
                                window.open(`storage.php?offer_id=${offerId}`, '_blank');
                            }
                        } else {
                            // Fallback: nur mit offer_id
                            window.open(`storage.php?offer_id=${offerId}`, '_blank');
                        }
                    })
                    .catch(error => {
                        // Fallback: nur mit offer_id
                        window.open(`storage.php?offer_id=${offerId}`, '_blank');
                    });
            } else {
                // P2-Angebot - öffne Details-Modal
                window.viewP2OfferDetails(offerId);
            }
        }
        
        // Lade Badge beim Seitenaufruf
        $(document).ready(function() {
            setTimeout(() => {
                loadOffersNotificationsBadge();
            }, 1000);
        });
        
        function loadOffersNotificationsBadge() {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                $('#offersNotificationsBadge').hide();
                return;
            }
            
            Promise.all([
                fetch(`../api/p2_tire_offers.php?action=get_all&autohaus_id=${autohausId}`).then(r => r.json()).catch(() => ({ success: false, offers: [] })),
                fetch(`../api/wheelset_offers.php?action=get_all&autohaus_id=${autohausId}`).then(r => r.json()).catch(() => ({ success: false, offers: [] }))
            ])
            .then(([p2Data, resyData]) => {
                let allOffers = [];
                if (p2Data.success && p2Data.offers) {
                    allOffers = allOffers.concat(p2Data.offers);
                }
                if (resyData.success && resyData.offers) {
                    allOffers = allOffers.concat(resyData.offers);
                }
                updateOffersNotificationsBadges(allOffers);
            })
            .catch(error => {
            });
        }
        
        // Lade Badge wenn Autohaus gewechselt wird
        $(document).on('change', '#autohausSelect', function() {
            loadOffersNotificationsBadge();
        });

        // ============ LOADING FUNKTION ============
        function showLoading(show, text = null, subtext = null) {
            const overlay = $('#loadingOverlay');
            const loadingText = $('#loadingText');
            const loadingSubtext = $('#loadingSubtext');
            
            // Stelle sicher, dass z-index immer der höchste ist
            overlay.css('z-index', '9999999');
            
            if (show) {
                if (text) loadingText.text(text);
                else loadingText.text('Wird geladen...');
                
                if (subtext) loadingSubtext.text(subtext);
                else loadingSubtext.text('Bitte warten Sie einen Moment');
                
                // Setze z-index explizit beim Anzeigen
                overlay.css('z-index', '9999999').addClass('active');
            } else {
                overlay.removeClass('active');
                // Reset nach Animation
                setTimeout(() => {
                    loadingText.text('Wird geladen...');
                    loadingSubtext.text('Bitte warten Sie einen Moment');
                }, 300);
            }
        }
    </script>
</body>
</html>

