<?php
/**
 * Admin Benachrichtigungen Management
 * Modulares System zum Verwalten, Erstellen und Versenden von Benachrichtigungen
 */

session_start();
require_once '../config/database.php';
require_once '../includes/push_notification_helper.php';
require_once 'includes/helpers.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator';
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$current_user_autohaus_name = null;

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.role, u.autohaus_id, u.username, a.name as autohaus_name
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_role = $user_data['role'] ?? 'admin';
    $current_user_autohaus_id = $user_data['autohaus_id'];
    $current_user_autohaus_name = $user_data['autohaus_name'];
    $current_user_name = $user_data['username'] ?? 'Administrator';
}

// Get user theme preference
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Handle actions
$action = $_GET['action'] ?? 'list';
$message = '';
$message_type = '';

// Hole Nachricht aus Session (POST-Redirect-GET Pattern)
if (isset($_SESSION['notification_message'])) {
    $message = $_SESSION['notification_message'];
    $message_type = $_SESSION['notification_message_type'] ?? 'info';
    unset($_SESSION['notification_message']);
    unset($_SESSION['notification_message_type']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['send_notification'])) {
        // Neue Benachrichtigung senden
        $user_ids = $_POST['user_ids'] ?? [];
        // Konvertiere user_ids String zu Array falls nötig
        if (is_string($user_ids)) {
            $user_ids = explode(',', $user_ids);
            $user_ids = array_filter(array_map('trim', $user_ids));
        }
        // Stelle sicher, dass es ein Array ist
        if (!is_array($user_ids)) {
            $user_ids = [];
        }
        
        $title = $_POST['title'] ?? '';
        $message_text = $_POST['message'] ?? '';
        $type = $_POST['type'] ?? 'info';
        $url = $_POST['url'] ?? '/profile.php';
        $send_push = isset($_POST['send_push']) && $_POST['send_push'] == '1';
        
        if (empty($user_ids) || empty($title) || empty($message_text)) {
            $_SESSION['notification_message'] = 'Bitte füllen Sie alle Felder aus.';
            $_SESSION['notification_message_type'] = 'error';
        } else {
            $success_count = 0;
            $error_count = 0;
            $notification_logs = [];
            
            // Prüfe ob notification_logs Tabelle existiert
            $checkTable = $conn->query("SHOW TABLES LIKE 'notification_logs'");
            $hasLogTable = $checkTable->rowCount() > 0;
            
            foreach ($user_ids as $user_id) {
                try {
                    if ($send_push) {
                        $result = createNotificationWithPush($conn, $user_id, $title, $message_text, $type, $url);
                    } else {
                        // Nur Datenbank-Benachrichtigung ohne Push
                        $stmt = $conn->prepare("
                            INSERT INTO user_notifications (user_id, type, title, message, is_read, created_at)
                            VALUES (?, ?, ?, ?, 0, NOW())
                        ");
                        $stmt->execute([$user_id, $type, $title, $message_text]);
                        $notification_id = $conn->lastInsertId();
                        $result = ['success' => true, 'notification_id' => $notification_id, 'push_sent' => false];
                    }
                    
                    if ($result['success']) {
                        $success_count++;
                        
                        // Log erstellen
                        if ($hasLogTable && isset($result['notification_id'])) {
                            // Hole Geräte-Informationen
                            $stmt = $conn->prepare("
                                SELECT id, endpoint, user_agent, device_info, is_active
                                FROM push_subscriptions
                                WHERE user_id = ? AND is_active = 1
                            ");
                            $stmt->execute([$user_id]);
                            $devices = $stmt->fetchAll(PDO::FETCH_ASSOC);
                            
                            $push_success = 0;
                            $push_failed = 0;
                            if (isset($result['push_result'])) {
                                $push_success = $result['push_result']['sent'] ?? 0;
                                $push_failed = ($result['push_result']['total'] ?? 0) - $push_success;
                            }
                            
                            $stmt = $conn->prepare("
                                INSERT INTO notification_logs 
                                (notification_id, user_id, sent_by, sent_method, title, message, type, url, push_sent, push_success_count, push_failed_count, push_devices)
                                VALUES (?, ?, ?, 'manual', ?, ?, ?, ?, ?, ?, ?, ?)
                            ");
                            $stmt->execute([
                                $result['notification_id'],
                                $user_id,
                                $current_user_id,
                                $title,
                                $message_text,
                                $type,
                                $url,
                                $send_push ? 1 : 0,
                                $push_success,
                                $push_failed,
                                json_encode($devices)
                            ]);
                        }
                    } else {
                        $error_count++;
                    }
                } catch (Exception $e) {
                    error_log('Error sending notification: ' . $e->getMessage());
                    $error_count++;
                }
            }
            
            if ($success_count > 0) {
                $_SESSION['notification_message'] = "$success_count Benachrichtigung(en) erfolgreich gesendet.";
                $_SESSION['notification_message_type'] = 'success';
            }
            if ($error_count > 0) {
                $_SESSION['notification_message'] = ($_SESSION['notification_message'] ?? '') . ($_SESSION['notification_message'] ? ' ' : '') . "$error_count Fehler aufgetreten.";
                $_SESSION['notification_message_type'] = $success_count > 0 ? 'warning' : 'error';
            }
            
            // POST-Redirect-GET Pattern: Weiterleitung nach POST
            header('Location: notifications.php?sent=1');
            exit;
        }
    } elseif (isset($_POST['delete_notification'])) {
        // Benachrichtigung löschen
        $notification_id = $_POST['notification_id'] ?? null;
        if ($notification_id) {
            try {
                $stmt = $conn->prepare("DELETE FROM user_notifications WHERE id = ?");
                $stmt->execute([$notification_id]);
                $_SESSION['notification_message'] = 'Benachrichtigung erfolgreich gelöscht.';
                $_SESSION['notification_message_type'] = 'success';
            } catch (Exception $e) {
                $_SESSION['notification_message'] = 'Fehler beim Löschen: ' . $e->getMessage();
                $_SESSION['notification_message_type'] = 'error';
            }
            header('Location: notifications.php');
            exit;
        }
    } elseif (isset($_POST['mark_read'])) {
        // Als gelesen markieren
        $notification_id = $_POST['notification_id'] ?? null;
        if ($notification_id) {
            try {
                $stmt = $conn->prepare("UPDATE user_notifications SET is_read = 1 WHERE id = ?");
                $stmt->execute([$notification_id]);
                $_SESSION['notification_message'] = 'Benachrichtigung als gelesen markiert.';
                $_SESSION['notification_message_type'] = 'success';
            } catch (Exception $e) {
                $_SESSION['notification_message'] = 'Fehler: ' . $e->getMessage();
                $_SESSION['notification_message_type'] = 'error';
            }
            header('Location: notifications.php');
            exit;
        }
    }
}

// Filters
$filter_type = $_GET['type'] ?? 'all';
$filter_read = $_GET['read'] ?? 'all';
$search_query = $_GET['search'] ?? '';

// Build query
$where_conditions = [];
$params = [];

if ($filter_type !== 'all') {
    $where_conditions[] = "n.type = ?";
    $params[] = $filter_type;
}

if ($filter_read !== 'all') {
    $where_conditions[] = "n.is_read = ?";
    $params[] = $filter_read === 'read' ? 1 : 0;
}

if (!empty($search_query)) {
    $where_conditions[] = "(n.title LIKE ? OR n.message LIKE ? OR u.email LIKE ? OR u.first_name LIKE ? OR u.last_name LIKE ?)";
    $search_param = '%' . $search_query . '%';
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param, $search_param]);
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Get all autohauser for filter
$stmt = $conn->query("SELECT id, name FROM autohaus WHERE is_active = 1 ORDER BY name");
$autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get all users for selection (inkl. Profilbild und Autohaus)
$stmt = $conn->query("
    SELECT u.id, u.email, u.first_name, u.last_name, u.profile_image, u.profile_image_url, 
           u.preferred_dealership_id, a.name as autohaus_name
    FROM users u
    LEFT JOIN autohaus a ON u.preferred_dealership_id = a.id
    ORDER BY u.last_name, u.first_name
");
$all_users = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get notifications with filters
$stmt = $conn->prepare("
    SELECT n.*, u.email, u.first_name, u.last_name,
           (SELECT COUNT(*) FROM notification_logs WHERE notification_id = n.id) as has_log
    FROM user_notifications n
    LEFT JOIN users u ON n.user_id = u.id
    $where_clause
    ORDER BY n.created_at DESC
    LIMIT 200
");
$stmt->execute($params);
$notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Prüfe ob Details angezeigt werden sollen
$show_details = isset($_GET['details']) && $_GET['details'];
$details_notification_id = $_GET['notification_id'] ?? null;
$notification_details = null;
$notification_logs = [];

if ($show_details && $details_notification_id) {
    // Hole Benachrichtigungs-Details
    $stmt = $conn->prepare("
        SELECT n.*, u.email, u.first_name, u.last_name
        FROM user_notifications n
        LEFT JOIN users u ON n.user_id = u.id
        WHERE n.id = ?
    ");
    $stmt->execute([$details_notification_id]);
    $notification_details = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Hole Log-Details falls vorhanden
    $checkTable = $conn->query("SHOW TABLES LIKE 'notification_logs'");
    if ($checkTable->rowCount() > 0) {
        $stmt = $conn->prepare("
            SELECT nl.*, au.username as sent_by_name
            FROM notification_logs nl
            LEFT JOIN admin_users au ON nl.sent_by = au.id
            WHERE nl.notification_id = ?
            ORDER BY nl.created_at DESC
        ");
        $stmt->execute([$details_notification_id]);
        $notification_logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}

// Get notification statistics
$stmt = $conn->query("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN is_read = 0 THEN 1 ELSE 0 END) as unread,
        SUM(CASE WHEN is_read = 1 THEN 1 ELSE 0 END) as read_count,
        COUNT(DISTINCT user_id) as unique_users,
        SUM(CASE WHEN type = 'info' THEN 1 ELSE 0 END) as type_info,
        SUM(CASE WHEN type = 'success' THEN 1 ELSE 0 END) as type_success,
        SUM(CASE WHEN type = 'warning' THEN 1 ELSE 0 END) as type_warning,
        SUM(CASE WHEN type = 'error' THEN 1 ELSE 0 END) as type_error
    FROM user_notifications
");
$stats = $stmt->fetch(PDO::FETCH_ASSOC);

$current_page = 'notifications';
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Benachrichtigungen verwalten - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #003057;
            --primary-light: #0056b3;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --light: #f8f9fa;
            --dark: #343a40;
            --border: #dee2e6;
            --shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 8px 24px rgba(0, 0, 0, 0.15);
        }

        .admin-container {
            display: flex;
            min-height: 100vh;
            background: #f5f7fa;
        }

        .admin-main {
            flex: 1;
            padding: 2rem;
            margin-left: 300px;
        }

        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }

        .admin-header h1 {
            font-size: 2rem;
            font-weight: 700;
            color: var(--dark);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .admin-header h1 i {
            color: var(--primary);
        }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            box-shadow: var(--shadow);
            transition: transform 0.2s, box-shadow 0.2s;
            border: 1px solid var(--border);
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .stat-icon {
            width: 60px;
            height: 60px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.5rem;
            flex-shrink: 0;
        }

        .stat-icon.info { background: var(--info); }
        .stat-icon.success { background: var(--success); }
        .stat-icon.warning { background: var(--warning); }
        .stat-icon.danger { background: var(--danger); }

        .stat-content {
            flex: 1;
            min-width: 0;
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--dark);
            line-height: 1.2;
        }

        .stat-label {
            color: #6c757d;
            font-size: 0.9rem;
            margin-top: 0.25rem;
        }

        /* Filters */
        .filters-section {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            box-shadow: var(--shadow);
            border: 1px solid var(--border);
        }

        .filters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .filter-group label {
            font-weight: 600;
            color: var(--dark);
            font-size: 0.9rem;
        }

        .filter-group select,
        .filter-group input {
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 0.9rem;
            transition: border-color 0.2s;
        }

        .filter-group select:focus,
        .filter-group input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 48, 87, 0.1);
        }

        /* Notifications Table */
        .notifications-card {
            background: white;
            border-radius: 12px;
            box-shadow: var(--shadow);
            border: 1px solid var(--border);
            overflow: hidden;
        }

        .card-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .card-header h2 {
            margin: 0;
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--dark);
        }

        .table {
            width: 100%;
            border-collapse: collapse;
        }

        .table thead {
            background: #f8f9fa;
        }

        .table th {
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: var(--dark);
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border-bottom: 2px solid var(--border);
        }

        .table td {
            padding: 1rem;
            border-bottom: 1px solid var(--border);
            vertical-align: middle;
        }

        .table tbody tr:hover {
            background: #f8f9fa;
        }

        .badge {
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.85rem;
            font-weight: 500;
            display: inline-block;
        }

        .badge-info { background: #d1ecf1; color: #0c5460; }
        .badge-success { background: #d4edda; color: #155724; }
        .badge-warning { background: #fff3cd; color: #856404; }
        .badge-error { background: #f8d7da; color: #721c24; }

        .btn {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 8px;
            font-size: 0.9rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-light);
        }

        .btn-success {
            background: var(--success);
            color: white;
        }

        .btn-danger {
            background: var(--danger);
            color: white;
        }

        .btn-sm {
            padding: 0.25rem 0.5rem;
            font-size: 0.85rem;
        }

        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }

        .modal.active {
            display: flex;
        }

        .modal-content {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            max-width: 900px;
            width: 95%;
            max-height: 95vh;
            overflow-y: auto;
            box-shadow: var(--shadow-lg);
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border);
        }

        .modal-header h2 {
            margin: 0;
            font-size: 1.5rem;
            font-weight: 600;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #6c757d;
            padding: 0;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: background 0.2s;
        }

        .modal-close:hover {
            background: #f8f9fa;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--dark);
        }

        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 0.9rem;
            transition: border-color 0.2s;
            font-family: inherit;
        }

        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 48, 87, 0.1);
        }

        .form-group textarea {
            resize: vertical;
            min-height: 100px;
        }

        .form-group small {
            display: block;
            margin-top: 0.25rem;
            color: #6c757d;
            font-size: 0.85rem;
        }

        .form-actions {
            display: flex;
            gap: 1rem;
            justify-content: flex-end;
            margin-top: 2rem;
        }

        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .alert-warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }

        /* User Cards */
        .user-card {
            background: white;
            border: 2px solid var(--border);
            border-radius: 12px;
            padding: 1rem;
            cursor: pointer;
            transition: all 0.2s;
            position: relative;
            display: flex;
            flex-direction: column;
            align-items: center;
            text-align: center;
        }

        .user-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: var(--shadow);
        }

        .user-card.selected {
            border-color: var(--primary);
            background: #e3f2fd;
        }

        .user-card-checkbox {
            position: absolute;
            top: 0.5rem;
            right: 0.5rem;
        }

        .user-card-checkbox input[type="checkbox"] {
            width: 20px;
            height: 20px;
            cursor: pointer;
            accent-color: var(--primary);
        }

        .user-card-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.5rem;
            margin-bottom: 0.75rem;
            overflow: hidden;
            position: relative;
        }

        .user-card-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 50%;
        }

        .user-card-avatar .avatar-initials {
            position: absolute;
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
        }

        .user-card-info {
            width: 100%;
        }

        .user-card-name {
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 0.25rem;
            font-size: 0.9rem;
        }

        .user-card-email {
            font-size: 0.75rem;
            color: #6c757d;
            word-break: break-word;
        }

        .pagination-btn {
            padding: 0.5rem 1rem;
            border: 1px solid var(--border);
            background: white;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.2s;
        }

        .pagination-btn:hover:not(:disabled) {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }

        .pagination-btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .user-card.hidden {
            display: none;
        }
    </style>
</head>
<body>
    <?php include 'includes/sidebar.php'; ?>
    
    <div class="admin-container">
        <main class="admin-main">
            <div class="admin-header">
                <h1><i class="fas fa-bell"></i> Benachrichtigungen verwalten</h1>
                <button class="btn btn-primary" onclick="showSendModal()">
                    <i class="fas fa-paper-plane"></i> Neue Benachrichtigung
                </button>
            </div>

            <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
            <?php endif; ?>

            <!-- Statistics -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon info">
                        <i class="fas fa-bell"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['total']; ?></div>
                        <div class="stat-label">Gesamt</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon danger">
                        <i class="fas fa-envelope-open"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['unread']; ?></div>
                        <div class="stat-label">Ungelesen</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon success">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['read_count']; ?></div>
                        <div class="stat-label">Gelesen</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon warning">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['unique_users']; ?></div>
                        <div class="stat-label">Benutzer</div>
                    </div>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters-section">
                <form method="GET" style="display: contents;">
                    <div class="filters-grid">
                        <div class="filter-group">
                            <label>Typ</label>
                            <select name="type" onchange="this.form.submit()">
                                <option value="all" <?php echo $filter_type === 'all' ? 'selected' : ''; ?>>Alle</option>
                                <option value="info" <?php echo $filter_type === 'info' ? 'selected' : ''; ?>>Info</option>
                                <option value="success" <?php echo $filter_type === 'success' ? 'selected' : ''; ?>>Erfolg</option>
                                <option value="warning" <?php echo $filter_type === 'warning' ? 'selected' : ''; ?>>Warnung</option>
                                <option value="error" <?php echo $filter_type === 'error' ? 'selected' : ''; ?>>Fehler</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label>Status</label>
                            <select name="read" onchange="this.form.submit()">
                                <option value="all" <?php echo $filter_read === 'all' ? 'selected' : ''; ?>>Alle</option>
                                <option value="unread" <?php echo $filter_read === 'unread' ? 'selected' : ''; ?>>Ungelesen</option>
                                <option value="read" <?php echo $filter_read === 'read' ? 'selected' : ''; ?>>Gelesen</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label>Suche</label>
                            <input type="text" name="search" value="<?php echo htmlspecialchars($search_query); ?>" placeholder="Titel, Nachricht, Benutzer...">
                        </div>
                        <div class="filter-group" style="justify-content: flex-end;">
                            <label>&nbsp;</label>
                            <button type="submit" class="btn btn-primary" style="width: 100%;">
                                <i class="fas fa-search"></i> Suchen
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Notifications Table -->
            <div class="notifications-card">
                <div class="card-header">
                    <h2>Benachrichtigungen</h2>
                    <span style="color: #6c757d; font-size: 0.9rem;"><?php echo count($notifications); ?> Einträge</span>
                </div>
                <table class="table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Benutzer</th>
                            <th>Titel</th>
                            <th>Typ</th>
                            <th>Status</th>
                            <th>Erstellt</th>
                            <th>Aktionen</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($notifications)): ?>
                        <tr>
                            <td colspan="7" style="text-align: center; padding: 3rem; color: #6c757d;">
                                Keine Benachrichtigungen gefunden
                            </td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($notifications as $notif): ?>
                        <tr>
                            <td><?php echo $notif['id']; ?></td>
                            <td>
                                <?php 
                                $user_name = trim(($notif['first_name'] ?? '') . ' ' . ($notif['last_name'] ?? ''));
                                echo htmlspecialchars($user_name ?: $notif['email']); 
                                ?>
                            </td>
                            <td><?php echo htmlspecialchars($notif['title']); ?></td>
                            <td>
                                <span class="badge badge-<?php echo $notif['type'] ?: 'info'; ?>">
                                    <?php echo htmlspecialchars($notif['type'] ?: 'info'); ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($notif['is_read']): ?>
                                    <span style="color: var(--success);"><i class="fas fa-check-circle"></i> Gelesen</span>
                                <?php else: ?>
                                    <span style="color: var(--danger);"><i class="fas fa-circle"></i> Ungelesen</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo date('d.m.Y H:i', strtotime($notif['created_at'])); ?></td>
                            <td>
                                <div style="display: flex; gap: 0.5rem; align-items: center;">
                                    <?php if (isset($notif['has_log']) && $notif['has_log'] > 0): ?>
                                    <a href="?details=1&notification_id=<?php echo $notif['id']; ?>" class="btn btn-sm" style="background: var(--info); color: white;" title="Details anzeigen">
                                        <i class="fas fa-info-circle"></i>
                                    </a>
                                    <?php endif; ?>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Wirklich löschen?');">
                                        <input type="hidden" name="notification_id" value="<?php echo $notif['id']; ?>">
                                        <?php if (!$notif['is_read']): ?>
                                        <button type="submit" name="mark_read" class="btn btn-success btn-sm" title="Als gelesen markieren">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        <?php endif; ?>
                                        <button type="submit" name="delete_notification" class="btn btn-danger btn-sm" title="Löschen">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>

    <!-- Send Notification Modal -->
    <div id="sendModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Neue Benachrichtigung senden</h2>
                <button class="modal-close" onclick="closeSendModal()">&times;</button>
            </div>
            
            <form method="POST" id="sendNotificationForm">
                <div class="form-group">
                    <label>Empfänger *</label>
                    <div id="selectedUsersInfo" style="margin-bottom: 1rem; padding: 0.75rem; background: #e3f2fd; border-radius: 8px; display: none;">
                        <strong style="color: var(--primary);">Ausgewählt: <span id="selectedCount">0</span> Benutzer</strong>
                        <div id="selectedUsersList" style="margin-top: 0.5rem; font-size: 0.85rem; color: #666;"></div>
                    </div>
                    <input type="hidden" name="user_ids[]" id="selectedUserIds" required>
                    
                    <!-- Autohaus Filter -->
                    <div style="margin-bottom: 1rem;">
                        <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #343a40;">Filter nach Autohaus:</label>
                        <div style="display: flex; flex-wrap: wrap; gap: 0.5rem;">
                            <button type="button" class="autohaus-filter-btn active" data-autohaus-id="all" onclick="filterByAutohaus('all')" style="padding: 0.5rem 1rem; background: var(--primary); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 0.9rem;">
                                <i class="fas fa-building"></i> Alle Autohäuser
                            </button>
                            <button type="button" class="autohaus-filter-btn" data-autohaus-id="none" onclick="filterByAutohaus('none')" style="padding: 0.5rem 1rem; background: #6c757d; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 0.9rem;">
                                <i class="fas fa-times-circle"></i> Kein Autohaus
                            </button>
                            <?php foreach ($autohauser as $autohaus): ?>
                            <button type="button" class="autohaus-filter-btn" data-autohaus-id="<?php echo $autohaus['id']; ?>" onclick="filterByAutohaus(<?php echo $autohaus['id']; ?>)" style="padding: 0.5rem 1rem; background: #6c757d; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 0.9rem;">
                                <i class="fas fa-building"></i> <?php echo htmlspecialchars($autohaus['name']); ?>
                            </button>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Quick Actions -->
                    <div style="margin-bottom: 1rem; display: flex; gap: 0.5rem;">
                        <button type="button" onclick="selectAllVisible()" style="padding: 0.5rem 1rem; background: var(--success); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 0.9rem;">
                            <i class="fas fa-check-double"></i> Alle sichtbaren auswählen
                        </button>
                        <button type="button" onclick="deselectAll()" style="padding: 0.5rem 1rem; background: #6c757d; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 0.9rem;">
                            <i class="fas fa-times"></i> Alle abwählen
                        </button>
                    </div>
                    
                    <!-- Search -->
                    <div style="margin-bottom: 1rem;">
                        <input type="text" id="userSearch" placeholder="Benutzer suchen..." style="width: 100%; padding: 0.75rem; border: 1px solid var(--border); border-radius: 8px;">
                    </div>
                    
                    <!-- User Cards Grid -->
                    <div id="usersGrid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 1rem; max-height: 500px; overflow-y: auto; padding: 0.5rem; border: 1px solid var(--border); border-radius: 8px; background: #f8f9fa;">
                        <?php 
                        $users_per_page = 12;
                        $total_users = count($all_users);
                        $total_pages = ceil($total_users / $users_per_page);
                        $current_page = 1;
                        $start = 0;
                        $end = min($users_per_page, $total_users);
                        
                        for ($i = $start; $i < $end; $i++): 
                            $user = $all_users[$i];
                            $name = trim(($user['first_name'] ?? '') . ' ' . ($user['last_name'] ?? ''));
                            $display_name = $name ?: $user['email'];
                        ?>
                        <div class="user-card" data-user-id="<?php echo $user['id']; ?>" data-user-name="<?php echo htmlspecialchars($display_name); ?>" data-user-email="<?php echo htmlspecialchars($user['email']); ?>" data-autohaus-id="<?php echo $user['preferred_dealership_id'] ?? 'none'; ?>" onclick="toggleUserSelection(<?php echo $user['id']; ?>)">
                            <div class="user-card-checkbox">
                                <input type="checkbox" class="user-checkbox" value="<?php echo $user['id']; ?>" id="user_<?php echo $user['id']; ?>" onclick="event.stopPropagation(); toggleUserSelection(<?php echo $user['id']; ?>)">
                                <label for="user_<?php echo $user['id']; ?>" onclick="event.stopPropagation();"></label>
                            </div>
                            <div class="user-card-avatar">
                                <?php 
                                $profile_image = $user['profile_image_url'] ?? $user['profile_image'] ?? null;
                                $initials = strtoupper(substr($user['first_name'] ?? '', 0, 1) . substr($user['last_name'] ?? '', 0, 1));
                                if ($profile_image && file_exists('../' . ltrim($profile_image, '/'))) {
                                    echo '<img src="../' . htmlspecialchars(ltrim($profile_image, '/')) . '" alt="' . htmlspecialchars($display_name) . '" onerror="this.style.display=\'none\'; this.nextElementSibling.style.display=\'flex\';">';
                                    echo '<div class="avatar-initials" style="display: none;">' . htmlspecialchars($initials) . '</div>';
                                } else {
                                    echo '<div class="avatar-initials">' . htmlspecialchars($initials ?: '?') . '</div>';
                                }
                                ?>
                            </div>
                            <div class="user-card-info">
                                <div class="user-card-name"><?php echo htmlspecialchars($display_name); ?></div>
                                <div class="user-card-email"><?php echo htmlspecialchars($user['email']); ?></div>
                                <?php if (!empty($user['autohaus_name'])): ?>
                                <div class="user-card-autohaus" style="font-size: 0.75rem; color: #6c757d; margin-top: 0.25rem;">
                                    <i class="fas fa-building"></i> <?php echo htmlspecialchars($user['autohaus_name']); ?>
                                </div>
                                <?php else: ?>
                                <div class="user-card-autohaus" style="font-size: 0.75rem; color: #999; margin-top: 0.25rem;">
                                    <i class="fas fa-times-circle"></i> Kein Autohaus
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endfor; ?>
                    </div>
                    
                    <!-- Pagination -->
                    <div class="pagination" style="display: flex; justify-content: center; align-items: center; gap: 0.5rem; margin-top: 1rem;">
                        <button type="button" class="pagination-btn" id="prevPage" onclick="changePage(-1)" disabled>
                            <i class="fas fa-chevron-left"></i>
                        </button>
                        <span id="pageInfo" style="padding: 0.5rem 1rem; background: white; border-radius: 8px; border: 1px solid var(--border);">
                            Seite <span id="currentPage">1</span> von <span id="totalPagesDisplay"><?php echo $total_pages; ?></span>
                        </span>
                        <button type="button" class="pagination-btn" id="nextPage" onclick="changePage(1)" <?php echo $total_pages > 1 ? '' : 'disabled'; ?>>
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    </div>
                    
                    <input type="hidden" id="totalPages" value="<?php echo $total_pages; ?>">
                    <input type="hidden" id="usersPerPage" value="<?php echo $users_per_page; ?>">
                    <input type="hidden" id="allUsersData" value='<?php echo json_encode($all_users); ?>'>
                    <input type="hidden" id="autohauserData" value='<?php echo json_encode($autohauser); ?>'>
                </div>
                
                <div class="form-group">
                    <label>Titel *</label>
                    <input type="text" name="title" required>
                </div>
                
                <div class="form-group">
                    <label>Nachricht *</label>
                    <textarea name="message" required rows="4"></textarea>
                </div>
                
                <div class="form-group">
                    <label>Typ</label>
                    <select name="type">
                        <option value="info">Info</option>
                        <option value="success">Erfolg</option>
                        <option value="warning">Warnung</option>
                        <option value="error">Fehler</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>URL (optional)</label>
                    <input type="text" name="url" value="/profile.php">
                </div>
                
                <div class="form-group">
                    <label style="display: flex; align-items: center; cursor: pointer;">
                        <input type="checkbox" name="send_push" value="1" checked style="margin-right: 0.5rem; width: auto;">
                        <span>Push-Benachrichtigung senden</span>
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="button" onclick="closeSendModal()" class="btn btn-secondary">Abbrechen</button>
                    <button type="submit" name="send_notification" class="btn btn-primary">
                        <i class="fas fa-paper-plane"></i> Senden
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
    let currentPage = 1;
    let allUsers = [];
    let selectedUsers = new Set();
    let filteredUsers = [];
    let currentAutohausFilter = 'all';

    // Load users data
    document.addEventListener('DOMContentLoaded', function() {
        const usersData = document.getElementById('allUsersData');
        if (usersData) {
            allUsers = JSON.parse(usersData.value);
            filteredUsers = allUsers;
            renderUsers();
        }

        // Search functionality
        const searchInput = document.getElementById('userSearch');
        if (searchInput) {
            searchInput.addEventListener('input', function(e) {
                const searchTerm = e.target.value.toLowerCase();
                filteredUsers = allUsers.filter(user => {
                    const name = ((user.first_name || '') + ' ' + (user.last_name || '')).trim().toLowerCase();
                    const email = (user.email || '').toLowerCase();
                    return name.includes(searchTerm) || email.includes(searchTerm);
                });
                currentPage = 1;
                renderUsers();
            });
        }

        // Checkbox change handler
        document.addEventListener('change', function(e) {
            if (e.target.classList.contains('user-checkbox')) {
                const userId = e.target.value;
                toggleUserSelection(userId);
            }
        });
        
        // Initial filter application
        applyFilters();
    });
    
    function applyFilters() {
        const searchTerm = (document.getElementById('userSearch')?.value || '').toLowerCase();
        filteredUsers = allUsers.filter(user => {
            // Search filter
            const name = ((user.first_name || '') + ' ' + (user.last_name || '')).trim().toLowerCase();
            const email = (user.email || '').toLowerCase();
            const matchesSearch = !searchTerm || name.includes(searchTerm) || email.includes(searchTerm);
            
            // Autohaus filter
            const userAutohausId = user.preferred_dealership_id ? user.preferred_dealership_id.toString() : 'none';
            let matchesAutohaus = false;
            if (currentAutohausFilter === 'all') {
                matchesAutohaus = true;
            } else if (currentAutohausFilter === 'none') {
                matchesAutohaus = userAutohausId === 'none';
            } else {
                matchesAutohaus = userAutohausId === currentAutohausFilter.toString();
            }
            
            return matchesSearch && matchesAutohaus;
        });
        currentPage = 1;
        renderUsers();
    }

    function renderUsers() {
        const usersPerPage = parseInt(document.getElementById('usersPerPage').value) || 12;
        const totalPages = Math.ceil(filteredUsers.length / usersPerPage);
        const start = (currentPage - 1) * usersPerPage;
        const end = start + usersPerPage;
        const pageUsers = filteredUsers.slice(start, end);

        const grid = document.getElementById('usersGrid');
        grid.innerHTML = '';

        pageUsers.forEach(user => {
            const name = ((user.first_name || '') + ' ' + (user.last_name || '')).trim() || user.email;
            const isSelected = selectedUsers.has(user.id.toString());
            
            // Profilbild oder Initialen
            const profileImage = user.profile_image_url || user.profile_image || null;
            const initials = ((user.first_name || '').charAt(0) + (user.last_name || '').charAt(0)).toUpperCase() || '?';
            let avatarHtml = '';
            if (profileImage) {
                avatarHtml = `<img src="../${profileImage.replace(/^\//, '')}" alt="${escapeHtml(name)}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><div class="avatar-initials" style="display: none;">${escapeHtml(initials)}</div>`;
            } else {
                avatarHtml = `<div class="avatar-initials">${escapeHtml(initials)}</div>`;
            }
            
            const autohausId = user.preferred_dealership_id ? user.preferred_dealership_id.toString() : 'none';
            const autohausName = user.autohaus_name || null;
            
            const card = document.createElement('div');
            card.className = 'user-card' + (isSelected ? ' selected' : '');
            card.dataset.userId = user.id;
            card.dataset.userName = name;
            card.dataset.userEmail = user.email;
            card.dataset.autohausId = autohausId;
            card.onclick = function(e) {
                if (e.target.type !== 'checkbox' && e.target.tagName !== 'LABEL') {
                    toggleUserSelection(user.id);
                }
            };
            card.style.cursor = 'pointer';
            card.innerHTML = `
                <div class="user-card-checkbox">
                    <input type="checkbox" class="user-checkbox" value="${user.id}" id="user_${user.id}" ${isSelected ? 'checked' : ''} onclick="event.stopPropagation(); toggleUserSelection(${user.id})">
                    <label for="user_${user.id}" onclick="event.stopPropagation();"></label>
                </div>
                <div class="user-card-avatar">
                    ${avatarHtml}
                </div>
                <div class="user-card-info">
                    <div class="user-card-name">${escapeHtml(name)}</div>
                    <div class="user-card-email">${escapeHtml(user.email)}</div>
                    ${autohausName ? `<div class="user-card-autohaus" style="font-size: 0.75rem; color: #6c757d; margin-top: 0.25rem;"><i class="fas fa-building"></i> ${escapeHtml(autohausName)}</div>` : `<div class="user-card-autohaus" style="font-size: 0.75rem; color: #999; margin-top: 0.25rem;"><i class="fas fa-times-circle"></i> Kein Autohaus</div>`}
                </div>
            `;
            grid.appendChild(card);
        });

        // Update pagination
        const currentPageEl = document.getElementById('currentPage');
        const prevBtn = document.getElementById('prevPage');
        const nextBtn = document.getElementById('nextPage');
        const pageInfo = document.getElementById('pageInfo');
        
        if (currentPageEl) currentPageEl.textContent = currentPage;
        if (prevBtn) prevBtn.disabled = currentPage === 1;
        if (nextBtn) nextBtn.disabled = currentPage >= totalPages || totalPages === 0;
        
        // Update total pages display
        if (pageInfo) {
            const totalPagesDisplay = document.getElementById('totalPagesDisplay');
            if (totalPagesDisplay) {
                totalPagesDisplay.textContent = totalPages || 1;
            } else {
                pageInfo.innerHTML = `Seite <span id="currentPage">${currentPage}</span> von <span id="totalPagesDisplay">${totalPages || 1}</span>`;
            }
        }
    }

    function toggleUserSelection(userId) {
        const userIdStr = userId.toString();
        const checkbox = document.getElementById(`user_${userId}`);
        const userCard = checkbox?.closest('.user-card');
        
        if (checkbox) {
            if (checkbox.checked) {
                selectedUsers.delete(userIdStr);
                checkbox.checked = false;
                if (userCard) userCard.classList.remove('selected');
            } else {
                selectedUsers.add(userIdStr);
                checkbox.checked = true;
                if (userCard) userCard.classList.add('selected');
            }
            updateSelectedUsersInfo();
        }
    }
    
    function filterByAutohaus(autohausId) {
        currentAutohausFilter = autohausId;
        
        // Update button styles
        document.querySelectorAll('.autohaus-filter-btn').forEach(btn => {
            btn.classList.remove('active');
            btn.style.background = '#6c757d';
        });
        
        const activeBtn = document.querySelector(`[data-autohaus-id="${autohausId}"]`);
        if (activeBtn) {
            activeBtn.classList.add('active');
            activeBtn.style.background = 'var(--primary)';
        }
        
        applyFilters();
    }
    
    function selectAllVisible() {
        const usersPerPage = parseInt(document.getElementById('usersPerPage').value) || 12;
        const start = (currentPage - 1) * usersPerPage;
        const end = start + usersPerPage;
        const pageUsers = filteredUsers.slice(start, end);
        
        pageUsers.forEach(user => {
            selectedUsers.add(user.id.toString());
        });
        
        renderUsers();
        updateSelectedUsersInfo();
    }
    
    function deselectAll() {
        selectedUsers.clear();
        renderUsers();
        updateSelectedUsersInfo();
    }
    
    function applyFilters() {
        const searchTerm = (document.getElementById('userSearch')?.value || '').toLowerCase();
        filteredUsers = allUsers.filter(user => {
            // Search filter
            const name = ((user.first_name || '') + ' ' + (user.last_name || '')).trim().toLowerCase();
            const email = (user.email || '').toLowerCase();
            const matchesSearch = !searchTerm || name.includes(searchTerm) || email.includes(searchTerm);
            
            // Autohaus filter
            const userAutohausId = user.preferred_dealership_id ? user.preferred_dealership_id.toString() : 'none';
            let matchesAutohaus = false;
            if (currentAutohausFilter === 'all') {
                matchesAutohaus = true;
            } else if (currentAutohausFilter === 'none') {
                matchesAutohaus = userAutohausId === 'none';
            } else {
                matchesAutohaus = userAutohausId === currentAutohausFilter.toString();
            }
            
            return matchesSearch && matchesAutohaus;
        });
        currentPage = 1;
        renderUsers();
    }
    
    function changePage(direction) {
        const usersPerPage = parseInt(document.getElementById('usersPerPage').value) || 12;
        const totalPages = Math.ceil(filteredUsers.length / usersPerPage);
        const newPage = currentPage + direction;
        
        if (newPage >= 1 && newPage <= totalPages && totalPages > 0) {
            currentPage = newPage;
            renderUsers();
        }
    }

    function updateSelectedUsersInfo() {
        const selectedIds = Array.from(selectedUsers);
        const selectedUserIdsInput = document.getElementById('selectedUserIds');
        const infoDiv = document.getElementById('selectedUsersInfo');
        const countSpan = document.getElementById('selectedCount');
        const listDiv = document.getElementById('selectedUsersList');

        if (selectedIds.length > 0) {
            selectedUserIdsInput.value = selectedIds.join(',');
            infoDiv.style.display = 'block';
            countSpan.textContent = selectedIds.length;
            
            // Show selected user names
            const selectedNames = selectedIds.map(id => {
                const user = allUsers.find(u => u.id.toString() === id);
                if (user) {
                    const name = ((user.first_name || '') + ' ' + (user.last_name || '')).trim();
                    return name || user.email;
                }
                return '';
            }).filter(n => n);
            
            listDiv.innerHTML = selectedNames.slice(0, 5).join(', ') + (selectedNames.length > 5 ? ' ...' : '');
        } else {
            selectedUserIdsInput.value = '';
            infoDiv.style.display = 'none';
        }
    }

    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    function showSendModal() {
        document.getElementById('sendModal').classList.add('active');
    }

    function closeSendModal() {
        document.getElementById('sendModal').classList.remove('active');
    }

    // Close modal on outside click
    document.getElementById('sendModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeSendModal();
        }
    });

    // Close modal on ESC key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeSendModal();
        }
    });

    // Form validation
    const form = document.getElementById('sendNotificationForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            if (selectedUsers.size === 0) {
                e.preventDefault();
                alert('Bitte wählen Sie mindestens einen Benutzer aus.');
                return false;
            }
            
            // Set all selected user IDs as hidden inputs
            selectedUsers.forEach(userId => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'user_ids[]';
                input.value = userId;
                this.appendChild(input);
            });
        });
    }

    function closeDetailsModal() {
        window.location.href = 'notifications.php';
    }
    </script>

    <!-- Details Modal -->
    <?php if ($show_details && $notification_details): ?>
    <div id="detailsModal" class="modal active">
        <div class="modal-content" style="max-width: 800px;">
            <div class="modal-header">
                <h2><i class="fas fa-info-circle"></i> Benachrichtigungs-Details</h2>
                <button class="modal-close" onclick="closeDetailsModal()">&times;</button>
            </div>
            
            <div style="margin-bottom: 2rem;">
                <h3 style="margin-bottom: 1rem; color: var(--dark);">Grundinformationen</h3>
                <div style="display: grid; gap: 1rem;">
                    <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem; padding: 0.75rem; background: #f8f9fa; border-radius: 8px;">
                        <strong>ID:</strong>
                        <span><?php echo $notification_details['id']; ?></span>
                    </div>
                    <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem; padding: 0.75rem; background: #f8f9fa; border-radius: 8px;">
                        <strong>Benutzer:</strong>
                        <span><?php 
                            $user_name = trim(($notification_details['first_name'] ?? '') . ' ' . ($notification_details['last_name'] ?? ''));
                            echo htmlspecialchars($user_name ?: $notification_details['email']); 
                        ?></span>
                    </div>
                    <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem; padding: 0.75rem; background: #f8f9fa; border-radius: 8px;">
                        <strong>Titel:</strong>
                        <span><?php echo htmlspecialchars($notification_details['title']); ?></span>
                    </div>
                    <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem; padding: 0.75rem; background: #f8f9fa; border-radius: 8px;">
                        <strong>Nachricht:</strong>
                        <span><?php echo htmlspecialchars($notification_details['message']); ?></span>
                    </div>
                    <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem; padding: 0.75rem; background: #f8f9fa; border-radius: 8px;">
                        <strong>Typ:</strong>
                        <span><span class="badge badge-<?php echo $notification_details['type'] ?: 'info'; ?>"><?php echo htmlspecialchars($notification_details['type'] ?: 'info'); ?></span></span>
                    </div>
                    <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem; padding: 0.75rem; background: #f8f9fa; border-radius: 8px;">
                        <strong>Erstellt:</strong>
                        <span><?php echo date('d.m.Y H:i:s', strtotime($notification_details['created_at'])); ?></span>
                    </div>
                    <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem; padding: 0.75rem; background: #f8f9fa; border-radius: 8px;">
                        <strong>Status:</strong>
                        <span><?php echo $notification_details['is_read'] ? '<span style="color: var(--success);">✓ Gelesen</span>' : '<span style="color: var(--danger);">○ Ungelesen</span>'; ?></span>
                    </div>
                </div>
            </div>

            <?php if (!empty($notification_logs)): ?>
            <div style="margin-bottom: 2rem;">
                <h3 style="margin-bottom: 1rem; color: var(--dark);">Versand-Details</h3>
                <?php foreach ($notification_logs as $log): ?>
                <div style="background: white; border: 1px solid var(--border); border-radius: 8px; padding: 1.5rem; margin-bottom: 1rem;">
                    <div style="display: grid; gap: 0.75rem;">
                        <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem;">
                            <strong>Gesendet von:</strong>
                            <span><?php echo htmlspecialchars($log['sent_by_name'] ?? 'System'); ?></span>
                        </div>
                        <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem;">
                            <strong>Methode:</strong>
                            <span><?php 
                                $methods = ['manual' => 'Manuell', 'trigger' => 'Trigger', 'automatic' => 'Automatisch'];
                                echo $methods[$log['sent_method']] ?? $log['sent_method'];
                            ?></span>
                        </div>
                        <?php if ($log['trigger_name']): ?>
                        <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem;">
                            <strong>Trigger:</strong>
                            <span><code><?php echo htmlspecialchars($log['trigger_name']); ?></code></span>
                        </div>
                        <?php endif; ?>
                        <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem;">
                            <strong>Push gesendet:</strong>
                            <span><?php echo $log['push_sent'] ? '<span style="color: var(--success);">✓ Ja</span>' : '<span style="color: #6c757d;">○ Nein</span>'; ?></span>
                        </div>
                        <?php if ($log['push_sent']): ?>
                        <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem;">
                            <strong>Push-Ergebnis:</strong>
                            <span>
                                <span style="color: var(--success);">✓ <?php echo $log['push_success_count']; ?> erfolgreich</span>
                                <?php if ($log['push_failed_count'] > 0): ?>
                                <span style="color: var(--danger); margin-left: 1rem;">✗ <?php echo $log['push_failed_count']; ?> fehlgeschlagen</span>
                                <?php endif; ?>
                            </span>
                        </div>
                        <?php endif; ?>
                        <div style="display: grid; grid-template-columns: 150px 1fr; gap: 1rem;">
                            <strong>Gesendet am:</strong>
                            <span><?php echo date('d.m.Y H:i:s', strtotime($log['created_at'])); ?></span>
                        </div>
                        
                        <?php if ($log['push_devices']): ?>
                        <?php 
                        $devices = json_decode($log['push_devices'], true);
                        if ($devices && is_array($devices) && !empty($devices)):
                        ?>
                        <div style="margin-top: 1rem; padding-top: 1rem; border-top: 1px solid var(--border);">
                            <strong style="display: block; margin-bottom: 0.75rem;">Geräte (<?php echo count($devices); ?>):</strong>
                            <div style="display: grid; gap: 0.5rem;">
                                <?php foreach ($devices as $device): ?>
                                <div style="padding: 0.75rem; background: #f8f9fa; border-radius: 6px; font-size: 0.9rem;">
                                    <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 0.5rem;">
                                        <div>
                                            <strong>Gerät #<?php echo $device['id']; ?></strong>
                                            <span style="color: var(--<?php echo $device['is_active'] ? 'success' : 'danger'; ?>); margin-left: 0.5rem;">
                                                <?php echo $device['is_active'] ? '✓ Aktiv' : '○ Inaktiv'; ?>
                                            </span>
                                        </div>
                                    </div>
                                    <div style="color: #6c757d; font-size: 0.85rem; margin-top: 0.5rem;">
                                        <div><strong>User Agent:</strong> <?php echo htmlspecialchars(substr($device['user_agent'] ?? '', 0, 80)); ?>...</div>
                                        <?php if ($device['device_info']): ?>
                                        <?php $device_info = json_decode($device['device_info'], true); ?>
                                        <?php if ($device_info): ?>
                                        <div style="margin-top: 0.25rem;"><strong>Platform:</strong> <?php echo htmlspecialchars($device_info['platform'] ?? 'Unbekannt'); ?></div>
                                        <?php endif; ?>
                                        <?php endif; ?>
                                        <div style="margin-top: 0.25rem; word-break: break-all;"><strong>Endpoint:</strong> <?php echo htmlspecialchars(substr($device['endpoint'] ?? '', 0, 100)); ?>...</div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php else: ?>
            <div style="padding: 2rem; text-align: center; color: #6c757d;">
                <i class="fas fa-info-circle" style="font-size: 2rem; margin-bottom: 1rem; display: block;"></i>
                <p>Keine Versand-Details verfügbar. Diese Benachrichtigung wurde möglicherweise automatisch erstellt.</p>
            </div>
            <?php endif; ?>
            
            <div class="form-actions">
                <button type="button" onclick="closeDetailsModal()" class="btn btn-primary">Schließen</button>
            </div>
        </div>
    </div>
    <?php endif; ?>
</body>
</html>
