<?php
/**
 * Admin Login mit 2FA per WhatsApp
 */

session_start();
require_once '../config/database.php';
require_once 'includes/auth_2fa.php';
require_once 'includes/whatsapp_2fa.php';

// Redirect if already logged in and 2FA verified
if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] && is2FAVerified()) {
    header('Location: index.php');
    exit;
}

$error_message = '';
$step = $_GET['step'] ?? 'login'; // login, 2fa
$show_2fa = false;
$phone_hint = '';

// Prüfe ob Gerät vertrauenswürdig ist
$deviceIdentifier = generateDeviceIdentifier();
$trustedUser = checkTrustedDevice($deviceIdentifier);
if ($trustedUser && !isset($_GET['force_login'])) {
    // Auto-Login mit vertrauenswürdigem Gerät
    $_SESSION['admin_logged_in'] = true;
    $_SESSION['admin_user_id'] = $trustedUser['user_id'];
    $_SESSION['admin_username'] = $trustedUser['username'];
    $_SESSION['2fa_verified'] = true;
    $_SESSION['2fa_required'] = false;
    $_SESSION['auto_login'] = true;
    
    header('Location: index.php');
    exit;
}

if ($_POST) {
    $action = $_POST['action'] ?? 'login';
    
    if ($action === 'login') {
        $username = $_POST['username'] ?? '';
        $password = $_POST['password'] ?? '';
        $trustDevice = isset($_POST['trust_device']) && $_POST['trust_device'] === '1';
        
        if ($username && $password) {
            // Rate Limiting prüfen
            if (!checkLoginAttempts($username)) {
                $error_message = 'Zu viele Login-Versuche. Bitte versuchen Sie es in 15 Minuten erneut.';
            } else {
                $db = new Database();
                $conn = $db->getConnection();
                
                // Erstelle admin_user_2fa Tabelle falls nicht vorhanden
                $conn->exec("
                    CREATE TABLE IF NOT EXISTS admin_user_2fa (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        user_id INT NOT NULL UNIQUE,
                        two_factor_secret VARCHAR(32),
                        two_factor_method VARCHAR(20) DEFAULT 'whatsapp',
                        two_factor_enabled TINYINT(1) DEFAULT 1,
                        phone VARCHAR(20) DEFAULT NULL,
                        phone_verified TINYINT(1) DEFAULT 0,
                        phone_verified_at DATETIME NULL,
                        phone_verification_error VARCHAR(255) NULL,
                        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        INDEX idx_user_id (user_id)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
                ");
                
                // Stelle sicher, dass Verifizierungs-Spalten existieren
                try {
                    $tableExists = $conn->query("SHOW TABLES LIKE 'admin_user_2fa'")->rowCount() > 0;
                    if ($tableExists) {
                        $columns = $conn->query("SHOW COLUMNS FROM admin_user_2fa")->fetchAll(PDO::FETCH_COLUMN);
                        if (!in_array('phone_verified', $columns)) {
                            $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verified TINYINT(1) DEFAULT 0");
                        }
                        if (!in_array('phone_verified_at', $columns)) {
                            $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verified_at DATETIME NULL");
                        }
                        if (!in_array('phone_verification_error', $columns)) {
                            $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verification_error VARCHAR(255) NULL");
                        }
                    }
                } catch (PDOException $e) {
                    error_log("Phone verification columns check: " . $e->getMessage());
                }
                
                // Hole Benutzer mit 2FA-Daten und Verifizierungsstatus
                // WICHTIG: Telefonnummer ist in admin_users.phone gespeichert (wie in whatsappp2kiel)
                $stmt = $conn->prepare("
                    SELECT au.id, au.username, au.password_hash, au.email, au.autohaus_id, au.phone,
                           tfa.two_factor_enabled, tfa.two_factor_method,
                           COALESCE(tfa.phone_verified, 0) as phone_verified
                    FROM admin_users au
                    LEFT JOIN admin_user_2fa tfa ON au.id = tfa.user_id
                    WHERE au.username = ? AND au.is_active = 1
                ");
                $stmt->execute([$username]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($user && password_verify($password, $user['password_hash'])) {
                    clearLoginAttempts($username);
                    
                    // Prüfe ob Telefonnummer vorhanden ist (trim für Whitespace)
                    $phone = !empty($user['phone']) ? trim($user['phone']) : '';
                    $hasPhone = !empty($phone);
                    
                    // Prüfe ob 2FA aktiviert ist
                    $has2FA = !empty($user['two_factor_enabled']) && $user['two_factor_enabled'] == 1;
                    
                    // Wenn Telefonnummer vorhanden ist, aber 2FA nicht aktiviert, aktiviere es automatisch
                    if ($hasPhone && !$has2FA) {
                        require_once 'includes/whatsapp_2fa.php';
                        setWhatsApp2FA($user['id'], $phone, false); // Kein Code senden, nur 2FA aktivieren
                        $has2FA = true;
                    }
                    
                    // WICHTIG: Prüfe ob Telefonnummer verifiziert ist
                    $isPhoneVerified = !empty($user['phone_verified']) && $user['phone_verified'] == 1;
                    
                    if ($hasPhone) {
                        // Prüfe zuerst ob Telefonnummer verifiziert ist
                        if (!$isPhoneVerified) {
                            $error_message = 'Ihre Telefonnummer ist noch nicht verifiziert. Bitte kontaktieren Sie den Administrator, um die Telefonnummer zu verifizieren, bevor Sie sich anmelden können.';
                        } else {
                            // 2FA erforderlich - sende Code
                            $sendResult = generateAndSendWhatsApp2FA($user['id'], $phone, $user['autohaus_id']);
                            
                            if ($sendResult['success']) {
                                // Setze Session für 2FA-Verifizierung
                                $_SESSION['pending_user_id'] = $user['id'];
                                $_SESSION['pending_username'] = $user['username'];
                                $_SESSION['pending_autohaus_id'] = $user['autohaus_id'];
                                $_SESSION['pending_device_identifier'] = $deviceIdentifier;
                                $_SESSION['pending_trust_device'] = $trustDevice;
                                
                                $show_2fa = true;
                                $step = '2fa';
                                $phone_hint = '***' . substr($phone, -4);
                            } else {
                                $error_message = 'Fehler beim Senden des 2FA-Codes: ' . ($sendResult['error'] ?? 'Unbekannter Fehler');
                            }
                        }
                    } else {
                        // Keine Telefonnummer vorhanden
                        $error_message = 'Für diesen Benutzer ist keine Telefonnummer hinterlegt. Bitte kontaktieren Sie den Administrator, um eine Telefonnummer für 2FA einzutragen.';
                    }
                } else {
                    recordFailedLogin($username);
                    $error_message = 'Ungültige Anmeldedaten';
                }
            }
        } else {
            $error_message = 'Bitte geben Sie Benutzername und Passwort ein';
        }
    } elseif ($action === 'verify_2fa') {
        $code = $_POST['code'] ?? '';
        $trustDevice = isset($_POST['trust_device']) && $_POST['trust_device'] === '1';
        
        if (!isset($_SESSION['pending_user_id'])) {
            $error_message = 'Kein ausstehender Login gefunden';
            $step = 'login';
        } elseif (empty($code) || strlen($code) !== 6) {
            $error_message = 'Bitte geben Sie den 6-stelligen Code ein';
            $step = '2fa';
            $show_2fa = true;
        } else {
            $userId = $_SESSION['pending_user_id'];
            $codeValid = verifyWhatsApp2FACode($userId, $code);
            
            if ($codeValid) {
                // 2FA erfolgreich - vollständiger Login
                session_regenerate_id(true);
                
                $_SESSION['admin_logged_in'] = true;
                $_SESSION['admin_user_id'] = $userId;
                $_SESSION['admin_username'] = $_SESSION['pending_username'];
                $_SESSION['2fa_verified'] = true;
                $_SESSION['2fa_required'] = true;
                
                // Update last login
                $db = new Database();
                $conn = $db->getConnection();
                $stmt = $conn->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = ?");
                $stmt->execute([$userId]);
                
                // Registriere Gerät als vertrauenswürdig wenn gewünscht
                $deviceIdentifier = $_SESSION['pending_device_identifier'] ?? generateDeviceIdentifier();
                if ($trustDevice && !empty($deviceIdentifier)) {
                    $deviceName = $_SERVER['HTTP_USER_AGENT'] ?? 'Unbekanntes Gerät';
                    if (registerTrustedDevice($userId, $deviceIdentifier, $deviceName, null)) {
                        setTrustedDeviceCookie($deviceIdentifier);
                    }
                }
                
                // Lösche pending Session-Daten
                unset($_SESSION['pending_user_id']);
                unset($_SESSION['pending_username']);
                unset($_SESSION['pending_autohaus_id']);
                unset($_SESSION['pending_device_identifier']);
                unset($_SESSION['pending_trust_device']);
                
                header('Location: index.php');
                exit;
            } else {
                recordFailedLogin($_SESSION['pending_username'] ?? '');
                $error_message = 'Ungültiger 2FA-Code';
                $step = '2fa';
                $show_2fa = true;
                $phone_hint = '***' . substr($_SESSION['pending_phone'] ?? '', -4);
            }
        }
    } elseif ($action === 'resend_code') {
        if (isset($_SESSION['pending_user_id'])) {
            $db = new Database();
            $conn = $db->getConnection();
            
            // Erstelle admin_user_2fa Tabelle falls nicht vorhanden
            $conn->exec("
                CREATE TABLE IF NOT EXISTS admin_user_2fa (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT NOT NULL UNIQUE,
                    two_factor_secret VARCHAR(32),
                    two_factor_method VARCHAR(20) DEFAULT 'whatsapp',
                    two_factor_enabled TINYINT(1) DEFAULT 1,
                    phone VARCHAR(20) DEFAULT NULL,
                    phone_verified TINYINT(1) DEFAULT 0,
                    phone_verified_at DATETIME NULL,
                    phone_verification_error VARCHAR(255) NULL,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    INDEX idx_user_id (user_id)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
            ");
            
            $stmt = $conn->prepare("
                SELECT au.email, au.phone, au.autohaus_id
                FROM admin_users au
                LEFT JOIN admin_user_2fa tfa ON au.id = tfa.user_id
                WHERE au.id = ?
            ");
            $stmt->execute([$_SESSION['pending_user_id']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Verwende admin_users.phone (wie in whatsappp2kiel)
            $phone = !empty($user['phone']) ? trim($user['phone']) : '';
            
            if (!empty($phone)) {
                $sendResult = generateAndSendWhatsApp2FA($_SESSION['pending_user_id'], $phone, $user['autohaus_id']);
                if ($sendResult['success']) {
                    $error_message = 'Code wurde erneut gesendet';
                } else {
                    $error_message = 'Fehler beim Senden: ' . ($sendResult['error'] ?? 'Unbekannter Fehler');
                }
            }
            $step = '2fa';
            $show_2fa = true;
            $phone_hint = '***' . substr($user['phone'] ?? '', -4);
        }
    }
}

// Wenn step=2fa in URL, zeige 2FA-Formular
if ($step === '2fa' && isset($_SESSION['pending_user_id'])) {
    $show_2fa = true;
    $db = new Database();
    $conn = $db->getConnection();
    
    // Erstelle admin_user_2fa Tabelle falls nicht vorhanden
    $conn->exec("
        CREATE TABLE IF NOT EXISTS admin_user_2fa (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL UNIQUE,
            two_factor_secret VARCHAR(32),
            two_factor_method VARCHAR(20) DEFAULT 'whatsapp',
            two_factor_enabled TINYINT(1) DEFAULT 1,
            phone VARCHAR(20) DEFAULT NULL,
            phone_verified TINYINT(1) DEFAULT 0,
            phone_verified_at DATETIME NULL,
            phone_verification_error VARCHAR(255) NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_user_id (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    // Verwende admin_users.phone (wie in whatsappp2kiel)
    $stmt = $conn->prepare("
        SELECT phone FROM admin_users WHERE id = ?
    ");
    $stmt->execute([$_SESSION['pending_user_id']]);
    $tfaData = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($tfaData && !empty($tfaData['phone'])) {
        $phone_hint = '***' . substr(trim($tfaData['phone']), -4);
    }
}
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <title>Boxenstop Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="login-overlay">
        <div class="login-modal">
            <div class="login-header">
                <div class="login-logo">
                    <i class="fas fa-car"></i>
                </div>
                <h1>Boxenstop</h1>
                <p class="login-subtitle">Admin-Dashboard</p>
            </div>

            <?php if ($error_message): ?>
                <div class="login-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if (!$show_2fa): ?>
            <!-- Login Formular -->
            <form method="POST" class="login-form">
                <input type="hidden" name="action" value="login">
                
                <div class="form-group">
                    <label for="username">
                        <i class="fas fa-user"></i>
                        Benutzername
                    </label>
                    <input
                        type="text"
                        id="username"
                        name="username"
                        value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>"
                        placeholder="Benutzername eingeben"
                        required
                        autofocus
                    />
                </div>

                <div class="form-group">
                    <label for="password">
                        <i class="fas fa-lock"></i>
                        Passwort
                    </label>
                    <input
                        type="password"
                        id="password"
                        name="password"
                        placeholder="Passwort eingeben"
                        required
                    />
                </div>

                <div class="form-group checkbox-group">
                    <label class="checkbox-label">
                        <input
                            type="checkbox"
                            name="trust_device"
                            value="1"
                        />
                        <span class="checkmark"></span>
                        <span class="checkbox-text">
                            <i class="fas fa-shield-alt"></i>
                            Dieses Gerät merken (keine erneute Anmeldung erforderlich)
                        </span>
                    </label>
                </div>

                <button type="submit" class="btn-login">
                    <i class="fas fa-sign-in-alt"></i>
                    Anmelden
                </button>
            </form>
            <?php else: ?>
            <!-- 2FA Formular -->
            <form method="POST" class="login-form" id="2fa-form">
                <input type="hidden" name="action" value="verify_2fa">
                
                <div class="tfa-info">
                    <div class="tfa-icon">
                        <i class="fab fa-whatsapp"></i>
                    </div>
                    <h2>Zwei-Faktor-Authentifizierung</h2>
                    <p class="tfa-message">Ein 6-stelliger Code wurde per WhatsApp gesendet</p>
                    <?php if ($phone_hint): ?>
                        <p class="tfa-hint">Gesendet an: ***<?php echo htmlspecialchars($phone_hint); ?></p>
                    <?php endif; ?>
                </div>

                <div class="form-group">
                    <label for="code">
                        <i class="fas fa-key"></i>
                        Sicherheitscode
                    </label>
                    <input
                        type="text"
                        id="code"
                        name="code"
                        placeholder="6-stelliger Code"
                        required
                        autofocus
                        maxlength="6"
                        pattern="[0-9]{6}"
                        class="code-input"
                        oninput="this.value = this.value.replace(/\D/g, '').slice(0, 6)"
                    />
                </div>

                <div class="form-group checkbox-group">
                    <label class="checkbox-label">
                        <input
                            type="checkbox"
                            name="trust_device"
                            value="1"
                            checked
                        />
                        <span class="checkmark"></span>
                        <span class="checkbox-text">
                            <i class="fas fa-shield-alt"></i>
                            Gerät merken
                        </span>
                    </label>
                </div>

                <button type="submit" class="btn-login" id="verify-btn">
                    <i class="fas fa-check"></i>
                    Bestätigen
                </button>

                <div class="tfa-actions">
                    <button 
                        type="submit" 
                        formaction="login.php" 
                        formmethod="POST" 
                        name="action" 
                        value="resend_code"
                        class="btn-link"
                    >
                        Code erneut senden
                    </button>
                    <a href="login.php" class="btn-link">
                        Zurück zum Login
                    </a>
                </div>
            </form>
            <?php endif; ?>

            <div class="login-footer">
                <p>Geschützer Bereich - Nur für autorisierte Mitarbeiter</p>
            </div>
        </div>
    </div>

    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Oxygen', 'Ubuntu', 'Cantarell', 'Fira Sans', 'Droid Sans', 'Helvetica Neue', sans-serif;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        .login-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, #0a1628 0%, #1a2942 50%, #243b55 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 9999;
            padding: 1rem;
        }

        .login-modal {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 24px;
            padding: 2.5rem;
            width: 100%;
            max-width: 420px;
            box-shadow: 0 25px 80px rgba(0, 0, 0, 0.4), 0 0 60px rgba(59, 130, 246, 0.15);
            backdrop-filter: blur(20px);
            animation: slideUp 0.4s ease-out;
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .login-header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .login-logo {
            width: 72px;
            height: 72px;
            background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
            border-radius: 18px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.35);
        }

        .login-logo i {
            font-size: 32px;
            color: white;
        }

        .login-header h1 {
            font-size: 1.75rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0 0 0.25rem;
        }

        .login-subtitle {
            color: #64748b;
            font-size: 0.95rem;
            margin: 0;
        }

        .login-form {
            display: flex;
            flex-direction: column;
            gap: 1.25rem;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .form-group label {
            font-size: 0.875rem;
            font-weight: 600;
            color: #374151;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .form-group label i {
            color: #6b7280;
            width: 16px;
        }

        .form-group input[type="text"],
        .form-group input[type="password"] {
            padding: 0.875rem 1rem;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            font-size: 1rem;
            transition: all 0.2s ease;
            background: white;
        }

        .form-group input:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 4px rgba(59, 130, 246, 0.1);
        }

        .code-input {
            text-align: center;
            font-size: 1.5rem !important;
            letter-spacing: 0.5rem;
            font-weight: 700;
        }

        .checkbox-group {
            margin-top: 0.5rem;
        }

        .checkbox-label {
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
            cursor: pointer;
            user-select: none;
        }

        .checkbox-label input {
            display: none;
        }

        .checkmark {
            width: 22px;
            height: 22px;
            border: 2px solid #d1d5db;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
            flex-shrink: 0;
        }

        .checkbox-label input:checked + .checkmark {
            background: #3b82f6;
            border-color: #3b82f6;
        }

        .checkbox-label input:checked + .checkmark::after {
            content: '✓';
            color: white;
            font-size: 14px;
            font-weight: bold;
        }

        .checkbox-text {
            font-size: 0.9rem;
            color: #4b5563;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .checkbox-text i {
            color: #10b981;
        }

        .btn-login {
            padding: 1rem;
            background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            transition: all 0.2s ease;
            box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
            margin-top: 0.5rem;
        }

        .btn-login:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.4);
        }

        .btn-login:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .login-error {
            background: #fef2f2;
            color: #dc2626;
            padding: 0.875rem 1rem;
            border-radius: 10px;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            border: 1px solid #fecaca;
            margin-bottom: 1rem;
        }

        .tfa-info {
            text-align: center;
            margin-bottom: 1.5rem;
        }

        .tfa-icon {
            width: 64px;
            height: 64px;
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
        }

        .tfa-icon i {
            font-size: 28px;
            color: white;
        }

        .tfa-info h2 {
            font-size: 1.25rem;
            color: #1e293b;
            margin: 0 0 0.5rem;
        }

        .tfa-message {
            color: #64748b;
            margin: 0;
        }

        .tfa-hint {
            color: #6b7280;
            font-size: 0.85rem;
            margin: 0.5rem 0 0;
        }

        .tfa-actions {
            display: flex;
            justify-content: space-between;
            gap: 1rem;
            margin-top: 1rem;
        }

        .btn-link {
            background: none;
            border: none;
            color: #3b82f6;
            font-size: 0.9rem;
            cursor: pointer;
            padding: 0.5rem;
            transition: color 0.2s;
            text-decoration: none;
        }

        .btn-link:hover:not(:disabled) {
            color: #1d4ed8;
            text-decoration: underline;
        }

        .btn-link:disabled {
            color: #9ca3af;
            cursor: not-allowed;
        }

        .login-footer {
            text-align: center;
            margin-top: 2rem;
            padding-top: 1.5rem;
            border-top: 1px solid #e5e7eb;
        }

        .login-footer p {
            color: #9ca3af;
            font-size: 0.8rem;
            margin: 0;
        }

        @media (max-width: 480px) {
            .login-modal {
                padding: 1.5rem;
                margin: 0.5rem;
            }

            .login-header h1 {
                font-size: 1.5rem;
            }

            .tfa-actions {
                flex-direction: column;
                gap: 0.5rem;
            }
        }
    </style>

    <script>
        // Code-Input: Nur Zahlen, max 6 Zeichen
        document.addEventListener('DOMContentLoaded', function() {
            const codeInput = document.getElementById('code');
            if (codeInput) {
                codeInput.addEventListener('input', function(e) {
                    this.value = this.value.replace(/\D/g, '').slice(0, 6);
                    const verifyBtn = document.getElementById('verify-btn');
                    if (verifyBtn) {
                        verifyBtn.disabled = this.value.length !== 6;
                    }
                });
            }
        });
    </script>
</body>
</html>
