<?php
/**
 * 🚀 INTELLIGENT ADMIN DASHBOARD - Next Generation
 * Ultra-modernes Dashboard mit KI-gestützten Empfehlungen und individueller Konfiguration
 */

session_start();
require_once '../config/database_optimized.php';
require_once 'includes/helpers.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$current_user_autohaus_name = null;

if ($current_user_id) {
    $user_data = $db->getUserData($current_user_id);
    if ($user_data) {
        $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
        $current_user_role = $user_data['role'] ?? 'admin';
        $current_user_autohaus_id = $user_data['autohaus_id'];
        $current_user_autohaus_name = $user_data['autohaus_name'];
    }
}

// Check if user needs dashboard setup
$needs_setup = false;
$user_dashboard_config = null;
$show_setup_modal = false;

if ($current_user_id) {
    // First check if user has the needs_dashboard_setup flag
    $stmt = $conn->prepare("SELECT needs_dashboard_setup FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $user_needs_setup = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($user_needs_setup && $user_needs_setup['needs_dashboard_setup'] == 1) {
        $needs_setup = true;
        $show_setup_modal = true;
    } else {
        // Check if user has dashboard configuration
        $stmt = $conn->prepare("SELECT config_data FROM user_dashboard_config WHERE user_id = ?");
        $stmt->execute([$current_user_id]);
        $result = $stmt->fetch();
        
        if ($result) {
            $user_dashboard_config = json_decode($result['config_data'], true);
        } else {
            $needs_setup = true;
            $show_setup_modal = true;
        }
    }
}

// Handle storage autohaus filter change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_storage_autohaus_filter') {
    if (isset($_POST['autohaus_id'])) {
        $autohaus_id = $_POST['autohaus_id'];
        // Handle empty string (show all)
        if ($autohaus_id === '' || $autohaus_id === null) {
            unset($_SESSION['dashboard_storage_autohaus_id']);
            $autohaus_id = null;
        } else {
            $autohaus_id = (int)$autohaus_id;
            $_SESSION['dashboard_storage_autohaus_id'] = $autohaus_id;
        }
        
        // Debug
        error_log("Storage Filter Updated: " . ($autohaus_id ?? 'all'));
        
        // Return JSON response for AJAX
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'autohaus_id' => $autohaus_id]);
        exit;
    }
}

// Handle setup form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_dashboard_config') {
    try {
        $config = [
            'layout' => $_POST['layout'] ?? 'default',
            'widgets' => json_decode($_POST['widgets'] ?? '[]', true),
            'quick_actions' => [], // Quick Actions are now managed via quick_actions_manager.php
            'show_welcome' => (isset($_POST['show_welcome']) && $_POST['show_welcome'] == '1') ? true : false,
            'auto_refresh' => (isset($_POST['auto_refresh']) && $_POST['auto_refresh'] == '1') ? true : false,
            'refresh_interval' => intval($_POST['refresh_interval'] ?? 300)
        ];
        
        $stmt = $conn->prepare("
            INSERT INTO user_dashboard_config (user_id, config_data, created_at, updated_at) 
            VALUES (?, ?, NOW(), NOW())
            ON DUPLICATE KEY UPDATE 
            config_data = VALUES(config_data), updated_at = NOW()
        ");
        $stmt->execute([$current_user_id, json_encode($config)]);
        
        $stmt = $conn->prepare("UPDATE admin_users SET needs_dashboard_setup = 0 WHERE id = ?");
        $stmt->execute([$current_user_id]);
        
        // Reload page to show new configuration
        header('Location: index.php');
        exit;
    } catch (Exception $e) {
        $error_message = 'Fehler: ' . $e->getMessage();
    }
}

// Get user's dashboard configuration
$dashboard_layout = $user_dashboard_config['layout'] ?? 'default';
$enabled_widgets = $user_dashboard_config['widgets'] ?? ['today_appointments', 'upcoming_events', 'quick_actions', 'notifications', 'activity_feed', 'whatsapp_stats', 'whatsapp_activity'];
// Quick Actions are now managed via quick_actions_manager.php
$enabled_quick_actions = [];
$show_welcome = $user_dashboard_config['show_welcome'] ?? true;
$auto_refresh = $user_dashboard_config['auto_refresh'] ?? false;
$refresh_interval = $user_dashboard_config['refresh_interval'] ?? 300;

// Get widget order from config
$widget_order = $user_dashboard_config['widget_order'] ?? [];

// Get notifications count and data
$notifications_count = 0;
$notifications = [];
$upcoming_appointments_count = 0;
$events_soon_count = 0;
$upcoming_appointments_data = [];
$events_soon_data = [];
$all_notifications = [];

// Benachrichtigungen immer laden (nicht mehr abhängig von Widget)
try {
    // Get upcoming appointments in next 24 hours with details - gefiltert nach Autohaus
    $appointments_query = "
        SELECT COUNT(*) as count
        FROM appointments 
        WHERE slot_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
        AND status IN ('booked', 'vorbereitet', 'completed')
    ";
    $appointments_params = [];
    
    if ($current_user_autohaus_id) {
        $appointments_query .= " AND autohaus_id = ?";
        $appointments_params[] = $current_user_autohaus_id;
    }
    
    $upcoming_appointments_result = $db->fetchAll($appointments_query, $appointments_params);
    $upcoming_appointments_count = intval($upcoming_appointments_result[0]['count'] ?? 0);
    $notifications_count += $upcoming_appointments_count;
    
    // Get detailed appointment data (ALLE Termine, gefiltert nach Autohaus)
    if ($upcoming_appointments_count > 0) {
        $appointments_detail_query = "
            SELECT 
                id,
                customer_name, 
                license_plate, 
                slot_time,
                slot_date,
                status,
                autohaus_id,
                'appointment' as notification_type
            FROM appointments 
            WHERE slot_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
            AND status IN ('booked', 'vorbereitet', 'completed')
        ";
        $appointments_detail_params = [];
        
        if ($current_user_autohaus_id) {
            $appointments_detail_query .= " AND autohaus_id = ?";
            $appointments_detail_params[] = $current_user_autohaus_id;
        }
        
        $appointments_detail_query .= " ORDER BY slot_time ASC";
        $upcoming_appointments_data = $db->fetchAll($appointments_detail_query, $appointments_detail_params);
    }
    
    // Get events starting soon - gefiltert nach Autohaus (wenn Events Autohaus haben)
    $events_query = "
        SELECT COUNT(*) as count
        FROM events 
        WHERE event_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
        AND is_active = 1
    ";
    $events_params = [];
    
    // Prüfe ob Events-Tabelle autohaus_id hat
    try {
        $check_events_autohaus = $db->fetchAll("SHOW COLUMNS FROM events LIKE 'autohaus_id'");
        if (!empty($check_events_autohaus) && $current_user_autohaus_id) {
            $events_query .= " AND autohaus_id = ?";
            $events_params[] = $current_user_autohaus_id;
        }
    } catch (Exception $e) {
        // Spalte existiert nicht, keine Filterung
    }
    
    $events_soon_result = $db->fetchAll($events_query, $events_params);
    $events_soon_count = intval($events_soon_result[0]['count'] ?? 0);
    $notifications_count += $events_soon_count;
    
    // Get detailed event data (ALLE Events, gefiltert nach Autohaus)
    if ($events_soon_count > 0) {
        $events_detail_query = "
            SELECT 
                id, 
                name, 
                event_date,
                'event' as notification_type
            FROM events 
            WHERE event_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
            AND is_active = 1
        ";
        $events_detail_params = [];
        
        // Prüfe ob Events-Tabelle autohaus_id hat
        try {
            $check_events_autohaus = $db->fetchAll("SHOW COLUMNS FROM events LIKE 'autohaus_id'");
            if (!empty($check_events_autohaus) && $current_user_autohaus_id) {
                $events_detail_query .= " AND autohaus_id = ?";
                $events_detail_params[] = $current_user_autohaus_id;
            }
        } catch (Exception $e) {
            // Spalte existiert nicht, keine Filterung
        }
        
        $events_detail_query .= " ORDER BY event_date ASC";
        $events_soon_data = $db->fetchAll($events_detail_query, $events_detail_params);
    }
    
    // Kombiniere alle Benachrichtigungen in einem Array
    $all_notifications = [];
    if (!empty($upcoming_appointments_data)) {
        foreach ($upcoming_appointments_data as $apt) {
            $all_notifications[] = $apt;
        }
    }
    if (!empty($events_soon_data)) {
        foreach ($events_soon_data as $evt) {
            $all_notifications[] = $evt;
        }
    }
    
    // Sortiere nach Datum/Zeit
    usort($all_notifications, function($a, $b) {
        $dateA = $a['notification_type'] === 'appointment' 
            ? ($a['slot_date'] . ' ' . $a['slot_time']) 
            : $a['event_date'];
        $dateB = $b['notification_type'] === 'appointment' 
            ? ($b['slot_date'] . ' ' . $b['slot_time']) 
            : $b['event_date'];
        return strtotime($dateA) - strtotime($dateB);
    });
} catch (Exception $e) {
    error_log("Error loading notifications: " . $e->getMessage());
    $upcoming_appointments_count = 0;
    $events_soon_count = 0;
    $notifications_count = 0;
    $all_notifications = [];
}

// Get recent activity for activity feed
$recent_activity = [];
if (isWidgetEnabled('activity_feed', $user_dashboard_config)) {
    $recent_activity = $db->fetchAll("
        SELECT 
            'appointment' as type,
            CONCAT('Termin erstellt: ', customer_name) as description,
            created_at as timestamp
        FROM appointments 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        ORDER BY created_at DESC
        LIMIT 10
    ");
}

// Get quick stats data
$quick_stats_data = [];
if (isWidgetEnabled('quick_stats', $user_dashboard_config)) {
    // Get this week's appointments
    $this_week_appointments = $db->fetchAll("
        SELECT COUNT(*) as count
        FROM appointments 
        WHERE slot_date >= DATE_SUB(CURDATE(), INTERVAL WEEKDAY(CURDATE()) DAY)
        AND slot_date < DATE_ADD(CURDATE(), INTERVAL 7 - WEEKDAY(CURDATE()) DAY)
    ");
    
    // Get this month's appointments
    $this_month_appointments = $db->fetchAll("
        SELECT COUNT(*) as count
        FROM appointments 
        WHERE MONTH(slot_date) = MONTH(CURDATE())
        AND YEAR(slot_date) = YEAR(CURDATE())
    ");
    
    // Get active events
    $active_events_count = $db->fetchAll("
        SELECT COUNT(*) as count
        FROM events 
        WHERE is_active = 1
        AND event_date >= CURDATE()
    ");
    
    $quick_stats_data = [
        'this_week' => intval($this_week_appointments[0]['count'] ?? 0),
        'this_month' => intval($this_month_appointments[0]['count'] ?? 0),
        'active_events' => intval($active_events_count[0]['count'] ?? 0)
    ];
}

// Get task reminders data from dashboard_tasks table
$task_reminders = [];
$user_tasks = [];
if (isWidgetEnabled('task_reminders', $user_dashboard_config)) {
    try {
        // Check if dashboard_tasks table exists
        $table_exists = $db->fetchAll("
            SELECT COUNT(*) as count 
            FROM information_schema.tables 
            WHERE table_schema = DATABASE() 
            AND table_name = 'dashboard_tasks'
        ");
        
        if (intval($table_exists[0]['count'] ?? 0) > 0) {
            // Get user's tasks
            $user_tasks = $db->fetchAll("
                SELECT t.*, 
                       u1.username as assigned_to_name,
                       u2.username as created_by_name
                FROM dashboard_tasks t
                LEFT JOIN admin_users u1 ON t.assigned_to = u1.id
                LEFT JOIN admin_users u2 ON t.created_by = u2.id
                WHERE t.is_active = 1
                AND t.status IN ('pending', 'in_progress')
                AND (t.user_id IS NULL OR t.user_id = ? OR t.assigned_to = ?)
                AND (t.due_date IS NULL OR t.due_date >= CURDATE())
                ORDER BY 
                    CASE t.priority 
                        WHEN 'urgent' THEN 1
                        WHEN 'high' THEN 2
                        WHEN 'medium' THEN 3
                        WHEN 'low' THEN 4
                    END,
                    t.due_date ASC,
                    t.due_time ASC
                LIMIT 10
            ", [$current_user_id, $current_user_id]);
        }
        
        // Get appointments needing attention (tomorrow)
        $tomorrow_appointments = $db->fetchAll("
            SELECT COUNT(*) as count
            FROM appointments 
            WHERE slot_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
            AND status IN ('booked', 'vorbereitet')
        ");
        
        // Get upcoming events this week
        $upcoming_events_week = $db->fetchAll("
            SELECT COUNT(*) as count
            FROM events 
            WHERE event_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
            AND is_active = 1
        ");
        
        $task_reminders = [
            'tomorrow_appointments' => intval($tomorrow_appointments[0]['count'] ?? 0),
            'upcoming_events' => intval($upcoming_events_week[0]['count'] ?? 0),
            'tasks_count' => count($user_tasks)
        ];
    } catch (Exception $e) {
        error_log("Error loading task reminders: " . $e->getMessage());
        $task_reminders = [
            'tomorrow_appointments' => 0,
            'upcoming_events' => 0,
            'tasks_count' => 0
        ];
        $user_tasks = [];
    }
}

// Get system health data
$system_health = [];
if (isWidgetEnabled('system_health', $user_dashboard_config)) {
    // Get total appointments
    $total_appointments = $db->fetchAll("SELECT COUNT(*) as count FROM appointments");
    
    // Get total events
    $total_events = $db->fetchAll("SELECT COUNT(*) as count FROM events WHERE is_active = 1");
    
    // Get total users
    $total_users = $db->fetchAll("SELECT COUNT(*) as count FROM admin_users WHERE is_active = 1");
    
    // Get recent errors (if error log exists)
    $system_health = [
        'total_appointments' => intval($total_appointments[0]['count'] ?? 0),
        'total_events' => intval($total_events[0]['count'] ?? 0),
        'total_users' => intval($total_users[0]['count'] ?? 0),
        'status' => 'healthy' // Can be enhanced with actual system checks
    ];
}

// Get WhatsApp statistics data
$whatsapp_stats = [];
$whatsapp_recent_logs = [];
$whatsapp_template_status = [];

if (isWidgetEnabled('whatsapp_stats', $user_dashboard_config) || 
    isWidgetEnabled('whatsapp_activity', $user_dashboard_config) || 
    isWidgetEnabled('whatsapp_templates', $user_dashboard_config)) {
    try {
        // Check if whatsapp_logs table exists
        $tableCheck = $db->fetchAll("
            SELECT COUNT(*) as count 
            FROM information_schema.tables 
            WHERE table_schema = DATABASE() 
            AND table_name = 'whatsapp_logs'
        ");
        
        if (intval($tableCheck[0]['count'] ?? 0) > 0) {
            // Get WhatsApp statistics
            if (isWidgetEnabled('whatsapp_stats', $user_dashboard_config)) {
                $whatsappStatsParams = [];
                $whatsappStatsQuery = "
                    SELECT 
                        COUNT(*) as total,
                        SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) as successful,
                        SUM(CASE WHEN status = 'error' OR status = 'failed' THEN 1 ELSE 0 END) as failed,
                        SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today,
                        SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as this_week,
                        SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as this_month
                    FROM whatsapp_logs
                    WHERE 1=1
                ";
                
                if ($current_user_autohaus_id) {
                    $whatsappStatsQuery .= " AND autohaus_id = ?";
                    $whatsappStatsParams[] = intval($current_user_autohaus_id);
                }
                
                $whatsappStatsResult = $db->fetchAll($whatsappStatsQuery, $whatsappStatsParams);
                if (!empty($whatsappStatsResult)) {
                    $stats = $whatsappStatsResult[0];
                    $total = intval($stats['total'] ?? 0);
                    $successful = intval($stats['successful'] ?? 0);
                    $success_rate = $total > 0 ? round(($successful / $total) * 100, 1) : 0;
                    
                    $whatsapp_stats = [
                        'total' => $total,
                        'successful' => $successful,
                        'failed' => intval($stats['failed'] ?? 0),
                        'today' => intval($stats['today'] ?? 0),
                        'this_week' => intval($stats['this_week'] ?? 0),
                        'this_month' => intval($stats['this_month'] ?? 0),
                        'success_rate' => $success_rate
                    ];
                }
            }
            
            // Get recent WhatsApp logs
            if (isWidgetEnabled('whatsapp_activity', $user_dashboard_config)) {
                $whatsappLogsParams = [];
                $whatsappLogsQuery = "
                    SELECT wl.*, a.name as autohaus_name
                    FROM whatsapp_logs wl
                    LEFT JOIN autohaus a ON wl.autohaus_id = a.id
                    WHERE 1=1
                ";
                
                if ($current_user_autohaus_id) {
                    $whatsappLogsQuery .= " AND wl.autohaus_id = ?";
                    $whatsappLogsParams[] = intval($current_user_autohaus_id);
                }
                
                $whatsappLogsQuery .= " ORDER BY wl.created_at DESC LIMIT 10";
                $whatsapp_recent_logs = $db->fetchAll($whatsappLogsQuery, $whatsappLogsParams);
            }
        }
        
        // Get WhatsApp template status
        if (isWidgetEnabled('whatsapp_templates', $user_dashboard_config)) {
            $templateStatusParams = [];
            $templateStatusQuery = "
                SELECT 
                    action_type,
                    COUNT(*) as count,
                    SUM(CASE WHEN is_active = 1 AND is_enabled = 1 THEN 1 ELSE 0 END) as active_enabled,
                    SUM(CASE WHEN is_active = 1 AND is_enabled = 0 THEN 1 ELSE 0 END) as active_disabled,
                    SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive
                FROM whatsapp_template_configs
                WHERE 1=1
            ";
            
            if ($current_user_autohaus_id) {
                $templateStatusQuery .= " AND autohaus_id = ?";
                $templateStatusParams[] = intval($current_user_autohaus_id);
            }
            
            $templateStatusQuery .= " GROUP BY action_type";
            $templateStatusResult = $db->fetchAll($templateStatusQuery, $templateStatusParams);
            
            $whatsapp_template_status = [
                'total_configs' => 0,
                'active_enabled' => 0,
                'active_disabled' => 0,
                'inactive' => 0,
                'by_action' => []
            ];
            
            foreach ($templateStatusResult as $row) {
                $whatsapp_template_status['total_configs'] += intval($row['count'] ?? 0);
                $whatsapp_template_status['active_enabled'] += intval($row['active_enabled'] ?? 0);
                $whatsapp_template_status['active_disabled'] += intval($row['active_disabled'] ?? 0);
                $whatsapp_template_status['inactive'] += intval($row['inactive'] ?? 0);
                $whatsapp_template_status['by_action'][$row['action_type']] = [
                    'total' => intval($row['count'] ?? 0),
                    'active_enabled' => intval($row['active_enabled'] ?? 0),
                    'active_disabled' => intval($row['active_disabled'] ?? 0),
                    'inactive' => intval($row['inactive'] ?? 0)
                ];
            }
        }
    } catch (Exception $e) {
        error_log("Error loading WhatsApp data: " . $e->getMessage());
        $whatsapp_stats = [];
        $whatsapp_recent_logs = [];
        $whatsapp_template_status = [];
    }
}


// Helper function to check if widget is enabled
function isWidgetEnabled($widget_key, $user_config) {
    if (!isset($user_config['widgets']) || !is_array($user_config['widgets'])) {
        return false;
    }
    return in_array($widget_key, $user_config['widgets']);
}

// Get dashboard statistics
$stats = $db->getDashboardStats($current_user_autohaus_id, 'admin', true);

// Get today's appointments
$todayAppointments = $db->fetchAll("
    SELECT a.*, e.name as event_name
    FROM appointments a
    LEFT JOIN events e ON a.event_id = e.id
    WHERE a.slot_date = CURDATE()
    ORDER BY a.slot_time ASC
    LIMIT 10
");

// Get upcoming events
$upcomingEvents = $db->fetchAll("
    SELECT e.*, 
           COUNT(a.id) as appointment_count,
           COUNT(CASE WHEN a.status = 'booked' THEN 1 END) as booked_count
    FROM events e
    LEFT JOIN appointments a ON e.id = a.event_id
    WHERE e.event_date >= CURDATE() AND e.is_active = 1
    GROUP BY e.id
    ORDER BY e.event_date ASC
    LIMIT 5
");

// Get performance metrics
// REMOVED: Performance metrics query has been removed as requested

// Get recent storage activities - filtered by autohaus_id
// NOTE: Storage data is loaded via API (RESY), not from database
// The data will be loaded via JavaScript/AJAX in the widget
$selected_autohaus_id = null;

// Check if user has autohaus_id assigned
if ($current_user_autohaus_id) {
    // User has autohaus assigned, use it
    $selected_autohaus_id = $current_user_autohaus_id;
} else {
    // User has no autohaus assigned, check session for selected autohaus
    if (isset($_SESSION['dashboard_storage_autohaus_id'])) {
        $selected_autohaus_id = $_SESSION['dashboard_storage_autohaus_id'];
    }
    // If still no autohaus selected, show all (or first one)
}

// Storage data will be loaded via JavaScript from API
// Set empty array for now - will be populated by AJAX
$recentStorage = [];

// Get all autohauser for dropdown (if user has no autohaus assigned)
$all_autohauser = [];
if (!$current_user_autohaus_id) {
    $stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE is_active = 1 ORDER BY name ASC");
    $stmt->execute();
    $all_autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Get Quick Actions from database
// Get user's selected quick actions from quick_actions_manager
$quick_actions = [];
if ($current_user_id) {
    // Load user's personal quick actions first
    $stmt = $conn->prepare("
        SELECT * FROM quick_actions 
        WHERE user_id = ? AND is_active = 1
        ORDER BY position_order ASC, id ASC
    ");
    $stmt->execute([$current_user_id]);
    $user_quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Load global quick actions (user_id IS NULL)
    $stmt = $conn->prepare("
        SELECT * FROM quick_actions 
        WHERE user_id IS NULL AND is_active = 1
        ORDER BY position_order ASC, id ASC
    ");
    $stmt->execute();
    $global_quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Combine user actions first, then global actions
    $quick_actions = array_merge($user_quick_actions, $global_quick_actions);
} else {
    // Fallback: Load all active quick actions
    $stmt = $conn->prepare("SELECT * FROM quick_actions WHERE is_active = 1 ORDER BY position_order ASC");
    $stmt->execute();
    $quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Intelligent User Scoring System
// REMOVED: calculateIntelligentScore function has been removed as requested

// Calculate intelligent user score for current user (Deniz Uludag)
// REMOVED: Scoring system has been removed as requested

// Get user activity log
$userActivity = $db->fetchAll("
    SELECT 
        'appointment' as type,
        'Termin erstellt' as action,
        customer_name as description,
        created_at
    FROM appointments 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    ORDER BY created_at DESC
    LIMIT 10
");

// Setup data for modal
$available_widgets = [
    'today_appointments' => [
        'title' => 'Heutige Termine',
        'description' => 'Zeigt alle Termine für heute an',
        'icon' => 'fas fa-calendar-day',
        'color' => '#007AFF',
        'category' => 'essential',
        'importance' => 'high'
    ],
    'upcoming_events' => [
        'title' => 'Kommende Events',
        'description' => 'Zeigt die nächsten Events an',
        'icon' => 'fas fa-calendar-alt',
        'color' => '#34C759',
        'category' => 'operational',
        'importance' => 'medium'
    ],
    'quick_actions' => [
        'title' => 'Schnellaktionen',
        'description' => 'Ihre personalisierten Schnellzugriffe',
        'icon' => 'fas fa-rocket',
        'color' => '#AF52DE',
        'category' => 'essential',
        'importance' => 'high'
    ],
    'recent_storage' => [
        'title' => 'Letzte Lageraktivitäten',
        'description' => 'Zeigt die neuesten Einlagerungen an',
        'icon' => 'fas fa-warehouse',
        'color' => '#FF9500',
        'category' => 'operational',
        'importance' => 'medium'
    ],
    'notifications' => [
        'title' => 'Benachrichtigungen',
        'description' => 'Wichtige Updates und Erinnerungen',
        'icon' => 'fas fa-bell',
        'color' => '#EF4444',
        'category' => 'essential',
        'importance' => 'high'
    ],
    'activity_feed' => [
        'title' => 'Aktivitäts-Feed',
        'description' => 'Zeigt Ihre letzten Aktivitäten an',
        'icon' => 'fas fa-stream',
        'color' => '#06B6D4',
        'category' => 'operational',
        'importance' => 'medium'
    ],
    'smart_recommendations' => [
        'title' => 'KI-Empfehlungen',
        'description' => 'Intelligente Vorschläge basierend auf Ihrem Verhalten',
        'icon' => 'fas fa-brain',
        'color' => '#8B5CF6',
        'category' => 'analytics',
        'importance' => 'low'
    ],
    'quick_stats' => [
        'title' => 'Schnellstatistiken',
        'description' => 'Wichtige Kennzahlen auf einen Blick',
        'icon' => 'fas fa-chart-pie',
        'color' => '#10B981',
        'category' => 'analytics',
        'importance' => 'medium'
    ],
    'task_reminders' => [
        'title' => 'Aufgaben & Erinnerungen',
        'description' => 'Ihre offenen Aufgaben und Termine',
        'icon' => 'fas fa-tasks',
        'color' => '#F59E0B',
        'category' => 'essential',
        'importance' => 'high'
    ],
    'system_health' => [
        'title' => 'System-Status',
        'description' => 'Übersicht über Systemzustand und Performance',
        'icon' => 'fas fa-heartbeat',
        'color' => '#6366F1',
        'category' => 'analytics',
        'importance' => 'low'
    ],
    'whatsapp_stats' => [
        'title' => 'WhatsApp-Statistiken',
        'description' => 'Übersicht über gesendete Nachrichten und Erfolgsrate',
        'icon' => 'fab fa-whatsapp',
        'color' => '#25D366',
        'category' => 'operational',
        'importance' => 'high'
    ],
    'whatsapp_activity' => [
        'title' => 'WhatsApp-Aktivitäten',
        'description' => 'Letzte gesendete WhatsApp-Nachrichten',
        'icon' => 'fas fa-paper-plane',
        'color' => '#128C7E',
        'category' => 'operational',
        'importance' => 'medium'
    ],
    'whatsapp_templates' => [
        'title' => 'WhatsApp-Template-Status',
        'description' => 'Status der zugewiesenen Templates',
        'icon' => 'fas fa-file-alt',
        'color' => '#075E54',
        'category' => 'operational',
        'importance' => 'medium'
    ]
];

$available_layouts = [
    'default' => [
        'name' => 'Standard Layout',
        'description' => 'Zwei-Spalten Layout mit Hauptinhalt und Sidebar',
        'best_for' => 'Tägliche Arbeit, ausgewogene Übersicht',
        'screen_size' => 'Desktop & Tablet'
    ],
    'compact' => [
        'name' => 'Kompaktes Layout',
        'description' => 'Ein-Spalten Layout für fokussierte Arbeit',
        'best_for' => 'Mobile Geräte, fokussierte Aufgaben',
        'screen_size' => 'Mobile & Tablet'
    ],
    'wide' => [
        'name' => 'Breites Layout',
        'description' => 'Drei-Spalten Layout für maximale Übersicht',
        'best_for' => 'Administratoren, große Bildschirme',
        'screen_size' => 'Desktop & Ultrawide'
    ]
];

// REMOVED: Quick action templates are now managed via quick_actions_manager.php

// Default config for new users
if (!$user_dashboard_config) {
    $user_dashboard_config = [
        'layout' => 'default',
        'widgets' => ['today_appointments', 'upcoming_events', 'quick_actions', 'notifications', 'activity_feed', 'whatsapp_stats', 'whatsapp_activity'],
        'quick_actions' => [], // Quick Actions are now managed via quick_actions_manager.php
        'show_welcome' => true,
        'auto_refresh' => false,
        'refresh_interval' => 300
    ];
}

// Ensure user_dashboard_config is always available for the modal
$modal_config = $user_dashboard_config;
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <meta name="robots" content="noindex, nofollow">
    <title>🚀 Dashboard • Boxenstop</title>
    
    <!-- Ultra Modern Design System -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    
    <style>
        /* 🚀 ULTRA MODERN DESIGN SYSTEM - 2024 */
        :root {
            /* Modern Color Palette */
            --primary: #6366f1;
            --primary-dark: #4f46e5;
            --secondary: #8b5cf6;
            --accent: #06b6d4;
            --success: #10b981;
            --warning: #f59e0b;
            --error: #ef4444;
            
            /* Neutral Colors */
            --gray-50: #f9fafb;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-300: #d1d5db;
            --gray-400: #9ca3af;
            --gray-500: #6b7280;
            --gray-600: #4b5563;
            --gray-700: #374151;
            --gray-800: #1f2937;
            --gray-900: #111827;
            
            /* Glass Effects */
            --glass-bg: rgba(255, 255, 255, 0.1);
            --glass-border: rgba(255, 255, 255, 0.2);
            --glass-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            
            /* Spacing */
            --space-1: 0.25rem;
            --space-2: 0.5rem;
            --space-3: 0.75rem;
            --space-4: 1rem;
            --space-5: 1.25rem;
            --space-6: 1.5rem;
            --space-8: 2rem;
            --space-10: 2.5rem;
            --space-12: 3rem;
            --space-16: 4rem;
            --space-20: 5rem;
            
            /* Border Radius */
            --radius-sm: 0.375rem;
            --radius-md: 0.5rem;
            --radius-lg: 0.75rem;
            --radius-xl: 1rem;
            --radius-2xl: 1.5rem;
            --radius-3xl: 2rem;
            
            /* Shadows */
            --shadow-sm: 0 1px 2px rgba(0, 0, 0, 0.05);
            --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.1);
            --shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.1);
            --shadow-2xl: 0 25px 50px rgba(0, 0, 0, 0.25);
        }


        /* Reset & Base */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: #1e293b;
            line-height: 1.6;
            overflow-x: hidden;
        }


        /* Main Container */
        .admin-container {
            display: flex;
            min-height: 100vh;
        }

        .admin-main {
            flex: 1;
            padding: var(--space-8);
            background: transparent;
            min-height: 100vh;
            transition: margin-left 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            margin-left: 300px;
        }

        .smart-sidebar.collapsed + .admin-main {
            margin-left: 80px;
        }

        /* Dashboard Layouts */
        .dashboard-grid {
            display: grid;
            gap: var(--space-6);
            min-height: calc(100vh - var(--space-16));
        }

        .dashboard-grid.layout-default {
            grid-template-columns: 1fr;
            grid-template-areas: 
                "header"
                "main";
        }

        .dashboard-grid.layout-compact {
            grid-template-columns: 1fr;
            grid-template-areas: 
                "header"
                "main";
        }

        .dashboard-grid.layout-wide {
            grid-template-columns: 1fr 1fr;
            grid-template-areas: 
                "header header"
                "main main";
        }

        /* Header */
        .dashboard-header {
            grid-area: header;
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-2xl);
            padding: var(--space-8);
            margin-bottom: var(--space-6);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
        }


        .dashboard-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--secondary), var(--accent));
            opacity: 0.8;
        }

        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
            z-index: 1;
        }

        .header-left h1 {
            font-size: clamp(2rem, 4vw, 3rem);
            font-weight: 800;
            color: #1e293b;
            margin-bottom: var(--space-2);
            letter-spacing: -0.02em;
        }


        .header-left p {
            color: #64748b;
            font-size: 1.1rem;
        }


        .user-info {
            display: flex;
            align-items: center;
            gap: var(--space-4);
            background: rgba(255, 255, 255, 0.4);
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-xl);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.3);
        }


        .user-avatar {
            width: 48px;
            height: 48px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            font-weight: 700;
        }

        .user-details h3 {
            color: #1e293b;
            font-weight: 600;
            margin-bottom: var(--space-1);
        }


        .user-details p {
            color: #64748b;
            font-size: 0.875rem;
        }


        .header-actions {
            display: flex;
            gap: var(--space-3);
        }

        .btn-header {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            padding: var(--space-3) var(--space-5);
            color: #1e293b;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
        }

        .btn-header:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        /* Main Content Area */
        .main-content {
            grid-area: main;
            display: grid;
            gap: var(--space-6);
        }

        .sidebar-content {
            grid-area: sidebar;
            display: grid;
            gap: var(--space-6);
        }

        /* Widget System */
        .widget {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-2xl);
            padding: var(--space-6);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
            transition: all 0.3s ease;
        }


        .widget:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-xl);
        }

        .widget::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .widget:hover::before {
            opacity: 1;
        }

        .widget-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--space-4);
        }

        .widget-title {
            font-size: 1.25rem;
            font-weight: 700;
            color: #1e293b;
            display: flex;
            align-items: center;
            gap: var(--space-3);
        }


        .widget-icon {
            width: 32px;
            height: 32px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1rem;
            color: white;
        }

        .widget-actions {
            display: flex;
            gap: var(--space-2);
        }

        .widget-btn {
            width: 36px;
            height: 36px;
            border: 2px solid rgba(148, 163, 184, 0.3);
            border-radius: var(--radius-lg);
            background: rgba(255, 255, 255, 0.9);
            color: #475569;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.875rem;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            position: relative;
        }

        .widget-btn:hover {
            background: rgba(255, 255, 255, 1);
            border-color: rgba(99, 102, 241, 0.5);
            color: #6366f1;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
        }

        .widget-btn:active {
            transform: translateY(0);
        }


        /* Special styling for move buttons - VISUALLY DISTINCT */
        .widget-btn[onclick*="moveWidgetUp"],
        .widget-btn[onclick*="moveWidgetDown"] {
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.15), rgba(139, 92, 246, 0.15));
            border: 2px solid rgba(99, 102, 241, 0.5);
            color: #6366f1;
            font-weight: 700;
            box-shadow: 0 2px 8px rgba(99, 102, 241, 0.2);
        }
        
        .widget-btn[onclick*="moveWidgetUp"]:hover,
        .widget-btn[onclick*="moveWidgetDown"]:hover {
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.3), rgba(139, 92, 246, 0.3));
            border-color: rgba(99, 102, 241, 0.8);
            color: #4f46e5;
            transform: translateY(-2px) scale(1.05);
            box-shadow: 0 4px 16px rgba(99, 102, 241, 0.4);
        }
        
        .widget-btn[onclick*="moveWidgetUp"]:active,
        .widget-btn[onclick*="moveWidgetDown"]:active {
            transform: translateY(0) scale(1);
        }
        

        /* Tooltip for move buttons */
        .widget-btn[onclick*="moveWidget"]::after {
            content: attr(title);
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%) translateY(-8px);
            background: rgba(0, 0, 0, 0.95);
            color: white;
            padding: 8px 14px;
            border-radius: 8px;
            font-size: 0.75rem;
            white-space: nowrap;
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.3s, transform 0.3s;
            z-index: 1000;
            margin-bottom: 6px;
            font-weight: 600;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
        }
        
        .widget-btn[onclick*="moveWidget"]::before {
            content: '';
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%) translateY(-2px);
            border: 6px solid transparent;
            border-top-color: rgba(0, 0, 0, 0.95);
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.3s, transform 0.3s;
            z-index: 1000;
            margin-bottom: 0;
        }
        
        .widget-btn[onclick*="moveWidget"]:hover::after,
        .widget-btn[onclick*="moveWidget"]:hover::before {
            opacity: 1;
            transform: translateX(-50%) translateY(0);
        }

        /* Statistics Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--space-4);
            margin-bottom: var(--space-6);
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.6);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            text-align: center;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }


        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1), rgba(255, 255, 255, 0.05));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .stat-card:hover::before {
            opacity: 1;
        }

        .stat-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
        }

        .stat-number {
            font-size: 2.5rem;
            font-weight: 800;
            color: #1e293b;
            margin-bottom: var(--space-2);
            position: relative;
            z-index: 1;
        }


        .stat-label {
            color: #64748b;
            font-weight: 600;
            position: relative;
            z-index: 1;
        }


        .stat-icon {
            position: absolute;
            top: var(--space-4);
            right: var(--space-4);
            width: 40px;
            height: 40px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            opacity: 0.3;
        }

        /* Quick Actions */
        .quick-actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: var(--space-4);
        }

        .quick-action {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            text-align: center;
            transition: all 0.3s ease;
            text-decoration: none;
            color: inherit;
            position: relative;
            overflow: hidden;
        }


        .quick-action::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1), rgba(255, 255, 255, 0.05));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .quick-action:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
            text-decoration: none;
            color: inherit;
        }

        .quick-action:hover::before {
            opacity: 1;
        }

        .quick-action-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: white;
            margin: 0 auto var(--space-3);
            position: relative;
            z-index: 1;
        }

        .quick-action-title {
            font-weight: 600;
            color: #1e293b;
            position: relative;
            z-index: 1;
        }


        /* Quick Action Empty State */
        .quick-action-empty {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: var(--space-6);
            text-align: center;
            color: var(--label-secondary);
            border: 2px dashed var(--separator-non-opaque);
            border-radius: var(--radius-lg);
            transition: all 0.2s ease;
        }

        .quick-action-empty:hover {
            border-color: var(--system-blue);
            color: var(--system-blue);
        }

        .quick-action-empty .quick-action-icon {
            margin-bottom: var(--space-3);
            opacity: 0.6;
        }

        .quick-action-empty .quick-action-title {
            font-size: 14px;
            font-weight: 500;
        }

.user-select option {
    background: #1e293b;
    color: white;
    padding: var(--space-2);
}

/* Storage Autohaus Select */
.storage-autohaus-select {
    width: 100%;
    padding: var(--space-3) var(--space-4);
    border: 1px solid rgba(255, 255, 255, 0.3);
    border-radius: var(--radius-md);
    background: rgba(255, 255, 255, 0.6);
    color: #1e293b;
    font-size: 0.875rem;
    cursor: pointer;
    transition: all 0.3s ease;
    backdrop-filter: blur(10px);
}

.storage-autohaus-select:hover {
    border-color: var(--primary);
    background: rgba(255, 255, 255, 0.8);
}

.storage-autohaus-select:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.storage-autohaus-select option {
    background: #1e293b;
    color: white;
    padding: var(--space-2);
}


/* Modal Styles */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100vw;
    height: 100vh;
    background: rgba(0, 0, 0, 0.7);
    backdrop-filter: blur(10px);
    z-index: 10000;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: var(--space-4);
    box-sizing: border-box;
}

.modal-content {
    background: rgba(255, 255, 255, 0.95);
    border-radius: var(--radius-xl);
    border: 1px solid rgba(255, 255, 255, 0.3);
    box-shadow: var(--shadow-xl);
    max-width: 90vw;
    max-height: 90vh;
    width: auto;
    min-width: 600px;
    overflow-y: auto;
    backdrop-filter: blur(20px);
    margin: auto;
    position: relative;
}


.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: var(--space-5);
    border-bottom: 1px solid rgba(255, 255, 255, 0.2);
}

.modal-header h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
}


.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: #64748b;
    cursor: pointer;
    padding: var(--space-2);
    border-radius: var(--radius-md);
    transition: all 0.3s ease;
}

.modal-close:hover {
    background: rgba(255, 255, 255, 0.2);
    color: #1e293b;
}


.modal-form {
    padding: 0;
}

/* Responsive Modal */
@media (max-width: 768px) {
    .modal {
        padding: var(--space-2);
    }
    
    .modal-content {
        min-width: 95vw;
        max-width: 95vw;
        max-height: 95vh;
    }
}

@media (max-width: 480px) {
    .modal-content {
        min-width: 100vw;
        max-width: 100vw;
        max-height: 100vh;
        border-radius: 0;
    }
}

/* Score Details Modal */
.score-details-content {
    padding: var(--space-4);
}

        .score-category {
            background: rgba(255, 255, 255, 0.6);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            margin-bottom: var(--space-4);
            border: 1px solid rgba(255, 255, 255, 0.3);
            transition: all 0.3s ease;
        }

        .score-category:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .category-header {
            display: flex;
            align-items: center;
            gap: var(--space-4);
            margin-bottom: var(--space-4);
        }

        .category-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            flex-shrink: 0;
        }

        .category-info {
            flex: 1;
        }

        .category-info h4 {
            font-size: 1.125rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: var(--space-1);
        }

        .category-info p {
            color: #64748b;
            font-size: 0.875rem;
        }

        .category-score {
            font-size: 1.5rem;
            font-weight: 800;
            color: #1e293b;
            background: rgba(255, 255, 255, 0.8);
            padding: var(--space-2) var(--space-4);
            border-radius: var(--radius-lg);
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        .category-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--space-3);
        }

        .detail-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--space-2) var(--space-3);
            background: rgba(255, 255, 255, 0.4);
            border-radius: var(--radius-md);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .detail-label {
            font-size: 0.875rem;
            color: #64748b;
            font-weight: 500;
        }

        .detail-value {
            font-size: 0.875rem;
            font-weight: 700;
            color: #1e293b;
        }

        .score-tips {
            background: linear-gradient(135deg, #10b981, #059669);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            color: white;
            margin-top: var(--space-6);
        }

        .score-tips h4 {
            font-size: 1.125rem;
            font-weight: 700;
            margin-bottom: var(--space-3);
            display: flex;
            align-items: center;
            gap: var(--space-2);
        }

        .score-tips ul {
            list-style: none;
            padding: 0;
        }

        .score-tips li {
            padding: var(--space-2) 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
        }

        .score-tips li:last-child {
            border-bottom: none;
        }

        .score-tips strong {
            color: rgba(255, 255, 255, 0.9);
        }

        /* Activity Feed */
        .activity-item {
            display: flex;
            align-items: center;
            gap: var(--space-3);
            padding: var(--space-3);
            background: rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            margin-bottom: var(--space-2);
            transition: all 0.3s ease;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .activity-item:hover {
            background: rgba(255, 255, 255, 0.4);
        }

        .activity-icon {
            width: 32px;
            height: 32px;
            border-radius: var(--radius-md);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.875rem;
            color: white;
        }

        .activity-content {
            flex: 1;
        }

        .activity-title {
            color: #1e293b;
            font-weight: 600;
            font-size: 0.875rem;
            margin-bottom: var(--space-1);
        }

        .activity-description {
            color: #64748b;
            font-size: 0.75rem;
        }

        .activity-time {
            color: #94a3b8;
            font-size: 0.75rem;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .admin-main {
                padding: var(--space-4);
            }

            .dashboard-grid.layout-default,
            .dashboard-grid.layout-wide {
                grid-template-columns: 1fr;
                grid-template-areas: 
                    "header"
                    "main"
                    "sidebar"
                    "widgets";
            }

            .header-content {
                flex-direction: column;
                gap: var(--space-4);
                text-align: center;
            }

            .stats-grid {
                grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            }

            .quick-actions-grid {
                grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
            }
        }

        /* Loading States */
        .loading {
            opacity: 0.6;
            pointer-events: none;
        }

        .loading::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 20px;
            height: 20px;
            margin: -10px 0 0 -10px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-top: 2px solid white;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Animations */
        .fade-in {
            animation: fadeIn 0.6s ease-out;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .slide-in {
            animation: slideIn 0.6s ease-out;
        }

        @keyframes slideIn {
            from { opacity: 0; transform: translateX(-20px); }
            to { opacity: 1; transform: translateX(0); }
        }

        /* Header Statistics */
        .header-stats {
            display: flex;
            gap: var(--space-4);
            margin-top: var(--space-4);
            flex-wrap: wrap;
        }

        .header-stat-item {
            display: flex;
            align-items: center;
            gap: var(--space-2);
            background: rgba(255, 255, 255, 0.4);
            padding: var(--space-2) var(--space-4);
            border-radius: var(--radius-lg);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            transition: all 0.3s ease;
        }

        .header-stat-item:hover {
            background: rgba(255, 255, 255, 0.6);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .header-stat-icon {
            width: 32px;
            height: 32px;
            border-radius: var(--radius-md);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.875rem;
            color: white;
        }

        .header-stat-content {
            display: flex;
            flex-direction: column;
        }

        .header-stat-number {
            font-size: 1.25rem;
            font-weight: 700;
            color: #1e293b;
            line-height: 1.2;
        }

        .header-stat-label {
            font-size: 0.75rem;
            color: #64748b;
            font-weight: 500;
        }

        /* Header Quick Actions */
        .header-quick-actions {
            margin-top: var(--space-4);
            padding-top: var(--space-4);
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }


        .notifications-dropdown-body::-webkit-scrollbar {
            width: 8px;
        }

        .notifications-dropdown-body::-webkit-scrollbar-track {
            background: rgba(0, 0, 0, 0.05);
            border-radius: 4px;
        }

        .notifications-dropdown-body::-webkit-scrollbar-thumb {
            background: rgba(0, 0, 0, 0.2);
            border-radius: 4px;
        }

        .notifications-dropdown-body::-webkit-scrollbar-thumb:hover {
            background: rgba(0, 0, 0, 0.3);
        }

        .notification-dropdown-item {
            transition: all 0.2s ease;
        }

        .notification-dropdown-item:hover {
            background: rgba(99, 102, 241, 0.05) !important;
        }

        .notification-filter-btn.active {
            background: var(--primary) !important;
            color: white !important;
            border-color: var(--primary) !important;
        }

        .notification-filter-btn:hover {
            background: rgba(99, 102, 241, 0.1) !important;
        }

        .header-quick-actions-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: #64748b;
            margin-bottom: var(--space-3);
            display: flex;
            align-items: center;
            gap: var(--space-2);
        }

        .header-quick-actions-grid {
            display: flex;
            gap: var(--space-2);
            flex-wrap: wrap;
        }

        .header-quick-action {
            display: flex;
            align-items: center;
            gap: var(--space-2);
            padding: var(--space-2) var(--space-3);
            background: rgba(255, 255, 255, 0.4);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            text-decoration: none;
            color: inherit;
            transition: all 0.3s ease;
            font-size: 0.875rem;
            font-weight: 500;
        }

        .header-quick-action:hover {
            background: rgba(255, 255, 255, 0.6);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
            text-decoration: none;
            color: inherit;
        }

        .header-quick-action-icon {
            width: 24px;
            height: 24px;
            border-radius: var(--radius-md);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.75rem;
            color: white;
        }

        /* Quick Actions Modal Styles */
        .quick-actions-modal {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-2xl);
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }

        .quick-actions-header {
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1));
        }

        .quick-action-card {
            animation: slideIn 0.3s ease-out;
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .quick-action-card:hover .quick-action-drag-handle {
            opacity: 1 !important;
        }

        #quick-actions-search:focus {
            outline: none;
            border-color: rgba(99, 102, 241, 0.5);
            background: rgba(255, 255, 255, 0.15);
        }


        .icon-picker-btn.selected {
            border: 2px solid #3b82f6 !important;
            background: rgba(59, 130, 246, 0.3) !important;
        }

        .color-picker-btn.selected {
            border: 3px solid white !important;
            box-shadow: 0 0 0 2px rgba(255, 255, 255, 0.5) !important;
        }

        /* Setup Modal */
        .setup-modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            backdrop-filter: blur(10px);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: var(--space-4);
            animation: fadeIn 0.3s ease-out;
        }

        .setup-modal {
            background: rgba(255, 255, 255, 0.9);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-2xl);
            max-width: 1000px;
            width: 100%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-2xl);
            position: relative;
            animation: slideIn 0.3s ease-out;
        }


        .setup-modal-header {
            padding: var(--space-8);
            text-align: center;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1));
            position: relative;
            overflow: hidden;
        }

        .setup-modal-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--secondary), var(--accent));
        }

        .setup-modal-header h2 {
            font-size: 2rem;
            font-weight: 800;
            color: #1e293b;
            margin-bottom: var(--space-2);
            position: relative;
            z-index: 1;
        }


        .setup-modal-header p {
            color: #64748b;
            font-size: 1.1rem;
            position: relative;
            z-index: 1;
        }


        .setup-section {
            padding: var(--space-6);
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            transition: background 0.3s ease;
        }

        .setup-section:hover {
            background: rgba(99, 102, 241, 0.05);
        }


        .setup-section:last-child {
            border-bottom: none;
        }

        .setup-section h3 {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: var(--space-4);
            display: flex;
            align-items: center;
            gap: var(--space-3);
        }

        .setup-section h3::before {
            content: '';
            width: 4px;
            height: 24px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            border-radius: 2px;
        }


        .setup-section .layout-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: var(--space-4);
        }

        .setup-section .layout-card {
            background: rgba(255, 255, 255, 0.6);
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            user-select: none;
            pointer-events: auto;
        }


        .setup-section .layout-card:hover {
            border-color: #3b82f6;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.2);
        }

        .setup-section .layout-card:active {
            transform: translateY(0);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .layout-card.selected {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.2);
        }

        .setup-section .layout-card.selected::after {
            content: '✓';
            position: absolute;
            top: var(--space-3);
            right: var(--space-3);
            width: 24px;
            height: 24px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.875rem;
        }

        .setup-section .layout-preview {
            width: 100%;
            height: 80px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: var(--radius-lg);
            margin-bottom: var(--space-3);
            position: relative;
            overflow: hidden;
        }

        /* Layout Preview Visualizations */
        .setup-section .layout-preview.default::before {
            content: '';
            position: absolute;
            top: 10px;
            left: 10px;
            right: 10px;
            bottom: 10px;
            background: linear-gradient(90deg, 
                rgba(59, 130, 246, 0.3) 0%, 
                rgba(59, 130, 246, 0.3) 65%, 
                rgba(59, 130, 246, 0.1) 65%, 
                rgba(59, 130, 246, 0.1) 100%);
            border-radius: var(--radius-sm);
        }

        .setup-section .layout-preview.compact::before {
            content: '';
            position: absolute;
            top: 10px;
            left: 10px;
            right: 10px;
            bottom: 10px;
            background: rgba(59, 130, 246, 0.3);
            border-radius: var(--radius-sm);
        }

        .setup-section .layout-preview.wide::before {
            content: '';
            position: absolute;
            top: 10px;
            left: 10px;
            right: 10px;
            bottom: 10px;
            background: linear-gradient(90deg, 
                rgba(59, 130, 246, 0.3) 0%, 
                rgba(59, 130, 246, 0.3) 30%, 
                rgba(59, 130, 246, 0.2) 30%, 
                rgba(59, 130, 246, 0.2) 65%, 
                rgba(59, 130, 246, 0.1) 65%, 
                rgba(59, 130, 246, 0.1) 100%);
            border-radius: var(--radius-sm);
        }

        .setup-section .layout-name {
            font-size: 1.125rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: var(--space-2);
        }


        .setup-section .layout-description {
            color: #64748b;
            font-size: 0.875rem;
            margin-bottom: var(--space-2);
        }


        .setup-section .layout-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.75rem;
            color: rgba(255, 255, 255, 0.6);
        }

        .setup-section .layout-tag {
            background: rgba(255, 255, 255, 0.2);
            padding: var(--space-1) var(--space-2);
            border-radius: var(--radius-sm);
        }

        .setup-section .widget-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: var(--space-4);
        }

        .setup-section .widget-card {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            padding: var(--space-4);
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            user-select: none;
            pointer-events: auto;
        }


        .setup-section .widget-card:hover {
            border-color: #3b82f6;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.2);
        }

        .setup-section .widget-card:active {
            transform: translateY(0);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .widget-card.selected {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.2);
        }

        .setup-section .widget-card.selected::after {
            content: '✓';
            position: absolute;
            top: var(--space-3);
            right: var(--space-3);
            width: 20px;
            height: 20px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.75rem;
        }

        .setup-section .widget-header {
            display: flex;
            align-items: center;
            gap: var(--space-3);
        }

        .setup-section .widget-icon {
            width: 40px;
            height: 40px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1rem;
            color: white;
        }

        .setup-section .widget-info h4 {
            font-size: 1rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: var(--space-1);
        }


        .setup-section .widget-info p {
            color: #64748b;
            font-size: 0.875rem;
        }


        .setup-section .widget-importance {
            margin-left: auto;
            padding: var(--space-1) var(--space-2);
            border-radius: var(--radius-sm);
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }

        .setup-section .importance-high {
            background: rgba(239, 68, 68, 0.2);
            color: #fca5a5;
        }

        .setup-section .importance-medium {
            background: rgba(245, 158, 11, 0.2);
            color: #fcd34d;
        }

        .setup-section .importance-low {
            background: rgba(107, 114, 128, 0.2);
            color: #d1d5db;
        }

        .setup-section .quick-actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: var(--space-4);
        }

        .setup-section .quick-action-card {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            padding: var(--space-4);
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: center;
            position: relative;
            user-select: none;
            pointer-events: auto;
        }


        .setup-section .quick-action-card:hover {
            border-color: #3b82f6;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.2);
        }

        .setup-section .quick-action-card:active {
            transform: translateY(0);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .quick-action-card.selected {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.2);
        }

        .setup-section .quick-action-card.selected::after {
            content: '✓';
            position: absolute;
            top: var(--space-3);
            right: var(--space-3);
            width: 20px;
            height: 20px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.75rem;
        }

        .setup-section .quick-action-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            margin: 0 auto var(--space-3);
        }

        .setup-section .quick-action-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: #1e293b;
        }


        .setup-section .form-group {
            margin-bottom: var(--space-4);
        }

        .setup-section .form-group label {
            display: block;
            font-size: 0.875rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: var(--space-2);
        }


        .setup-section .form-control {
            width: 100%;
            padding: var(--space-3);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            font-size: 0.875rem;
            transition: all 0.3s ease;
        }

        .setup-section .form-control:hover {
            border-color: #3b82f6;
            background: rgba(255, 255, 255, 0.8);
        }


        .setup-section .form-control:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            background: rgba(255, 255, 255, 0.8);
        }


        .setup-section .checkbox-group {
            display: flex;
            align-items: center;
            gap: var(--space-3);
            margin-bottom: var(--space-3);
        }

        .setup-section .checkbox {
            width: 24px;
            height: 24px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-sm);
            cursor: pointer;
            position: relative;
            transition: all 0.3s ease;
            flex-shrink: 0;
            background: rgba(255, 255, 255, 0.1);
            user-select: none;
            pointer-events: auto;
        }


        .setup-section .checkbox.checked {
            background: #3b82f6;
            border-color: #3b82f6;
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .checkbox:hover {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
            transform: scale(1.05);
        }


        .setup-section .checkbox.checked::after {
            content: '✓';
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            color: white;
            font-size: 0.875rem;
            font-weight: 700;
        }

        .setup-section .checkbox-label {
            color: #1e293b;
            cursor: pointer;
            font-weight: 500;
            font-size: 0.875rem;
            user-select: none;
            transition: color 0.2s ease;
        }

        .setup-section .checkbox-label:hover {
            color: #3b82f6;
        }


        .setup-section .setup-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: var(--space-6);
            padding-top: var(--space-4);
            border-top: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-section .btn {
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-lg);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
            border: none;
            font-size: 1rem;
            min-height: 48px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        /* Button Classes */
        .btn {
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-lg);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
            border: none;
            font-size: 1rem;
            min-height: 48px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        /* Ensure buttons are properly styled */
        button.btn {
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-lg);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
            border: none;
            font-size: 1rem;
            min-height: 48px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            box-shadow: var(--shadow-lg);
        }

        button.btn-primary {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            box-shadow: var(--shadow-lg);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
            background: linear-gradient(135deg, #2563eb, #1e40af);
        }

        button.btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
            background: linear-gradient(135deg, #2563eb, #1e40af);
        }

        .btn-secondary {
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        button.btn-secondary {
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        .btn-secondary:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        button.btn-secondary:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        .setup-section .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            box-shadow: var(--shadow-lg);
        }

        .setup-section .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
            background: linear-gradient(135deg, #2563eb, #1e40af);
        }

        .setup-section .btn-secondary {
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        .setup-section .btn-secondary:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        @media (max-width: 768px) {
            .setup-modal {
                margin: var(--space-4);
                max-height: calc(100vh - var(--space-8));
            }

            .setup-section .layout-grid,
            .setup-section .widget-grid,
            .setup-section .quick-actions-grid {
                grid-template-columns: 1fr;
            }

            .setup-section .setup-actions {
                flex-direction: column;
                gap: var(--space-3);
            }

            .setup-section .btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-container">
        <!-- Include Unified Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <main class="admin-main">
            <div class="dashboard-grid layout-<?php echo $dashboard_layout; ?>">
                <!-- Header -->
                <header class="dashboard-header fade-in">
                    <div class="header-content">
                        <div class="header-left">
                            <h1>Dashboard • Boxenstop</h1>
                            <p>Willkommen zurück, <?php echo htmlspecialchars($current_user_name); ?>!</p>
                            
                            <!-- Statistics in Header -->
                            <div class="header-stats">
                                <div class="header-stat-item">
                                    <div class="header-stat-icon" style="background: var(--success);">
                                        <i class="fas fa-calendar-day"></i>
                                    </div>
                                    <div class="header-stat-content">
                                        <div class="header-stat-number"><?php echo $stats['today_appointments']; ?></div>
                                        <div class="header-stat-label">Heutige Termine</div>
                                    </div>
                                </div>
                                <div class="header-stat-item">
                                    <div class="header-stat-icon" style="background: var(--primary);">
                                        <i class="fas fa-calendar-alt"></i>
                                    </div>
                                    <div class="header-stat-content">
                                        <div class="header-stat-number"><?php echo $stats['total_events']; ?></div>
                                        <div class="header-stat-label">Aktive Events</div>
                                    </div>
                                </div>
                                <div class="header-stat-item">
                                    <div class="header-stat-icon" style="background: var(--accent);">
                                        <i class="fas fa-calendar-check"></i>
                                    </div>
                                    <div class="header-stat-content">
                                        <div class="header-stat-number"><?php echo $stats['total_appointments']; ?></div>
                                        <div class="header-stat-label">Alle Termine</div>
                                    </div>
                                </div>
                                <div class="header-stat-item">
                                    <div class="header-stat-icon" style="background: var(--warning);">
                                        <i class="fas fa-warehouse"></i>
                                    </div>
                                    <div class="header-stat-content">
                                        <div class="header-stat-number" id="storage-activities-count">
                                            <i class="fas fa-spinner fa-spin" style="font-size: 1rem;"></i>
                                        </div>
                                        <div class="header-stat-label">Lageraktivitäten</div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Quick Actions in Header -->
                            <?php if (isWidgetEnabled('quick_actions', $user_dashboard_config)): ?>
                                <div class="header-quick-actions" id="header-quick-actions-container">
                                    <div class="header-quick-actions-title">
                                        <i class="fas fa-rocket"></i>
                                        Schnellaktionen
                                        <button onclick="openQuickActionsModal()" style="margin-left: auto; background: none; border: none; color: inherit; opacity: 0.7; transition: opacity 0.3s; cursor: pointer;" onmouseover="this.style.opacity='1'" onmouseout="this.style.opacity='0.7'" title="Verwalten">
                                            <i class="fas fa-cog"></i>
                                        </button>
                                    </div>
                                    <div class="header-quick-actions-grid" id="header-quick-actions-grid">
                                        <?php if (!empty($quick_actions)): ?>
                                            <?php foreach (array_slice($quick_actions, 0, 6) as $action): ?>
                                                <a href="<?php echo htmlspecialchars($action['url']); ?>" class="header-quick-action" title="<?php echo htmlspecialchars($action['title']); ?>">
                                                    <div class="header-quick-action-icon" style="background: <?php echo htmlspecialchars($action['background_color'] ?? $action['color'] ?? '#007AFF'); ?>;">
                                                        <i class="<?php echo htmlspecialchars($action['icon']); ?>"></i>
                                                    </div>
                                                    <span><?php echo htmlspecialchars($action['title']); ?></span>
                                                </a>
                                            <?php endforeach; ?>
                                        <?php else: ?>
                                            <div style="text-align: center; padding: var(--space-4); color: var(--label-secondary); font-size: 0.875rem;">
                                                <i class="fas fa-info-circle" style="margin-right: var(--space-2);"></i>
                                                Keine Schnellaktionen. <a href="quick_actions_manager.php" style="color: var(--primary); text-decoration: underline;">Erste erstellen</a>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="header-actions">
                            <button class="btn-header" onclick="openSetupModal()">
                                <i class="fas fa-cog"></i>
                                Anpassen
                            </button>
                        </div>
                    </div>
                </header>


                <!-- Main Content Area - Optimized Widget Layout -->
                <div class="main-content">
                    <!-- Priority Widgets Row (Most Important) -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(350px, 1fr)); gap: var(--space-6); margin-bottom: var(--space-6);">
                        <!-- Today's Appointments -->
                        <?php if (isWidgetEnabled('today_appointments', $user_dashboard_config)): ?>
                            <div class="widget fade-in" id="widget-today-appointments" data-widget-type="today_appointments">
                                <div class="widget-header">
                                    <h3 class="widget-title">
                                        <div class="widget-icon" style="background: var(--success);">
                                            <i class="fas fa-calendar-day"></i>
                                        </div>
                                        Heutige Termine
                                    </h3>
                                    <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-today-appointments')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-today-appointments')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <a href="appointments.php?filter=today" class="widget-btn">
                                            <i class="fas fa-external-link-alt"></i>
                                        </a>
                            </div>
                        </div>
                                <?php if (empty($todayAppointments)): ?>
                                    <div style="text-align: center; padding: var(--space-8);">
                                        <div style="width: 80px; height: 80px; margin: 0 auto var(--space-4); background: rgba(99, 102, 241, 0.1); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
                                            <i class="fas fa-calendar-times" style="font-size: 2.5rem; color: var(--primary); opacity: 0.6;"></i>
                                        </div>
                                        <h4 style="color: #1e293b; margin-bottom: var(--space-2); font-weight: 600;">
                                            Keine Termine für heute
                                        </h4>
                                        <p style="color: #64748b; font-size: 0.875rem; margin-bottom: var(--space-4);">
                                            Sie haben heute keine geplanten Termine. Genießen Sie den freien Tag!
                                        </p>
                                        <a href="appointments.php?action=create" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2);">
                                            <i class="fas fa-plus"></i>
                                            Neuen Termin erstellen
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($todayAppointments as $appointment): ?>
                                        <div class="activity-item">
                                            <div class="activity-icon" style="background: var(--success);">
                                                <i class="fas fa-user"></i>
                                            </div>
                                            <div class="activity-content">
                                                <div class="activity-title"><?php echo htmlspecialchars($appointment['customer_name']); ?></div>
                                                <div class="activity-description"><?php echo htmlspecialchars($appointment['license_plate']); ?></div>
                                            </div>
                                            <div class="activity-time">
                                                <?php echo date('H:i', strtotime($appointment['slot_time'])); ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                    </div>
                <?php endif; ?>


                        <!-- Task Reminders Widget -->
                        <?php if (isWidgetEnabled('task_reminders', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-task-reminders" data-widget-type="task_reminders">
                            <div class="widget-header">
                                <h3 class="widget-title">
                                        <div class="widget-icon" style="background: var(--warning);">
                                            <i class="fas fa-tasks"></i>
                                    </div>
                                        Aufgaben & Erinnerungen
                                </h3>
                                <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-task-reminders')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-task-reminders')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <a href="tasks_manager.php" class="widget-btn" title="Aufgaben verwalten">
                                        <i class="fas fa-external-link-alt"></i>
                                    </a>
                                </div>
                            </div>
                                <div id="task-reminders-container">
                                    <?php 
                                    $has_tasks = !empty($user_tasks) || ($task_reminders['tomorrow_appointments'] ?? 0) > 0 || ($task_reminders['upcoming_events'] ?? 0) > 0;
                                    ?>
                                    <?php if ($has_tasks): ?>
                                        <?php if (!empty($user_tasks)): ?>
                                            <?php foreach (array_slice($user_tasks, 0, 3) as $task): ?>
                                                <?php
                                                $priority_colors = [
                                                    'urgent' => '#EF4444',
                                                    'high' => '#F59E0B',
                                                    'medium' => '#3B82F6',
                                                    'low' => '#6B7280'
                                                ];
                                                $priority_color = $priority_colors[$task['priority'] ?? 'medium'] ?? '#3B82F6';
                                                $status_icons = [
                                                    'pending' => 'fa-clock',
                                                    'in_progress' => 'fa-spinner',
                                                    'completed' => 'fa-check-circle',
                                                    'cancelled' => 'fa-times-circle'
                                                ];
                                                $status_icon = $status_icons[$task['status'] ?? 'pending'] ?? 'fa-clock';
                                                $hex = str_replace('#', '', $priority_color);
                                                $r = hexdec(substr($hex, 0, 2));
                                                $g = hexdec(substr($hex, 2, 2));
                                                $b = hexdec(substr($hex, 4, 2));
                                                ?>
                                                <div class="activity-item" style="background: rgba(<?php echo $r; ?>, <?php echo $g; ?>, <?php echo $b; ?>, 0.1); border-left: 3px solid <?php echo $priority_color; ?>;">
                                                    <div class="activity-icon" style="background: <?php echo $priority_color; ?>;">
                                                        <i class="fas <?php echo $status_icon; ?>"></i>
                                        </div>
                                        <div class="activity-content">
                                                        <div class="activity-title">
                                                            <?php echo htmlspecialchars($task['title']); ?>
                                                            <?php if ($task['priority'] === 'urgent'): ?>
                                                                <span style="background: var(--error); color: white; padding: 2px 6px; border-radius: 4px; font-size: 0.75rem; margin-left: var(--space-2);">Dringend</span>
                                                            <?php endif; ?>
                                                        </div>
                                                        <div class="activity-description">
                                                            <?php if ($task['due_date']): ?>
                                                                Fällig: <?php echo date('d.m.Y', strtotime($task['due_date'])); ?>
                                                                <?php if ($task['due_time']): ?>
                                                                    um <?php echo date('H:i', strtotime($task['due_time'])); ?>
                                                                <?php endif; ?>
                                                            <?php elseif ($task['description']): ?>
                                                                <?php echo htmlspecialchars(substr($task['description'], 0, 50)); ?><?php echo strlen($task['description']) > 50 ? '...' : ''; ?>
                                                            <?php endif; ?>
                                                        </div>
                                        </div>
                                        <div class="activity-time">
                                                        <button class="widget-btn" onclick="completeTask(<?php echo $task['id']; ?>)" title="Als erledigt markieren" style="width: 32px; height: 32px;">
                                                            <i class="fas fa-check"></i>
                                                        </button>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                                        
                                        <?php if (($task_reminders['tomorrow_appointments'] ?? 0) > 0): ?>
                                            <div class="activity-item" style="background: rgba(245, 158, 11, 0.1); border-left: 3px solid var(--warning);">
                                                <div class="activity-icon" style="background: var(--warning);">
                                                    <i class="fas fa-calendar-day"></i>
                                                </div>
                                                <div class="activity-content">
                                                    <div class="activity-title">Morgen: <?php echo $task_reminders['tomorrow_appointments']; ?> Termine</div>
                                                    <div class="activity-description">Bereiten Sie sich auf morgen vor</div>
                                                </div>
                        </div>
                    <?php endif; ?>

                                        <?php if (($task_reminders['upcoming_events'] ?? 0) > 0): ?>
                                            <div class="activity-item" style="background: rgba(59, 130, 246, 0.1); border-left: 3px solid var(--primary);">
                                                <div class="activity-icon" style="background: var(--primary);">
                                                    <i class="fas fa-calendar-alt"></i>
                                                </div>
                                                <div class="activity-content">
                                                    <div class="activity-title"><?php echo $task_reminders['upcoming_events']; ?> Events diese Woche</div>
                                                    <div class="activity-description">Überprüfen Sie Ihre Event-Vorbereitungen</div>
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <div style="margin-top: var(--space-4); text-align: center;">
                                            <a href="tasks_manager.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2); width: 100%; justify-content: center;">
                                                <i class="fas fa-tasks"></i>
                                                Aufgaben verwalten
                                            </a>
                                        </div>
                                    <?php else: ?>
                                        <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                                            <i class="fas fa-check-circle" style="font-size: 2rem; margin-bottom: var(--space-4); color: var(--success); opacity: 0.5;"></i>
                                            <p>Keine offenen Aufgaben</p>
                                            <p style="font-size: 0.875rem; margin-top: var(--space-2); opacity: 0.7;">Alles ist auf dem neuesten Stand</p>
                                            <a href="tasks_manager.php" class="btn btn-primary" style="margin-top: var(--space-4); display: inline-flex; align-items: center; gap: var(--space-2);">
                                                <i class="fas fa-plus"></i>
                                                Neue Aufgabe erstellen
                                            </a>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Secondary Widgets Row (Operational) -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(350px, 1fr)); gap: var(--space-6); margin-bottom: var(--space-6);">
                    <!-- Upcoming Events -->
                    <?php if (isWidgetEnabled('upcoming_events', $user_dashboard_config)): ?>
                            <div class="widget fade-in" id="widget-upcoming-events" data-widget-type="upcoming_events">
                            <div class="widget-header">
                                <h3 class="widget-title">
                                    <div class="widget-icon" style="background: var(--primary);">
                                        <i class="fas fa-calendar-alt"></i>
                                    </div>
                                    Kommende Events
                                </h3>
                                <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-upcoming-events')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-upcoming-events')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                    <a href="events.php" class="widget-btn">
                                        <i class="fas fa-external-link-alt"></i>
                                    </a>
                                </div>
                            </div>
                            <?php if (empty($upcomingEvents)): ?>
                                <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                                    <i class="fas fa-calendar-plus" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                                    <p>Keine kommenden Events</p>
                                        <a href="events.php?action=create" class="btn btn-primary" style="margin-top: var(--space-4); display: inline-flex; align-items: center; gap: var(--space-2);">
                                        <i class="fas fa-plus"></i>
                                        Event erstellen
                                    </a>
                                </div>
                            <?php else: ?>
                                <?php foreach ($upcomingEvents as $event): ?>
                                    <div class="activity-item">
                                        <div class="activity-icon" style="background: var(--primary);">
                                            <i class="fas fa-calendar"></i>
                                        </div>
                                        <div class="activity-content">
                                            <div class="activity-title"><?php echo htmlspecialchars($event['name']); ?></div>
                                            <div class="activity-description"><?php echo date('d.m.Y', strtotime($event['event_date'])); ?></div>
                                        </div>
                                        <div class="activity-time">
                                            <?php echo $event['booked_count']; ?>/<?php echo $event['appointment_count']; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                        <!-- Activity Feed Widget -->
                        <?php if (isWidgetEnabled('activity_feed', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-activity-feed" data-widget-type="activity_feed">
                                <div class="widget-header">
                                    <h3 class="widget-title">
                                        <div class="widget-icon" style="background: var(--accent);">
                                            <i class="fas fa-stream"></i>
                </div>
                                        Aktivitäts-Feed
                                    </h3>
                                    <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-activity-feed')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-activity-feed')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <a href="appointments.php" class="widget-btn" title="Alle Aktivitäten anzeigen">
                                            <i class="fas fa-external-link-alt"></i>
                                        </a>
                                    </div>
                                </div>
                                <div id="activity-feed-container">
                                    <?php if (!empty($recent_activity)): ?>
                                        <?php foreach (array_slice($recent_activity, 0, 5) as $activity): ?>
                                            <div class="activity-item">
                                                <div class="activity-icon" style="background: var(--accent);">
                                                    <i class="fas fa-<?php echo $activity['type'] === 'appointment' ? 'calendar-check' : 'circle'; ?>"></i>
                                                </div>
                                                <div class="activity-content">
                                                    <div class="activity-title"><?php echo htmlspecialchars($activity['description']); ?></div>
                                                    <div class="activity-description"><?php echo date('d.m.Y H:i', strtotime($activity['timestamp'])); ?></div>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                                            <i class="fas fa-stream" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                                            <p>Keine Aktivitäten in den letzten 7 Tagen</p>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Recent Storage -->
                        <?php if (isWidgetEnabled('recent_storage', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-recent-storage" data-widget-type="recent_storage">
                            <div class="widget-header">
                                <h3 class="widget-title">
                                        <div class="widget-icon" style="background: var(--warning);">
                                            <i class="fas fa-warehouse"></i>
                                    </div>
                                        Letzte Lageraktivitäten
                                        <?php if ($current_user_autohaus_name): ?>
                                            <span style="font-size: 0.875rem; font-weight: 500; color: var(--gray-500); margin-left: var(--space-2);">
                                                (<?php echo htmlspecialchars($current_user_autohaus_name); ?>)
                                            </span>
                                        <?php endif; ?>
                                </h3>
                                <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-recent-storage')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-recent-storage')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <a href="storage.php" class="widget-btn">
                                            <i class="fas fa-external-link-alt"></i>
                                    </a>
                                </div>
                            </div>
                                
                                <!-- Autohaus Dropdown (only if user has no autohaus assigned) -->
                                <?php if (!$current_user_autohaus_id && !empty($all_autohauser)): ?>
                                    <div style="margin-bottom: var(--space-4);">
                                        <label for="storage-autohaus-filter" style="display: block; margin-bottom: var(--space-2); font-size: 0.875rem; font-weight: 600; color: #64748b;">
                                            <i class="fas fa-building"></i> Autohaus auswählen
                                        </label>
                                        <select id="storage-autohaus-filter" onchange="filterStorageByAutohaus(this.value)" 
                                                class="storage-autohaus-select">
                                            <option value="">Alle Autohäuser</option>
                                            <?php foreach ($all_autohauser as $autohaus): ?>
                                                <option value="<?php echo $autohaus['id']; ?>" 
                                                        <?php echo ($selected_autohaus_id == $autohaus['id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($autohaus['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                        </div>
                                <?php endif; ?>
                                
                                <!-- Storage Activities Container (loaded via AJAX) -->
                                <div id="storage-activities-container">
                                    <div style="text-align: center; padding: var(--space-4); color: rgba(255, 255, 255, 0.8);">
                                        <i class="fas fa-spinner fa-spin" style="font-size: 1.5rem; margin-bottom: var(--space-2);"></i>
                                        <p style="font-size: 0.875rem;">Lade Lageraktivitäten...</p>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- WhatsApp Statistics Widget -->
                        <?php if (isWidgetEnabled('whatsapp_stats', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-whatsapp-stats" data-widget-type="whatsapp_stats" style="background: #ffffff; border: 1px solid #e5e7eb; box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);">
                                <div class="widget-header">
                                    <h3 class="widget-title" style="color: #111827;">
                                        <div class="widget-icon" style="background: #25D366; border-radius: 6px;">
                                            <i class="fab fa-whatsapp"></i>
                                        </div>
                                        WhatsApp-Statistiken
                                        <?php if ($current_user_autohaus_name): ?>
                                            <span style="font-size: 0.875rem; font-weight: 400; color: #6b7280; margin-left: var(--space-2);">
                                                (<?php echo htmlspecialchars($current_user_autohaus_name); ?>)
                                            </span>
                                        <?php endif; ?>
                                    </h3>
                                    <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-whatsapp-stats')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-whatsapp-stats')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <a href="whatsapp_logs.php" class="widget-btn" title="Alle Logs anzeigen">
                                            <i class="fas fa-external-link-alt"></i>
                                        </a>
                                    </div>
                                </div>
                                <div class="stats-grid" style="grid-template-columns: repeat(2, 1fr); margin-bottom: var(--space-4); gap: var(--space-3);">
                                    <div class="stat-card" style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 8px; padding: var(--space-4);">
                                        <div class="stat-icon" style="background: #25D366; width: 40px; height: 40px; border-radius: 8px;">
                                            <i class="fas fa-paper-plane"></i>
                                        </div>
                                        <div class="stat-number" style="color: #111827; font-weight: 700;"><?php echo $whatsapp_stats['today'] ?? 0; ?></div>
                                        <div class="stat-label" style="color: #6b7280; font-size: 0.875rem;">Heute gesendet</div>
                                    </div>
                                    <div class="stat-card" style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 8px; padding: var(--space-4);">
                                        <div class="stat-icon" style="background: #128C7E; width: 40px; height: 40px; border-radius: 8px;">
                                            <i class="fas fa-chart-line"></i>
                                        </div>
                                        <div class="stat-number" style="color: #111827; font-weight: 700;"><?php echo $whatsapp_stats['success_rate'] ?? 0; ?>%</div>
                                        <div class="stat-label" style="color: #6b7280; font-size: 0.875rem;">Erfolgsrate</div>
                                    </div>
                                </div>
                                <div class="stats-grid" style="grid-template-columns: repeat(3, 1fr); gap: var(--space-2);">
                                    <div class="stat-card" style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 6px; padding: var(--space-3); text-align: center;">
                                        <div class="stat-number" style="font-size: 1.5rem; color: #111827; font-weight: 700;"><?php echo $whatsapp_stats['total'] ?? 0; ?></div>
                                        <div class="stat-label" style="color: #6b7280; font-size: 0.75rem;">Gesamt</div>
                                    </div>
                                    <div class="stat-card" style="background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; padding: var(--space-3); text-align: center;">
                                        <div class="stat-number" style="font-size: 1.5rem; color: #10b981; font-weight: 700;"><?php echo $whatsapp_stats['successful'] ?? 0; ?></div>
                                        <div class="stat-label" style="color: #6b7280; font-size: 0.75rem;">Erfolgreich</div>
                                    </div>
                                    <div class="stat-card" style="background: #fef2f2; border: 1px solid #fecaca; border-radius: 6px; padding: var(--space-3); text-align: center;">
                                        <div class="stat-number" style="font-size: 1.5rem; color: #ef4444; font-weight: 700;"><?php echo $whatsapp_stats['failed'] ?? 0; ?></div>
                                        <div class="stat-label" style="color: #6b7280; font-size: 0.75rem;">Fehlgeschlagen</div>
                                    </div>
                                </div>
                                <div style="margin-top: var(--space-4); display: flex; gap: var(--space-2); border-top: 1px solid #e5e7eb; padding-top: var(--space-4);">
                                    <a href="whatsapp_logs.php" class="btn btn-primary" style="flex: 1; text-align: center; background: #ffffff; border: 1px solid #e5e7eb; color: #111827; padding: var(--space-2) var(--space-4); border-radius: 6px; font-weight: 500; transition: all 0.2s;">
                                        <i class="fas fa-list"></i> Logs anzeigen
                                    </a>
                                    <a href="whatsapp_reminders_admin.php" class="btn btn-primary" style="flex: 1; text-align: center; background: #ffffff; border: 1px solid #e5e7eb; color: #111827; padding: var(--space-2) var(--space-4); border-radius: 6px; font-weight: 500; transition: all 0.2s;">
                                        <i class="fas fa-cog"></i> Templates
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- WhatsApp Activity Widget -->
                        <?php if (isWidgetEnabled('whatsapp_activity', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-whatsapp-activity" data-widget-type="whatsapp_activity" style="background: #ffffff; border: 1px solid #e5e7eb; box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);">
                                <div class="widget-header">
                                    <h3 class="widget-title" style="color: #111827;">
                                        <div class="widget-icon" style="background: #128C7E; border-radius: 6px;">
                                            <i class="fas fa-paper-plane"></i>
                                        </div>
                                        WhatsApp-Aktivitäten
                                        <?php if ($current_user_autohaus_name): ?>
                                            <span style="font-size: 0.875rem; font-weight: 400; color: #6b7280; margin-left: var(--space-2);">
                                                (<?php echo htmlspecialchars($current_user_autohaus_name); ?>)
                                            </span>
                                        <?php endif; ?>
                                    </h3>
                                    <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-whatsapp-activity')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-whatsapp-activity')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <a href="whatsapp_logs.php" class="widget-btn" title="Alle Aktivitäten anzeigen">
                                            <i class="fas fa-external-link-alt"></i>
                                        </a>
                                    </div>
                                </div>
                                <div id="whatsapp-activity-container">
                                    <?php if (!empty($whatsapp_recent_logs)): ?>
                                        <?php foreach (array_slice($whatsapp_recent_logs, 0, 5) as $log): ?>
                                            <?php
                                            $statusColor = ($log['status'] ?? '') === 'success' ? '#10b981' : '#ef4444';
                                            $statusIcon = ($log['status'] ?? '') === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle';
                                            $actionLabels = [
                                                'confirmation' => 'Bestätigung',
                                                'reminder_24h' => 'Erinnerung 24h',
                                                'reminder_48h' => 'Erinnerung 48h',
                                                'approved' => 'Genehmigt',
                                                'rejected' => 'Abgelehnt',
                                                'cancellation' => 'Storniert',
                                                'rescheduled' => 'Verschoben',
                                                'test_send' => 'Test'
                                            ];
                                            $actionLabel = $actionLabels[$log['action_type'] ?? ''] ?? ucfirst($log['action_type'] ?? 'Unbekannt');
                                            ?>
                                            <div class="activity-item" style="background: #f9fafb; border-left: 3px solid <?php echo $statusColor; ?>; border-radius: 6px; padding: var(--space-3); margin-bottom: var(--space-2);">
                                                <div class="activity-icon" style="background: <?php echo $statusColor; ?>; width: 32px; height: 32px; border-radius: 6px;">
                                                    <i class="fas <?php echo $statusIcon; ?>"></i>
                                                </div>
                                                <div class="activity-content">
                                                    <div class="activity-title" style="color: #111827; font-weight: 600;">
                                                        <?php echo htmlspecialchars($actionLabel); ?>
                                                        <?php if ($log['customer_name']): ?>
                                                            - <?php echo htmlspecialchars($log['customer_name']); ?>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="activity-description" style="color: #6b7280; font-size: 0.875rem;">
                                                        <?php echo htmlspecialchars($log['phone_number'] ?? '-'); ?>
                                                        <?php if ($log['message']): ?>
                                                            <br><small style="opacity: 0.8;"><?php echo htmlspecialchars(substr($log['message'], 0, 50)); ?><?php echo strlen($log['message']) > 50 ? '...' : ''; ?></small>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                <div class="activity-time" style="color: #9ca3af; font-size: 0.75rem;">
                                                    <?php echo date('H:i', strtotime($log['created_at'] ?? 'now')); ?>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                        <div style="margin-top: var(--space-4); text-align: center; border-top: 1px solid #e5e7eb; padding-top: var(--space-4);">
                                            <a href="whatsapp_logs.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2); width: 100%; justify-content: center; background: #ffffff; border: 1px solid #e5e7eb; color: #111827; padding: var(--space-2) var(--space-4); border-radius: 6px; font-weight: 500;">
                                                <i class="fas fa-list"></i>
                                                Alle Logs anzeigen
                                            </a>
                                        </div>
                                    <?php else: ?>
                                        <div style="text-align: center; padding: var(--space-8); color: #6b7280;">
                                            <i class="fab fa-whatsapp" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.3;"></i>
                                            <p style="color: #111827; font-weight: 500;">Keine WhatsApp-Aktivitäten</p>
                                            <p style="font-size: 0.875rem; margin-top: var(--space-2); color: #6b7280;">Noch keine Nachrichten gesendet</p>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- WhatsApp Template Status Widget -->
                        <?php if (isWidgetEnabled('whatsapp_templates', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-whatsapp-templates" data-widget-type="whatsapp_templates" style="background: #ffffff; border: 1px solid #e5e7eb; box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);">
                                <div class="widget-header">
                                    <h3 class="widget-title" style="color: #111827;">
                                        <div class="widget-icon" style="background: #075E54; border-radius: 6px;">
                                            <i class="fas fa-file-alt"></i>
                                        </div>
                                        Template-Status
                                        <?php if ($current_user_autohaus_name): ?>
                                            <span style="font-size: 0.875rem; font-weight: 400; color: #6b7280; margin-left: var(--space-2);">
                                                (<?php echo htmlspecialchars($current_user_autohaus_name); ?>)
                                            </span>
                                        <?php endif; ?>
                                    </h3>
                                    <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-whatsapp-templates')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-whatsapp-templates')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <a href="whatsapp_reminders_admin.php" class="widget-btn" title="Templates verwalten">
                                            <i class="fas fa-external-link-alt"></i>
                                        </a>
                                    </div>
                                </div>
                                <div id="whatsapp-templates-container">
                                    <?php if (!empty($whatsapp_template_status) && $whatsapp_template_status['total_configs'] > 0): ?>
                                        <div class="stats-grid" style="grid-template-columns: repeat(3, 1fr); margin-bottom: var(--space-4); gap: var(--space-2);">
                                            <div class="stat-card" style="background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; padding: var(--space-3); text-align: center;">
                                                <div class="stat-number" style="font-size: 1.75rem; color: #10b981; font-weight: 700;"><?php echo $whatsapp_template_status['active_enabled'] ?? 0; ?></div>
                                                <div class="stat-label" style="color: #6b7280; font-size: 0.75rem;">Aktiv & Enabled</div>
                                            </div>
                                            <div class="stat-card" style="background: #fffbeb; border: 1px solid #fde68a; border-radius: 6px; padding: var(--space-3); text-align: center;">
                                                <div class="stat-number" style="font-size: 1.75rem; color: #f59e0b; font-weight: 700;"><?php echo $whatsapp_template_status['active_disabled'] ?? 0; ?></div>
                                                <div class="stat-label" style="color: #6b7280; font-size: 0.75rem;">Aktiv & Disabled</div>
                                            </div>
                                            <div class="stat-card" style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 6px; padding: var(--space-3); text-align: center;">
                                                <div class="stat-number" style="font-size: 1.75rem; color: #6b7280; font-weight: 700;"><?php echo $whatsapp_template_status['inactive'] ?? 0; ?></div>
                                                <div class="stat-label" style="color: #6b7280; font-size: 0.75rem;">Inaktiv</div>
                                            </div>
                                        </div>
                                        
                                        <?php if (!empty($whatsapp_template_status['by_action'])): ?>
                                            <div style="font-size: 0.875rem; font-weight: 600; margin-bottom: var(--space-2); color: #374151;">Nach Aktion:</div>
                                            <?php 
                                            $actionLabels = [
                                                'confirmation' => 'Bestätigung',
                                                'reminder_24h' => 'Erinnerung 24h',
                                                'reminder_48h' => 'Erinnerung 48h',
                                                'approved' => 'Genehmigt',
                                                'rejected' => 'Abgelehnt',
                                                'cancellation' => 'Storniert',
                                                'rescheduled' => 'Verschoben'
                                            ];
                                            foreach (array_slice($whatsapp_template_status['by_action'], 0, 5) as $actionType => $actionData): 
                                                $actionLabel = $actionLabels[$actionType] ?? ucfirst($actionType);
                                            ?>
                                                <div class="activity-item" style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 6px; margin-bottom: var(--space-2); padding: var(--space-3);">
                                                    <div class="activity-content" style="flex: 1;">
                                                        <div class="activity-title" style="color: #111827; font-weight: 600; font-size: 0.875rem;"><?php echo htmlspecialchars($actionLabel); ?></div>
                                                        <div class="activity-description" style="display: flex; gap: var(--space-3); font-size: 0.75rem; margin-top: var(--space-1);">
                                                            <span style="color: #10b981;">✓ <?php echo $actionData['active_enabled']; ?> aktiv</span>
                                                            <span style="color: #f59e0b;">⚠ <?php echo $actionData['active_disabled']; ?> disabled</span>
                                                            <span style="color: #6b7280;">○ <?php echo $actionData['inactive']; ?> inaktiv</span>
                                                        </div>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                        
                                        <div style="margin-top: var(--space-4); text-align: center; border-top: 1px solid #e5e7eb; padding-top: var(--space-4);">
                                            <a href="whatsapp_reminders_admin.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2); width: 100%; justify-content: center; background: #ffffff; border: 1px solid #e5e7eb; color: #111827; padding: var(--space-2) var(--space-4); border-radius: 6px; font-weight: 500;">
                                                <i class="fas fa-cog"></i>
                                                Templates verwalten
                                            </a>
                                        </div>
                                    <?php else: ?>
                                        <div style="text-align: center; padding: var(--space-8); color: #6b7280;">
                                            <i class="fas fa-file-alt" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.3;"></i>
                                            <p style="color: #111827; font-weight: 500;">Keine Template-Konfigurationen</p>
                                            <p style="font-size: 0.875rem; margin-top: var(--space-2); color: #6b7280;">Konfigurieren Sie Ihre WhatsApp-Templates</p>
                                            <a href="whatsapp_reminders_admin.php" class="btn btn-primary" style="margin-top: var(--space-4); display: inline-flex; align-items: center; gap: var(--space-2); background: #ffffff; border: 1px solid #e5e7eb; color: #111827; padding: var(--space-2) var(--space-4); border-radius: 6px; font-weight: 500;">
                                                <i class="fas fa-plus"></i>
                                                Templates konfigurieren
                                            </a>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>

                    </div>

                    <!-- Analytics Widgets Row (Optional) -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(350px, 1fr)); gap: var(--space-6);">
                        <!-- Quick Stats Widget -->
                        <?php if (isWidgetEnabled('quick_stats', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-quick-stats" data-widget-type="quick_stats">
                                <div class="widget-header">
                                    <h3 class="widget-title">
                                        <div class="widget-icon" style="background: var(--success);">
                                            <i class="fas fa-chart-pie"></i>
                                        </div>
                                        Schnellstatistiken
                                    </h3>
                                    <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-quick-stats')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-quick-stats')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <button class="widget-btn" onclick="refreshQuickStats()" title="Aktualisieren">
                                            <i class="fas fa-sync-alt"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="stats-grid" style="grid-template-columns: repeat(3, 1fr);">
                                    <div class="stat-card">
                                        <div class="stat-icon" style="background: var(--primary);">
                                            <i class="fas fa-calendar-week"></i>
                                        </div>
                                        <div class="stat-number"><?php echo $quick_stats_data['this_week'] ?? 0; ?></div>
                                        <div class="stat-label">Diese Woche</div>
                                    </div>
                                    <div class="stat-card">
                                        <div class="stat-icon" style="background: var(--accent);">
                                            <i class="fas fa-calendar-alt"></i>
                                        </div>
                                        <div class="stat-number"><?php echo $quick_stats_data['this_month'] ?? 0; ?></div>
                                        <div class="stat-label">Dieser Monat</div>
                                    </div>
                                    <div class="stat-card">
                                        <div class="stat-icon" style="background: var(--secondary);">
                                            <i class="fas fa-calendar-check"></i>
                                        </div>
                                        <div class="stat-number"><?php echo $quick_stats_data['active_events'] ?? 0; ?></div>
                                        <div class="stat-label">Aktive Events</div>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- System Health Widget -->
                        <?php if (isWidgetEnabled('system_health', $user_dashboard_config)): ?>
                            <div class="widget slide-in" id="widget-system-health" data-widget-type="system_health">
                                <div class="widget-header">
                                    <h3 class="widget-title">
                                        <div class="widget-icon" style="background: var(--primary);">
                                            <i class="fas fa-heartbeat"></i>
                                        </div>
                                        System-Status
                                    </h3>
                                    <div class="widget-actions">
                                        <button class="widget-btn" onclick="moveWidgetUp('widget-system-health')" title="Nach oben">
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button class="widget-btn" onclick="moveWidgetDown('widget-system-health')" title="Nach unten">
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <button class="widget-btn" onclick="refreshSystemHealth()" title="Aktualisieren">
                                            <i class="fas fa-sync-alt"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="stats-grid" style="grid-template-columns: repeat(3, 1fr);">
                                    <div class="stat-card">
                                        <div class="stat-icon" style="background: var(--primary);">
                                            <i class="fas fa-calendar-check"></i>
                                        </div>
                                        <div class="stat-number"><?php echo $system_health['total_appointments'] ?? 0; ?></div>
                                        <div class="stat-label">Termine gesamt</div>
                                    </div>
                                    <div class="stat-card">
                                        <div class="stat-icon" style="background: var(--success);">
                                            <i class="fas fa-calendar-alt"></i>
                                        </div>
                                        <div class="stat-number"><?php echo $system_health['total_events'] ?? 0; ?></div>
                                        <div class="stat-label">Aktive Events</div>
                                    </div>
                                    <div class="stat-card">
                                        <div class="stat-icon" style="background: var(--accent);">
                                            <i class="fas fa-users"></i>
                                        </div>
                                        <div class="stat-number"><?php echo $system_health['total_users'] ?? 0; ?></div>
                                        <div class="stat-label">Aktive Benutzer</div>
                                    </div>
                                </div>
                                <div style="margin-top: var(--space-4); padding: var(--space-3); background: rgba(16, 185, 129, 0.1); border-radius: var(--radius-lg); border-left: 3px solid var(--success);">
                                    <div style="display: flex; align-items: center; gap: var(--space-2);">
                                        <i class="fas fa-check-circle" style="color: var(--success);"></i>
                                        <span style="font-weight: 600; color: var(--success);">System Status: <?php echo ucfirst($system_health['status'] ?? 'healthy'); ?></span>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Setup Modal -->
    <?php if ($show_setup_modal): ?>
        <div id="setupModal" class="setup-modal-overlay">
            <div class="setup-modal">
                <div class="setup-modal-header">
                    <h2>🚀 Dashboard Setup</h2>
                    <p>Lassen Sie uns Ihr persönliches Dashboard einrichten!</p>
                </div>
                
                <form method="POST" id="dashboardSetupForm">
                    <input type="hidden" name="action" value="save_dashboard_config">
                    
                    <!-- Layout Selection -->
                    <div class="setup-section">
                        <h3>Layout auswählen</h3>
                        <div class="layout-grid">
                            <?php foreach ($available_layouts as $layout_id => $layout): ?>
                                <div class="layout-card <?php echo $modal_config['layout'] === $layout_id ? 'selected' : ''; ?>" 
                                     data-layout="<?php echo $layout_id; ?>">
                                    <div class="layout-preview <?php echo $layout_id; ?>"></div>
                                    <div class="layout-name"><?php echo htmlspecialchars($layout['name']); ?></div>
                                    <div class="layout-description"><?php echo htmlspecialchars($layout['description']); ?></div>
                                    <div class="layout-meta">
                                        <span class="layout-tag"><?php echo htmlspecialchars($layout['best_for']); ?></span>
                                        <span><?php echo htmlspecialchars($layout['screen_size']); ?></span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <input type="hidden" name="layout" id="selectedLayout" value="<?php echo htmlspecialchars($modal_config['layout']); ?>">
                    </div>

                    <!-- Widget Selection -->
                    <div class="setup-section">
                        <h3>Widgets auswählen</h3>
                        <p style="color: #64748b; font-size: 0.875rem; margin-bottom: var(--space-4);">
                            Wählen Sie die Widgets aus, die Sie auf Ihrem Dashboard sehen möchten. Sie können diese später jederzeit ändern.
                        </p>
                        <div class="widget-grid">
                            <?php foreach ($available_widgets as $widget_id => $widget): ?>
                                <div class="widget-card <?php echo in_array($widget_id, $modal_config['widgets']) ? 'selected' : ''; ?>" 
                                     data-widget="<?php echo $widget_id; ?>"
                                     onclick="toggleWidget(this, '<?php echo $widget_id; ?>')">
                                    <div class="widget-header">
                                        <div class="widget-icon" style="background: <?php echo htmlspecialchars($widget['color']); ?>;">
                                            <i class="<?php echo htmlspecialchars($widget['icon']); ?>"></i>
                                        </div>
                                        <div class="widget-info">
                                            <h4><?php echo htmlspecialchars($widget['title']); ?></h4>
                                            <p><?php echo htmlspecialchars($widget['description']); ?></p>
                                        </div>
                                        <span class="widget-importance importance-<?php echo $widget['importance']; ?>">
                                            <?php echo $widget['importance']; ?>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <input type="hidden" name="widgets" id="selectedWidgets" value="<?php echo htmlspecialchars(json_encode($modal_config['widgets'])); ?>">
                    </div>

                    <!-- Quick Actions Setup -->
                    <!-- Quick Actions Manager Link -->
                    <div class="setup-section">
                        <h3>Schnellaktionen verwalten</h3>
                        <p>Verwalte deine personalisierten Schnellaktionen über den Quick Actions Manager.</p>
                        <a href="quick_actions_manager.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2);">
                            <i class="fas fa-rocket"></i>
                            Schnellaktionen Manager öffnen
                        </a>
                    </div>

                    <!-- Additional Settings -->
                    <div class="setup-section">
                        <h3>Dashboard-Einstellungen</h3>

                        <div class="checkbox-group">
                            <div class="checkbox <?php echo ($modal_config['show_welcome'] === true || $modal_config['show_welcome'] === 1 || $modal_config['show_welcome'] === '1') ? 'checked' : ''; ?>" data-checkbox="show_welcome" onclick="toggleCheckbox(this, 'showWelcome')"></div>
                            <label class="checkbox-label" onclick="toggleCheckbox(document.querySelector('[data-checkbox=\"show_welcome\"]'), 'showWelcome')">Willkommensnachricht anzeigen</label>
                        </div>
                        <input type="hidden" name="show_welcome" id="showWelcome" value="<?php echo ($modal_config['show_welcome'] === true || $modal_config['show_welcome'] === 1 || $modal_config['show_welcome'] === '1') ? '1' : '0'; ?>">

                        <div class="checkbox-group">
                            <div class="checkbox <?php echo ($modal_config['auto_refresh'] === true || $modal_config['auto_refresh'] === 1 || $modal_config['auto_refresh'] === '1') ? 'checked' : ''; ?>" data-checkbox="auto_refresh" onclick="toggleCheckbox(this, 'autoRefresh')"></div>
                            <label class="checkbox-label" onclick="toggleCheckbox(document.querySelector('[data-checkbox=\"auto_refresh\"]'), 'autoRefresh')">Automatische Aktualisierung</label>
                        </div>
                        <input type="hidden" name="auto_refresh" id="autoRefresh" value="<?php echo ($modal_config['auto_refresh'] === true || $modal_config['auto_refresh'] === 1 || $modal_config['auto_refresh'] === '1') ? '1' : '0'; ?>">

                        <div class="form-group" id="refreshIntervalGroup" style="<?php echo $modal_config['auto_refresh'] ? '' : 'display: none;'; ?>">
                            <label>Aktualisierungsintervall</label>
                            <select name="refresh_interval" class="form-control">
                                <option value="60" <?php echo $modal_config['refresh_interval'] == 60 ? 'selected' : ''; ?>>1 Minute</option>
                                <option value="300" <?php echo $modal_config['refresh_interval'] == 300 ? 'selected' : ''; ?>>5 Minuten</option>
                                <option value="600" <?php echo $modal_config['refresh_interval'] == 600 ? 'selected' : ''; ?>>10 Minuten</option>
                                <option value="1800" <?php echo $modal_config['refresh_interval'] == 1800 ? 'selected' : ''; ?>>30 Minuten</option>
                            </select>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="setup-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeSetupModal()">
                            <i class="fas fa-times"></i>
                            Überspringen
                        </button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            Dashboard einrichten
                        </button>
                    </div>
                </form>
            </div>
        </div>
    <?php endif; ?>

    <!-- Score Details Modal -->
    <div id="scoreDetailsModal" class="modal" style="display: none;">
        <div class="modal-content" style="max-width: 800px;">
                    <div class="modal-header">
                        <h3>🎯 Ihr persönlicher Performance-Score</h3>
                        <button class="modal-close" onclick="closeScoreDetails()">&times;</button>
                    </div>
            <div class="modal-form">
                <div class="score-details-content">
                    <!-- Termine Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #10b981;">
                                <i class="fas fa-calendar-check"></i>
                            </div>
                            <div class="category-info">
                                <h4>📅 Termine & Kundenbetreuung</h4>
                                <p>Wie gut verwalten Sie Termine und betreuen Ihre Kunden?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['appointments'], 1); ?>/40</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Alle Termine (letzte 3 Monate):</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['total_appointments'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">✅ Erfolgreich abgeschlossen:</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['completed_appointments'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🎯 Erfolgsquote:</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['completion_rate'] ?? 0; ?>%</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🕒 Aktuelle Aktivität (letzter Monat):</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['recent_appointments'] ?? 0; ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Events Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #3b82f6;">
                                <i class="fas fa-calendar-alt"></i>
                            </div>
                            <div class="category-info">
                                <h4>🎪 Events & Veranstaltungen</h4>
                                <p>Wie aktiv sind Sie bei der Organisation von Events?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['events'], 1); ?>/25</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Alle Events (letzte 3 Monate):</span>
                                <span class="detail-value"><?php echo $score_details['events']['total_events'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🟢 Aktuell laufende Events:</span>
                                <span class="detail-value"><?php echo $score_details['events']['active_events'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">📅 Geplante Events:</span>
                                <span class="detail-value"><?php echo $score_details['events']['upcoming_events'] ?? 0; ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Lager Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #f59e0b;">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <div class="category-info">
                                <h4>📦 Lager & Einlagerungen</h4>
                                <p>Wie gut organisieren Sie das Lager und die Einlagerungen?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['storage'], 1); ?>/15</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Alle Lageraktivitäten (letzte 3 Monate):</span>
                                <span class="detail-value"><?php echo $score_details['storage']['total_activities'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🕒 Aktuelle Aktivitäten (letzter Monat):</span>
                                <span class="detail-value"><?php echo $score_details['storage']['recent_activities'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🏷️ Verschiedene Lagerplätze:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['unique_storage_numbers'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">👥 Verschiedene Kunden:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['unique_customers'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">📥 Eingelagerte Artikel:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['stored_items'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">📤 Abgeholte Artikel:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['retrieved_items'] ?? 0; ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Effizienz Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #8b5cf6;">
                                <i class="fas fa-tachometer-alt"></i>
                            </div>
                            <div class="category-info">
                                <h4>⚡ Arbeits-Effizienz</h4>
                                <p>Wie schnell und qualitativ arbeiten Sie?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['efficiency'], 1); ?>/10</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">🎯 Aufgaben-Abschlussrate:</span>
                                <span class="detail-value"><?php echo $score_details['efficiency']['completion_rate'] ?? 0; ?>%</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🕒 Aktuelle Arbeitsaktivität:</span>
                                <span class="detail-value"><?php echo $score_details['efficiency']['recent_activity'] ?? 0; ?> Termine</span>
                            </div>
                        </div>
                    </div>

                    <!-- Konsistenz Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #06b6d4;">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <div class="category-info">
                                <h4>📈 Arbeits-Konsistenz</h4>
                                <p>Wie regelmäßig und kontinuierlich arbeiten Sie?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['consistency'], 1); ?>/10</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Regelmäßige Arbeitsaktivität:</span>
                                <span class="detail-value"><?php echo $score_details['consistency']['activity_consistency'] ?? 0; ?>%</span>
                            </div>
                        </div>
                    </div>

                    <!-- Score Tipps -->
                    <div class="score-tips">
                        <h4>💡 So verbessern Sie Ihren Score:</h4>
                        <ul>
                            <li><strong>📅 Termine & Kundenbetreuung:</strong> Schließen Sie mehr Termine erfolgreich ab und reduzieren Sie Stornierungen. Betreuen Sie Ihre Kunden aktiv.</li>
                            <li><strong>🎪 Events & Veranstaltungen:</strong> Erstellen Sie regelmäßig neue Events und halten Sie sie aktiv. Planen Sie Veranstaltungen im Voraus.</li>
                            <li><strong>📦 Lager & Einlagerungen:</strong> Dokumentieren Sie alle Lageraktivitäten regelmäßig. Organisieren Sie das Lager effizient.</li>
                            <li><strong>⚡ Arbeits-Effizienz:</strong> Arbeiten Sie kontinuierlich und schließen Sie Aufgaben schnell ab. Verbessern Sie Ihre Arbeitsgeschwindigkeit.</li>
                            <li><strong>📈 Arbeits-Konsistenz:</strong> Bleiben Sie regelmäßig aktiv und arbeiten Sie kontinuierlich. Vermeiden Sie längere Pausen.</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- JavaScript -->
    <script src="../assets/js/main.js"></script>
    <script src="assets/js/admin.js"></script>
    <script src="assets/js/unified-sidebar.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            console.log('🚀 Intelligent Dashboard loaded');
            
            // Load storage activities on page load
            <?php if ($selected_autohaus_id): ?>
                loadStorageActivities(<?php echo $selected_autohaus_id; ?>);
            <?php elseif (!$current_user_autohaus_id): ?>
                // User has no autohaus, wait for selection
                console.log('No autohaus selected, waiting for user selection...');
            <?php endif; ?>
            
            // Auto-refresh if enabled
            <?php if ($auto_refresh): ?>
                setInterval(() => {
                    refreshDashboard();
                }, <?php echo $refresh_interval * 1000; ?>);
            <?php endif; ?>
            
            // Animate elements on load
            const elements = document.querySelectorAll('.fade-in, .slide-in');
            elements.forEach((element, index) => {
                element.style.animationDelay = `${index * 0.1}s`;
            });
            
            // Initialize setup modal for new users
            <?php if ($show_setup_modal): ?>
                console.log('New user detected, initializing setup modal...');
                setTimeout(() => {
                    initializeSetupModal();
                }, 500);
            <?php endif; ?>
        });

        function refreshDashboard() {
            location.reload();
        }

        function refreshStats() {
            const btn = event.target.closest('.widget-btn');
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            
            setTimeout(() => {
                location.reload();
            }, 1000);
        }



        // Setup Modal Functions
        function openSetupModal() {
            // Get current user config from PHP
            const currentConfig = {
                layout: '<?php echo $user_dashboard_config['layout']; ?>',
                widgets: <?php echo json_encode($user_dashboard_config['widgets']); ?>,
                quick_actions: [], // Quick Actions are now managed via quick_actions_manager.php
                show_welcome: <?php echo $user_dashboard_config['show_welcome'] ? 'true' : 'false'; ?>,
                auto_refresh: <?php echo $user_dashboard_config['auto_refresh'] ? 'true' : 'false'; ?>,
                refresh_interval: <?php echo $user_dashboard_config['refresh_interval']; ?>
            };
            
            // Create modal dynamically with current config
            const modalHTML = `
                <div id="setupModal" class="setup-modal-overlay">
                    <div class="setup-modal">
                        <div class="setup-modal-header">
                            <h2>⚙️ Dashboard Konfiguration</h2>
                            <p>Passen Sie Ihr Dashboard nach Ihren Wünschen an</p>
                        </div>
                        
                        <form method="POST" id="dashboardSetupForm">
                            <input type="hidden" name="action" value="save_dashboard_config">
                            
                            <!-- Layout Selection -->
                            <div class="setup-section">
                                <h3>Layout auswählen</h3>
                                <div class="layout-grid">
                                    <div class="layout-card ${currentConfig.layout === 'default' ? 'selected' : ''}" data-layout="default">
                                        <div class="layout-preview default"></div>
                                        <div class="layout-name">Standard Layout</div>
                                        <div class="layout-description">Zwei-Spalten Layout mit Hauptinhalt und Sidebar</div>
                                        <div class="layout-meta">
                                            <span class="layout-tag">Tägliche Arbeit</span>
                                            <span>Desktop & Tablet</span>
                                        </div>
                                    </div>
                                    <div class="layout-card ${currentConfig.layout === 'compact' ? 'selected' : ''}" data-layout="compact">
                                        <div class="layout-preview compact"></div>
                                        <div class="layout-name">Kompaktes Layout</div>
                                        <div class="layout-description">Ein-Spalten Layout für fokussierte Arbeit</div>
                                        <div class="layout-meta">
                                            <span class="layout-tag">Mobile Geräte</span>
                                            <span>Mobile & Tablet</span>
                                        </div>
                                    </div>
                                    <div class="layout-card ${currentConfig.layout === 'wide' ? 'selected' : ''}" data-layout="wide">
                                        <div class="layout-preview wide"></div>
                                        <div class="layout-name">Breites Layout</div>
                                        <div class="layout-description">Drei-Spalten Layout für maximale Übersicht</div>
                                        <div class="layout-meta">
                                            <span class="layout-tag">Administratoren</span>
                                            <span>Desktop & Ultrawide</span>
                                        </div>
                                    </div>
                                </div>
                                <input type="hidden" name="layout" id="selectedLayout" value="${currentConfig.layout}">
                            </div>

                            <!-- Widget Selection -->
                            <div class="setup-section">
                                <h3>Widgets auswählen</h3>
                                <div class="widget-grid">
                                    <div class="widget-card ${currentConfig.widgets.includes('today_appointments') ? 'selected' : ''}" data-widget="today_appointments">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #007AFF;">
                                                <i class="fas fa-calendar-day"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Heutige Termine</h4>
                                                <p>Zeigt alle Termine für heute an</p>
                                            </div>
                                            <span class="widget-importance importance-high">high</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('upcoming_events') ? 'selected' : ''}" data-widget="upcoming_events">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #34C759;">
                                                <i class="fas fa-calendar-alt"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Kommende Events</h4>
                                                <p>Zeigt die nächsten Events an</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('quick_actions') ? 'selected' : ''}" data-widget="quick_actions">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #AF52DE;">
                                                <i class="fas fa-rocket"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Schnellaktionen</h4>
                                                <p>Ihre personalisierten Schnellzugriffe</p>
                                            </div>
                                            <span class="widget-importance importance-high">high</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('recent_storage') ? 'selected' : ''}" data-widget="recent_storage">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #FF9500;">
                                                <i class="fas fa-warehouse"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Letzte Lageraktivitäten</h4>
                                                <p>Zeigt die neuesten Einlagerungen an</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('notifications') ? 'selected' : ''}" data-widget="notifications">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #EF4444;">
                                                <i class="fas fa-bell"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Benachrichtigungen</h4>
                                                <p>Wichtige Updates und Erinnerungen</p>
                                            </div>
                                            <span class="widget-importance importance-high">high</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('activity_feed') ? 'selected' : ''}" data-widget="activity_feed">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #06B6D4;">
                                                <i class="fas fa-stream"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Aktivitäts-Feed</h4>
                                                <p>Zeigt Ihre letzten Aktivitäten an</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('smart_recommendations') ? 'selected' : ''}" data-widget="smart_recommendations">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #8B5CF6;">
                                                <i class="fas fa-brain"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>KI-Empfehlungen</h4>
                                                <p>Intelligente Vorschläge basierend auf Ihrem Verhalten</p>
                                            </div>
                                            <span class="widget-importance importance-low">low</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('quick_stats') ? 'selected' : ''}" data-widget="quick_stats">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #10B981;">
                                                <i class="fas fa-chart-pie"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Schnellstatistiken</h4>
                                                <p>Wichtige Kennzahlen auf einen Blick</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('task_reminders') ? 'selected' : ''}" data-widget="task_reminders">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #F59E0B;">
                                                <i class="fas fa-tasks"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Aufgaben & Erinnerungen</h4>
                                                <p>Ihre offenen Aufgaben und Termine</p>
                                            </div>
                                            <span class="widget-importance importance-high">high</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('system_health') ? 'selected' : ''}" data-widget="system_health">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #6366F1;">
                                                <i class="fas fa-heartbeat"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>System-Status</h4>
                                                <p>Übersicht über Systemzustand und Performance</p>
                                            </div>
                                            <span class="widget-importance importance-low">low</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('whatsapp_stats') ? 'selected' : ''}" data-widget="whatsapp_stats">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #25D366;">
                                                <i class="fab fa-whatsapp"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>WhatsApp-Statistiken</h4>
                                                <p>Übersicht über gesendete Nachrichten und Erfolgsrate</p>
                                            </div>
                                            <span class="widget-importance importance-high">high</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('whatsapp_activity') ? 'selected' : ''}" data-widget="whatsapp_activity">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #128C7E;">
                                                <i class="fas fa-paper-plane"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>WhatsApp-Aktivitäten</h4>
                                                <p>Letzte gesendete WhatsApp-Nachrichten</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('whatsapp_templates') ? 'selected' : ''}" data-widget="whatsapp_templates">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #075E54;">
                                                <i class="fas fa-file-alt"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>WhatsApp-Template-Status</h4>
                                                <p>Status der zugewiesenen Templates</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                </div>
                                <input type="hidden" name="widgets" id="selectedWidgets" value='${JSON.stringify(currentConfig.widgets)}'>
                            </div>

                            <!-- Quick Actions Setup -->
                            <div class="setup-section">
                                <!-- Quick Actions Manager Link -->
                                <div class="setup-section">
                                    <h3>Schnellaktionen verwalten</h3>
                                    <p>Verwalte deine personalisierten Schnellaktionen über den Quick Actions Manager.</p>
                                    <a href="quick_actions_manager.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2);">
                                        <i class="fas fa-rocket"></i>
                                        Schnellaktionen Manager öffnen
                                    </a>
                                </div>
                            </div>

                            <!-- Additional Settings -->
                            <div class="setup-section">
                                <h3>Dashboard-Einstellungen</h3>

                                <div class="checkbox-group">
                                    <div class="checkbox ${(currentConfig.show_welcome === true || currentConfig.show_welcome === 1 || currentConfig.show_welcome === '1') ? 'checked' : ''}" data-checkbox="show_welcome" onclick="toggleCheckbox(this, 'showWelcome')"></div>
                                    <label class="checkbox-label" onclick="toggleCheckbox(document.querySelector('[data-checkbox=\"show_welcome\"]'), 'showWelcome')">Willkommensnachricht anzeigen</label>
                                </div>
                                <input type="hidden" name="show_welcome" id="showWelcome" value="${(currentConfig.show_welcome === true || currentConfig.show_welcome === 1 || currentConfig.show_welcome === '1') ? '1' : '0'}">

                                <div class="checkbox-group">
                                    <div class="checkbox ${(currentConfig.auto_refresh === true || currentConfig.auto_refresh === 1 || currentConfig.auto_refresh === '1') ? 'checked' : ''}" data-checkbox="auto_refresh" onclick="toggleCheckbox(this, 'autoRefresh')"></div>
                                    <label class="checkbox-label" onclick="toggleCheckbox(document.querySelector('[data-checkbox=\"auto_refresh\"]'), 'autoRefresh')">Automatische Aktualisierung</label>
                                </div>
                                <input type="hidden" name="auto_refresh" id="autoRefresh" value="${(currentConfig.auto_refresh === true || currentConfig.auto_refresh === 1 || currentConfig.auto_refresh === '1') ? '1' : '0'}">

                                <div class="form-group" id="refreshIntervalGroup" style="${currentConfig.auto_refresh ? '' : 'display: none;'}">
                                    <label>Aktualisierungsintervall</label>
                                    <select name="refresh_interval" class="form-control">
                                        <option value="60" ${currentConfig.refresh_interval == 60 ? 'selected' : ''}>1 Minute</option>
                                        <option value="300" ${currentConfig.refresh_interval == 300 ? 'selected' : ''}>5 Minuten</option>
                                        <option value="600" ${currentConfig.refresh_interval == 600 ? 'selected' : ''}>10 Minuten</option>
                                        <option value="1800" ${currentConfig.refresh_interval == 1800 ? 'selected' : ''}>30 Minuten</option>
                                    </select>
                                </div>
                            </div>

                            <!-- Action Buttons -->
                            <div class="setup-actions">
                                <button type="button" class="btn btn-secondary" onclick="closeSetupModal()">
                                    <i class="fas fa-times"></i>
                                    Abbrechen
                                </button>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i>
                                    Änderungen speichern
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            `;
            
            document.body.insertAdjacentHTML('beforeend', modalHTML);
            
            // Initialize modal event listeners
            initializeSetupModal();
        }

        function closeSetupModal() {
            const modal = document.getElementById('setupModal');
            if (modal) {
                modal.style.animation = 'fadeOut 0.3s ease-in';
                setTimeout(() => {
                    modal.remove();
                }, 300);
            }
        }

        function initializeSetupModal() {
            const modal = document.getElementById('setupModal');
            if (!modal) {
                console.error('Setup modal not found!');
                return;
            }

            console.log('Initializing setup modal...');
            console.log('Modal found:', modal);
            console.log('Modal children:', modal.children.length);

            // Layout selection
            const layoutCards = modal.querySelectorAll('.layout-card');
            const selectedLayoutInput = modal.querySelector('#selectedLayout');
            
            console.log('Found layout cards:', layoutCards.length);
            
            layoutCards.forEach((card, index) => {
                console.log(`Setting up layout card ${index}:`, card);
                // Remove existing listeners
                card.removeEventListener('click', handleLayoutClick);
                // Add new listener
                card.addEventListener('click', handleLayoutClick);
                console.log(`Layout card ${index} event listener added`);
            });

            function handleLayoutClick(e) {
                e.preventDefault();
                e.stopPropagation();
                console.log('Layout card clicked:', this.dataset.layout);
                layoutCards.forEach(c => c.classList.remove('selected'));
                this.classList.add('selected');
                selectedLayoutInput.value = this.dataset.layout;
                console.log('Selected layout:', selectedLayoutInput.value);
            }

            // Widget selection
            const widgetCards = modal.querySelectorAll('.widget-card');
            const selectedWidgetsInput = modal.querySelector('#selectedWidgets');
            let selectedWidgets = JSON.parse(selectedWidgetsInput.value || '[]');
            
            console.log('Found widget cards:', widgetCards.length);
            
            widgetCards.forEach((card, index) => {
                console.log(`Setting up widget card ${index}:`, card);
                card.removeEventListener('click', handleWidgetClick);
                card.addEventListener('click', handleWidgetClick);
                console.log(`Widget card ${index} event listener added`);
            });

            function handleWidgetClick(e) {
                e.preventDefault();
                e.stopPropagation();
                const widgetId = this.dataset.widget;
                console.log('Widget card clicked:', widgetId);
                
                if (this.classList.contains('selected')) {
                    this.classList.remove('selected');
                    selectedWidgets = selectedWidgets.filter(id => id !== widgetId);
                } else {
                    this.classList.add('selected');
                    selectedWidgets.push(widgetId);
                }
                
                selectedWidgetsInput.value = JSON.stringify(selectedWidgets);
                console.log('Selected widgets:', selectedWidgets);
            }
            
            // Global toggleWidget function for onclick handlers
            window.toggleWidget = function(card, widgetId) {
                handleWidgetClick.call(card, { preventDefault: () => {}, stopPropagation: () => {} });
            };

            // Quick actions selection
            const quickActionCards = modal.querySelectorAll('.quick-action-card');
            const selectedQuickActionsInput = modal.querySelector('#selectedQuickActions');
            let selectedQuickActions = JSON.parse(selectedQuickActionsInput.value || '[]');
            
            console.log('Found quick action cards:', quickActionCards.length);
            
            quickActionCards.forEach((card, index) => {
                console.log(`Setting up quick action card ${index}:`, card);
                card.removeEventListener('click', handleQuickActionClick);
                card.addEventListener('click', handleQuickActionClick);
                console.log(`Quick action card ${index} event listener added`);
            });

            function handleQuickActionClick(e) {
                e.preventDefault();
                e.stopPropagation();
                const actionId = this.dataset.action;
                console.log('Quick action card clicked:', actionId);
                
                if (this.classList.contains('selected')) {
                    this.classList.remove('selected');
                    selectedQuickActions = selectedQuickActions.filter(id => id !== actionId);
                } else {
                    this.classList.add('selected');
                    selectedQuickActions.push(actionId);
                }
                
                selectedQuickActionsInput.value = JSON.stringify(selectedQuickActions);
            }

            // Checkbox handling - now handled by toggleCheckbox function
            // Initialize refresh interval visibility
            const autoRefreshCheckbox = modal.querySelector('[data-checkbox="auto_refresh"]');
            const refreshIntervalGroup = modal.querySelector('#refreshIntervalGroup');
            if (autoRefreshCheckbox && refreshIntervalGroup) {
                refreshIntervalGroup.style.display = autoRefreshCheckbox.classList.contains('checked') ? 'block' : 'none';
            }

            // Form submission
            const form = modal.querySelector('#dashboardSetupForm');
            if (form) {
                form.addEventListener('submit', function(e) {
                    e.preventDefault();
                    
                    const submitBtn = this.querySelector('button[type="submit"]');
                    const originalText = submitBtn.innerHTML;
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Speichern...';
                    submitBtn.disabled = true;
                    
                    setTimeout(() => {
                        this.submit();
                    }, 500);
                });
            }

            // Close modal on overlay click
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    console.log('Overlay clicked, closing modal');
                    closeSetupModal();
                }
            });

            console.log('Setup modal initialized successfully');
        }

        // Score Details Functions - Nur für Admin Deniz Uludag
        function showScoreDetails() {
            // Check if current user is admin Deniz Uludag
            <?php if ($current_user_role !== 'admin' || $current_user_name !== 'Deniz Uludag'): ?>
            console.log('❌ Access denied: Only admin Deniz Uludag can view score details');
            alert('Zugriff verweigert: Nur Admin Deniz Uludag kann Score-Details anzeigen');
            return;
            <?php endif; ?>
            
            console.log('🎯 Opening score details modal...');
            const modal = document.getElementById('scoreDetailsModal');
            if (modal) {
                console.log('✅ Modal found:', modal);
                modal.style.display = 'block';
                modal.style.opacity = '0';
                modal.style.transform = 'scale(0.9)';
                
                // Smooth fade in animation
                setTimeout(() => {
                    modal.style.transition = 'all 0.3s ease-out';
                    modal.style.opacity = '1';
                    modal.style.transform = 'scale(1)';
                }, 10);
                
                // Animate score breakdown bars
                setTimeout(() => {
                    const breakdownBars = modal.querySelectorAll('.breakdown-fill');
                    console.log('📊 Animating breakdown bars:', breakdownBars.length);
                    breakdownBars.forEach((bar, index) => {
                        const width = bar.style.width;
                        bar.style.width = '0%';
                        bar.style.transition = 'width 1s ease-out';
                        setTimeout(() => {
                            bar.style.width = width;
                        }, index * 100);
                    });
                }, 300);
            } else {
                console.error('❌ Score details modal not found!');
            }
        }

        function closeScoreDetails() {
            console.log('🎯 Closing score details modal...');
            const modal = document.getElementById('scoreDetailsModal');
            if (modal) {
                modal.style.transition = 'all 0.3s ease-in';
                modal.style.opacity = '0';
                modal.style.transform = 'scale(0.9)';
                setTimeout(() => {
                    modal.style.display = 'none';
                }, 300);
            }
        }

        // Close modal when clicking outside
        window.addEventListener('click', function(event) {
            const modal = document.getElementById('scoreDetailsModal');
            if (event.target === modal) {
                closeScoreDetails();
            }
        });

        // Ensure button click works
        document.addEventListener('DOMContentLoaded', function() {
            const scoreDetailsBtn = document.querySelector('.btn-score-details');
            if (scoreDetailsBtn) {
                console.log('✅ Score details button found');
                scoreDetailsBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('🎯 Button clicked!');
                    showScoreDetails();
                });
            } else {
                console.log('⚠️ Score details button not found');
            }
        });

        // REMOVED: Scoring JavaScript functions have been removed as requested

        // Toggle Checkbox Function
        function toggleCheckbox(checkbox, inputId) {
            if (!checkbox) return;
            
            const input = document.getElementById(inputId);
            if (!input) return;
            
            if (checkbox.classList.contains('checked')) {
                checkbox.classList.remove('checked');
                input.value = '0';
            } else {
                checkbox.classList.add('checked');
                input.value = '1';
            }
            
            // Special handling for auto_refresh checkbox
            if (inputId === 'autoRefresh') {
                const refreshIntervalGroup = document.getElementById('refreshIntervalGroup');
                if (refreshIntervalGroup) {
                    refreshIntervalGroup.style.display = checkbox.classList.contains('checked') ? 'block' : 'none';
                }
            }
        }

        // Load Storage Activities from API
        function loadStorageActivities(autohausId) {
            const container = document.getElementById('storage-activities-container');
            if (!container) return;
            
            // Show loading state
            container.innerHTML = `
                <div style="text-align: center; padding: var(--space-4); color: rgba(255, 255, 255, 0.8);">
                    <i class="fas fa-spinner fa-spin" style="font-size: 1.5rem; margin-bottom: var(--space-2);"></i>
                    <p style="font-size: 0.875rem;">Lade Lageraktivitäten...</p>
                </div>
            `;
            
            if (!autohausId) {
                container.innerHTML = `
                    <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                        <i class="fas fa-warehouse" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                        <p>Keine Lageraktivitäten</p>
                        <p style="font-size: 0.875rem; margin-top: var(--space-2); opacity: 0.7;">
                            Bitte wählen Sie ein Autohaus aus, um Aktivitäten anzuzeigen
                        </p>
                    </div>
                `;
                // Update stats count to 0
                updateStorageCount(0);
                return;
            }
            
            // Load wheelsets from API
            fetch(`../api/get_resy_wheelsets.php?autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    console.log('📦 Storage API Response:', data);
                    
                    // Update storage count in statistics widget
                    const storageCount = (data.success && data.wheelsets) ? data.wheelsets.length : 0;
                    updateStorageCount(storageCount);
                    
                    if (data.success && data.wheelsets && data.wheelsets.length > 0) {
                        // Sort by eingangsdatum (newest first) and take first 5
                        const sortedWheelsets = data.wheelsets
                            .sort((a, b) => {
                                const dateA = a.eingangsdatum ? new Date(a.eingangsdatum.split('.').reverse().join('-')) : new Date(0);
                                const dateB = b.eingangsdatum ? new Date(b.eingangsdatum.split('.').reverse().join('-')) : new Date(0);
                                return dateB - dateA;
                            })
                            .slice(0, 5);
                        
                        if (sortedWheelsets.length > 0) {
                            let html = '';
                            sortedWheelsets.forEach(wheelset => {
                                const customerName = wheelset.haltername || wheelset.kunde || 'Unbekannt';
                                const licensePlate = wheelset.kennzeichen || '';
                                const satznummer = wheelset.satznummer || '';
                                const eingangsdatum = wheelset.eingangsdatum || '';
                                
                                html += `
                                    <div class="activity-item">
                                        <div class="activity-icon" style="background: var(--warning);">
                                            <i class="fas fa-box"></i>
                                        </div>
                                        <div class="activity-content">
                                            <div class="activity-title">${escapeHtml(customerName)}</div>
                                            <div class="activity-description">
                                                ${escapeHtml(licensePlate)}
                                                ${satznummer ? ' • ' + escapeHtml(satznummer) : ''}
                                                ${eingangsdatum ? '<br><small style="opacity: 0.7;">Eingang: ' + escapeHtml(eingangsdatum) + '</small>' : ''}
                                            </div>
                                        </div>
                                        <div class="activity-time">
                                            ${eingangsdatum ? formatDate(eingangsdatum) : ''}
                                        </div>
                                    </div>
                                `;
                            });
                            container.innerHTML = html;
                        } else {
                            showNoStorageMessage(container);
                        }
                    } else {
                        showNoStorageMessage(container);
                    }
                })
                .catch(error => {
                    console.error('❌ Error loading storage activities:', error);
                    container.innerHTML = `
                        <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                            <i class="fas fa-exclamation-triangle" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                            <p>Fehler beim Laden der Daten</p>
                            <p style="font-size: 0.875rem; margin-top: var(--space-2); opacity: 0.7;">
                                Bitte versuchen Sie es später erneut
                            </p>
                        </div>
                    `;
                    // Update stats count to show error
                    updateStorageCount('-');
                });
        }
        
        // Update storage count in statistics widget
        function updateStorageCount(count) {
            const countElement = document.getElementById('storage-activities-count');
            if (countElement) {
                if (count === '-') {
                    countElement.innerHTML = '-';
                } else {
                    countElement.textContent = count;
                }
            }
        }
        
        function showNoStorageMessage(container) {
            container.innerHTML = `
                <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                    <i class="fas fa-warehouse" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                    <p>Keine Lageraktivitäten</p>
                    <p style="font-size: 0.875rem; margin-top: var(--space-2); opacity: 0.7;">
                        Keine Aktivitäten für das ausgewählte Autohaus gefunden.
                    </p>
                </div>
            `;
        }
        
        function escapeHtml(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        function formatDate(dateString) {
            if (!dateString) return '';
            // Convert DD.MM.YYYY to DD.MM format
            const parts = dateString.split('.');
            if (parts.length >= 2) {
                return parts[0] + '.' + parts[1];
            }
            return dateString;
        }
        
        // WhatsApp Widgets werden jetzt direkt aus PHP geladen
        // Keine zusätzliche JavaScript-Funktion mehr erforderlich

        // Mark all notifications as read

        // Refresh quick stats
        function refreshQuickStats() {
            location.reload();
        }

        // Refresh system health
        function refreshSystemHealth() {
            location.reload();
        }

        // Task Manager Functions
        function openTaskManager() {
            window.open('tasks_manager.php', '_blank', 'width=1200,height=800');
        }

        // Widget Order Management
        function moveWidgetUp(widgetId) {
            const widget = document.getElementById(widgetId);
            if (!widget) return;
            
            const parent = widget.parentElement;
            const prevSibling = widget.previousElementSibling;
            
            if (prevSibling && prevSibling.classList.contains('widget')) {
                parent.insertBefore(widget, prevSibling);
                saveWidgetOrder();
            }
        }

        function moveWidgetDown(widgetId) {
            const widget = document.getElementById(widgetId);
            if (!widget) return;
            
            const parent = widget.parentElement;
            const nextSibling = widget.nextElementSibling;
            
            if (nextSibling && nextSibling.classList.contains('widget')) {
                parent.insertBefore(nextSibling, widget);
                saveWidgetOrder();
            }
        }

        function saveWidgetOrder() {
            const widgets = document.querySelectorAll('.widget[id^="widget-"]');
            const widgetOrder = [];
            
            widgets.forEach(widget => {
                widgetOrder.push(widget.id);
            });
            
            fetch('api/save_widget_order.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ widget_order: widgetOrder })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Widget-Reihenfolge gespeichert');
                }
            })
            .catch(error => {
                console.error('Fehler beim Speichern der Widget-Reihenfolge:', error);
            });
        }

        // Load Quick Actions Dynamically
        function loadQuickActions() {
            const grid = document.getElementById('header-quick-actions-grid');
            if (!grid) return;
            
            fetch('api/quick_actions_api.php?action=list')
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.actions) {
                        const actions = data.actions.slice(0, 6);
                        if (actions.length > 0) {
                            grid.innerHTML = actions.map(action => `
                                <a href="${escapeHtml(action.url)}" class="header-quick-action" title="${escapeHtml(action.title)}">
                                    <div class="header-quick-action-icon" style="background: ${escapeHtml(action.background_color || action.color || '#007AFF')};">
                                        <i class="${escapeHtml(action.icon)}"></i>
                                    </div>
                                    <span>${escapeHtml(action.title)}</span>
                                </a>
                            `).join('');
                        } else {
                            grid.innerHTML = `
                                <div style="text-align: center; padding: var(--space-4); color: var(--label-secondary); font-size: 0.875rem;">
                                    <i class="fas fa-info-circle" style="margin-right: var(--space-2);"></i>
                                    Keine Schnellaktionen. <a href="quick_actions_manager.php" style="color: var(--primary); text-decoration: underline;">Erste erstellen</a>
                                </div>
                            `;
                        }
                    }
                })
                .catch(error => {
                    console.error('Fehler beim Laden der Schnellaktionen:', error);
                });
        }

        // Quick Actions Modal
        function openQuickActionsModal() {
            // Load quick actions
            fetch('api/quick_actions_api.php?action=list')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showQuickActionsModal(data.actions);
                    }
                })
                .catch(error => {
                    console.error('Fehler beim Laden der Schnellaktionen:', error);
                });
        }

        function showQuickActionsModal(actions) {
            const availableIcons = [
                'fas fa-home', 'fas fa-dashboard', 'fas fa-chart-bar', 'fas fa-users',
                'fas fa-calendar-alt', 'fas fa-tasks', 'fas fa-envelope', 'fas fa-phone',
                'fas fa-car', 'fas fa-tools', 'fas fa-warehouse', 'fas fa-clipboard-list',
                'fas fa-search', 'fas fa-download', 'fas fa-upload', 'fas fa-print',
                'fas fa-share-alt', 'fas fa-edit', 'fas fa-trash-alt', 'fas fa-star',
                'fas fa-heart', 'fas fa-bookmark', 'fas fa-flag', 'fas fa-rocket',
                'fas fa-cog', 'fas fa-user-cog', 'fas fa-database', 'fas fa-server',
                'fas fa-sign-out-alt', 'fas fa-plus', 'fas fa-external-link-alt', 'fas fa-calendar-check'
            ];
            
            const presetColors = [
                { name: 'Blau', value: '#007AFF' },
                { name: 'Lila', value: '#5856D6' },
                { name: 'Pink', value: '#AF52DE' },
                { name: 'Rot', value: '#FF2D92' },
                { name: 'Orange', value: '#FF3B30' },
                { name: 'Gelb', value: '#FF9500' },
                { name: 'Grün', value: '#34C759' },
                { name: 'Türkis', value: '#00C7BE' },
                { name: 'Himmelblau', value: '#64D2FF' },
                { name: 'Schwarz', value: '#1D1D1F' },
                { name: 'Grau', value: '#86868B' }
            ];
            
            const modalHTML = `
                <div id="quickActionsModal" class="setup-modal-overlay" style="display: flex;">
                    <div class="quick-actions-modal" style="max-width: 900px; max-height: 90vh; overflow: hidden; display: flex; flex-direction: column;">
                        <div class="quick-actions-header" style="padding: var(--space-6); border-bottom: 1px solid rgba(255, 255, 255, 0.1); display: flex; justify-content: space-between; align-items: center; flex-shrink: 0;">
                            <div>
                                <h2 style="margin: 0; font-size: 1.5rem; font-weight: 700; display: flex; align-items: center; gap: var(--space-2);">
                                    <span style="font-size: 1.75rem;">🚀</span>
                                    Schnellaktionen verwalten
                                </h2>
                                <p style="margin: var(--space-2) 0 0 0; font-size: 0.875rem; opacity: 0.7;">
                                    ${actions.length} Schnellaktion${actions.length !== 1 ? 'en' : ''} verfügbar
                                </p>
                            </div>
                            <button class="btn-close" onclick="closeQuickActionsModal()" style="background: rgba(255, 255, 255, 0.1); border: none; width: 40px; height: 40px; border-radius: var(--radius-lg); color: inherit; cursor: pointer; transition: all 0.3s;">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                        
                        <div class="quick-actions-body" style="flex: 1; overflow-y: auto; padding: var(--space-6);">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: var(--space-4);">
                                <div style="display: flex; gap: var(--space-2); align-items: center;">
                                    <input type="text" id="quick-actions-search" placeholder="🔍 Schnellaktionen durchsuchen..." 
                                           style="padding: var(--space-3) var(--space-4); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: var(--radius-lg); background: rgba(255, 255, 255, 0.1); color: inherit; width: 300px; font-size: 0.875rem;"
                                           onkeyup="filterQuickActions(this.value)">
                                </div>
                                <div style="display: flex; gap: var(--space-2);">
                                    <button class="btn btn-secondary" onclick="showTemplatesSection()" style="display: flex; align-items: center; gap: var(--space-2);">
                                        <i class="fas fa-layer-group"></i>
                                        Vorlagen
                                    </button>
                                    <button class="btn btn-primary" onclick="showCreateQuickActionForm()" style="display: flex; align-items: center; gap: var(--space-2);">
                                        <i class="fas fa-plus"></i>
                                        Neue erstellen
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Templates Section -->
                            <div id="templates-section" style="display: none; margin-bottom: var(--space-6); padding: var(--space-4); background: rgba(99, 102, 241, 0.1); border-radius: var(--radius-xl); border: 1px solid rgba(99, 102, 241, 0.3);">
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: var(--space-4);">
                                    <h3 style="margin: 0; font-size: 1.125rem; font-weight: 600; display: flex; align-items: center; gap: var(--space-2);">
                                        <i class="fas fa-layer-group"></i>
                                        Schnellaktionen-Vorlagen
                                    </h3>
                                    <button onclick="hideTemplatesSection()" style="background: none; border: none; color: inherit; cursor: pointer; opacity: 0.7; transition: opacity 0.3s;" onmouseover="this.style.opacity='1'" onmouseout="this.style.opacity='0.7'">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                <p style="margin: 0 0 var(--space-4) 0; font-size: 0.875rem; opacity: 0.8;">
                                    Wählen Sie eine Vorlage aus, um schnell eine neue Schnellaktion zu erstellen:
                                </p>
                                <div id="templates-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: var(--space-3);">
                                    <!-- Templates werden hier dynamisch eingefügt -->
                                </div>
                            </div>
                            
                            <div id="quick-actions-list" style="display: grid; gap: var(--space-3);">
                                ${actions.length === 0 ? `
                                    <div style="text-align: center; padding: var(--space-8); background: rgba(255, 255, 255, 0.05); border-radius: var(--radius-xl); border: 2px dashed rgba(255, 255, 255, 0.2);">
                                        <i class="fas fa-rocket" style="font-size: 3rem; opacity: 0.3; margin-bottom: var(--space-4);"></i>
                                        <h3 style="margin-bottom: var(--space-2);">Keine Schnellaktionen</h3>
                                        <p style="opacity: 0.7; margin-bottom: var(--space-4);">Erstellen Sie Ihre erste Schnellaktion für schnellen Zugriff</p>
                                        <button class="btn btn-primary" onclick="showCreateQuickActionForm()">
                                            <i class="fas fa-plus"></i>
                                            Erste Schnellaktion erstellen
                                        </button>
                                    </div>
                                ` : actions.map((action, index) => `
                                    <div class="quick-action-card" data-id="${action.id}" data-title="${action.title.toLowerCase()}" data-url="${action.url.toLowerCase()}"
                                         style="background: rgba(255, 255, 255, 0.08); border: 1px solid rgba(255, 255, 255, 0.1); border-radius: var(--radius-xl); padding: var(--space-4); transition: all 0.3s; cursor: move; position: relative;"
                                         onmouseenter="this.style.background='rgba(255, 255, 255, 0.12)'; this.style.transform='translateY(-2px)'; this.style.boxShadow='0 8px 24px rgba(0, 0, 0, 0.2)';"
                                         onmouseleave="this.style.background='rgba(255, 255, 255, 0.08)'; this.style.transform='translateY(0)'; this.style.boxShadow='none';">
                                        <div style="display: flex; align-items: center; gap: var(--space-4);">
                                            <div class="quick-action-drag-handle" style="cursor: grab; padding: var(--space-2); opacity: 0.5; transition: opacity 0.3s;" 
                                                 onmouseenter="this.style.opacity='1'" onmouseleave="this.style.opacity='0.5'">
                                                <i class="fas fa-grip-vertical" style="font-size: 1.25rem;"></i>
                                            </div>
                                            <div class="quick-action-icon-preview" style="background: ${action.background_color || '#007AFF'}; width: 56px; height: 56px; border-radius: var(--radius-xl); display: flex; align-items: center; justify-content: center; color: white; font-size: 1.5rem; box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2); flex-shrink: 0;">
                                                <i class="${action.icon || 'fas fa-circle'}"></i>
                                            </div>
                                            <div style="flex: 1; min-width: 0;">
                                                <div style="font-weight: 700; font-size: 1.125rem; margin-bottom: var(--space-1); word-break: break-word;">
                                                    ${action.title}
                                                </div>
                                                <div style="font-size: 0.875rem; opacity: 0.6; word-break: break-all; font-family: monospace;">
                                                    ${action.url}
                                                </div>
                                            </div>
                                            <div style="display: flex; gap: var(--space-2); flex-shrink: 0;">
                                                <button class="quick-action-btn" onclick="editQuickActionInline(${action.id})" title="Bearbeiten"
                                                        style="width: 40px; height: 40px; border-radius: var(--radius-lg); background: rgba(59, 130, 246, 0.2); border: 1px solid rgba(59, 130, 246, 0.3); color: #3b82f6; cursor: pointer; transition: all 0.3s; display: flex; align-items: center; justify-content: center;"
                                                        onmouseenter="this.style.background='rgba(59, 130, 246, 0.3)'; this.style.transform='scale(1.1)'"
                                                        onmouseleave="this.style.background='rgba(59, 130, 246, 0.2)'; this.style.transform='scale(1)'">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button class="quick-action-btn" onclick="deleteQuickAction(${action.id})" title="Löschen"
                                                        style="width: 40px; height: 40px; border-radius: var(--radius-lg); background: rgba(239, 68, 68, 0.2); border: 1px solid rgba(239, 68, 68, 0.3); color: #ef4444; cursor: pointer; transition: all 0.3s; display: flex; align-items: center; justify-content: center;"
                                                        onmouseenter="this.style.background='rgba(239, 68, 68, 0.3)'; this.style.transform='scale(1.1)'"
                                                        onmouseleave="this.style.background='rgba(239, 68, 68, 0.2)'; this.style.transform='scale(1)'">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                            
                            <div id="quick-action-form" style="display: none; margin-top: var(--space-6); padding: var(--space-6); background: linear-gradient(135deg, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1)); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: var(--radius-2xl); backdrop-filter: blur(20px);">
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: var(--space-4);">
                                    <h3 id="form-title" style="margin: 0; font-size: 1.25rem; font-weight: 700;">Neue Schnellaktion erstellen</h3>
                                    <button onclick="cancelQuickActionForm()" style="background: none; border: none; color: inherit; cursor: pointer; opacity: 0.7; transition: opacity 0.3s;" onmouseenter="this.style.opacity='1'" onmouseleave="this.style.opacity='0.7'">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                <form id="quick-action-form-element" onsubmit="saveQuickAction(event)">
                                    <input type="hidden" id="action-id" name="id">
                                    
                                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: var(--space-4); margin-bottom: var(--space-4);">
                                        <div class="form-group">
                                            <label style="display: block; margin-bottom: var(--space-2); font-weight: 600; font-size: 0.875rem;">Titel *</label>
                                            <input type="text" name="title" id="action-title" required 
                                                   style="width: 100%; padding: var(--space-3); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: var(--radius-lg); background: rgba(255, 255, 255, 0.1); color: inherit; font-size: 1rem;"
                                                   placeholder="z.B. Neues Event">
                                        </div>
                                        <div class="form-group">
                                            <label style="display: block; margin-bottom: var(--space-2); font-weight: 600; font-size: 0.875rem;">URL *</label>
                                            <input type="text" name="url" id="action-url" required 
                                                   style="width: 100%; padding: var(--space-3); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: var(--radius-lg); background: rgba(255, 255, 255, 0.1); color: inherit; font-size: 1rem;"
                                                   placeholder="z.B. events.php?action=create">
                                        </div>
                                    </div>
                                    
                                    <div class="form-group" style="margin-bottom: var(--space-4);">
                                        <label style="display: block; margin-bottom: var(--space-2); font-weight: 600; font-size: 0.875rem;">Icon auswählen</label>
                                        <div style="display: grid; grid-template-columns: repeat(8, 1fr); gap: var(--space-2); max-height: 200px; overflow-y: auto; padding: var(--space-2); background: rgba(0, 0, 0, 0.2); border-radius: var(--radius-lg);">
                                            ${availableIcons.map(icon => `
                                                <button type="button" class="icon-picker-btn" data-icon="${icon}" onclick="selectIcon('${icon}')"
                                                        style="width: 40px; height: 40px; border-radius: var(--radius-md); background: rgba(255, 255, 255, 0.1); border: 2px solid transparent; color: inherit; cursor: pointer; transition: all 0.3s; display: flex; align-items: center; justify-content: center;"
                                                        onmouseenter="this.style.background='rgba(255, 255, 255, 0.2)'; this.style.transform='scale(1.1)'"
                                                        onmouseleave="this.style.background='rgba(255, 255, 255, 0.1)'; this.style.transform='scale(1)'">
                                                    <i class="${icon}"></i>
                                                </button>
                                            `).join('')}
                                        </div>
                                        <input type="hidden" name="icon" id="action-icon" value="fas fa-circle">
                                        <div id="selected-icon-preview" style="margin-top: var(--space-2); display: flex; align-items: center; gap: var(--space-2);">
                                            <span style="font-size: 0.875rem; opacity: 0.7;">Ausgewähltes Icon:</span>
                                            <div id="icon-preview" style="width: 32px; height: 32px; border-radius: var(--radius-md); background: #007AFF; display: flex; align-items: center; justify-content: center; color: white;">
                                                <i class="fas fa-circle"></i>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group" style="margin-bottom: var(--space-4);">
                                        <label style="display: block; margin-bottom: var(--space-2); font-weight: 600; font-size: 0.875rem;">Farbe auswählen</label>
                                        <div style="display: grid; grid-template-columns: repeat(6, 1fr); gap: var(--space-2); margin-bottom: var(--space-3);">
                                            ${presetColors.map(color => `
                                                <button type="button" class="color-picker-btn" data-color="${color.value}" onclick="selectColor('${color.value}')"
                                                        style="width: 100%; height: 48px; border-radius: var(--radius-lg); background: ${color.value}; border: 3px solid transparent; cursor: pointer; transition: all 0.3s; position: relative;"
                                                        onmouseenter="this.style.transform='scale(1.05)'; this.style.boxShadow='0 4px 12px rgba(0, 0, 0, 0.3)'"
                                                        onmouseleave="this.style.transform='scale(1)'; this.style.boxShadow='none'"
                                                        title="${color.name}">
                                                    <span style="position: absolute; bottom: 4px; left: 50%; transform: translateX(-50%); font-size: 0.75rem; color: ${color.value === '#1D1D1F' || color.value === '#86868B' ? 'white' : 'white'}; text-shadow: 0 1px 2px rgba(0,0,0,0.5);">${color.name}</span>
                                                </button>
                                            `).join('')}
                                        </div>
                                        <div style="display: flex; align-items: center; gap: var(--space-3);">
                                            <input type="color" name="color" id="action-color" value="#007AFF" 
                                                   style="width: 80px; height: 48px; border-radius: var(--radius-lg); border: 2px solid rgba(255, 255, 255, 0.2); cursor: pointer;">
                                            <div style="flex: 1;">
                                                <div style="font-size: 0.875rem; opacity: 0.7; margin-bottom: var(--space-1);">Oder eigene Farbe wählen:</div>
                                                <input type="text" id="color-hex-input" value="#007AFF" placeholder="#007AFF"
                                                       style="width: 100%; padding: var(--space-2); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: var(--radius-md); background: rgba(255, 255, 255, 0.1); color: inherit; font-family: monospace;"
                                                       onchange="document.getElementById('action-color').value = this.value; updateColorPreview(this.value);">
                                            </div>
                                        </div>
                                        <div id="color-preview" style="margin-top: var(--space-3); padding: var(--space-3); background: #007AFF; border-radius: var(--radius-lg); text-align: center; color: white; font-weight: 600;">
                                            Vorschau
                                        </div>
                                    </div>
                                    
                                    <div style="display: flex; gap: var(--space-3); margin-top: var(--space-6);">
                                        <button type="submit" class="btn btn-primary" style="flex: 1; padding: var(--space-4); font-size: 1rem; font-weight: 600;">
                                            <i class="fas fa-save"></i>
                                            Speichern
                                        </button>
                                        <button type="button" class="btn btn-secondary" onclick="cancelQuickActionForm()" style="padding: var(--space-4);">
                                            Abbrechen
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            // Remove existing modal if any
            const existingModal = document.getElementById('quickActionsModal');
            if (existingModal) {
                existingModal.remove();
            }
            
            document.body.insertAdjacentHTML('beforeend', modalHTML);
            
            // Initialize color preview
            updateColorPreview('#007AFF');
            
            // Initialize icon selection
            selectIcon('fas fa-circle');
            
            // Add event listeners for color picker
            const colorInput = document.getElementById('action-color');
            const colorHexInput = document.getElementById('color-hex-input');
            
            if (colorInput) {
                colorInput.addEventListener('input', function(e) {
                    colorHexInput.value = e.target.value;
                    updateColorPreview(e.target.value);
                });
            }
            
            if (colorHexInput) {
                colorHexInput.addEventListener('input', function(e) {
                    const hex = e.target.value;
                    if (/^#[0-9A-F]{6}$/i.test(hex)) {
                        document.getElementById('action-color').value = hex;
                        updateColorPreview(hex);
                    }
                });
            }
        }
        
        function selectIcon(icon) {
            document.getElementById('action-icon').value = icon;
            document.querySelectorAll('.icon-picker-btn').forEach(btn => {
                btn.style.border = '2px solid transparent';
                btn.style.background = 'rgba(255, 255, 255, 0.1)';
            });
            const selectedBtn = document.querySelector(`[data-icon="${icon}"]`);
            if (selectedBtn) {
                selectedBtn.style.border = '2px solid #3b82f6';
                selectedBtn.style.background = 'rgba(59, 130, 246, 0.3)';
            }
            document.getElementById('icon-preview').innerHTML = `<i class="${icon}"></i>`;
        }
        
        function selectColor(color) {
            document.getElementById('action-color').value = color;
            document.getElementById('color-hex-input').value = color;
            updateColorPreview(color);
            document.querySelectorAll('.color-picker-btn').forEach(btn => {
                btn.style.border = '3px solid transparent';
            });
            const selectedBtn = document.querySelector(`[data-color="${color}"]`);
            if (selectedBtn) {
                selectedBtn.style.border = '3px solid white';
                selectedBtn.style.boxShadow = '0 0 0 2px rgba(255, 255, 255, 0.5)';
            }
        }
        
        function updateColorPreview(color) {
            const preview = document.getElementById('color-preview');
            if (preview) {
                preview.style.background = color;
            }
        }
        
        function filterQuickActions(searchTerm) {
            const cards = document.querySelectorAll('.quick-action-card');
            const term = searchTerm.toLowerCase();
            cards.forEach(card => {
                const title = card.dataset.title || '';
                const url = card.dataset.url || '';
                if (title.includes(term) || url.includes(term)) {
                    card.style.display = '';
                } else {
                    card.style.display = 'none';
                }
            });
        }
        
        function editQuickActionInline(id) {
            fetch(`api/quick_actions_api.php?action=list`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const action = data.actions.find(a => a.id == id);
                        if (action) {
                            document.getElementById('form-title').textContent = 'Schnellaktion bearbeiten';
                            document.getElementById('action-id').value = action.id;
                            document.getElementById('action-title').value = action.title;
                            document.getElementById('action-url').value = action.url;
                            document.getElementById('action-icon').value = action.icon || 'fas fa-circle';
                            document.getElementById('action-color').value = action.background_color || '#007AFF';
                            document.getElementById('color-hex-input').value = action.background_color || '#007AFF';
                            
                            selectIcon(action.icon || 'fas fa-circle');
                            selectColor(action.background_color || '#007AFF');
                            
                            document.getElementById('quick-action-form').style.display = 'block';
                            document.getElementById('quick-action-form').scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                        }
                    }
                });
        }

        function closeQuickActionsModal() {
            const modal = document.getElementById('quickActionsModal');
            if (modal) {
                modal.style.animation = 'fadeOut 0.3s ease-in';
                setTimeout(() => {
                    modal.remove();
                }, 300);
            }
        }

        function showCreateQuickActionForm() {
            document.getElementById('form-title').textContent = 'Neue Schnellaktion erstellen';
            document.getElementById('action-id').value = '';
            document.getElementById('action-title').value = '';
            document.getElementById('action-url').value = '';
            document.getElementById('action-icon').value = 'fas fa-circle';
            document.getElementById('action-color').value = '#007AFF';
            document.getElementById('color-hex-input').value = '#007AFF';
            
            selectIcon('fas fa-circle');
            selectColor('#007AFF');
            
            document.getElementById('quick-action-form').style.display = 'block';
            document.getElementById('quick-action-form').scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }

        function deleteQuickAction(id) {
            if (!confirm('Schnellaktion wirklich löschen?')) return;
            
            const formData = new FormData();
            formData.append('action', 'delete');
            formData.append('id', id);
            
            fetch('api/quick_actions_api.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Reload modal and header actions
                    loadQuickActions();
                    openQuickActionsModal(); // Reload modal
                } else {
                    alert('Fehler: ' + data.error);
                }
            });
        }

        function saveQuickAction(event) {
            event.preventDefault();
            
            const formData = new FormData(event.target);
            const actionId = formData.get('id');
            formData.append('action', actionId ? 'update' : 'create');
            
            // Show loading state
            const submitBtn = event.target.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Speichern...';
            submitBtn.disabled = true;
            
            fetch('api/quick_actions_api.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show success message
                    const form = document.getElementById('quick-action-form');
                    form.style.background = 'linear-gradient(135deg, rgba(16, 185, 129, 0.2), rgba(5, 150, 105, 0.2))';
                    form.style.borderColor = 'rgba(16, 185, 129, 0.5)';
                    
                    // Reload header actions immediately
                    loadQuickActions();
                    
                    setTimeout(() => {
                        openQuickActionsModal(); // Reload modal
                    }, 500);
                } else {
                    alert('Fehler: ' + data.error);
                    submitBtn.innerHTML = originalText;
                    submitBtn.disabled = false;
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Fehler beim Speichern');
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        }

        function cancelQuickActionForm() {
            const form = document.getElementById('quick-action-form');
            form.style.animation = 'fadeOut 0.3s ease-in';
            setTimeout(() => {
                form.style.display = 'none';
                form.style.animation = '';
                form.style.background = 'linear-gradient(135deg, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1))';
                form.style.borderColor = 'rgba(255, 255, 255, 0.2)';
            }, 300);
        }

        // Templates für Seiten, die nicht in der Sidebar sind
        const quickActionTemplates = [
            { title: 'Profil', url: 'profile.php', icon: 'fas fa-user', color: '#007AFF' },
            { title: 'Aufgaben Manager', url: 'tasks_manager.php', icon: 'fas fa-tasks', color: '#F59E0B' },
            { title: 'Performance', url: 'performance.php', icon: 'fas fa-chart-line', color: '#8B5CF6' },
            { title: 'Audit Log', url: 'audit_log.php', icon: 'fas fa-clipboard-list', color: '#6366F1' },
            { title: 'Cache Management', url: 'cache_management.php', icon: 'fas fa-database', color: '#10B981' },
            { title: 'RESY Import', url: 'import_resy_data.php', icon: 'fas fa-upload', color: '#06B6D4' },
            { title: 'Project Server', url: 'project_server.php', icon: 'fas fa-server', color: '#EF4444' },
            { title: 'Schnellaktionen Manager', url: 'quick_actions_manager.php', icon: 'fas fa-rocket', color: '#AF52DE' }
        ];

        function showTemplatesSection() {
            const section = document.getElementById('templates-section');
            const grid = document.getElementById('templates-grid');
            if (!section || !grid) return;
            
            // Render templates
            grid.innerHTML = quickActionTemplates.map((template, index) => `
                <div class="template-card" data-template-index="${index}"
                     style="background: rgba(255, 255, 255, 0.08); border: 1px solid rgba(255, 255, 255, 0.1); border-radius: var(--radius-lg); padding: var(--space-4); cursor: pointer; transition: all 0.3s; text-align: center;"
                     onmouseenter="this.style.background='rgba(255, 255, 255, 0.12)'; this.style.transform='translateY(-2px)'; this.style.boxShadow='0 8px 24px rgba(0, 0, 0, 0.2)';"
                     onmouseleave="this.style.background='rgba(255, 255, 255, 0.08)'; this.style.transform='translateY(0)'; this.style.boxShadow='none';">
                    <div style="width: 48px; height: 48px; border-radius: var(--radius-lg); background: ${escapeHtml(template.color)}; display: flex; align-items: center; justify-content: center; color: white; font-size: 1.25rem; margin: 0 auto var(--space-2);">
                        <i class="${escapeHtml(template.icon)}"></i>
                    </div>
                    <div style="font-weight: 600; font-size: 0.875rem; margin-bottom: var(--space-1);">${escapeHtml(template.title)}</div>
                    <div style="font-size: 0.75rem; opacity: 0.6; font-family: monospace; word-break: break-all;">${escapeHtml(template.url)}</div>
                </div>
            `).join('');
            
            // Add click event listeners
            grid.querySelectorAll('.template-card').forEach((card, index) => {
                card.addEventListener('click', () => {
                    createFromTemplate(quickActionTemplates[index]);
                });
            });
            
            section.style.display = 'block';
            section.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }

        function hideTemplatesSection() {
            const section = document.getElementById('templates-section');
            if (section) {
                section.style.display = 'none';
            }
        }

        function createFromTemplate(template) {
            // Fill form with template data
            document.getElementById('form-title').textContent = 'Neue Schnellaktion erstellen';
            document.getElementById('action-id').value = '';
            document.getElementById('action-title').value = template.title;
            document.getElementById('action-url').value = template.url;
            document.getElementById('action-icon').value = template.icon;
            document.getElementById('action-color').value = template.color;
            document.getElementById('color-hex-input').value = template.color;
            
            selectIcon(template.icon);
            selectColor(template.color);
            
            // Hide templates section and show form
            hideTemplatesSection();
            document.getElementById('quick-action-form').style.display = 'block';
            document.getElementById('quick-action-form').scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }

        function completeTask(taskId) {
            if (!confirm('Möchten Sie diese Aufgabe als erledigt markieren?')) {
                return;
            }
            
            fetch('api/tasks_manager.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=complete&id=${taskId}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Reload the task container
                    location.reload();
                } else {
                    alert('Fehler: ' + (data.error || 'Unbekannter Fehler'));
                }
            })
            .catch(error => {
                console.error('Error completing task:', error);
                alert('Fehler beim Markieren der Aufgabe als erledigt');
            });
        }


        // Storage Autohaus Filter Function
        function filterStorageByAutohaus(autohausId) {
            console.log('🔍 Filtering storage by autohaus:', autohausId);
            
            // Show loading state
            const widget = document.querySelector('.widget:has(#storage-autohaus-filter)');
            if (widget) {
                widget.style.opacity = '0.6';
                widget.style.pointerEvents = 'none';
            }
            
            // Save selection to session via AJAX
            const formData = new URLSearchParams();
            formData.append('action', 'update_storage_autohaus_filter');
            formData.append('autohaus_id', autohausId || '');
            
            fetch('index.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: formData.toString()
            })
            .then(response => {
                console.log('📡 Response status:', response.status);
                return response.json();
            })
            .then(data => {
                console.log('✅ Response data:', data);
                if (data.success) {
                    // Reload page to show filtered results
                    // Update storage activities without reloading page
                    console.log('🔄 Updating storage activities...');
                    loadStorageActivities(autohausId || null);
                    if (widget) {
                        widget.style.opacity = '1';
                        widget.style.pointerEvents = 'auto';
                    }
                } else {
                    console.error('❌ Error updating filter:', data);
                    alert('Fehler beim Aktualisieren des Filters. Bitte versuchen Sie es erneut.');
                    if (widget) {
                        widget.style.opacity = '1';
                        widget.style.pointerEvents = 'auto';
                    }
                }
            })
            .catch(error => {
                console.error('❌ Fetch error:', error);
                alert('Fehler beim Laden der Daten. Bitte versuchen Sie es erneut.');
                if (widget) {
                    widget.style.opacity = '1';
                    widget.style.pointerEvents = 'auto';
                }
            });
        }

    </script>
</body>
</html>


