<?php
/**
 * WhatsApp 2FA - 2FA per WhatsApp über HelloMateo API
 * Für Boxenstop Admin-System
 */

/**
 * Generiert einen 6-stelligen 2FA-Code
 */
function generateWhatsApp2FACode() {
    return str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
}

/**
 * Speichert einen 2FA-Code für WhatsApp-Verifizierung
 */
function saveWhatsApp2FACode($userId, $code) {
    require_once __DIR__ . '/../../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Erstelle Tabelle falls nicht vorhanden
    $conn->exec("
        CREATE TABLE IF NOT EXISTS admin_2fa_codes (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            code VARCHAR(6) NOT NULL,
            type VARCHAR(20) DEFAULT 'whatsapp',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            expires_at DATETIME NOT NULL,
            used TINYINT(1) DEFAULT 0,
            INDEX idx_user_code (user_id, code),
            INDEX idx_expires (expires_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    // Lösche alte Codes für diesen Benutzer
    $stmt = $conn->prepare("DELETE FROM admin_2fa_codes WHERE user_id = ?");
    $stmt->execute([$userId]);
    
    // Speichere neuen Code (gültig für 10 Minuten)
    $expiresAt = date('Y-m-d H:i:s', time() + 600);
    $stmt = $conn->prepare("
        INSERT INTO admin_2fa_codes (user_id, code, expires_at)
        VALUES (?, ?, ?)
    ");
    return $stmt->execute([$userId, $code, $expiresAt]);
}

/**
 * Generiert und sendet 2FA-Code per WhatsApp über Template
 * 1:1 aus whatsappp2kiel übernommen
 */
function generateAndSendWhatsApp2FA($userId, $phone, $autohausId = null) {
    $code = generateWhatsApp2FACode();
    
    // Speichere Code in Datenbank
    if (!saveWhatsApp2FACode($userId, $code)) {
        return ['success' => false, 'error' => 'Fehler beim Speichern des Codes'];
    }
    
    // Sende Code per WhatsApp über HelloMateo API mit Template
    try {
        require_once __DIR__ . '/../../config/database.php';
        
        // Template-ID für 2FA-Code
        $templateId = 'f570d54a-3ee9-476c-b553-72bbd72790ff';
        
        // Channel-ID für B&E Autocentrum Kiel
        $channelId = 'd8251c7f-22f7-4c6f-b632-12469f4dc0a3';
        
        // Hole API-Key aus Datenbank (Autohaus ID 1 oder übergebenes autohausId)
        $db = new Database();
        $conn = $db->getConnection();
        
        $apiKey = null;
        if ($autohausId) {
            // Versuche API-Key aus autohaus Tabelle zu holen
            try {
                $checkStmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_api_key'");
                if ($checkStmt->rowCount() > 0) {
                    $stmt = $conn->prepare("SELECT whatsapp_api_key FROM autohaus WHERE id = ?");
                    $stmt->execute([$autohausId]);
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    if ($result && !empty(trim($result['whatsapp_api_key']))) {
                        $apiKey = trim($result['whatsapp_api_key']);
                    }
                }
            } catch (Exception $e) {
                error_log("Fehler beim Laden des Autohaus API-Keys: " . $e->getMessage());
            }
        }
        
        // Fallback: Hole API-Key aus settings oder Autohaus ID 1
        if (empty($apiKey)) {
            // Versuche Autohaus ID 1
            try {
                $checkStmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_api_key'");
                if ($checkStmt->rowCount() > 0) {
                    $stmt = $conn->prepare("SELECT whatsapp_api_key FROM autohaus WHERE id = 1");
                    $stmt->execute();
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    if ($result && !empty(trim($result['whatsapp_api_key']))) {
                        $apiKey = trim($result['whatsapp_api_key']);
                    }
                }
            } catch (Exception $e) {
                // Ignoriere Fehler
            }
        }
        
        // Fallback: Hole aus settings Tabelle
        if (empty($apiKey)) {
            try {
                $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_api_key'");
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($result && !empty(trim($result['setting_value']))) {
                    $apiKey = trim($result['setting_value']);
                }
            } catch (Exception $e) {
                // Ignoriere Fehler
            }
        }
        
        if (empty($apiKey)) {
            return ['success' => false, 'error' => 'API-Key nicht gefunden'];
        }
        
        // Formatiere Telefonnummer
        $formattedPhone = preg_replace('/[^0-9+]/', '', $phone);
        if (!str_starts_with($formattedPhone, '+')) {
            if (str_starts_with($formattedPhone, '0')) {
                $formattedPhone = '+49' . substr($formattedPhone, 1);
            } else {
                $formattedPhone = '+49' . $formattedPhone;
            }
        }
        
        error_log("WhatsApp 2FA: Sende Code {$code} an {$formattedPhone} mit Template {$templateId}");
        
        // Direkter API-Aufruf mit placeholder_values (1:1 aus whatsappp2kiel)
        $requestData = [
            'to' => [['handle' => $formattedPhone]],
            'template_id' => $templateId,
            'from' => $channelId,
            'placeholder_values' => [$code]  // Array mit dem Code als erstes Element
        ];
        
        error_log("WhatsApp 2FA: Request Data = " . json_encode($requestData));
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://integration.getmateo.com/api/v1/send_message');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        error_log("WhatsApp 2FA: HTTP Code = {$httpCode}");
        error_log("WhatsApp 2FA: Response = " . substr($response, 0, 500));
        
        if ($error) {
            return ['success' => false, 'error' => 'cURL Fehler: ' . $error];
        }
        
        $responseData = json_decode($response, true);
        
        if ($httpCode >= 200 && $httpCode < 300) {
            // Prüfe ob Message-ID vorhanden ist
            $messageId = $responseData['message']['id'] ?? null;
            if ($messageId) {
                return ['success' => true, 'message' => '2FA-Code wurde per WhatsApp gesendet'];
            } else {
                return ['success' => false, 'error' => 'Keine Message-ID in Antwort'];
            }
        } else {
            $errorMsg = $responseData['message'] ?? $responseData['error'] ?? 'API-Fehler';
            return ['success' => false, 'error' => $errorMsg];
        }
    } catch (Exception $e) {
        error_log("WhatsApp 2FA Exception: " . $e->getMessage());
        return ['success' => false, 'error' => 'Fehler beim Senden: ' . $e->getMessage()];
    }
}

/**
 * Verifiziert WhatsApp 2FA-Code
 */
function verifyWhatsApp2FACode($userId, $code) {
    $code = preg_replace('/\s+/', '', $code);
    
    if (strlen($code) !== 6 || !ctype_digit($code)) {
        return false;
    }
    
    require_once __DIR__ . '/../../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Suche gültigen Code
    $stmt = $conn->prepare("
        SELECT id FROM admin_2fa_codes 
        WHERE user_id = ? AND code = ? AND expires_at > NOW() AND used = 0
        ORDER BY created_at DESC
        LIMIT 1
    ");
    $stmt->execute([$userId, $code]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($result) {
        // Markiere Code als verwendet
        $updateStmt = $conn->prepare("UPDATE admin_2fa_codes SET used = 1 WHERE id = ?");
        $updateStmt->execute([$result['id']]);
        return true;
    }
    
    return false;
}

/**
 * Sendet einen Verifizierungscode per WhatsApp (ohne automatische Verifizierung)
 */
function sendPhoneVerificationCode($userId, $phone, $autohausId = null) {
    // Sende Code per WhatsApp
    $result = generateAndSendWhatsApp2FA($userId, $phone, $autohausId);
    
    require_once __DIR__ . '/../../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Erweitere Tabelle um Verifizierungsstatus falls nicht vorhanden
    try {
        $tableExists = $conn->query("SHOW TABLES LIKE 'admin_user_2fa'")->rowCount() > 0;
        if ($tableExists) {
            $columns = $conn->query("SHOW COLUMNS FROM admin_user_2fa")->fetchAll(PDO::FETCH_COLUMN);
            
            if (!in_array('phone_verified', $columns)) {
                $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verified TINYINT(1) DEFAULT 0");
            }
            if (!in_array('phone_verified_at', $columns)) {
                $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verified_at DATETIME NULL");
            }
            if (!in_array('phone_verification_error', $columns)) {
                $conn->exec("ALTER TABLE admin_user_2fa ADD COLUMN phone_verification_error VARCHAR(255) NULL");
            }
        }
    } catch (PDOException $e) {
        error_log("Phone verification columns check: " . $e->getMessage());
    }
    
    // Setze Status zurück auf "nicht verifiziert" wenn Code gesendet wurde
    if ($result['success']) {
        $stmt = $conn->prepare("
            UPDATE admin_user_2fa 
            SET phone_verified = 0, 
                phone_verified_at = NULL,
                phone_verification_error = NULL,
                updated_at = NOW()
            WHERE user_id = ?
        ");
        $stmt->execute([$userId]);
    } else {
        // Speichere Fehler
        $stmt = $conn->prepare("
            UPDATE admin_user_2fa 
            SET phone_verified = 0,
                phone_verification_error = ?,
                updated_at = NOW()
            WHERE user_id = ?
        ");
        $stmt->execute([$result['error'] ?? 'Unbekannter Fehler', $userId]);
    }
    
    return $result;
}

/**
 * Verifiziert eine Telefonnummer durch Überprüfung des eingegebenen Codes
 */
function verifyPhoneNumberWithCode($userId, $code) {
    require_once __DIR__ . '/../../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Verifiziere den Code
    $isValid = verifyWhatsApp2FACode($userId, $code);
    
    if ($isValid) {
        // Code ist gültig - markiere als verifiziert
        $stmt = $conn->prepare("
            UPDATE admin_user_2fa 
            SET phone_verified = 1, 
                phone_verified_at = NOW(),
                phone_verification_error = NULL,
                updated_at = NOW()
            WHERE user_id = ?
        ");
        $stmt->execute([$userId]);
        
        return ['success' => true, 'message' => 'Telefonnummer wurde erfolgreich verifiziert'];
    } else {
        // Code ist ungültig
        $stmt = $conn->prepare("
            UPDATE admin_user_2fa 
            SET phone_verified = 0,
                phone_verification_error = 'Ungültiger Verifizierungscode',
                updated_at = NOW()
            WHERE user_id = ?
        ");
        $stmt->execute([$userId]);
        
        return ['success' => false, 'error' => 'Ungültiger Verifizierungscode'];
    }
}

/**
 * Setzt 2FA-Methode auf WhatsApp und sendet Verifizierungscode
 */
function setWhatsApp2FA($userId, $phone, $sendVerificationCode = true, $autohausId = null) {
    require_once __DIR__ . '/../../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Erstelle Tabelle falls nicht vorhanden
    $conn->exec("
        CREATE TABLE IF NOT EXISTS admin_user_2fa (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL UNIQUE,
            two_factor_secret VARCHAR(32),
            two_factor_method VARCHAR(20) DEFAULT 'whatsapp',
            two_factor_enabled TINYINT(1) DEFAULT 1,
            phone VARCHAR(20) DEFAULT NULL,
            phone_verified TINYINT(1) DEFAULT 0,
            phone_verified_at DATETIME NULL,
            phone_verification_error VARCHAR(255) NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_user_id (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    // Speichere WhatsApp 2FA Einstellungen (noch nicht verifiziert)
    $stmt = $conn->prepare("
        INSERT INTO admin_user_2fa (user_id, two_factor_method, two_factor_enabled, phone, phone_verified)
        VALUES (?, 'whatsapp', 1, ?, 0)
        ON DUPLICATE KEY UPDATE two_factor_method = 'whatsapp', two_factor_enabled = 1, phone = VALUES(phone), phone_verified = 0, phone_verified_at = NULL, phone_verification_error = NULL, updated_at = NOW()
    ");
    $stmt->execute([$userId, $phone]);
    
    // Sende Verifizierungscode wenn gewünscht (aber noch nicht verifizieren!)
    if ($sendVerificationCode) {
        return sendPhoneVerificationCode($userId, $phone, $autohausId);
    }
    
    return ['success' => true, 'message' => '2FA wurde gesetzt. Verifizierungscode wird gesendet.'];
}

/**
 * Prüft ob 2FA für einen Benutzer aktiviert ist
 */
function is2FAEnabled($userId) {
    require_once __DIR__ . '/../../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Erstelle Tabelle falls nicht vorhanden
    $conn->exec("
        CREATE TABLE IF NOT EXISTS admin_user_2fa (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL UNIQUE,
            two_factor_secret VARCHAR(32),
            two_factor_method VARCHAR(20) DEFAULT 'whatsapp',
            two_factor_enabled TINYINT(1) DEFAULT 1,
            phone VARCHAR(20) DEFAULT NULL,
            phone_verified TINYINT(1) DEFAULT 0,
            phone_verified_at DATETIME NULL,
            phone_verification_error VARCHAR(255) NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_user_id (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    $stmt = $conn->prepare("
        SELECT two_factor_enabled 
        FROM admin_user_2fa 
        WHERE user_id = ? AND two_factor_enabled = 1
    ");
    $stmt->execute([$userId]);
    return $stmt->fetch() !== false;
}

/**
 * Holt 2FA-Daten eines Benutzers
 */
function get2FAData($userId) {
    require_once __DIR__ . '/../../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Erstelle Tabelle falls nicht vorhanden
    $conn->exec("
        CREATE TABLE IF NOT EXISTS admin_user_2fa (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL UNIQUE,
            two_factor_secret VARCHAR(32),
            two_factor_method VARCHAR(20) DEFAULT 'whatsapp',
            two_factor_enabled TINYINT(1) DEFAULT 1,
            phone VARCHAR(20) DEFAULT NULL,
            phone_verified TINYINT(1) DEFAULT 0,
            phone_verified_at DATETIME NULL,
            phone_verification_error VARCHAR(255) NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_user_id (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    $stmt = $conn->prepare("
        SELECT two_factor_secret, two_factor_method, two_factor_enabled, phone, phone_verified, phone_verified_at, phone_verification_error
        FROM admin_user_2fa 
        WHERE user_id = ?
    ");
    $stmt->execute([$userId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
