<?php
/**
 * Helper-Funktionen für Template-Konfigurationen und Custom Fields
 */

/**
 * Extrahiert Kundendaten aus Appointment für Custom Fields
 */
function extractCustomerFieldsForTemplate($appointment, $customFieldsMapping = []) {
    $customerName = $appointment['customer_name'] ?? '';
    
    // Intelligente Namenserkennung: Versuche Vor- und Nachname zu trennen
    $nameParts = explode(' ', trim($customerName));
    $firstName = !empty($nameParts) ? $nameParts[0] : $customerName;
    $lastName = !empty($nameParts) && count($nameParts) > 1 ? end($nameParts) : ($customerName !== $firstName ? $customerName : '');
    
    // Falls kein Nachname gefunden, versuche aus user_id zu holen
    if (empty($lastName) && !empty($appointment['user_id'])) {
        try {
            require_once __DIR__ . '/../../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            $stmt = $conn->prepare("SELECT first_name, last_name FROM users WHERE id = ?");
            $stmt->execute([$appointment['user_id']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($user) {
                if (!empty($user['first_name'])) $firstName = $user['first_name'];
                if (!empty($user['last_name'])) $lastName = $user['last_name'];
            }
        } catch (Exception $e) {
            // Fehler ignorieren - verwende Fallback
        }
    }
    
    // Fallback: Wenn last_name leer ist, verwende den vollständigen Namen
    // HelloMateo erwartet immer einen Wert für custom_fields.last_name
    if (empty($lastName)) {
        $lastName = $customerName ?: 'Kunde';
    }
    
    // Fallback: Wenn first_name leer ist, verwende den vollständigen Namen
    if (empty($firstName)) {
        $firstName = $customerName ?: 'Kunde';
    }
    
    // Formatiere Datum und Zeit
    $appointmentDate = '';
    if (!empty($appointment['slot_date'])) {
        $dateObj = DateTime::createFromFormat('Y-m-d', $appointment['slot_date']);
        if ($dateObj) {
            $appointmentDate = $dateObj->format('d.m.Y');
        } else {
            $appointmentDate = $appointment['slot_date'];
        }
    } elseif (!empty($appointment['event_date'])) {
        $dateObj = DateTime::createFromFormat('Y-m-d', $appointment['event_date']);
        if ($dateObj) {
            $appointmentDate = $dateObj->format('d.m.Y');
        } else {
            $appointmentDate = $appointment['event_date'];
        }
    }
    
    $appointmentTime = '';
    if (!empty($appointment['slot_time'])) {
        $timeObj = DateTime::createFromFormat('H:i:s', $appointment['slot_time']);
        if ($timeObj) {
            $appointmentTime = $timeObj->format('H:i');
        } else {
            $appointmentTime = $appointment['slot_time'];
        }
    } elseif (!empty($appointment['start_time'])) {
        $timeObj = DateTime::createFromFormat('H:i:s', $appointment['start_time']);
        if ($timeObj) {
            $appointmentTime = $timeObj->format('H:i');
        } else {
            $appointmentTime = $appointment['start_time'];
        }
    }
    
    // Basis-Felder mit automatischer Formatierung
    $fields = [
        'first_name' => $firstName,
        'last_name' => $lastName,
        'full_name' => $customerName,
        'email' => $appointment['email'] ?? $appointment['user_email'] ?? '',
        'phone' => $appointment['phone'] ?? $appointment['user_phone_number'] ?? '',
        'license_plate' => $appointment['license_plate'] ?? '',
        'storage_number' => $appointment['storage_number'] ?? '',
        'appointment_date' => $appointmentDate,
        'appointment_time' => $appointmentTime,
        'autohaus_name' => $appointment['autohaus_name'] ?? $appointment['autohaus_short_name'] ?? '',
    ];
    
    // Erstelle Custom Fields Mapping für HelloMateo
    $customFields = [];
    
    // Standard-Mapping falls keins angegeben
    if (empty($customFieldsMapping)) {
        $customFieldsMapping = [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ];
    }
    
    // Mappe Template-Platzhalter zu Datenfeldern
    foreach ($customFieldsMapping as $templatePlaceholder => $dataField) {
        if (isset($fields[$dataField]) && $fields[$dataField] !== '') {
            // Nur setzen wenn Wert vorhanden und nicht leer
            $customFields[$templatePlaceholder] = $fields[$dataField];
        } elseif (isset($fields[$dataField]) && $dataField === 'last_name' && empty($fields[$dataField])) {
            // Spezialfall: last_name ist leer, aber Template erwartet es - verwende Fallback
            $customFields[$templatePlaceholder] = $customerName ?: 'Kunde';
        } elseif (isset($fields[$dataField]) && $dataField === 'first_name' && empty($fields[$dataField])) {
            // Spezialfall: first_name ist leer, aber Template erwartet es - verwende Fallback
            $customFields[$templatePlaceholder] = $customerName ?: 'Kunde';
        }
    }
    
    return $customFields;
}

/**
 * Verfügbare Aktionstypen für Templates
 */
function getAvailableActionTypes() {
    return [
        'reminder_24h' => [
            'label' => 'Erinnerung 24h vorher',
            'send_timing_hours' => -24,
            'default_timing' => '24h_before',
            'description' => 'Automatische Erinnerung 24 Stunden vor dem Termin'
        ],
        'reminder_48h' => [
            'label' => 'Erinnerung 48h vorher',
            'send_timing_hours' => -48,
            'default_timing' => '48h_before',
            'description' => 'Automatische Erinnerung 48 Stunden vor dem Termin'
        ],
        'confirmation' => [
            'label' => 'Termin-Bestätigung (sofort)',
            'send_timing_hours' => 0,
            'default_timing' => 'immediate',
            'description' => 'Wird sofort nach erfolgreicher Buchung gesendet'
        ],
        'approved' => [
            'label' => 'Termin genehmigt',
            'send_timing_hours' => 0,
            'default_timing' => 'immediate',
            'description' => 'Wird gesendet, wenn ein Termin manuell genehmigt wird'
        ],
        'rejected' => [
            'label' => 'Termin abgelehnt',
            'send_timing_hours' => 0,
            'default_timing' => 'immediate',
            'description' => 'Wird gesendet, wenn ein Termin abgelehnt wird'
        ],
        'cancellation' => [
            'label' => 'Termin storniert',
            'send_timing_hours' => 0,
            'default_timing' => 'immediate',
            'description' => 'Wird gesendet, wenn ein Termin storniert wird'
        ],
        'rescheduled' => [
            'label' => 'Termin verschoben',
            'send_timing_hours' => 0,
            'default_timing' => 'immediate',
            'description' => 'Wird gesendet, wenn ein Termin verschoben wird'
        ],
        'reminder_1h' => [
            'label' => 'Erinnerung 1h vorher',
            'send_timing_hours' => -1,
            'default_timing' => '1h_before',
            'description' => 'Automatische Erinnerung 1 Stunde vor dem Termin'
        ],
        'reminder_2h' => [
            'label' => 'Erinnerung 2h vorher',
            'send_timing_hours' => -2,
            'default_timing' => '2h_before',
            'description' => 'Automatische Erinnerung 2 Stunden vor dem Termin'
        ],
        'reminder_12h' => [
            'label' => 'Erinnerung 12h vorher',
            'send_timing_hours' => -12,
            'default_timing' => '12h_before',
            'description' => 'Automatische Erinnerung 12 Stunden vor dem Termin'
        ],
    ];
}

/**
 * Verfügbare Custom Fields für Mapping
 */
function getAvailableCustomFields() {
    return [
        'first_name' => 'Vorname',
        'last_name' => 'Nachname',
        'full_name' => 'Vollständiger Name',
        'email' => 'E-Mail',
        'phone' => 'Telefonnummer',
        'license_plate' => 'Kennzeichen',
        'storage_number' => 'Einlagerungsnummer',
        'appointment_date' => 'Termindatum',
        'appointment_time' => 'Terminzeit',
        'autohaus_name' => 'Autohaus-Name',
    ];
}

/**
 * Standard Custom Fields Mapping für verschiedene Aktionen
 */
function getDefaultCustomFieldsMapping($actionType) {
    $defaults = [
        'reminder_24h' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
        'reminder_48h' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
        'reminder_12h' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
        'reminder_2h' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
        'reminder_1h' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
        'confirmation' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
        'approved' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
        'rejected' => [
            'custom_fields.last_name' => 'last_name',
        ],
        'cancellation' => [
            'custom_fields.last_name' => 'last_name',
        ],
        'rescheduled' => [
            'custom_fields.last_name' => 'last_name',
            'custom_fields.first_name' => 'first_name',
        ],
    ];
    
    return $defaults[$actionType] ?? ['custom_fields.last_name' => 'last_name'];
}

