<?php
/**
 * Intelligente Drucker-Erkennung
 * Erkennt Drucker automatisch mit mehreren Methoden und Fallbacks
 */

/**
 * Erkennt einen Drucker automatisch mit mehreren Methoden
 * 
 * @param string $printer_name_local Erwarteter lokaler Druckername (z.B. PRT17_BUEK-Server)
 * @param string $printer_name_server Druckername auf Server (z.B. BUEK-PRT17)
 * @param array $alternative_names Alternative Druckernamen zum Suchen
 * @return array ['found' => bool, 'name' => string, 'method' => string, 'details' => array]
 */
function smartDetectPrinter($printer_name_local, $printer_name_server, $alternative_names = []) {
    $result = [
        'found' => false,
        'name' => null,
        'method' => null,
        'details' => []
    ];
    
    // Methode 1: PowerShell Get-Printer (schnellste und zuverlässigste Methode)
    $ps_cmd = 'powershell -NoProfile -ExecutionPolicy Bypass -Command "Get-Printer | Select-Object Name, PrinterStatus, PortName, DriverName | ConvertTo-Json" 2>&1';
    exec($ps_cmd, $ps_output, $ps_code);
    
    if ($ps_code === 0 && !empty($ps_output)) {
        $json_output = implode("\n", $ps_output);
        $printers = json_decode($json_output, true);
        
        if (json_last_error() === JSON_ERROR_NONE && is_array($printers)) {
            // Normalisiere Array (kann einzelnes Objekt oder Array sein)
            if (isset($printers['Name'])) {
                $printers = [$printers];
            }
            
            foreach ($printers as $printer) {
                $name = trim($printer['Name'] ?? '');
                $port = trim($printer['PortName'] ?? '');
                $status = trim($printer['PrinterStatus'] ?? '');
                
                if (empty($name)) continue;
                
                // Priorität 1: Exakte Übereinstimmung mit erwartetem Namen
                if (strcasecmp($name, $printer_name_local) === 0) {
                    $result['found'] = true;
                    $result['name'] = $name;
                    $result['method'] = 'PowerShell Get-Printer (exakt)';
                    $result['details'] = [
                        'port' => $port,
                        'status' => $status,
                        'driver' => $printer['DriverName'] ?? ''
                    ];
                    return $result;
                }
                
                // Priorität 2: Teilübereinstimmung mit erwartetem Namen
                if (stripos($name, $printer_name_local) !== false) {
                    // Prüfe Port - muss Netzwerkport sein
                    if (stripos($port, 'LPT') === false && stripos($port, 'COM') === false) {
                        $result['found'] = true;
                        $result['name'] = $name;
                        $result['method'] = 'PowerShell Get-Printer (Teilübereinstimmung)';
                        $result['details'] = [
                            'port' => $port,
                            'status' => $status,
                            'driver' => $printer['DriverName'] ?? ''
                        ];
                        return $result;
                    }
                }
                
                // Priorität 3: Alternative Namen
                foreach ($alternative_names as $alt_name) {
                    if (stripos($name, $alt_name) !== false) {
                        // Prüfe Port
                        if (stripos($port, 'LPT') === false && stripos($port, 'COM') === false) {
                            $result['found'] = true;
                            $result['name'] = $name;
                            $result['method'] = 'PowerShell Get-Printer (Alternative)';
                            $result['details'] = [
                                'port' => $port,
                                'status' => $status,
                                'driver' => $printer['DriverName'] ?? ''
                            ];
                            return $result;
                        }
                    }
                }
            }
        }
    }
    
    // Methode 2: PowerShell Get-Printer (einfache Liste)
    $ps_cmd2 = 'powershell -NoProfile -ExecutionPolicy Bypass -Command "Get-Printer | Select-Object -ExpandProperty Name" 2>&1';
    exec($ps_cmd2, $ps_output2, $ps_code2);
    
    if ($ps_code2 === 0 && !empty($ps_output2)) {
        foreach ($ps_output2 as $line) {
            $line_trimmed = trim($line);
            if (empty($line_trimmed)) continue;
            
            // Exakte Übereinstimmung
            if (strcasecmp($line_trimmed, $printer_name_local) === 0) {
                $result['found'] = true;
                $result['name'] = $line_trimmed;
                $result['method'] = 'PowerShell Get-Printer Liste (exakt)';
                return $result;
            }
            
            // Teilübereinstimmung
            if (stripos($line_trimmed, $printer_name_local) !== false) {
                $result['found'] = true;
                $result['name'] = $line_trimmed;
                $result['method'] = 'PowerShell Get-Printer Liste (Teilübereinstimmung)';
                return $result;
            }
        }
    }
    
    // Methode 3: WMIC (Windows Management Instrumentation) - Fallback
    $wmic_cmd = 'wmic printer get name /format:list 2>&1';
    exec($wmic_cmd, $wmic_output, $wmic_code);
    
    if ($wmic_code === 0 && !empty($wmic_output)) {
        foreach ($wmic_output as $line) {
            if (stripos($line, 'Name=') === 0) {
                $name = trim(substr($line, 5));
                if (empty($name)) continue;
                
                // Exakte Übereinstimmung
                if (strcasecmp($name, $printer_name_local) === 0) {
                    $result['found'] = true;
                    $result['name'] = $name;
                    $result['method'] = 'WMIC (exakt)';
                    return $result;
                }
                
                // Teilübereinstimmung
                if (stripos($name, $printer_name_local) !== false) {
                    $result['found'] = true;
                    $result['name'] = $name;
                    $result['method'] = 'WMIC (Teilübereinstimmung)';
                    return $result;
                }
            }
        }
    }
    
    // Methode 4: Windows print-Befehl (nur zum Testen, nicht zum Erkennen)
    // Diese Methode wird nicht verwendet, da sie keine Liste zurückgibt
    
    return $result;
}

/**
 * Prüft, ob ein Drucker-Port korrekt konfiguriert ist
 * 
 * @param string $printer_name Druckername
 * @return array ['valid' => bool, 'port' => string, 'error' => string]
 */
function checkPrinterPort($printer_name) {
    $result = [
        'valid' => false,
        'port' => null,
        'error' => null
    ];
    
    $ps_cmd = sprintf(
        'powershell -NoProfile -ExecutionPolicy Bypass -Command "try { $p = Get-Printer -Name \'%s\' -ErrorAction Stop; Write-Host $p.PortName } catch { Write-Host \"ERROR: $_\" }" 2>&1',
        str_replace("'", "''", $printer_name)
    );
    
    exec($ps_cmd, $output, $code);
    $port = trim(implode("", $output));
    
    if ($code !== 0 || stripos($port, 'ERROR') !== false) {
        $result['error'] = $port;
        return $result;
    }
    
    $result['port'] = $port;
    
    // Prüfe, ob Port gültig ist (nicht LPT, COM, FILE, etc.)
    $invalid_ports = ['LPT', 'COM', 'FILE:', 'PORTPROMPT'];
    foreach ($invalid_ports as $invalid) {
        if (stripos($port, $invalid) !== false) {
            $result['error'] = "Ungültiger Port: $port (sollte Netzwerkport sein)";
            return $result;
        }
    }
    
    $result['valid'] = true;
    return $result;
}

/**
 * Prüft den Drucker-Status
 * 
 * @param string $printer_name Druckername
 * @return array ['online' => bool, 'status' => string, 'details' => array]
 */
function checkPrinterStatus($printer_name) {
    $result = [
        'online' => false,
        'status' => 'Unknown',
        'details' => []
    ];
    
    $ps_cmd = sprintf(
        'powershell -NoProfile -ExecutionPolicy Bypass -Command "try { $p = Get-Printer -Name \'%s\' -ErrorAction Stop; Write-Host \"Status=$($p.PrinterStatus)\"; Write-Host \"WorkOffline=$($p.WorkOffline)\"; Write-Host \"Port=$($p.PortName)\" } catch { Write-Host \"ERROR: $_\" }" 2>&1',
        str_replace("'", "''", $printer_name)
    );
    
    exec($ps_cmd, $output, $code);
    
    if ($code !== 0) {
        $result['status'] = 'Error';
        $result['details']['error'] = implode("\n", $output);
        return $result;
    }
    
    $status = 'Unknown';
    $work_offline = false;
    $port = '';
    
    foreach ($output as $line) {
        if (stripos($line, 'Status=') === 0) {
            $status = trim(substr($line, 7));
        } elseif (stripos($line, 'WorkOffline=') === 0) {
            $work_offline = strcasecmp(trim(substr($line, 12)), 'True') === 0;
        } elseif (stripos($line, 'Port=') === 0) {
            $port = trim(substr($line, 5));
        }
    }
    
    $result['status'] = $status;
    $result['details'] = [
        'work_offline' => $work_offline,
        'port' => $port
    ];
    
    // Drucker ist online, wenn Status "Normal" ist und nicht offline
    $result['online'] = (stripos($status, 'Normal') !== false || stripos($status, 'Ready') !== false) && !$work_offline;
    
    return $result;
}







