<?php
/**
 * P2 API-Helper für Kunden, Aufträge und Reifenangebote
 * Kommunikation mit der P2 API (app.py_backendp2)
 * Unterstützt Multi-Autohaus-System mit individueller API-Konfiguration pro Autohaus
 */

class P2API {
    private $baseUrl;
    private $apiKey;
    private $apiKeyRequired;
    private $autohausId;
    private $conn;
    
    public function __construct($autohausId = null) {
        $this->autohausId = $autohausId;
        
        // Lade Datenbankverbindung
        require_once dirname(__DIR__) . '/../config/database.php';
        $db = new Database();
        $this->conn = $db->getConnection();
        
        // Lade API-Konfiguration aus Datenbank
        $this->loadApiConfig();
    }
    
    /**
     * Lädt API-Konfiguration aus der Datenbank für das Autohaus
     */
    private function loadApiConfig() {
        // Fallback-Werte (Standard P2 API)
        $this->baseUrl = 'http://100.82.207.72:5001/api';
        $this->apiKey = 'bungert-p2-api-key-2024-default';
        $this->apiKeyRequired = false;
        
        // Wenn keine Autohaus-ID angegeben, versuche aus Session zu holen
        if (!$this->autohausId && isset($_SESSION['current_autohaus_id'])) {
            $this->autohausId = $_SESSION['current_autohaus_id'];
        }
        
        // Lade Konfiguration aus Datenbank
        if ($this->autohausId && $this->conn) {
            try {
                $stmt = $this->conn->prepare("
                    SELECT tire_offers_api_url, tire_offers_api_key, tire_offers_api_key_required 
                    FROM autohaus 
                    WHERE id = ? AND is_active = 1
                ");
                $stmt->execute([$this->autohausId]);
                $config = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($config) {
                    // Verwende konfigurierte Werte wenn vorhanden
                    if (!empty($config['tire_offers_api_url'])) {
                        $this->baseUrl = $config['tire_offers_api_url'];
                    }
                    if (!empty($config['tire_offers_api_key'])) {
                        $this->apiKey = $config['tire_offers_api_key'];
                    }
                    if (isset($config['tire_offers_api_key_required'])) {
                        $this->apiKeyRequired = (bool)$config['tire_offers_api_key_required'];
                    }
                }
            } catch (Exception $e) {
                error_log("P2API::loadApiConfig Error: " . $e->getMessage());
                // Verwende Fallback-Werte bei Fehler
            }
        }
    }
    
    /**
     * Führt einen API-Call durch mit Retry-Mechanismus
     */
    public function apiCall($endpoint, $method = 'GET', $params = [], $timeout = 5, $maxRetries = 2) {
        $baseUrl = $this->baseUrl . $endpoint;
        $lastError = null;
        $lastHttpCode = 0;
        
        for ($attempt = 0; $attempt <= $maxRetries; $attempt++) {
            try {
                $url = $baseUrl;
                if ($method === 'GET' && !empty($params)) {
                    $url .= '?' . http_build_query($params);
                }
                
                $headers = [
                    'Content-Type: application/json',
                ];
                
                // API-Key nur mitschicken wenn aktiviert
                if ($this->apiKeyRequired) {
                    $headers[] = 'X-API-Key: ' . $this->apiKey;
                }
                
                $ch = curl_init();
                
                curl_setopt_array($ch, [
                    CURLOPT_URL => $url,
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_HTTPHEADER => $headers,
                    CURLOPT_TIMEOUT => $timeout,
                    CURLOPT_CONNECTTIMEOUT => min($timeout, 5),
                    CURLOPT_FRESH_CONNECT => ($attempt > 0),
                    CURLOPT_FORBID_REUSE => false,
                    CURLOPT_SSL_VERIFYPEER => false,
                    CURLOPT_SSL_VERIFYHOST => false,
                ]);
                
                if ($method === 'POST' || $method === 'PUT') {
                    if ($method === 'PUT') {
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                    } else {
                        curl_setopt($ch, CURLOPT_POST, true);
                    }
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
                }
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $curlError = curl_error($ch);
                $curlErrno = curl_errno($ch);
                curl_close($ch);
                
                // Erfolgreiche Antwort
                if ($response !== false && $curlError === '' && $httpCode >= 200 && $httpCode < 400) {
                    // Prüfe ob Response HTML ist statt JSON
                    if (strpos(trim($response), '<') === 0 || strpos(strtolower($response), '<html') !== false) {
                        error_log("P2API: HTML-Response erhalten statt JSON. URL: $url");
                        throw new Exception("API antwortet mit HTML statt JSON");
                    }
                    
                    $data = json_decode($response, true);
                    
                    if (json_last_error() !== JSON_ERROR_NONE) {
                        error_log("P2API: JSON-Parse-Fehler. URL: $url, Response: " . substr($response, 0, 200));
                        throw new Exception("Ungültige JSON-Antwort von API: " . json_last_error_msg());
                    }
                    
                    return $data;
                }
                
                // Fehlerbehandlung
                if ($curlError) {
                    $lastError = "cURL Fehler ($curlErrno): $curlError";
                    if ($curlErrno === CURLE_OPERATION_TIMEDOUT || $curlErrno === CURLE_COULDNT_CONNECT) {
                        if ($attempt < $maxRetries) {
                            usleep(500000 * ($attempt + 1));
                            continue;
                        }
                    }
                    throw new Exception($lastError);
                }
                
                if ($httpCode >= 400) {
                    $lastHttpCode = $httpCode;
                    throw new Exception("API-Fehler (HTTP $httpCode): " . substr($response, 0, 200));
                }
                
            } catch (Exception $e) {
                $lastError = $e->getMessage();
                
                if (strpos($lastError, 'timed out') !== false) {
                    if ($attempt < $maxRetries) {
                        error_log("P2API: Retry $attempt/$maxRetries nach Timeout für $url");
                        usleep(500000 * ($attempt + 1));
                        continue;
                    }
                }
                
                if ($attempt >= $maxRetries) {
                    $errorDetails = "Fehler nach " . ($maxRetries + 1) . " Versuchen";
                    if ($lastHttpCode > 0) {
                        $errorDetails .= " (HTTP $lastHttpCode)";
                    }
                    $errorDetails .= ": $lastError";
                    throw new Exception($errorDetails);
                }
            }
        }
        
        throw new Exception("API-Call fehlgeschlagen: $lastError");
    }
    
    /**
     * Sucht Kunden nach Name
     */
    public function searchCustomers($searchTerm = '', $limit = 1000) {
        try {
            $params = ['limit' => $limit];
            if (!empty($searchTerm)) {
                $params['search'] = trim($searchTerm);
            }
            
            $result = $this->apiCall('/kunden', 'GET', $params);
            
            if (isset($result['success']) && $result['success']) {
                return $result['data'] ?? [];
            }
            
            return [];
        } catch (Exception $e) {
            error_log("P2API::searchCustomers Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt einen Kunden nach ID
     */
    public function getCustomerById($kundenAdrId) {
        try {
            $result = $this->apiCall("/kunden/by-id/$kundenAdrId", 'GET');
            
            if (isset($result['success']) && $result['success']) {
                return $result['data'] ?? null;
            }
            
            return null;
        } catch (Exception $e) {
            error_log("P2API::getCustomerById Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Holt Aufträge für ein Kennzeichen
     */
    public function getOrdersByLicensePlate($licensePlate, $days = 2000, $limit = 500) {
        try {
            if (empty($licensePlate)) {
                error_log("P2API::getOrdersByLicensePlate: Leeres Kennzeichen");
                return [];
            }
            
            $encodedLicense = rawurlencode(trim($licensePlate));
            error_log("P2API::getOrdersByLicensePlate: Suche nach '$licensePlate' (encoded: '$encodedLicense')");
            
            $result = $this->apiCall("/auftraege/by-license/$encodedLicense", 'GET', [
                'days' => $days,
                'limit' => $limit
            ], 10, 1); // Erhöhter Timeout für langsamere Abfragen
            
            error_log("P2API::getOrdersByLicensePlate: API-Antwort - success: " . (isset($result['success']) ? 'true' : 'false') . ", count: " . (isset($result['count']) ? $result['count'] : 'N/A'));
            
            if (isset($result['success']) && $result['success']) {
                $data = $result['data'] ?? [];
                error_log("P2API::getOrdersByLicensePlate: Gefunden " . count($data) . " Aufträge");
                if (isset($result['debug'])) {
                    error_log("P2API::getOrdersByLicensePlate: Debug-Info: " . json_encode($result['debug']));
                }
                return $data;
            }
            
            error_log("P2API::getOrdersByLicensePlate: API-Antwort nicht erfolgreich");
            return [];
        } catch (Exception $e) {
            error_log("P2API::getOrdersByLicensePlate Error: " . $e->getMessage());
            error_log("P2API::getOrdersByLicensePlate Error Stack: " . $e->getTraceAsString());
            return [];
        }
    }
    
    /**
     * Holt Aufträge für eine Fahrgestellnummer (VIN/FIN)
     */
    public function getOrdersByVin($vin, $days = 2000, $limit = 500) {
        try {
            if (empty($vin)) {
                return [];
            }
            
            $encodedVin = rawurlencode(trim($vin));
            $result = $this->apiCall("/auftraege/by-vin/$encodedVin", 'GET', [
                'days' => $days,
                'limit' => $limit
            ]);
            
            if (isset($result['success']) && $result['success']) {
                return $result['data'] ?? [];
            }
            
            return [];
        } catch (Exception $e) {
            error_log("P2API::getOrdersByVin Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt Aufträge für die letzten 6 Ziffern der Fahrgestellnummer
     */
    public function getOrdersByVinLast6($vinLast6, $days = 2000, $limit = 500) {
        try {
            if (empty($vinLast6)) {
                return [];
            }
            
            // Extrahiere nur die letzten 6 Ziffern
            $vinLast6Clean = preg_replace('/[^0-9]/', '', $vinLast6);
            if (strlen($vinLast6Clean) < 6) {
                return [];
            }
            $vinLast6Clean = substr($vinLast6Clean, -6);
            
            $encodedVin = rawurlencode($vinLast6Clean);
            $result = $this->apiCall("/auftraege/by-vin-last6/$encodedVin", 'GET', [
                'days' => $days,
                'limit' => $limit
            ]);
            
            if (isset($result['success']) && $result['success']) {
                return $result['data'] ?? [];
            }
            
            return [];
        } catch (Exception $e) {
            error_log("P2API::getOrdersByVinLast6 Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt Aufträge für einen Kunden (KundenAdrID)
     * OPTIMIERT: Lädt mehr Aufträge und filtert zuverlässig
     */
    public function getOrdersByCustomer($kundenAdrId, $days = 2000, $limit = 2000) {
        try {
            // P2 API hat keinen direkten Endpoint für Kunden-Aufträge
            // Wir müssen alle Aufträge laden und filtern
            // Erhöhe Limit für bessere Abdeckung
            $result = $this->apiCall('/auftraege', 'GET', [
                'days' => $days,
                'limit' => $limit
            ]);
            
            if (isset($result['success']) && $result['success']) {
                $allOrders = $result['data'] ?? [];
                error_log("P2API::getOrdersByCustomer: Geladen " . count($allOrders) . " Aufträge, suche nach KundenAdrID: $kundenAdrId");
                
                // Debug: Zeige erste Aufträge mit KundenAdrID
                if (count($allOrders) > 0) {
                    $firstOrder = $allOrders[0];
                    error_log("P2API::getOrdersByCustomer: Erster Auftrag - KundenAdrID: " . ($firstOrder['KundenAdrID'] ?? 'NICHT GESETZT') . ", Keys: " . implode(', ', array_keys($firstOrder)));
                }
                
                // Filtere nach KundenAdrID - prüfe verschiedene Feldnamen und Typen
                $filtered = array_filter($allOrders, function($order) use ($kundenAdrId) {
                    // Prüfe verschiedene mögliche Feldnamen
                    $orderKundenAdrId = null;
                    if (isset($order['KundenAdrID'])) {
                        $orderKundenAdrId = $order['KundenAdrID'];
                    } elseif (isset($order['kunden_adr_id'])) {
                        $orderKundenAdrId = $order['kunden_adr_id'];
                    } elseif (isset($order['KundenAdrId'])) {
                        $orderKundenAdrId = $order['KundenAdrId'];
                    }
                    
                    if ($orderKundenAdrId === null) {
                        return false;
                    }
                    
                    // Konvertiere beide zu Integer für Vergleich
                    $match = (int)$orderKundenAdrId === (int)$kundenAdrId;
                    
                    if ($match) {
                        error_log("P2API::getOrdersByCustomer: ✅ Match gefunden - AuftragNr: " . ($order['AuftragNr'] ?? 'N/A') . ", KundenAdrID: $orderKundenAdrId");
                    }
                    
                    return $match;
                });
                
                $filteredOrders = array_values($filtered);
                error_log("P2API::getOrdersByCustomer: ✅ Gefiltert " . count($filteredOrders) . " Aufträge für Kunde $kundenAdrId");
                
                // Wenn keine Aufträge gefunden, aber Kunde hat AnzahlAuftraege > 0, versuche mit höherem Limit
                if (count($filteredOrders) === 0 && $limit < 5000) {
                    error_log("P2API::getOrdersByCustomer: ⚠️ Keine Aufträge gefunden, versuche mit höherem Limit (5000)");
                    return $this->getOrdersByCustomer($kundenAdrId, $days, 5000);
                }
                
                return $filteredOrders;
            }
            
            error_log("P2API::getOrdersByCustomer: ❌ API-Antwort nicht erfolgreich");
            return [];
        } catch (Exception $e) {
            error_log("P2API::getOrdersByCustomer Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt einen einzelnen Auftrag nach Auftragsnummer
     */
    public function getOrderById($auftragNr) {
        try {
            // Verwende den by-order-number Endpoint, da es keinen direkten /auftraege/<int> Endpoint gibt
            $result = $this->apiCall("/auftraege/by-order-number/$auftragNr", 'GET', [
                'limit' => 1
            ]);
            
            if (isset($result['success']) && $result['success']) {
                $data = $result['data'] ?? [];
                // Stelle sicher, dass es ein Array ist
                if (!is_array($data)) {
                    // Wenn es ein einzelnes Objekt ist, konvertiere zu Array
                    $data = [$data];
                }
                // Gib das erste Element zurück (oder null wenn leer)
                return !empty($data) ? $data[0] : null;
            }
            
            return null;
        } catch (Exception $e) {
            error_log("P2API::getOrderById Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Holt Aufträge nach Auftragsnummer (unterstützt Format "X - XXXXXX")
     */
    public function getOrdersByOrderNumber($orderNumber, $limit = 10) {
        try {
            $encoded = rawurlencode(trim($orderNumber));
            $result = $this->apiCall("/auftraege/by-order-number/$encoded", 'GET', [
                'limit' => $limit
            ]);
            
            if (isset($result['success']) && $result['success']) {
                $data = $result['data'] ?? [];
                // Stelle sicher, dass es ein Array ist
                if (!is_array($data)) {
                    // Wenn es ein einzelnes Objekt ist, konvertiere zu Array
                    $data = [$data];
                }
                return $data;
            }
            
            return [];
        } catch (Exception $e) {
            error_log("P2API::getOrdersByOrderNumber Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt Auftragspositionen für einen Auftrag
     */
    public function getOrderPositions($auftragNr) {
        try {
            $result = $this->apiCall("/auftraege/positionen/$auftragNr", 'GET');
            
            if (isset($result['success']) && $result['success']) {
                return $result['data'] ?? [];
            }
            
            return [];
        } catch (Exception $e) {
            error_log("P2API::getOrderPositions Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Health-Check für API-Verfügbarkeit
     */
    public function healthCheck() {
        try {
            $result = $this->apiCall('/system/health', 'GET', [], 2, 1);
            return isset($result['status']) && $result['status'] === 'ok';
        } catch (Exception $e) {
            return false;
        }
    }
}

