<?php
/**
 * Authentifizierung mit 2FA-Unterstützung
 * Für Boxenstop Admin-System
 * Mit Unterstützung für Trusted Devices (dauerhafte Anmeldung)
 */

// Session starten falls noch nicht gestartet
if (session_status() === PHP_SESSION_NONE) {
    // Sichere Session-Konfiguration
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 1 : 0);
    ini_set('session.cookie_samesite', 'Strict');
    ini_set('session.use_strict_mode', 1);
    ini_set('session.gc_maxlifetime', 3600); // 1 Stunde Session-Lebensdauer
    
    session_start();
}

/**
 * Prüft ob Benutzer eingeloggt ist
 */
function isAdminLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true
        && isset($_SESSION['admin_user_id']);
}

/**
 * Prüft ob 2FA verifiziert wurde
 */
function is2FAVerified() {
    // Wenn 2FA nicht aktiviert ist, gilt es als verifiziert
    if (!isset($_SESSION['2fa_required']) || $_SESSION['2fa_required'] === false) {
        return true;
    }
    return isset($_SESSION['2fa_verified']) && $_SESSION['2fa_verified'] === true;
}

/**
 * Erfordert Login
 */
function requireAdminLogin() {
    if (!isAdminLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

/**
 * Erfordert vollständige Authentifizierung (Login + 2FA)
 */
function requireFullAuth() {
    requireAdminLogin();
    if (!is2FAVerified()) {
        header('Location: login.php?step=2fa');
        exit;
    }
}

/**
 * Prüft Login-Versuche (Rate Limiting)
 */
function checkLoginAttempts($username) {
    $cacheFile = sys_get_temp_dir() . '/admin_login_attempts_' . md5($username) . '.json';
    
    if (!file_exists($cacheFile)) {
        return true;
    }
    
    $data = json_decode(file_get_contents($cacheFile), true);
    if (!$data) {
        return true;
    }
    
    // Bereinige alte Einträge (älter als 15 Minuten)
    $data['attempts'] = array_filter($data['attempts'] ?? [], function($timestamp) {
        return $timestamp > time() - 900;
    });
    
    // Mehr als 5 Versuche in 15 Minuten = blockiert
    if (count($data['attempts']) >= 5) {
        return false;
    }
    
    return true;
}

/**
 * Erfasst fehlgeschlagenen Login
 */
function recordFailedLogin($username) {
    $cacheFile = sys_get_temp_dir() . '/admin_login_attempts_' . md5($username) . '.json';
    
    $data = ['attempts' => []];
    if (file_exists($cacheFile)) {
        $data = json_decode(file_get_contents($cacheFile), true) ?: ['attempts' => []];
    }
    
    $data['attempts'][] = time();
    file_put_contents($cacheFile, json_encode($data));
}

/**
 * Löscht Login-Versuche nach erfolgreichem Login
 */
function clearLoginAttempts($username) {
    $cacheFile = sys_get_temp_dir() . '/admin_login_attempts_' . md5($username) . '.json';
    if (file_exists($cacheFile)) {
        unlink($cacheFile);
    }
}

/**
 * Generiert ein sicheres Token
 */
function generateSecureToken($length = 64) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Generiert einen Geräte-Identifier
 */
function generateDeviceIdentifier() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    $acceptLanguage = $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '';
    
    $fingerprint = $userAgent . $ip . $acceptLanguage;
    return hash('sha256', $fingerprint);
}

/**
 * Prüft ob ein Gerät als vertrauenswürdig registriert ist
 */
function checkTrustedDevice($deviceIdentifier) {
    if (empty($deviceIdentifier)) {
        return false;
    }
    
    try {
        require_once __DIR__ . '/../../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        
        // Erstelle Tabelle falls nicht vorhanden
        $conn->exec("
            CREATE TABLE IF NOT EXISTS admin_trusted_devices (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                device_identifier VARCHAR(255) NOT NULL,
                device_name VARCHAR(100) DEFAULT 'Unbekanntes Gerät',
                device_type VARCHAR(50) DEFAULT NULL,
                ip_address VARCHAR(45) DEFAULT NULL,
                user_agent TEXT DEFAULT NULL,
                is_active TINYINT(1) DEFAULT 1,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                last_used DATETIME DEFAULT CURRENT_TIMESTAMP,
                expires_at DATETIME DEFAULT NULL,
                UNIQUE KEY uk_device (device_identifier),
                INDEX idx_user_id (user_id),
                INDEX idx_device_identifier (device_identifier)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
        
        // Suche nach dem Gerät
        $stmt = $conn->prepare("
            SELECT td.*, au.id as user_id, au.username
            FROM admin_trusted_devices td
            JOIN admin_users au ON td.user_id = au.id
            WHERE td.device_identifier = ?
            AND td.is_active = 1
            AND (td.expires_at IS NULL OR td.expires_at > NOW())
            AND au.is_active = 1
        ");
        $stmt->execute([$deviceIdentifier]);
        $device = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($device) {
            // Aktualisiere last_used
            $updateStmt = $conn->prepare("
                UPDATE admin_trusted_devices 
                SET last_used = NOW() 
                WHERE id = ?
            ");
            $updateStmt->execute([$device['id']]);
            
            return [
                'user_id' => $device['user_id'],
                'username' => $device['username'],
                'device_name' => $device['device_name']
            ];
        }
        
        return false;
    } catch (Exception $e) {
        error_log("checkTrustedDevice Error: " . $e->getMessage());
        return false;
    }
}

/**
 * Registriert ein Gerät als vertrauenswürdig
 */
function registerTrustedDevice($userId, $deviceIdentifier, $deviceName = 'Unbekanntes Gerät', $expiresInDays = null) {
    if (empty($deviceIdentifier) || empty($userId)) {
        return false;
    }
    
    try {
        require_once __DIR__ . '/../../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        
        // Erstelle Tabelle falls nicht vorhanden
        $conn->exec("
            CREATE TABLE IF NOT EXISTS admin_trusted_devices (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                device_identifier VARCHAR(255) NOT NULL,
                device_name VARCHAR(100) DEFAULT 'Unbekanntes Gerät',
                device_type VARCHAR(50) DEFAULT NULL,
                ip_address VARCHAR(45) DEFAULT NULL,
                user_agent TEXT DEFAULT NULL,
                is_active TINYINT(1) DEFAULT 1,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                last_used DATETIME DEFAULT CURRENT_TIMESTAMP,
                expires_at DATETIME DEFAULT NULL,
                UNIQUE KEY uk_device (device_identifier),
                INDEX idx_user_id (user_id),
                INDEX idx_device_identifier (device_identifier)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
        
        $expiresAt = $expiresInDays ? date('Y-m-d H:i:s', strtotime("+{$expiresInDays} days")) : null;
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? null;
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
        
        // Prüfe ob Gerät bereits registriert ist
        $checkStmt = $conn->prepare("SELECT id FROM admin_trusted_devices WHERE device_identifier = ?");
        $checkStmt->execute([$deviceIdentifier]);
        
        if ($checkStmt->fetch()) {
            // Update existing
            $stmt = $conn->prepare("
                UPDATE admin_trusted_devices 
                SET user_id = ?, device_name = ?, ip_address = ?, user_agent = ?, 
                    is_active = 1, last_used = NOW(), expires_at = ?
                WHERE device_identifier = ?
            ");
            return $stmt->execute([$userId, $deviceName, $ipAddress, $userAgent, $expiresAt, $deviceIdentifier]);
        } else {
            // Insert new
            $stmt = $conn->prepare("
                INSERT INTO admin_trusted_devices 
                (user_id, device_identifier, device_name, ip_address, user_agent, expires_at)
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            return $stmt->execute([$userId, $deviceIdentifier, $deviceName, $ipAddress, $userAgent, $expiresAt]);
        }
    } catch (Exception $e) {
        error_log("registerTrustedDevice Error: " . $e->getMessage());
        return false;
    }
}

/**
 * Setzt das Cookie für vertrauenswürdiges Gerät
 */
function setTrustedDeviceCookie($token, $expiresInDays = 365) {
    $expires = time() + ($expiresInDays * 24 * 60 * 60);
    $secure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
    
    setcookie('admin_trusted_device_token', $token, [
        'expires' => $expires,
        'path' => '/',
        'secure' => $secure,
        'httponly' => true,
        'samesite' => 'Strict'
    ]);
}
