<?php
/**
 * Holt Template-Informationen von HelloMateo API
 */

header('Content-Type: application/json');
require_once '../config/database.php';

session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    echo json_encode(['success' => false, 'error' => 'Nicht angemeldet']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$templateId = trim($input['template_id'] ?? ''); // Template-ID trimmen
$apiKey = trim($input['api_key'] ?? ''); // API-Key trimmen
$autohausId = $input['autohaus_id'] ?? null;

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // WICHTIG: Wenn force_use_provided_key gesetzt ist, verwende IMMER den übergebenen API-Key
    // Das stellt sicher, dass der getestete API-Key verwendet wird (aus localStorage)
    $forceUseProvided = isset($input['force_use_provided_key']) && $input['force_use_provided_key'];
    
    // Wenn force_use_provided_key gesetzt ist, verwende NUR den übergebenen Key
    if ($forceUseProvided) {
        // Verwende den übergebenen API-Key direkt - KEIN Fallback zur DB
        // Das stellt sicher, dass der getestete API-Key verwendet wird
        if (empty($apiKey)) {
            echo json_encode([
                'success' => false,
                'error' => 'Kein API-Key übergeben. Bitte testen Sie zuerst die API-Verbindung mit dem "Test"-Button.',
                'template_id' => $templateId,
                'debug' => 'force_use_provided_key ist gesetzt, aber api_key ist leer'
            ]);
            exit;
        }
    } elseif (empty($apiKey) && !isset($input['api_key'])) {
        // Nur wenn api_key nicht im Request vorhanden ist, hole aus DB
        if ($autohausId) {
            // Autohaus-spezifischer API-Key
            $stmt = $conn->prepare("SELECT whatsapp_api_key FROM autohaus WHERE id = ?");
            $stmt->execute([$autohausId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            $apiKey = $autohaus['whatsapp_api_key'] ?? '';
        }
        
        // Falls immer noch leer, hole globalen API-Key
        if (empty($apiKey)) {
            $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_api_key'");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $apiKey = $result['setting_value'] ?? '';
        }
    }
    
    if (empty($templateId)) {
        echo json_encode([
            'success' => false,
            'error' => 'Keine Template-ID angegeben'
        ]);
        exit;
    }
    
    if (empty($apiKey)) {
        echo json_encode([
            'success' => false,
            'error' => 'Kein API-Key konfiguriert',
            'template_id' => $templateId
        ]);
        exit;
    }
    
    // API-Key bereinigen (Leerzeichen, Zeilenumbrüche entfernen)
    $apiKey = trim($apiKey);
    
    // Debug: Log API-Key Info (nur Länge, nicht der Key selbst)
    error_log("Template-Info Request - Template-ID: $templateId, API-Key Länge: " . strlen($apiKey) . ", Force: " . ($forceUseProvided ? 'yes' : 'no'));
    
    // Hole Template-Informationen von HelloMateo API
    // Laut HelloMateo-Dokumentation: Base URL ist https://integration.getmateo.com/api/v1
    // Versuche verschiedene mögliche Endpunkte für Templates
    $endpoints = [
        'https://integration.getmateo.com/api/v1/templates/' . urlencode($templateId),
        'https://integration.getmateo.com/api/v1/message-templates/' . urlencode($templateId),
        'https://integration.getmateo.com/api/v1/template/' . urlencode($templateId),
        'https://api.hellomateo.com/v1/templates/' . urlencode($templateId),
        'https://api.hellomateo.com/v1/message-templates/' . urlencode($templateId),
    ];
    
    $templateData = null;
    $lastError = null;
    $allErrors = [];
    $responseData = null;
    
    foreach ($endpoints as $index => $url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        $curlInfo = curl_getinfo($ch);
        curl_close($ch);
        
        // Debug: Speichere Response für Analyse
        if ($index === 0) {
            $errorResponse = null;
            if ($response) {
                $errorResponse = json_decode($response, true);
            }
            $responseData = [
                'url' => $url,
                'http_code' => $httpCode,
                'response_preview' => substr($response, 0, 500),
                'response_full_length' => strlen($response),
                'curl_error' => $curlError,
                'error_response' => $errorResponse
            ];
            
            // Log für Debugging
            error_log("Template API - URL: $url, HTTP: $httpCode");
            if ($errorResponse) {
                error_log("Template API Error: " . json_encode($errorResponse));
            }
        }
        
        if ($httpCode === 200 && !empty($response)) {
            $decoded = json_decode($response, true);
            if ($decoded && (isset($decoded['id']) || isset($decoded['name']) || isset($decoded['template_id']) || isset($decoded['data']))) {
                $templateData = $decoded;
                break;
            }
        } else {
            $errorMsg = "HTTP $httpCode";
            if ($curlError) {
                $errorMsg .= ": $curlError";
            }
            if ($response) {
                $errorResponse = json_decode($response, true);
                // Laut HelloMateo API-Dokumentation: {code, message, hint, status}
                if ($errorResponse) {
                    if (isset($errorResponse['message'])) {
                        $errorMsg .= " - " . $errorResponse['message'];
                    }
                    if (isset($errorResponse['hint'])) {
                        $errorMsg .= " (" . $errorResponse['hint'] . ")";
                    }
                    if (isset($errorResponse['code'])) {
                        $errorMsg .= " [Code: " . $errorResponse['code'] . "]";
                    }
                    // Fallback für andere Formate
                    if (isset($errorResponse['error'])) {
                        $errorMsg .= " - " . $errorResponse['error'];
                    }
                }
            }
            $allErrors[] = $errorMsg;
            $lastError = $errorMsg;
            
            // Bei 401/403: Stoppe weitere Versuche, da der API-Key ungültig ist
            if ($httpCode === 401 || $httpCode === 403) {
                break;
            }
        }
    }
    
    // Falls kein Template gefunden wurde, versuche alle Templates aufzulisten
    // WICHTIG: Prüfe zuerst, ob der API-Key für Templates berechtigt ist
    if (!$templateData) {
        $listEndpoints = [
            'https://integration.getmateo.com/api/v1/templates',
            'https://integration.getmateo.com/api/v1/message-templates',
            'https://api.hellomateo.com/v1/templates',
        ];
        
        foreach ($listEndpoints as $listUrl) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $listUrl);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $apiKey,
                'Content-Type: application/json',
                'Accept: application/json'
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            
        $listResponse = curl_exec($ch);
        $listHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $listCurlError = curl_error($ch);
        curl_close($ch);
        
        // Bei 401/403: Stoppe weitere Versuche
        if ($listHttpCode === 401 || $listHttpCode === 403) {
            $errorResponse = json_decode($listResponse, true);
            $lastError = "HTTP $listHttpCode";
            if ($errorResponse && isset($errorResponse['message'])) {
                $lastError .= " - " . $errorResponse['message'];
                if (isset($errorResponse['hint'])) {
                    $lastError .= " (" . $errorResponse['hint'] . ")";
                }
            }
            break;
        }
        
        if ($listHttpCode === 200 && !empty($listResponse)) {
                $templatesList = json_decode($listResponse, true);
                if ($templatesList) {
                    // Suche nach dem Template in der Liste
                    if (isset($templatesList['data']) && is_array($templatesList['data'])) {
                        foreach ($templatesList['data'] as $template) {
                            if (isset($template['id']) && $template['id'] === $templateId) {
                                $templateData = $template;
                                break 2;
                            }
                            if (isset($template['name']) && $template['name'] === $templateId) {
                                $templateData = $template;
                                break 2;
                            }
                        }
                    } elseif (is_array($templatesList)) {
                        foreach ($templatesList as $template) {
                            if (isset($template['id']) && $template['id'] === $templateId) {
                                $templateData = $template;
                                break 2;
                            }
                            if (isset($template['name']) && $template['name'] === $templateId) {
                                $templateData = $template;
                                break 2;
                            }
                        }
                    }
                }
            }
        }
    }
    
    if ($templateData) {
        // Formatiere Template-Daten für die Anzeige
        $formattedData = [
            'success' => true,
            'template_id' => $templateId,
            'name' => $templateData['name'] ?? $templateData['template_name'] ?? 'Unbekannt',
            'status' => $templateData['status'] ?? $templateData['template_status'] ?? 'Unbekannt',
            'language' => $templateData['language'] ?? $templateData['template_language'] ?? 'de',
            'category' => $templateData['category'] ?? $templateData['template_category'] ?? 'UTILITY',
            'components' => $templateData['components'] ?? $templateData['body'] ?? [],
            'body' => $templateData['body'] ?? $templateData['message'] ?? '',
            'header' => $templateData['header'] ?? null,
            'footer' => $templateData['footer'] ?? null,
            'buttons' => $templateData['buttons'] ?? [],
            'raw_data' => $templateData
        ];
        
        // Extrahiere Text aus Components falls vorhanden
        if (isset($templateData['components']) && is_array($templateData['components'])) {
            foreach ($templateData['components'] as $component) {
                if (isset($component['type']) && $component['type'] === 'BODY') {
                    $formattedData['body'] = $component['text'] ?? $component['format'] ?? '';
                }
                if (isset($component['type']) && $component['type'] === 'HEADER') {
                    $formattedData['header'] = $component['text'] ?? $component['format'] ?? '';
                }
                if (isset($component['type']) && $component['type'] === 'FOOTER') {
                    $formattedData['footer'] = $component['text'] ?? '';
                }
            }
        }
        
        echo json_encode($formattedData);
    } else {
        // Detaillierte Fehlerinformationen
        $errorMessage = 'Template nicht gefunden oder API-Fehler';
        if ($lastError) {
            $errorMessage = $lastError; // Verwende die genaue Fehlermeldung von der API
        }
        
        // Spezielle Behandlung für Authentifizierungsfehler
        $isAuthError = false;
        foreach ($allErrors as $err) {
            if (strpos($err, '401') !== false || strpos($err, '403') !== false || 
                strpos($err, 'Invalid token') !== false || strpos($err, 'invalid') !== false) {
                $isAuthError = true;
                break;
            }
        }
        
        // WICHTIG: Laut HelloMateo-Dokumentation gibt es keinen Template-Endpunkt
        // Die API unterstützt nur das Senden von Nachrichten, nicht das Abrufen von Template-Details
        if ($isAuthError) {
            // Prüfe ob der API-Key für Messages funktioniert (das ist der Haupt-Endpunkt)
            $canSendMessages = false;
            $testMessageUrl = 'https://integration.getmateo.com/api/v1/messages';
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $testMessageUrl);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'OPTIONS');
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $apiKey,
                'Content-Type: application/json',
                'Accept: application/json'
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_NOBODY, true);
            $testResponse = curl_exec($ch);
            $testHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if (in_array($testHttpCode, [200, 204, 405])) {
                $canSendMessages = true;
            }
            
            if ($canSendMessages) {
                // API-Key funktioniert für Messages, aber Template-Endpunkt existiert nicht
                $errorMessage = 'Template-Informationen können nicht abgerufen werden.';
            } else {
                $errorMessage = 'Authentifizierungsfehler: Der API-Key ist ungültig oder fehlt.';
                if ($lastError) {
                    $errorMessage .= ' (' . $lastError . ')';
                }
            }
        }
        
        // Info-Meldung: Template-ID kann trotzdem verwendet werden
        $infoMessage = 'Die HelloMateo API unterstützt keinen Endpunkt zum Abrufen von Template-Informationen (laut Dokumentation). ';
        $infoMessage .= 'Templates werden nur beim Senden von Nachrichten über /api/v1/messages verwendet. ';
        $infoMessage .= 'Die Template-ID kann trotzdem gespeichert werden und wird beim Senden von WhatsApp-Nachrichten automatisch verwendet.';
        
        $debugInfo = [
            'success' => false,
            'error' => $errorMessage,
            'template_id' => $templateId,
            'all_errors' => $allErrors,
            'info' => $isAuthError && isset($canSendMessages) && $canSendMessages
                ? $infoMessage
                : ($isAuthError 
                    ? 'Bitte überprüfen Sie den API-Key. Der API-Key muss korrekt sein und darf keine Leerzeichen enthalten.'
                    : 'Bitte überprüfen Sie die Template-ID. Die Template-ID wird beim Senden von Nachrichten verwendet.'),
            'template_id_saved' => true, // Template-ID kann trotzdem gespeichert werden
            'note' => $isAuthError && isset($canSendMessages) && $canSendMessages
                ? 'Die Template-ID wird beim Senden von WhatsApp-Nachrichten automatisch verwendet. Template-Informationen können nicht abgerufen werden, da die API keinen entsprechenden Endpunkt bietet.'
                : 'Die Template-ID kann gespeichert werden.',
            'debug' => [
                'api_key_length' => strlen($apiKey),
                'api_key_prefix' => substr($apiKey, 0, 30) . '...',
                'api_key_suffix' => '...' . substr($apiKey, -10),
                'tested_endpoints' => count($endpoints),
                'first_response' => $responseData,
                'is_auth_error' => $isAuthError,
                'can_send_messages' => isset($canSendMessages) ? $canSendMessages : null
            ]
        ];
        
        // Nur in Development-Modus: Zeige detaillierte Debug-Infos
        if (isset($_GET['debug']) || (isset($input['debug']) && $input['debug'])) {
            echo json_encode($debugInfo, JSON_PRETTY_PRINT);
        } else {
            // Produktions-Modus: Zeige nur wichtige Infos
            unset($debugInfo['debug']);
            echo json_encode($debugInfo);
        }
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => 'Fehler: ' . $e->getMessage()
    ]);
}
?>

