<?php
/**
 * Events Management
 */

session_start();
require_once '../config/database.php';
require_once 'includes/helpers.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data with autohaus information
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$current_user_autohaus_name = null;
$can_switch_autohaus = false;

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.role, u.autohaus_id, u.can_switch_autohaus, a.name as autohaus_name
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_role = $user_data['role'] ?? 'admin';
    $current_user_autohaus_id = $user_data['autohaus_id'];
    $current_user_autohaus_name = $user_data['autohaus_name'];
    $can_switch_autohaus = $user_data['can_switch_autohaus'] == 1;
}

// Get user theme preference
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

$action = $_GET['action'] ?? 'list';
$event_id = $_GET['id'] ?? null;

// Handle form submissions
if ($_POST) {
    if ($action === 'create' || $action === 'edit') {
        handleEventForm($conn, $action, $event_id, $current_user_autohaus_id, $current_user_role, $can_switch_autohaus);
    }
}

// Handle delete action (GET request)
if ($action === 'delete' && $event_id) {
    error_log("Attempting to delete event with ID: $event_id");
    handleEventDelete($conn, $event_id, $current_user_autohaus_id, $current_user_role, $can_switch_autohaus);
}

// Get events list (filtered by autohaus)
$events = getEvents($conn, $current_user_autohaus_id, $current_user_role, $can_switch_autohaus);

// Get autohaus list for forms
$stmt = $conn->prepare("SELECT * FROM autohaus WHERE is_active = 1 ORDER BY name ASC");
$stmt->execute();
$autohauser = $stmt->fetchAll();

// Get event data for editing
$event = null;
if ($action === 'edit' && $event_id) {
    $stmt = $conn->prepare("SELECT * FROM events WHERE id = ?");
    $stmt->execute([$event_id]);
    $event = $stmt->fetch();
}

function getEvents($conn, $autohaus_id = null, $user_role = 'admin', $can_switch_autohaus = false) {
    $where_clause = "WHERE 1=1";
    $params = [];
    
    // If user is not admin or can't switch autohaus, filter by their autohaus
    if ($user_role !== 'admin' || !$can_switch_autohaus) {
        if ($autohaus_id) {
            $where_clause .= " AND e.autohaus_id = ?";
            $params[] = $autohaus_id;
        } else {
            // User has no autohaus assigned, show no events
            $where_clause .= " AND e.autohaus_id IS NULL";
        }
    }
    
    $stmt = $conn->prepare("
        SELECT e.*, 
               a.name as autohaus_name,
               a.short_name as autohaus_short_name,
               COUNT(ap.id) as appointment_count,
               COUNT(CASE WHEN ap.status = 'booked' THEN 1 END) as booked_count
        FROM events e
        LEFT JOIN autohaus a ON e.autohaus_id = a.id
        LEFT JOIN appointments ap ON e.id = ap.event_id
        $where_clause
        GROUP BY e.id
        ORDER BY e.event_date DESC
    ");
    $stmt->execute($params);
    return $stmt->fetchAll();
}

function handleEventForm($conn, $action, $event_id, $user_autohaus_id = null, $user_role = 'admin', $can_switch_autohaus = false) {
    // Check autohaus permissions
    $autohaus_id = $_POST['autohaus_id'] ?: null;
    
    // If user is not admin or can't switch autohaus, force their autohaus
    if ($user_role !== 'admin' || !$can_switch_autohaus) {
        $autohaus_id = $user_autohaus_id;
    }
    
    $data = [
        'name' => $_POST['name'],
        'event_date' => $_POST['event_date'],
        'start_time' => $_POST['start_time'],
        'end_time' => $_POST['end_time'],
        'slot_duration' => $_POST['slot_duration'],
        'max_monteure' => $_POST['max_monteure'],
        'pause_start' => $_POST['pause_start'] ?: null,
        'pause_end' => $_POST['pause_end'] ?: null,
        'blocked_start' => $_POST['blocked_start'] ?: null,
        'blocked_end' => $_POST['blocked_end'] ?: null,
        'is_active' => isset($_POST['is_active']) ? 1 : 0,
        'autohaus_id' => $autohaus_id
    ];
    
    try {
        if ($action === 'create') {
            $stmt = $conn->prepare("
                INSERT INTO events (name, event_date, start_time, end_time, slot_duration, 
                                  max_monteure, pause_start, pause_end, blocked_start, blocked_end, is_active, autohaus_id)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute(array_values($data));
            $message = 'Event erfolgreich erstellt';
        } else {
            $stmt = $conn->prepare("
                UPDATE events 
                SET name = ?, event_date = ?, start_time = ?, end_time = ?, slot_duration = ?,
                    max_monteure = ?, pause_start = ?, pause_end = ?, blocked_start = ?, 
                    blocked_end = ?, is_active = ?, autohaus_id = ?
                WHERE id = ?
            ");
            $values = array_values($data);
            $values[] = $event_id;
            $stmt->execute($values);
            $message = 'Event erfolgreich aktualisiert';
        }
        
        $_SESSION['success_message'] = $message;
        header('Location: events.php');
        exit;
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = 'Fehler: ' . $e->getMessage();
    }
}

function handleEventDelete($conn, $event_id, $user_autohaus_id = null, $user_role = 'admin', $can_switch_autohaus = false) {
    try {
        error_log("handleEventDelete called with event_id: $event_id, user_role: $user_role, can_switch: " . ($can_switch_autohaus ? 'true' : 'false'));
        
        // Check if user can delete this event
        $where_clause = "WHERE id = ?";
        $params = [$event_id];
        
        if ($user_role !== 'admin' || !$can_switch_autohaus) {
            if ($user_autohaus_id) {
                $where_clause .= " AND autohaus_id = ?";
                $params[] = $user_autohaus_id;
            } else {
                $where_clause .= " AND autohaus_id IS NULL";
            }
        }
        
        error_log("SQL: DELETE FROM events $where_clause with params: " . print_r($params, true));
        
        // Check if event has appointments
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM appointments WHERE event_id = ?");
        $stmt->execute([$event_id]);
        $result = $stmt->fetch();
        
        if ($result['count'] > 0) {
            error_log("Cannot delete event: appointments exist");
            throw new Exception('Event kann nicht gelöscht werden, da bereits Termine existieren');
        }
        
        $stmt = $conn->prepare("DELETE FROM events $where_clause");
        $stmt->execute($params);
        
        error_log("Delete executed, rows affected: " . $stmt->rowCount());
        
        if ($stmt->rowCount() === 0) {
            $_SESSION['error_message'] = 'Event nicht gefunden oder keine Berechtigung zum Löschen';
        } else {
            $_SESSION['success_message'] = 'Event erfolgreich gelöscht';
        }
        
        header('Location: events.php');
        exit;
        
    } catch (Exception $e) {
        error_log("Error in handleEventDelete: " . $e->getMessage());
        $_SESSION['error_message'] = 'Fehler: ' . $e->getMessage();
    }
}

// Get event for editing
$event = null;
if ($action === 'edit' && $event_id) {
    $stmt = $conn->prepare("SELECT * FROM events WHERE id = ?");
    $stmt->execute([$event_id]);
    $event = $stmt->fetch();
}
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Events verwalten - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="admin-body">
    <div class="admin-container">
        <!-- Include Unified Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <main class="admin-main">
            <!-- Events Header with Hero Section -->
            <header class="events-header">
                <div class="events-hero">
                    <div class="events-hero-content">
                        <div class="events-title-section">
                            <h1 class="events-title">
                                <i class="fas fa-calendar-star"></i> Events verwalten
                            </h1>
                            <p class="events-subtitle">Verwalten Sie Boxenstop Events und deren Einstellungen</p>
                            <div class="events-meta">
                                <div class="meta-item">
                                    <i class="fas fa-calendar-alt"></i>
                                    <span><?php echo count($events); ?> Events</span>
                                </div>
                                <div class="meta-item">
                                    <i class="fas fa-building"></i>
                                    <span><?php echo $current_user_autohaus_name ?: 'Alle Autohäuser'; ?></span>
                                </div>
                                <div class="meta-item">
                                    <i class="fas fa-users"></i>
                                    <span>Event-Management</span>
                                </div>
                            </div>
                        </div>
                        <div class="events-actions">
                            <a href="events.php?action=create" class="btn-primary">
                                <i class="fas fa-plus"></i>
                                Neues Event
                            </a>
                        </div>
                    </div>
                </div>
            </header>

            <!-- Events Content -->
            <div class="events-content">
                <!-- Messages -->
                <?php if (isset($_SESSION['success_message'])): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
                    </div>
                <?php endif; ?>

                <?php if (isset($_SESSION['error_message'])): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
                    </div>
                <?php endif; ?>

            <?php if ($action === 'create' || $action === 'edit'): ?>
                <!-- Event Form -->
                <div class="content-card">
                    <div class="card-header">
                        <h3><?php echo $action === 'create' ? 'Neues Event erstellen' : 'Event bearbeiten'; ?></h3>
                        <a href="events.php" class="ios-btn ios-btn-secondary">
                            <i class="fas fa-arrow-left"></i>
                            Zurück
                        </a>
                    </div>
                    <div class="card-content">
                        <form method="POST" class="event-form">
                            <div class="form-grid">
                                <div class="form-group">
                                    <label for="name">Event Name *</label>
                                    <input type="text" id="name" name="name" required 
                                           value="<?php echo htmlspecialchars($event['name'] ?? ''); ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="autohaus_id">Autohaus *</label>
                                    <?php if ($current_user_role === 'admin' && $can_switch_autohaus): ?>
                                        <select id="autohaus_id" name="autohaus_id" required>
                                            <option value="">Autohaus auswählen...</option>
                                            <?php foreach ($autohauser as $autohaus): ?>
                                                <option value="<?php echo $autohaus['id']; ?>" 
                                                        <?php echo ($event['autohaus_id'] ?? '') == $autohaus['id'] ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($autohaus['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php else: ?>
                                        <select id="autohaus_id" name="autohaus_id" required disabled>
                                            <option value="<?php echo $current_user_autohaus_id; ?>" selected>
                                                <?php echo htmlspecialchars($current_user_autohaus_name ?: 'Kein Autohaus zugeordnet'); ?>
                                            </option>
                                        </select>
                                        <input type="hidden" name="autohaus_id" value="<?php echo $current_user_autohaus_id; ?>">
                                        <small class="form-text text-muted">Sie können nur Events für Ihr zugeordnetes Autohaus erstellen.</small>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="form-group">
                                    <label for="event_date">Datum *</label>
                                    <input type="date" id="event_date" name="event_date" required 
                                           value="<?php echo $event['event_date'] ?? ''; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="start_time">Startzeit *</label>
                                    <input type="time" id="start_time" name="start_time" required 
                                           value="<?php echo $event['start_time'] ?? ''; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="end_time">Endzeit *</label>
                                    <input type="time" id="end_time" name="end_time" required 
                                           value="<?php echo $event['end_time'] ?? ''; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="slot_duration">Slot-Dauer (Minuten) *</label>
                                    <input type="number" id="slot_duration" name="slot_duration" required 
                                           min="15" max="120" value="<?php echo $event['slot_duration'] ?? '30'; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="max_monteure">Max. Monteure *</label>
                                    <input type="number" id="max_monteure" name="max_monteure" required 
                                           min="1" max="20" value="<?php echo $event['max_monteure'] ?? '5'; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="pause_start">Pausenstart</label>
                                    <input type="time" id="pause_start" name="pause_start" 
                                           value="<?php echo $event['pause_start'] ?? ''; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="pause_end">Pausenende</label>
                                    <input type="time" id="pause_end" name="pause_end" 
                                           value="<?php echo $event['pause_end'] ?? ''; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="blocked_start">Blockierte Zeit Start</label>
                                    <input type="time" id="blocked_start" name="blocked_start" 
                                           value="<?php echo $event['blocked_start'] ?? ''; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="blocked_end">Blockierte Zeit Ende</label>
                                    <input type="time" id="blocked_end" name="blocked_end" 
                                           value="<?php echo $event['blocked_end'] ?? ''; ?>">
                                </div>
                                
                                <div class="form-group full-width">
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="is_active" value="1" 
                                               <?php echo ($event['is_active'] ?? 1) ? 'checked' : ''; ?>>
                                        <span>Event ist aktiv</span>
                                    </label>
                                </div>
                            </div>
                            
                            <div class="form-actions">
                                <a href="events.php" class="ios-btn ios-btn-secondary">Abbrechen</a>
                                <button type="submit" class="ios-btn ios-btn-primary">
                                    <i class="fas fa-floppy-disk"></i>
                                    <?php echo $action === 'create' ? 'Event erstellen' : 'Änderungen speichern'; ?>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <!-- Events List -->
                <div class="content-card">
                    <div class="card-header">
                        <h3>Alle Events</h3>
                        <div class="header-actions">
                            <a href="events.php?action=create" class="ios-btn ios-btn-primary">
                                <i class="fas fa-plus"></i>
                                Neues Event
                            </a>
                        </div>
                    </div>
                    <div class="card-content">
                        <?php if (empty($events)): ?>
                            <div class="empty-state">
                                <i class="fas fa-calendar-plus"></i>
                                <p>Keine Events vorhanden</p>
                                <a href="events.php?action=create" class="ios-btn ios-btn-primary">Erstes Event erstellen</a>
                            </div>
                        <?php else: ?>
                            <div class="events-table">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>Name</th>
                                            <th>Autohaus</th>
                                            <th>Datum</th>
                                            <th>Zeit</th>
                                            <th>Monteure</th>
                                            <th>Termine</th>
                                            <th>Status</th>
                                            <th>Aktionen</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($events as $event): ?>
                                            <tr>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($event['name']); ?></strong>
                                                </td>
                                                <td>
                                                    <?php if ($event['autohaus_name']): ?>
                                                        <span class="autohaus-badge">
                                                            <i class="fas fa-building"></i>
                                                            <?php echo htmlspecialchars($event['autohaus_name']); ?>
                                                        </span>
                                                    <?php else: ?>
                                                        <span class="text-muted">Kein Autohaus</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo date('d.m.Y', strtotime($event['event_date'])); ?></td>
                                                <td>
                                                    <?php echo date('H:i', strtotime($event['start_time'])); ?> - 
                                                    <?php echo date('H:i', strtotime($event['end_time'])); ?>
                                                </td>
                                                <td><?php echo $event['max_monteure']; ?></td>
                                                <td>
                                                    <span class="appointment-count">
                                                        <?php echo $event['booked_count']; ?>/<?php echo $event['appointment_count']; ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="status-badge status-<?php echo $event['is_active'] ? 'active' : 'inactive'; ?>">
                                                        <?php echo $event['is_active'] ? 'Aktiv' : 'Inaktiv'; ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <a href="events.php?action=edit&id=<?php echo $event['id']; ?>" 
                                                           class="ios-btn ios-btn-secondary" title="Bearbeiten">
                                                            <i class="fas fa-pencil"></i>
                                                        </a>
                                                        <a href="appointments.php?event_id=<?php echo $event['id']; ?>" 
                                                           class="ios-btn ios-btn-primary" title="Termine anzeigen">
                                                            <i class="fas fa-calendar-check"></i>
                                                        </a>
                                                        <?php if ($event['appointment_count'] == 0): ?>
                                                            <button onclick="showDeleteEventModal(<?php echo $event['id']; ?>, '<?php echo htmlspecialchars($event['name']); ?>', '<?php echo date('d.m.Y', strtotime($event['event_date'])); ?>')" 
                                                                    class="ios-btn ios-btn-danger" title="Löschen">
                                                                <i class="fas fa-trash-can"></i>
                                                            </button>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </main>
    </div>

    <script src="../assets/js/main.js"></script>
    <script src="assets/js/admin.js"></script>
    <script src="assets/js/unified-sidebar.js"></script>
    
    
    <style>
        /* Apple-Volvo Hybrid Design System */
        :root {
            /* Volvo Brand Colors */
            --volvo-blue: #0066CC;
            --volvo-blue-dark: #004C99;
            --volvo-blue-light: #E6F2FF;
            --volvo-gray: #2C3E50;
            --volvo-gray-light: #ECF0F1;
            --volvo-accent: #F39C12;
            
            /* Apple-inspired Neutrals */
            --apple-white: #FFFFFF;
            --apple-gray-1: #F2F2F7;
            --apple-gray-2: #E5E5EA;
            --apple-gray-3: #D1D1D6;
            --apple-gray-4: #C7C7CC;
            --apple-gray-5: #AEAEB2;
            --apple-gray-6: #8E8E93;
            --apple-gray-7: #636366;
            --apple-gray-8: #48484A;
            --apple-gray-9: #3A3A3C;
            --apple-gray-10: #1C1C1E;
            
            /* Design System Variables */
            --color-primary: var(--volvo-blue);
            --color-primary-dark: var(--volvo-blue-dark);
            --color-primary-light: var(--volvo-blue-light);
            --color-accent: var(--volvo-accent);
            
            --surface-primary: var(--apple-white);
            --surface-secondary: var(--apple-gray-1);
            --surface-tertiary: var(--apple-gray-2);
            --surface-quaternary: var(--apple-gray-3);
            
            --text-primary: var(--apple-gray-10);
            --text-secondary: var(--apple-gray-7);
            --text-tertiary: var(--apple-gray-6);
            --text-quaternary: var(--apple-gray-5);
            
            --border-color: var(--apple-gray-3);
            --border-color-light: var(--apple-gray-2);
            --border-color-dark: var(--apple-gray-4);
            
            /* Typography */
            --font-family: 'SF Pro Display', 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            --font-size-xs: 12px;
            --font-size-sm: 14px;
            --font-size-base: 16px;
            --font-size-lg: 18px;
            --font-size-xl: 20px;
            --font-size-2xl: 24px;
            --font-size-3xl: 30px;
            --font-size-4xl: 36px;
            
            --font-weight-light: 300;
            --font-weight-normal: 400;
            --font-weight-medium: 500;
            --font-weight-semibold: 600;
            --font-weight-bold: 700;
            --font-weight-extrabold: 800;
            
            /* Spacing (8pt Grid) */
            --spacing-xs: 4px;
            --spacing-sm: 8px;
            --spacing-md: 16px;
            --spacing-lg: 24px;
            --spacing-xl: 32px;
            --spacing-2xl: 48px;
            --spacing-3xl: 64px;
            --spacing-4xl: 96px;
            
            /* Border Radius */
            --border-radius: 8px;
            --border-radius-sm: 6px;
            --border-radius-lg: 12px;
            --border-radius-xl: 16px;
            --border-radius-2xl: 24px;
            --border-radius-full: 50%;
            
            /* Shadows */
            --shadow-sm: 0 1px 2px rgba(0, 0, 0, 0.05);
            --shadow: 0 1px 3px rgba(0, 0, 0, 0.1), 0 1px 2px rgba(0, 0, 0, 0.06);
            --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.07), 0 2px 4px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.1), 0 4px 6px rgba(0, 0, 0, 0.05);
            --shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.1), 0 10px 10px rgba(0, 0, 0, 0.04);
            --shadow-2xl: 0 25px 50px rgba(0, 0, 0, 0.25);
            
            /* Transitions */
            --transition-fast: 150ms cubic-bezier(0.4, 0, 0.2, 1);
            --transition-normal: 300ms cubic-bezier(0.4, 0, 0.2, 1);
            --transition-slow: 500ms cubic-bezier(0.4, 0, 0.2, 1);
        }

        /* Dark Theme Variables */
        [data-theme="dark"] {
            --surface-primary: var(--apple-gray-10);
            --surface-secondary: var(--apple-gray-9);
            --surface-tertiary: var(--apple-gray-8);
            --surface-quaternary: var(--apple-gray-7);
            
            --text-primary: var(--apple-white);
            --text-secondary: var(--apple-gray-3);
            --text-tertiary: var(--apple-gray-4);
            --text-quaternary: var(--apple-gray-5);
            
            --border-color: var(--apple-gray-7);
            --border-color-light: var(--apple-gray-8);
            --border-color-dark: var(--apple-gray-6);
        }

        /* Base Styles */
        body {
            font-family: var(--font-family);
            background: linear-gradient(135deg, var(--surface-secondary) 0%, var(--surface-tertiary) 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
            transition: all var(--transition-normal);
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        .admin-container {
            display: flex;
            min-height: 100vh;
        }

        /* Layout */
        .admin-main {
            flex: 1;
            padding: var(--spacing-xl);
            overflow-y: auto;
            background: var(--surface-secondary);
            transition: margin-left var(--transition-normal);
            min-height: 100vh;
        }

        .modern-sidebar.collapsed + .admin-main {
            margin-left: 80px;
        }

        /* Events Header */
        .events-header {
            margin-bottom: var(--spacing-2xl);
        }

        .events-hero {
            background: linear-gradient(135deg, var(--color-primary) 0%, var(--color-primary-dark) 100%);
            border-radius: var(--border-radius-xl);
            padding: var(--spacing-3xl);
            position: relative;
            overflow: hidden;
            box-shadow: var(--shadow-xl);
        }

        .events-hero::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grid" width="10" height="10" patternUnits="userSpaceOnUse"><path d="M 10 0 L 0 0 0 10" fill="none" stroke="rgba(255,255,255,0.1)" stroke-width="0.5"/></pattern></defs><rect width="100" height="100" fill="url(%23grid)"/></svg>');
            opacity: 0.3;
        }

        .events-hero-content {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            position: relative;
            z-index: 1;
        }

        .events-title-section {
            flex: 1;
        }

        .events-title {
            font-size: var(--font-size-4xl);
            font-weight: var(--font-weight-bold);
            color: var(--apple-white);
            margin: 0 0 var(--spacing-md) 0;
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }

        .events-title i {
            font-size: var(--font-size-3xl);
            opacity: 0.9;
        }

        .events-subtitle {
            font-size: var(--font-size-lg);
            color: rgba(255, 255, 255, 0.8);
            margin: 0 0 var(--spacing-lg) 0;
            font-weight: var(--font-weight-normal);
        }

        .events-meta {
            display: flex;
            gap: var(--spacing-xl);
            flex-wrap: wrap;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            color: rgba(255, 255, 255, 0.9);
            font-size: var(--font-size-sm);
            font-weight: var(--font-weight-medium);
        }

        .meta-item i {
            font-size: var(--font-size-base);
            opacity: 0.8;
        }

        .events-actions {
            display: flex;
            gap: var(--spacing-md);
            align-items: center;
        }

        .btn-primary {
            background: rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--border-radius);
            padding: var(--spacing-sm) var(--spacing-lg);
            color: var(--apple-white);
            font-size: var(--font-size-sm);
            font-weight: var(--font-weight-semibold);
            font-family: var(--font-family);
            cursor: pointer;
            transition: all var(--transition-normal);
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            backdrop-filter: blur(10px);
            text-decoration: none;
        }

        .btn-primary:hover {
            background: rgba(255, 255, 255, 0.3);
            border-color: rgba(255, 255, 255, 0.5);
            transform: translateY(-1px);
        }

        /* Events Content */
        .events-content {
            background: var(--surface-primary);
            border-radius: var(--border-radius-xl);
            padding: var(--spacing-2xl);
            box-shadow: var(--shadow-lg);
            border: 1px solid var(--border-color-light);
        }

        /* Content Card */
        .content-card {
            background: var(--surface-primary);
            border-radius: var(--border-radius-lg);
            box-shadow: var(--shadow-md);
            border: 1px solid var(--border-color-light);
            overflow: hidden;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-lg) var(--spacing-xl);
            border-bottom: 1px solid var(--border-color-light);
            background: var(--surface-secondary);
        }

        .card-header h3 {
            margin: 0;
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            font-family: var(--font-family);
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }

        /* iOS 26 Button Design */
        .ios-btn {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            padding: var(--spacing-sm) var(--spacing-md);
            border: none;
            border-radius: var(--border-radius-lg);
            font-family: var(--font-family);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-semibold);
            cursor: pointer;
            transition: all var(--transition-fast);
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(20px);
            box-shadow: var(--shadow-sm);
            text-decoration: none;
            min-height: 32px;
        }

        .ios-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0.05) 100%);
            opacity: 0;
            transition: opacity var(--transition-fast);
        }

        .ios-btn:hover::before {
            opacity: 1;
        }

        .ios-btn:active {
            transform: scale(0.95);
        }

        .ios-btn i {
            font-size: var(--font-size-sm);
            transition: transform var(--transition-fast);
        }

        .ios-btn:hover i {
            transform: scale(1.1);
        }

        /* iOS Button Variants */
        .ios-btn-primary {
            background: linear-gradient(135deg, var(--color-primary) 0%, var(--color-primary-dark) 100%);
            color: var(--apple-white);
            box-shadow: 0 4px 12px rgba(0, 102, 204, 0.3);
        }

        .ios-btn-primary:hover {
            box-shadow: 0 6px 16px rgba(0, 102, 204, 0.4);
            transform: translateY(-1px);
        }

        .ios-btn-secondary {
            background: rgba(142, 142, 147, 0.12);
            color: var(--text-primary);
            border: 1px solid rgba(142, 142, 147, 0.2);
        }

        .ios-btn-secondary:hover {
            background: rgba(142, 142, 147, 0.18);
            border-color: rgba(142, 142, 147, 0.3);
        }

        .ios-btn-danger {
            background: linear-gradient(135deg, #FF3B30 0%, #D70015 100%);
            color: var(--apple-white);
            box-shadow: 0 4px 12px rgba(255, 59, 48, 0.3);
        }

        .ios-btn-danger:hover {
            box-shadow: 0 6px 16px rgba(255, 59, 48, 0.4);
        }

        /* Form Styles */
        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: var(--spacing-lg);
            margin-top: var(--spacing-lg);
        }

        .form-group {
            display: flex;
            flex-direction: column;
        }

        .form-group label {
            font-weight: var(--font-weight-medium);
            color: var(--text-primary);
            margin-bottom: var(--spacing-sm);
            font-size: var(--font-size-sm);
            font-family: var(--font-family);
        }

        .form-group input,
        .form-group select {
            padding: var(--spacing-sm) var(--spacing-md);
            border: 1px solid var(--border-color);
            border-radius: var(--border-radius);
            font-size: var(--font-size-sm);
            font-family: var(--font-family);
            background: var(--surface-primary);
            color: var(--text-primary);
            transition: all var(--transition-normal);
        }

        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px rgba(0, 102, 204, 0.1);
        }

        .checkbox-label {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            cursor: pointer;
            font-weight: var(--font-weight-medium);
            color: var(--text-primary);
        }

        .form-actions {
            display: flex;
            justify-content: flex-end;
            gap: var(--spacing-md);
            margin-top: var(--spacing-xl);
            padding-top: var(--spacing-lg);
            border-top: 1px solid var(--border-color-light);
        }

        /* Table Styles */
        .events-table {
            overflow-x: auto;
        }

        .events-table table {
            width: 100%;
            border-collapse: collapse;
            font-size: var(--font-size-sm);
        }

        .events-table th {
            background: var(--surface-tertiary);
            color: var(--text-primary);
            font-weight: var(--font-weight-semibold);
            padding: var(--spacing-md) var(--spacing-sm);
            text-align: left;
            border-bottom: 2px solid var(--border-color);
            font-family: var(--font-family);
        }

        .events-table td {
            padding: var(--spacing-md) var(--spacing-sm);
            border-bottom: 1px solid var(--border-color-light);
            color: var(--text-primary);
            vertical-align: middle;
        }

        .events-table tbody tr:hover {
            background: var(--surface-secondary);
        }

        .events-table tbody tr:nth-child(even) {
            background: var(--surface-primary);
        }

        .events-table tbody tr:nth-child(even):hover {
            background: var(--surface-secondary);
        }

        /* Action Buttons */
        .action-buttons {
            display: flex;
            gap: var(--spacing-xs);
            flex-wrap: wrap;
        }

        /* Status Badge */
        .status-badge {
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-semibold);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .status-badge.status-active {
            background: rgba(52, 199, 89, 0.1);
            color: #34C759;
            border: 1px solid rgba(52, 199, 89, 0.2);
        }

        .status-badge.status-inactive {
            background: rgba(142, 142, 147, 0.1);
            color: var(--text-secondary);
            border: 1px solid rgba(142, 142, 147, 0.2);
        }

        /* Autohaus Badge */
        .autohaus-badge {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            padding: var(--spacing-xs) var(--spacing-sm);
            background: rgba(0, 102, 204, 0.1);
            color: var(--color-primary);
            border-radius: var(--border-radius-lg);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-medium);
            border: 1px solid rgba(0, 102, 204, 0.2);
        }

        .autohaus-badge i {
            font-size: var(--font-size-xs);
        }

        .text-muted {
            color: var(--text-tertiary);
            font-style: italic;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: var(--spacing-3xl);
            color: var(--text-secondary);
        }

        .empty-state i {
            font-size: var(--font-size-4xl);
            margin-bottom: var(--spacing-lg);
            opacity: 0.5;
        }

        .empty-state p {
            font-size: var(--font-size-lg);
            margin-bottom: var(--spacing-lg);
        }

        /* Alert Styles */
        .alert {
            padding: var(--spacing-md) var(--spacing-lg);
            border-radius: var(--border-radius);
            margin-bottom: var(--spacing-lg);
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-weight: var(--font-weight-medium);
            font-size: var(--font-size-sm);
        }

        .alert-success {
            background: #D1FAE5;
            color: #065F46;
            border: 1px solid #A7F3D0;
        }

        .alert-error {
            background: #FEE2E2;
            color: #991B1B;
            border: 1px solid #FECACA;
        }

        /* Responsive Design */
        @media (max-width: 1200px) {
            .events-hero-content {
                flex-direction: column;
                gap: var(--spacing-lg);
            }
            
            .events-actions {
                width: 100%;
                justify-content: flex-start;
            }
        }

        @media (max-width: 768px) {
            .admin-main {
                padding: var(--spacing-md);
            }
            
            .events-hero {
                padding: var(--spacing-xl);
            }
            
            .events-title {
                font-size: var(--font-size-2xl);
            }
            
            .events-meta {
                gap: var(--spacing-md);
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .action-buttons {
                flex-direction: column;
                gap: var(--spacing-sm);
            }
            
            .ios-btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>

    <!-- Delete Event Modal -->
    <div id="delete-event-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>
                    <i class="fas fa-trash"></i>
                    Event löschen
                </h3>
                <button class="modal-close" onclick="closeDeleteEventModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <div style="text-align: center; margin-bottom: 2rem;">
                    <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #dc3545 0%, #c82333 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1.5rem;">
                        <i class="fas fa-trash" style="color: #fff; font-size: 2rem;"></i>
                    </div>
                    <h4 style="margin: 0 0 1rem 0; color: #dc3545;">Event wirklich löschen?</h4>
                    <p style="margin: 0; color: #6c757d;">
                        Diese Aktion kann nicht rückgängig gemacht werden!
                    </p>
                </div>
                
                <div style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; margin-bottom: 1.5rem;">
                    <div style="margin-bottom: 1rem;">
                        <div style="color: #495057; font-weight: 600; margin-bottom: 0.5rem;">Event:</div>
                        <div style="color: #003057; font-size: 1.1rem; font-weight: 500;" id="delete-event-name"></div>
                    </div>
                    <div style="border-top: 1px solid #e9ecef; padding-top: 1rem; margin-top: 1rem;">
                        <div style="color: #495057; font-weight: 600; margin-bottom: 0.5rem;">Datum:</div>
                        <div style="color: #0056b3; font-size: 1rem; font-weight: 500;" id="delete-event-date"></div>
                    </div>
                </div>
                
                <div style="display: flex; justify-content: flex-end; gap: 1rem; border-top: 2px solid #e9ecef; padding-top: 1.5rem; margin-top: 1.5rem;">
                    <button onclick="closeDeleteEventModal()" style="padding: 0.75rem 1.5rem; background: #6c757d; color: #fff; border: none; border-radius: 8px; cursor: pointer; font-size: 1rem; font-weight: 500;">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button onclick="confirmDeleteEvent()" style="padding: 0.75rem 1.5rem; background: #dc3545; color: #fff; border: none; border-radius: 8px; cursor: pointer; font-size: 1rem; font-weight: 500; box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);">
                        <i class="fas fa-trash"></i> Endgültig löschen
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Store event ID for deletion
        let eventToDelete = null;

        // Show delete event modal
        function showDeleteEventModal(eventId, eventName, eventDate) {
            eventToDelete = eventId;
            document.getElementById('delete-event-name').textContent = eventName;
            document.getElementById('delete-event-date').textContent = eventDate;
            document.getElementById('delete-event-modal').style.display = 'block';
        }

        // Close delete event modal
        function closeDeleteEventModal() {
            document.getElementById('delete-event-modal').style.display = 'none';
            eventToDelete = null;
        }

        // Confirm delete event
        function confirmDeleteEvent() {
            if (!eventToDelete) {
                alert('Fehler: Kein Event zum Löschen ausgewählt');
                return;
            }

            // Redirect to delete action
            window.location.href = `events.php?action=delete&id=${eventToDelete}`;
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const deleteModal = document.getElementById('delete-event-modal');
            if (event.target === deleteModal) {
                closeDeleteEventModal();
            }
        }
    </script>

    <style>
        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 10000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0, 0, 0, 0.4);
            backdrop-filter: blur(8px);
        }

        .modal-content {
            background-color: #fff;
            margin: 5% auto;
            padding: 0;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 500px;
            width: 90%;
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem;
            border-bottom: 2px solid #e9ecef;
        }

        .modal-header h3 {
            margin: 0;
            color: #003057;
            font-size: 1.25rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 8px;
            transition: all 0.3s ease;
        }

        .modal-close:hover {
            background: #f8f9fa;
            color: #003057;
        }

        .modal-body {
            padding: 1.5rem;
        }
    </style>
</body>
</html>
