<?php
/**
 * Debug-Script: Zeigt vollständige HelloMateo API-Antwort beim Senden einer WhatsApp-Nachricht
 */

session_start();

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../api/autohaus_whatsapp_notifications.php';

header('Content-Type: application/json');

// Test-Daten
$testData = [
    'customer_name' => 'Uludag',
    'phone' => '01782465658',
    'email' => 'test@example.com',
    'license_plate' => 'TEST-123',
    'autohaus_id' => 1,
];

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Prüfe und erstelle whatsapp_logs Tabelle und Spalten falls nötig
    $whatsappNotifications = new AutohausWhatsAppNotifications($testData['autohaus_id']);
    $whatsappNotifications->ensureWhatsAppLogsTableAndColumnsExist();

    // Finde ein aktives Event für das Autohaus
    $stmt = $conn->prepare("
        SELECT id, name, event_date, start_time, end_time, slot_duration, max_monteure
        FROM events 
        WHERE autohaus_id = ? AND is_active = 1 AND event_date >= CURDATE()
        ORDER BY event_date ASC, start_time ASC
        LIMIT 1
    ");
    $stmt->execute([$testData['autohaus_id']]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$event) {
        echo json_encode(['success' => false, 'error' => 'Kein aktives Event für dieses Autohaus gefunden']);
        exit;
    }

    // Finde einen verfügbaren Slot
    $eventDate = $event['event_date'];
    $startTime = strtotime($event['start_time']);
    $endTime = strtotime($event['end_time']);
    $slotDuration = $event['slot_duration'] * 60;
    $maxMonteure = $event['max_monteure'];

    $bookedSlotsStmt = $conn->prepare("
        SELECT slot_time, COUNT(*) as booked_count
        FROM appointments 
        WHERE event_id = ? AND slot_date = ? AND status IN ('booked', 'approved', 'vorbereitet')
        GROUP BY slot_time
    ");
    $bookedSlotsStmt->execute([$event['id'], $eventDate]);
    $bookedSlots = $bookedSlotsStmt->fetchAll(PDO::FETCH_KEY_PAIR);

    $availableSlot = null;
    $currentTime = $startTime;

    while ($currentTime < $endTime) {
        $slotTime = date('H:i:s', $currentTime);
        $bookedCount = $bookedSlots[$slotTime] ?? 0;

        if ($bookedCount < $maxMonteure) {
            $availableSlot = $slotTime;
            break;
        }
        $currentTime += $slotDuration;
    }
    
    if (!$availableSlot) {
        echo json_encode([
            'success' => false,
            'error' => 'Kein verfügbarer Slot für dieses Event gefunden'
        ]);
        exit;
    }
    
    // Erstelle Termin
    $stmt = $conn->prepare("
        INSERT INTO appointments (
            event_id, autohaus_id, customer_name, license_plate, phone, email, 
            slot_time, slot_date, status, source
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'booked', 'admin_debug')
    ");
    
    $stmt->execute([
        $event['id'],
        $testData['autohaus_id'],
        $testData['customer_name'],
        $testData['license_plate'],
        $testData['phone'],
        $testData['email'],
        $availableSlot,
        $eventDate
    ]);
    
    $appointmentId = $conn->lastInsertId();
    
    // Hole vollständige Appointment-Daten für WhatsApp
    $stmt = $conn->prepare("
        SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
               ah.name as autohaus_name, ah.short_name as autohaus_short_name
        FROM appointments a
        LEFT JOIN events e ON a.event_id = e.id
        LEFT JOIN autohaus ah ON a.autohaus_id = ah.id
        WHERE a.id = ?
    ");
    $stmt->execute([$appointmentId]);
    $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Sende WhatsApp-Bestätigung mit detailliertem Logging
    $whatsapp = new AutohausWhatsAppNotifications($testData['autohaus_id']);
    
    // Aktiviere detailliertes Logging
    error_log("=== DEBUG: WhatsApp Send Start ===");
    error_log("Appointment ID: " . $appointmentId);
    error_log("Phone: " . $testData['phone']);
    error_log("Autohaus ID: " . $testData['autohaus_id']);
    
    $result = $whatsapp->sendAppointmentNotification($appointmentId, 'confirmation');
    
    error_log("=== DEBUG: WhatsApp Send Result ===");
    error_log("Result: " . json_encode($result, JSON_PRETTY_PRINT));
    
    // Hole die letzten Logs aus whatsapp_logs
    $logStmt = $conn->prepare("
        SELECT * FROM whatsapp_logs 
        WHERE appointment_id = ? 
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    $logStmt->execute([$appointmentId]);
    $log = $logStmt->fetch(PDO::FETCH_ASSOC);
    
    // Lese Debug-Log-Datei
    $debugLogFile = __DIR__ . '/../logs/whatsapp_contact_debug.log';
    $debugLog = '';
    if (file_exists($debugLogFile)) {
        // Lese die letzten 100 Zeilen
        $lines = file($debugLogFile);
        $debugLog = implode('', array_slice($lines, -100));
    }
    
    // Lese PHP Error Log (letzte 50 Zeilen mit "HelloMateoAPI")
    $errorLogFile = ini_get('error_log');
    $errorLog = '';
    if ($errorLogFile && file_exists($errorLogFile)) {
        $lines = file($errorLogFile);
        $relevantLines = array_filter($lines, function($line) {
            return strpos($line, 'HelloMateoAPI') !== false || 
                   strpos($line, 'WhatsApp') !== false ||
                   strpos($line, 'sendMessage') !== false;
        });
        $errorLog = implode('', array_slice($relevantLines, -50));
    }
    
    echo json_encode([
        'success' => $result['success'],
        'appointment_id' => $appointmentId,
        'whatsapp_result' => $result,
        'log_entry' => $log,
        'debug_log' => $debugLog,
        'error_log_snippet' => $errorLog,
        'message' => $result['success'] 
            ? 'WhatsApp-Nachricht gesendet (prüfe Logs für Details)' 
            : 'Fehler beim Senden: ' . ($result['error'] ?? 'Unbekannt')
    ], JSON_PRETTY_PRINT);

} catch (Exception $e) {
    echo json_encode([
        'success' => false, 
        'error' => $e->getMessage(), 
        'trace' => $e->getTraceAsString()
    ], JSON_PRETTY_PRINT);
}
?>




